// kmeans.c
// Evan Lord
// Created: July 13, 2007
// Last Modified: July 31, 2007


#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <mcheck.h>
#include "constants.h"
#include "struct_def.h"


void CheckConstraintsSatisfied(struct DATA *);
int ComparisonFunction(const void *, const void *);
float CVQEDistance(int, int, struct DATA *);
float Distance(float *, float *, int);
void PrintData(struct DATA *);
void UpdateClosestClusters(struct DATA *);
void UpdateConstraints(struct SUPER_POINT *, struct DATA *);
int ViolateConstraints(struct SUPER_POINT *, int, struct DATA *);


float * distance_array;


// This function calculates the constrained vector quantization error.
// It receives the DATA struct containing the current cluster assignments
// and updates the struct with the CVQE

float CalculateCVQE(struct DATA * data, struct SUPER_POINT * super_point_one, int int_one, struct SUPER_POINT * super_point_two, int int_two)
{

  int int_loop_count;                 // loop counter
  
  int int_cluster_one;
  int int_cluster_two;

  int int_point_one;
  int int_point_two;
  int int_further_point;

  int int_next_centroid;
  int int_closest_centroid;

  float CVQEj;                       // CVQE of individual cluster
  float CVQE;

  float flt_point_one_distance;
  float flt_point_two_distance;
  float flt_shortest_distance;
  float flt_next_distance;

  struct SUPER_POINT * cur_super;     // current "super point" in cluster

  struct DATA_POINT * cur_node;       // current data point in cluster

  struct CONSTRAINT * cur_constraint; // current constraint in which data point is involved

  CVQE = 0;                     // initialize the VQE to zero for this iteration
  
  // loop through all clusters
  for(int_loop_count = 0; int_loop_count < data->int_num_clusters; int_loop_count++){ 
    
    CVQEj = 0;   // initialize the individual cluster VQE to zero;
    
    if(int_loop_count == int_one){
      cur_node = super_point_one->head;  // start at the first data point in the current "super point"	  
      while(cur_node != NULL){     // and loop through all data points in the current "super point"
	
	//printf("\n\n%d\n\n", cur_node->int_index);

	CVQEj = CVQEj + Distance(data->data_array[cur_node->int_index], data->cluster_centroids[int_loop_count], data->int_dimensions);

	cur_constraint = data->ML_constraint_involvement[cur_node->int_index][HEAD];
	while(cur_constraint != NULL){
	  int_cluster_one = data->ML_constraints_array[cur_constraint->int_constraint_index][2];
	  int_cluster_two = data->ML_constraints_array[cur_constraint->int_constraint_index][3];

	  if((int_cluster_one != int_cluster_two) && (int_cluster_one != -1) && (int_cluster_two != -1) && (data->ML_constraints_array[cur_constraint->int_constraint_index][4] == 0)){

	    CVQEj = CVQEj + Distance(data->cluster_centroids[int_cluster_one], data->cluster_centroids[int_cluster_two], data->int_dimensions);


	    data->ML_constraints_array[cur_constraint->int_constraint_index][4] = 1;
	  }	 

	  cur_constraint = cur_constraint->pNext;
	  
	}
	 
	cur_constraint = data->CL_constraint_involvement[cur_node->int_index][HEAD];
	
	while(cur_constraint != NULL){

	  int_cluster_one = data->CL_constraints_array[cur_constraint->int_constraint_index][2];
	  int_cluster_two = data->CL_constraints_array[cur_constraint->int_constraint_index][3];
	  if((int_cluster_one == int_cluster_two) && (int_cluster_one != -1) && (data->CL_constraints_array[cur_constraint->int_constraint_index][4] == 0)){

	    CVQEj = CVQEj + Distance(data->cluster_centroids[int_cluster_two], data->cluster_centroids[data->closest_clusters[int_cluster_two]], data->int_dimensions);

	    data->CL_constraints_array[cur_constraint->int_constraint_index][4] = 1;
	  }
	  cur_constraint = cur_constraint->pNext;
	}
	cur_node = cur_node->pSetNext;
      }
    }
    
    if(int_loop_count == int_two){
      cur_node = super_point_two->head;  // start at the first data point in the current "super point"	  
      while(cur_node != NULL){     // and loop through all data points in the current "super point"
	CVQEj = CVQEj + Distance(data->data_array[cur_node->int_index], data->cluster_centroids[int_loop_count], data->int_dimensions);


	cur_constraint = data->ML_constraint_involvement[cur_node->int_index][HEAD];
	while(cur_constraint != NULL){
	  int_cluster_one = data->ML_constraints_array[cur_constraint->int_constraint_index][2];
	  int_cluster_two = data->ML_constraints_array[cur_constraint->int_constraint_index][3];
	  if((int_cluster_one != int_cluster_two) && (int_cluster_one != -1) && (int_cluster_two != -1)){
	    //&& (data->ML_constraints_array[cur_constraint->int_constraint_index][4] == 0)){
	    CVQEj = CVQEj + Distance(data->cluster_centroids[int_cluster_one], data->cluster_centroids[int_cluster_two], data->int_dimensions);


	    //data->ML_constraints_array[cur_constraint->int_constraint_index][4] = 1;
	  }
	  cur_constraint = cur_constraint->pNext;
	}
	
	cur_constraint = data->CL_constraint_involvement[cur_node->int_index][HEAD];
	
	while(cur_constraint != NULL){
	  int_cluster_one = data->CL_constraints_array[cur_constraint->int_constraint_index][2];
	  int_cluster_two = data->CL_constraints_array[cur_constraint->int_constraint_index][3];
	  if((int_cluster_one == int_cluster_two) && (int_cluster_one != -1) && (data->CL_constraints_array[cur_constraint->int_constraint_index][4] == 0)){
	    CVQEj = CVQEj + Distance(data->cluster_centroids[int_cluster_two], data->cluster_centroids[data->closest_clusters[int_cluster_two]], data->int_dimensions);


	    data->CL_constraints_array[cur_constraint->int_constraint_index][4] = 1;
	  }
	  cur_constraint = cur_constraint->pNext;
	}
	cur_node = cur_node->pSetNext;
      }
    }
    
    // point to head "super point" of cluster list
    cur_super = data->clusters[int_loop_count];

    if(cur_super != NULL){     // if there is at least one "super point" in the current cluster
      
      while(cur_super != NULL){      // loop through list of all "super points" in current cluster
	 
	if((cur_super != super_point_one) && (cur_super != super_point_two)){
	  
	  cur_node = cur_super->head;  // start at the first data point in the current "super point"
	  
	  while(cur_node != NULL){     // and loop through all data points in the current "super point"
	    
	    
	    CVQEj = CVQEj + Distance(data->data_array[cur_node->int_index], data->cluster_centroids[int_loop_count], data->int_dimensions);
	    
	    cur_constraint = data->ML_constraint_involvement[cur_node->int_index][HEAD];
	    
	    while(cur_constraint != NULL){
	      int_cluster_one = data->ML_constraints_array[cur_constraint->int_constraint_index][2];
	      int_cluster_two = data->ML_constraints_array[cur_constraint->int_constraint_index][3];
	      if((int_cluster_one != int_cluster_two) && (int_cluster_one != -1) && (int_cluster_two != -1)){
		//&& (data->ML_constraints_array[cur_constraint->int_constraint_index][4] == 0)){
		CVQEj = CVQEj + Distance(data->cluster_centroids[int_cluster_one], data->cluster_centroids[int_cluster_two], data->int_dimensions);

		//data->ML_constraints_array[cur_constraint->int_constraint_index][4] = 1;
	      }
	      cur_constraint = cur_constraint->pNext;
	    }
	    
	    cur_constraint = data->CL_constraint_involvement[cur_node->int_index][HEAD];
	    
	    while(cur_constraint != NULL){
	      int_point_one = data->CL_constraints_array[cur_constraint->int_constraint_index][0];
	      int_point_two = data->CL_constraints_array[cur_constraint->int_constraint_index][1];
	      int_cluster_one = data->CL_constraints_array[cur_constraint->int_constraint_index][2];
	      int_cluster_two = data->CL_constraints_array[cur_constraint->int_constraint_index][3];
	      if((int_cluster_one == int_cluster_two) && (int_cluster_one != -1)){ 

		if(data->int_algorithm_version == 2){
		  CVQEj = CVQEj + Distance(data->cluster_centroids[int_cluster_two], data->cluster_centroids[data->closest_clusters[int_cluster_two]], data->int_dimensions);
		}
		else{
		  if(data->int_algorithm_version == 3){
		    flt_point_one_distance = Distance(data->cluster_centroids[int_cluster_one], data->super_point_centroids[int_point_one], data->int_dimensions);
		    flt_point_two_distance = Distance(data->cluster_centroids[int_cluster_one], data->super_point_centroids[int_point_two], data->int_dimensions);
		    if(flt_point_one_distance >= flt_point_two_distance){
		      int_further_point = int_point_one;
		    }
		    else{
		      int_further_point = int_point_two;
		    }
		    int_next_centroid = (int_cluster_one + 1) % data->int_num_clusters;
		    flt_shortest_distance = Distance(data->cluster_centroids[int_next_centroid], data->super_point_centroids[int_further_point], data->int_dimensions);
		    int_closest_centroid = int_next_centroid;
		    for(int_loop_count = 1; int_loop_count < (data->int_num_clusters - 1); int_loop_count++){
		      int_next_centroid = (int_loop_count + int_cluster_one + 1) % data->int_num_clusters;
		      flt_next_distance = Distance(data->cluster_centroids[int_next_centroid], data->super_point_centroids[int_further_point], data->int_dimensions);
		      if(flt_next_distance < flt_shortest_distance){
			flt_shortest_distance = flt_next_distance;
			int_closest_centroid = int_next_centroid;
		      }
		    }
		    CVQEj = CVQEj + flt_shortest_distance;
		  }
		}

	      }
	      cur_constraint = cur_constraint->pNext;
	    }
	    
	    cur_node = cur_node->pSetNext;      // move to the next data point in this "super point"
	    
	  }
	}

	cur_super = cur_super->pClusterNext;  // move to the next "super point" in the cluster

	
      }
    } 
     
    CVQEj = CVQEj / 2;     // divide by two to get the individual cluster VQE
      
    CVQE = CVQE + CVQEj;  // add the individual cluster VQE to the total VQE
  }

  return CVQE;
}


// This function calculates the vector quantization error.  It receives
// the DATA struct containing the current cluster assignments and updates
// the struct with the VQE

void CalculateVQE(struct DATA * data)
{
  int int_loop_count;                // loop counter
  
  float VQEj;                       // VQE of individual cluster

  struct SUPER_POINT * cur_super;    // current "super point" in cluster

  struct DATA_POINT * cur_node;      // current data point in cluster

  data->VQE = 0;                     // initialize the VQE to zero for this iteration
  
  // loop through all clusters
  for(int_loop_count = 0; int_loop_count < data->int_num_clusters; int_loop_count++){ 
    
    // point to head "super point" of cluster list
    cur_super = data->clusters[int_loop_count];
    
    VQEj = 0;   // initialize the individual cluster VQE to zero;
    
    if(cur_super != NULL){     // if there is at least one "super point" in the current cluster
      
      while(cur_super != NULL){      // loop through list of all "super points" in current cluster
	    
	cur_node = cur_super->head;  // start at the first data point in the current "super point"
	
	while(cur_node != NULL){     // and loop through all data points in the current "super point"
	  
	  // square the distance between each point and the cluster centroid and maintain a sum of the results
	  VQEj = VQEj + Distance(data->data_array[cur_node->int_index], data->cluster_centroids[int_loop_count], data->int_dimensions) * Distance(data->data_array[cur_node->int_index], data->cluster_centroids[int_loop_count], data->int_dimensions);
	  
	  cur_node = cur_node->pSetNext;      // move to the next data point in this "super point"
	  
	}
	
	cur_super = cur_super->pClusterNext;  // move to the next "super point" in the cluster
      }
      
      VQEj = VQEj / 2;     // divide by two to get the individual cluster VQE
      
      data->VQE = data->VQE + VQEj;  // add the individual cluster VQE to the total VQE
    } 
  }
}


int ComparisonFunction(const void * index_one, const void * index_two)
{
  if(distance_array[*(int *)index_one] < distance_array[*(int *)index_two]){
    return -1;
  }
  else{
    if(distance_array[*(int *)index_one] == distance_array[*(int *)index_two]){
      return 0;
    }
    else{
      return 1;
    }
  }
}


// This function randomly assigns each "super point" to one of k clusters.  
// It then calculates the coordinates of the k centroids based on those assignments.  
// The function receives the DATA struct and updates it with the cluster centroids.

void InitializeCentroids(struct DATA * data)
{
  // pointers for linked list manipulation
  struct SUPER_POINT * new_super;
  struct SUPER_POINT * prev_super;
  struct SUPER_POINT * cur_super;
  struct SUPER_POINT * next_super;

  // loop counters
  int int_loop_count;
  int int_outer_loop;
  int int_inner_loop;
   
  int int_num_nodes;                   // number of data points in a cluster

  float temp[data->int_dimensions];   // array used in calculating centroid coordinates

  // allocate memory for the array to hold the centroid coordinates
  if((data->cluster_centroids = (float **)malloc(data->int_num_clusters * sizeof(float *))) == NULL){
    fprintf(stderr, "\nMemory allocation error.  Closing program.\n\n");
    exit(1);
  }
  else{
    for(int_loop_count = 0; int_loop_count < data->int_num_clusters; int_loop_count++){
      if((data->cluster_centroids[int_loop_count] = (float *)malloc(data->int_dimensions * sizeof(float))) == NULL){
	fprintf(stderr, "\nMemory allocation error.  Closing program.\n\n");
	exit(1);
      }
    }
  }

  if(data->int_algorithm_version == 2){
    // allocate memory to hold the closest clusters list (h(i))
    if((data->closest_clusters = (int *)malloc(data->int_num_clusters * sizeof(int ))) == NULL){
      fprintf(stderr, "\nMemory allocation error.  Closing program.\n\n");
      exit(1);
    }
    for(int_loop_count = 0; int_loop_count < data->int_num_clusters; int_loop_count++){
      data->closest_clusters[int_loop_count] = -1;  
    }    
  }
  
  /***************************************Unassign points*********************************/

  cur_super = data->super_point_head;       // start at first "super point" in linked list
  while(cur_super != NULL){                 // walk entire list of "super points"
    cur_super->int_cluster = -1;            // set cluster assignment to -1
    cur_super->pClusterNext = NULL;         // set cluster next pointer to NULL
    cur_super->pClusterPrev = NULL;         // set cluster previous pointer to NULL
    cur_super = cur_super->pNext;           // move to next "super point" in linked list
  }
  // loop over all clusters
  for(int_loop_count = 0; int_loop_count < data->int_num_clusters; int_loop_count++){
    data->clusters[int_loop_count] = NULL;  // set each cluster list to NULL
  }

  /*********************************Assign points to clusters*******************************/

  cur_super = data->super_point_head;       // start at the first "super point"
  while(cur_super != NULL){                 // and walk the entire list of all "super points"
    
    prev_super = cur_super->pClusterPrev;   // mark the previous "super point" in cluster list
    next_super = cur_super->pClusterNext;   // mark the next "super point" in cluster list
    
    cur_super->int_cluster = rand()%data->int_num_clusters;  // randomly assign "super point"

    // update pointers to add point to new cluster list
    if(data->clusters[cur_super->int_cluster] == NULL){      // if there are no "super points"
	                                                     // in the new cluster list
      cur_super->pClusterPrev = NULL;
      cur_super->pClusterNext = NULL;
      data->clusters[cur_super->int_cluster] = cur_super;    // make the reassigned "super point"
	                                                     // the head of the cluster list
    }
    else{                                                    // if there are "super points"
	                                                     // currently in the new cluster lis
      new_super = data->clusters[cur_super->int_cluster];    // mark the current head
      new_super->pClusterPrev = cur_super;                   // and connect the reassigned "super
      cur_super->pClusterPrev = NULL;                        // point" to the current head
      cur_super->pClusterNext = new_super;
      data->clusters[cur_super->int_cluster] = cur_super;    // make the reassigned point the
	                                                     // new head of the cluster list
    }
    
    cur_super = cur_super->pNext;    // move to the next "super point" in the list
    
  }

  /************************************Calculate centroids***********************************/

  // loop through all clusters
  for(int_outer_loop = 0; int_outer_loop < data->int_num_clusters; int_outer_loop++){
    
    cur_super = data->clusters[int_outer_loop];    // start at head of current cluster list

    if(cur_super != NULL){                         // if the cluster constains at least one point
      
      // initialize the temp array to zero
      for(int_loop_count = 0; int_loop_count < data->int_dimensions; int_loop_count++){
	temp[int_loop_count] = 0;
      }
      
      int_num_nodes = 0;    // initialize number of nodes in this cluster to zero

      while(cur_super != NULL){        // walk entire list of "super points" in this cluster
	
	// add the weighted super point coordinates to the temp array
	for(int_loop_count = 0; int_loop_count < data->int_dimensions; int_loop_count++){
	  temp[int_loop_count] = temp[int_loop_count] + cur_super->int_weight * data->super_point_centroids[cur_super->int_super_point_num][int_loop_count];
	}

	int_num_nodes = int_num_nodes + cur_super->int_weight;     // maintain the total weight of the cluster
	cur_super = cur_super->pClusterNext;                       // move to the next "super point" in cluster
      }

      // divide sum of weighted coordinates by the weight of the cluster to get centroid coordinates
      for(int_loop_count = 0; int_loop_count < data->int_dimensions; int_loop_count++){
	
	temp[int_loop_count] = temp[int_loop_count] / int_num_nodes;
	
      }
      
      // store the centroid coordinates in the appropriate array
      for(int_inner_loop = 0; int_inner_loop < data->int_dimensions; int_inner_loop++){
	data->cluster_centroids[int_outer_loop][int_inner_loop] = temp[int_inner_loop];
      }
    }
  }


  /**********************************Unassign points****************************************/

  cur_super = data->super_point_head;   // start at first "super point" in linked list
  while(cur_super != NULL){             // walk entire list of "super points"
    cur_super->int_cluster = -1;        // set cluster assignment to -1
    cur_super->pClusterNext = NULL;     // set cluster next pointer to NULL
    cur_super->pClusterPrev = NULL;     // set cluster previous pointer to NULL
    cur_super = cur_super->pNext;       // move to next "super point" in list
  }
  // loop over all clusters
  for(int_loop_count = 0; int_loop_count < data->int_num_clusters; int_loop_count++){
    data->clusters[int_loop_count] = NULL;    // set head of cluster list to NULL
  }

  // Update the closest clusters array (h(i)) if doing CVQE-K-Means
  if(data->int_algorithm_version == 2){
    UpdateClosestClusters(data);
  }

}


// This function performs the K-Means algorithm.  It receives the DATA struct containing
// the data to be analyzed and updates it with the final centroid assignments for each of the 
// data points as well as the final centroid coordinates.
  
int Kmeans(struct DATA * data_to_analyze)
{
  // pointers used in linked list manipulation
  
  struct SUPER_POINT * super_one;
  struct SUPER_POINT * super_two;
  struct SUPER_POINT * cur_super;
  struct SUPER_POINT * new_super;
  struct SUPER_POINT * prev_super;
  struct SUPER_POINT * next_super;

  struct DATA_POINT * cur_node;
  struct DATA_POINT * point_one;
  struct DATA_POINT * point_two;
    
  // loop counters
  int int_outer_loop;
  int int_inner_loop;
  int int_loop_count;

  int int_iteration;            // current iteration of the algorithm
  int int_num_nodes;            // number of points assigned to a given cluster
  int int_num_changes;          // number of points reassigned during one iteration
  
  float temp[data_to_analyze->int_dimensions]; // array used for centroid recalculation

  //float shortest_distance;     // shortest distance between a "super point" and all centroids
  //float distance;              // distance between a "super point" and a centroid
  
  int int_cur_cluster;          // index of the cluster to which a "super point" belongs
  int int_cur_cluster_one;
  int int_cur_cluster_two;
  //int int_constraint_violated;  // flag set if a particular cluster assignment violates a constraint
  int int_valid_cluster;        // flag set if there exists a valid cluster assignment

  int int_other_centroid; 

  float flt_current_CVQE;
  float flt_minimum_CVQE;

  char garbage;

  int index_array[data_to_analyze->int_num_clusters];
 
  int int_cluster_index;
  //int int_best_cluster;

  if((distance_array = (float *)malloc(data_to_analyze->int_num_clusters * sizeof(float))) == NULL){
    fprintf(stderr, "\nMemory allocation error.  Closing program.\n\n");
    exit(1);
  }
     
  int_num_nodes = 0;
  int_valid_cluster = 1;
  int_iteration = 0;    // no iterations of the algorithm have been performed

  int_num_changes = data_to_analyze->int_num_data_points;   // initially all data points are
                                                            // "reassigned" 
  // iterate until no points are reassigned
  while(int_num_changes > TERM_CRITERION * data_to_analyze->int_num_data_points / 100){


    int_iteration++;        // increment iteration number

    int_num_changes = 0;    // reset number of changes to zero


    //PrintData(data_to_analyze);
     

    /*************************Unassign points*********************************/
    
    cur_super = data_to_analyze->super_point_head; // start at first "super point" in linked list
    while(cur_super != NULL){                      // walk entire list of "super points"
      cur_super->pClusterNext = NULL;              // set cluster next pointer to NULL
      cur_super->pClusterPrev = NULL;              // set cluster previous pointer to NULL
      cur_super = cur_super->pNext;                // move to next "super point" in linked list
    }
    // loop over all clusters
    for(int_loop_count = 0; int_loop_count < data_to_analyze->int_num_clusters; int_loop_count++){
      data_to_analyze->clusters[int_loop_count] = NULL;       // set each cluster list to NULL
    }

    for(int_loop_count = 0; int_loop_count < data_to_analyze->int_num_ML_constraints; int_loop_count++){
      data_to_analyze->ML_constraints_array[int_loop_count][2] = -1;
      data_to_analyze->ML_constraints_array[int_loop_count][3] = -1;
    }
    
    for(int_loop_count = 0; int_loop_count < data_to_analyze->int_num_CL_constraints; int_loop_count++){
      data_to_analyze->CL_constraints_array[int_loop_count][2] = -1;
      data_to_analyze->CL_constraints_array[int_loop_count][3] = -1;
    }


    /************************Reassign Data Points************************/
    
    /************************CVQE constrained point assignment**********************************/
    if(data_to_analyze->int_algorithm_version == 2){ 
      for(int_loop_count = 0; int_loop_count < data_to_analyze->int_num_ML_constraints; int_loop_count++){

	super_one = data_to_analyze->ML_constraint_ptrs[int_loop_count][0];
	super_two = data_to_analyze->ML_constraint_ptrs[int_loop_count][1];

	point_one = super_one->head;
	point_two = super_two->head;

	int_cur_cluster_one = super_one->int_cluster;
	int_cur_cluster_two = super_two->int_cluster;

	flt_minimum_CVQE = CalculateCVQE(data_to_analyze, super_one, 0, super_two, 0);
       
	super_one->int_cluster = 0;
	super_two->int_cluster = 0;
	for(int_outer_loop = 0; int_outer_loop < data_to_analyze->int_num_clusters; int_outer_loop++){
	  for(int_inner_loop = 0; int_inner_loop < data_to_analyze->int_num_clusters; int_inner_loop++){
	    flt_current_CVQE = CalculateCVQE(data_to_analyze, super_one, int_outer_loop, super_two, int_inner_loop);
	    if(flt_current_CVQE < flt_minimum_CVQE){
	      flt_minimum_CVQE = flt_current_CVQE;
	      super_one->int_cluster = int_outer_loop;
	      super_two->int_cluster = int_inner_loop;
	    }
	    
	  }
	  
	}


	// if the cluster assignment has changed
	if(int_cur_cluster_one != super_one->int_cluster){
	  // update the number of changes in this iteration
	  int_num_changes = int_num_changes + super_one->int_weight;  

	  cur_node = super_one->head;
	  while(cur_node != NULL){
	    cur_node->int_cluster = super_one->int_cluster;
	    cur_node = cur_node->pSetNext;
	  }
	}

	// remove "super point" from cluster list
	prev_super = super_one->pClusterPrev;
	next_super = super_one->pClusterNext;
	
	if((prev_super == NULL) && (next_super != NULL)){
	  super_one->pClusterNext = NULL;
	  data_to_analyze->clusters[int_cur_cluster_one] = next_super;
	  next_super->pClusterPrev = NULL;
	}
	else{
	  if((prev_super != NULL) && (next_super == NULL)){
	    super_one->pClusterPrev = NULL;
	    prev_super->pClusterNext = NULL;
	  }
	  else{
	    if((prev_super != NULL) && (next_super != NULL)){
	      super_one->pClusterPrev = NULL;
	      super_one->pClusterNext = NULL;
	      prev_super->pClusterNext = next_super;
	      next_super->pClusterPrev = prev_super;
	    }
	  }
	}



	// update pointers to add point to new cluster list
	if(data_to_analyze->clusters[super_one->int_cluster] == NULL){   // if there are no "super points"
	                                                                   // in the new cluster list
	super_one->pClusterPrev = NULL;
	super_one->pClusterNext = NULL;
	data_to_analyze->clusters[super_one->int_cluster] = super_one; // make the reassigned "super point"
	                                                               // the head of the cluster list
	}
	else{                                                            // if there are "super points"
	                                                               // currently in the new cluster list
	  new_super = data_to_analyze->clusters[super_one->int_cluster]; // mark the current head
	  new_super->pClusterPrev = super_one;                           // and connect the reassigned "super
	  super_one->pClusterNext = new_super;                           // point" to the current head
	  super_one->pClusterPrev = NULL;
	  data_to_analyze->clusters[super_one->int_cluster] = super_one; // make the reassigned point the
	  // new head of the cluster list
	}

	// Update the constraints array
	UpdateConstraints(super_one, data_to_analyze);
	
	// if the cluster assignment has changed
	if(int_cur_cluster_two != super_two->int_cluster){
	  // update the number of changes in this iteration
	  int_num_changes = int_num_changes + super_two->int_weight;   

	  cur_node = super_two->head;
	  while(cur_node != NULL){
	    cur_node->int_cluster = super_two->int_cluster;
	    cur_node = cur_node->pSetNext;
	  }
	}
	
	// remove "super point" from cluster list
	prev_super = super_two->pClusterPrev;
	next_super = super_two->pClusterNext;
	
	if((prev_super == NULL) && (next_super != NULL)){
	  super_two->pClusterNext = NULL;
	  data_to_analyze->clusters[int_cur_cluster_two] = next_super;
	  next_super->pClusterPrev = NULL;
	}
	else{
	  if((prev_super != NULL) && (next_super == NULL)){
	    super_two->pClusterPrev = NULL;
	    prev_super->pClusterNext = NULL;
	  }
	  else{
	    if((prev_super != NULL) && (next_super != NULL)){
	      super_two->pClusterPrev = NULL;
	      super_two->pClusterNext = NULL;
	      prev_super->pClusterNext = next_super;
	      next_super->pClusterPrev = prev_super;
	    }
	  }
	}
 
	// update pointers to add point to new cluster list
	if(data_to_analyze->clusters[super_two->int_cluster] == NULL){   // if there are no "super points"
	  // in the new cluster list
	  super_two->pClusterPrev = NULL;
	  super_two->pClusterNext = NULL;
	  data_to_analyze->clusters[super_two->int_cluster] = super_two; // make the reassigned "super point"
	  // the head of the cluster list
	}
	else{                                                            // if there are "super points"
	  // currently in the new cluster list
	  new_super = data_to_analyze->clusters[super_two->int_cluster]; // mark the current head
	  new_super->pClusterPrev = super_two;                           // and connect the reassigned "super
	  super_two->pClusterNext = new_super;                           // point" to the current head
	  super_two->pClusterPrev = NULL;
	  data_to_analyze->clusters[super_two->int_cluster] = super_two; // make the reassigned point the
	  // new head of the cluster list
	}
	
  

	// Update the constraints array
	UpdateConstraints(super_two, data_to_analyze);


	
      }
      

      for(int_loop_count = 0; int_loop_count < data_to_analyze->int_num_CL_constraints; int_loop_count++){

	super_one = data_to_analyze->CL_constraint_ptrs[int_loop_count][0];
	super_two = data_to_analyze->CL_constraint_ptrs[int_loop_count][1];
	
	//super_one = data_to_analyze->CL_constraint_ptrs_one[int_loop_count];
	//super_two = data_to_analyze->CL_constraint_ptrs_two[int_loop_count];

	int_cur_cluster_one = super_one->int_cluster;
	int_cur_cluster_two = super_two->int_cluster;



	flt_minimum_CVQE = CalculateCVQE(data_to_analyze, super_one, 0, super_two, 0);
	super_one->int_cluster = 0;
	super_two->int_cluster = 0;
	for(int_outer_loop = 0; int_outer_loop < data_to_analyze->int_num_clusters; int_outer_loop++){
	  for(int_inner_loop = 0; int_inner_loop < data_to_analyze->int_num_clusters; int_inner_loop++){
	    flt_current_CVQE = CalculateCVQE(data_to_analyze, super_one, int_outer_loop, super_two, int_inner_loop);
	    if(flt_current_CVQE < flt_minimum_CVQE){
	      flt_minimum_CVQE = flt_current_CVQE;
	      super_one->int_cluster = int_outer_loop;
	      super_two->int_cluster = int_inner_loop;
	    }
	  }
	}

	// if the cluster assignment has changed
	if(int_cur_cluster_one != super_one->int_cluster){
	  // update the number of changes in this iteration
	  int_num_changes = int_num_changes + super_one->int_weight;

	  cur_node = super_one->head;
	  while(cur_node != NULL){
	    cur_node->int_cluster = super_one->int_cluster;
	    cur_node = cur_node->pSetNext;
	  }
	}

	// remove "super point" from cluster list
	prev_super = super_one->pClusterPrev;
	next_super = super_one->pClusterNext;
	
	if((prev_super == NULL) && (next_super != NULL)){
	  super_one->pClusterNext = NULL;
	  data_to_analyze->clusters[int_cur_cluster_one] = next_super;
	  next_super->pClusterPrev = NULL;
	}
	else{
	  if((prev_super != NULL) && (next_super == NULL)){
	    super_one->pClusterPrev = NULL;
	    prev_super->pClusterNext = NULL;
	  }
	  else{
	    if((prev_super != NULL) && (next_super != NULL)){
	      super_one->pClusterPrev = NULL;
	      super_one->pClusterNext = NULL;
	      prev_super->pClusterNext = next_super;
	      next_super->pClusterPrev = prev_super;
	    }
	  }
	}

	// update pointers to add point to new cluster list
	if(data_to_analyze->clusters[super_one->int_cluster] == NULL){   // if there are no "super points"
	                                                               // in the new cluster list
	super_one->pClusterPrev = NULL;
	super_one->pClusterNext = NULL;
	data_to_analyze->clusters[super_one->int_cluster] = super_one; // make the reassigned "super point"
	                                                               // the head of the cluster list
	}
	else{                                                            // if there are "super points"
	                                                               // currently in the new cluster list
	  new_super = data_to_analyze->clusters[super_one->int_cluster]; // mark the current head
	  new_super->pClusterPrev = super_one;                           // and connect the reassigned "super
	  super_one->pClusterNext = new_super;                           // point" to the current head
	  super_one->pClusterPrev = NULL;
	  data_to_analyze->clusters[super_one->int_cluster] = super_one; // make the reassigned point the
	  // new head of the cluster list
	}
	
	// Update the constraints array
	UpdateConstraints(super_one, data_to_analyze);

	// if the cluster assignment has changed
	if(int_cur_cluster_two != super_two->int_cluster){
	  // update the number of changes in this iteration
	  int_num_changes = int_num_changes + super_two->int_weight;  

	  cur_node = super_two->head;
	  while(cur_node != NULL){
	    cur_node->int_cluster = super_two->int_cluster;
	    cur_node = cur_node->pSetNext;
	  }
	}

	// remove "super point" from cluster list
	prev_super = super_two->pClusterPrev;
	next_super = super_two->pClusterNext;
       
	if((prev_super == NULL) && (next_super != NULL)){
	  super_two->pClusterNext = NULL;
	  data_to_analyze->clusters[int_cur_cluster_two] = next_super;
	  next_super->pClusterPrev = NULL;
	}
	else{
	  if((prev_super != NULL) && (next_super == NULL)){
	    super_two->pClusterPrev = NULL;
	    prev_super->pClusterNext = NULL;
	  }
	  else{
	    if((prev_super != NULL) && (next_super != NULL)){
	      super_two->pClusterPrev = NULL;
	      super_two->pClusterNext = NULL;
	      prev_super->pClusterNext = next_super;
	      next_super->pClusterPrev = prev_super;
	    }
	  }
	}

	// update pointers to add point to new cluster list
	if(data_to_analyze->clusters[super_two->int_cluster] == NULL){   // if there are no "super points"
	                                                               // in the new cluster list
	super_two->pClusterPrev = NULL;
	super_two->pClusterNext = NULL;
	data_to_analyze->clusters[super_two->int_cluster] = super_two; // make the reassigned "super point"
	                                                               // the head of the cluster list
	}
	else{                                                            // if there are "super points"
	                                                               // currently in the new cluster list
	  new_super = data_to_analyze->clusters[super_two->int_cluster]; // mark the current head
	  new_super->pClusterPrev = super_two;                           // and connect the reassigned "super
	  super_two->pClusterNext = new_super;                           // point" to the current head
	  super_two->pClusterPrev = NULL;
	  data_to_analyze->clusters[super_two->int_cluster] = super_two; // make the reassigned point the
	  // new head of the cluster list
	}
	
	// Update the constraints array
	UpdateConstraints(super_two, data_to_analyze);
      }
    }
    /***************************************************************************************************/

    cur_super = data_to_analyze->super_point_head; // start at head of linked list of "super points"
    while((cur_super != NULL) && (cur_super->int_constraint != 0)){
      cur_super = cur_super->pNext;
    }
    while(cur_super != NULL){                      // walk entire list of "super points"
    
      for(int_loop_count = 0; int_loop_count < data_to_analyze->int_num_clusters; int_loop_count++){
	index_array[int_loop_count] = int_loop_count;
      }

      //prev_super = cur_super->pClusterPrev;   // mark the "super points" on either side of 
      //next_super = cur_super->pClusterNext;   // the current "super point" in its cluster list
    
      int_cur_cluster = cur_super->int_cluster;   // remember the current cluster   
      cur_super->int_cluster = 0;      // and assign current "super point" to cluster zero
    
      // calculate the distance between the "super point" and the centroid of cluster zero  
      //if(data_to_analyze->int_algorithm_version == 2){
      //shortest_distance = CVQEDistance(cur_super->int_super_point_num, 0, data_to_analyze);
      //}
      //else{
      //shortest_distance = Distance(data_to_analyze->super_point_centroids[cur_super->int_super_point_num], data_to_analyze->cluster_centroids[cur_super->int_cluster], data_to_analyze->int_dimensions);
	//}
    
      //if(data_to_analyze->int_algorithm_version == 1){
	// determine if assignment to cluster 0 results in a constraint violation
	//int_constraint_violated = ViolateConstraints(cur_super, 0, data_to_analyze);
	
	//if(!int_constraint_violated){    // if a constraint is not violated
	//  int_valid_cluster = 1;         // remember that there is at least one valid assignment
	//}
	//else{                            // if a constraint is violated
	//  int_valid_cluster = 0;         // remember that there are currently no valid assignments
	//}
      //}	
      

      
      //printf("%d %d %d %d ", cur_super->int_super_point_num, 0, int_constraint_violated, int_valid_cluster);
      //scanf("%c", &chr_garbage);



      // calculate the distances between the "super point"and the centroids of all other clusters
      for(int_loop_count = 0; int_loop_count < data_to_analyze->int_num_clusters; int_loop_count++){     
	//if(data_to_analyze->int_algorithm_version == 2){
	//distance = CVQEDistance(cur_super->int_super_point_num, int_loop_count, data_to_analyze);
	//}
	//else{
	distance_array[int_loop_count] = Distance(data_to_analyze->super_point_centroids[cur_super->int_super_point_num], data_to_analyze->cluster_centroids[int_loop_count], data_to_analyze->int_dimensions);
	  //} 
      
	  //if(data_to_analyze->int_algorithm_version == 1){
	  // determine if assignment to current cluster results in a constraint violation
	  //int_constraint_violated = ViolateConstraints(cur_super, int_loop_count, data_to_analyze);

	  //if(((distance < shortest_distance) || (int_valid_cluster == 0)) && (!int_constraint_violated)){
	  //shortest_distance = distance;             // update the shortest distance to cluster centroid
	  //cur_super->int_cluster = int_loop_count;  // retain the cluster number of the closest centroid
	  //int_valid_cluster = 1;                    // indicate that there is at least one valid assignment
	  //}
	  //}
	  //else{
	  //if(distance < shortest_distance){
	  //shortest_distance = distance;             // update the shortest distance to cluster centroid
	  //cur_super->int_cluster = int_loop_count;  // retain the cluster number of the closest centroid
	  //}

	  //}



	//printf("%d %d %d %d ", cur_super->int_super_point_num, int_loop_count, int_constraint_violated, int_valid_cluster);
	//scanf("%c", &chr_garbage);	
	
      }

      qsort(index_array, data_to_analyze->int_num_clusters, sizeof(int), ComparisonFunction);
     
      if(data_to_analyze->int_algorithm_version == 1){
	int_cluster_index = 0;

	
	while((int_cluster_index < data_to_analyze->int_num_clusters) && (ViolateConstraints(cur_super, index_array[int_cluster_index], data_to_analyze))){
	  int_cluster_index++;
	}
	
	/*
	while((int_cluster_index < data_to_analyze->int_num_clusters) && (AlternateViolateConstraints(cur_super, index_array[int_cluster_index], data_to_analyze))){
	  int_cluster_index++;
	}
	*/

	if(int_cluster_index == data_to_analyze->int_num_clusters){
	  return 0;
	}
	else{
	  cur_super->int_cluster = index_array[int_cluster_index];
	}
      }
      else{
	cur_super->int_cluster = index_array[0];
      }
    
      // if the cluster assignment has changed
      if(int_cur_cluster != cur_super->int_cluster){
	
	// update the number of changes in this iteration
	int_num_changes = int_num_changes + cur_super->int_weight;          
	
      }

      // update pointers to add point to new cluster list
      if(data_to_analyze->clusters[cur_super->int_cluster] == NULL){   // if there are no "super points"
	                                                               // in the new cluster list
	cur_super->pClusterPrev = NULL;
	cur_super->pClusterNext = NULL;
	data_to_analyze->clusters[cur_super->int_cluster] = cur_super; // make the reassigned "super point"
	                                                               // the head of the cluster list
      }
      else{                                                            // if there are "super points"
	                                                               // currently in the new cluster list
	new_super = data_to_analyze->clusters[cur_super->int_cluster]; // mark the current head
	new_super->pClusterPrev = cur_super;                           // and connect the reassigned "super
	cur_super->pClusterNext = new_super;                           // point" to the current head
	cur_super->pClusterPrev = NULL;
	data_to_analyze->clusters[cur_super->int_cluster] = cur_super; // make the reassigned point the
	                                                               // new head of the cluster list
      }
      
      // Update the constraints array
      UpdateConstraints(cur_super, data_to_analyze);
      
      
      cur_node = cur_super->head;          // start at head of linked list of data points
                                           // assigned to current "super point"
      while(cur_node != NULL){             // and walk the entire list
	cur_node->int_cluster = cur_super->int_cluster;    // update cluster assignment of data point
	cur_node = cur_node->pSetNext;                     // move to next data point
      }   
       
      cur_super = cur_super->pNext;
      while((cur_super != NULL) && (cur_super->int_constraint != 0)){
	cur_super = cur_super->pNext;
      }      
    }
    
    
    /*********************Recalculate Centroids*********************/

    // loop through all clusters
    for(int_outer_loop = 0; int_outer_loop < data_to_analyze->int_num_clusters; int_outer_loop++){
      
      // point to head "super point" of current cluster list
      cur_super = data_to_analyze->clusters[int_outer_loop];
      
      if(cur_super != NULL){   // if there is at least one "super point" in the current cluster
	
	// initialize the temp array to zero
	for(int_loop_count = 0; int_loop_count < data_to_analyze->int_dimensions; int_loop_count++){
	  temp[int_loop_count] = 0;
	}
	
	int_num_nodes = 0;    // set the number of "super points" in this cluster to zero
	
	while(cur_super != NULL){   // loop through list of all super points in current cluster
	  
	  // multiply coordinates of "super point" by its weight and add to the temp array
	  for(int_loop_count = 0; int_loop_count < data_to_analyze->int_dimensions; int_loop_count++){
	    temp[int_loop_count] = temp[int_loop_count] + cur_super->int_weight * data_to_analyze->super_point_centroids[cur_super->int_super_point_num][int_loop_count];
	  }
	  
	  // update the number of data points in the current cluster by adding the weight
	  int_num_nodes = int_num_nodes + cur_super->int_weight;
  
	  cur_super = cur_super->pClusterNext;    // move to the next "super point" in the cluster list
	}
      }

      // if doing CVQE-K-Means, take into account coordinates of centroids involved in violated constraints
      if(data_to_analyze->int_algorithm_version == 2){
	
	// loop through first column of the array of ML constraints
	for(int_inner_loop = 0; int_inner_loop < data_to_analyze->int_num_ML_constraints; int_inner_loop++){
	  // if g(i) for the current constraint is equal to the current cluster
	  if(data_to_analyze->ML_constraints_array[int_inner_loop][2] == int_outer_loop){
	    // get the value of g'(i) for the current constraint
	    int_other_centroid = data_to_analyze->ML_constraints_array[int_inner_loop][3];
	    // if g(i) and g'(i) are different
	    if(int_outer_loop != int_other_centroid){
	      // add the coordinates of centroid g(i) to the temp array
	      for(int_loop_count = 0; int_loop_count < data_to_analyze->int_dimensions; int_loop_count++){
		temp[int_loop_count] = temp[int_loop_count] + data_to_analyze->cluster_centroids[int_other_centroid][int_loop_count];
	      }
	      int_num_nodes++;   // and increment the number of points used in the centroid calculation
	    }
	  }
	}
	
	// loop through second column of the array of ML constraints
	for(int_inner_loop = 0; int_inner_loop < data_to_analyze->int_num_ML_constraints; int_inner_loop++){
	  // if g'(i) for the current constraint is equal to the current cluster
	  if(data_to_analyze->ML_constraints_array[int_inner_loop][3] == int_outer_loop){
	    // get the value of g(i) for the current constraint
	    int_other_centroid = data_to_analyze->ML_constraints_array[int_inner_loop][2];
	    // if g(i) and g'(i) are different
	    if(int_outer_loop != int_other_centroid){
	      // add the coordinates of centroid g(i) to the temp array
	      for(int_loop_count = 0; int_loop_count < data_to_analyze->int_dimensions; int_loop_count++){
		temp[int_loop_count] = temp[int_loop_count] + data_to_analyze->cluster_centroids[int_other_centroid][int_loop_count];
	      }
	      int_num_nodes++;   // and increment the number of points used in the centroid calculation
	    }
	  }
	}

	// loop through first column of the array of CL constraints
	for(int_inner_loop = 0; int_inner_loop < data_to_analyze->int_num_CL_constraints; int_inner_loop++){
	  // if g(i) for the current constraint is equal to the current cluster
	  if(data_to_analyze->CL_constraints_array[int_inner_loop][2] == int_outer_loop){
	    // get the value of g'(i) for the current constraint
	    int_other_centroid = data_to_analyze->CL_constraints_array[int_inner_loop][3];
	    // if g(i) and g'(i) are the same
	    if(int_outer_loop == int_other_centroid){
	      // add the coordinates of the centroid closest to centroid g'(i) to the temp array
	      for(int_loop_count = 0; int_loop_count < data_to_analyze->int_dimensions; int_loop_count++){
		temp[int_loop_count] = temp[int_loop_count] + data_to_analyze->cluster_centroids[data_to_analyze->closest_clusters[int_other_centroid]][int_loop_count];
	      }
	      int_num_nodes++;   // and increment the number of points used in the centroid calculation
	    }
	  }
	}

	// loop through second column of the array of CL constraints
	for(int_inner_loop = 0; int_inner_loop < data_to_analyze->int_num_CL_constraints; int_inner_loop++){
	  // if g'(i) for the current constraint is equal to the current cluster
	  if(data_to_analyze->CL_constraints_array[int_inner_loop][3] == int_outer_loop){
	    // get the value of g(i) for the current constraint
	    int_other_centroid = data_to_analyze->CL_constraints_array[int_inner_loop][2];
	    // if g(i) and g'(i) are the same
	    if(int_outer_loop == int_other_centroid){
	      // add the coordinates of the centroid closest to centroid g'(i) to the temp array
	      for(int_loop_count = 0; int_loop_count < data_to_analyze->int_dimensions; int_loop_count++){
		temp[int_loop_count] = temp[int_loop_count] + data_to_analyze->cluster_centroids[data_to_analyze->closest_clusters[int_other_centroid]][int_loop_count];
	      }
	      int_num_nodes++;   // and increment the number of points used in the centroid calculation
	    }
	  }
	}
      }

      // divide the temp array by the number of points in the current cluster to get the centroid
      for(int_loop_count = 0; int_loop_count < data_to_analyze->int_dimensions; int_loop_count++){
	if(int_num_nodes != 0){
	  temp[int_loop_count] = temp[int_loop_count] / int_num_nodes;
	}
      }
      
      // update the current cluster centroid coordinates with the temp array
      for(int_inner_loop = 0; int_inner_loop < data_to_analyze->int_dimensions; int_inner_loop++){
	if(int_num_nodes != 0){
	  data_to_analyze->cluster_centroids[int_outer_loop][int_inner_loop] = temp[int_inner_loop];
	}
      }
    }

    if(data_to_analyze->int_algorithm_version == 2){
      UpdateClosestClusters(data_to_analyze);
    }

    /*************************Calculate VQE or CVQE*******************************/

    if(data_to_analyze->int_algorithm_version == 2){
      data_to_analyze->CVQE = CalculateCVQE(data_to_analyze, NULL, -1, NULL, -1);
      printf("%d\t%d\t%f\n", int_iteration, int_num_changes, data_to_analyze->CVQE);

      if(isnan(data_to_analyze->CVQE)){
	printf("Not a number.  Hit Enter to continue.");
	scanf("%c", &garbage);
      }
    }
    else{
      CalculateVQE(data_to_analyze);
      printf("%d\t%d\t%f\n", int_iteration, int_num_changes, data_to_analyze->VQE);

      if(isnan(data_to_analyze->VQE)){
	printf("Not a number.  Hit Enter to continue.");
	scanf("%c", &garbage);
      }
    }

  }

  CheckConstraintsSatisfied(data_to_analyze);

  return 1;           
}
