/* This file is part of the RC compiler.
   Copyright (C) 2000-2001 The Regents of the University of California.

RC is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

RC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with RC; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include "parser.h"
#include "rc.h"
#include "fix.h"
#include "edit.h"
#include "cfg.h"
#include "live.h"
#include "dd_list.h"
#include "unparse.h"
#include "c-parse.h"
#include "c-lex.h"
#include "constants.h"
#include "vars.h"
#include "semantics.h"
#include "callcc1.h"
#include "flags.h"
#include "builtins.h"
#include "optrc.h"
#include "chkopt.h"
#include "AST_utils.h"
// hchen
#include <inttypes.h>
#include <unistd.h>
#include "mops_constants.h"

#undef RCDEBUG
// static FILE *debug_output;

data_declaration internal_rctypeof_decl;

tag_declaration region_decl;
type region_type, ptrint_type, rc_adjust_fn_type;
asttype ptrint_type_ast;

bool flag_perturb;
long perturb_seed;

static bool rc_init(void)
{
  typelist rc_adjust_fn_args;
  tag_ref region_tag;
  declarator ptd;
  type_element ptm;
  region r;

  /* XXX: Should do some consistency checks on this symbol */
  internal_rctypeof_decl = lookup_global_id("internal_rctypeof");
  if (!internal_rctypeof_decl)
    return FALSE;

  r = newregion();

  ptrint_type = unsigned_long_type;

  rc_adjust_fn_args = new_typelist(r);
  typelist_append(rc_adjust_fn_args, ptr_void_type);
  typelist_append(rc_adjust_fn_args, int_type);
  rc_adjust_fn_type = make_function_type(size_t_type, rc_adjust_fn_args,
					 FALSE, FALSE);

  region_tag = new_struct_ref(r, dummy_location,
			      new_word(r, dummy_location,
				       str2cstring(r, "region")),
			      NULL, NULL, FALSE);
  region_decl = lookup_global_tag(region_tag);
  if (!region_decl)
    region_decl = declare_global_tag(region_tag);
  pending_xref_error(); /* In case someone declared an enum or union region */

  if (rc_pairs_array)
    region_type = unsigned_short_type; /* must have as much range as __rcid in regions.h */
  else
    region_type = make_pointer_type(make_tagged_type(region_decl));

  type2ast(r, dummy_location, ptrint_type, NULL, &ptd, &ptm);
  ptrint_type_ast = new_asttype(r, dummy_location, ptd, ptm);

  return TRUE;
}

static bool rc_init_once(void)
{
  static bool done = FALSE;

  if (!done && rc_init())
    done = TRUE;

  return done;
}

data_declaration declare_region_temporary(region r, compound_stmt block)
{
  return declare_temporary(r, block, region_type);
}

/*
static void print_vars(bitset vars, data_declaration *vmap)
{
  int vid;

  fprintf(debug_output, "VARS:");
  bitset_scan (vid, vars)
    {
      data_declaration vd = vmap[vid];

      fprintf(debug_output, " %s(%p)", vd->name, vd);
    }
  fprintf(debug_output, "\n");
}
*/

void add_rcop(region r, edge e, data_declaration v, bool increment)
{
  dd_list *vlist = increment ? &e->incvars : &e->decvars;

  if (!*vlist)
    *vlist = dd_new_list(r);
  assert(!dd_find(*vlist, v));
  dd_add_last(r, *vlist, v);
}

void insert_assignment_rcops(region r, function_decl fd, bitset vars)
{
  node parentptr *gnodes = fd->postorder_nodes;
  int nnodes = fd->postorder_size, i, vid;
  data_declaration *vmap = fd->vmap;

  /* Simple algorithm 1: On all nodes n:
     if n is an assignment to v:
       increment RC(v) on all edges from n to nodes where v is live on the way in
     for all v live on the way IN to n:
       decrement RC(v) on all edges from n to nodes where v is dead on the way in
  */
     
  /* Increment RC for all live parameters */
  assert(gnodes[nnodes - 1] == fd->cfg_entry);
  bitset_scan (vid, gnodes[nnodes - 1]->live_in)
    if (vmap[vid]->isparameter)
      add_rcop(r, gnodes[nnodes - 1]->edges_out, vmap[vid], TRUE);

  for (i = 0; i < nnodes; i++)
    {
      node n = gnodes[i];
      edge out;

      if (is_assign(n))
	{
	  assign a = CAST(assign, n);

	  if (is_identifier(a->arg1))
	    {
	      data_declaration id = CAST(identifier, a->arg1)->ddecl;

	      /* The test for membership in vars is not necessary, but
		 avoids useless work (only members of vars are marked
		 live) */
	      if (id->id != -1 && BIT_SETP(vars, id->id) &&
		  !zero_expression(a->arg2))
		for (out = n->edges_out; out; out = out->next_out)
		  if (BIT_SETP(out->to->live_in, id->id))
		    add_rcop(r, out, id, TRUE);
	    }
	}

      /* We need to decrement the RC of v in the following conditions:
	 - For all edges e, if v is live on the way out of source(e) and dead
	 on the way in to dest(e) (i.e., source(e) is a control-flow split
	 and the v is dead on the dest(e) branch). Decrement RC(v) on execution
	 of edge e.
	 - For all nodes n, if v is live on entrance to n and dead on
	 exit from n (i.e., n is the last use of v). Decrement RC(v) before
	 executing n.

	 As we currently only support addition of code on edges, we handle
	 the node case by adding the decrement RC(v) operation to all 
	 edges incoming to n. This is detected in the code below by the
	 test of liveonout (i.e. live on the way in to n and dead on the
	 way out of n is equivalent to live on the way out of a predecessor
	 of n and dead on the way out of n) */
      for (out = n->edges_out; out; out = out->next_out)
	{
	  bitset liveonin = out->to->live_in;
	  bitset liveonout = out->to->live_out;

	  bitset_scan (vid, n->live_out)
	    if (!BIT_SETP(liveonout, vid) || !BIT_SETP(liveonin, vid))
	      add_rcop(r, out, vmap[vid], FALSE);
	}
    }
}

void insert_functions_rcops(region r, function_decl fd, bitset vars)
{
  node parentptr *gnodes = fd->postorder_nodes;
  int nnodes = fd->postorder_size, i, vid;
  data_declaration *vmap = fd->vmap;

  /* Simple algorithm 2: On all nodes n:
     if n is a function call that may delete a region, for all variables v
     in vars live at exit from n:
       increment RC(v) on all edges into n
       decrement RC(v) on all edges out of n
  */
     
  for (i = 0; i < nnodes; i++)
    {
      node n = gnodes[i];
      edge out, in;

      /* Only calls to 'deletes' function where local is live before and after are black */
	if (is_function_call(n) &&
	    type_deletes(function_call_type(CAST(function_call, n))))
	  bitset_scan (vid, n->live_out)
	    {
	      data_declaration v = vmap[vid];

	      for (in = n->edges_in; in; in = in->next_in)
		add_rcop(r, in, v, TRUE);
	      for (out = n->edges_out; out; out = out->next_out)
		add_rcop(r, out, v, FALSE);
	    }
    }
}

static void add_rc_fn(function_decl fd, declaration rc_fn)
{
  rc_fn->next = CAST(node, fd->rc_fns);
  fd->rc_fns = rc_fn;
}

data_declaration lookup_rc_fn(region r, location loc, function_decl fd, type t,
			      bool update, bool do_pointer)
{
  data_declaration adjust_fn = NULL;
  char *adjustname = NULL;
  const char *tagname = NULL;

  if (type_pointer(t))
    {
      if (do_pointer)
	{
	  assert(!update);
	  adjustname = "__rc_adjustptr";
	}
    }
  else
    {
      tag_declaration tag = type_tag(t);

      name_tag(tag);
      tagname = tag->name;
      adjustname = rstralloc(r, strlen(tagname) + 11);
      sprintf(adjustname, "rc_%s_%s", update ? "update" : "adjust", tagname);
    }

  if (adjustname)
    {
      adjust_fn = lookup_global_id(adjustname);

      if (!adjust_fn)
	{
	  declaration rcfn;
	  data_declaration *rcfn_decl;
	  struct data_declaration tempdecl;
	  type rcfn_type;

	  /* We still create a "bad" function when this error occurs to avoid
	     duplicate messages */
	  if (type_contains_union_with_pointers(t))
	    error_with_location(loc, "must provide %s function for struct or union %s", adjustname, tagname);

	  if (update)
	    {
	      update_function_decl updfn = new_update_function_decl(r, loc);
	      type ptr_to_t = make_pointer_type(t);
	      typelist rc_update_fn_args;

	      rcfn = CAST(declaration, updfn);
	      rcfn_decl = &updfn->rcfn_decl;

	      rc_update_fn_args = new_typelist(r);
	      typelist_append(rc_update_fn_args, ptr_to_t);
	      typelist_append(rc_update_fn_args, ptr_to_t);
	      rcfn_type = make_function_type(void_type, rc_update_fn_args,
					     FALSE, FALSE);
	    }
	  else
	    {
	      adjust_function_decl adjfn = new_adjust_function_decl(r, loc);

	      rcfn = CAST(declaration, adjfn);
	      rcfn_decl = &adjfn->rcfn_decl;
	      rcfn_type = rc_adjust_fn_type;
	    }

	  init_data_declaration(&tempdecl, rcfn, adjustname, rcfn_type);
	  tempdecl.kind = decl_function;
	  tempdecl.needsmemory = TRUE;
	  tempdecl.isused = TRUE;
	  tempdecl.ftype = function_static;
	  tempdecl.isinline = TRUE;

	  *rcfn_decl = adjust_fn = declare(global_env, &tempdecl, FALSE);
	  rcfn->type = t;

	  add_rc_fn(fd, rcfn);
	}
      else if (adjust_fn->kind != decl_function)
	{
	  error_with_location(loc, "%s is not an rc_adjust function for struct or union %s not found", adjustname, tagname);
	  adjust_fn = NULL;
	}
    }
  return adjust_fn;
}

static expression *build_rcop(region r, location loc, function_decl fd,
			      expression idv, bool increment, expression *lastrcop)
{
  adjust_rc rcop = new_adjust_rc(r, loc, idv, increment);

  rcop->type = void_type;
  if (!type_pointer(idv->type))
    {
#ifndef ASSUME_GCC
      if (type_array(idv->type))
	{
	XXX:
	  ensure presence of __rc_adjustarray, or add an adjust_array_function_decl
	  node to fd
	}
#endif
      rcop->rcfn_decl =
	lookup_rc_fn(r, loc, fd, type_array_of_base(idv->type), FALSE, TRUE);
    }

  set_parent(CASTSRPTR(node, &rcop->arg1), CAST(node, rcop));
  *lastrcop = CAST(expression, rcop);

  return (expression *)&rcop->next;
}

void generate_rcops(region r, function_decl fd)
{
  node parentptr *gnodes = fd->postorder_nodes;
  int nnodes = fd->postorder_size, i;
     
  for (i = 0; i < nnodes; i++)
    {
      node n = gnodes[i];
      edge out;

      for (out = n->edges_out; out; out = out->next_out)
	{
	  dd_list_pos v;
	  expression rcops = NULL, *lastrcop = &rcops;
	  location loc = out->from->location;

	  /* Collect all rcops on this edge */
	  if (out->incvars)
	    dd_scan (v, out->incvars)
	      {
		dd_list_pos decpos;
		data_declaration vv = DD_GET(data_declaration, v);

		/* Don't generate ++/-- on same var on same edge */
		if (out->decvars && (decpos = dd_find(out->decvars, vv)))
		  dd_remove(decpos);
		else
		  lastrcop = build_rcop(r, loc, fd, build_identifier(r, loc, vv),
					TRUE, lastrcop);
	      }
	  if (out->decvars)
	    dd_scan (v, out->decvars)
	      lastrcop = build_rcop(r, loc, fd, build_identifier(r, loc, DD_GET(data_declaration, v)),
				    FALSE, lastrcop);

	  if (rcops)
	    {
	      /* If more than 1, wrap in comma expression */
	      if (rcops->next)
		{
		  rcops = CAST(expression, new_comma(r, n->location, rcops));
		  rcops->type = void_type;
		}
	      add_expression_to_edge(r, out, rcops);
	    }
	}
    }
}

static void adjust_localrc(region r, edge e, data_declaration local, bool increment)
{
  /* Not cross_pointers because the variables concerned are on the stack, so
     pointers are all cross-region anyway */
  if (type_contains_pointers(local->type))
    add_rcop(r, e, local, increment);
}

void add_local_rc(region r, function_decl fd, bitset done)
{
  node parentptr *gnodes = fd->postorder_nodes;
  int nnodes = fd->postorder_size, i, vid;
  data_declaration *vmap = fd->vmap;
  edge entry_edge;
  bool qdeletes = type_deletes(fd->ddecl->type);

  /* Add non-assignment relayed RC adjustments for all local variables not 
     in done. Rules:
     - increment RC for all parameters on entry
     - decrement RC for all locals on edges from in-scope to out-of-scope
  */

  /* Increment RC for all parameters not in done */
  assert(gnodes[nnodes - 1] == fd->cfg_entry);
  entry_edge = gnodes[nnodes - 1]->edges_out;
  bitset_scan (vid, fd->cfg_entry->inscope_vars)
    if (!BIT_SETP(done, vid) && (qdeletes || vmap[vid]->addressed))
      {
	data_declaration parm = vmap[vid];

	assert(parm->isparameter);
	adjust_localrc(r, entry_edge, parm, TRUE);
      }

  for (i = 0; i < nnodes; i++)
    {
      node n = gnodes[i];
      edge out;

      for (out = n->edges_out; out; out = out->next_out)
	{
	  bitset stayinscope = out->to->inscope_vars;

	  bitset_scan (vid, n->inscope_vars)
	    {
	      data_declaration v = vmap[vid];

	      if (!BIT_SETP(done, vid) && !BIT_SETP(stayinscope, vid) &&
		  (qdeletes || v->addressed || type_array(v->type)))
		adjust_localrc(r, out, v, FALSE);
	    }
	}
    }
}

static bool array_access(expression e)
{
  expression nfe;

  /* Find the base pointer in a mess of plus nodes */
  while (is_plus(e))
    {
      plus add = CAST(plus, e);

      if (type_integer(add->arg1->type))
	e = add->arg2;
      else
	e = add->arg1;
    }
  /* We answer yes if we see an array that is accessed (possibly via a bunch
     of fields) from some variable */
  return type_array(e->type) && (nfe = ignore_fields(e), is_identifier(nfe));
}

/* Add RC ops at assignment statements except for local variables whose
   address is not taken.
   Note: destinations of asm statements are considered as having
     their address taken. It is up to the assembly code to make sure the
     RC is correct
*/
void add_other_rc(region r, function_decl fd)
{
  node parentptr *gnodes = fd->postorder_nodes;
  int nnodes = fd->postorder_size, i;
  bool qdeletes = type_deletes(fd->ddecl->type);

  /* Add RC adjustments for all assignments to globals or locals
     and via pointers
  */

  /* The AST has been edited, but the assign nodes are still there */
  for (i = 0; i < nnodes; i++)
    {
      node n = gnodes[i];
      
      if (is_assign(n) && type_contains_pointers(n->type))
	{
	  assign ae = CAST(assign, n);
	  expression stabiliser, update, base_dest, newarg1;
	  location loc = ae->location;
	  data_declaration temp;
	  node sameregion *pp = ae->parent_ptr;
	  node parent = ae->parent, next = ae->next;
	  bool aeused;

	  /* Skip optimised locals and writes to non-addressed local structs in
	     non-deletes functions */
	  base_dest = ignore_fields(ae->arg1);
	  if (is_identifier(base_dest))
	    {
	      identifier id = CAST(identifier, base_dest);

	      if (id->ddecl->islocal &&
		  (id->ddecl->interesting ||
		   (!qdeletes && !id->ddecl->addressed)))
		continue;
	    }
	  /* Skip the special "clear to NULL" initialisations (there is
	     no old value) */
	  if (ae->clearing)
	    continue;

	  aeused = expression_used(CAST(expression, ae));

	  ae->next = NULL;

	  temp = stabilise_lvalue(r, parent_block(n), ae->arg1, &newarg1,
				  &stabiliser);
	  ae->arg1 = newarg1;
	  if (!stabiliser || array_access(CAST(assign, stabiliser)->arg2))
	    {
	      /* This must have been an assignment (possiblty to a field)
		 of a global or local. ae->arg2 may be a complex expression
		 (with side effects). It can also be an array access (of
		 a local or gloabl array) */
	      global_rc rcop;

	      rcop = new_global_rc(r, loc, ae->arg1, ae->arg2);
	      rcop->sr_vertex = ae->sr_vertex;
	      rcop->rcfn_decl = lookup_rc_fn(r, loc, fd, ae->arg1->type, FALSE, FALSE);
	      ae->rcop = CAST(generic_rc, rcop);

	      if (stabiliser)
		update = build_comma(r, loc, stabiliser, CAST(expression, rcop));
	      else
		update = CAST(expression, rcop);
	    }
	  else
	    {
	      /* Assignment via a pointer. */
	      update_rc rcop = new_update_rc(r, loc, ae->arg1, ae->arg2, temp);

	      rcop->sr_vertex = ae->sr_vertex;
	      rcop->rcfn_decl = lookup_rc_fn(r, loc, fd, ae->arg1->type, !USE_RC_ADJUST_FOR_UPDATE, FALSE);
	      ae->rcop = CAST(generic_rc, rcop);

	      set_parent(CASTSRPTR(node, &rcop->arg1), CAST(node, rcop));
	      set_parent(CASTSRPTR(node, &rcop->arg2), CAST(node, rcop));
	      update = build_comma(r, loc, stabiliser, CAST(expression, rcop));
	    }
	  /* Only reproduce arg1 if result used (avoid warnings) */
	  if (aeused)
	    update = build_comma(r, loc, update,
				 copy_lvalue(r, loc, ae->arg1));
	  *pp = CAST(node, update);
	  set_parent(pp, parent);
	  AST_SET_NEXT(update, next);
	}
    }
}

/* Note: global_rc_temps_needed and update_rc_temps_needed depend on
   prt_global_rc and prt_update_rc in unparse.c, and also
   on stats_global_rc* in stats.c */
/*
static int global_rc_temps_needed(global_rc rcop)
{
  expression dest = rcop->arg1, src = rcop->arg2;
  binary brcop = CAST(binary, rcop);

  if (type_aggregate(dest->type) ||
      type_traditional(dest->type) ||
      type_sameregion(dest->type) ||
      type_parentptr(dest->type) ||
      known_traditional_write(brcop) || zero_expression(src))
    return 0;
  else if (!rc_stats_costs && rc_stats_detailed)
    return 3;
  else
    return 2;
}

static int update_rc_temps_needed(update_rc rcop)
{
  expression of = rcop->arg1;
  binary brcop = CAST(binary, rcop);

  if (type_aggregate(of->type) ||
      type_traditional(of->type) ||
      type_sameregion(of->type))
    return 0;
  else if (type_parentptr(of->type))
    {
#ifdef USE_CHECK_CHILD
      return 0;
#else
      if (rc_keepchecks ||
	  !(rc_nochecks ||
	    known_parentptr_write(rcop) || zero_expression(rcop->arg2)))
	return 2;
      else
	return 0;
#endif
    }
  else if (known_sameregion_write(rcop) || known_traditional_write(brcop) ||
	   zero_expression(rcop->arg2))
      return 2;
  else
      return 3;
}

static void declare_rcop_temps(region r, function_decl fd)
{
  generic_rc sameregion *rcops = fd->allrcops;
  generic_rc rcop;

  while ((rcop = *rcops++))
    {
      int ntemps, j;
      compound_stmt b;

      if (is_update_rc(rcop))
	ntemps = update_rc_temps_needed(CAST(update_rc, rcop));
      else
	ntemps = global_rc_temps_needed(CAST(global_rc, rcop));
      assert(ntemps <= MAXRCTEMPS);

      b = parent_block(CAST(node, rcop));
      for (j = 0; j < ntemps; j++)
	rcop->temps[j] = declare_region_temporary(r, b);
    }
}

static size_t copy_rcops(function_decl fd, generic_rc sameregion *rcops)
{
  node parentptr *gnodes = fd->postorder_nodes;
  int nnodes = fd->postorder_size, i, count = 0;

  // The assign nodes we are interested in have been replaced by update_rc
  // or global_rc, but the nodes list still contains the original assign
  // nodes. These assign nodes point to the update_rc/global_rc node 
  for (i = 0; i < nnodes; i++)
    {
      node n = gnodes[i];
      assign ae;
      generic_rc rcop;

      if (!is_assign(n))
	continue;

      ae = CAST(assign, n);

      if (!(rcop = ae->rcop))
	continue;

      if (rcops)
	rcops[count] = rcop;
      count++;
    }

  return count;
}

static void save_rcops(region r, function_decl fd)
{
  size_t count = copy_rcops(fd, NULL);
  generic_rc sameregion *rcops = rarrayalloc(r, count + 1, generic_rc sameregion);

  copy_rcops(fd, rcops);
  rcops[count] = NULL;
  fd->allrcops = rcops;
}
*/

expression rc_check_function_call(function_call fce)
{
  if (!rc_init_once())
    return CAST(expression, fce); /* not seen regions.h yet */

  if (is_identifier(fce->arg1))
    {
      identifier called = CAST(identifier, fce->arg1);

      if (called->ddecl == internal_rctypeof_decl)
	{
	  /* replace call internal_rctypeof(sizeof(foo)) by the appropriate
	     cleanup function for foo */
	  type allocated_type;
	  location loc = fce->location;


	  /* foo can be:
	     - array or function or variable-size: error
	     - contains no pointers: just return 0
	     - otherwise return appropriate rc_adjust fn */

	  if (!fce->args || fce->args->next
	      || !is_sizeof_type(fce->args))
	    error_with_location(loc, "don't call internal_rctypeof");
	  else
	    {
	      allocated_type = CAST(sizeof_type, fce->args)->asttype->type;

	      if (!type_size_cc(allocated_type))
		error_with_location(loc, "variable-sized types unsupported");
	      else if (type_function(allocated_type))
		error_with_location(loc, "cannot get type of a function");
	      else if (type_array(allocated_type))
		error_with_location(loc, "cannot get type of an array");
	      else if (type_contains_cross_pointers(allocated_type))
		{
		  data_declaration cleanup_fn =
		    lookup_rc_fn(parse_region, loc, current_function_decl,
				 allocated_type, FALSE, TRUE);
		  expression cleanup_id =
		    build_identifier(parse_region, loc, cleanup_fn);

		  return cleanup_id;
		}
	      else
		return build_zero(parse_region, loc);
	    }
	}
    }
  return CAST(expression, fce);
}

/*
static void clear_fd(function_decl fd)
{
  long i;

  fd->vmap = NULL;
  fd->addressed_labels = NULL;

  for (i = 0; i < fd->cfg_size; i++)
    {
      node n = fd->cfg_nodes[i];

      n->edges_in = n->edges_out = NULL;
      n->live_in = n->live_out = NULL;
      n->inscope_vars = NULL;

      if (is_statement(n))
	{
	  statement s = CAST(statement, n);

	  s->continue_dest = s->break_dest = NULL;
	}
    }
  fd->cfg_nodes = fd->postorder_nodes = fd->preorder_nodes = NULL;
  fd->cfg_entry = NULL;
}

void rc_process(declaration the_program) deletes
{
  declaration d;
  char *perturb = getenv("RCPERTURB");

  if (perturb)
    {
      flag_perturb = 1;
      perturb_seed = atol(perturb);

      srand48(perturb_seed);
    }

  builtins_init();
  if (!rc_init_once())
    {
      error("regions.h not included");
      exit(2);
    }

  if (rc_chkopt_perfile)
    chkopt_libfn_init();

    //   XXX: Parse tree needs to be fixed ahead of time :-(
    //See chkopt.c:has_interesting_pointers 
  scan_declaration (d, the_program)
    if (is_function_decl(d))
      {
	function_decl fd = CAST(function_decl, d);
	fix_parse_tree(parse_region, fd);
      }

  scan_declaration (d, the_program)
    if (is_function_decl(d))
      {
	function_decl fd = CAST(function_decl, d);
	data_declaration *vmap;
	region work = newsubregion(parse_region); //regionof(fd);
	int i;
	bitset interesting;

	build_cfg(work, fd);
	cfg_build_postorder(work, fd);

	AST_set_parents(CAST(node, fd));

	collect_variables(work, fd);

	// We optimise local pointers whose address is not taken 
	interesting = bclearb(new_bitset(work, fd->nlocals));
	vmap = fd->vmap;
	for (i = 0; i < fd->nlocals; i++)
	  {
	    data_declaration v = vmap[i];

	    if (v->kind == decl_variable && v->islocal && !v->addressed &&
		type_pointer(v->type))
	      {
		SET_BITB(interesting, i);
		v->interesting = TRUE;
	      }
	  }
	  
	compute_liveness(work, fd, interesting);

	if (rc_chkopt || rc_chkopt_perfile)
	  optimise_rc_checks(parse_region, fd, interesting);

	if (rc_locals && type_deletes(fd->ddecl->type))
	  switch (rc_algorithm)
	    {
	    case rc_optimal:
	      insert_optimal_rcops(work, fd, interesting);
	      break;
	    case rc_assignment:
	      insert_assignment_rcops(work, fd, interesting);
	      break;
	    case rc_functions:
	      insert_functions_rcops(work, fd, interesting);
	      break;
	    }

	    // Add RC code for locals whose address was taken 
	    if (rc_globals) // Counting these as globals for rc purposes
	  add_local_rc(work, fd, interesting);

	generate_rcops(parse_region, fd);

	if (rc_globals)
	  add_other_rc(parse_region, fd);

	save_rcops(parse_region, fd);

	clear_fd(fd);
	deleteregion(work);
      }

  if (errorcount)
    return;

  if (rc_collect_statistics)
    {
      extern double largest_optrc;

      if (largest_optrc)
	fprintf(stderr, "optrc: %.2f\n", largest_optrc);
    }

  if (rc_chkopt_perfile)
    optimise_rc_checks_perfile(the_program);

  scan_declaration (d, the_program)
    if (is_function_decl(d))
      {
	function_decl fd = CAST(function_decl, d);
	declare_rcop_temps(parse_region, fd);
      }

  if (flag_parse_only)
    unparse(stdout, the_program);
  else
    exit(exec_cc1(the_program));
}
*/

// BEGIN hchen's stuff

char *deadFunctions[] =
  { "exit", "_exit", "abort", "execl",
    "execlp", "execle", "execv", "execvp"
    //,"longjmp", "siglongjmp"
  };

const int numDeadFunctions = sizeof(deadFunctions) / sizeof(char*);

// hashtable
typedef struct
{
  void *key;
  uint32_t value;
} HashtableEntry;

typedef struct 
{
  unsigned int capacity, size;
  HashtableEntry *data;
} HashtableBucket;

typedef struct
{
  HashtableBucket *buckets;
  unsigned int capacity, numKeys;
  float loadFactor;
} Hashtable;

const unsigned int INIT_TABLE_CAPACITY = 9, // 3071,
  INIT_BUCKET_CAPACITY = 4;
const float INIT_LOAD_FACTOR = 0.75;

static Hashtable hashtable;

static void hashtableInit(Hashtable *table, int initialCapacity,
			  float initialLoadFactor)
{
  unsigned int i;
  HashtableBucket *buckets;

  table->capacity = initialCapacity;
  table->numKeys = 0;
  table->loadFactor = initialLoadFactor;
  if ((table->buckets = (HashtableBucket*)
       malloc(sizeof(HashtableBucket) * table->capacity)) == NULL)
  {
    perror("Not enough memory");
    exit(1);
  }
  buckets = table->buckets;
  for (i = 0; i < table->capacity; i++)
  {
    buckets->capacity = buckets->size = 0;
    buckets->data = NULL;
    buckets++;
  }
}

static void hashtableFinalize(Hashtable *table)
{
  unsigned int i;

  for (i = 0; i < table->capacity; i++)
    if ((table->buckets)[i].data != NULL)
    {
      free((table->buckets)[i].data);
    }

  //fprintf(stderr, "Number of keys: %d\n", table->numKeys);
  free(table->buckets);
  table->buckets = NULL;
  table->capacity = table->numKeys = 0;
}

static bool hashtablePut(Hashtable *table, void *key, uint32_t value);

static void hashtableRehash(Hashtable *table, int expansionRatio)
{
  HashtableBucket *oldBuckets, *buckets;
  HashtableEntry *entry;
  unsigned int oldCapacity, i, j;

  oldBuckets = table->buckets;
  oldCapacity = table->capacity;
  hashtableInit(table, table->capacity * expansionRatio + 1, table->loadFactor);
  for (i = 0; i < oldCapacity; i++)
  {
    buckets = oldBuckets + i;
    for (j = 0; j < buckets->size; j++)
    {
      entry = buckets->data + j;
      if (!hashtablePut(table, entry->key, entry->value))
      {
	perror("hashtableRehash(): Collision\n");
	exit(1);
      }
    }
    if (buckets->data != NULL)
      free(buckets->data);
  }
  free(oldBuckets);
}

static __inline uintptr_t hashCode(void *object)
{
  return (uintptr_t)object;
}

static bool hashtableFind(Hashtable *table, void *key,
			  HashtableBucket **bucketParam,
			  HashtableEntry **entryParam)
{
  unsigned int i, index;
  HashtableBucket *bucket;
  HashtableEntry *entry = NULL;
  
  index = hashCode(key) % table->capacity;
  bucket = table->buckets + index;
  for (i = 0; i < bucket->size; i++)
    if (((bucket->data)[i]).key == key)
    {
      entry = bucket->data + i;
      break;
    }

  if (bucketParam != NULL)
    *bucketParam = bucket;
  if (entryParam != NULL)
    *entryParam = entry;
  
  return entry != NULL;
}

static bool hashtableContainsKey(Hashtable *table, void *key)
{
  return hashtableFind(table, key, NULL, NULL);
}

static uint32_t hashtableGet(Hashtable *table, void *key)
{
  HashtableEntry *entry;

  if (!hashtableFind(table, key, NULL, &entry))
    return -1;

  return entry->value;
}

static bool hashtablePut(Hashtable *table, void *key, uint32_t value)
{
  HashtableBucket *bucket;

  if (hashtableFind(table, key, &bucket, NULL))
    return FALSE;

  if (table->numKeys >= 0x80000000L)
  {
    perror("Too many entries in hashtable.");
    exit(1);
  }

  if (bucket->size >= bucket->capacity)
  {
    if (bucket->capacity == 0)
    {
      bucket->capacity = INIT_BUCKET_CAPACITY;
      bucket->size = 0;
      if ((bucket->data = (HashtableEntry*)
	   malloc(sizeof(HashtableEntry) * bucket->capacity)) == NULL)
      {
	perror("Not enough memory");
	exit(1);
      }
    }
    else
    {
      bucket->capacity *= 2;
      if ((bucket->data = (HashtableEntry*)realloc
	   (bucket->data, sizeof(HashtableEntry) * bucket->capacity)) == NULL)
      {
	perror("Not enough memory");
	exit(1);
      }
      //fprintf(stderr, "Hashtable entry %d 's size is increased to %d\n",
      //      bucket - table->buckets, bucket->capacity);
    }
  }
  ((bucket->data)[bucket->size]).key = key;
  ((bucket->data)[bucket->size]).value = value;
  bucket->size++;

  table->numKeys++;
  if (table->numKeys > (int)(table->capacity * table->loadFactor))
  {
    //fprintf(stderr, "Rehash capacity=%d loadfactor=%f\n", table->capacity,
    // table->loadFactor);
    hashtableRehash(table, 2);
  }
  
  return TRUE;
}

/*
  Returns a unique 32-bit integer for p.  isFirstTime indicates whether 
  a unique integer has been assigned to p before (i.e. whether p
  is already in the hashtable).
 */
uint32_t getUniqueAddress(void *p, bool *isFirstTime)
{
  static uint32_t count = 0;

  if (count >= 0x7fffffffL)
  {
    perror("Too many addresses");
    exit(1);
  }
  
  if (hashtableContainsKey(&hashtable, p))
  {
    if (isFirstTime != NULL)
      *isFirstTime = FALSE;
    return hashtableGet(&hashtable, p);
  }
  else
  {
    if (isFirstTime != NULL)
      *isFirstTime = TRUE;
    hashtablePut(&hashtable, p, ++count);
    return count;
  }
}

// vector
typedef struct
{
  void **data;
  int size, capacity;
} Vector;

Vector variableDefinition, variableDeclaration, functionDefinition;

const int INIT_VECTOR_CAPACITY = 2;

void vectorInit(Vector *vector, int initCapacity)
{
  vector->capacity = initCapacity;
  if ((vector->data = malloc(sizeof(void*) * vector->capacity)) == NULL)
  {
    perror("Not enough memory\n");
    exit(1);
  }
  vector->size = 0;
}

void vectorAdd(Vector *vector, void *data)
{
  if (vector->size >= vector->capacity)
  {
    vector->capacity *= 2;
    if ((vector->data =
	 realloc(vector->data, sizeof(void*) * vector->capacity)) == NULL)
    {
      perror("Not enough memory\n");
      exit(1);
    }
  }
  vector->data[vector->size++] = data;
}

void *vectorGet(Vector *vector, int i)
{
  if (i >= 0 && i < vector->size)
    return vector->data[i];
  else
  {
    return NULL;
  }
}

void vectorFinalize(Vector *vector)
{
  free(vector->data);
  vector->capacity = vector->size = 0;
}
	       
static FILE *outputFile;

static void ast_output(const unsigned char *buf, int length)
{
  if (fwrite(buf, length, 1, outputFile) < 1)
  {
    fprintf(stderr, "Cannot write to output file");
    exit(1);
  }
}

static void ast_output_token(unsigned char token)
{
  ast_output(&token, 1);
}

static void ast_output_byte(unsigned char value)
{
  unsigned char buffer[2];

  buffer[0] = MOPS_BYTE;
  buffer[1] = value;
  ast_output(buffer, 2);
}

static void ast_output_int(uint32_t value)
{
  unsigned char buffer[5];
  
  buffer[0] = MOPS_INT;
  // big endian -- compatible with Java DataInputStream!
  buffer[1] = (unsigned char) (value >> 24);
  buffer[2] = (unsigned char) (value >> 16);
  buffer[3] = (unsigned char) (value >> 8);
  buffer[4] = (unsigned char) (value);

  ast_output(buffer, 5);
}

static void ast_output_string_basic(const char *str, int length)
{
  unsigned char buffer[5];
  uint32_t ulength = length;

  buffer[0] = MOPS_STRING;
  // big endian -- compatible with Java DataInputStream!
  buffer[1] = (unsigned char) (ulength >> 24);
  buffer[2] = (unsigned char) (ulength >> 16);
  buffer[3] = (unsigned char) (ulength >> 8);
  buffer[4] = (unsigned char) (ulength);

  ast_output(buffer, 5);
  ast_output(str, length);
}

static void ast_output_string(const char *str)
{
  ast_output_string_basic(str, strlen(str));
}

static void ast_output_cstring(const cstring s)
{
  ast_output_string_basic(s.data, s.length);
}

// if node "d" is not in hashtable, return true;
// otherwise, print out "*d" and return false;
static bool ast_output_node_start(declaration d)
{
  bool isFirstTime;
  uint32_t address;

  address = getUniqueAddress(d, &isFirstTime);
  if (isFirstTime)
  {
    ast_output_token(MOPS_LP);
    ast_output_int(address);
    ast_output_byte(d->kind);
    return TRUE;
  }
  else
  {
    ast_output_token(MOPS_ADDRESS);
    ast_output_int(address);
    return FALSE;
  }
}

void process_toplevel_declarations(declaration dlist);
void process_toplevel_declaration(declaration d);
void process_asm_decl(asm_decl d);
void process_extension_decl(extension_decl d);
bool should_process_data_decl(data_decl d);
void process_data_decl(data_decl d);
void process_ellipsis_decl(ellipsis_decl d);
void process_function_decl(function_decl d, declaration entry, declaration exit);
void process_variable_decl(variable_decl d);
void process_declarator(declarator d, type_element elements, attribute attributes);
void process_simple_declarator(declarator d, int container_priority);
void process_type_elements(type_element elements);
void process_type_element(type_element em);
void process_typename(typename tname);
void process_typeof_expr(typeof_expr texpr);
void process_typeof_type(typeof_type ttype);
void process_attribute(attribute a);
void process_rid(rid r);
void process_qualifier(qualifier q);
void process_tag_ref(tag_ref sr);
void process_fields(declaration flist);
void process_enumerators(declaration elist);
void process_field_declaration(declaration d);
void process_field_extension_decl(extension_decl d);
void process_field_data_decl(data_decl d);
void process_field_decl(field_decl fd);
void process_enumerator(enumerator ed);
void process_parameters(declaration parms);
void process_parameter(declaration parm);
void process_asttype(asttype t);
void process_word(word w);

void process_expressions(expression elist, bool isfirst);
void process_expression(expression e, int context_priority);
void process_comma(comma e, int context_priority);
void process_sizeof_type(sizeof_type e, int context_priority);
void process_alignof_type(alignof_type e, int context_priority);
void process_label_address(label_address e, int context_priority);
void process_cast(cast e, int context_priority);
void process_cast_list(cast_list e, int context_priority);
void process_conditional(conditional e, int context_priority);
void process_identifier(identifier e, int context_priority);
void process_compound_expr(compound_expr e, int context_priority);
void process_function_call(function_call e, int context_priority);
void process_array_ref(array_ref e, int context_priority);
void process_field_ref(field_ref e, int context_priority);
void process_unary(unary e, int context_priority);
void process_binary(binary e, int context_priority);
void process_init_list(init_list e, int context_priority);
void process_init_index(init_index e, int context_priority);
void process_init_field(init_field e, int context_priority);
//void process_signed_cst(signed_cst e, int context_priority);
//void process_unsigned_cst(unsigned_cst e, int context_priority);
void process_lexical_cst(lexical_cst e, int context_priority);
void process_string(string e, int context_priority);
void process_adjust_rc(adjust_rc rcop, int context_priority);
void process_global_rc(global_rc rcop, int context_priority);
void process_update_rc(update_rc rcop, int context_priority);
void process_rc_fns(declaration rc_fns);
void process_parameter_declarations(declaration dlist);
void process_parameter_declaration(declaration d);

void process_statement(statement s);
void process_compound_stmt(compound_stmt s);
void process_compound_declarations(declaration dlist);
void process_compound_declaration(declaration d);
void process_asm_stmt(asm_stmt s);
void process_asm_stmt_plain(asm_stmt s);
void process_asm_operands(asm_operand olist);
void process_asm_operand(asm_operand o);
void process_if_stmt(if_stmt s);
void process_labeled_stmt(labeled_stmt s);
void process_expression_stmt(expression_stmt s);
void process_while_stmt(while_stmt s);
void process_dowhile_stmt(while_stmt s);
void process_switch_stmt(switch_stmt s);
void process_for_stmt(for_stmt s);
void process_break_stmt(break_stmt s);
void process_continue_stmt(continue_stmt s);
void process_return_stmt(return_stmt s);
void process_goto_stmt(goto_stmt s);
void process_computed_goto_stmt(computed_goto_stmt s);
void process_empty_stmt(empty_stmt s);

void process_label(label l);
void process_id_label(id_label l);
void process_case_label(case_label l);
void process_default_label(default_label l);

void process_regionof(expression e);
void process_function_entry(declaration d, declarator dc);
void process_function_exit(declaration d, declarator dc);

#define PRTCASE(type, x) case kind_ ## type: process_ ## type(CAST(type, (x))); break

void process_toplevel_declarations(declaration dlist)
{
  declaration d;

  //if (!ast_output_node_start((declaration)dlist)) return;
  scan_declaration (d, dlist)
    process_toplevel_declaration(d);
  //ast_output_token(MOPS_RP);
}

void process_toplevel_declaration(declaration d)
{
  //if (!ast_output_node_start((declaration)d)) return;
  // startline();
  switch (d->kind)
    {
      PRTCASE(asm_decl, d);
      PRTCASE(data_decl, d);
      //PRTCASE(function_decl, d);
      case kind_function_decl:
	process_function_decl(CAST(function_decl, d), NULL, NULL);
	break;
      PRTCASE(extension_decl, d);
    default: assert(0); break;
    }
  //ast_output_token(MOPS_RP);
}

/* Invariant: all declarations end with ; */
void process_asm_decl(asm_decl d)
{
  /*
  if (!ast_output_node_start((declaration)d)) return;
  process_asm_stmt(d->asm_stmt);
  ast_output_token(MOPS_RP);
  */
}

void process_extension_decl(extension_decl d)
{
  /*
  if (!ast_output_node_start((declaration)d)) return;
  set_location(d->location);
  output("__extension__ ");
  */
  process_toplevel_declaration(d->decl);
  /*
  ast_output_token(MOPS_RP);
  */
}

void process_ellipsis_decl(ellipsis_decl d)
{
  /*
  if (!ast_output_node_start((declaration)d)) return;
  set_location(d->location);
  output("...");
  ast_output_token(MOPS_RP);
  */
}

// daw: Side-effect free.  Returns TRUE if we should call process_data_decl(). 
// hchen: Decide if we should print data_decl.
// We should print out a data_decl if it contains at least:
// - the declaration of a variable that is used in the program, or
// - the definition(non-external declaration) of a global variable
bool should_process_data_decl(data_decl d)
{
  declaration vd;
  variable_decl vd2;
  data_declaration ddecl;
  bool isVariable = FALSE; 

  // decide if this node should be printed at all
  scan_declaration(vd, d->decls)
  {
    vd2 = CAST(variable_decl, vd);
    ddecl = vd2->ddecl;
    // print out data_declaration if
    // - it is used, or
    // - it is the definition of a global variable (which may be used in
    //   another module)
    if (ddecl != NULL && ddecl->kind == decl_variable &&
	(ddecl->isused || (ddecl->isexternalscope && !ddecl->isfilescoperef)))
    {
      isVariable = TRUE;
      break;
    }
  }

  return isVariable;
}

void process_data_decl(data_decl d)
{
  declaration vd;
  variable_decl vd2;
  data_declaration ddecl;

  if (!ast_output_node_start((declaration)d)) return;
  //process_type_elements(d->modifiers);
  //process_type_elements(CAST(type_element, d->attributes));

  scan_declaration (vd, d->decls)
  {
    vd2 = CAST(variable_decl, vd);
    ddecl = vd2->ddecl;
    if (ddecl != NULL && ddecl->kind == decl_variable &&
	(ddecl->isused || (ddecl->isexternalscope && !ddecl->isfilescoperef)))
    {
      process_variable_decl(vd2);
    }
    //if (vd->next)
    //output(", ");
  }
  //outputln(";");
  ast_output_token(MOPS_RP);
}

void process_parameter_declarations(declaration dlist)
{
  declaration d;

  //if (!ast_output_node_start((declaration)dlist)) return;
  scan_declaration (d, dlist)
    process_parameter_declaration(d);
  //ast_output_token(MOPS_RP);
}

void process_parameter_declaration(declaration d)
{
  //if (!ast_output_node_start((declaration)d)) return;
  //startline();
  switch (d->kind)
    {
      PRTCASE(data_decl, d);
      PRTCASE(ellipsis_decl, d);
    default: assert(0); break;
    }
  //ast_output_token(MOPS_RP);
}

void process_function_decl(function_decl d, declaration function_entry,
			   declaration function_exit)
{
  data_declaration ddecl;
  declarator decl;
  
  if (!ast_output_node_start((declaration)d)) return;
  //process_rc_fns(d->rc_fns);
  //process_data_declaration(d->ddecl);
  ddecl = d->ddecl;
  if (ddecl->ftype == function_normal // external scope
      && !ddecl->isinline  // not inline or extern inline
     )
  {
    decl = d->declarator;
    while (decl->kind != kind_function_declarator)
    {
      switch(decl->kind)
      {
	case kind_array_declarator:
	  decl = CAST(array_declarator, decl)->declarator;
	  break;

	case kind_pointer_declarator:
	  decl = CAST(pointer_declarator, decl)->declarator;
	  break;

	default:
	  assert(0);
      }
    }
    vectorAdd(&functionDefinition, decl);
  }
  process_declarator(d->declarator, d->qualifiers, d->attributes);
  //startline();
  process_parameter_declarations(d->old_parms);
  assert(is_compound_stmt(d->stmt));
  process_statement(d->stmt);
  //newline();
  if (function_entry != NULL)
  {
    process_function_entry(function_entry, d->declarator);
  }
  if (function_exit != NULL)
  {
    process_function_exit(function_exit, d->declarator);
  }
  //process_simple_declarator(CAST(function_declarator, d->declarator)->declarator, 1);
  ast_output_token(MOPS_RP);
  //ast_output_token(MOPS_EOL);
}


/* See process_rid. This is ugly: GCC has builtin signatures for memcpy, memcmp,
   strcpy, strcmp and strlen which use the const qualifier. If we drop the
   const qualifier when unparsing declarations for these functions, we get
   a warning from gcc. Thus we disable dropping the const while unparsing
   declarations of these functions (a more principled approach would be to
   annotate the const-nodes in the AST with an "ignore me" flag. To be 
   really principled, we should only ignore the consts that cause a problem
   for our initialisations). 
   In extra evil, we enable const qualifiers via a global. */
// static bool allow_const_evil_hack;

// this was not in the original rc.c.
void process_data_declaration(data_declaration d)
{
  const char *name;
  ast_kind kind;
  bool ret;

  kind = d->kind;
  d->kind = kind_data_declaration;
  ret = ast_output_node_start((declaration)d);
  d->kind = kind;
  if (!ret)
    return;
  //ast_output("%d %d ", d->isexternalscope, d->isfilescoperef);
  //ast_output("%d %d %d %d ", d->isused, d->vtype, d->kind, d->isparameter);
  if (d->isexternalscope)
  {
    name = d->name;
    if (d->isfilescoperef)
      vectorAdd(&variableDeclaration, (void*)name);
    else
      vectorAdd(&variableDefinition, (void*)name);
  }
  ast_output_token(MOPS_RP);
}

void process_variable_decl(variable_decl d)
{
  if (!ast_output_node_start((declaration)d)) return;
  /*
  if (d->ddecl == builtin_memcpy || d->ddecl == builtin_memcmp ||
      d->ddecl == builtin_strcpy || d->ddecl == builtin_strcmp ||
      d->ddecl == builtin_strlen)
    allow_const_evil_hack = TRUE;
  */
  // this was not in the original rc.c.
  if (d->ddecl != NULL)
    process_data_declaration(d->ddecl);
  if (d->declarator != NULL)
    process_declarator(d->declarator, NULL, d->attributes);
  /*
  allow_const_evil_hack = FALSE;

  if (d->asm_stmt)
    process_asm_stmt_plain(d->asm_stmt);
  */
  if (d->arg1)
    {
      //output(" = ");
      process_expression(d->arg1, 1);
    }
  //printf("%d ", d->ddecl->isexternalscope);

  ast_output_token(MOPS_RP);
}

void process_declarator(declarator d, type_element elements, attribute attributes)
{
  if (!ast_output_node_start((declaration)d)) return;
  process_type_elements(elements);
  process_type_elements(CAST(type_element, attributes));
  process_simple_declarator(d, 0);
  ast_output_token(MOPS_RP);
}

void process_simple_declarator(declarator d, int container_priority)
{
  //if (!ast_output_node_start((declaration)d)) return;
  if (d)
    switch (d->kind)
      {
      case kind_function_declarator:
	process_simple_declarator(CAST(function_declarator, d)->declarator, 1);
	process_parameters(CAST(function_declarator, d)->parms);
	break;
      case kind_array_declarator:
	{
	  array_declarator ad = CAST(array_declarator, d);

	  process_simple_declarator(ad->declarator, 1);
	  /*
	  if (!ad->arg1)
	    output("[]");
	  else
	    {
	      set_location(ad->arg1->location);
	      output("[");
	      process_expression(ad->arg1, -1);
	      output("]");
	    }
	  */
	  break;
	}
      case kind_pointer_declarator:
	{
	  
	  pointer_declarator pd = CAST(pointer_declarator, d);

	  /*
	  if (pd->qualifiers)
	    set_location(pd->qualifiers->location);
	  if (container_priority)
	    output("(");
	  output("*");
	  */
	  process_type_elements(pd->qualifiers);
	  process_simple_declarator(pd->declarator, 0);
	  /*
	  if (container_priority)
	    output(")");
	  */
	  break;
	}
      case kind_identifier_declarator:
	//set_location(d->location);
	  ast_output_cstring(CAST(identifier_declarator, d)->cstring);
	break;

      default: assert(0); break;
      }
  //ast_output_token(MOPS_RP);
}

void process_type_elements(type_element elements)
{
  type_element em;

  //if (!ast_output_node_start((declaration)elements)) return;
  scan_type_element (em, elements)
    {
      process_type_element(em);
      //output(" ");
    }
    //ast_output_token(MOPS_RP);
}

void process_type_element(type_element em)
{
  //if (!ast_output_node_start((declaration)em)) return;
  switch (em->kind)
    {
      PRTCASE(typename, em);
      PRTCASE(typeof_expr, em);
      PRTCASE(typeof_type, em);
      PRTCASE(attribute, em);
      PRTCASE(rid, em);
      PRTCASE(qualifier, em);
    default:
      if (is_tag_ref(em))
	process_tag_ref(CAST(tag_ref, em));
      else
	assert(0);
      break;
    }
  //ast_output_token(MOPS_RP);
}

void process_typename(typename tname)
{
  /*
  if (!ast_output_node_start((declaration)tname)) return;
  set_location(tname->location);
  output("%s", tname->ddecl->name);
  ast_output_token(MOPS_RP);
  */
}

void process_typeof_expr(typeof_expr texpr)
{
  /*
  if (!ast_output_node_start((declaration)texpr)) return;
  set_location(texpr->location);
  output("typeof(");
  process_expression(texpr->arg1, -1);
  output(")");
  ast_output_token(MOPS_RP);
  */
}

void process_typeof_type(typeof_type ttype)
{
  /*
  if (!ast_output_node_start((declaration)ttype)) return;
  set_location(ttype->location);
  output("typeof(");
  process_asttype(ttype->asttype);
  output(")");
  ast_output_token(MOPS_RP);
  */
}

void process_attribute(attribute a)
{
  /*
  if (!ast_output_node_start((declaration)a)) return;
  set_location(a->location);
  output("__attribute((");
  process_word(a->word1);
  if (a->word2 || a->args)
    {
      output("(");
      if (a->word2)
	process_word(a->word2);
      process_expressions(a->args, a->word2 == NULL);
      output(")");
    }
  output("))");
  ast_output_token(MOPS_RP);
  */
}

void process_rid(rid r)
{
  /*
  if (!ast_output_node_start((declaration)r)) return;
  set_location(r->location);
  output("%s", rid_name(r));
  ast_output_token(MOPS_RP);
  */
}

void process_qualifier(qualifier q)
{
  /*
  if (!ast_output_node_start((declaration)q)) return;
  // We drop all const's in the output to allow initialisers to be moved to
  // the start of the block. This is not a big issue, as we have already
  // type-checked the input program. 
  if (q->id == volatile_qualifier || (q->id == const_qualifier && allow_const_evil_hack))
    {
      set_location(q->location);
      output("%s", qualifier_name(q->id));
    }
  ast_output_token(MOPS_RP);
  */
}

void process_tag_ref(tag_ref tr)
{
  if (!ast_output_node_start((declaration)tr)) return;
  //set_location(tr->location);
  switch (tr->kind)
    {
      case kind_struct_ref: /*output("struct ");*/ break;
      case kind_union_ref: /*output("union ");*/ break;
      case kind_enum_ref: /*output("enum ");*/ break;
    default: assert(0);
    }

  if (tr->word1)
    process_word(tr->word1);
  if (tr->defined)
    {
      if (tr->kind == kind_enum_ref)
	process_enumerators(tr->fields);
      //else
      //process_fields(tr->fields);
    }
  if (tr->attributes)
    {
      //output(" ");
      process_type_elements(CAST(type_element, tr->attributes));
    }
  ast_output_token(MOPS_RP);
}

void process_enumerators(declaration elist)
{
  declaration d;

  //if (!ast_output_node_start((declaration)elist)) return;
  //output(" {");
  //indent();
  //startline();
  scan_declaration (d, elist)
    {
      process_enumerator(CAST(enumerator, d));
      //if (d->next)
      //output(", ");
    }
  //unindent();
  //startline();
  //output("}");
  //ast_output_token(MOPS_RP);
}

/*
void process_fields(declaration flist)
{
  declaration d;

  if (!ast_output_node_start((declaration)flist)) return;
  output(" {");
  indent();
  startline();
  scan_declaration (d, flist)
    process_field_declaration(d);
  unindent();
  startline();
  output("}");
  ast_output_token(MOPS_RP);
}

void process_field_declaration(declaration d)
{
  if (!ast_output_node_start((declaration)d)) return;
  if (is_extension_decl(d))
    process_field_extension_decl(CAST(extension_decl, d));
  else
    process_field_data_decl(CAST(data_decl, d));
  ast_output_token(MOPS_RP);
}

void process_field_extension_decl(extension_decl d)
{
  if (!ast_output_node_start((declaration)d)) return;
  set_location(d->location);
  output("__extension__ ");
  process_field_declaration(d->decl);
  ast_output_token(MOPS_RP);
}

void process_field_data_decl(data_decl d)
{
  declaration fd;

  if (!ast_output_node_start((declaration)d)) return;
  process_type_elements(d->modifiers);
  process_type_elements(CAST(type_element, d->attributes));

  scan_declaration (fd, d->decls)
    {
      process_field_decl(CAST(field_decl, fd));
      if (fd->next)
	output(", ");
    }
  outputln(";");
  ast_output_token(MOPS_RP);
}

void process_field_decl(field_decl fd)
{
  if (!ast_output_node_start((declaration)fd)) return;
  process_declarator(fd->declarator, NULL, fd->attributes);
  if (fd->arg1)
    {
      output(" : ");
      process_expression(fd->arg1, -1);
    }
  ast_output_token(MOPS_RP);
}
*/

void process_enumerator(enumerator ed)
{
  if (!ast_output_node_start((declaration)ed)) return;
  //set_location(ed->location);
  //output_cstring(ed->cstring);
  if (ed->arg1)
    {
      //output(" = ");
      process_expression(ed->arg1, 1);
    }
  ast_output_token(MOPS_RP);
}

void process_parameters(declaration parms)
{
  declaration d;

  //if (!ast_output_node_start((declaration)parms)) return;
  //output("(");
  scan_declaration (d, parms)
    {
      process_parameter(d);
      //if (d->next)
      //output(", ");
    }
  //output(")");
  //ast_output_token(MOPS_RP);
}

void process_parameter(declaration parm)
{
  if (!ast_output_node_start((declaration)parm)) return;
  switch (parm->kind)
    {
    case kind_oldidentifier_decl:
      //set_location(parm->location);
      ast_output_cstring(CAST(oldidentifier_decl, parm)->cstring);
      break;
    case kind_ellipsis_decl:
      //set_location(parm->location);
      //output("...");
      break;
    case kind_data_decl:
      {
	data_decl dd = CAST(data_decl, parm);
	variable_decl vd = CAST(variable_decl, dd->decls);
	process_type_elements(CAST(type_element, dd->attributes));
	// this was not in the original rc.c.
	if (vd->ddecl != NULL)
          process_data_declaration(vd->ddecl);
        if (vd->declarator != NULL)
          process_declarator(vd->declarator, dd->modifiers, vd->attributes);
	break;
      }
    default: assert(0); break;
    }
  ast_output_token(MOPS_RP);
}

void process_asttype(asttype t)
{
  if (!ast_output_node_start((declaration)t)) return;
  process_declarator(t->declarator, t->qualifiers, NULL);
  ast_output_token(MOPS_RP);
}

void process_word(word w)
{
  if (!ast_output_node_start((declaration)w)) return;
  //set_location(w->location);
  ast_output_cstring(w->cstring);
  ast_output_token(MOPS_RP);
}

void process_expressions(expression elist, bool isfirst)
{
  expression e;

  //if (!ast_output_node_start((declaration)elist)) return;
  scan_expression (e, elist)
    {
      //if (!isfirst) output(", ");
      //isfirst = FALSE;
      process_expression(e, 1); /* priority is that of assignment */
    }
  //ast_output_token(MOPS_RP);
}

#define PRTEXPR(type, x) case kind_ ## type: process_ ## type(CAST(type, (x)), context_priority); break

/* Context priorities are that of the containing operator, starting at 0
   for , going up to 14 for ->, .
   -1 is used for contexts with no priority restrictions */
void process_expression(expression e, int context_priority)
{
  //if (!ast_output_node_start((declaration)e)) return;
  switch (e->kind) 
    {
      PRTEXPR(comma, e);
      PRTEXPR(sizeof_type, e);
      PRTEXPR(alignof_type, e);
      PRTEXPR(label_address, e);
      PRTEXPR(cast, e);
      PRTEXPR(cast_list, e);
      PRTEXPR(conditional, e);
      PRTEXPR(identifier, e);
      PRTEXPR(compound_expr, e);
      PRTEXPR(function_call, e);
      PRTEXPR(array_ref, e);
      PRTEXPR(field_ref, e);
      PRTEXPR(init_list, e);
      PRTEXPR(init_index, e);
      PRTEXPR(init_field, e);
      //PRTEXPR(signed_cst, e);
      //PRTEXPR(unsigned_cst, e);
    case kind_string_cst:
      PRTEXPR(lexical_cst, e);
      PRTEXPR(string, e);
      PRTEXPR(adjust_rc, e);
      PRTEXPR(global_rc, e);
      PRTEXPR(update_rc, e);
    default: 
      if (is_unary(e))
	{
	  process_unary(CAST(unary, e), context_priority);
	  return;
	}
      assert(is_binary(e));
      process_binary(CAST(binary, e), context_priority);
      return;
    }
  //ast_output_token(MOPS_RP);
}

/*
#define OPEN(pri) \
  if (pri < context_priority) \
    output("(")

#define CLOSE(pri) \
  if (pri < context_priority) \
    output(")")
*/

void process_comma(comma e, int context_priority)
{
  if (!ast_output_node_start((declaration)e)) return;
  //OPEN(0);
  process_expressions(e->arg1, TRUE);
  //CLOSE(0);
  ast_output_token(MOPS_RP);
}

void process_sizeof_type(sizeof_type e, int context_priority)
{
  /*
  if (!ast_output_node_start((declaration)e)) return;
  set_location(e->location);
  output("sizeof(");
  process_asttype(e->asttype);
  output(")");
  ast_output_token(MOPS_RP);
  */
}

void process_alignof_type(alignof_type e, int context_priority)
{
  /*
  if (!ast_output_node_start((declaration)e)) return;
  set_location(e->location);
  output("__alignof__(");
  process_asttype(e->asttype);
  output(")");
  ast_output_token(MOPS_RP);
  */
}

void process_label_address(label_address e, int context_priority)
{
  /*
  if (!ast_output_node_start((declaration)e)) return;
  set_location(e->location);
  output("&&");
  process_id_label(e->id_label);
  ast_output_token(MOPS_RP);
  */
}

void process_cast(cast e, int context_priority)
{
  if (!ast_output_node_start((declaration)e)) return;
  /*
  OPEN(13);
  set_location(e->location);
  output("(");
  process_asttype(e->asttype);
  output(")");
  */
  process_expression(e->arg1, 13);
  // CLOSE(13);
  ast_output_token(MOPS_RP);
}

void process_cast_list(cast_list e, int context_priority)
{
  if (!ast_output_node_start((declaration)e)) return;
  /*
  OPEN(13);
  set_location(e->location);
  output("(");
  process_asttype(e->asttype);
  output(")");
  */
  process_init_list(CAST(init_list, e->init_expr), 1);
  //CLOSE(13);
  ast_output_token(MOPS_RP);
}

void process_conditional(conditional e, int context_priority)
{
  if (!ast_output_node_start((declaration)e)) return;
  //OPEN(2);
  process_expression(e->condition, 2);
  //output(" ? ");
  if (e->arg1)
    process_expression(e->arg1, 2);
  //output(" : ");
  process_expression(e->arg2, 2);
  //CLOSE(2);
  ast_output_token(MOPS_RP);
}

void process_identifier(identifier e, int context_priority)
{
  if (!ast_output_node_start((declaration)e)) return;
  //set_location(e->location);
  ast_output_cstring(e->cstring);
  // pointer to its declaration, useful for handling variables
  if (e->ddecl->kind == decl_variable) // || e->ddecl->kind == decl_constant)
    if (ast_output_node_start((declaration)e->ddecl))
    {
      ast_output_token(MOPS_RP);
      fprintf(stderr,
	      "ddecl %u is referenced but not defined yet in identifier %u\n",
	      getUniqueAddress(e->ddecl, NULL), getUniqueAddress(e, NULL));
    }
  ast_output_token(MOPS_RP);
}

void process_compound_expr(compound_expr e, int context_priority)
{
  if (!ast_output_node_start((declaration)e)) return;
  //set_location(e->location);
  //output("(");
  process_compound_stmt(CAST(compound_stmt, e->stmt));
  //output(")");
  ast_output_token(MOPS_RP);
}

void process_function_call(function_call e, int context_priority)
{
  if (!ast_output_node_start((declaration)e)) return;
  /* Check for regionof and rewrite as direct access to page->region array */
  /*
  if (call_to(regionof_decl, e) && e->args && !e->args->next)
    {
      output("(");
      if (rc_pairs_array)
	output("__rcregions[");
      process_regionof(e->args);
      if (rc_pairs_array)
	output("]");
      output(")");
    }
  else 
    {
  */
      process_expression(e->arg1, 14);
      //output("(");
      if (e->args != NULL)
        process_expressions(e->args, TRUE);
      //output(")");
      //  }
  ast_output_token(MOPS_RP);
}

void process_array_ref(array_ref e, int context_priority)
{
  if (!ast_output_node_start((declaration)e)) return;
  process_expression(e->arg1, 14);
  //output("[");
  process_expression(e->arg2, -1);
  //output("]");
  ast_output_token(MOPS_RP);
}

void process_field_ref(field_ref e, int context_priority)
{
  if (!ast_output_node_start((declaration)e)) return;
  /* Reconstruct -> for nicer output */
  if (is_dereference(e->arg1))
    {
      process_expression(CAST(dereference, e->arg1)->arg1, 14);
      ast_output_string("->");
    }
  else
    {
      process_expression(e->arg1, 14);
      ast_output_string(".");
    }
  ast_output_cstring(e->cstring);
  ast_output_token(MOPS_RP);
}

void process_unary(unary e, int context_priority)
{
  char *op = NULL, *postop = NULL;
  int pri = 0;

  if (!ast_output_node_start((declaration)e)) return;
  /* Yuck. Evil hack because gcc is broken (breaks the a[i] === *(a+i)
     rule when a is a non-lvalue array). So we undo our earlier rewrite
     (from fix.c) of a[i] as *(a+i). Note that gcc doesn't allow i[a] in
     this case (bozos at work?) */
  /*
  if (is_dereference(e) && is_plus(e->arg1))
    {
      plus derefed = CAST(plus, e->arg1);

      if (type_array(derefed->arg1->type))
	{
	  process_array_ref(derefed, context_priority);
	  return;
	}
    }
  */
  switch (e->kind)
    {
    case kind_dereference: op = "*"; break;
    case kind_extension_expr: op = "__extension__ "; break;
      /* Higher priority for sizeof/alignof expr because we must
	 add parens around sizeof cast_expr (e.g. sizeof((char)x), not
	 sizeof (char)x */
    case kind_sizeof_expr: op = "sizeof "; pri = 14; break;
    case kind_alignof_expr: op = "__alignof__ "; pri = 14; break;
    case kind_realpart: op = "__real__ "; break;
    case kind_imagpart: op = "__imag__ "; break;
    case kind_address_of: op = "&"; break;
    case kind_unary_minus: op = "-"; break;
    case kind_unary_plus: op = "+"; break;
    case kind_preincrement: op = "++"; break;
    case kind_predecrement: op = "--"; break;
    case kind_postincrement: postop = "++"; break;
    case kind_postdecrement: postop = "--"; break;
    case kind_conjugate: case kind_bitnot: op = "~"; break;
    case kind_not: op = "!"; break;
    default: assert(0); return;
    }
  // OPEN(13);
  // set_location(e->location);
  if (op)
    {
      ast_output_string(op);
      /*
      if (is_unary(e->arg1))
	output(" "); // Catch weirdness such as - - x
      if (!pri)
	pri = 13;
      */
    }
  if (postop)
    ast_output_string(postop);
  process_expression(e->arg1, pri ? pri : 14);
  //CLOSE(13);
  ast_output_token(MOPS_RP);
}

/*
  This was defined in unparse.c.  To make ld happy, comment out this copy
static const char *binary_op_name(ast_kind kind)
{
  switch (kind)
    {
    case kind_plus: return "+"; 
    case kind_minus: return "-"; 
    case kind_times: return "*"; 
    case kind_divide: return "/"; 
    case kind_modulo: return "%"; 
    case kind_lshift: return "<<"; 
    case kind_rshift: return ">>"; 
    case kind_leq: return "<="; 
    case kind_geq: return ">="; 
    case kind_lt: return "<"; 
    case kind_gt: return ">"; 
    case kind_eq: return "=="; 
    case kind_ne: return "!="; 
    case kind_bitand: return "&"; 
    case kind_bitor: return "|"; 
    case kind_bitxor: return "^"; 
    case kind_andand: return "&&"; 
    case kind_oror: return "||"; 
    case kind_assign: return "="; 
    case kind_plus_assign: return "+="; 
    case kind_minus_assign: return "-="; 
    case kind_times_assign: return "*="; 
    case kind_divide_assign: return "/="; 
    case kind_modulo_assign: return "%="; 
    case kind_lshift_assign: return "<<="; 
    case kind_rshift_assign: return ">>="; 
    case kind_bitand_assign: return "&="; 
    case kind_bitor_assign: return "|="; 
    case kind_bitxor_assign: return "^="; 
    default: assert(0); return "<bad>";
    }
}
*/

void process_binary(binary e, int context_priority)
{
  int pri, lpri, rpri;
  const char *op = binary_op_name(e->kind);
  //bool perturb = FALSE;

  if (!ast_output_node_start((declaration)e)) return;
  switch (e->kind)
    {
    case kind_plus: case kind_minus:
      lpri = 11; pri = 11; rpri = 12; break;
    case kind_times: case kind_divide: case kind_modulo:
      lpri = 12; pri = 12; rpri = 13; break;
    case kind_lshift: case kind_rshift:
      lpri = 10; pri = 10; rpri = 11; break;
    case kind_leq: case kind_geq: case kind_lt: case kind_gt:
      lpri = 9; pri = 9; rpri = 10; break;
    case kind_eq: case kind_ne:
      lpri = 8; pri = 8; rpri = 9; break;
    case kind_bitand:
      lpri = 7; pri = 7; rpri = 8; break;
    case kind_bitor:
      lpri = 5; pri = 5; rpri = 6; break;
    case kind_bitxor:
      lpri = 6; pri = 6; rpri = 7; break;
    case kind_andand:
      lpri = 4; pri = 4; rpri = 5; break;
    case kind_oror:
      lpri = 3; pri = 3; rpri = 4; break;
    case kind_assign: case kind_plus_assign: case kind_minus_assign: 
    case kind_times_assign: case kind_divide_assign: case kind_modulo_assign:
    case kind_lshift_assign: case kind_rshift_assign: case kind_bitand_assign:
    case kind_bitor_assign: case kind_bitxor_assign:
      lpri = 13; pri = 1; rpri = 1; break;
    default: assert(0); return;
    }
  /*
  OPEN(pri);
  if (flag_perturb && e->kind == kind_assign)
    perturb = drand48() >= 0.8;
  if (perturb)
    output("(perturb_counter++, ");
  */
  ast_output_string(op);
  process_expression(e->arg1, lpri);
  //set_location(e->location);
  process_expression(e->arg2, rpri);
  //if (perturb)
  //  output(")");
  //CLOSE(pri);

  //if (e->kind == kind_assign && type_pointer(e->type) && rc_chkopt)
  //output_chkopt_debug(e);
  ast_output_token(MOPS_RP);
}

/*
static char *cst_suffix(type t)
{
  if (type_equal(t, unsigned_int_type))
    return "U";
  if (type_equal(t, long_type))
    return "L";
  if (type_equal(t, unsigned_long_type))
    return "UL";
  if (type_equal(t, long_long_type))
    return "LL";
  if (type_equal(t, unsigned_long_long_type))
    return "ULL";
  return "";
}

void process_signed_cst(signed_cst e, int context_priority)
{
  if (!ast_output_node_start((declaration)e)) return;
  //set_location(e->location);
  // XXX: broken for >32bit csts 
  //output("%ld%s", (long)e->signedcst, cst_suffix(e->type));
  ast_output("%ld ", (long)e->signedcst);
  ast_output_token(MOPS_RP);
}

void process_unsigned_cst(unsigned_cst e, int context_priority)
{
  if (!ast_output_node_start((declaration)e)) return;
  //set_location(e->location);
  // XXX: broken for >32bit csts
  //output("%lu%s", (unsigned long)e->unsignedcst, cst_suffix(e->type));
  ast_output("%lu ", (unsigned long)e->unsignedcst);
  ast_output_token(MOPS_RP);
}
*/
void process_lexical_cst(lexical_cst e, int context_priority)
{
  if (!ast_output_node_start((declaration)e)) return;
  //set_location(e->location);
  ast_output_cstring(e->cstring);
  ast_output_token(MOPS_RP);
}

void process_string(string e, int context_priority)
{
  expression s;

  if (!ast_output_node_start((declaration)e)) return;
  scan_expression (s, e->strings)
    process_expression(s, 1);
  ast_output_token(MOPS_RP);
}

void process_adjust_rc(adjust_rc rcop, int context_priority){}
void process_global_rc(global_rc rcop, int context_priority){}
void process_update_rc(update_rc rcop, int context_priority){}
/*
void process_regionof(expression e)
{
  if (!ast_output_node_start((declaration)e)) return;
  output("__rcregionmap[(");
  process_asttype(ptrint_type_ast);
  output(")");
  process_expression(e, 13);
  output(" >> " RPAGELOG_STRING "]");
  ast_output_token(MOPS_RP);
}

void process_regionof_name(const char *name)
{
  if (!ast_output_node_start((declaration)char)) return;
  output("__rcregionmap[(");
  process_asttype(ptrint_type_ast);
  output(")%s >> " RPAGELOG_STRING "]", name);
  ast_output_token(MOPS_RP);
}

void process_regionof_closure(process_closure closure)
{
  if (!ast_output_node_start((declaration)closure)) return;
  output("__rcregionmap[(");
  process_asttype(ptrint_type_ast);
  output(")(");
  CALL0(closure);
  output(") >> " RPAGELOG_STRING "]");
  ast_output_token(MOPS_RP);
}

void process_globalrc_adjust(char *by)
{
  if (rc_pairs_array)
    output("[__rcs[0]]"); // Remember, a[b] is the same as b[a]
  else if (rc_pairs_from)
    output("->id[zeroregion.rc]"); // Remember, a[b] is the same as b[a] 
  else if (rc_pairs)
    output("->rc[0]");
  else
    output("->rc");

  // Looks nicer ;-)
  if (strcmp(by, "1") == 0)
    output("++");
  else if (strcmp(by, "-1") == 0)
    output("--");
  else
    output("+= %s", by);
}

void process_updaterc_adjust(char *by, const char *base)
{
  if (rc_pairs_array)
    output("[__rcs[%s]]", base);
  else if (rc_pairs_from)
    output("->id[%s->rc]", base);
  else if (rc_pairs)
    output("->rc[%s->id]", base);
  else
    output("->rc");

  // Looks nicer ;-)
  if (strcmp(by, "1") == 0)
    output("++");
  else if (strcmp(by, "-1") == 0)
    output("--");
  else
    output(" += %s", by);
}

void process_nelements(expression array)
{
  if (!ast_output_node_start((declaration)array)) return;
  type basetype = array->type;

  output("sizeof ");
  process_expression(array, 13);
  output(" / sizeof ");
  while (type_array(basetype))
    {
      output("*");
      basetype = type_array_of(basetype);
    }
  process_expression(array, 13);
  ast_output_token(MOPS_RP);
}

void process_adjust_rc(adjust_rc rcop, int context_priority)
{
  if (!ast_output_node_start((declaration)rcop)) return;
  expression of = rcop->arg1;

  set_location(rcop->location);

  output("(");
  indent();

  if (rc_collect_statistics)
    {
      stats_adjust_rc(rcop);
      output(", ");
    }

  if (type_aggregate(of->type))
    {
#if 1
      output("%s(& ", rcop->rcfn_decl->name);
      process_expression(of, 13);
      output(", %d)", rcop->incrementrc ? 1 : -1);
#else
      output("0");
#endif
    }
  else if (type_array(of->type))
    {
#ifdef ASSUME_GCC
      output("({ int __rctmp;");
      indent();
      newline();
      output("for (__rctmp = 0; __rctmp < ");
      process_nelements(of);
      output("; __rctmp++) ");
      output("%s((", rcop->rcfn_decl->name);
      process_expression(of, 13);
      output(") + __rctmp, %d); })", rcop->incrementrc ? 1 : -1);
      unindent();
#else
      output("__rc_adjustarray(");
      process_expression(of, 13);
      output(", ");
      process_nelements(of);
      output(", %s, %d)", rcop->rcfn_decl->name, rcop->incrementrc ? 1 : -1);
#endif
    }
  else
    {
      process_regionof(of);
      process_globalrc_adjust(rcop->incrementrc ? "1" : "-1");
    }
  if (rc_collect_statistics_end)
    {
      output(", ");
      stats_adjust_rc_end(rcop);
    }

  output(")");
  unindent();
  ast_output_token(MOPS_RP);
}

void process_update_rc(update_rc rcop, int context_priority)
{
  if (!ast_output_node_start((declaration)rcop)) return;
  binary brcop = CAST(binary, rcop);
  expression of = rcop->arg1;

  set_location(rcop->location);
  output("(");
  indent();

  if (rc_collect_statistics)
    {
      stats_update_rc(rcop);
      output(", ");
    }

  if (type_aggregate(of->type))
    {
#if 1
      output("%s(&", rcop->rcfn_decl->name);
      process_expression(of, 1);
      output(", &");
      process_expression(rcop->arg2, 1);
      output("), ");
#endif
    }
  else if (type_parentptr(of->type))
    {
      qwrites++;
      if (known_parentptr_write(rcop))
	safe_qwrites++;

      if (rc_keepchecks || !(rc_nochecks || known_parentptr_write(rcop) || zero_expression(rcop->arg2)))
	{
#if 0
	  output("check_child(");
	  process_regionof_name(rcop->ddecl->name);
	  output(", ");
	  process_regionof(rcop->arg2);
	  output("), ");
#else
	  const char *rs = rcop->temp1->name, *ofs = rcop->temp3->name;

	  output("%s = ", rs);
	  process_regionof_name(rcop->ddecl->name);
	  output(", %s = ", ofs);
	  process_regionof(rcop->arg2);
	  output(", %s->rid >= %s->rid && %s->rid < %s->nextid ? 0 : abort(), ",
		 rs, ofs, rs, ofs);
#endif
	}
    }
  else if (type_traditional(of->type))
    {
      qwrites++;
      if (known_traditional_write(brcop))
	safe_qwrites++;

      if (rc_keepchecks || !(rc_nochecks || known_traditional_write(brcop)))
	{
	  process_regionof(rcop->arg2);
	  output(" != &zeroregion ? abort() : 0, ");
	}
    }
  else if (type_sameregion(of->type))
    {
      qwrites++;
      if (known_sameregion_write(rcop))
	safe_qwrites++;

      if (rc_keepchecks || !(rc_nochecks || known_sameregion_write(rcop) || zero_expression(rcop->arg2)))
	{
	  process_expression(rcop->arg2, 5);
	  output(" && ");
	  process_regionof(rcop->arg2);
	  output(" != ");
	  process_regionof_name(rcop->ddecl->name);
	  output(" ? abort() : 0, ");
	}
    }
  else if (known_sameregion_write(rcop) || known_traditional_write(brcop) ||
	   zero_expression(rcop->arg2))
    {
#ifdef NULLRC
      process_expression(of, 13);
      output(" ? (");
#endif
      output("%s = ", rcop->temp1->name);
      process_regionof(of);
      output(", %s = ", rcop->temp3->name);
      process_regionof_name(rcop->ddecl->name);
      output(", (%s == %s ? 0 : %s",
	     rcop->temp3->name, rcop->temp1->name, rcop->temp1->name);
      process_updaterc_adjust("-1", rcop->temp3->name);
      output(")");
#ifdef NULLRC
      output(") : 0");
#endif
      output(", ");
    }
  else
    {
#ifdef NULLRC
      process_expression(of, 13);
      output(" ? (");
#endif
#if 1
      output("%s = ", rcop->temp1->name);
      process_regionof(of);
      output(", %s = ", rcop->temp2->name);
      process_regionof(rcop->arg2);
      output(", %s == %s ? 0 : (%s = ", rcop->temp1->name, rcop->temp2->name,
	     rcop->temp3->name);
      process_regionof_name(rcop->ddecl->name);
      output(", (%s == %s ? 0 : %s", 
	     rcop->temp3->name, rcop->temp1->name, rcop->temp1->name);
      process_updaterc_adjust("-1", rcop->temp3->name);
      output("), (%s == %s ? 0 : %s",
	     rcop->temp3->name, rcop->temp2->name, rcop->temp2->name);
      process_updaterc_adjust("1", rcop->temp3->name);
      output("))");
#endif
#ifdef NULLRC
      output(") : (%s = ", rcop->temp2->name);
      process_regionof(rcop->arg2);
      output(", %s = ", rcop->temp3->name);
      process_regionof_name(rcop->ddecl->name);
      output(", (%s == %s ? 0 : %s",
	     rcop->temp3->name, rcop->temp2->name, rcop->temp2->name);
      process_updaterc_adjust("1", rcop->temp3->name);
      output("))");
#endif
      output(", ");
    }
  if (rc_collect_statistics_end)
    {
      stats_update_rc_end(rcop);
      output(", ");
    }

  process_expression(of, 13);
  output(" = ");
  process_expression(rcop->arg2, 1);

  output_chkopt_debug(CAST(binary, rcop));
  output(")");
  unindent();
  ast_output_token(MOPS_RP);
}

void process_global_rc(global_rc rcop, int context_priority)
{
  if (!ast_output_node_start((declaration)rcop)) return;
  binary brcop = CAST(binary, rcop);
  expression dest = rcop->arg1, src = rcop->arg2;
  bool stats_end = rc_collect_statistics_end;

  set_location(rcop->location);
  output("(");
  indent();

  if (rc_collect_statistics)
    {
      stats_end = stats_global_rc(rcop) || stats_end;
      output(", ");
    }
  
  if (type_aggregate(dest->type))
    {
#if 1
      output("%s(& ", rcop->rcfn_decl->name);
      process_expression(dest, 13);
      output(", -1), ");
#endif
      process_expression(dest, 13);
      output(" = ");
      process_expression(src, 1);
#if 1
      output(", %s(& ", rcop->rcfn_decl->name);
      process_expression(dest, 13);
      output(", 1)");
#endif
    }
  else if (type_traditional(dest->type) || type_sameregion(dest->type) ||
	   type_parentptr(dest->type))
    {
      qwrites++;
      if (known_traditional_write(brcop))
	safe_qwrites++;

      process_expression(dest, 13);
      output(" = ");
      process_expression(src, 1);
      if (rc_keepchecks || !(rc_nochecks || known_traditional_write(brcop) || zero_expression(src)))
	{
	  output(", ");
	  process_regionof(dest);
	  output(" != &zeroregion ? abort() : 0");
	}
    }
  else if (known_traditional_write(brcop) || zero_expression(src))
    {
#ifdef NULLRC
      process_expression(dest, 13);
      output(" ? ");
#endif
      process_regionof(dest);
      process_globalrc_adjust("-1");
#ifdef NULLRC
      output(" : 0");
#endif
      output(", ");
      process_expression(dest, 13);
      output(" = ");
      process_expression(src, 1);
    }
  else
    {
#ifdef NULLRC
      process_expression(dest, 13);
      output(" ? (");
#endif
      output("%s = ", rcop->temp1->name);
      process_regionof(dest);
      output(", ");
      process_expression(dest, 13);
      output(" = ");
      process_expression(src, 1);
      output(", %s = ", rcop->temp2->name);
      process_regionof(dest);
      output(", %s == %s ? 0 : (%s", rcop->temp1->name, rcop->temp2->name,
	     rcop->temp1->name);
      process_globalrc_adjust("-1");
      output(", %s", rcop->temp2->name);
      process_globalrc_adjust("1");
      output(")");
#ifdef NULLRC
      output(") : (");
      process_expression(dest, 13);
      output(" = ");
      process_expression(src, 1);
      output(", ");
      process_regionof(dest);
      process_globalrc_adjust("1");
      output(")");
#endif
    }
  if (stats_end)
    {
      output(", ");
      stats_global_rc_end(rcop);
    }
  
  output_chkopt_debug(CAST(binary, rcop));
  output(")");
  unindent();
  ast_output_token(MOPS_RP);
}

static void process_aggregate_type(type t)
{
  if (type_struct(t))
    output("struct ");
  else
    output("union ");
  output("%s", type_tag(t)->name);
}

static void print_var(process_closure closure)
{
  output("(*%s)", closure->name);
}

static void print_field(process_closure closure)
{
  CALL0(closure->parent);
  output(".%s", closure->name);
}

static void print_array(process_closure closure)
{
  CALL0(closure->parent);
  output("[%s]", closure->name);
}

typedef void (*prtrc_closure)(process_closure ptr);

static void process_rcfn(type basetype, process_closure base, prtrc_closure ptraction)
{
  if (type_pointer(basetype))
    {
      if (!(type_sameregion(basetype) || type_traditional(basetype) ||
	    type_parentptr(basetype)))
	ptraction(base);
    }
  else if (type_contains_cross_pointers(basetype))
    {
      if (type_array(basetype))
	{
	  char *indexname = rstrdup(tempregion, next_temporary());
	  struct process_closure parray;

	  parray.fn = print_array;
	  parray.parent = base;
	  parray.name = indexname;

	  outputln("{");
	  indent();
	  outputln("unsigned long %s;", indexname);
	  newline();
	  output("for (%s = 0; %s < (sizeof ", indexname, indexname);
	  CALL0(base);
	  output(") / sizeof *");
	  CALL0(base);
	  outputln("; %s++)", indexname);

	  indent();
	  process_rcfn(type_array_of(basetype), &parray, ptraction);
	  unindent();

	  unindent();
	  outputln("}");
	}
      else
	{
	  tag_declaration tag = type_tag(basetype);
	  field_declaration field;
	  struct process_closure pfield;

	  // unions must be handled by programmer-provided rc_adjust fns 
	  // We now allow unions and just produce bogus code. An error has
	  //   already been signalled 
	  //assert(type_struct(basetype));

	  pfield.fn = print_field;
	  pfield.parent = base;
      
	  outputln("{");
	  indent();

	  for (field = tag->fieldlist; field; field = field->next)
	    {
	      pfield.name = field->name;
	      process_rcfn(field->type, &pfield, ptraction);
	    }
	  unindent();
	  outputln("}");
	}
    }
}

static void process_adjust_ptr(process_closure ptr)
{
  if (rc_safe_adjustfn)
    {
      output("__rcr1 = ");
      process_regionof_closure(ptr);
      outputln(";");

      output("if (__rcr1 != base) __rcr1");
    }
  else
      process_regionof_closure(ptr);

  process_globalrc_adjust("by");
  outputln(";");
}

void process_adjust_function_decl(adjust_function_decl d)
{
  if (!ast_output_node_start((declaration)d)) return;
  struct process_closure pvar;

  outputln("static size_t %s(void *x, int by)", d->rcfn_decl->name);
  outputln("{");
  indent();
  if (type_pointer(d->type))
    outputln("void **p = x;");
  else
    {
      process_aggregate_type(d->type);
      outputln("*p = x;");
    }
  newline();

  if (rc_safe_adjustfn)
    {
      output("regionid __rcr1, base = ");
      process_regionof_name("p");
      outputln(";");
      newline();
    }

#if 1
  pvar.fn = print_var;
  pvar.name = "p";
  process_rcfn(d->type, &pvar, process_adjust_ptr);
#endif

  outputln("return sizeof *p;");
  unindent();
  outputln("}");
  newline();
  ast_output_token(MOPS_RP);
}

static bool evilhack;

static void print_evil(process_closure closure)
{
  if (evilhack)
    output("(*new)");
  else
    output("(*old)");
}

static void process_update_ptr(process_closure ptr)
{
  outputln("{");
  indent();

  output("__rcr1 = ");
  process_regionof_closure(ptr);
  outputln(";");
  output("__rcr2 = ");
  evilhack = TRUE;
  process_regionof_closure(ptr);
  evilhack = FALSE;
  outputln(";");

  outputln("if (__rcr1 != __rcr2)");
  outputln("{");
  indent();

  output("if (__rcr1 != base) __rcr1");
  process_updaterc_adjust("-1", "base");
  outputln(";");

  output("if (__rcr2 != base) __rcr2");
  process_updaterc_adjust("1", "base");
  outputln(";");

  unindent();
  outputln("}");

  unindent();
  outputln("}");
}

void process_update_function_decl(update_function_decl d)
{
  if (!ast_output_node_start((declaration)d)) return;
  struct process_closure pvar;

  outputln("static void %s(", d->rcfn_decl->name);
  process_aggregate_type(d->type);
  output(" *old, ");
  process_aggregate_type(d->type);
  outputln(" *new)");

  outputln("{");
  indent();

  output("regionid __rcr1, __rcr2, base = ");
  process_regionof_name("old");
  outputln(";");
  newline();

  pvar.fn = print_evil;
  process_rcfn(d->type, &pvar, process_update_ptr);

  unindent();
  outputln("}");
  newline();
  ast_output_token(MOPS_RP);
}

void process_adjustarray_function_decl(adjustarray_function_decl d)
{
  if (!ast_output_node_start((declaration)d)) return;
  output("\
static void __rc_adjustarray(void *array, int nelems, adjust_fn *adjustelem, int by)\n\
{\n\
  char *x = array;\n\
\n\
  while (nelems--)\n\
    x += adjustelem(x, by);\n\
  ast_output_token(MOPS_RP);
}\n\n");
}

void process_rc_fn(declaration rc_fn)
{
  //if (!ast_output_node_start((declaration)rc_fn)) return;
  switch (rc_fn->kind)
    {
      PRTCASE(adjust_function_decl, rc_fn);
      PRTCASE(adjustarray_function_decl, rc_fn);
      PRTCASE(update_function_decl, rc_fn);
    default: assert(0); return;
    }
  //ast_output_token(MOPS_RP);
}

void process_rc_fns(declaration rc_fns)
{
  declaration rc_fn;

  if (!ast_output_node_start((declaration)rc_fns)) return;
  scan_declaration (rc_fn, rc_fns)
    process_rc_fn(rc_fn);
  ast_output_token(MOPS_RP);
}
*/

void process_init_list(init_list e, int context_priority)
{
  if (!ast_output_node_start((declaration)e)) return;
  //set_location(e->location);
  //output("{ ");
  process_expressions(e->args, TRUE);
  //output(" }");
  ast_output_token(MOPS_RP);
}

void process_init_index(init_index e, int context_priority)
{
  if (!ast_output_node_start((declaration)e)) return;
  //set_location(e->location);
  //output("[");
  process_expression(e->arg1, 1);
  if (e->arg2)
    {
      //output(" ... ");
      process_expression(e->arg2, 1);
    }
  //output("] ");
  process_expression(e->init_expr, 1);
  ast_output_token(MOPS_RP);
}

void process_init_field(init_field e, int context_priority)
{
  if (!ast_output_node_start((declaration)e)) return;
  process_word(e->word1);
  //output(" : ");
  process_expression(e->init_expr, 1);
  ast_output_token(MOPS_RP);
}

void process_statement(statement s)
{
  //if (!ast_output_node_start((declaration)s)) return;
  switch (s->kind)
    {
      PRTCASE(asm_stmt, s);
      PRTCASE(compound_stmt, s);
      PRTCASE(if_stmt, s);
      PRTCASE(labeled_stmt, s);
      PRTCASE(expression_stmt, s);
      PRTCASE(while_stmt, s);
      PRTCASE(dowhile_stmt, s);
      PRTCASE(switch_stmt, s);
      PRTCASE(for_stmt, s);
      PRTCASE(break_stmt, s);
      PRTCASE(continue_stmt, s);
      PRTCASE(return_stmt, s);
      PRTCASE(goto_stmt, s);
      PRTCASE(computed_goto_stmt, s);
      PRTCASE(empty_stmt, s);
    default: assert(0); return;
    }
  //ast_output_token(MOPS_RP);
}

static void process_as_compound(statement s)
{
  /*
  if (!is_compound_stmt(s))
    outputln("{");
  */
  process_statement(s);
  /*
  if (!is_compound_stmt(s))
    {
      startline();
      outputln("}");
    }
  */
}

void process_compound_stmt(compound_stmt s)
{
  statement s1;

  if (!ast_output_node_start((declaration)s)) return;
  /*
  set_location(s->location);
  outputln("{");
  indent();
  if (s->id_labels)
    {
      id_label l;

      output("__label__ ");
      scan_id_label (l, s->id_labels)
	{
	  process_id_label(l);
	  if (l->next) 
	    output(", ");
	}
      outputln(";");
    }
  */
  if (s->decls)
    {
      process_compound_declarations(s->decls);
      //newline();
    }

  scan_statement (s1, s->stmts)
    process_statement(s1);

  //unindent();
  //outputln("}");
  ast_output_token(MOPS_RP);
}

void process_compound_declarations(declaration dlist)
{
  declaration d;

  //if (!ast_output_node_start((declaration)dlist)) return;
  scan_declaration (d, dlist)
    process_compound_declaration(d);

  //ast_output_token(MOPS_RP);
}

void process_compound_declaration(declaration d)
{
  //if (!ast_output_node_start((declaration)d)) return;
  //startline();
  switch (d->kind)
    {
      PRTCASE(data_decl, d);
      PRTCASE(extension_decl, d);
      //PRTCASE(function_decl, d);
      case kind_function_decl:
	process_function_decl(CAST(function_decl, d), NULL, NULL);
	break;
    default: assert(0); break;
    }
  //ast_output_token(MOPS_RP);
}

void process_asm_stmt(asm_stmt s){}
/*
void process_asm_stmt(asm_stmt s)
{
  if (!ast_output_node_start((declaration)s)) return;
  process_asm_stmt_plain(s);
  output(";");
  ast_output_token(MOPS_RP);
}

void process_asm_stmt_plain(asm_stmt s)
{
  if (!ast_output_node_start((declaration)s)) return;
  set_location(s->location);
  output(" __asm ");
  if (s->qualifiers)
    process_type_elements(s->qualifiers);
  output("(");
  process_expression(s->arg1, -1);
  if (s->asm_operands1 || s->asm_operands2 || s->asm_clobbers)
    {
      output(" : ");
      process_asm_operands(s->asm_operands1);

      if (s->asm_operands2 || s->asm_clobbers)
	{
	  output(" : ");
	  process_asm_operands(s->asm_operands2);

	  if (s->asm_clobbers)
	    {
	      output(" : ");
	      process_expressions(CAST(expression, s->asm_clobbers), TRUE);
	    }
	}
    }
  output(")");
  ast_output_token(MOPS_RP);
}

void process_asm_operands(asm_operand olist)
{
  asm_operand o;

  if (!ast_output_node_start((declaration)olist)) return;
  scan_asm_operand (o, olist)
    {
      process_asm_operand(o);
      if (o->next)
	output(", ");
    }
  ast_output_token(MOPS_RP);
}

void process_asm_operand(asm_operand o)
{
  if (!ast_output_node_start((declaration)o)) return;
  process_string(o->string, -1);
  output("(");
  process_expression(o->arg1, -1);
  output(")");
  ast_output_token(MOPS_RP);
}
*/

void process_if_stmt(if_stmt s)
{
  if (!ast_output_node_start((declaration)s)) return;
  //set_location(s->location);
  //output("if (");
  process_expression(s->condition, 2); /* Force parens around assignments */
  //output(") ");
  //indent();
  process_as_compound(s->stmt1);
  //unindent();
  if (s->stmt2)
    {
      //startline();
      //output("else ");
      //indent();
      process_as_compound(s->stmt2);
      //unindent();
    }
  ast_output_token(MOPS_RP);
}

void process_labeled_stmt(labeled_stmt s)
{
  if (!ast_output_node_start((declaration)s)) return;
  //process_label(s->label);
  //output(": ");
  //indent();
  process_statement(s->stmt);
  //unindent();
  ast_output_token(MOPS_RP);
}

void process_expression_stmt(expression_stmt s)
{
  if (!ast_output_node_start((declaration)s)) return;
  process_expression(s->arg1, -1);
  //outputln(";");
  ast_output_token(MOPS_RP);
}

void process_while_stmt(while_stmt s)
{
  if (!ast_output_node_start((declaration)s)) return;
  //set_location(s->location);
  //output("while (");
  process_expression(s->condition, 2); /* Force parens around assignments */
  //output(") ");
  //indent();
  process_statement(s->stmt);
  //unindent();
  ast_output_token(MOPS_RP);
}

void process_dowhile_stmt(while_stmt s)
{
  if (!ast_output_node_start((declaration)s)) return;
  //set_location(s->location);
  //output("do ");
  //indent();
  process_statement(s->stmt);
  //unindent();
  //startline();
  //output("while (");
  process_expression(s->condition, 2); /* Force parens around assignments */
  //outputln(");");
  ast_output_token(MOPS_RP);
}

void process_switch_stmt(switch_stmt s)
{
  if (!ast_output_node_start((declaration)s)) return;
  //set_location(s->location);
  //output("switch (");
  process_expression(s->condition, 2); /* Force parens around assignments */
  //output(") ");
  //indent();
  process_statement(s->stmt);
  //unindent();
  ast_output_token(MOPS_RP);
}

void process_for_stmt(for_stmt s)
{
  if (!ast_output_node_start((declaration)s)) return;
  //set_location(s->location);
  //output("for (");
  if (s->arg1)
    process_expression(s->arg1, -1);
  //output("; ");
  if (s->arg2)
    process_expression(s->arg2, -1);
  //output("; ");
  if (s->arg3)
    process_expression(s->arg3, -1);
  //output(") ");
  //indent();
  process_statement(s->stmt);
  //unindent();
  ast_output_token(MOPS_RP);
}  

void process_break_stmt(break_stmt s)
{
  /*
  if (!ast_output_node_start((declaration)s)) return;
  set_location(s->location);
  outputln("break;");
  ast_output_token(MOPS_RP);
  */
}

void process_continue_stmt(continue_stmt s)
{
  /*
  if (!ast_output_node_start((declaration)s)) return;
  set_location(s->location);
  outputln("continue;");
  ast_output_token(MOPS_RP);
  */
}

void process_return_stmt(return_stmt s)
{
  if (!ast_output_node_start((declaration)s)) return;
  //set_location(s->location);
  if (s->arg1)
    {
      //output("return ");
      process_expression(s->arg1, -1);
      //outputln(";");
    }
  // else
  //  outputln("return;");
  ast_output_token(MOPS_RP);
}

void process_goto_stmt(goto_stmt s)
{
  /*
  if (!ast_output_node_start((declaration)s)) return;
  set_location(s->location);
  output("goto ");
  process_id_label(s->id_label);
  outputln(";");
  ast_output_token(MOPS_RP);
  */
}

void process_computed_goto_stmt(computed_goto_stmt s)
{
  if (!ast_output_node_start((declaration)s)) return;
  //set_location(s->location);
  //output("goto *");
  process_expression(s->arg1, -1);
  //outputln(";");
  ast_output_token(MOPS_RP);
}

void process_empty_stmt(empty_stmt s)
{
  if (!ast_output_node_start((declaration)s)) return;
  //set_location(s->location);
  //outputln(";");
  ast_output_token(MOPS_RP);
}

/*
void process_label(label l)
{
  //if (!ast_output_node_start((declaration)l)) return;
  switch (l->kind)
    {
      PRTCASE(id_label, l);
      PRTCASE(case_label, l);
      PRTCASE(default_label, l);
    default: assert(0); return;
    }
    //ast_output_token(MOPS_RP);
}

void process_id_label(id_label l)
{
  if (!ast_output_node_start((declaration)l)) return;
  set_location(l->location);
  output_cstring(l->cstring);
  ast_output_token(MOPS_RP);
}
*/

void process_case_label(case_label l)
{
  if (!ast_output_node_start((declaration)l)) return;
  //set_location(l->location);
  //output("case ");
  process_expression(l->arg1, 1);
  if (l->arg2)
    {
      //output(" ... ");
      process_expression(l->arg2, 1);
    }
  ast_output_token(MOPS_RP);
}

/*
void process_default_label(default_label l)
{
  if (!ast_output_node_start((declaration)l)) return;
  set_location(l->location);
  output("default");
  ast_output_token(MOPS_RP);
}
*/

// Added by hchen
void process_function_entry(declaration n, declarator entry)
{
  bool ret;
  ast_kind kind;

  kind = n->kind;
  n->kind = kind_function_entry;
  ret = ast_output_node_start(n);
  n->kind = kind;
  if (!ret)
    return;
  process_declarator(entry, NULL, NULL);
  ast_output_token(MOPS_RP);
}

void process_function_exit(declaration n, declarator entry)
{
  bool ret;
  ast_kind kind;

  kind = n->kind;
  n->kind = kind_function_exit;
  ret = ast_output_node_start(n);
  n->kind = kind;
  if (!ret)
    return;
  process_declarator(entry, NULL, NULL);
  ast_output_token(MOPS_RP);
}
// END

/*
Print out CFG.  In function_decl, nodes are statements.  But in the printout
of this function, nodes are program positions (right before a statement is
executed), and edges are statements.
*/
char *getAbsoluteFileName(char *cwd, char *fileName)
{
  static char buf[1024];

  // Warning: For performance reasons, I don't check if
  // strlen(cwd) + strlen(fileName) < size(buf)
  if (fileName[0] == '/')
  {
    return fileName;
  }
  else
  {
    strcpy(buf, cwd);
    strcat(buf, fileName);
    return buf;
  }
}

void process_cfg(function_decl fd)
{
  node *gnodes = fd->postorder_nodes;
  int nnodes = fd->postorder_size;
  node n, n2;
  edge e;
  int i, j;
  node exitnode = CAST(node, fd);
  char *functionName, *sourceFileName;
  bool hasEntry = FALSE, hasExit = FALSE, isDeadFunction;
  char *ptr, *str;
  char cwd[1024], buf[4096];
  declaration function_entry, function_exit;
  int nodeType;

  // create declarations for function entry and function exit
  function_entry = malloc(sizeof(declaration));
  function_exit = malloc(sizeof(declaration));
  if (function_entry == NULL || function_exit == NULL)
  {
    perror("Not enough memory");
    exit(1);
  }
  
  functionName =
    CAST(identifier_declarator,fd->fdeclarator->declarator)->cstring.data;
  // get the current working directory
  if (getcwd(cwd, sizeof(cwd) - 2) == NULL)
  {
    perror("getcwd()");
    exit(1);
  }
  sourceFileName = (fd->cfg_entry->location).filename;
  if ((strlen(cwd) + strlen(sourceFileName)) >= (sizeof(buf) - 2))
  {
    fprintf(stderr, "Directory name or file name too long");
    exit(1);
  }
  bzero(buf, sizeof(buf));
  if (sourceFileName[0] != '/') {
    strcpy(buf, cwd);
    strcat(buf, "/");
  }
  strcat(buf, sourceFileName);
  ast_output_token(MOPS_FB);
  ast_output_string(functionName);
  ast_output_string(buf);
  ast_output_token(MOPS_EOL);

  // print AST
  process_function_decl(fd, function_entry, function_exit);
  ast_output_token(MOPS_EOL);
  
  for (i = 0; i < nnodes; i++)
  {
    n = gnodes[i];
    if (n == fd->cfg_entry)
    {
      nodeType = MOPS_NN;
      if (hasEntry)
	fprintf(stderr, "process_cfg(): duplicate entry node in function %s\n",
		functionName);
      else
	hasEntry = TRUE;
    }
    else if (n == exitnode)
    {
      nodeType = MOPS_NX;
      if (hasExit)
	fprintf(stderr, "process_cfg(): duplicate exit node in function %s\n",
		functionName);
      else
        hasExit = TRUE;
    }
    else
    {
      nodeType = MOPS_N;
    }
    ast_output_token(nodeType);
    ast_output_int(getUniqueAddress(n, NULL));
    str = (n->location).filename;
    if (strcmp(str, sourceFileName) == 0)
    {
      ast_output_int((n->location).lineno);
    }
    else
    {
      if (str[0] == '/')
      {
	sprintf(buf, "%s:%ld", str, (n->location).lineno);
      }
      else
      {
	sprintf(buf, "%s/%s:%ld", cwd, str, (n->location).lineno);
      }
      ast_output_string(buf);
    }
    ast_output_token(MOPS_EOL);

    // detect dead functions and create final node for them.
    isDeadFunction = FALSE;
    if (is_function_call(n))
    {
      functionName = ((identifier)((function_call)n)->arg1)->cstring.data;
      for (j = 0; j < numDeadFunctions; j++)
	if (strcmp(functionName, deadFunctions[j]) == 0)
	{
	  isDeadFunction = TRUE;
	  break;
	}
    }
    if (!isDeadFunction)
    {
      for (e = n->edges_out; e; e = e->next_out)
      {
	if (n == fd->cfg_entry)
	{
	  n2 = (node)function_entry;
	}
	else if (e->to == exitnode)
	{
	  n2 = (node)function_exit;
	}
	else
	{
	  n2 = n;
	}
	ast_output_token(MOPS_E);
	ast_output_int(getUniqueAddress(n, NULL));
	ast_output_int(getUniqueAddress(e->to, NULL));
	ast_output_int(getUniqueAddress(n2, NULL));
	ast_output_token(MOPS_EOL);
      }
    }
    else
    {
      // Warning: Intentionally leak a byte of memory in order to
      // get a unique address
      if ((ptr = (char*)malloc(1)) == NULL)
      {
	fprintf(stderr, "Not enough memory\n");
	exit(1);
      }
      if (n == fd->cfg_entry)
      {
	n2 = (node)function_entry;
      }
      else
      {
	n2 = n;
      }
      ast_output_token(MOPS_N);
      ast_output_int(getUniqueAddress(ptr, NULL));
      str = (n->location).filename;
      if (strcmp(str, sourceFileName) == 0)
      {
	ast_output_int((n->location).lineno);
      }
      else
      {
	sprintf(buf, "%s:%ld", str, (n->location).lineno);
	ast_output_string(buf);
      }
      ast_output_token(MOPS_EOL);
      ast_output_token(MOPS_E);
      ast_output_int(getUniqueAddress(n, NULL));
      ast_output_int(getUniqueAddress(ptr, NULL));
      ast_output_int(getUniqueAddress(n2, NULL));
      ast_output_token(MOPS_EOL);
    }
  }

  if (!hasEntry || !hasExit)
  {
    /*
    fprintf(stderr, "process_cfg(): missing");
    if (!hasEntry)
      fprintf(stderr, " entry ");
    if (!hasExit)
      fprintf(stderr, " exit ");
    fprintf(stderr, "node in function %s\n", functionName);
    */
  }
  ast_output_token(MOPS_FE);
  // Do not free function_entry and function_exit so that each function_decl
  // has a unqiue function_entry and function_node node.
  // free(function_entry);
  // free(function_exit);
}

void rc_process(declaration the_program, FILE *outputFileHandle) deletes
{
  declaration d;
  int i;
  declarator decl;
  char *str;

  outputFile = outputFileHandle;
  // write magic string to identify CFG files
  ast_output_string(CFG_MAGIC_STRING);
  ast_output_token(MOPS_EOL);
  
  hashtableInit(&hashtable, INIT_TABLE_CAPACITY, INIT_LOAD_FACTOR);
  vectorInit(&variableDefinition, INIT_VECTOR_CAPACITY);
  vectorInit(&variableDeclaration, INIT_VECTOR_CAPACITY);
  vectorInit(&functionDefinition, INIT_VECTOR_CAPACITY);

  scan_declaration (d, the_program)
    if (is_function_decl (d))
    {
      function_decl fd = CAST(function_decl, d);
      //fix_parse_tree(parse_region, fd);
    }
  scan_declaration (d, the_program)
  {
    if (is_function_decl(d))
      {
	region work = newregion();
	function_decl fd = CAST(function_decl, d);
	build_cfg(work, fd);
	cfg_build_postorder(work, fd);
	// AST_set_parents(CAST(node, fd));

	// print CFG with pointers to AST
	process_cfg(fd);
	deleteregion(work);
        ast_output_token(MOPS_EOL);
      }
    else if (is_data_decl(d))
    {
      if (should_process_data_decl(CAST(data_decl, d)))
      {
        ast_output_token(MOPS_D);
	// ast_output_token(MOPS_EOL);
        process_data_decl(CAST(data_decl, d));
        ast_output_token(MOPS_EOL);
      }
    }
    else
    {
      //fprintf(stderr, "rc_process(): declaration type %x is unprocessed\n",
      //      (d->kind & 0x3ffff));
    }
  }

  hashtableFinalize(&hashtable);

  if (functionDefinition.size > 0)
  {
    ast_output_token(MOPS_LF);
    for (i = 0; i < functionDefinition.size; i++)
    {
      if ((decl = vectorGet(&functionDefinition, i)) == NULL)
      {
	fprintf(stderr, "Error: vectorGet() returns NULL");
	exit(1);
      }
      ast_output_cstring(CAST(identifier_declarator,
		       CAST(function_declarator, decl)->declarator)->cstring);
    }
    ast_output_token(MOPS_EOL);
  }
  if (variableDefinition.size > 0)
  {
    ast_output_token(MOPS_LC);
    for (i = 0; i < variableDefinition.size; i++)
    {
      if ((str = (char*)vectorGet(&variableDefinition, i)) == NULL)
      {
	fprintf(stderr, "Error: vectorGet() returns NULL");
	exit(1);
      }
      ast_output_string(str);
    }
    ast_output_token(MOPS_EOL);
  }
  if (variableDeclaration.size > 0)
  {
    ast_output_token(MOPS_LU);
    for (i = 0; i < variableDeclaration.size; i++)
    {
      if ((str = (char*)vectorGet(&variableDeclaration, i)) == NULL)
      {
	fprintf(stderr, "Error: vectorGet() returns NULL");
	exit(1);
      }
      ast_output_string(str);
    }
    ast_output_token(MOPS_EOL);
  }
  vectorFinalize(&functionDefinition);
  vectorFinalize(&variableDefinition);
  vectorFinalize(&variableDeclaration);
  ast_output_token(MOPS_EOF);
  ast_output_token(MOPS_EOL);
  fclose(outputFile);
  /*
  if (flag_parse_only)
    unparse(stdout, the_program);
  else
    exit(exec_cc1(the_program));
  */
}
