// $Id: ModelChecker.java,v 1.11 2004/07/25 00:52:47 hchen Exp $

import java.io.*;
import java.util.*;

/**
 * Model checking a PDA
 * <ol>
 * <li>Construct a P-automaton <tt>PA</tt> from the PDA <tt>P</tt>.</li>
 * <li>Compute post*(PA).</li>
 * <li>Construct a directed graph <tt>G</tt> from post*(PA).  Each vertex in G
 *     is a transition, whose source state is a state in the PDA P,
 *     in post*(PA).  Each edge (u,v) in G denotes that the transition u
 *     causes the transition v to be added in post*(PA).  A source vertex in
 *     G is a transition whose source state is an initial state in P.
 *     A sink vertex in G is a transition whose source state is a final state
 *     in P.</li> 
 * </ol>
 * Recover a class of error traces, i.e. all the error traces that share
 * a common prefix.  Note: each element in an error trace is a tuple
 * (state, program point).
 * <ol>
 * <li>Find a shortest path from a source vertex to a sink vertex in G.</li>
 * <li>On the path find the first vertex that always reaches a sink vertex.</li>
 * <li>Delete all the vertices that are reachable from the vertex found in the previous step.</li>
 * </ol>
 */
public class ModelChecker
{
  public ModelChecker()
  {
    fsaTransTemp = new FsaTransition();
    criticalEdgeTemp = new ExplicitEdge();
    sourceNodes = new HashSet();
    sinkNodes = new HashSet();
  }

  void init(Pda pda, HashSet criticalEdges)
  {
    Iterator iter;
    Vector nodes;
    TransitionNode node;
    
    composedPda = pda;
    this.criticalEdges = criticalEdges;

    // Create the pAutomaton from composedPda
    pAutomaton = createPAutomaton(pda);

    // Compute the sourceNodes and sinkNodes of the transition graph
    // derived from the pAutomaton.  sourceNodes have been computed in
    // createPAutomaton.  Here we compute the sinkNodes
    iter = pAutomaton.getInitialStates().iterator();
    while (iter.hasNext())
    {
      fsaTransTemp.state0 = ((IntHashBase.Entry)iter.next()).getKey();
      // Assume that the first PolyHashtable in pAutomaton indexes on state0
      pAutomaton.getTransitions(fsaTransTemp, pAutomaton.getState0Index(),
				sinkNodes);
    }

    // Compute the reachability of each node in the transition graph
    // i.e. whether a node inevitably/never/maybe reaches a sink node?
    // reachability();

    // Compute the shortest path from the source nodes.
    // Assume that node.distance = -1 for all nodes
    nodes = new Vector();
    iter = getSourceNodes().iterator();
    while (iter.hasNext())
    {
      node = (TransitionNode)iter.next();
      node.distance = 0;
      node.parent = null;
      nodes.add(node);
    }
    Util.dijkstra(nodes, true, null);
  }

  public void clear()
  {
    // composedPda belongs to the caller
    composedPda = null;
    if (pAutomaton != null)
    {
      pAutomaton.clear();
      pAutomaton = null;
    }
    sourceNodes.clear();
    sinkNodes.clear();
    // the caller may want to keep criticalEdges
    criticalEdges = null;
  }
  
  // Find all unsafe transitions in the PAutomaton
  
  // Find one unsafe path.  The caller must clear path
  // Invariant: if (node.distance >= 0) then
  // node.parent != null and node is reachable from a source node whose
  // distance is >= 0
  // if (node.parent != null)
  // then node is a source node or node.distance >= 0
  public boolean findPath(Vector path)
  {
    Vector inEdges, outEdges, nodeQueue, unreachableNodes, stack, list;
    HashSet sinkNodes;
    Hashtable visitedEdges;
    TransitionNode node, node2, lastNode, parentNode;
    TransitionEdge edge, parent, oldEdge;
    Iterator iter;
    ExplicitEdge criticalEdge;
    int i, j, distance, minVisit, minIndex, value = -1;
    Integer enterFunction, exitFunction, functionCall;
    Object object;
    final int MAX_VISIT = 10;

    // Find the sink node that is the closest to the source
    sinkNodes = getSinkNodes();
    iter = sinkNodes.iterator();
    if (!iter.hasNext())
      // no sinkNode
    {
      return false;
    }
    
    // Find the sink node that is closed to a source node
    node = (TransitionNode)iter.next();
    if (node.getDistance() < 0)
	Util.die(Util.INTERNAL, "sinkNodes contains unreachable nodes");
    /*
    if (node.getDistance() >= 0)
      Util.stderr.println(composedPda.getStateLabelString(node.state0) + " " +
		       composedPda.getStackLabel((Integer)node.input) + " " +
		       node.getDistance());
    */
    while (iter.hasNext())
    {
      node2 = (TransitionNode)iter.next();
      if (node2.getDistance() < 0)
	Util.die(Util.INTERNAL, "sinkNodes contains unreachable nodes");
      if (node2.compareTo(node) < 0)
        node = node2;
      /*
      if (node2.getDistance() >= 0)
	Util.stderr.println(composedPda.getStateLabelString(node2.state0) + " " +
			 composedPda.getStackLabel((Integer)node2.input) + " " +
			 node2.getDistance());
      */
    }
    if (node.getDistance() < 0)
    {
      // no sinkNode is reachable
      Util.die(Util.INTERNAL, "No sink node is reachable");
      return false;
    }
    
    /*
    // Recover a path that ends in node
    // On the path, the srcNode of criticalEdge is the last node whose
    // reachability is not ALWAYS.
    criticalEdge = null;
    for (edge = node.getParent(); edge != null; edge = node.getParent())
    {
      node = edge.getSrcNode();
      if (//node.getReachability() != TransitionNode.ALWAYS
	  
	  && criticalEdge == null)
      {
	criticalEdge = edge;
      }
      path.add(edge);
    }

    // Remove an incoming edge to node
    if (criticalEdge != null)
    {
      node = criticalEdge.getDstNode();
      criticalEdge.remove();
    }
    */

    // Recover the shortest path
    stack = new Vector();
    list = new Vector();
    lastNode = node;
    edge = (TransitionEdge)node.getParent();
    enterFunction = new Integer(ENTER_FUNCTION);
    exitFunction = new Integer(EXIT_FUNCTION);
    visitedEdges = new Hashtable();
    while (edge != null)
    {
      // Util.stderr.print("(" + visitedEdges.size() + " ");
      // Util.stderr.print(visitedEdges.size() + ") ");
      parentNode = (TransitionNode)edge.getSrcNode();
      functionCall = null;
      if (node.state1 != parentNode.state1)
	// node and parentNode aren't in the same call frame
      {
	fsaTransTemp.state0 = parentNode.state1;
	pAutomaton.getTransitions(fsaTransTemp, pAutomaton.getState0Index(),
				  list);
	for (i = 0; i < list.size(); i++)
	{
	  node2 = (TransitionNode)list.get(i);
	  if (node2.state1 == node.state1)
	    // node is in the caller and parentNode is in the callee
	  {
	    functionCall = exitFunction;
	    stack.add(node);
	    break;
	  }
	}
	list.clear();
	if (functionCall == null)
	{
	  if (stack.size() > 0)
	  {
	    node2 = (TransitionNode)stack.remove(stack.size() - 1);
	    if (parentNode.state1 != node2.state1)
	    {
	      Util.warn(Util.INFO, Util.INTERNAL,
			"searching for another parentNode ");
	      inEdges = node.getInEdges();
	      for (i = 0; i < inEdges.size(); i++)
	      {
		if (((TransitionNode)((TransitionEdge)inEdges.get(i)).
		     getSrcNode()).state1 == node2.state1)
		  break;
	      }
	      if (i >= inEdges.size())
	      {
		Util.die(Util.INTERNAL,
			 "Cannot find the function entry corresponding to the function exit");
	      }
	      edge = (TransitionEdge)inEdges.get(i);
	      parentNode = (TransitionNode)edge.getSrcNode();
	    }
	  }
	  // verify that node is in the callee and parentNode in the caller
	  fsaTransTemp.state0 = node.state1;
	  pAutomaton.getTransitions(fsaTransTemp, pAutomaton.getState0Index(),
				    list);
	  for (i = 0; i < list.size(); i++)
	  {
	    node2 = (TransitionNode)list.get(i);
	    if (node2.state1 == parentNode.state1)
	    {
	      functionCall = enterFunction;
	      break;
	    }
	  }
	  list.clear();
	  if (functionCall == null)
	  {
	    Util.die(Util.INTERNAL, "Two adjacent transitions in the P-Automaton are not in the same call frame or in adjacent caller and callee");
	  }
	}
      }
      if (functionCall != null)
      {
	path.add(functionCall);
	//Util.stderr.println(functionCall == enterFunction ?
	// "enterFunction" : "exitFunction");
      }
      path.add(edge);
      if (visitedEdges.containsKey(edge))
      {
	visitedEdges.put(edge,
	  new Integer(((Integer)visitedEdges.get(edge)).intValue() + 1));
      }
      else
      {
	visitedEdges.put(edge, new Integer(1));
      }
      node = parentNode;
      edge = (TransitionEdge)node.getParent();
      if (edge != null && visitedEdges.containsKey(edge))
      {
	minVisit = MAX_VISIT;
	minIndex = -1;
	inEdges = node.getInEdges();
	for (i = 0; i < inEdges.size(); i++)
	{
	  edge = (TransitionEdge)inEdges.get(i);
	  if (!visitedEdges.containsKey(edge))
	  {
	    break;
	  }
	  else if ((value = ((Integer)visitedEdges.get(edge)).intValue())
		   < minVisit)
	  {
	    minVisit = value;
	    minIndex = i;
	  }
	}
	if (i >= inEdges.size())
	{
	  if (minIndex == -1)
	  {
	    Util.warn(Util.ERROR, Util.INTERNAL, "Fail to backtrack an error trace");
	    stack.clear();
	    visitedEdges.clear();
	    path.clear();
 	    return false;
	  }
	  else
	  {
	    edge = (TransitionEdge)inEdges.get(minIndex);
	    // Util.stderr.println("visit=" + value);
	  }
	}
      }
    }
    
    stack.clear();
    visitedEdges.clear();
    /*
    if (edge != null)
    {
      Util.warn(Util.ERROR, Util.INTERNAL, "Fail to backtrack a complete path because shortest path for FSA does not work for PDA");
      // Cfg.transformPath requires that a path must begin with the entrance
      // in a function
      for (i = path.size() - 1; i >= 0; i--)
      {
	object = path.remove(i);
	if (object instanceof Integer && (Integer)object == enterFunction)
	  break;
      }
    }
    */
    // Remove the last edge on the path 
    edge = (TransitionEdge)lastNode.getParent();
    if (edge != null)
    {
      criticalEdge = new ExplicitEdge((TransitionNode)edge.getSrcNode(),
				      lastNode);
      /*
      Util.stderr.println("\n" +
	  criticalEdge.srcNode.state0 + ", " +
	  criticalEdge.srcNode.input + ", " +
	  criticalEdge.srcNode.state1 + " -> " +
	  criticalEdge.dstNode.state0 + ", " +
	  criticalEdge.dstNode.input + ", " +
	  criticalEdge.dstNode.state1 + "\n" +
	  composedPda.getStateLabelString(criticalEdge.srcNode.state0) + " " +
	  composedPda.getStackLabel(((Integer)criticalEdge.srcNode.input).intValue()) +
          " -> " +
	  composedPda.getStateLabelString(criticalEdge.dstNode.state0) + " " +
			  composedPda.getStackLabel(((Integer)criticalEdge.dstNode.input).intValue()));
      */
      if (criticalEdges.contains(criticalEdge))
      {
	Util.die(Util.INTERNAL, "A removed critical edge reappears");
      }
      criticalEdges.add(criticalEdge);
      edge.remove();
    }

    // recompute the distances of all successors of node
    lastNode.parent = null;
    lastNode.distance = -1;
    // collect all the nodes whose transitive parent is the removed node
    nodeQueue = new Vector();
    nodeQueue.add(lastNode);
    for (i = 0; i < nodeQueue.size(); i++)
    {
      node = (TransitionNode)nodeQueue.get(i);
      outEdges = node.getOutEdges();
      for (j = 0; j < outEdges.size(); j++)
      {
	edge = (TransitionEdge)outEdges.get(j);
	node2 = (TransitionNode)edge.getDstNode();
	if (node2.parent == edge)
	{
	  node2.parent = null;
	  node2.distance = -1;
	  nodeQueue.add(node2);
	  // Need not put node2 into a HashSet because node2.parent==null
	  // so it cannot be reinserted into nodeQueue.
	}
      }
    }

    // compute a tentative distance for every node in nodeQueue
    for (i = 0; i < nodeQueue.size(); i++)
    {
      node = (TransitionNode)nodeQueue.get(i);
      distance = node.distance;
      parent = (TransitionEdge)node.parent;
      inEdges = node.getInEdges();
      for (j = 0; j < inEdges.size(); j++)
      {
	edge = (TransitionEdge)inEdges.get(j);
	node2 = (TransitionNode)edge.getSrcNode();
	if (node2.distance >= 0 && node.compareTo(node2.distance + 1) > 0)
	{
	  distance = node2.distance + 1;
	  parent = edge;
	}
      }
      if (distance != node.distance)
      {
	node.distance = distance;
	node.parent = parent;
      }
    }

    // Compute shortest distance for every node in nodeQueue
    unreachableNodes = new Vector();
    Util.dijkstra(nodeQueue, true, unreachableNodes);

    // Remove all unreachable nodes from sinkNodes
    for (i = 0; i < unreachableNodes.size(); i++)
      sinkNodes.remove(unreachableNodes.get(i));

    return true;
  }

  /**
   * Write the path
   */
  public void writePath(Vector path, Output writer) throws IOException
  {
    TransitionEdge edge;
    TransitionNode node;
    Object object;
    int i, level, functionCall;

    if (path.size() == 0)
      return;
    object = path.get(0);
    if (object instanceof Integer)
    {
      // The first element in "path" is ENTER_FUNCTION or EXIT_FUNCTION
      node = (TransitionNode)((TransitionEdge)path.get(1)).getDstNode();
    }
    else
    {
      // The first element in "path" is a TransitionEdge object
      node = (TransitionNode)((TransitionEdge)object).getDstNode();
    }
    writer.writeString(composedPda.getExplicitStateLabel(node.state0));
    writer.writeToken(Constants.EOL);
    level = 1;
    edge = (TransitionEdge)path.get(path.size() - 1);
    node = (TransitionNode)edge.getSrcNode();
    writer.writeString(composedPda.getStateLabelString(node.state0));
    writer.writeString(composedPda.getStackLabel(((Integer)node.input).intValue()));
    writer.writeInt(level);
    writer.writeToken(Constants.EOL);
    for (i = path.size() - 1; i >= 0; i--)
    {
      edge = (TransitionEdge)path.get(i);
      node = (TransitionNode)edge.getDstNode();
      functionCall = SAME_FUNCTION;
      if (i > 0)
      {
	object = path.get(i - 1);
	if (object instanceof Integer)
	{
	  functionCall = ((Integer)object).intValue();
	  switch(functionCall)
	  {
	    case ENTER_FUNCTION:
	      level++;
	      break;

	    case EXIT_FUNCTION:
	      level--;
	      break;

	    default:
	      Util.die(Util.INTERNAL, "Unknown enumeration value");
	  }
	  i--;
	}
      }
      if (functionCall == EXIT_FUNCTION)
      {
	writer.writeInt(0);
      }
      else
      {
	writer.writeInt(edge.getLabel().getAddress());
      }
      writer.writeToken(Constants.EOL);
      writer.writeString(composedPda.getStateLabelString(node.state0));
      writer.writeString(composedPda.getStackLabel(((Integer)node.input).intValue()));
      writer.writeInt(level);
      writer.writeToken(Constants.EOL);
    }
    writer.flush();
  }
  
  /**
   * Create a P-automaton to describe the stack configuration of the PDA
   * States of P-automaton are states of PDA
   * Inputs of P-automaton are stack symbols of PDA
   * See paper for details
   * Note: postStar() works only if:
   * - initialStackSymbols contains only one element which is also the entry
   *   point of a function
   * - Each function entry point has no incoming edges
   */
  Fsa createPAutomaton(Pda pda)
  {
    BitSet[] keyFieldsArray;
    Fsa fsa;
    Iterator iter;
    int state0, state1, finalState, i;
    int state;
    String label;
    StateLabel stateLabel;
    TransitionNode transition;
    Vector stackSymbols;

    stackSymbols = pda.getInitialStackSymbols();
    if (stackSymbols.size() <= 0)
      Util.die(Util.INTERNAL, "initialStackSymbols.size() <= 0");

    keyFieldsArray = new BitSet[2];
    keyFieldsArray[0] = new BitSet();
    keyFieldsArray[0].set(0);
    keyFieldsArray[1] = new BitSet();
    keyFieldsArray[1].set(0);
    keyFieldsArray[1].set(1);
    // fsa is the P-Automaton
    fsa = new Fsa(keyFieldsArray);
    
    // Shallow copy stateLabels
    for (i = 0; i < pda.getStateLabels().size(); i++)
    {
      stateLabel = (StateLabel)pda.getStateLabels().get(i);
      fsa.stateLabels.add(new StateLabel(stateLabel.size, stateLabel.labels));
    }

    // Create initial states. The set of initial states of the
    // P-automaton is the set of final states of the PDA
    iter = pda.getFinalStates().iterator();
    while (iter.hasNext())
    {
      state = ((IntHashBase.Entry)iter.next()).getKey();
      fsa.addInitialState(state);
    }

    // Create one final state for the P-Automaton
    finalState = fsa.getNewState();
    fsa.addFinalState(finalState);

    // Create a path from each initial state to the final state with all
    // initial stack symbols on.  The first stack symbol is adjacent to the
    // initial states.  The last stack symbol is adjacent to the final state.
    iter = pda.getInitialStates().iterator();
    while (iter.hasNext())
    {
      state1 = finalState;
      for (i = stackSymbols.size() - 1; i > 0; i--)
      {
	state0 = fsa.getNewState();
	fsa.addState(state0);
	transition = new TransitionNode(state0, (Integer)stackSymbols.get(i),
					state1);
	fsa.addTransition(transition);
	state1 = state0;
      }
      state = ((IntHashBase.Entry)iter.next()).getKey();
      fsa.addState(state);
      transition = new TransitionNode(state, (Integer)stackSymbols.get(i),
				      state1);
      fsa.addTransition(transition);
      sourceNodes.add(transition);
    }

    // poststar.  See paper for details
    postStar(fsa, pda);

    // Transforms input symbols of P-automaton from integers to strings
    // Remember input symbols of P-automaton are stack symbols of PDA
    /*
    iter = fsa.iterator();
    while (iter.hasNext())
    {
      fsaTransition = (FsaTransition)iter.next();
      fsaTransition.input = getStackLabel((Integer)fsaTransition.input);
    }
    */
    
    return fsa;
  }

  /**
   * Add an FsaTransitionTrace and its parent to rel or trans if they
   * do not already exist.  See paper for explanation of rel and trans
   *
   * @param isToTrans If true, add new FsaTransitionTrace to trans.
   *   If false, add it to rel
   */
  protected void addTransParent(Fsa rel, Fsa trans, int state0,
    Object input, int state1, TransitionNode parent, Ast ast,
    boolean isToTrans)
  {
    TransitionNode t;
    TransitionEdge edge;
    Vector outEdges;
    int i;

    if (parent != null)
    {
      criticalEdgeTemp.srcNode.state0 = parent.state0;
      criticalEdgeTemp.srcNode.input = parent.input;
      criticalEdgeTemp.srcNode.state1 = parent.state1;
      criticalEdgeTemp.dstNode.state0 = state0;
      criticalEdgeTemp.dstNode.input = input;
      criticalEdgeTemp.dstNode.state1 = state1;
      if (criticalEdges.contains(criticalEdgeTemp))
      {
	return;
      }
    }
    fsaTransTemp.state0 = state0;
    fsaTransTemp.input = input;
    fsaTransTemp.state1 = state1;
    if ((t = (TransitionNode)rel.getTransition(fsaTransTemp)) == null &&
	(t = (TransitionNode)trans.getTransition(fsaTransTemp)) == null)
    {
      t = new TransitionNode(state0, input, state1);
      if (isToTrans)
      {
	trans.addTransition(t);
      }
      else
      {
	rel.addTransition(t);
      }
    }
    if (parent != null)
    {
      // Detect if the edge to be added already exists
      outEdges = parent.getOutEdges();
      for (i = 0; i < outEdges.size(); i++)
      {
	edge = (TransitionEdge)outEdges.get(i);
	if (edge.getDstNode() == t)
	  // Only allow one edge between any pair of nodes
	  // It's ok to disallow inserting a new edge that has the same
	  // src and dst nodes but a different AST as an existing edge
	{
	  return;
	}
      }
      edge = new TransitionEdge(ast, parent, t);
    }
  }

  /**
   * PostStar() operation.  See paper for details
   * Indices: rel is indexed by (state0) and (state0, input), and
   * trans is indexed by (state0, input).
   */
  protected void postStar(Fsa rel, Pda pda)
  {
    Fsa trans;
    Hashtable newStates; // q_r in paper
    IntHashtable eps, // eps() in paper
      states;
    TransitionNode fsaTransition, t;
    PdaTransition pdaTransition, pdaTransPattern;
    Iterator iter;
    int state, state2;
    int pdaHashIndex;
    IntHashtable.Entry entry;
    Ast ast;
    BitSet keyFields;
    Vector allTransitions, allTransitions2;
    int i, j;

    // trans is indexed by (state0, input)
    keyFields = new BitSet();
    keyFields.set(0);
    keyFields.set(1);
    trans = new Fsa(new BitSet[] { keyFields });
    // make FsaTransition happy
    trans.addInitialDimension();

    // compute fsaHashIndex and pdaHashIndex
    keyFields.clear();
    keyFields.set(0);
    keyFields.set(3);
    pdaHashIndex = pda.getHashtableIndex(keyFields);
    /*
    keyFields.clear();
    keyFields.set(0);
    fsaHashIndex = rel.getHashtableIndex(keyFields);
    */
    
    // compute trans
    allTransitions = new Vector();
    allTransitions2 = new Vector();
    rel.getAllTransitions(allTransitions);
    for (i = 0; i < allTransitions.size(); i++)
    {
      fsaTransition = (TransitionNode)allTransitions.get(i);
      if (pda.getStates().contains(fsaTransition.state0))
      {
	trans.addTransition(fsaTransition);
	rel.removeTransitionObject(fsaTransition);
      }
    }

    // compute eps
    eps = new IntHashtable(64);
    newStates = new Hashtable();
    allTransitions.clear();
    pda.getAllTransitions(allTransitions);
    for (i = 0; i < allTransitions.size(); i++)
    {
      pdaTransition = (PdaTransition)allTransitions.get(i);
      if (pdaTransition.stack2 != PdaTransition.EPSILON)
      {
	state = rel.getNewState();
	rel.addState(state);
	newStates.put(pdaTransition, new Integer(state));
	eps.put(state, new IntHashtable(16));
      }
    }

    pdaTransPattern = new PdaTransition();
    while ((t = (TransitionNode)trans.popTransition()) != null)
    {
      if (rel.containsTransitionObject(t))
      {
	Util.die(Util.INTERNAL, "rel & trans intersect: " + t.state0 + " " + t.input + " " + t.state1);
      }
      rel.addTransition(t);
      pdaTransPattern.state0 = t.state0;
      pdaTransPattern.stack0 = ((Integer)t.input).intValue();
      allTransitions.clear();
      pda.getTransitions(pdaTransPattern, pdaHashIndex, allTransitions);

      for (i = 0; i < allTransitions.size(); i++)
      {
	pdaTransition = (PdaTransition)allTransitions.get(i);
	if (pdaTransition.stack1 == PdaTransition.EPSILON)
	{
	  state = t.state1;
	  states = (IntHashtable)eps.get(state);
	  if (states == null)
	  {
	    if (!rel.getFinalStates().contains(state))
	      Util.die(Util.INTERNAL, "transition into a non-final and non-intermediate state.");
	    // Now, state must be a final state of rel
	    continue;
	  }
	  state2 = pdaTransition.state1;
	  if (!states.containsKey(state2))
	  {
	    states.put(state2, t);
	    if (rel.getFinalStates().contains(t.state1))
	      rel.addFinalState(pdaTransition.state1);
	    fsaTransTemp.state0 = t.state1;
	    allTransitions2.clear();
	    rel.getTransitions(fsaTransTemp, rel.getState0Index(),
			       allTransitions2);
	    for (j = 0; j < allTransitions2.size(); j++)
	    {
	      fsaTransition = (TransitionNode)allTransitions2.get(j);
	      addTransParent(rel, trans, pdaTransition.state1,
			     fsaTransition.input, fsaTransition.state1, t,
			     (Ast)pdaTransition.input, true);
	    }
	  }
	  else
	  {
	    fsaTransition = (TransitionNode)states.get(state2);
	    Util.warn(Util.INFO, Util.FILE_FORMAT,
	      "Multiple exit nodes exit in the same function: "
	      + pda.getStackLabel(((Integer)fsaTransition.input).intValue())
	      + " and "
	      + pda.getStackLabel(((Integer)t.input).intValue()));
	  }
	}
	else if (pdaTransition.stack2 == PdaTransition.EPSILON)
	{
	  addTransParent(rel, trans, pdaTransition.state1,
			 new Integer(pdaTransition.stack1), t.state1, t,
			 (Ast)pdaTransition.input, true);
	}
	else
	{
	  state = ((Integer)newStates.get(pdaTransition)).intValue();
	  if (state != t.state1)
	  {
	    addTransParent(rel, trans, pdaTransition.state1,
		           new Integer(pdaTransition.stack1),
		           state, t, (Ast)pdaTransition.input, true);
	    addTransParent(rel, trans, state,
			   new Integer(pdaTransition.stack2), t.state1,
			   null, null, false);
	    states = (IntHashtable)eps.get(state);
	    iter = states.iterator();
	    while (iter.hasNext())
	    {
	      entry = (IntHashtable.Entry)iter.next();
	      addTransParent(rel, trans, entry.getKey(),
			     new Integer(pdaTransition.stack2), t.state1,
			     (TransitionNode)entry.getValue(),
			     (Ast)pdaTransition.input, true);
	    }
	  }
	  else
	  {
	    Util.warn(Util.INFO, Util.INTERNAL, 
              "Recursive function call ignored: " +
	      pda.getStackLabel(pdaTransition.stack0) + " -> " +
	      pda.getStackLabel(pdaTransition.stack1) + " " +
	      pda.getStackLabel(pdaTransition.stack2));
	  }
	}
      }
    }
    
    // clean up
    trans.clear();
    newStates.clear();
    eps.clear();
    allTransitions.clear();
    allTransitions2.clear();
  }

  // Assume that all TransitionNode.distance == -1 initially.
  protected final HashSet getSourceNodes()
  {
    return sourceNodes;
  }

  protected final HashSet getSinkNodes()
  {
    return sinkNodes;
  }

  // The composed PDA, indexed by state0 and input
  Pda composedPda;

  // The P-Automaton created from the composed PDA
  Fsa pAutomaton;

  // The source nodes and sink nodes of the transition graph
  HashSet sourceNodes, sinkNodes;

  // Contains ExplicitEdge objects that are removed in findPath().
  // This is passed in from init(), but we add new elements to it in findPath().
  // The caller passes criticalEdges to another call to init(), 
  HashSet criticalEdges;

  // For temporary use, avoiding creating lots of temporary objects
  FsaTransition fsaTransTemp;

  // For temporary use, avoiding creating lots of temporary objects
  ExplicitEdge criticalEdgeTemp;

  static final int SAME_FUNCTION = 0, ENTER_FUNCTION = 1, EXIT_FUNCTION = 2;

  static int count;

  static
  {
    count = 0;
  }
}

  /*
  // Compute the reachability of each FsaTransitionNode
  protected void reachability()
  {
    int i;
    Vector nodeQueue, inEdges;
    HashSet neverTrans;
    Iterator iter;
    TransitionNode node, node2;

    neverTrans = new HashSet();
    pAutomaton.getAllTransitions(neverTrans);
    // Ensure that the reachability of each node is NEVER
    iter = neverTrans.iterator();
    while (iter.hasNext())
    {
      node = (TransitionNode)iter.next();
      if (node.reachability != TransitionNode.NEVER)
      {
	Util.warn(Util.INFO, Util.INTERNAL, "The initial reachability of a TransitionNode is not set to NEVER");
	node.reachability = TransitionNode.NEVER;
      }
    }
    
    nodeQueue = new Vector();
    // From neverTrans remove all the nodes that may reach a sink node
    iter = getSinkNodes().iterator();
    while (iter.hasNext())
    {
      node = (TransitionNode)iter.next();
      node.reachability = TransitionNode.ALWAYS;
      nodeQueue.add(node);
      neverTrans.remove(node);
    }
    while (nodeQueue.size() > 0)
    {
      node = (TransitionNode)nodeQueue.remove(nodeQueue.size() - 1);
      inEdges = node.getInEdges();
      for (i = 0; i < inEdges.size(); i++)
      {
	node2 = ((TransitionEdge)inEdges.get(i)).getSrcNode();
	if (node2.reachability != TransitionNode.ALWAYS)
	{
	  neverTrans.remove(node2);
	  nodeQueue.add(node2);
	  node2.reachability = TransitionNode.ALWAYS;
	}
      }
    }
    
    // Now neverTrans contains all the nodes that must not reach a
    // sink node.  Mark all the nodes that have been marked ALWAYS but
    // may reach a neverTrans MAYBE
    iter = neverTrans.iterator();
    while (iter.hasNext())
    {
      nodeQueue.add(iter.next());
    }
    while (nodeQueue.size() > 0)
    {
      node = (TransitionNode)nodeQueue.remove(nodeQueue.size() - 1);
      inEdges = node.getInEdges();
      for (i = 0; i < inEdges.size(); i++)
      {
	node2 = ((TransitionEdge)inEdges.get(i)).getSrcNode();
	if (node2.reachability != TransitionNode.NEVER)
	{
	  nodeQueue.add(node2);
	  node.reachability = TransitionNode.MAYBE;
	}
      }
    }

    neverTrans.clear();
  }
  */
