/* ====Zernike.h================================================================================
 *
 * Author: Patrice Koehl, August 2020
 * Department of Computer Science
 * University of California, Davis
 *
 * Computes rotation invariant Zernike moment for a 2D image
 *
 *
 * See:
 	C.-W. Chong, P. Raveendran, R. Mukudan "A comparative analysis of algorithms for 
	fast computation of Zernike moments", Pattern Recognition, 36, 731-742 (2003)

 * ============================================================================================ */

#ifndef _ZERNIKE_H_
#define _ZERNIKE_H_

/* ============================================================================================
   Includes
 * ============================================================================================ */

/* ============================================================================================
   The class
 * ============================================================================================ */

  class Zernike {
	public:
		// constructs table
		void compute_descriptors(int N, int Nx, int Ny, double *img, std::vector<double>& moments);

	private:

		int itype = 0; // only core of image (0), or full image scaled inside circle

		void compute_Rnp(int N, double rho, std::vector<double>& Rnp);
  };

/* ============================================================================================
   Construct the Zernike moments
 * ============================================================================================ */

  void Zernike::compute_descriptors(int N, int Nx, int Ny, double *img, std::vector<double>& moments)
  {
	int ndesc = 0;
	for(int i = 0; i <= N; i++) ndesc += i + 1;
	for(int i = 0; i < ndesc; i++) moments[i] = 0;

	std::vector<double> Znp;
	Znp.resize(2*ndesc);
	for(int i = 0; i < 2*ndesc; i++) Znp[i] = 0;

	std::vector<double> Rnp;
	int nR=0;
	for(int i = 0; i <= N; i++) nR += i/2 + 1;
	Rnp.resize(nR);

	double fac, c1x, c2x, c1y, c2y;
	if(itype==0) {
		fac = 1./((Nx-1)*(Ny-1));
		c1x = 2.0/(Nx-1);
		c2x = -1.;
		c1y = 2.0/(Ny-1);
		c2y = -1.;
	} else{
		fac = 2./(M_PI*(Nx-1)*(Ny-1));
		c1x = std::sqrt(2.0)/(Nx-1);
		c2x = -1./std::sqrt(2.0);
		c1y = std::sqrt(2.0)/(Ny-1);
		c2y = -1./std::sqrt(2.0);
	}

	double x, y, rho, ang, val;
	int ipos, iposR;

	for(int j = 0; j < Ny; j++) 
	{
		for(int i = 0; i < Nx; i++) 
		{
			x = c1x*i + c2x;
			y = c1y*j + c2y;

			rho = std::sqrt(x*x+y*y);
			ang = std::atan2(y,x);


			if(0< rho && rho <= 1.) {
				compute_Rnp(N, rho, Rnp);
				ipos = 0;
				iposR = 0;
				for (int n = 0; n <= N; n++) {
					for(int m = -n; m <= n; m++) {
						int p = std::abs(m);
						if((n-p)%2==0) {
							val = (n+1)*img[i+j*Nx]*Rnp[iposR+p/2];
							Znp[2*ipos] += val*std::cos(m*ang);
							Znp[2*ipos+1] -= val*std::sin(m*ang);
							ipos++;
						}
					}
					iposR += n/2 + 1;
				}
			}
		}
	}

	for(int i = 0; i < ndesc; i++) {
		x = Znp[2*i];
		y = Znp[2*i+1];
		rho = std::sqrt(x*x+y*y);
		moments[i] = fac*rho;
	}

  }

/* ============================================================================================
 * ============================================================================================ */

  void Zernike::compute_Rnp(int N, double rho, std::vector<double>& Rnp)
  {
	int nv, nval = 0;
	int *idx = new int[N+1];
	for(int i = 0; i <= N; i++) {
		idx[i] = nval;
		nval += i/2 + 1;
	}
	for(int i = 0; i < nval; i++) Rnp[i] = 0.;

	double A, B, K1, K2, K3, K4, rho2, rho_n;

	rho2 = rho*rho;
	rho_n = 1.0;

	int ipos = 0;
	int m;
	for(int n = 0; n <=N; n++)
	{
		nv = n/2 + 1;
		if(nv==1) {
			Rnp[ipos] = rho_n;
		} else if(nv==2) {
			Rnp[ipos] = n*rho_n - (n-1)*rho_n/rho2;
			Rnp[ipos+1] = rho_n;
		} else {
			m = 0;
			if(n%2!=0) m = 1;
			for (int p = 0; p <nv-2; p++) {
				K1 = (n+m)*(n-m)*(n-2);
				K2 = 4*n*(n-1)*(n-2);
				K3 = -2*m*m*(n-1) - 2*n*(n-1)*(n-2);
				K4 = - n*(n+m-2)*(n-m+2);
				A = (K2*rho2+K3)/K1;
				B = K4/K1;
				Rnp[ipos+p] = A*Rnp[idx[n-2]+p] + B*Rnp[idx[n-4]+p];
				m+=2;
			}
			Rnp[ipos+nv-2] = n*rho_n - (n-1)*rho_n/rho2;
			Rnp[ipos+nv-1] = rho_n;
		}
		rho_n *= rho;
		ipos += nv;
	}
  }

#endif
