#pragma once

#include <set>
#include <fstream>
#include <sstream>
#include <unordered_map>
#include <queue>
#include <iomanip>

#include "MeshData.h"
#include "Mesh.h"
#include "Soup.h"

class MeshIO {
public:
	// reads model from obj file
	static bool readOBJ(const std::string& fileName, Mesh& model, int flag_norm, Vector& cm, double *scale,
					 std::string& error);

	// reads model from off file
	static bool readOFF(const std::string& fileName, Mesh& model, int flag_norm, Vector& cm, double *scale,
					 std::string& error);

	// writes data to off file
	static bool writeOFF(const std::string& fileName, Mesh& model);

	// writes data to color off file
	static bool writeCOFF(const std::string& fileName, Mesh& model, int *color);

	// builds a halfedge mesh
	static bool buildMesh( const Soup& soup, Mesh& mesh, std::string& error);

	// centers model around origin and records radius
	static void normalize(Mesh& model, Vector &cm, double *scale);

private:
	// preallocates mesh elements
	static void preallocateElements(const Soup& soup, Mesh& mesh);

	// assigns indices to mesh elements
	static void indexElements(Mesh& mesh);

	// checks if mesh has isolated vertices
	static bool hasIsolatedVertices(const Mesh& mesh);

	// checks if mesh has non-manifold vertices
	static bool hasNonManifoldVertices(const Mesh& mesh);

	// reads data from obj file
	static bool readOBJ(std::istringstream& in, Mesh& model,
					 std::string& error);

	// reads data from off file
	static bool readOFF(std::istringstream& in, Mesh& model,
					 std::string& error);

	// writes data to off file
	static void writeOFF(std::ofstream& out, Mesh& model);

	// writes data to off file
	static void writeCOFF(std::ofstream& out, Mesh& model, int *color);
};

void MeshIO::preallocateElements(const Soup& soup, Mesh& mesh)
{
	// clear arrays
	mesh.vertices.clear();
	mesh.edges.clear();
	mesh.faces.clear();
	mesh.halfEdges.clear();
	mesh.boundaries.clear();

	// reserve space
	int nVertices = (int)soup.positions.size();
	int nEdges = soup.table.getSize();
	int nFaces = (int)soup.indices.size()/3;
	int nHalfedges = 2*nEdges;

	mesh.vertices.reserve(2*nVertices);
	mesh.edges.reserve(2*nEdges);
	mesh.faces.reserve(2*nFaces);
	mesh.halfEdges.reserve(2*nHalfedges);
}

void MeshIO::indexElements(Mesh& mesh)
{
	int index = 0;
	for (VertexIter v = mesh.vertices.begin(); v != mesh.vertices.end(); v++) {
		v->index = index++;
		v->indexN = -1;
	}

	index = 0;
	for (EdgeIter e = mesh.edges.begin(); e != mesh.edges.end(); e++) {
		e->index = index++;
	}

	index = 0;
	for (FaceIter f = mesh.faces.begin(); f != mesh.faces.end(); f++) {
		f->index = index++;
	}

	index = 0;
	for (HalfEdgeIter h = mesh.halfEdges.begin(); h != mesh.halfEdges.end(); h++) {
		h->index = index++;
	}

	index = 0;
	for (BoundaryIter b = mesh.boundaries.begin(); b != mesh.boundaries.end(); b++) {
		b->index = index++;
	}
}

bool MeshIO::hasIsolatedVertices(const Mesh& mesh)
{
	for (VertexCIter v = mesh.vertices.begin(); v != mesh.vertices.end(); v++) {
		if (v->he == mesh.halfEdges.end()) {
			return true;
		}
	}

	return false;
}

bool MeshIO::hasNonManifoldVertices(const Mesh& mesh)
{
	VertexData<int> adjacentFaces(mesh, 0);
	for (FaceCIter f = mesh.faces.begin(); f != mesh.faces.end(); f++) {
		HalfEdgeCIter h = f->he;
		do {
			adjacentFaces[h->vertex]++;

			h = h->next;
		} while (h != f->he);
	}

	for (BoundaryCIter b = mesh.boundaries.begin(); b != mesh.boundaries.end(); b++) {
		HalfEdgeCIter h = b->he;
		do {
			adjacentFaces[h->vertex]++;

			h = h->next;
		} while (h != b->he);
	}

	bool check = false;
	for (VertexCIter v = mesh.vertices.begin(); v != mesh.vertices.end(); v++) {
		if (adjacentFaces[v] != v->degree()) {
			std::cout << "Problem with vertex: " << v->index << std::endl;
			check = true;
		}
	}
	if(check) return true;

	return false;
}

bool MeshIO::buildMesh(const Soup& soup, Mesh& mesh, std::string& error)
{
	// preallocate elements
	preallocateElements(soup, mesh);

	// create and insert vertices
	int nVertices = (int)soup.positions.size();
	std::vector<VertexIter> indexToVertex(nVertices);
	for (int i = 0; i < nVertices; i++) {
		VertexIter v = mesh.vertices.insert(mesh.vertices.end(), Vertex());
		v->position = soup.positions[i];
		v->position2 = soup.positions2[i];
		v->he = mesh.halfEdges.end();
		indexToVertex[i] = v;
	}

	// create and insert halfedges, edges and "real" faces
	int tableSize = soup.table.getSize();
	std::vector<int> edgeCount(tableSize, 0);
	std::vector<HalfEdgeIter> existingHalfEdges(tableSize);
	std::vector<int> hasFlipEdge(2*tableSize, 0);

	for (int I = 0; I < (int)soup.indices.size(); I += 3) {
		// create new face
		FaceIter f = mesh.faces.insert(mesh.faces.end(), Face());

		// create a halfedge for each edge of the newly created face
		std::vector<HalfEdgeIter> halfEdges(3);
		for (int J = 0; J < 3; J++) {
			halfEdges[J] = mesh.halfEdges.insert(mesh.halfEdges.end(), HalfEdge());
			halfEdges[J]->index = (int)mesh.halfEdges.size() - 1;
		}

		// initialize the newly created halfedges
		for (int J = 0; J < 3; J++) {
			// current halfedge goes from vertex i to vertex j
			int K = (J + 1) % 3;
			int i = soup.indices[I + J];
			int j = soup.indices[I + K];

			// set the current halfedge's attributes
			HalfEdgeIter h = halfEdges[J];
			h->next = halfEdges[K];
			h->prev = halfEdges[(J + 3 - 1) % 3];
			h->onBoundary = false;
			hasFlipEdge[h->index] = 0;

			// point the new halfedge and vertex i to each other
			VertexIter v = indexToVertex[i];
			h->vertex = v;
			v->he = h;

			// point the new halfedge and face to each other
			h->face = f;
			f->he = h;

			int eIndex = soup.table.getIndex(i, j);
			if (edgeCount[eIndex] > 0) {
				// if a halfedge between vertex i and j has been created in the past,
				// then it is the flip halfedge of the current halfedge
				HalfEdgeIter flip = existingHalfEdges[eIndex];
				h->flip = flip;
				flip->flip = h;
				h->edge = flip->edge;

				hasFlipEdge[h->index] = 1;
				hasFlipEdge[flip->index] = 1;
				edgeCount[eIndex]++;

			} else {
				// create an edge and set its halfedge
				EdgeIter e = mesh.edges.insert(mesh.edges.end(), Edge());
				h->edge = e;
				e->he = h;

				// record the newly created edge and halfedge from vertex i to j
				existingHalfEdges[eIndex] = h;
				edgeCount[eIndex] = 1;
			}

			// check for non-manifold edges
			if (edgeCount[eIndex] > 2) {
				error = "Mesh has non-manifold edges.";
				std::cout << "Found a non-manifold edge" << std::endl;
				return false;
			}
		}
	}

	// create and insert boundary halfedges and "imaginary" faces for boundary cycles
	// also create and insert corners
	HalfEdgeIter end = mesh.halfEdges.end();
	for (HalfEdgeIter h = mesh.halfEdges.begin(); h != end; h++) {
		// if a halfedge has no flip halfedge, create a new face and
		// link it the corresponding boundary cycle
		if (!hasFlipEdge[h->index]) {
			// create new face
			FaceIter f = mesh.boundaries.insert(mesh.boundaries.end(), Face());

			// walk along boundary cycle
			std::vector<HalfEdgeIter> boundaryCycle;
			boundaryCycle.reserve(2*mesh.edges.size() - mesh.halfEdges.size());
			HalfEdgeIter he = h;
			do {
				// create a new halfedge
				HalfEdgeIter bH = mesh.halfEdges.insert(mesh.halfEdges.end(), HalfEdge());
				bH->index = (int)mesh.halfEdges.size() - 1;
				boundaryCycle.push_back(bH);

				// grab the next halfedge along the boundary that does not have a
				// flip halfedge
				HalfEdgeIter nextHe = he->next;
				while (hasFlipEdge[nextHe->index]) {
					nextHe = nextHe->flip->next;
				}

				// set the current halfedge's attributes
				bH->vertex = nextHe->vertex;
				bH->edge = he->edge;
				bH->onBoundary = true;

				// point the new halfedge and face to each other
				bH->face = f;
				f->he = bH;

				// point the new halfedge and he to each other
				bH->flip = he;
				he->flip = bH;

				// continue walk
				he = nextHe;
			} while (he != h);

			// link the cycle of boundary halfedges together
			int n = (int)boundaryCycle.size();
			for (int i = 0; i < n; i++) {
				boundaryCycle[i]->next = boundaryCycle[(i + n - 1) % n]; // boundary halfedges are linked in clockwise order
				boundaryCycle[i]->prev = boundaryCycle[(i + 1) % n];
				hasFlipEdge[boundaryCycle[i]->index] = 1;
				hasFlipEdge[boundaryCycle[i]->flip->index] = 1;
			}
		}

	}

	// index mesh elements
	indexElements(mesh);

	// check if mesh has isolated vertices
	if (hasIsolatedVertices(mesh)) {
		error = "Mesh has isolated vertices.";
		std::cout << "Mesh has isolated vertices." << std::endl;
		return false;
	}

	// check if mesh has non-manifold vertices
	if (hasNonManifoldVertices(mesh)) {
		error = "Mesh has non manifold vertices.";
		std::cout << "Mesh has non manifold vertices." << std::endl;
		return false;
	}

	return true;
}

bool MeshIO::readOBJ(std::istringstream& in, Mesh& model, std::string& error)
{
	Soup soup;
	std::string line;
	int nVertices = 0;
	bool seenFace = false;

	double x, y, z;
	double x2=0, y2=0, z2=0;
	while (std::getline(in, line)) {
		std::istringstream ss(line);
		std::string token;
		ss >> token;

		if (token == "v") {
			ss >> x >> y >> z;
			soup.positions.push_back(Vector(x, y, z));
			soup.positions2.push_back(Vector(x2, y2, z2));

			if (seenFace) {
				nVertices = 0;
				seenFace = false;
			}
			nVertices++;

		} else if (token == "f") {
			seenFace = true;
			int vertexCount = 0;

			while (ss >> token) {
				std::istringstream indexStream(token);
				std::string indexString;

				if (std::getline(indexStream, indexString, '/')) {
					int index = std::stoi(indexString);
					if (index < 0) index = nVertices + index;
					else index -= 1;

					vertexCount++;
					soup.indices.push_back(index);

				}
			}
		}
	}

	// construct table
	soup.table.construct(soup.positions.size(), soup.indices);

	// build halfedge meshes
	if (!buildMesh(soup, model, error)) {
		return false;
	}
	return true;
}

bool MeshIO::readOFF(std::istringstream& in, Mesh& model, std::string& error)
{
	Soup soup;
	std::string line;

	int nvert, nface, info;
	std::getline(in, line); // OFF or COFF; not needed
	std::getline(in, line); // info on mesh size
	std::istringstream ss(line);
	ss >> nvert >> nface >> info;

	double x, y, z;
	double x2=0, y2=0, z2=0;
	for(int i = 0; i < nvert; i++) {
		std::getline(in, line);
		std::istringstream ss(line);
		ss >> x >> y >> z;
		soup.positions.push_back(Vector(x, y, z));
		soup.positions2.push_back(Vector(x2, y2, z2));
	}

	int idx, jdx, kdx;
	for(int i = 0; i < nface; i++) {
		std::getline(in, line);
		std::istringstream ss(line);
		ss >> info >> idx >> jdx >> kdx;
		soup.indices.push_back(idx);
		soup.indices.push_back(jdx);
		soup.indices.push_back(kdx);
	}

	// construct table
	soup.table.construct(soup.positions.size(), soup.indices);

	// build halfedge mesh
	if (!buildMesh(soup, model, error)) {
		return false;
	}

	return true;
}

void MeshIO::normalize(Mesh& model, Vector& cm, double *scale)
{
	// compute center of mass
	int nVertices = 0;
	cm[0] = 0; cm[1] = 0; cm[2] = 0;
	for (VertexCIter v = model.vertices.begin(); v != model.vertices.end(); v++) {
		cm += v->position;
		nVertices++;
	}
	cm /= nVertices;

	// translate to origin
	for (VertexIter v = model.vertices.begin(); v != model.vertices.end(); v++) {
		v->position -= cm;
	}

	// rescale to unit Area
	double Area = 0.0;
	for (FaceIter f = model.faces.begin(); f != model.faces.end(); f++) {
		Area += f->area();
	}
	double Scale = std::sqrt(1.0/Area);
//	Scale = std::max(scale, 1.0);
//	Scale = 1.0;
	double radius = 0.0;
	for (VertexIter v = model.vertices.begin(); v != model.vertices.end(); v++) {
		v->position *= Scale;
		radius = std::max(radius, v->position.norm());
	}
	model.radius = radius;
	*scale = Scale;

}

bool MeshIO::readOBJ(const std::string& fileName, Mesh& model, int flag_norm, Vector& cm, double *scale, std::string& error)
{
	std::ifstream in(fileName.c_str());
	std::istringstream buffer;

	if (!in.is_open()) {
		return false;

	} else {
		buffer.str(std::string(std::istreambuf_iterator<char>(in),
							   std::istreambuf_iterator<char>()));
	}

	bool success = readOBJ(buffer, model, error);

	if (success && flag_norm == 1) {
		normalize(model, cm, scale);
	}

	in.close();

	return success;
}

bool MeshIO::readOFF(const std::string& fileName, Mesh& model, int flag_norm, Vector& cm, double *scale, std::string& error)
{
	std::ifstream in(fileName.c_str());
	std::istringstream buffer;

	if (!in.is_open()) {
		return false;

	} else {
		buffer.str(std::string(std::istreambuf_iterator<char>(in),
							   std::istreambuf_iterator<char>()));
	}

	bool success = readOFF(buffer, model, error);

	if (success && flag_norm == 1) {
		normalize(model, cm, scale);
	}

	in.close();

	return success;
}

void writeString(std::ofstream& out, const std::string& str)
{
	out.write(str.c_str(), str.size());
}

void MeshIO::writeOFF(std::ofstream& out, Mesh& model)
{
	writeString(out, "OFF\n");
	int nvert = model.vertices.size();
	int nface = model.faces.size();
	int info = 0;
	Vector p;
	writeString(out, std::to_string(nvert) + " " + std::to_string(nface) + " "
	+ std::to_string(info) +"\n");
	for (VertexCIter v = model.vertices.begin(); v != model.vertices.end(); v++) {
		p = v->position2;
		writeString(out, std::to_string(p.x) + " " + 
		std::to_string(p.y) + " " + std::to_string(p.z) + "\n");
	}

	for (FaceCIter f = model.faces.begin(); f != model.faces.end(); f++) {
		writeString(out, "3 ");

		HalfEdgeCIter he = f->he;
		HalfEdgeCIter fhe = he;
		do {
			VertexCIter v = he->vertex;
			int vIndex = v->referenceIndex == -1 ? v->index : v->referenceIndex;
			writeString(out, " " + std::to_string(vIndex));
			he = he->next;
		} while (he != fhe);

		writeString(out, "\n");
	}
}

void MeshIO::writeCOFF(std::ofstream& out, Mesh& model, int *color)
{
	std::string white=" 192 192 192 255";
	std::string red=" 255 0 0 255";
	writeString(out, "COFF\n");
	int nvert = model.vertices.size();
	int nface = model.faces.size();
	int info = 0;
	Vector p;
	writeString(out, std::to_string(nvert) + " " + std::to_string(nface) + " "
	+ std::to_string(info) +"\n");
	for (VertexCIter v = model.vertices.begin(); v != model.vertices.end(); v++) {
		p = v->position;
		if(color[v->index]==0) {
			writeString(out, std::to_string(p.x) + " " + 
			std::to_string(p.y) + " " + std::to_string(p.z) + white + "\n");
		} else {
			writeString(out, std::to_string(p.x) + " " + 
			std::to_string(p.y) + " " + std::to_string(p.z) + red + "\n");
		}
	}

	for (FaceCIter f = model.faces.begin(); f != model.faces.end(); f++) {
		writeString(out, "3 ");

		HalfEdgeCIter he = f->he;
		HalfEdgeCIter fhe = he;
		do {
			VertexCIter v = he->vertex;
			int vIndex = v->referenceIndex == -1 ? v->index : v->referenceIndex;
			writeString(out, " " + std::to_string(vIndex));
			he = he->next;
		} while (he != fhe);

		writeString(out, "\n");
	}
}
bool MeshIO::writeOFF(const std::string& fileName, Mesh& model)
{
	std::ofstream out(fileName.c_str());

	if (!out.is_open()) {
		return false;
	}

	MeshIO::writeOFF(out, model);
	out.close();

	return true;
}

bool MeshIO::writeCOFF(const std::string& fileName, Mesh& model, int *color)
{
	std::ofstream out(fileName.c_str());

	if (!out.is_open()) {
		return false;
	}

	MeshIO::writeCOFF(out, model, color);
	out.close();

	return true;
}
