/* ================================================================================
   ExpMap

   Set of functions used to represent a rotation using an exponential map

   It is (loosely) adapted from:
   F. Grassia, "Practical Parameterization of Rotations Using the Exponential Map",
   J. Grph. Tools, 3, 29-48 (1998)

   and from the code found at http://www.cs.cmu.edu/~spiff/exp-map

 ================================================================================ */

#ifndef _EXPMAP_H_
#define _EXPMAP_H_

   #include <math.h>

/* ================================================================================
   The class
 ================================================================================ */

   class EXPMAP {

	public:

		// Convert a vector v into a rotation matrix R
		void EM3_To_R(double *v, double R[3][3]);

		// Derivatives of the rotation matrix R wrt one component of vector v
		void dR_dVi(double *v, int i, double dRdvi[3][3]);

	private:

		// compute the norm of a vector
		double VNorm3(double *vec);

		// scale a vector
		void VScale3(double *vec, const double s, double *svec);

		// From quaternion to rotation matrix
		void Q_To_Matrix(double *q, double R[3][3]);

		// check parameterization of angle too close to 2*pi
		int Check_Param(double v[3], double *theta);

		// From the vector defining the Exp. Map to quaternion
		int EM_To_Q(double *, double *q, int reparam);

		// Base on quaternion representation of R, computes dR/dv_i
		void Partial_R_Partial_Vi(double *q, double *dqdvi, double dRdvi[3][3]);

		// derivative of quaternion q wrt one coordinate of EM vector v
		void Partial_Q_Partial_3V(double *v, int i, double *dqdx);

		double MinAngle = 1.e-6;
		double CutoffAngle = M_PI;

   };

/* ================================================================================
	compute the norm of a vector
 ================================================================================ */

   double EXPMAP::VNorm3(double *vec)
   {
	double norm = 0;
	for(int i = 0; i < 3; i++) {
		norm+= vec[i]*vec[i];
	}
	return std::sqrt(norm);
   }

/* ================================================================================
	scale a vector
 ================================================================================ */

   void EXPMAP::VScale3(double *v, double a, double *av)
   {
	for(int i = 0; i < 3; i++) {
		av[i] = a*v[i];
	}
   }

/* ================================================================================
	quaternion to rotation matrix
 ================================================================================ */

   void EXPMAP::Q_To_Matrix(double *q, double R[3][3])
   {
	double  xx, yy, zz;
	double  xy, xz, yz;
	double  wx, wy, wz;

	xx = 2.0*q[0]*q[0]; yy = 2.0*q[1]*q[1]; zz = 2.0*q[2]*q[2];
	xy = 2.0*q[0]*q[1]; xz = 2.0*q[0]*q[2]; yz = 2.0*q[1]*q[2];
	wx = 2.0*q[3]*q[0]; wy = 2.0*q[3]*q[1]; wz = 2.0*q[3]*q[2];
    
	R[0][0] = 1.0 - (yy+zz); R[0][1] = xy-wz;       R[0][2] = xz+wy;
	R[1][0] = xy+wz;         R[1][1] = 1.0-(xx+zz); R[1][2] = yz-wx;
	R[2][0] = xz-wy;         R[2][1] = yz+wx;       R[2][2] = 1.0-(xx+yy);
   }

/* ================================================================================
  'Check_Parameterization' To escape the vanishing derivatives at
  shells of 2PI rotations, we reparameterize to a rotation of (2PI -
  theta) about the opposite axis when we get too close to 2PI
 ================================================================================ */

   int EXPMAP::Check_Param(double *v, double *theta)
   {
	int rep = 0;

	*theta = VNorm3(v);

	if (*theta > CutoffAngle){
		double scl = *theta;
		if (*theta > 2*M_PI){	/* first get theta into range 0..2PI */
			*theta = std::fmod(*theta, 2*M_PI);
			scl = *theta/scl;
			VScale3(v, scl, v);
			rep = 1;
		}
		if (*theta > CutoffAngle){
			scl = *theta;
			*theta = 2*M_PI - *theta;
			scl = 1.0 - 2*M_PI/scl;
			VScale3(v, scl, v);
			rep = 1;
		}
	}

	return rep;
   }

/* ================================================================================
  'EM_To_Q' Convert a 3 DOF EM vector 'v' into its corresponding
  quaternion 'q'. If 'reparam' is non-zero, perform dynamic
  reparameterization, if necessary, storing the reparameterized EM in
  'v' and returning 1.  Returns 0 if no reparameterization was
  performed.
 ================================================================================ */

   int EXPMAP::EM_To_Q(double *v, double *q, int reparam)
   {

	int rep=0;
	double cosp, sinp, theta;

	if (reparam) {
		rep = Check_Param(v, &theta);
	} else {
		theta = VNorm3(v);
	}
    
	cosp = std::cos(.5*theta);
	sinp = std::sin(.5*theta);

	q[3] = cosp;
	if (theta < MinAngle) {
		VScale3(v, .5 - theta*theta/48.0, q);	/* Taylor Series for sinc */
	} else {
		VScale3(v, sinp/theta, q);
	}

	return rep;
   }

/* ================================================================================
  Converts a 3 DOF EM vector 'v' into a rotation matrix.
 ================================================================================ */

   void EXPMAP::EM3_To_R(double *v, double R[3][3])
   {
	double q[4];

	EM_To_Q(v, q, 0);
	Q_To_Matrix(q, R);
   }

/* ================================================================================
  'Partial_R_Partial_Vi' Given a quaternion 'q' computed from the
  current 2 or 3 degree of freedom EM vector 'v', and the partial
  derivative of the quaternion with respect to the i'th element of
  'v' in 'dqdvi' (computed using 'Partial_Q_Partial_3V' or
  'Partial_Q_Partial_2V'), compute and store in 'dRdvi' the i'th
  partial derivative of the rotation matrix 'R' with respect to the
  i'th element of 'v'.
 ================================================================================ */

   void EXPMAP::Partial_R_Partial_Vi(double *q, double *dqdvi, double dRdvi[3][3])
   {
	double    prod[9];
    
/* ================================================================================
    This efficient formulation is arrived at by writing out the
    entire chain rule product dRdq * dqdv in terms of 'q' and 
    noticing that all the entries are formed from sums of just
    nine products of 'q' and 'dqdv' 
 ================================================================================ */

	prod[0] = -4*q[0]*dqdvi[0];
	prod[1] = -4*q[1]*dqdvi[1];
	prod[2] = -4*q[2]*dqdvi[2];
	prod[3] = 2*(q[1]*dqdvi[0] + q[0]*dqdvi[1]);
	prod[4] = 2*(q[3]*dqdvi[2] + q[2]*dqdvi[3]);
	prod[5] = 2*(q[2]*dqdvi[0] + q[0]*dqdvi[2]);
	prod[6] = 2*(q[3]*dqdvi[1] + q[1]*dqdvi[3]);
	prod[7] = 2*(q[2]*dqdvi[1] + q[1]*dqdvi[2]);
	prod[8] = 2*(q[3]*dqdvi[0] + q[0]*dqdvi[3]);

	dRdvi[0][0] = prod[1] + prod[2];
	dRdvi[0][1] = prod[3] - prod[4];
	dRdvi[0][2] = prod[5] + prod[6];

	dRdvi[1][0] = prod[3] + prod[4];
	dRdvi[1][1] = prod[0] + prod[2];
	dRdvi[1][2] = prod[7] - prod[8];

	dRdvi[2][0] = prod[5] - prod[6];
	dRdvi[2][1] = prod[7] + prod[8];
	dRdvi[2][2] = prod[0] + prod[1];

   }

/* ================================================================================
 'Partial_Q_Partial_3V' Partial derivative of quaternion wrt i'th
 component of EM vector 'v'
 ================================================================================ */

   void EXPMAP::Partial_Q_Partial_3V(double *v, int i, double *dqdx)
   {
	double   theta = VNorm3(v);
	double   cosp = cos(.5*theta), sinp = sin(.5*theta);
    
/* ================================================================================
    This is an efficient implementation of the derivatives given
    in Appendix A of the paper with common subexpressions factored out 
 ================================================================================ */

	int i2 = (i+1)%3; const int i3 = (i+2)%3;

	if (theta < MinAngle){

		double Tsinc = 0.5 - theta*theta/48.0;
		double vTerm = v[i] * (theta*theta/40.0 - 1.0) / 24.0;

		dqdx[3]  = -.5*v[i]*Tsinc;
		dqdx[i]  = v[i]* vTerm + Tsinc;
		dqdx[i2] = v[i2]*vTerm;
		dqdx[i3] = v[i3]*vTerm;
	}
	else {
		double  ang   = 1.0/theta;
		double ang2   = ang*ang*v[i]; 
		double sang   = sinp*ang;
		double  cterm = ang2*(.5*cosp - sang);
	
		dqdx[i]  = cterm*v[i] + sang;
		dqdx[i2] = cterm*v[i2];
		dqdx[i3] = cterm*v[i3];
		dqdx[3] = -.5*v[i]*sang;
	}
   }

/* ================================================================================
  'Computes the i'th partial derivative of
  the rotation matrix with respect to EM parameter 'v', storing result
  in 'dRdvi'.  If 'v' is near a singularity, it will be dynamically
  reparameterized in place and the value 1 is returned; otherwise,
  0 is returned.
 ================================================================================ */

   void EXPMAP::dR_dVi(double *v, int i, double dRdvi[3][3])
   {

	double q[4], dqdvi[4];

	EM_To_Q(v, q, 1);

	Partial_Q_Partial_3V(v, i, dqdvi);
	Partial_R_Partial_Vi(q, dqdvi, dRdvi);

   }

#endif
