/* ====SIFT=======================================================================================
 *
 * Author: Patrice Koehl,  May 2020
 * Department of Computer Science
 * University of California, Davis
 *
 * Computes the SIFT descriptors of keypoints of an image
 *
 * based on: VLFEAT
 * Copyright (C) 2007-12 Andrea Vedaldi and Brian Fulkerson.
 * All rights reserved. See COPYING
 =============================================================================================== */

#ifndef _SIFT_H_
#define _SIFT_H_

 #include "math.h"
 #include "string.h"
 #include <iostream>
 #include "stdlib.h"
 #include <vector>
 #include "sift.hh"

/* ===============================================================================================
   The SIFT class
 =============================================================================================== */

  class SIFT {

	public:

		void compute_descriptors(int width, int height, sift_pix *grd, double x, double y, 
			double sigma, double angle0, std::vector<double>& descriptors);

	private:

		// Create a SIFT filter
		SiftFilt * sift_new_filter (int width, int height, int noctaves, int nlevels, 
			int o_min);

		// Delete a SIFT filter
		void sift_delete_filter (SiftFilt* f);

		// Compute SIFT descriptor on a specific pixel
		void sift_calc_raw_descriptor (SiftFilt const *f, sift_pix const* grad, 
			sift_pix *descr, int width, int height,
			double x, double y, double sigma, double angle0);

		// Normalize (L2-norm) a SIFT descriptor
		sift_pix normalize_histogram(sift_pix *begin, sift_pix *end);

		// small utility to compute floor of a float as a long int
		long int floor_f (float x);

		// small utility to bring an angle into [0, 2*Pi]
		float mod_2pi_f (float x);

		// transpose a descriptor
		void transpose_descriptor (sift_pix* dst, sift_pix* src);

  };

/* ===============================================================================================
 *
 * Compute the SIFT descriptor on one point P(x, y) of the image over one octave
 *
 * @param grad     image gradients (provided as pairs (amplitude, angle)
 * @param descr    SIFT descriptor (output).
 * @param width    image width.
 * @param height   image height.
 * @param x        keypoint x coordinate.
 * @param y        keypoint y coordinate.
 * @param sigma    keypoint scale.
 * @param angle0   keypoint orientation.
 =============================================================================================== */

  void SIFT::compute_descriptors(int width, int height, sift_pix *grad, double x, double y, 
	double sigma, double angle0, std::vector<double>& descriptors)
 {

	/* ====================================================================================
	transpose angles 
 	======================================================================================= */

	for (int i = 1 ; i < 2*width*height ; i+=2) {
		grad[i] = M_PI/2 - grad[i] ;
	}

	/* ====================================================================================
	Create a SIFT filter
 	======================================================================================= */

	SiftFilt * filt = 0 ;
	filt = sift_new_filter (width, height, -1, -1, 0);

	sift_pix  buf[128], rbuf[128] ;

	double th = M_PI / 2 - angle0;

	/* ====================================================================================
	Compute descriptor
 	======================================================================================= */

	sift_calc_raw_descriptor (filt, grad, buf, width, height, x, y, sigma, th);

	/* ====================================================================================
	Transpose, and generate descriptors as doubles
 	======================================================================================= */

	transpose_descriptor (rbuf, buf) ;

	double v;
	for(int j = 0; j < 128; j++) {
		v = 512.0 * rbuf[j];
		if(v > 255.0) v = 255.0;
//		descriptors[j]+=round(v);
		descriptors[j]+=v;
	}

	/* ====================================================================================
	Transpose, and generate descriptors as doubles
 	======================================================================================= */

	sift_delete_filter( filt); 

 }
/* ===============================================================================================
 * Create a new SIFT filter
 *
 * @param width    image width.
 * @param height   image height.
 * @param noctaves number of octaves.
 * @param nlevels  number of levels per octave.
 * @param o_min    first octave index.
 *
 * The function allocates and returns a new SIFT filter for the
 * specified image and scale space geometry.
 *
 * Setting @a O to a negative value sets the number of octaves to the
 * maximum possible value depending on the size of the image.
 *
 * return the new SIFT filter.
 =============================================================================================== */

  SiftFilt * SIFT::sift_new_filter (int width, int height,
	int noctaves, int nlevels, int o_min)
{
	SiftFilt *f =  new SiftFilt[1] ;

	int w   = SHIFT_LEFT (width,  -o_min) ;
	int h   = SHIFT_LEFT (height, -o_min) ;
	int nel = w * h ;

/* negative value O => calculate max. value */

	if (noctaves < 0) {
		noctaves = MAX (std::floor (log2 (MIN(width, height))) - o_min - 3, 1) ;
	}

	f-> width   = width ;
	f-> height  = height ;
	f-> O       = noctaves ;
	f-> S       = nlevels ;
	f-> o_min   = o_min ;
	f-> s_min   = -1 ;
	f-> s_max   = nlevels + 1 ;
	f-> o_cur   = o_min ;

	f-> temp    = new sift_pix[nel];
	f-> octave  = new sift_pix[nel*(f->s_max- f->s_min + 1)] ;
	f-> dog     = new sift_pix[nel*(f->s_max- f->s_min)] ;
	f-> grad    = new sift_pix[nel * 2 *(f->s_max - f->s_min)];

	f-> sigman  = 0.5 ;
	f-> sigmak  = std::pow (2.0, 1.0 / nlevels) ;
	f-> sigma0  = 1.6 * f->sigmak ;
	f-> dsigma0 = f->sigma0 * std::sqrt (1.0 - 1.0 / (f->sigmak*f->sigmak)) ;

	f-> gaussFilter = NULL ;
	f-> gaussFilterSigma = 0 ;
	f-> gaussFilterWidth = 0 ;

	f-> octave_width  = 0 ;
	f-> octave_height = 0 ;

	f-> keys     = 0 ;
	f-> nkeys    = 0 ;
	f-> keys_res = 0 ;

	f-> peak_thresh = 0.0 ;
	f-> edge_thresh = 10.0 ;
	f-> norm_thresh = 0.0 ;
	f-> magnif      = 3.0 ;
	f-> windowSize  = NBP / 2 ;

	f-> grad_o  = o_min - 1 ;

	return f ;
}

/* ===============================================================================================
 * Delete SIFT filter
 *
 * @param f SIFT filter to delete.
 *
 * The function frees the resources allocated by ::sift_new_filter().
 =============================================================================================== */

  void SIFT::sift_delete_filter (SiftFilt* f)
  {
	if (f) {
		if (f->keys) delete [] f->keys ;
		if (f->grad) delete [] f->grad ;
		if (f->dog) delete [] f->dog ;
		if (f->octave) delete [] f->octave ;
		if (f->temp) delete [] f->temp ;
		if (f->gaussFilter) delete [] f->gaussFilter ;
		delete [] f;
	}
  }

/* ===============================================================================================
 *
 * Run the SIFT descriptor on raw data
 *
 * @param f        SIFT filter.
 * @param grad     image gradients.
 * @param descr    SIFT descriptor (output).
 * @param width    image width.
 * @param height   image height.
 * @param x        keypoint x coordinate.
 * @param y        keypoint y coordinate.
 * @param sigma    keypoint scale.
 * @param angle0   keypoint orientation.
 *
 * The function runs the SIFT descriptor on raw data. Here @a image
 * is a 2 x @a width x @a height array (by convention, the memory
 * layout is a s such the first index is the fastest varying
 * one). The first @a width x @a height layer of the array contains
 * the gradient magnitude and the second the gradient angle (in
 * radians, between 0 and @f$ 2\pi @f$). @a x, @a y and @a sigma give
 * the keypoint center and scale respectively.
 *
 * In order to be equivalent to a standard SIFT descriptor the image
 * gradient must be computed at a smoothing level equal to the scale
 * of the keypoint. In practice, the actual SIFT algorithm makes the
 * following additional approximation, which influence the result:
 *
 * - Scale is discretized in @c S levels.
 * - The image is downsampled once for each octave (if you do this,
 *   the parameters @a x, @a y and @a sigma must be
 *   scaled too).
 *
 =============================================================================================== */

  void SIFT::sift_calc_raw_descriptor (SiftFilt const *f,
	sift_pix const* grad, sift_pix *descr, int width, int height,
	double x, double y, double sigma, double angle0)
  {

	double const magnif = f-> magnif ;

	int          w      = width ;
	int          h      = height ;
	int const    xo     = 2 ;         /* x-stride */
	int const    yo     = 2 * w ;     /* y-stride */

	int          xi     = (int) (x + 0.5) ;
	int          yi     = (int) (y + 0.5) ;

	double const st0    = std::sin (angle0) ;
	double const ct0    = std::cos (angle0) ;
	double const SBP    = magnif * sigma + EPSILON_D ;
	int    const W      = std::floor (std::sqrt(2.0) * SBP * (NBP + 1) / 2.0 + 0.5) ;

	int const binto = 1 ;          /* bin theta-stride */
	int const binyo = NBO * NBP ;  /* bin y-stride */
	int const binxo = NBO ;        /* bin x-stride */

	int bin, dxi, dyi ;
	sift_pix const *pt ;
	sift_pix       *dpt ;

/* ===============================================================================================
	check bounds
 =============================================================================================== */

	if(xi <  0 || xi >= w || yi <  0 || yi >= h - 1 ) return ;

/* ===============================================================================================
  	clear descriptor
 =============================================================================================== */

	memset (descr, 0, sizeof(sift_pix) * NBO*NBP*NBP) ;

/* ===============================================================================================
	Center the scale space and the descriptor on the current keypoint.
	Note that dpt is pointing to the bin of center (SBP/2,SBP/2,0).
 =============================================================================================== */

	pt  = grad + xi*xo + yi*yo ;
	dpt = descr + (NBP/2) * binyo + (NBP/2) * binxo ;

#undef atd
#define atd(dbinx,dbiny,dbint) *(dpt + (dbint)*binto + (dbiny)*binyo + (dbinx)*binxo)

/* ===============================================================================================
	Process pixels in the intersection of the image rectangle
	(1,1)-(M-1,N-1) and the keypoint bounding box.
 =============================================================================================== */

	for(dyi =  MAX(- W,   - yi   ) ; dyi <= MIN(+ W, h - yi -1) ; ++ dyi)
	{
		for(dxi =  MAX(- W,   - xi   ) ; dxi <= MIN(+ W, w - xi -1) ; ++ dxi) 
		{

			/* retrieve */
			sift_pix mod   = *( pt + dxi*xo + dyi*yo + 0 ) ;
			sift_pix angle = *( pt + dxi*xo + dyi*yo + 1 ) ;
			sift_pix theta = mod_2pi_f (angle - angle0) ;

			/* fractional displacement */
			sift_pix dx = xi + dxi - x;
			sift_pix dy = yi + dyi - y;

			/* get the displacement normalized w.r.t. the keypoint
			orientation and extension */
			sift_pix nx = ( ct0 * dx + st0 * dy) / SBP ;
			sift_pix ny = (-st0 * dx + ct0 * dy) / SBP ;
			sift_pix nt = NBO * theta / (2 * M_PI) ;

			/* Get the Gaussian weight of the sample. The Gaussian window
			* has a standard deviation equal to NBP/2. Note that dx and dy
			* are in the normalized frame, so that -NBP/2 <= dx <= NBP/2. */
			sift_pix const wsigma = f->windowSize ;
			sift_pix win = std::exp ((nx*nx + ny*ny)/(2.0 * wsigma * wsigma)) ;

			/* The sample will be distributed in 8 adjacent bins.
			We start from the ``lower-left'' bin. */
			int         binx = (int)floor_f (nx - 0.5) ;
			int         biny = (int)floor_f (ny - 0.5) ;
			int         bint = (int)floor_f (nt) ;
			sift_pix rbinx = nx - (binx + 0.5) ;
			sift_pix rbiny = ny - (biny + 0.5) ;
			sift_pix rbint = nt - bint ;
			int         dbinx ;
			int         dbiny ;
			int         dbint ;
			sift_pix weight;

			/* Distribute the current sample into the 8 adjacent bins*/
			for(dbinx = 0 ; dbinx < 2 ; ++dbinx) {
				for(dbiny = 0 ; dbiny < 2 ; ++dbiny) {
					for(dbint = 0 ; dbint < 2 ; ++dbint) {

						if (binx + dbinx >= - (NBP/2) &&
						binx + dbinx <    (NBP/2) &&
						biny + dbiny >= - (NBP/2) &&
						biny + dbiny <    (NBP/2) ) {
							weight = win
								* mod
								* std::abs (1 - dbinx - rbinx)
								* std::abs (1 - dbiny - rbiny)
								* std::abs (1 - dbint - rbint) ;

							atd(binx+dbinx,biny+dbiny,(bint+dbint)%NBO) += weight ;
						}

					}
				}
			}
		}
	}

/* ===============================================================================================
	Standard SIFT descriptors are normalized, truncated and normalized again
 =============================================================================================== */

	/* normalize L2 norm */
	sift_pix norm = normalize_histogram (descr, descr + NBO*NBP*NBP) ;

/* ===============================================================================================
	Set the descriptor to zero if it is lower than our
	norm_threshold.  We divide by the number of samples in the
	descriptor region because the Gaussian window used in the
	calculation of the descriptor is not normalized.
 =============================================================================================== */

	int numSamples = (MIN(W, w - xi -1) - MAX(-W, - xi) + 1) * 
			(MIN(W, h - yi -1) - MAX(-W, - yi) + 1) ;

	if(f-> norm_thresh && norm < f-> norm_thresh * numSamples) {
		for(bin = 0; bin < NBO*NBP*NBP ; ++ bin)
			descr [bin] = 0;
	} else {
		/* truncate at 0.2. */
		for(bin = 0; bin < NBO*NBP*NBP ; ++ bin) {
			if (descr [bin] > 0.2) descr [bin] = 0.2;
		}

		/* normalize again. */
		normalize_histogram (descr, descr + NBO*NBP*NBP) ;
	}
  }

/* ===============================================================================================
  @brief Floor and convert to integer
 @param x argument.
 @return Similar to @c (int) floor(x)
 =============================================================================================== */

  long int SIFT::floor_f (float x)
  {
	long int xi = (long int) x ;
	if (x >= 0 || (float) xi == x) return xi ;
	else return xi - 1 ;
  }
 
/* ===============================================================================================
 * @brief Normalizes in norm L_2 a descriptor
 * @param begin begin of histogram.
 * @param end   end of histogram.
 =============================================================================================== */

  sift_pix SIFT::normalize_histogram(sift_pix *begin, sift_pix *end)
  {
	sift_pix* iter ;
	sift_pix  norm = 0.0 ;

	for (iter = begin ; iter != end ; ++ iter) {
		norm += (*iter) * (*iter) ;
	}

	norm = std::sqrt(norm) + EPSILON_F ;

	for (iter = begin; iter != end ; ++ iter) {
		*iter /= norm ;
	}

	return norm;
  }
/* ===============================================================================================
 * mod_2pi(x, 2 * PI)
 *
 * @param x input value.
 * @return mod(x, 2 * PI)
 *
 * The function is optimized for small absolute values of @a x.
 *
 * The result is guaranteed to be not smaller than 0. However, due to
 * finite numerical precision and rounding errors, the result can be
 * equal to 2 * PI (for instance, if @c x is a very small negative
 * number).
 =============================================================================================== */

  float SIFT::mod_2pi_f (float x)
  {
	while (x > (float)(2 * M_PI)) x -= (float) (2 * M_PI) ;
	while (x < 0.0F) x += (float) (2 * M_PI);
	return x ;
  }

/* ===============================================================================================
 * Transpose descriptor
 *
 * @param dst destination buffer.
 * @param src source buffer.
 *
 * The function writes to @a dst the transpose of the SIFT descriptor
 * @a src. The transpose is defined as the descriptor that one
 * obtains from computing the normal descriptor on the transposed
 * image.
 =============================================================================================== */

  void SIFT::transpose_descriptor (sift_pix* dst, sift_pix* src)
  {

	for (int j = 0 ; j < NBP ; ++j) {
		int jp = NBP - 1 - j ;
		for (int i = 0 ; i < NBP ; ++i) {
			int o  = NBO * i + NBP*NBO * j  ;
			int op = NBO * i + NBP*NBO * jp ;
			dst [op] = src[o] ;
			for (int t = 1 ; t < NBO ; ++t) {
				dst [NBO - t + op] = src [t + o] ;
			}
		}
	}
  }

#endif
