
/** @brief SIFT filter pixel type */
  typedef float sift_pix ;

/** ------------------------------------------------------------------
 ** @brief SIFT filter keypoint
 **
 ** This structure represent a keypoint as extracted by the SIFT
 ** filter ::SiftFilt.
 **/

  typedef struct _SiftKeypoint
  {
	int o ;           /**< o coordinate (octave). */

	int ix ;          /**< Integer unnormalized x coordinate. */
	int iy ;          /**< Integer unnormalized y coordinate. */
	int is ;          /**< Integer s coordinate. */

	float x ;     /**< x coordinate. */
	float y ;     /**< y coordinate. */
	float s ;     /**< s coordinate. */
	float sigma ; /**< scale. */
  } SiftKeypoint ;

/** ------------------------------------------------------------------
 ** @brief SIFT filter
 **
 ** This filter implements the SIFT detector and descriptor.
 **/

  typedef struct _SiftFilt
  {
	double sigman ;       /**< nominal image smoothing. */
	double sigma0 ;       /**< smoothing of pyramid base. */
	double sigmak ;       /**< k-smoothing */
	double dsigma0 ;      /**< delta-smoothing. */

	int width ;           /**< image width. */
	int height ;          /**< image height. */
	int O ;               /**< number of octaves. */
	int S ;               /**< number of levels per octave. */
	int o_min ;           /**< minimum octave index. */
	int s_min ;           /**< minimum level index. */
	int s_max ;           /**< maximum level index. */
	int o_cur ;           /**< current octave. */

	sift_pix *temp ;   /**< temporary pixel buffer. */
	sift_pix *octave ; /**< current GSS data. */
	sift_pix *dog ;    /**< current DoG data. */
	int octave_width ;    /**< current octave width. */
	int octave_height ;   /**< current octave height. */

	sift_pix *gaussFilter ;  /**< current Gaussian filter */
	double gaussFilterSigma ;   /**< current Gaussian filter std */
	long long gaussFilterWidth ;  /**< current Gaussian filter width */

	SiftKeypoint* keys ;/**< detected keypoints. */
	int nkeys ;           /**< number of detected keypoints. */
	int keys_res ;        /**< size of the keys buffer. */

	double peak_thresh ;  /**< peak threshold. */
	double edge_thresh ;  /**< edge threshold. */
	double norm_thresh ;  /**< norm threshold. */
	double magnif ;       /**< magnification factor. */
	double windowSize ;   /**< size of Gaussian window (in spatial bins) */

	sift_pix *grad ;   /**< GSS gradient data. */
	int grad_o ;          /**< GSS gradient data octave. */

 } SiftFilt ;

 #define SHIFT_LEFT(x,n) (((n)>=0)?((x)<<(n)):((x)>>-(n)))
 #define MAX(x,y) (((x)>(y))?(x):(y))
 #define MIN(x,y) (((x)<(y))?(x):(y))

 #define NBO 8
 #define NBP 4

 #define LOG_OF_2 0.693147180559945
 #define log2(x) (std::log(x)/LOG_OF_2)
 
 #define EPSILON_D 2.220446049250313e-16
 #define EPSILON_F 1.19209290E-07F
