/*================================================================================================
  HessianGo.h
  Version 1: 9/29/2019

  Purpose: Sets of routine for defining the Hessian for the Go potential and 
	   computing Hessian - Vect products

Copyright (c) Patrice Koehl.

================================================================================================== */

#ifndef _HESSIANGO_H_
#define _HESSIANGO_H_


/*================================================================================================
 Includes
================================================================================================== */

#include <math.h>
#include <cstdlib>
#include <cstring>
#include <vector>
#include "Links.h"

/*================================================================================================
 BLAS prototypes
================================================================================================== */

extern "C" {
	void daxpy_(int * n ,double *alpha , double * X, int *incx, double * Y,int *incy);

        void dsyevd_(char * JOBZ, char * UPLO, int *N, double *A, int *LDA, double *W, 
	double *WORK, int *LWORK, int *IWORK, int *LIWORK, int *INFO);

        void dsyev_(char * JOBZ, char * UPLO, int *N, double *A, int *LDA, double *W,
        double *WORK, int *LWORK, int *INFO);

	void dgemv_(char * trans, int * m, int * n, double * alpha, double *A,
        int *lda, double * X, int * incx, double *beta, double * Y, int * incy);

	double ddot_(int * n, double * u, int * incu, double * v, int *incv);

}

/*================================================================================================
 Class for Hessian Go
================================================================================================== */

  class HessianGo {

	public:
		// load all information on Hessian in data structure
		void loadHessian(int N, std::vector<Links> & ListPair, 
		double *Uij, double *U1ij, double *Uijk, double *Uijkl, int nthreads);

		// build non-bonded part of Hessian
		void buildUij(std::vector<Atoms>& atoms, std::vector<Links>& List, 
		int potential, double *Uij);

		// build bond part of Hessian
		void buildU1ij(std::vector<Atoms>& atoms, int N, double *U1ij);

		// build angle part of Hessian
		void buildUijk(std::vector<Atoms>& atoms, int N, double *Uijk);

		// build dihedral angle part of Hessian
		void buildUijkl(std::vector<Atoms>& atoms, int N, double *Uijkl);

		// Set force constants
		void buildK(std::vector<Links> & ListPair, double *Kconst);

		// Set force constants
		void buildK2(std::vector<Links> & ListPair, std::vector<Atoms>& atoms, 
		double *Kconst, int type);

		// build full Hessian matrix
		void fullHessian(std::vector<Atoms>& atoms, std::vector<Links>& List, double *Uij, 
		double *U1ij, double *Uijk, double *Uijkl, double *hessian);

		// Diagonalize full Hessian
		void fullEigen(int natoms, double *eigVal, double *eigVect);

		// Check eigenvalues / eigenvectors
		void checkEigen(int N, int NE, double *hessian, double *eigVal, 
		double *eigVect, double *err, double *Emin, double *Emax, double *Emean, 
		int nthreads);

		// Rescale eigenvectors to account for mass
		void rescaleEigVect(std::vector<Atoms>& atoms, int NE, double *eigVect);

		// Compute Go energy
		void goEnergy(std::vector<Atoms>& atoms, double *newconf, int N, double K_bond, 
		double K_angle, double K_dih1, double K_dih3, 
		double *U_bond, double *U_angle, double *U_dih, double *U_Go);

	private:

		// filter list using cutoff
		void filterList(std::vector<Atoms>& atoms, std::vector<Links>& List1, 
		std::vector<Links>& List2, double cutoff);

		// update full Hessian matrix
		void update_hessian(int N, double *hessian, int i, int j, double *u, double *v, double fact);
  };


/*================================================================================================
 LoadHessian: set hessians struct to store information about Hessian
================================================================================================== */

void HessianGo::loadHessian(int N, std::vector<Links> & ListPair, double *Uij, double *U1ij, 
	double *Uijk, double *Uijkl, int nthreads)
{
	for(int i = 0; i < nthreads; i++) 
	{
		hessians[i].Ncoord    = N;
		hessians[i].ListPair  = ListPair;
		hessians[i].Uij       = Uij;
		hessians[i].U1ij      = U1ij;
		hessians[i].Uijk      = Uijk;
		hessians[i].Uijkl     = Uijkl;
		hessians[i].Npair     = ListPair.size();
		double *Vect = new double[N];
		hessians[i].Vect2     = Vect;
	}
}

/*================================================================================================
 Filter list
================================================================================================== */

void HessianGo::filterList(std::vector<Atoms>& atoms, std::vector<Links>& List1, 
	std::vector<Links>& List2, double cutoff)
{
	double d;
	double cutoff2 = cutoff*cutoff;
	int Npair = List1.size();

	for (int pair = 0; pair < Npair; pair++)
	{
		d = List1[pair].rij0;
		d = d*d;

		if(d<cutoff2) {
			List2.push_back(List1[pair]);
		}
	}
}

/*================================================================================================
 Tensor vectors that defines Hessian for elastic network as well as non-bonded terms for
 Go potential
================================================================================================== */

void HessianGo::buildUij(std::vector<Atoms>& atoms, std::vector<Links>& List, 
	int potential, double *Uij)
{
	int i1, j1;
	double dx, dy, dz, d;
	double fact;
	int Npair = List.size();

	for (int pair = 0; pair < Npair; pair++)
	{
		i1 = List[pair].atm1;
		j1 = List[pair].atm2;

		dx = atoms[i1].coord[0] - atoms[j1].coord[0];
		dy = atoms[i1].coord[1] - atoms[j1].coord[1];
		dz = atoms[i1].coord[2] - atoms[j1].coord[2];

		fact = std::sqrt(std::sqrt(atoms[i1].mass * atoms[j1].mass));

		d = dx*dx + dy*dy + dz*dz;
		if(potential == 1) d = std::sqrt(d);
		d = 1.0/(d*fact);

		Uij[3*pair]   = dx*d;
		Uij[3*pair+1] = dy*d;
		Uij[3*pair+2] = dz*d;
	}
}

/*================================================================================================
 Tensor vectors that defines Hessian for bond terms for Go potential
================================================================================================== */

void HessianGo::buildU1ij(std::vector<Atoms>& atoms, int N, double *U1ij)
{
	int i1, j1;
	double dx, dy, dz, d;
	double fact;

	for (int pair = 0; pair < N-1; pair++)
	{
		i1 = pair;
		j1 = pair+1;

		if(atoms[i1].chainid == atoms[j1].chainid)
		{
			dx = atoms[i1].coord[0] - atoms[j1].coord[0];
			dy = atoms[i1].coord[1] - atoms[j1].coord[1];
			dz = atoms[i1].coord[2] - atoms[j1].coord[2];

			fact = std::sqrt(std::sqrt(atoms[i1].mass * atoms[j1].mass));

			d = dx*dx + dy*dy + dz*dz;
			d = std::sqrt(d);
			d = 1.0/(d*fact);

			U1ij[3*pair]   = dx*d;
			U1ij[3*pair+1] = dy*d;
			U1ij[3*pair+2] = dz*d;
		} else {
			U1ij[3*pair]   = 0;
			U1ij[3*pair+1] = 0;
			U1ij[3*pair+2] = 0;
		}
	}
}

/*================================================================================================
 Tensor vectors that defines Hessian for angle terms for Go potential
================================================================================================== */

void HessianGo::buildUijk(std::vector<Atoms>& atoms, int N, double *Uijk)
{
	int i1, j1, k1;
	double r1, r2, a1, a2, a3, d;
	double cosine, sine, alpha, beta1, beta2;
	double d1[3], d2[3], u[3], v[3];

	for (int ang = 0; ang < N-2; ang++)
	{
		i1 = ang;
		j1 = ang+1;
		k1 = ang+2;

		if((atoms[i1].chainid == atoms[j1].chainid) && (atoms[i1].chainid == atoms[k1].chainid) ) 
		{
			for (int i = 0; i < 3; i++)
			{
				a1 = atoms[i1].coord[i];
				a2 = atoms[j1].coord[i];
				a3 = atoms[k1].coord[i];
				d1[i]   = a2 - a1;
				d2[i]   = a2 - a3;
			}
			r1 = d1[0]*d1[0] + d1[1]*d1[1] + d1[2]*d1[2];
			r2 = d2[0]*d2[0] + d2[1]*d2[1] + d2[2]*d2[2];
			d  = d1[0]*d2[0] + d1[1]*d2[1] + d1[2]*d2[2];
			r1 = std::sqrt(r1);
			r2 = std::sqrt(r2);
			cosine = d/(r1*r2);
			sine   = std::sqrt(1-cosine*cosine);
			alpha  = 1.0/(sine*r1*r2);
			beta1  = 1.0*cosine/(sine*r1*r1);
			beta2  = 1.0*cosine/(sine*r2*r2);

			for(int i = 0; i < 3; i++)
			{
				u[i] = alpha*d2[i]-beta1*d1[i];
				v[i] = alpha*d1[i]-beta2*d2[i];
			}

			Uijk[9*ang]   = u[0];
			Uijk[9*ang+1] = u[1];
			Uijk[9*ang+2] = u[2];
			Uijk[9*ang+3] = -u[0]-v[0];
			Uijk[9*ang+4] = -u[1]-v[1];
			Uijk[9*ang+5] = -u[2]-v[2];
			Uijk[9*ang+6] = v[0];
			Uijk[9*ang+7] = v[1];
			Uijk[9*ang+8] = v[2];
		} else {
			for (int i = 0; i < 9 ; i++ )
			{
				Uijk[9*ang+i]=0;
			}
		}
	}
}

/*================================================================================================
 Tensor vectors that defines Hessian for dihedral terms for Go potential
================================================================================================== */

void HessianGo::buildUijkl(std::vector<Atoms>& atoms, int N, double *Uijkl)
{
	int i1, j1, k1, l1;
	double r, rm, rn, a1, a2, a3, a4;
	double alpha, beta;
	double d1[3], d2[3], d3[3], xm[3], xn[3], u[3], v[3], t[3];

	for (int dih = 0; dih < N-3; dih++)
	{
		i1 = dih;
		j1 = dih+1;
		k1 = dih+2;
		l1 = dih+3;

		if((atoms[i1].chainid == atoms[j1].chainid) && (atoms[i1].chainid == atoms[k1].chainid) &&
		(atoms[i1].chainid == atoms[l1].chainid))
		{
			for (int i = 0; i < 3; i++)
			{
				a1 = atoms[i1].coord[i];
				a2 = atoms[j1].coord[i];
				a3 = atoms[k1].coord[i];
				a4 = atoms[l1].coord[i];
				d1[i]   = a2 - a1;
				d2[i]   = a2 - a3;
				d3[i]   = a4 - a3;
			}
			r  = d2[0]*d2[0] + d2[1]*d2[1] + d2[2]*d2[2];
			a1 = d1[0]*d2[0] + d1[1]*d2[1] + d1[2]*d2[2];
			a2 = d2[0]*d3[0] + d2[1]*d3[1] + d2[2]*d3[2];
			xm[0] = d1[1]*d2[2] - d1[2]*d2[1];
			xm[1] = -d1[0]*d2[2] + d1[2]*d2[0];
			xm[2] = d1[0]*d2[1] - d1[1]*d2[0];
			xn[0] = d2[1]*d3[2] - d2[2]*d3[1];
			xn[1] = -d2[0]*d3[2] + d2[2]*d3[0];
			xn[2] = d2[0]*d3[1] - d2[1]*d3[0];
			rm    = xm[0]*xm[0] + xm[1]*xm[1] + xm[2]*xm[2];
			rn    = xn[0]*xn[0] + xn[1]*xn[1] + xn[2]*xn[2];
			r  = std::sqrt(r);
			alpha = a1/(r*r);
			beta  = a2/(r*r);

			for( int i = 0; i < 3; i++)
			{
				u[i] = r*xm[i]/rm;
				t[i] = -r*xn[i]/rn;
				v[i] = (alpha-1.0)*u[i] -beta*t[i];
			}

			Uijkl[12*dih]    = u[0];
			Uijkl[12*dih+1]  = u[1];
			Uijkl[12*dih+2]  = u[2];
			Uijkl[12*dih+3]  = v[0];
			Uijkl[12*dih+4]  = v[1];
			Uijkl[12*dih+5]  = v[2];
			Uijkl[12*dih+6]  = -(u[0]+v[0]+t[0]);
			Uijkl[12*dih+7]  = -(u[1]+v[1]+t[1]);
			Uijkl[12*dih+8]  = -(u[2]+v[2]+t[2]);
			Uijkl[12*dih+9]  = t[0];
			Uijkl[12*dih+10] = t[1];
			Uijkl[12*dih+11] = t[2];
		} else
		{
			for (int i = 0; i < 12; i++)
			{
				Uijkl[12*dih+i] = 0;
			}
		}
	}
}

/*================================================================================================
 From tensor to full matrix: update with one interaction
================================================================================================== */

void HessianGo::update_hessian(int N, double *hessian, int i, int j, double *u, double *v, double fact)
{
	int i1=3*i;
	int j1=3*j;
	hessian[i1*N + j1 ] += fact*u[0]*v[0];
	hessian[i1*N + j1 + 1] += fact*u[0]*v[1];
	hessian[i1*N + j1 + 2] += fact*u[0]*v[2];
	hessian[i1*N + N + j1 ] += fact*u[1]*v[0];
	hessian[i1*N + N + j1 + 1] += fact*u[1]*v[1];
	hessian[i1*N + N + j1 + 2] += fact*u[1]*v[2];
	hessian[i1*N + 2*N + j1 ] += fact*u[2]*v[0];
	hessian[i1*N + 2*N + j1 + 1] += fact*u[2]*v[1];
	hessian[i1*N + 2*N + j1 + 2] += fact*u[2]*v[2];
}

/*+================================================================================================
 Define full hessian matrix
================================================================================================== */

void HessianGo::fullHessian(std::vector<Atoms>& atoms, std::vector<Links>& List, double *Uij, 
	double *U1ij, double *Uijk, double *Uijkl, double *hessian)
{
	int npairs = List.size();
	int natoms = atoms.size();
	int i1, j1, k1, l1;

	int N = 3*natoms;
	std::memset(hessian, 0, N*N*sizeof(double));

	double u[3], v[3], w[3], t[3];
	double kconst;

	if(U1ij) {
		for(int atom = 0; atom < natoms-1; atom++)
		{
			i1 = atom;
			j1 = atom + 1;
			kconst = atoms[atom].k_bond;
			for (int i = 0; i < 3; i++)
			{
				u[i] = U1ij[3*atom+i];
				v[i] = -U1ij[3*atom+i];
			}
			update_hessian(N, hessian, i1, j1, u, v, kconst);
			update_hessian(N, hessian, j1, i1, v, u, kconst);
		}
	}

	if(Uijk) {
		for(int atom = 0; atom < natoms-2; atom++)
		{
			i1 = atom;
			j1 = atom + 1;
			k1 = atom + 2;
			kconst = atoms[atom].k_angle;
			for (int i = 0; i < 3; i++)
			{
				u[i] = Uijk[9*atom+i];
				v[i] = Uijk[9*atom+3+i];
				w[i] = Uijk[9*atom+6+i];
			}
			update_hessian(N, hessian, i1, j1, u, v, kconst);
			update_hessian(N, hessian, j1, i1, v, u, kconst);
			update_hessian(N, hessian, i1, k1, u, w, kconst);
			update_hessian(N, hessian, k1, i1, w, u, kconst);
			update_hessian(N, hessian, j1, k1, v, w, kconst);
			update_hessian(N, hessian, k1, j1, w, v, kconst);
		}
	}

	if(Uijkl) {
		for(int atom = 0; atom < natoms-3; atom++)
		{
			i1 = atom;
			j1 = atom + 1;
			k1 = atom + 2;
			l1 = atom + 3;
			kconst = atoms[atom].k_dihed;
			for (int i = 0; i < 3; i++)
			{
				u[i] = -Uijkl[12*atom+i];
				v[i] = -Uijkl[12*atom+3+i];
				w[i] = -Uijkl[12*atom+6+i];
				t[i] = -Uijkl[12*atom+9+i];
			}
			update_hessian(N, hessian, i1, j1, u, v, kconst);
			update_hessian(N, hessian, j1, i1, v, u, kconst);
			update_hessian(N, hessian, i1, k1, u, w, kconst);
			update_hessian(N, hessian, k1, i1, w, u, kconst);
			update_hessian(N, hessian, i1, l1, u, t, kconst);
			update_hessian(N, hessian, l1, i1, t, u, kconst);
			update_hessian(N, hessian, j1, k1, v, w, kconst);
			update_hessian(N, hessian, k1, j1, w, v, kconst);
			update_hessian(N, hessian, j1, l1, v, t, kconst);
			update_hessian(N, hessian, l1, j1, t, v, kconst);
			update_hessian(N, hessian, k1, l1, w, t, kconst);
			update_hessian(N, hessian, l1, k1, t, w, kconst);
		}
	}


	for(int pair = 0; pair < npairs; pair++)
	{
		i1 = List[pair].atm1;
		j1 = List[pair].atm2;
		kconst = List[pair].kconst;
		for (int i = 0; i < 3; i++)
		{
			u[i] = Uij[3*pair+i];
			v[i] = -Uij[3*pair+i];
		}

		if(i1 < 0 || i1 > natoms-1) {
			std::cout << "problem with a link: i1 = " << i1 << " j1 = " << j1 << std::endl;
			exit(1);
		}
		if(j1 < 0 || j1 > natoms-1) {
			std::cout << "problem with a link: i1 = " << i1 << " j1 = " << j1 << std::endl;
			exit(1);
		}

		update_hessian(N, hessian, i1, j1, u, v, kconst);
		update_hessian(N, hessian, j1, i1, v, u, kconst);

	}

	double sum;
	int Ncol;
	for(int i = 0; i < natoms; i++)
	{
		for(int l = 0; l < 3; l++) {
			Ncol = 3*i+l;
			for(int k = 0; k < 3; k++)
			{
				sum = 0.0;
				for(int j = k; j < N; j = j + 3) 
				{
					sum = sum + hessian[Ncol*N+j];
				}
				hessian[Ncol*N + 3*i + k] = -sum;
			}
		}
	}
}

/*================================================================================================
 goEnergy: computes the bonded energy for the Go potential
================================================================================================== */

void HessianGo::goEnergy(std::vector<Atoms>& atoms, double *newconf, int N, double K_bond, 
	double K_angle, double K_dih1, double K_dih3, 
	double *U_bond, double *U_angle, double *U_dih, double *U_Go)
{
	int i1, j1, k1, l1;
	double dx, dy, dz, d, r0, r;

/*================================================================================================
 	Bond energy for Go potential
================================================================================================== */

	double bond = 0;
	int nb = 0;
	for (int nbond = 0; nbond < N-1; nbond++)
	{
		i1 = nbond;
		j1 = nbond+1;

		if(atoms[i1].chainid == atoms[j1].chainid)
		{
			dx = atoms[i1].coord[0] - atoms[j1].coord[0];
			dy = atoms[i1].coord[1] - atoms[j1].coord[1];
			dz = atoms[i1].coord[2] - atoms[j1].coord[2];

			r0 = dx*dx + dy*dy + dz*dz;
			r0 = std::sqrt(r0);

			dx = newconf[3*i1] - newconf[3*j1];
			dy = newconf[3*i1+1] - newconf[3*j1+1];
			dz = newconf[3*i1+2] - newconf[3*j1+2];

			r  = dx*dx + dy*dy + dz*dz;
			r = std::sqrt(r);

			bond = bond + (r-r0)*(r-r0);
			nb++;
		}
	}
	bond = K_bond*bond;
	*U_bond = std::sqrt(bond/nb);

/*================================================================================================
 	Angle energy for Go potential
================================================================================================== */

	double a1, a2, a3, a4;
	double r1, r2, cosine;
	double theta0, theta;
	double d1[3], d2[3], d3[3];

	double ang = 0;
	for (int nang = 0; nang < N-2; nang++)
	{
		i1 = nang;
		j1 = nang+1;
		k1 = nang+2;

		if((atoms[i1].chainid == atoms[j1].chainid) && (atoms[i1].chainid == atoms[k1].chainid) ) 
		{

			for (int i = 0; i < 3; i++)
			{
				a1 = atoms[i1].coord[i];
				a2 = atoms[j1].coord[i];
				a3 = atoms[k1].coord[i];
				d1[i]   = a2 - a1;
				d2[i]   = a2 - a3;
			}
			r1 = d1[0]*d1[0] + d1[1]*d1[1] + d1[2]*d1[2];
			r2 = d2[0]*d2[0] + d2[1]*d2[1] + d2[2]*d2[2];
			d  = d1[0]*d2[0] + d1[1]*d2[1] + d1[2]*d2[2];
			r1 = std::sqrt(r1);
			r2 = std::sqrt(r2);
			cosine = d/(r1*r2);
			theta0 = std::acos(cosine);

			for (int i = 0; i < 3; i++)
			{
				a1 = newconf[3*i1+i];
				a2 = newconf[3*j1+i];
				a3 = newconf[3*k1+i];
				d1[i]   = a2 - a1;
				d2[i]   = a2 - a3;
			}
			r1 = d1[0]*d1[0] + d1[1]*d1[1] + d1[2]*d1[2];
			r2 = d2[0]*d2[0] + d2[1]*d2[1] + d2[2]*d2[2];
			d  = d1[0]*d2[0] + d1[1]*d2[1] + d1[2]*d2[2];
			r1 = std::sqrt(r1);
			r2 = std::sqrt(r2);
			cosine = d/(r1*r2);
			theta = std::acos(cosine);

			if( std::abs(theta-theta0) > 6) {
				std::cout << "Angle problem: i,j,k: " << i1 << " " << j1 << " " << k1 << " theta0, theta: " << theta0 << " " << theta << std::endl;
			}

			ang = ang + (theta-theta0)*(theta-theta0);
		}
	}
	ang = K_angle*ang;
	*U_angle = ang;

/*================================================================================================
 	Dihedral energy for Go potential
================================================================================================== */

	double xm[3], xn[3];
	double dih = 0;

	for (int ndih = 0; ndih < N-3; ndih++)
	{
		i1 = ndih;
		j1 = ndih+1;
		k1 = ndih+2;
		l1 = ndih+3;

		if((atoms[i1].chainid == atoms[j1].chainid) && (atoms[i1].chainid == atoms[k1].chainid) &&
		(atoms[i1].chainid == atoms[l1].chainid))
		{
			for (int i = 0; i < 3; i++)
			{
				a1 = atoms[i1].coord[i];
				a2 = atoms[j1].coord[i];
				a3 = atoms[k1].coord[i];
				a4 = atoms[l1].coord[i];
				d1[i]   = a2 - a1;
				d2[i]   = a2 - a3;
				d3[i]   = a4 - a3;
			}
			xm[0] = d1[1]*d2[2] - d1[2]*d2[1];
			xm[1] = -d1[0]*d2[2] + d1[2]*d2[0];
			xm[2] = d1[0]*d2[1] - d1[1]*d2[0];
			xn[0] = d2[1]*d3[2] - d2[2]*d3[1];
			xn[1] = -d2[0]*d3[2] + d2[2]*d3[0];
			xn[2] = d2[0]*d3[1] - d2[1]*d3[0];
			r1    = xm[0]*xm[0] + xm[1]*xm[1] + xm[2]*xm[2];
			r2    = xn[0]*xn[0] + xn[1]*xn[1] + xn[2]*xn[2];
			d     = xm[0]*xn[0] + xm[1]*xn[1] + xm[2]*xn[2];
			r1    = std::sqrt(r1);
			r2    = std::sqrt(r2);
			cosine = d/(r1*r2);
			theta0  = std::acos(cosine);

			for (int i = 0; i < 3; i++)
			{
				a1 = newconf[3*i1+i];
				a2 = newconf[3*j1+i];
				a3 = newconf[3*k1+i];
				a4 = newconf[3*l1+i];
				d1[i]   = a2 - a1;
				d2[i]   = a2 - a3;
				d3[i]   = a4 - a3;
			}
			xm[0] = d1[1]*d2[2] - d1[2]*d2[1];
			xm[1] = -d1[0]*d2[2] + d1[2]*d2[0];
			xm[2] = d1[0]*d2[1] - d1[1]*d2[0];
			xn[0] = d2[1]*d3[2] - d2[2]*d3[1];
			xn[1] = -d2[0]*d3[2] + d2[2]*d3[0];
			xn[2] = d2[0]*d3[1] - d2[1]*d3[0];
			r1    = xm[0]*xm[0] + xm[1]*xm[1] + xm[2]*xm[2];
			r2    = xn[0]*xn[0] + xn[1]*xn[1] + xn[2]*xn[2];
			d     = xm[0]*xn[0] + xm[1]*xn[1] + xm[2]*xn[2];
			r1    = std::sqrt(r1);
			r2    = std::sqrt(r2);
			cosine = d/(r1*r2);
			theta  = std::acos(cosine);

			dih = dih + K_dih1*std::cos(theta-theta0) +
			K_dih3*std::cos(3*(theta-theta0));
		}
	}
	*U_dih = dih;

	*U_Go = bond + ang + dih;

}

/*================================================================================================
 fullEigen : builds full eigen from its sparse representation, and diagonalize it using
	     LAPACK algorithms

================================================================================================== */

void HessianGo::fullEigen(int natoms, double *eigVal, double *eigVect) 
{
/*================================================================================================
	Input:
		natoms    : number of atoms
		List	  : list of all interaction pairs
		eigVect   : Full hessian
	Output:
		eigVal    : the 3*natoms eigenvalues
		eigVect   : the corresponding eigenvectors
================================================================================================== */

/*================================================================================================
	Declare some variables
================================================================================================== */

	int lwork, liwork, info;
	int isizeopt;
	int N = 3*natoms;

	double sizeopt;

	char U = 'U';
	char V = 'V';

/*================================================================================================
	Create all temporary arrays
================================================================================================== */

	lwork = -1;
	liwork = -1;
	dsyevd_(&V, &U, &N, eigVect, &N, eigVal, &sizeopt, &lwork, &isizeopt, &liwork, &info);

	lwork = (int) sizeopt;
	liwork = isizeopt;
	int *iwork = new int[liwork];
	double *dwork = new double[lwork];

/*================================================================================================
	Compute eigenvalues / eigenvectors
================================================================================================== */

	dsyevd_(&V, &U, &N, eigVect, &N, eigVal, dwork, &lwork, iwork, &liwork, &info);

/*================================================================================================
	Clean up all temporary arrays
================================================================================================== */

	delete [] dwork;
	delete [] iwork;

}

/*================================================================================================
  checkEigen
================================================================================================== */

void HessianGo::checkEigen(int N, int NE, double *hessian, double *eigVal, double *eigVect, 
	double *err, double *Emin, double *Emax, double *Emean, int nthreads)
{
/*================================================================================================
	Input:
		N         : number of rows in the matrix
		NE        : number of eigenpairs to be analyzed
		eigVal    : the NE eigenvalues
		eigVect   : the corresponding eigenvectors
		nthreads  : number of threads for parallel computation
	Output:
		err	  : list of errors
================================================================================================== */

/*================================================================================================
	Declare some variables
================================================================================================== */

	int inc = 1;

	double norm;
	double alpha;

/*================================================================================================
	Create all temporary arrays
================================================================================================== */

	double *Work     = new double[N];

/*================================================================================================
	Check each eigenpair
================================================================================================== */

	*Emean = 0;

	double beta = 0.0;
	char NoTrans = 'N';



	for(int i = 0; i < NE; i++)
	{
/*================================================================================================
		Check for convergence 
================================================================================================== */

		alpha = 1.0;
        	dgemv_(&NoTrans, &N, &N, &alpha, hessian, &N, 
                        &eigVect[N*i], &inc, &beta, Work, &inc);
		alpha = -eigVal[i];
		daxpy_(&N, &alpha, &eigVect[N*i], &inc, Work, &inc);
		norm = ddot_(&N, Work, &inc, Work, &inc);
		alpha = std::sqrt(norm);
		err[i] = alpha;
		*Emean = *Emean + alpha;
		if(i==0) {
			*Emin = alpha;
			*Emax = alpha;
		} else {
			*Emin = std::min(*Emin, alpha);
			*Emax = std::max(*Emax, alpha);
		}
	}
	*Emean = *Emean/NE;
	delete [] Work;
}

/*================================================================================================
 Build force constant array
================================================================================================== */

  void HessianGo::buildK(std::vector<Links> & ListPair, double *Kconst)
  {

	int npair = ListPair.size();

	for(int i = 0; i < npair; i++) {
		ListPair[i].kconst = Kconst[i];
	}

  }

/*================================================================================================
 Build force constant array
================================================================================================== */

  void HessianGo::buildK2(std::vector<Links> & ListPair, std::vector<Atoms>& atoms, 
	double *Kconst, int type)
  {

	int npair = ListPair.size();

	double kconstij, ki, kj;

	for(int i = 0; i < npair; i++) {
		int iat = ListPair[i].atm1;
		int jat = ListPair[i].atm2;
		ki = Kconst[iat];
		kj = Kconst[jat];
		atoms[iat].kconst = ki;
		atoms[jat].kconst = kj;
		if(type==1) {
			kconstij = (ki+kj)/2.;
		} else {
			kconstij = std::sqrt(ki*kj);
		}
		ListPair[i].kconst = kconstij;
	}
  }

/*================================================================================================
 Rescale eigvect: rescale eigenvectors based on mass of atoms
================================================================================================== */

void HessianGo::rescaleEigVect(std::vector<Atoms>& atoms, int NE, double *eigVect)
{
	int N = 3*atoms.size();
	double alpha;
	double *Temp = new double[N];

	for(int i = 0; i < N/3; i++) {
		alpha = 1.0/std::sqrt(atoms[i].mass);
		Temp[3*i] = alpha;
		Temp[3*i+1] = alpha;
		Temp[3*i+2] = alpha;
	}

	for(int i = 0; i < NE; i++)
	{
		for(int j = 0; j < N; j++) {
			eigVect[N*i + j] = eigVect[N*i +j]*Temp[j];
		}
	}

	delete [] Temp;
}
#endif
