c	Copyright (C) 2008 Patrice Koehl
c
c	This library is free software; you can redistribute it and/or
c	modify it under the terms of the GNU Lesser General Public
c	License as published by the Free Software Foundation; either
c	version 2.1 of the License, or (at your option) any later version.
c
c	This library is distributed in the hope that it will be useful,
c	but WITHOUT ANY WARRANTY; without even the implied warranty of
c	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
c	Lesser General Public License for more details.
c
c	You should have received a copy of the GNU Lesser General Public
c	License along with this library; if not, write to the Free Software
c	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
c
c ============================================================================
c ============================================================================
c
c       This set of programs computes the product f(A)X, where A is a symmetric
c       real matrix, X is a vector, and f a function
c       The standard way would be to diagonalise A:
c               A = V D tV
c       and then
c               f(A) = V f(D) tV
c       where f(D) is easily computed as being the matrix (f(d_i)) where
c       d_i are the eigenvalues of A.
c       Finally:
c               f(A)X = V f(D) tV X
c
c       But:
c       this is cumbersome when the matrix A is large, even if it is sparse,
c       as the matrix V is dense. Needs a lot of memory!!
c
c       We use a restarted Lanczos approximation instead.
c
c       This program follows the method described in:
c       "A restarted Lanczos approximation to functions of a symmetric
c       matrix" by Ilic, Turner and Simpson
c
c ============================================================================
c ============================================================================
c	ComputeMatVect.f
c
c       This program is the driver to the subroutines that computes the 
c       product f(A) X
c
c ============================================================================
c ============================================================================
c
        subroutine ComputeMatvect(func,ndim,spr_jdx,spr_row,
     1          spr_hessian,m,X,Y,V,T_eigvect,T_eigval)
c
c ============================================================================
c       Define input/output
c ============================================================================
c
c       Input:
c               ndim    : size of A, X and Y (A is supposed to be symmetric)
c
c               spr_jdx,spr_row,spr_hessian: A is stored in sparse matrix
c                                            format (in fact, row compressed)
c               m       : size of the Krylov subspace considered
c               X       : input vector
c
c       Output:
c               Y       : output vector such that Y = f(A)X
c
c
c ============================================================================
c       Declare variables
c ============================================================================
c
	integer	m,ndim
c
	integer spr_jdx(*)
	integer spr_row(*)
c
        real*8  beta,betam1,rho
        real*8  func
        real*8  dnrm2
c
	real*8	spr_hessian(*)
        real*8  v_mp1(*)
        real*8  X(ndim),Y(ndim)
        real*8  V(ndim,m)
        real*8  T_eigvect(m,m)
        real*8  T_eigval(m)
c
        external func, dnrm2
c
	pointer	(ptr_v_mp1,v_mp1)
c
	common /space_comp/	ptr_v_mp1
c
c ============================================================================
c       Compute approximation Y of f(A)X
c ============================================================================
c
c       Compute norm of input vector:
c       =============================
c
        beta = dnrm2(ndim,X,1)
c
c       Normalise input vector:
c       =======================
c
        call set_all_zeros(ndim,v_mp1)
        betam1 = 1.d0/beta
        call daxpy(ndim,betam1,X,1,v_mp1,1)
c
c       Initialise vector Y:
c       ====================
c
        call set_all_zeros(ndim,Y)
c
c       Compute Lanczos decomposition: get V, and eigenvalues
c       and eigenvectors of matrix T
c
        call compute_lanczos(ndim,spr_jdx,spr_row,spr_hessian,
     1                  m,rho,v_mp1,V,T_eigval,T_eigvect)
c
c       Update estimate of f(A)X
c
        call update_f(func,ndim,m,Y,beta,V,T_eigval,
     1          T_eigvect)
c
        return
        end
c
c ============================================================================
c ============================================================================
c	set_all_zeros.f
c
c       This subroutine initialises a vector to 0
c
c ============================================================================
c ============================================================================
c
        subroutine set_all_zeros(n,x)
c
        integer i,n
c
        real*8  x(n)
c
        do 100 i = 1,n
                x(i) = 0.d0
100     continue
c
        return
        end
c ============================================================================
c ============================================================================
c	Update_f.f
c
c       This subroutine updates the estimate of f(A)X, using the
c       relationship:
c
c       Y_new = Y + rho*V*T_vect*F_k*tT_vect*e1
c
c       where:
c               rho     is a scalar (initialised to the norm of X)
c               V       is the base of the Lanczos decomposition
c               T_vect  are the eigenvectors of the tridiagonal
c                       matrix T obtained from the Lanczos decomposition
c               F_k     is F(T_eigen), where T_eigen is the vector
c                       of eigenvalues of T, and F is the function
c                       considered
c               e1      is a vector with 1 as first element and 0 otherwise
c
c ============================================================================
c ============================================================================
c
        subroutine update_f(func,ndim,m,Y,rho,V,T_eig,T_vect)
c
        integer ndim,m,i
c
        real*8  func
        real*8  zero,rho
c
        real*8  Y(ndim)
        real*8  V(ndim,m),T_vect(m,m)
        real*8  T_eig(m)
        real*8  temp(*),temp2(*)
c
        character*1 trans
c
        external func
c
	pointer	(ptr_temp,temp)
	pointer (ptr_temp2,temp2)
c
	common /space_update/ ptr_temp,ptr_temp2
c
        trans = 'N'
        zero = 0.d0
c
        do 100 i = 1,m
                temp2(i) = 0.d0
                temp(i) = func(T_eig(i))*T_vect(1,i)
100     continue
c
        call dgemv(trans,m,m,1.d0,T_vect,m,temp,1,zero,
     1          temp2,1)
c
        call dgemv(trans,ndim,m,rho,V,ndim,temp2,1,zero,
     1          Y,1)
c
        return
        end
c
c ============================================================================
c ============================================================================
c
c	Compute_lanczos.f
c
c       This subroutine computes a base of the Krylov subspace
c       associated with a matrix A, using the Lanczos algorithm
c       provided in ARPACK
c
c       A V = V T + resid* t_e_m
c
c ============================================================================
c ============================================================================
c
        subroutine compute_lanczos(ndim,spr_jdx,spr_row,spr_hessian,
     1                  m,beta,v_seed,V,T_eigval,T_eigvect)
c
c ============================================================================
c
c       Input:
c               ndim    : size of the real symmetric matrix A
c               spr_jdx, spr_row : pointers for the sparse storage
c                                  of A (compressed row format)
c               spr_hessian      : non zero entries of A
c               m               : size of the Krylov subspace considered
c
c       Input/Output:
c               v_seed          : on input, v_seed is the vector
c                                 used to seed the Krylov subspace
c                                 on output, v_seed is the residual
c                                 that can be used for subsequent
c                                 call of compute_lanczos
c       Output:
c               beta            : norm of v_seed
c               V               : orthonormal base of the Krylov
c                                 subspace
c               T_eigval        : eigenvalues of the matrix T
c               T_eigvect       : eigenvectors of the matrix T
c
c
c ============================================================================
c     Routines called:
c
c     dsaupd  ARPACK reverse communication interface routine.
c     dseupd  ARPACK routine that returns Ritz values and (optionally)
c             Ritz vectors.
c     dnrm2   Level 1 BLAS that computes the norm of a vector.
c     daxpy   Level 1 BLAS that computes y <- alpha*x+y.
c
c ============================================================================
c Declarations:
c ============================================================================
c
c Local arrays
c ============================================================================
c
        real*8  workl(*)
        real*8  workd(*)
c
        integer iparam(11),ipntr(11)
c
c Local scalars
c ============================================================================
c
        integer i,nval
        integer ido,n,nev,ncv,lworkl,info,ierr,j,nx,ishfts
        integer maxitr,mode1
c
        real*8  tol
c
        character       bmat*1,which*2
c
c External functions
c ============================================================================
c
        real*8  dnrm2
c
c Input/Output:
c ============================================================================
c
        integer ndim,m
c
        integer spr_jdx(*),spr_row(*)
c
        real*8  beta
        real*8  spr_hessian(*)
        real*8  v_seed(*)
        real*8  V(ndim,m)
        real*8  T_eigval(m)
        real*8  T_eigvect(m,m)
c
	pointer (ptr_workl,workl)
	pointer	(ptr_workd,workd)
c
	common /space_lanczos/ ptr_workl,ptr_workd
c
c ============================================================================
c Set dimensions for this problem
c (see header in dsaupd for the meaning of each variable)
c ============================================================================
c
        nx = ndim
        n = nx
c
        nev = m/2
        ncv = m
c
        bmat = 'I'
        which = 'SM'
c
        lworkl = ncv*(ncv+8)
        tol = 1.d-10
        info = 1
        ido = 0
c
        ishfts = 1
        maxitr = 300
        mode1 = 1
c
        iparam(1) = ishfts
        iparam(3) = maxitr
        iparam(7) = mode1
c
c
c ============================================================================
c Main loop to get Lanczos decomposition
c ============================================================================
c
        i = 0
100     continue
                i = i + 1
c
c               Call dsauhd repeatedly to build Krylov subspace
c               ===================================================
c
                call dsauhd(ido,bmat,n,which,nev,tol,v_seed,ncv,V,
     1                  ndim,iparam,ipntr,workd,workl,lworkl,info)
c
c               Take action indicated by parameter IDO
c               ===================================================
c
                if(ido.eq.-1.or.ido.eq.1) then
c
c                       Perform matrix-vector multiplication
c                       ====================================
c
                        call sprsym_cr_mult(spr_hessian,spr_row,
     1                  spr_jdx,workd(ipntr(1)),workd(ipntr(2)),
     2                  ndim)
c
                        goto 100
c
                endif
c
c       We have either reached convergence, or there is an error
c       =========================================================
c
        if(info.lt.0) then
c
                write(6,*) ' '
                write(6,*) 'Error with dsaupd: info = ',info
                write(6,*) 'Check documentation in dsaupd'
                write(6,*) ' '
c
        else
c
c               No fatal error: now compute eigenvectors of T
c               ===============================================
c
                call dsteqr('Identity',ncv,workl(ncv+1),workl(2),
     1          workl(2*ncv+1),ncv,workl(2*ncv+ncv*ncv+1),ierr)
c
c               First, store eigenvalues of T:
c               ==============================
c
                nval = ncv
                do 200 i = 1,ncv
                        T_eigval(i) = workl(nval+i)
200             continue
c
c               Store eigenvectors of T:
c               ==========================
c
                nval = 2*ncv+1
                do 400 j = 1,ncv
                        do 300 i = 1,ncv
                                T_eigvect(i,j)=workl(nval)
                                nval = nval + 1
300                     continue
400             continue
c
                beta = dnrm2(ndim,v_seed,1)
c
c               Now we should be done
c               =====================
c
        endif
c
        return
        end
