/*================================================================================================
  Tools.h
  Version 1: 12/1/2017

Copyright (c) Patrice Koehl.

>>> SOURCE LICENSE >>>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

>>> END OF LICENSE >>>

================================================================================================== */

#ifndef _TOOLS_H_
#define _TOOLS_H_

/*================================================================================================
 Includes
================================================================================================== */

#include <math.h>
#include <cstdlib>

/*================================================================================================
  Prototypes for BLAS and LAPACK
================================================================================================== */

extern "C" {

	void daxpy_(int * n ,double *alpha , double * X, int *incx, double * Y,int *incy);
	double dnrm2_(int * n, double * X, int *incx);
	void dscal_(int * n, double * alpha, double * X, int *incx);
	void dcopy_(int * n, double * X, int *incx, double * Y, int *incy);
	double ddot_(int * n, double * u, int * incu, double * v, int *incv);

        void dsyev(char * JOBZ, char * UPLO, int *N, double *A, int *LDA, double *W,
        double *WORK, int *LWORK, int *INFO);

	void bestfitm_(double *coord1, int *natom1, double *coord2, int *natom2, int *natoms, 
	int *List1, int *List2, double *rmsd, int *ierr, double *mass1, double *mass2);
}

 class Tools {

	public:
		// Compute b-factor
		void computeBfact(std::vector<Atoms>& atoms, double *rigid, int nmode1, int nmode2, 
		double *eigVal, double *eigVect, double *bfact);

		// compare experimental and computed b-factors
		void compareBfact(std::vector<Atoms>& atoms, double *bfact, double *rms, double *correl);

		// compute overlap between structure differences and eigenvectors
		void computeOverlap(int nmode1, int nmode2, double *eigVect,
		std::vector<Atoms>& atoms1, std::vector<Atoms>& atoms2, double *rms0, 
		double *rms, double *overlap);

		double findScale(std::vector<Atoms>& atoms);

		void scaleBfact(std::vector<Atoms>& atoms, double scale);

		void scaleEig(int nmodes, double *eigVal, double scale);

		void computeCovar(int natoms, int nmode1, int nmode2, double *eigVal, double *eigVect,
		double *Covar);

		void computeCorrel(int natoms, double *Covar, double *Correl);

		void updateCovar(int natoms, double *Covar, double *X, double fact);


	private:

		double rigidBestfit(std::vector<Atoms>& atoms1, std::vector<Atoms>& atoms2, double *rms);

		void get3x3(int natoms, double *mat, int i, int j, double *mat3);

  };

/*================================================================================================
 Performs mass-weighted bestfit between two conformations of a molecule
================================================================================================== */

double Tools::rigidBestfit(std::vector<Atoms>& atoms1, std::vector<Atoms>& atoms2, double *rms)
{
	int natoms = atoms1.size();
	int N = 3*natoms;
	double *coord1 = new double[N];
	double *coord2 = new double[N];
	double *mass1  = new double[natoms];
	double *mass2  = new double[natoms];
	int *List      = new int[natoms];
	double rmsd;
	int ierr;

	for(int i = 0; i < natoms; i++)
	{
		for (int j = 0; j < 3; j++)
		{
			coord1[3*i+j] = atoms1[i].coord[j];
			coord2[3*i+j] = atoms2[i].coord[j];
		}
		mass1[i] = atoms1[i].mass;
		mass2[i] = atoms2[i].mass;
		List[i] = i+1;
	}

	bestfitm_(coord1, &natoms, coord2, &natoms, &natoms, List, List, &rmsd, &ierr, mass1, mass2);

	std::cout << " " << std::endl;
	std::cout << "rmsd Start - Target " << rmsd << std::endl;
	std::cout << " " << std::endl;

	*rms = rmsd;

	for(int i = 0; i < natoms; i++)
	{
		atoms2[i].setXYZ(coord2[3*i],coord2[3*i+1],coord2[3*i+2]);
	}

	delete [] coord1;
	delete [] coord2;
	delete [] mass1;
	delete [] mass2;
	delete [] List;

	return rmsd;
}

/*================================================================================================
 CompareBfact: compare computed and experimental B-factors
================================================================================================== */

void Tools::compareBfact(std::vector<Atoms>& atoms, double *bfact, double *rms, double *correl)
{
	double Se, See, Sc, Scc, Sec;
	double val_exp, val_cal;
	double val1, val2, val;

	Se  = 0;
	See = 0;
	Sc  = 0;
	Scc = 0;
	Sec = 0;

	int Natoms = atoms.size();

	for(int i = 0; i < Natoms; i++)
	{
		val_exp = atoms[i].bfact;
		val_cal = bfact[i];
		Se  = Se + val_exp;
		See = See + val_exp*val_exp;
		Sc  = Sc + val_cal;
		Scc = Scc + val_cal*val_cal;
		Sec = Sec + val_exp*val_cal;
	}

	if(See!=0 && Scc!=0) {
		*rms    = std::sqrt( (See+Scc-2*Sec)/Natoms );
		val1 = Natoms*See-Se*Se;
		val2 = Natoms*Scc-Sc*Sc;
		val = std::abs(val1*val2);
		val = std::sqrt(val);
		if(val==0) {
			*correl = 0;
		} else {
			*correl = (Natoms*Sec - Se*Sc)/val;
		}
	} else {
		*rms = 0.0;
		*correl = 0.0;
	}
}

/*================================================================================================
 ComputeBfact: compute B-factors and compare with experimental B-factors
================================================================================================== */

void Tools::computeBfact(std::vector<Atoms>& atoms, double *rigid, int nmode1, int nmode2, double *eigVal, double *eigVect,
	double *bfact)
{
	int Natoms = atoms.size();
	memset(bfact, 0, Natoms*sizeof(double));

	double kT = 0.593;
	double facb = 8.0*kT*M_PI*M_PI/3.0;

	double x,y,z,val, val2;

        double c[3];
	for(int i = 0; i < Natoms; i++)
	{
		for(int k = 0; k < 3; k++) c[k] = atoms[i].coord[k];
		val = rigid[0] + rigid[1]*c[0] + rigid[2]*c[1] + rigid[3]*c[2];
		val += rigid[4]*c[0]*c[0] + rigid[5]*c[0]*c[1] + rigid[6]*c[0]*c[2];
		val += rigid[7]*c[1]*c[1] + rigid[8]*c[1]*c[2] + rigid[9]*c[2]*c[2];
		bfact[i] = facb*val;
	}

	for (int i = nmode1; i < nmode2;i++)
	{
		val = facb/eigVal[i];
		for(int j = 0; j < Natoms; j++)
		{
			x = eigVect[i*3*Natoms+3*j];
			y = eigVect[i*3*Natoms+3*j+1];
			z = eigVect[i*3*Natoms+3*j+2];
			val2 = (x*x+y*y+z*z)*val;
			bfact[j] = bfact[j]+val2;
		}
	}
}

/*================================================================================================
 ComputeOverlap: compute overlap between modes and molecule displacement
================================================================================================== */

void Tools::computeOverlap(int nmode1, int nmode2, double *eigVect,
	std::vector<Atoms>& atoms1, std::vector<Atoms>& atoms2, double *rms0, double *rms, double *overlap)
{
	int Natoms = atoms1.size();
	int N = 3*Natoms;
	int inc = 1;
	double rms_val;

	rigidBestfit(atoms1, atoms2, &rms_val);
	*rms0 = rms_val;

	double *vect1 = new double[N];

	for(int i=0; i < Natoms; i++)
	{
		for(int j = 0; j < 3; j++)
		{
			vect1[3*i+j] = atoms2[i].coord[j]-atoms1[i].coord[j];
		}
	}

	double norm1 = dnrm2_(&N, vect1, &inc);

	for(int i = 0; i < nmode1; i++)
	{
		rms[i] = rms_val;
	}
	double sum_c2 = 0;
	double c, rms2;
	double norm2;

	for(int i = nmode1; i < nmode2; i++)
	{
		c = ddot_(&N, &eigVect[N*i], &inc, vect1, &inc);
		norm2 = dnrm2_(&N, &eigVect[N*i], &inc);
		overlap[i] = std::abs(c)/(norm1*norm2);
		sum_c2 = sum_c2 + c*c;
		rms2 = rms_val*rms_val - sum_c2/Natoms;
		rms2 = std::abs(rms2);
		rms[i] = std::sqrt(rms2);
	}

	delete [] vect1;
}

/*================================================================================================
  FInd scale for Bfactors, so that they are in [0, 100];
================================================================================================== */

  double Tools::findScale(std::vector<Atoms>& atoms) 
  {	
	int natoms = atoms.size();
	double maxB=0;
	for(int i = 0; i < natoms; i++) {
		maxB = std::max(maxB, atoms[i].bfact);
	}

	double scale = 100./maxB;
	return scale;
  }
	
/*================================================================================================
  scale Bfactors by "scale"
================================================================================================== */

  void Tools::scaleBfact(std::vector<Atoms>& atoms, double scale) 
  {
	int natoms = atoms.size();
	for(int i = 0; i < natoms; i++) {
		atoms[i].bfact *= scale;
	}
  }

/*================================================================================================
  scale eigen values
================================================================================================== */

  void Tools::scaleEig(int nmodes, double *eigval, double scale) 
  {
	for(int i = 0; i < nmodes; i++) {
		eigval[i] *= scale;
	}
  }

/*================================================================================================
  Builds covariance matrix 
================================================================================================== */

  void Tools::computeCovar(int natoms, int nmode1, int nmode2, double *eigVal, double *eigVect,
		double *Covar)
  {

	int N = 3*natoms;
	int inc = 1;

	double *X = new double[N];

	memset(Covar, 0, 9*natoms*natoms*sizeof(double));

	double fact;
	for(int k = nmode1; k < nmode2; k++) {

		dcopy_(&N, &eigVect[k*N], &inc, X, &inc);
		fact = 1.0/eigVal[k];
		updateCovar(natoms, Covar, X, fact);

	}

	delete [] X;
  }


/*================================================================================================
  Builds correlation matrix from covaration matrix
================================================================================================== */

  void Tools::computeCorrel(int natoms, double *Covar, double *Correl)
  {

	double *diag = new double[natoms];
	double *mat3 = new double[9];
	for(int i = 0; i < natoms; i++) {
		get3x3(natoms, Covar, i, i, mat3);
		diag[i] = mat3[0] + mat3[4] + mat3[8];
	}

	double Dij;
	for(int j = 0; j < natoms; j++) {
		for(int i = 0; i < natoms; i++) {
			get3x3(natoms, Covar, i, j, mat3);
			Dij = mat3[0] + mat3[4] + mat3[8];
			Correl[i+j*natoms] = Dij/std::sqrt(diag[i]*diag[j]);
		}
	}

	delete [] diag;
  }

/*================================================================================================
 Get small 3x3 matrix from full covariance matrix
================================================================================================== */

void Tools::get3x3(int natoms, double *mat, int i, int j, double *mat3)
{
	int N = 3*natoms;
	int i1=3*i;
	int j1=3*j;
	mat3[0] = mat[j1*N + i1 ];
	mat3[1] = mat[j1*N + i1 + 1];
	mat3[2] = mat[j1*N + i1 + 2];
	mat3[3] = mat[j1*N + N + i1 ];
	mat3[4] = mat[j1*N + N + i1 + 1];
	mat3[5] = mat[j1*N + N + i1 + 2];
	mat3[6] = mat[j1*N + 2*N + i1 ];
	mat3[7] = mat[j1*N + 2*N + i1 + 1];
	mat3[8] = mat[j1*N + 2*N + i1 + 2];
}

/*================================================================================================
 Update full matrix
================================================================================================== */

void Tools::updateCovar(int natoms, double *Covar, double *X, double fact)
{
	int N = 3*natoms;
	for(int j = 0; j < 3*natoms; j++) {
		for(int i = 0; i < 3*natoms; i++) {
			Covar[i+j*N] += fact*X[i]*X[j];
		}
	}
}
#endif
