/* ===============================================================================================
  ConformalError.h

   Small functions that checks the conformality of a mesh given an input mesh

   Author:  Patrice Koehl
   Date:    06/05/2019
   Version: 1
   =============================================================================================== */

#pragma once

  /* ===============================================================================================
   Includes
   =============================================================================================== */

  #include <vector>
  #include <algorithm>

  /* =============================================================================================
     Define class
   =============================================================================================== */

  class ConformalError{

  public:
	// Checks cross ratios and angles
	static void checkConformal(Mesh& mesh);

	// Cross ratio error (mean and max)
	static double crossRatioError(Mesh& mesh, double *errMax);

	// quasi conformal error (whole mesh)
	static Vector quasiConformalError(Mesh& mesh);

  private:
	// Angular error (mean and max)
	static double angleError(Mesh& mesh, double *errMax);

	// quasi conformal error (one triangle)
	static Vector quasiConformalError(std::vector<Vector>& p1, std::vector<Vector>& p2);

  };

  /* =============================================================================================
     main function: check cross ratios and angles
   =============================================================================================== */

  void ConformalError::checkConformal(Mesh& mesh)
  {
	double maxc, maxa;
	double crossdefect = crossRatioError(mesh, &maxc);
	double angledefect = angleError(mesh, &maxa);
	Vector r = quasiConformalError(mesh);
	double qmean = r[2];
	double qmax  = r[1];

	std::cout << "Checking conformality of final map: " << std::endl;
	std::cout << "===================================" << std::endl;
	std::cout << " " << std::endl;
	std::cout << "RMS defect for stretch over all triangles   : " << qmean << std::endl;
	std::cout << "Largest quasi conformal error               : " << qmax << std::endl;
	std::cout << "RMS defect for cross ratios                 : " << crossdefect << std::endl;
	std::cout << "Largest defect for cross ratios             : " << maxc << std::endl;
	std::cout << "Average defect for angles                   : " << angledefect << std::endl;
	std::cout << "Largest defect for angles                   : " << maxa << std::endl;
	std::cout << " " << std::endl;

 }
  /* ===== Cross ratio Defects ======================================================================
   * Computed by considering the two triangles incident to (ij): (ijk) and (ijl)
   * c_ij = ( lik/ljk) * (ljl/lil)
   *
   * From B. Springborn, P. Shroeder, and U. Pinkall (2008). Conformal equivalence of triangular meshes.
   *	  ACM Trans. On Graphics, 
   ==================================================================================================*/

  double ConformalError::crossRatioError(Mesh& mesh, double *errMax)
  {

	double l_ik, l_jk;
	double l_il, l_jl;
	double l_ik0, l_jk0;
	double l_il0, l_jl0;

	double CR, CR0;

	int n_edge = mesh.edges.size();

	double diff;
	double dmax = 0;
	double defect = 0;

        HalfEdgeIter he, he2;
        VertexIter v_i, v_j;
	Vector a0, b0, c0, d0;
	Vector a, b, c, d;

	/* Iterate over all edges */
	for (EdgeIter e = mesh.edges.begin(); e != mesh.edges.end(); e++)
	{
		he = e->he;
		he2 = he->flip;

		v_i = he->vertex;
		v_j = he2->vertex;

		a0 = v_i->position;
		b0 = he->next->vertex->position;
		c0 = he->prev->vertex->position;
		d0 = he2->prev->vertex->position;

		l_jk0 = (b0-c0).norm();
		l_ik0 = (a0-c0).norm();
		l_jl0 = (b0-d0).norm();
		l_il0 = (a0-d0).norm();

		a = v_i->position2;
		b = he->next->vertex->position2;
		c = he->prev->vertex->position2;
		d = he2->prev->vertex->position2;

		l_jk = (b-c).norm();
		l_ik = (a-c).norm();
		l_jl = (b-d).norm();
		l_il = (a-d).norm();

		if(l_jk > 0 && l_il > 0) {

			/* Calculate cross ratio CR */
			CR = (l_ik/l_jk) * (l_jl/l_il);
			CR0 = (l_ik0/l_jk0) * (l_jl0/l_il0);

			diff = std::abs(CR-CR0);
		} else {
			diff = 0.0;
		}

		dmax = std::max(dmax, diff);
		defect = defect + diff*diff;

	}
	defect = std::sqrt( defect/n_edge);
	*errMax  = dmax;
	return defect;
  }

  /* ===============================================================================================
   AngleDefect: computes the angle defects between two representations of a mesh

   For a triangle (i,j,k), computes 3 angles a0,b0,c0 and a,b,c for the two representations
   of the mesh; angle defect is : |a0-a| + |bo-b| + |c0-c|
   The angles are given as frations of the total angle at a vertex

   From:
	L. Ju, M.K. Hurdal, J. Stern, K. Rehm, K. Schaper, D. Rottenberg (2005)
	Quantitative evaluation of three cortical surface flattening methods, 28, 869-880.
   =============================================================================================== */

  double ConformalError::angleError(Mesh& mesh, double *maxA)
  {

	double lAB, lBC, lCA;
	double lAB0, lBC0, lCA0;
	double alpha_A, alpha_B, alpha_C;
	double alpha_A0, alpha_B0, alpha_C0;
	double val;

	int n_vertices = mesh.vertices.size();
	double *angles = new double[n_vertices];
	double *angles0 = new double[n_vertices];

	HalfEdgeIter hAB, hBC, hCA;
	VertexIter v1, v2, v3;
	int idxA, idxB, idxC;
	Vector a,b,c;
	Vector a0,b0,c0;

	for(int i = 0; i < n_vertices; i++) {
		angles[i] = 0.0;
		angles0[i] = 0.0;
	}

	for(FaceIter f_iter = mesh.faces.begin(); f_iter != mesh.faces.end(); f_iter++)
	{
		hAB =f_iter->he;
		hBC =hAB->next;
		hCA =hBC->next;

		v1 = hAB->vertex;
		v2 = hBC->vertex;
		v3 = hCA->vertex;

		idxA = v1->index;
		idxB = v2->index;
		idxC = v3->index;

		a = v1->position2;
		b = v2->position2;
		c = v3->position2;
		lAB = (a-b).norm();
		lBC = (b-c).norm();
		lCA = (c-a).norm();

		if(lAB == 0 || lBC == 0 || lCA == 0) continue;

		a0 = v1->position;
		b0 = v2->position;
		c0 = v3->position;
		lAB0 = (a0-b0).norm();
		lBC0 = (b0-c0).norm();
		lCA0 = (c0-a0).norm();

		val = (lAB0*lAB0 + lCA0*lCA0 - lBC0*lBC0)/(2.0*lAB0*lCA0);
		if(val > 1.0) val = 1.0; if (val < -1.0) val = -1.0;
		alpha_A0 = std::acos(val);
		val = (lAB0*lAB0 + lBC0*lBC0 - lCA0*lCA0)/(2.0*lAB0*lBC0);
		if(val > 1.0) val = 1.0; if (val < -1.0) val = -1.0;
		alpha_B0 = std::acos(val);
		val = (lCA0*lCA0 + lBC0*lBC0 - lAB0*lAB0)/(2.0*lCA0*lBC0);
		if(val > 1.0) val = 1.0; if (val < -1.0) val = -1.0;
		alpha_C0 = std::acos(val);

		val = (lAB*lAB + lCA*lCA - lBC*lBC)/(2.0*lAB*lCA);
		if(val > 1.0) val = 1.0; if (val < -1.0) val = -1.0;
		alpha_A = std::acos(val);
		val = (lAB*lAB + lBC*lBC - lCA*lCA)/(2.0*lAB*lBC);
		if(val > 1.0) val = 1.0; if (val < -1.0) val = -1.0;
		alpha_B = std::acos(val);
		val = (lCA*lCA + lBC*lBC - lAB*lAB)/(2.0*lCA*lBC);
		if(val > 1.0) val = 1.0; if (val < -1.0) val = -1.0;
		alpha_C = std::acos(val);

		angles[idxA] += alpha_A; angles[idxB] += alpha_B; angles[idxC] += alpha_C;
		angles0[idxA] += alpha_A0; angles0[idxB] += alpha_B0; angles0[idxC] += alpha_C0;

	}

	double diff;
	double diffA, diffB, diffC;
	double totdefect = 0;
	int nface = mesh.faces.size();
	double dmax = 0;

	for(FaceIter f_iter = mesh.faces.begin(); f_iter != mesh.faces.end(); f_iter++)
	{
		hAB =f_iter->he;
		hBC =hAB->next;
		hCA =hBC->next;

		v1 = hAB->vertex;
		v2 = hBC->vertex;
		v3 = hCA->vertex;

		idxA = v1->index;
		idxB = v2->index;
		idxC = v3->index;

		if(angles[idxA] == 0 || angles[idxB] == 0 || angles[idxC] == 0) continue;

		a = v1->position2;
		b = v2->position2;
		c = v3->position2;
		lAB = (a-b).norm();
		lBC = (b-c).norm();
		lCA = (c-a).norm();

		if(lAB == 0 || lBC == 0 || lCA == 0) continue;

		a0 = v1->position;
		b0 = v2->position;
		c0 = v3->position;
		lAB0 = (a0-b0).norm();
		lBC0 = (b0-c0).norm();
		lCA0 = (c0-a0).norm();

		val = (lAB0*lAB0 + lCA0*lCA0 - lBC0*lBC0)/(2.0*lAB0*lCA0);
		if(val > 1.0) val = 1.0; if (val < -1.0) val = -1.0;
		alpha_A0 = std::acos(val)/angles0[idxA];
		val = (lAB0*lAB0 + lBC0*lBC0 - lCA0*lCA0)/(2.0*lAB0*lBC0);
		if(val > 1.0) val = 1.0; if (val < -1.0) val = -1.0;
		alpha_B0 = std::acos(val)/angles0[idxB];
		val = (lCA0*lCA0 + lBC0*lBC0 - lAB0*lAB0)/(2.0*lCA0*lBC0);
		if(val > 1.0) val = 1.0; if (val < -1.0) val = -1.0;
		alpha_C0 = std::acos(val)/angles0[idxC];

		val = (lAB*lAB + lCA*lCA - lBC*lBC)/(2.0*lAB*lCA);
		if(val > 1.0) val = 1.0; if (val < -1.0) val = -1.0;
		alpha_A = std::acos(val)/angles[idxA];
		val = (lAB*lAB + lBC*lBC - lCA*lCA)/(2.0*lAB*lBC);
		if(val > 1.0) val = 1.0; if (val < -1.0) val = -1.0;
		alpha_B = std::acos(val)/angles[idxB];
		val = (lCA*lCA + lBC*lBC - lAB*lAB)/(2.0*lCA*lBC);
		if(val > 1.0) val = 1.0; if (val < -1.0) val = -1.0;
		alpha_C = std::acos(val)/angles[idxC];

		diffA = std::abs(alpha_A0-alpha_A);
		diffB = std::abs(alpha_B0-alpha_B);
		diffC = std::abs(alpha_C0-alpha_C);
		dmax = std::max(dmax, diffA);
		dmax = std::max(dmax, diffB);
		dmax = std::max(dmax, diffC);
		diff = diffA + diffB + diffC;

		totdefect += diff;
	}

	delete [] angles; delete [] angles0;

	totdefect = totdefect/(3*nface);
	*maxA = dmax;
	return totdefect;

  }

  /* ===============================================================================================
  Quasi conformal error for a face
   =============================================================================================== */

Vector ConformalError::quasiConformalError(std::vector<Vector>& p, std::vector<Vector>& q)
{
	// compute edge vectors
	Vector u1 = p[1] - p[0];
	Vector u2 = p[2] - p[0];

	Vector v1 = q[1] - q[0];
	Vector v2 = q[2] - q[0];

	// compute orthonormal bases
	Vector e1 = u1; e1.normalize();
	Vector e2 = (u2 - dot(u2, e1)*e1); e2.normalize();

	Vector f1 = v1; f1.normalize();
	Vector f2 = (v2 - dot(v2, f1)*f1); f2.normalize();

	// project onto bases
	p[0] = Vector(0, 0, 0);
	p[1] = Vector(dot(u1, e1), dot(u1, e2), 0);
	p[2] = Vector(dot(u2, e1), dot(u2, e2), 0);

	q[0] = Vector(0, 0, 0);
	q[1] = Vector(dot(v1, f1), dot(v1, f2), 0);
	q[2] = Vector(dot(v2, f1), dot(v2, f2), 0);

	double A = 2.0*cross(u1, u2).norm();

	Vector Ss = (q[0]*(p[1].y - p[2].y) + q[1]*(p[2].y - p[0].y) + q[2]*(p[0].y - p[1].y)) / A;
	Vector St = (q[0]*(p[2].x - p[1].x) + q[1]*(p[0].x - p[2].x) + q[2]*(p[1].x - p[0].x)) / A;
	double a = dot(Ss, Ss);
	double b = dot(Ss, St);
	double c = dot(St, St);
	double det = sqrt(pow(a-c, 2) + 4.0*b*b);
	double Gamma = sqrt(0.5*(a + c + det));
	double gamma = sqrt(0.5*std::abs((a + c - det)));

	if (Gamma < gamma) std::swap(Gamma, gamma);

	double L2A = 0.5*(a+c);
	double Linf = Gamma;

	Vector r = Vector(Gamma/gamma, L2A, Linf);
	return r;
}

  /* ===============================================================================================
  Quasi conformal error for a whole mesh
   =============================================================================================== */

Vector ConformalError::quasiConformalError(Mesh& model)
{
	double totalQc = 0.0;
	double maxQc = -std::numeric_limits<double>::infinity();
	double totalArea = 0.0;
	double TOL = 1.e-8;

	std::vector<Vector> p(3), q(3);
	Vector r;
	for (FaceCIter f = model.faces.begin(); f != model.faces.end(); f++) {
		HalfEdgeCIter he = f->he;
		int j = 0;
		do {
			p[j] = he->vertex->position;
			q[j] = he->vertex->position2;
			j++;
			he = he->next;
		} while(he != f->he);

		double area0 = cross(p[1]-p[0], p[2]-p[0]).norm();
		double area = cross(q[1]-q[0], q[2]-q[0]).norm();
		if(area0 > TOL && area > TOL) {
			r = quasiConformalError(p, q);
		} else {
			r = Vector(0.,0.,0.);
		}
		totalArea += area;
		totalQc += r[0]*area;
		maxQc = std::max(maxQc, r[2]);
	}

//	totalQc= std::sqrt(totalQc/totalArea);
	return Vector(0, maxQc, totalQc/totalArea);
}
