/* ===============================================================================================
   Map2Sphere: Conformal mapping of a genus zero surface represented by a triangular mesh onto
               the sphere.
               Different methods are implemented:
		1) Intrinsic methods:
			1.a: Tutte embedding
			1.b: Yamabe flow (Springborn et al, 2007)
			1.c: Ricci flow
                2) Extrinsic method:
			2.a: conformalized Mean Curvature Flow

   Author:  Patrice Koehl
   Date:    5/10/2019
   Version: 1
   =============================================================================================== */

/* ===============================================================================================
   Includes
   =============================================================================================== */

#include "Map2Sphere-CL.h"

/* ===============================================================================================
   Main program
   =============================================================================================== */

int main(int argc, char **argv)
{

/*	==========================================================================================
	Show usage if needed
	========================================================================================== */

	if( argc < 2 )
	{
		usage(argv);
		return -1;
	}

	std::string input = argv[1];
	if( input == "-h" || input == "-help" )
	{
		usage(argv);
		return -1;
	}

/*	==========================================================================================
	Read in all inputs (some values may have been preset)
	========================================================================================== */

	std::string INfile;
	std::string OUTfile="test.off";
	int ntype = 1;
	int nimp = 0;
	double dt = 0.1;

        if (!parse_args(argc, argv, &INfile, &OUTfile, &ntype, &nimp, &dt)) return 1;

/*	==========================================================================================
	Read in mesh from input file
	========================================================================================== */

	Mesh model;

	std::string error;
	std::size_t found = INfile.find("obj");
	if(found !=std::string::npos) {
		MeshIO::readOBJ(INfile, model, error);
	} else {
		found = INfile.find("off");
		if(found !=std::string::npos) {
			MeshIO::readOFF(INfile, model, error);
/*			for(VertexIter v = model.vertices.begin(); v != model.vertices.end(); v++) {
				v->position2 = v->position;
			}
			MeshIO::write(OUTfile, model);
*/
		} else {
			std::cout << " " << std::endl;
			std::cout << "Input file format not recognized; program can only read OFF and OBJ files" << std::endl;
			std::cout << " " << std::endl;
			exit(1);
		}
	}

	int n3 = 0;
	for(VertexIter v = model.vertices.begin(); v!= model.vertices.end(); v++) {
		if(v->degree() <= 3) n3++;
	}
	std::cout << "Number of vertices of degree 3: " << n3 << std::endl;

/*	==========================================================================================
	Characteristics of the mesh
	========================================================================================== */

	int nvertices = model.vertices.size();
	int nfaces    = model.faces.size();
	int nbound    = model.boundaries.size();
	int euler     = model.eulerCharacteristic();
	double genus  = (2-euler)/2;

	if(nbound>0) {
		std::cout << "here ..." << std::endl;
		for (BoundaryIter b = model.boundaries.begin(); b != model.boundaries.end(); b++) {
			std::cout << "Vertex on boundary: " << b->he->vertex->index << std::endl;
		}
	}

	std::cout << " " << std::endl;
	std::cout << "Number of vertices in mesh         : " << nvertices << std::endl;
	std::cout << "Number of faces in mesh            : " << nfaces << std::endl;
	std::cout << "Number of boundaries in mesh       : " << nbound << std::endl;
	std::cout << "Euler characteristics              : " << euler << std::endl;
	std::cout << "Genus                              : " << genus << std::endl;

	double Area, Vol, Sphericity;
	MeshGeometry::measureMesh(model, &Area, &Vol);
	Sphericity = std::pow(M_PI,1.0/3.0) * std::pow(6.0*Vol,2.0/3.0) / Area;
			
	std::cout << " " << std::endl;
	std::cout << "Total surface area of scaled mesh  : " << Area << std::endl;
	std::cout << "Total volume of the mesh           : " << Vol << std::endl;
	std::cout << "Sphericity of the mesh             : " << Sphericity << std::endl;
	std::cout << " " << std::endl;

	double lmin = 100, lmax=0;
	double length;
	for(EdgeIter e = model.edges.begin(); e != model.edges.end(); e++)
	{
		length = (e->he->vertex->position - e->he->flip->vertex->position).norm();
		lmin = std::min(length, lmin);
		lmax = std::max(length, lmax);
	}
	double fmin = Area; double fmax = 0;
	double area;
	for(FaceIter f = model.faces.begin(); f != model.faces.end(); f++)
	{
		area = f->area();
		if(area < 1.e-15) {
			std::cout << "warning: very small face with close to 0 surface area!" << std::endl;
			VertexIter v0 = f->he->vertex;
			VertexIter v1 = f->he->next->vertex;
			VertexIter v2 = f->he->next->next->vertex;
			int i0 = v0->index;
			int i1 = v1->index;
			int i2 = v2->index;
			Vector p0 = v0->position;
			Vector p1 = v1->position;
			Vector p2 = v2->position;
			std::cout << "Vertices: " << i0 << " " << i1 << " " << i2 << " " << std::endl;
			std::cout << "Vertex 1: " << p0[0] << " " << p0[1] << " " << p0[2] << std::endl;
			std::cout << "Vertex 2: " << p1[0] << " " << p1[1] << " " << p1[2] << std::endl;
			std::cout << "Vertex 3: " << p2[0] << " " << p2[1] << " " << p2[2] << std::endl;
		}
		fmin = std::min(fmin, f->area());
		fmax = std::max(fmax, f->area());
	}
	std::cout << "Smallest edge: " << lmin << std::endl;
	std::cout << "Largest edge : " << lmax << std::endl;
	std::cout << "Smallest face: " << fmin << std::endl;
	std::cout << "Largest face : " << fmax << std::endl;
	std::cout << " " << std::endl;

/*	==========================================================================================
	Perform parametrization:
		ntype= 1: 	Barycentric Tutte
		ntype= 2:	cotan Tutte
		ntype= 3:	Mean Value Tutte
		ntype= 4:	Authalic Tutte
		ntype= 5:	Ricci Flow
		ntype= 6:	Yamabe Flow
		ntype= 7:	Yamabe Flow (with iDT)
		ntype= 8:	cMCF Flow
		ntype= 9:	Willmore flow
	========================================================================================== */

	Tutte tutte;
	Ricci ricci;
	Yamabe yamabe;
	YamabeIDT yamabeIDT;
	cMCF mcf;
	Willmore willmore;

	int v_type = 0;
	VertexIter pole;
	bool SUCCESS = true;
	if(ntype< 5) {
		pole = tutte.planarTutte(model, v_type, ntype, &SUCCESS);
	} else if(ntype==5) {
		pole = ricci.euclideanRicci(model, v_type, &SUCCESS);
	} else if(ntype==6) {
		pole = yamabe.yamabeFlow(model, v_type, &SUCCESS);
	} else if(ntype==7) {
		pole = yamabeIDT.yamabeFlowIDT(model, v_type, &SUCCESS);
	} else if(ntype==8) {
		int niter_max = 1000;
		double TOL = 1.e-6;
		int ftype = 0;
		int type = 0;
		double S, Q;
		double Q_old = 0.0;
		Vector r;
		mcf.initFlow(model, ftype, dt);
		for(int i = 0; i < niter_max; i++) {
			r = mcf.solveOneStep(model, i, dt, type);
			S = r[0]; Q = r[1];
			if(std::abs(Q-Q_old) < TOL) break;
			Q_old = Q;
		}
		mcf.stopFlow(model);
		if(std::abs(S-1.0) < TOL) SUCCESS = true;
	} else if(ntype==9) {
		int niter_max = 50;
		double TOL1 = 1.e-6;
		double TOL2 = 1.e-3;
		double S, Q;
		double Q_old = 0.0;
		int btype = 0;
		Vector r;
		std::cout << "dt = " << dt << std::endl;
		willmore.initFlow(model, dt);

		for(int i = 0; i < niter_max; i++) {
			r = willmore.solveOneStep(model, i, dt, btype);
			S = r[0]; Q = r[1];
			if(std::abs(Q-Q_old) < TOL1 || (std::abs(S-1.0) < TOL2) ) break;
			Q_old = Q;
		}
		willmore.stopFlow(model);
		if(std::abs(S-1.0) < TOL2) SUCCESS = true;

	}
//	exit(1);

/*	==========================================================================================
	Normalize and check conformality
	========================================================================================== */

	if(SUCCESS) {
		Normalize::normalize(model);
		ConformalError::checkConformal(model);
//		double scale = std::sqrt(Area/(4*M_PI));
//		mcf.scaleMesh(model, scale);
	}

/*	==========================================================================================
	Improve conformality with discrete Beltrami flow
	========================================================================================== */

	Beltrami beltrami;

	if(SUCCESS && nimp==1) {
		beltrami.beltramiFlow(model);
		ConformalError::checkConformal(model);
	} else if(SUCCESS && nimp==2) {
		int niter_max = 2;
		double TOL = 1.e-6;
		double dt2 = 0.2;
		int ftype = 1;
		int type = 0;
		double S, Q;
		double Q_old = 0.0;
		Vector r;
		mcf.initFlow(model, ftype, dt);
		for(int i = 0; i < niter_max; i++) {
			r = mcf.solveOneStep(model, i, dt2, type);
			S = r[0]; Q = r[1];
			if(std::abs(Q-Q_old) < TOL) break;
//			if(std::abs(S-1.0) < TOL) break;
			Q_old = Q;
		}
		mcf.stopFlow(model);
		ConformalError::checkConformal(model);
	}

/*	==========================================================================================
	Write file
	========================================================================================== */

	if(SUCCESS) MeshIO::write(OUTfile, model);

	return 0;

}

/* ===============================================================================================
   Usage
   =============================================================================================== */

static void usage(char** argv)
{
    std::cout << "\n\n" <<std::endl;
    std::cout << "     " << "================================================================================================"<<std::endl;
    std::cout << "     " << "================================================================================================"<<std::endl;
    std::cout << "     " << "=                                                                                              ="<<std::endl;
    std::cout << "     " << "=                                         Map2Sphere                                           ="<<std::endl;
    std::cout << "     " << "=                                                                                              ="<<std::endl;
    std::cout << "     " << "=     This program reads in a 3D surface represented by a triangular mesh, checks that         ="<<std::endl;
    std::cout << "     " << "=     it is a manifold and that its genus is 0, and map it conformally onto the sphere         ="<<std::endl;
    std::cout << "     " << "=                                                                                              ="<<std::endl;
    std::cout << "     " << "=     Usage is:                                                                                ="<<std::endl;
    std::cout << "     " << "=          Map2Sphere.exe -i INFILE -o OUTFILE -n type                                         ="<<std::endl;
    std::cout << "     " << "=     where:                                                                                   ="<<std::endl;
    std::cout << "     " << "=                 -i INFILE       --> Input Mesh file (OBJ or OFF format)                      ="<<std::endl;
    std::cout << "     " << "=                 -o OUTFILE      --> Ouput Mesh file in OFF format                            ="<<std::endl;
    std::cout << "     " << "=                 -n type         --> Parametrization type:                                    ="<<std::endl;
    std::cout << "     " << "=                                      n = 1: Tutte, barycentric                               ="<<std::endl;
    std::cout << "     " << "=                                      n = 2: Tutte, cotan weights                             ="<<std::endl;
    std::cout << "     " << "=                                      n = 3: Tutte, mean value                                ="<<std::endl;
    std::cout << "     " << "=                                      n = 4: Tutte, authalic                                  ="<<std::endl;
    std::cout << "     " << "=                                      n = 5: Ricci flow                                       ="<<std::endl;
    std::cout << "     " << "=                                      n = 6: Yamabe flow                                      ="<<std::endl;
    std::cout << "     " << "=                                      n = 7: Yamabe flow (with iDT)                           ="<<std::endl;
    std::cout << "     " << "=                                      n = 8: cMCF                                             ="<<std::endl;
    std::cout << "     " << "=                                      n = 9: conf. Willmore flow                              ="<<std::endl;
    std::cout << "     " << "=                 -r reftype      --> refinement type:                                         ="<<std::endl;
    std::cout << "     " << "=                                      r = 0: No refinement                                    ="<<std::endl;
    std::cout << "     " << "=                                      r = 1: quasi conformal discrete flow                    ="<<std::endl;
    std::cout << "     " << "=                                      r = 2: cMCF flow                                        ="<<std::endl;
    std::cout << "     " << "=                 -t time step    --> time step for flow methods (def. 0.1)                    ="<<std::endl;
    std::cout << "     " << "================================================================================================"<<std::endl;
    std::cout << "     " << "================================================================================================"<<std::endl;
    std::cout << "\n\n" <<std::endl;
}

/* ===============================================================================================
   Parse Argument from command line:

   =============================================================================================== */

bool parse_args(int argc, char **argv, std::string *INfile, std::string *OUTfile, int *ntype, int *rtype, double *dt) 
{
//
// Make sure we have at least two parameters....
//
	std::string param;
	if (argc == 1)
	{
		return false;
	}
	else
	{
		for (int i = 1; i < argc - 1; i = i + 2)
		{
			param = argv[i];

			if (param == "-i") {
				*INfile = argv[i + 1];
			}
			else if (param == "-o") {
				*OUTfile = argv[i + 1];
			}
			else if (param == "-n") {
				*ntype = std::atoi(argv[i + 1]);
			}
			else if (param == "-r") {
				*rtype = std::atoi(argv[i + 1]);
			}
			else if (param == "-t") {
				*dt = std::atof(argv[i + 1]);
			}
		}
  	}
	return true;
}
