/* trackball.c    */

/* E. Angel, Interactive Computer Graphics */
/* A Top-Down Approach with OpenGL, Third Edition */
/* Addison-Wesley Longman, 2003 */

/* Alias obj reader based on glm.c/glm.h */

/* Last modified: 1/27/05 */
/* O. Staadt, UC Davis */

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <GL/glut.h>
#include "glm.h"

#define bool int
#define false 0
#define true 1
#define M_PI 3.14159


//globals
int 	winWidth, winHeight;

float 	angle = 0.0, axis[3], trans[3];
bool 	trackingMouse = false;
bool 	redrawContinue = false;
bool    trackballMove = false;

char*	model_file = NULL;
GLMmodel	*model;



//init function
void func_init(void)
{
	model = glmReadOBJ(model_file);
	glmUnitize(model);
	glmFacetNormals(model);
	glmVertexNormals(model, 90.0);
	
	glEnable(GL_LIGHTING);
	glEnable(GL_LIGHT0);
	glEnable(GL_DEPTH_TEST); 
	glClearColor(0.0, 0.0, 0.0, 1.0);
}



/*----------------------------------------------------------------------*/
/* These functions implement a simple trackball-like motion control.    */
/*----------------------------------------------------------------------*/

float lastPos[3] = {0.0F, 0.0F, 0.0F};
int curx, cury;
int startX, startY;

void trackball_ptov(int x, int y, int width, int height, float v[3])
{
    float d, a;

    /* project x,y onto a hemi-sphere centered within width, height */
    v[0] = (2.0F*x - width) / width;
    v[1] = (height - 2.0F*y) / height;
    d = (float) sqrt(v[0]*v[0] + v[1]*v[1]);
    v[2] = (float) cos((M_PI/2.0F) * ((d < 1.0F) ? d : 1.0F));
    a = 1.0F / (float) sqrt(v[0]*v[0] + v[1]*v[1] + v[2]*v[2]);
    v[0] *= a;
    v[1] *= a;
    v[2] *= a;
}

void func_action_startMotion(int x, int y)
{

    trackingMouse = true;
    redrawContinue = false;
    startX = x; startY = y;
    curx = x; cury = y;
    trackball_ptov(x, y, winWidth, winHeight, lastPos);
	trackballMove=true;
}

void func_action_stopMotion(int x, int y)
{

    trackingMouse = false;

    if (startX != x || startY != y) {
	redrawContinue = true;
    } else {
	angle = 0.0F;
	redrawContinue = false;
	trackballMove = false;
    }
}

void func_action_exit(void)
{
	glmDelete(model);
	exit(0);
}

//callback functions
void callback_display(void)
{
    glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);

    /* view transform */

    if (trackballMove) {
	    glRotatef(angle, axis[0], axis[1], axis[2]);
    }
	
	glmDraw(model, GLM_SMOOTH);

    glutSwapBuffers();
}


void callback_reshape(int w, int h)
{
	glViewport(0, 0, w, h);
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluPerspective(60.0, (GLfloat)h/(GLfloat)w, 1.0, 128.0);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	glTranslatef(0.0, 0.0, -3.0);
	winHeight = h;
	winWidth = w;
}

void callback_mouse(int button, int state, int x, int y)
{
	if(button==GLUT_RIGHT_BUTTON) func_action_exit();
	if(button==GLUT_LEFT_BUTTON) {
		switch(state) 
		{
		case GLUT_DOWN:
			y=winHeight-y;
			func_action_startMotion( x,y);
			break;
		case GLUT_UP:
			func_action_stopMotion( x,y);
			break;
		} 
	}
}

void callback_motion(int x, int y)
{
    float curPos[3], dx, dy, dz;

    trackball_ptov(x, y, winWidth, winHeight, curPos);
if(trackingMouse)
{
    dx = curPos[0] - lastPos[0];
    dy = curPos[1] - lastPos[1];
    dz = curPos[2] - lastPos[2];

    if (dx || dy || dz) {
	angle = 90.0F * sqrt(dx*dx + dy*dy + dz*dz);

	axis[0] = lastPos[1]*curPos[2] - lastPos[2]*curPos[1];
	axis[1] = lastPos[2]*curPos[0] - lastPos[0]*curPos[2];
	axis[2] = lastPos[0]*curPos[1] - lastPos[1]*curPos[0];

	lastPos[0] = curPos[0];
	lastPos[1] = curPos[1];
	lastPos[2] = curPos[2];
    }
} 
    glutPostRedisplay();
}

void callback_spinCube(void)
{
    if (redrawContinue) glutPostRedisplay();
}


//main
int main(int argc, char **argv)
{
	glutInit(&argc, argv);

    //test & get argument:
	if (argc < 2) {
		printf ("usage: trackball model-file.obj\n");
		printf ("exiting ...\n");
		return 1;
	}

	model_file = argv[1];

    //init glut window:
	glutInitDisplayMode(GLUT_DOUBLE | GLUT_RGB | GLUT_DEPTH);
    glutInitWindowSize(500, 500);
    glutCreateWindow("Trackball");
	
    //init all the rest:
	func_init();
	
    //register callback function:
    glutDisplayFunc(callback_display);
	glutReshapeFunc(callback_reshape);
    glutIdleFunc(callback_spinCube);
    glutMouseFunc(callback_mouse);
    glutMotionFunc(callback_motion);
	
    glutMainLoop();
}
