program meas2

  use DQMC_SHEET
  use DQMC_HUBBARD
  use DQMC_CFG
  use DQMC_GTAU
  
  implicit none

  ! variables
  type(config)  :: cfg
  type(Hubbard) :: Hub                     ! Hubbard model
  type(GTAU)    :: tau
  integer       :: nx, ny, nz, n, L, nxy, nzl
  integer       :: nBin, nIter, ntausk
  integer       :: iBin, cnt, avg, err
  integer       :: i, j, k
  real(wp), pointer :: Gavg(:,:), sgn(:)
  real          :: tt1, tt2
  integer,pointer   :: path(:,:), lpath(:)
  integer       :: npath, Lmaxpath
  character(len=slen) :: fname
  logical       :: vname

  ! ... Executable ...

  ! --------------- !
  ! Configuration   !
  ! --------------- !

  call CPU_TIME(tt1)
  ! read parameters from config file
  call DQMC_Read_Config(cfg, STDIN)
  call CFG_Get(cfg, "fname", fname)

  ! Initialize the geometry
  call CFG_Get(cfg, "nx", nx)
  call CFG_Get(cfg, "ny", ny)
  call CFG_Get(cfg, "nz", nz)
  nxy = nx*ny
  n  = nxy*nz
  call DQMC_Init_Sheet(nx, ny, nz, Hub%S)  

  ! Initialize the rest data
  call DQMC_Hub_Config(Hub, cfg)
  Hub%meas2 = .false.

  call DQMC_Gtau_Init(Hub%n, Hub%L, TAU_T0, &
       Hub%SB%nOrth, Hub%G_up%nWrap, tau, Hub%B, Hub%WS)

  L      = Hub%L
  call CFG_Get(cfg, "nbin", nBin)
  iBin   = 1
  cnt    = 0
  avg    = nBin + 1
  err    = nBin + 2

  ! Initialize meas data
  nzl    = nz*(L+1)

  allocate(Gavg(nzl, nBin+2))
  allocate(sgn(nBin+2))
  Gavg = ZERO
  sgn  = ZERO
  
  ! initialize paths
  npath    = 3
  allocate(lpath(npath))
  lpath(1)  = Hub%S%dim(1)/2+1
  lpath(2)  = Hub%S%dim(2)/2+1
  lpath(3)  = min(Hub%S%dim(1)/2+1,Hub%S%dim(2)/2+1)
  Lmaxpath=0
  do i=1, npath
    Lmaxpath=max(Lmaxpath,lpath(i))
  enddo
  allocate(path(npath, Lmaxpath))

  ! ---------- !
  ! Execution  !
  ! ---------- !
  ! Execution MC loop
  ! Warmup sweep

  do i = 1, Hub%nWarm
     ! The second parameter means no measurement should be made.
     call DQMC_Hub_Sweep(Hub, NO_MEAS0)
  end do
  
  ! We divide all the measurement into nBin,
  ! each having nPass/nBin pass.
  call CFG_Get(cfg, "tausk", ntausk)
  nIter  = Hub%nPass/nBin/ntausk
  do i = 1, nBin
     do j = 1, nIter
        do k = 1, ntausk
           call DQMC_Hub_Sweep(Hub, Hub%nMeas)
        end do
        call Measure()
     end do
     ! Accumulate results for each bin
     call Average() 
     call DQMC_Phy0_Avg(Hub%P0)
  end do

  ! average
  call GetError()
  call DQMC_Phy0_GetErr(Hub%P0)

  ! -------------- !
  ! Postprocessing !
  ! -------------- !
  call DQMC_Hub_OutputParam(Hub, STDOUT)
  write(STDOUT, FMT_DBLINE) 
  call DQMC_Phy0_Print(Hub%P0, Hub%S, STDOUT)
  write(STDOUT, FMT_DBLINE) 
  call PrintOut(STDOUT, Gavg, "sheet, tau, Gavg(sheet, tau) +- error")
  write(STDOUT, FMT_DBLINE)
  call cpu_time(tt2)
  write(STDOUT, *) "Execution time=", tt2-tt1

  call Dump_result()

  ! paths 	
!  call DQMC_Sort_SpinSpin(Hub%S, path, npath, lpath, Lmaxpath)
!  open(unit=44, file=trim(fname)//"_path.opt", action="write" )
!  call DQMC_Print_SpinSpin(Hub%S, Hub%P0, path, npath, lpath, Lmaxpath, 44)

  ! Clean up the used storage
  call DQMC_Hub_Free(Hub)
  call DQMC_Config_Free(cfg)
  call DQMC_Gtau_Free(tau)
  deallocate(Gavg, sgn, path, lpath)

contains

  ! -------------------- !
  ! supporting functios  !
  ! ---------------------!

  subroutine Measure()
    ! Purpose
    ! =======
    !    compute G(i, tau) for time dependent G_up and down
    !
    ! ... Local variables ...
    integer :: i, j, k, gi, idx, L1
    real(wp) ::factor

    ! ... Executable ...
    ! compute the first row of G tau
    L1 = L + 1
    sgn(avg) = Hub%G_up%sgn*Hub%G_dn%sgn
    factor = HALF*sgn(avg)/nxy
    Gavg(:,avg) = ZERO

    ! For each time slice
    do i = 1, L
       call DQMC_MakeGtau(tau, Hub%G_up, Hub%G_dn, 1, i-1)
       idx = 0
       ! For each sheet j
       do j = 1, nz
          gi = i+(j-1)*L1
          ! For each site k on sheet j 
          do k = idx+1, idx+nxy
             Gavg(gi, avg)  = Gavg(gi, avg)  + &
                  tau%upt0(k, k) + tau%dnt0(k,k)
          end do
          idx = idx + nxy
       end do
    end do

    ! special case for L+1
    do i = L1, nzl, L1
       Gavg(i,avg) = Gavg(i,avg)  + 2*nxy - Gavg(i-L1+1,avg)
    end do

    ! average
    Gavg(:, iBin) = Gavg(:, iBin) +  (Gavg(:,avg)*factor)

    sgn(iBin)     = sgn(iBin) + sgn(avg)
    cnt = cnt + 1
    
  end subroutine Measure

  ! ----------------------------------------------------------------- !

  subroutine Average()
    ! Purpose
    ! =======
    !    partially averge Green's function
    !
    ! ... Local variables ...
    real(wp) :: factor
    integer  :: i
    real(wp) :: noise(nz)
    real(wp), parameter :: ratio = 10D-4

    ! ... Executable ...
    
    ! average all the values
    factor = ONE/cnt
    Gavg(:, iBin) = Gavg(:, iBin)*factor
    sgn (ibin)    = sgn(iBin)*factor

    ! reset counter and adavence to next bin
    cnt = 0
    iBin = iBin + 1

  end subroutine Average

  ! ----------------------------------------------------------------- !

  subroutine Dump_result()
    ! Purpose
    ! =======
    !   Output results
    !
    ! ... Local variables ...
    integer :: f1, f2
    integer :: i

    ! ... Executable ...

    
    ! open files
    if (vname) then
       f1 = 13
       f2 = 14
    else
       f1 = STDOUT
       f2 = STDOUT
    end if
    open(unit=f1, file=trim(fname)//".1.opt", action="write" )
    open(unit=f2, file=trim(fname)//".2.opt", action="write" )
    
    ! output results to files
    write(f1, *) 'Results for Nl,Nc,Nbin,Nskip,Nmeas = '
    write(f1, *) L, n, nBin, ntausk, nIter
    write(f1, *) 't1,t2,t3,mu1,mu2,u1,u2,beta,filling = '
    write(f1, "(9f8.4)") Hub%t(1:3), Hub%mu(1:2), Hub%U(1:2), Hub%dtau*L, ONE
    
    write(f2, *) 'Results for Nl,Nc,Nbin,Nskip, Nmeas ='
    write(f2, *) L, n, nBin, ntausk, nIter
    write(f2, *) 't1,t2,t3,mu1,mu2,u1,u2,beta,filling = '
    write(f2, "(9f8.4)") Hub%t(1:3), Hub%mu(1:2), Hub%U(1:2), Hub%dtau*L, ONE

    do i = 1, nBin
       write(f1, *) ' '
       write(f1, *) ' G(tau,1) for bin ', i
       write(f1, 401) Gavg(1:L, i), sgn(i)
       write(f2, *) ' '
       write(f2, *) ' G(tau,2) for bin ', i
       write(f2, 401) Gavg(L+2:2*L+1, i), sgn(i)
    end do

401 format(1x,f11.8,1x,f11.8,1x,f11.8,1x,f11.8,1x,f11.8,1x,f11.8)

  end subroutine Dump_result


  ! ----------------------------------------------------------------- !

  subroutine GetError()
    ! Purpose
    ! =======
    !    Compute std err and average
    !
    ! ... Local variables ...
    integer  :: i
    real(wp) :: sum_sgn, z(nBin), y(nBin)

    ! ... Executable ...
    
    call DQMC_JackKnife(nBin, sgn(avg), sgn(err), sgn(1:nBin), &
         y, z, sum_sgn)

    do i = 1, nzl
       ! average all the values
       call DQMC_SignJackKnife(nBin, Gavg(i, avg), Gavg(i, err), &
            Gavg(i, 1:nBin), y, z, sum_sgn)
    end do

  end subroutine GetError

  ! ----------------------------------------------------------------- !
  
  subroutine PrintOut(OPT, G, title)
    ! Purpose
    ! =======
    !    Print out values
    !
    ! Argument
    ! ========
    integer, intent(in)  :: OPT
    real(wp), intent(in) :: G(:,:)
    character(*), intent(in) :: title

    ! ... Local variables ...
    integer :: i, j, idx
    character(*), parameter :: OPTFMT = "(i5,i5,f20.16,'   +- ', f20.16)"

    write(OPT,*) title
    idx = 1
    do j = 1, nz
       do i = 0, L
          write(OPT,OPTFMT) j, i, G(idx,avg), G(idx,err)
          idx = idx + 1
       end do
    end do

  end subroutine PrintOut


end program meas2
