program meas2_mpi

  use DQMC_SHEET
  use DQMC_HUBBARD
  use DQMC_CONFIG
  use DQMC_GTAU
  use DQMC_MPI
  
  implicit none

  ! variables
  type(config)  :: cfg
  type(Hubbard) :: Hub                     ! Hubbard model
  type(GTAU)    :: tau
  type(MPI_SIMOR) :: sim
  integer       :: nx, ny, nz, n, L, nxy, nzl
  integer       :: nBin, nIter, ntausk, nData
  integer       :: iBin, cnt, avg, err
  integer       :: i, j, k
  real(wp), pointer :: Gavg(:,:), Gacc(:,:) 
  real, pointer :: data(:,:), data_acc(:,:)
  real          :: tt1, tt2
  integer       :: ierr, count, stat(MPI_STATUS_SIZE)

  integer, parameter :: f1=13, f2=15
  character(len=namelen) :: fname
  logical :: vname
  
  ! --------------- !
  ! Configuration   !
  ! --------------- !
  call DQMC_MPI_Init(sim, PLEVEL_1)
  call CPU_TIME(tt1)

  ! read parameters from config file
  call GetArg(1, fname)
  open (unit = f1, file = fname)
  call DQMC_Config_Read(cfg, f1, CONFIG_CONFIG)
  close(f1)

  ! Initialize the geometry
  nx = CFG_Get(cfg, PARAM_NX)
  ny = CFG_Get(cfg, PARAM_NY)
  nz = CFG_Get(cfg, PARAM_NZ)
  nxy = nx*ny
  n  = nxy*nz
  call DQMC_Init_Sheet(nx, ny, nz, Hub%S)  

  ! Initialize the rest data
  call DQMC_Hub_Config(Hub, cfg)
  Hub%meas2 = .false.

  call DQMC_Gtau_Init(Hub%n, Hub%L, TAU_T0, &
       Hub%SB%nOrth, Hub%G_up%nWrap, tau, Hub%B, Hub%WS)

  L  = Hub%L
  nBin   = CFG_Get(cfg, PARAM_nBin)
  iBin   = 1
  cnt    = 0
  avg    = nBin + 1
  err    = nBin + 2

  ! Initialize meas data
  nzl = nz*(L+1)

  allocate(Gavg(nzl+1, nBin+2))

  Gavg = ZERO

  ! ---------- !
  ! Execution  !
  ! ---------- !
  ! Execution MC loop
  do i = 1, Hub%nWarm
     ! The second parameter means no measurement should be made.
     call DQMC_Hub_Sweep(Hub, NO_MEAS0)
  end do

  ! We divide all the measurement into nBin,
  ! each having nPass/nBin pass.
  ntausk = CFG_Get(cfg, PARAM_tausk)
  nIter  = Hub%nPass/nBin/ntausk
  Hub%meas2 = .false.
  do i = 1, nBin
     do j = 1, nIter
        do k = 1, ntausk
           call DQMC_Hub_Sweep(Hub, Hub%nMeas)
        end do
        call Measure()
     end do
     ! Accumulate results for each bin
     call Average() 
     call DQMC_Phy0_Avg(Hub%P0)
  end do

  ! -------------- !
  ! Postprocessing !
  ! -------------- !
  nData = Hub%P0%nData
  allocate(data(nData, nBin))
  call DQMC_Phy0_Pack(Hub%P0, data, 1)
  
  count = (nzl+1) * nbin
  if (DQMC_MPI_Is_Root(sim, CHANNEL_AGGR)) then

     ! open files for Maxent
     vname = CFG_Get(cfg, PARAM_FNAME, fname)
     open(unit=f1, file=trim(fname)//".1.opt", action="write" )
     open(unit=f2, file=trim(fname)//".2.opt", action="write" )

     allocate(Gacc(nzl+1, nBin+2))
     Gacc = Gavg

     write(f1, *) 'Results for Nl,Nc,Nbin,Nskip,Nmeas, NProc= '
     write(f1, *) L, n, nBin*sim%size, ntausk, nIter, sim%size
     write(f1, *) 't1,t2,t3,mu1,mu2,u1,u2,beta,filling = '
     write(f1, "(9f8.4)") Hub%t(1:3), Hub%mu(1:2), Hub%U(1:2), Hub%dtau*L, ONE
     
     write(f2, *) 'Results for Nl,Nc,Nbin,Nskip, Nmeas, NProc ='
     write(f2, *) L, n, nBin*sim%size, ntausk, nIter, sim%size
     write(f2, *) 't1,t2,t3,mu1,mu2,u1,u2,beta,filling = '
     write(f2, "(9f8.4)") Hub%t(1:3), Hub%mu(1:2), Hub%U(1:2), Hub%dtau*L, ONE
     
     call Dump_result(sim%aggr_root)

     do i = 1, sim%size - 1
        call MPI_RECV (Gavg, count, MPI_DOUBLE_PRECISION, &
             i, i, sim%aggr_comm, stat, ierr)
        call Dump_result(i)
        Gacc = Gacc + Gavg
     end do
     close(f1)
     close(f2)

     ! ---- legacy output ----
     open(unit=f1, file=trim(fname)//".BASIC", action="write" )
     call DQMC_Hub_OutputParam(Hub, f1)
     write(f1, FMT_DBLINE) 
  
     ! for p0
     allocate(data_acc(nData, nBin+2))
     data_acc = data
     do i = 1, sim%size - 1
        call MPI_RECV (data, nData, MPI_REAL, &
             i, i, sim%aggr_comm, stat, ierr)
        data_acc = data_acc + data
     end do
     data_acc = data_acc/sim%size
     
     call DQMC_Phy0_Pack(Hub%P0, data_acc, 1)
     call DQMC_Phy0_GetErr(Hub%P0)
     call DQMC_Phy0_Print(Hub%P0, Hub%S, f1)
     write(f1, FMT_DBLINE) 

     ! for tdm
     call GetError()
     call PrintOut(f1, Gacc, "sheet, tau, Gavg(sheet, tau) +- error")
     call cpu_time(tt2)
     write(f1, *) "Execution time=", tt2-tt1
     close(f1)
     deallocate(Gacc)
     deallocate(data_acc)

  else
     ! ---- non-root node ----
     call MPI_SEND(Gavg, count, MPI_DOUBLE_PRECISION, &
          sim%aggr_root, sim%aggr_rank, sim%aggr_comm, ierr)
     call MPI_SEND(data, nData, MPI_REAL, &
          sim%aggr_root, sim%aggr_rank, sim%aggr_comm, ierr)

  end if

  ! Clean up the used storage
  call DQMC_Hub_Free(Hub)
  call DQMC_Config_Free(cfg)

  call DQMC_MPI_Final(sim)
contains

  ! -------------------- !
  ! supporting functios  !
  ! ---------------------!

  subroutine Measure()
    ! Purpose
    ! =======
    !    compute G(i, tau) for time dependent G_up and down
    !
    ! ... Local variables ...
    integer :: i, j, k, gi, idx, L1
    real(wp) ::factor, sgn

    ! ... Executable ...
    ! compute the first row of G tau
    L1 = L + 1
    sgn = Hub%G_up%sgn*Hub%G_dn%sgn
    factor = HALF*sgn/nxy
    Gavg(:,avg) = ZERO

    ! For each time slice
    do i = 1, L
       call DQMC_MakeGtau(tau, Hub%G_up, Hub%G_dn, 1, i-1)
       idx = 0
       ! For each sheet j
       do j = 1, nz
          gi = i+(j-1)*L1
          ! For each site k on sheet j 
          do k = idx+1, idx+nxy
             Gavg(gi, avg)  = Gavg(gi, avg)  + &
                  tau%upt0(k,k) + tau%dnt0(k,k)
          end do
          idx = idx + nxy
       end do
    end do
    
    ! special case for L+1
    do i = L1, nzl, L1
       Gavg(i,avg) = Gavg(i,avg)  + 2*nxy - Gavg(i-L1+1,avg)
    end do
    
    ! average
    Gavg(1:nzl, iBin) = Gavg(1:nzl, iBin) +  (Gavg(1:nzl,avg)*factor)
    Gavg(nzl+1, iBin) = Gavg(nzl+1, iBin) + sgn 
    
    cnt = cnt + 1
    
  end subroutine Measure

  ! ----------------------------------------------------------------- !

  subroutine Average()
    ! Purpose
    ! =======
    !    partially averge Green's function
    !
    ! ... Local variables ...
    real(wp) :: factor
    integer  :: i
    real(wp) :: noise(nz)
    real(wp), parameter :: ratio = 10D-4

    ! ... Executable ...
    
    ! average all the values
    factor = ONE/cnt
    Gavg(:, iBin) = Gavg(:, iBin)*factor

    ! reset counter and adavence to next bin
    cnt = 0
    iBin = iBin + 1

  end subroutine Average

  ! ----------------------------------------------------------------- !

  subroutine Dump_result(p)
    ! Purpose
    ! =======
    !   Output results
    !
    integer, intent(in) :: p

    ! ... Local variables ...
    integer :: i, j

    ! ... Executable ...

    ! output results to files
    do i = 1, nBin
       write(f1, *) ' '
       write(f1, *) ' G(tau,1) for bin ', i, " from ", p
       write(f1, 401) Gavg(1:L, i), Gavg(nzl+1, i)
       write(f2, *) ' '
       write(f2, *) ' G(tau,2) for bin ', i, " from ", p
       write(f2, 401) Gavg(L+2:2*L+1, i), Gavg(nzl+1, i)
    end do

401 format(1x,f11.8,1x,f11.8,1x,f11.8,1x,f11.8,1x,f11.8,1x,f11.8)

  end subroutine Dump_result

  ! ----------------------------------------------------------------- !

  subroutine GetError()
    ! Purpose
    ! =======
    !    Compute std err and average
    !
    ! ... Local variables ...
    integer  :: i, sgn_idx
    real(wp) :: sum_sgn, z(nBin), y(nBin), factor

    ! ... Executable ...
    factor = ONE/sim%size
    Gacc = Gacc * factor
    sgn_idx = nzl + 1
    
    call DQMC_JackKnife(nBin, Gacc(sgn_idx, avg), Gacc(sgn_idx, err), &
         Gacc(sgn_idx, 1:nBin), y, z, sum_sgn)

    do i = 1, nzl
       call DQMC_SignJackKnife(nBin, Gacc(i, avg), Gacc(i, err), &
            Gacc(i, 1:nBin), y, z, sum_sgn)
    end do

  end subroutine GetError

  ! ----------------------------------------------------------------- !
  
  subroutine PrintOut(OPT, G, title)
    ! Purpose
    ! =======
    !    Print out values
    !
    ! Argument
    ! ========
    integer, intent(in)  :: OPT
    real(wp), intent(in) :: G(:,:)
    character(*), intent(in) :: title

    ! ... Local variables ...
    integer :: i, j, idx
    character(*), parameter :: OPTFMT = "(i5,i5,f20.16,'   +- ', f20.16)"

    write(OPT,*) title
    idx = 1
    do j = 1, nz
       do i = 0, L
          write(OPT,OPTFMT) j, i, G(idx,avg), G(idx,err)
          idx = idx + 1
       end do
    end do
    write(OPT, FMT_DBLINE)

  end subroutine PrintOut

end program meas2_mpi
