%
% INSYDRV - INdefinite SYmmetric Lanczos DRiVer
%
%------------------------------------------------------------------------------
%     Variable Descriptions
%     =====================
%     Set by user:
%     ------------
%     AltForm        (logical) Determines linearization type used for quadratic
%                              eigenvalue problems.  See the remarks in the
%                              ``Further Details'' section.
%     Diagnostics    (logical) Turns on diagnostic computation/calculations. 
%     ERcorr         (logical) Turns on the Ericsson-Ruhe post processing step.
%                              See remarks in the ``Further Details'' section.
%     MustPreprocess (logical) If true, the initial vector is pre-multiplied
%                              by inv(A-sigma*B)*B.
%     iorthog        (string ) Reorthogonalization option: "none", "full",
%                              or "semi".
%     orthtol        (double ) Reorthogonalization tolerance.
%     convtol        (double ) Convergence tolerance.
%     bdtol          (double ) Breakdown tolerance.
%     m              (integer) Order of matrices in quadratic problems.
%     maxitr         (integer) Maximum number of Lanczos iterations.
%     n              (integer) Order of A and of B.
%     nwant          (integer) Number of wanted Ritz values.
%     sigma          (complex) Shift value.
%     testprob       (integer) Test problem identification number.
%     r              (double ) Initial Lanczos vector before scaling.
%
%     Problem dependent variables:
%     ----------------------------
%
%     probtype       (string ) probtype ='quadratic' if underlying problem is
%                              quadratic, and probtype ='linear   ' if 
%                              underlying problem is linear.
%     A              (double ) The A matrix. Not explicitly formed for 
%                              quadratic eigenvalue problems.
%     B              (double ) The B matrix. Not explicitly formed for 
%                              quadratic eigenvalue problems.
%     K              (double ) For quadratic problems, coefficient matrix of
%                              constant term.
%     C              (double ) For quadratic problems, coefficient matrix of
%                              linear term.
%     M              (double ) For quadratic problems, coefficient matrix of
%                              quadratic term.
%     L              (complex) Lower triangular factor of A-sigma*B (linear
%                              problems) or of K +sigma*C +sigma^2*M
%                              (quadratic problems).
%     U              (complex) Upper triangular factor of A-sigma*B (linear
%                              problems) or of K +sigma*C +sigma^2*M
%     aeig           (complex) Eigenvalues of (A,B).  If exact eigenvalues
%                              are known, aeig contains the exact eigenvalues.
%                              Otherwise, aeig contains the eigenvalues of 
%                              (A,B) as computed by the LAPACK routine DGEGV.
%
%     Returned from INSYLAN:
%     ----------------------
%     k              (integer) Number of Lanczos steps taken by INSYLAN.
%     nreorth        (integer) Number of reorthogonalization performed.
%     nconv          (integer) Number of converged Ritz values.
%     pritz          (complex) Eigenvalues of (T, Omega).
%     indcon         (integer) Indices of converged Ritz values.
%     Z              (complex) Eigenvectors of pencil (T, Omega).
%     Q              (complex) Matrix of Lanczos vectors.
%     res            (double ) Bound on estimated accuracy of pritz.
%     r              (complex) Most recent unscaled Lanczos vector.
%     alpha          (complex) Diagonal entries of T.
%     beta           (complex) Sub- and superdiagonal entries of T.
%     omega          (complex) Diagonal entries of Omega.
%     rcond          (double ) Reciprocal of the eigenvalue condition numbers
%                              of the eigenvalues of (T, Omega).
%     elossval       (double ) When semi-orthogonality is maintained,
%                              elossval(k) is the estimated loss of
%                              orthogonality at step k.
%     xlossval       (double ) When semi-orthogonality is maintained, and
%                              Diagnostics=true, xlossval(k) is the exact loss 
%                              of orthogonality at step k.
%
%     Post-processing after INSYLAN:
%     ------------------------------
%     X              (complex) Ritz vectors (linear problems).
%     X1             (complex) Ritz vectors (quadratic problems).
%     X2             (complex) Ritz vectors (quadratic problems).
%
%    
%     External Routines called: 
%     =========================
%     getmat, insylan
%
%     Intrinsic Routines called: 
%     ==========================
%     lu
%
%
%
%     Further Details
%     ===============
%
%     Linearization of quadratic problems:
%
%     Quadratic eigenvalue problems
%             K*x + (lam)*C*x + (lam)^2*M*x = 0
%     where K,C, and M are all symmetric may be converted to a linear 
%     eigenvalue problem in one of two ways:
%     
%       (I) If M is nonsingular:
%               [ -K  O ][ x1]       [ C  M ][x1]
%               [       ][   ] = lam*[      ][  ]
%               [  O  M ][ x2]       [ M  O ][x2],
%
%             where x1 = x and x2 = (lam)*x.
%
%       (II) If K is nonsingular:
%               [  O  K ][ x1]       [ K  O ][x1]
%               [       ][   ] = lam*[      ][  ]
%               [  K  C ][ x2]       [ O -M ][x2]
%
%             where x1 = x and x2 = (lam)*x.
%
%     The flag AltForm determines which linearization is used: if
%     Altform is false, then (I) is used; if Altform is true, then (II) is
%     used.
%
%     Ericsson-Ruhe Correction:
%     The Ericsson-Ruhe correction step attempts to improve the quality of
%     the Ritz vectors by creating a new Ritz vector from a linear combination
%     of the original Ritz vector and the last Lanczos vector.  For further
%     details, see
%     @article(ericsson:ruhe,
%       author  =  "T. Ericsson and Axel Ruhe",
%       title  =  "The Spectral Transformation {L}anczos Method of the 
%                  Numerical Solution of Large Sparse Generalized Symmetric 
%                  Eigenvalue Problems",
%       journal  =  "Mathematics of Computation",
%       year  =  "1980",
%       volume  =  "35",
%       pages  =  "1251-1268")
%
%
%     
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
clear; format compact; format short e; randn("seed",1999);
%
%--------Select test problem.
%
%testprob = 1; %----random linear
%testprob = 2; %----speaker107:
               %    quadratic eigenvalue problem from structural dynamics;
               %    order of quadratic problem: 107
%testprob = 3; %----random quadratic
%testprob = 4; %----shaft400
               %    quadratic eigenvalue problem from structural dynamics;
               %    order of quadratic problem: 400
               %    Both mass and damping matrix singular.
%testprob = 5; %----speaker334:
               %    quadratic eigenvalue problem from structural dynamics;
               %    order of quadratic problem: 334
%testprob = 6; %----defective symmetric pencil with prescribed eigenvalues;
               %    for now, will be complex if eigenvalues are complex, and
               %    each prescribed eigenvalue corresponds to 2 by 2 Jordan
               %    block of inv(K)*M.
%testprob = 7;  %----linear diagonal with singular B
%testprob = 8;  %----linear diagonal with nonsingular B
testprob = 9;  %----shaft80
%testprob = 10;  %----speaker538;
[probtype,n,m,sigma,MustPreprocess,AltForm,aeig,A,B,K,C,M] =...
    getmat(testprob);
m2 = 2*m;
%
%--------Initialize some parameters
%
maxitr = min(n, 100);         %-------Maximum Lanczos iterations
bdtol  = 1.e-14;              %-------Breakdown tolerance
convtol= 1.e-08;               %-------Convergence tolerance
%iorthog= 'none';              %-------Reorthogonalization flag:
%iorthog= 'semi';              %-------Reorthogonalization flag:
iorthog= 'full';              %-------Reorthogonalization flag:
                              %       iorthog = 'none' => no reorthogonalization
                              %       iorthog = 'full' ==> full orthogonality
                              %       iorthog = 'semi' ==> semi orthogonality
ERcorr = 1;                   %-------Flag to turn on Ericsson-Ruhe correction.
orthtol= 1.e-8;               %-------Reorthogonalization tolerance
nwant  =  5;                  %-------Number of desired eigenvalues
r      = ones(n,1);           %-------Initial vector.
%
Diagnostics = 1;              %-------Diagnostics flag: set to zero to turn off
                              %       expensive diagnostic computations such
                              %       as direct residual computations.
%
%--------Compute LU factorization of (A-sigma*B)
%
if (probtype=='linear   ')
   [L, U] = lu(A-sigma*B);
else
   [L, U] = lu(K + sigma*C + sigma^2*M);
end
%
%--------Call indefinite Lanczos procedure
%
[ k,nreorth,nconv,indcon,pritz,Z,res,alpha,beta,omega,r,Q,...
  rcond, elossval,xlossval ] = ...
   insylan( n, m, maxitr, nwant, probtype, AltForm, Diagnostics, ...
            MustPreprocess, iorthog, orthtol, convtol, bdtol, sigma, ...
            r, L, U, B, K, C, M );
%
%--------Compute eigenvectors and undo spectral transformation.
%
if (nconv > 0)
   ritz = 1./pritz + sigma;
   if (probtype=='linear   ')
      X = Q(:,1:k)*Z(:,indcon);
      for h = 1:nconv
          X(:,h) = X(:,h)/norm(X(:,h));
          dres(h) = norm(A*X(:,h) - ritz(indcon(h))*B*X(:,h));
          if (ERcorr)
             X(:,h) = X(:,h) + r/pritz(indcon(h))*Z(k,indcon(h));
             dresu(h) = dres(h);
             X(:,h) = X(:,h)/norm(X(:,h));
             dres(h) = norm(A*X(:,h) - ritz(indcon(h))*B*X(:,h),1);
          end
%          dres(h) = dres(h)/abs(ritz(indcon(h)));
          pdiff(h) = min( abs( pritz(indcon(h)) - (1./(aeig - sigma)) ) );
          diff(h) = min(abs( ritz(indcon(h)) - aeig ))/abs(ritz(indcon(h)));
      end
   else
%
%---------------The next for loop is for diagnostic purposes only; it
%               includes some expensive computations which should not be
%               included in a production code.
%
      if (Diagnostics)
         X = Q(:,1:k)*Z(:,indcon);
         for h=1:nconv
             normX(h) = norm(X(:,h));
             X(:,h) = X(:,h)/normX(h);
             w = zeros(m2,1);
             Gx = zeros(m2,1);
             Bx = zeros(m2,1);
             w(1:m) = C*X(1:m,h) + M*X(m+1:m2,h);
             w(m+1:m2) = M*X(1:m,h);
             Gx(1:m) = U \ ( L \ ( -w(1:m) - sigma*w(m+1:m2) ) );
             Gx(m+1:m2) = X(1:m,h) + sigma*Gx(1:m);
             pres(h) = norm( Gx - pritz(indcon(h))*X(:,h),2 );
             if (AltForm)
                Bx(1:m) = K*X(1:m,h);
                Bx(m+1:m2) = -M*X(m+1:m2,h);
             else
                Bx(1:m) = C*X(1:m,h) +M*X(m+1:m2,h);
                Bx(m+1:m2) = M*X(1:m,h);
             end
             condno(h) = norm(Bx) / abs( Bx.'*X(:,h) );
             errbnd(h) = condno(h)*pres(h);
             pdiff(h) = min( abs( pritz(indcon(h)) - (1./(aeig - sigma)) ) );
         end  
      end
%
%-------------End diagnostic code
%
%      X = Q(1:m,1:k)*Z(:,indcon);
%      X = Q(:,1:k)*Z(:,indcon);
      X1 = Q(1:m,1:k)*Z(:,indcon);
      X2 = Q(m+1:m2,1:k)*Z(:,indcon);
%      X1 = X(1:m,:);
%      X2 = X(m+1:m2,:);
      maxritz = max(abs(ritz(indcon)));
      for h = 1:nconv
          dres1(h) = norm(K*X1(:,h) + ritz(indcon(h))*C*X1(:,h)...
                         + ritz(indcon(h))^2*M*X1(:,h))/norm(X1(:,h));
          dres2(h) = norm(K*X2(:,h) + ritz(indcon(h))*C*X2(:,h)...
                         + ritz(indcon(h))^2*M*X2(:,h))/norm(X2(:,h));
          normX2h = norm(X2(:,h))
          normX1h = norm(X1(:,h))
          normXh = norm([ normX1h normX2h ])
          if (ERcorr)
             X1(:,h) = X1(:,h) + r(1:m)/pritz(indcon(h))*Z(k,indcon(h));
             X2(:,h) = X2(:,h) + r(m+1:m2)/pritz(indcon(h))*Z(k,indcon(h));
             dres1u(h) = dres1(h);
             dres2u(h) = dres2(h);
             dres1(h) = norm(K*X1(:,h) + ritz(indcon(h))*C*X1(:,h)...
                            + ritz(indcon(h))^2*M*X1(:,h))/norm(X1(:,h));
             dres2(h) = norm(K*X2(:,h) + ritz(indcon(h))*C*X2(:,h)...
                            + ritz(indcon(h))^2*M*X2(:,h))/norm(X2(:,h));
          end
          X2(:,h) = X2(:,h)/norm(X2(:,h));
          X1(:,h) = X1(:,h)/norm(X1(:,h));
          diff(h) = min(abs( ritz(indcon(h)) - aeig ));
          if (abs(ritz(indcon(h)))>0.001*maxritz)
             diff(h) = diff(h)/abs(ritz(indcon(h)));
          end
      end
   end
end
%
%--------Plot residuals, eigenvalues
%
%plot( real(ritz(indcon)), imag(ritz(indcon)), 'go' );
%hold on;
%plot( real(aeig), imag(aeig), 'r+' );
%hold off;
%pause;
%semilogy( dres, 'r+' );
%hold on;
%semilogy( res, 'go' );
%semilogy( diff, 'm*' );
%hold off;
