function [ k,nreorth,nconv,indcon,pritz,Z,res,alpha,beta,omega,r,Q,...
           rcond,elossval,xlossval]=...
   insylan( n, m, maxitr, nwant, probtype, AltForm, Diagnostics, ...
            MustPreprocess, iorthog, orthtol, convtol, bdtol, sigma, ...
            r, L, U, B, K, C, M )
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% INSYLAN - INdefinite SYmmetric LANczos process
%
%  Extract a few eigenpairs of (A-sigma*B)\B using indefinite symmetric
%  Lanczos procedure.
%
% function [ k,nreorth,nconv,indcon,pritz,Z,alpha,beta,omega,Q,...
%            rcond,xlossval,elossval,rcond ]=...
%   insylan( n, m, maxitr, nwant, probtype, AltForm, Diagnostics, ...
%            MustPreprocess, iorthog, orthtol, convtol, bdtol, sigma, ...
%            r, L, U, B, K, C, M )
%------------------------------------------------------------------------------
%     Variable Descriptions
%     =====================
%
%     Input variables:
%     ----------------
%     n              (integer) Order of A and of B.
%     m              (integer) Order of matrices in quadratic problems.
%                              see ``Further Details'' in file insydrv.m
%     maxitr         (integer) Maximum number of Lanczos iterations.
%     nwant          (integer) Number of wanted Ritz values.
%     probtype       (string ) probtype ='quadratic' if underlying problem is
%                              quadratic, and probtype ='linear   ' if 
%                              underlying problem is linear.
%     AltForm        (logical) Determines linearization type used for quadratic
%                              eigenvalue problems.  See the remarks in the
%                              ``Further Details'' section in file insydrv.m.
%     Diagnostics    (logical) Turns on diagnostic computation/calculations. 
%     MustPreprocess (logical) If true, the initial vector is pre-multiplied
%                              by inv(A-sigma*B)*B.
%     iorthog        (string ) Reorthogonalization option: "none", "full",
%                              or "semi".
%     orthtol        (double ) Reorthogonalization tolerance.
%     convtol        (double ) Convergence tolerance.
%     bdtol          (double ) Breakdown tolerance.
%     sigma          (complex) Shift value.
%     B              (double ) The B matrix. Not explicitly formed for 
%                              quadratic eigenvalue problems.
%     K              (double ) For quadratic problems, coefficient matrix of
%                              constant term.
%     C              (double ) For quadratic problems, coefficient matrix of
%                              linear term.
%     M              (double ) For quadratic problems, coefficient matrix of
%                              quadratic term.
%     L              (complex) Lower triangular factor of A-sigma*B (linear
%                              problems) or of K +sigma*C +sigma^2*M
%                              (quadratic problems).
%     U              (complex) Upper triangular factor of A-sigma*B (linear
%                              problems) or of K +sigma*C +sigma^2*M
%                              (quadratic problems).
%
%     Updated variables:
%     ------------------
%     r              (complex) Upon input, the first (unscaled) Lanczos
%                              vector, and upon output, the most recent
%                              Lanczos vector.
%
%     Output variables:
%     -----------------
%     k              (integer) Number of Lanczos steps taken by INSYLAN.
%     nreorth        (integer) Number of reorthogonalization performed.
%     nconv          (integer) Number of converged Ritz values.
%     indcon         (integer) Indices of converged Ritz values.
%     pritz          (complex) Eigenvalues of (T, Omega).
%     Z              (complex) Eigenvectors of pencil (T, Omega).
%     res            (double ) Bound on estimated accuracy of pritz.
%     alpha          (complex) Diagonal entries of T.
%     beta           (complex) Sub- and superdiagonal entries of T.
%     omega          (complex) Diagonal entries of Omega.
%     Q              (complex) Matrix of Lanczos vectors.
%     rcond          (double ) Reciprocal of the eigenvalue condition numbers
%                              of the eigenvalues of (T, Omega).
%     elossval       (double ) When semi-orthogonality is maintained,
%                              elossval(k) is the estimated loss of
%                              orthogonality at step k.
%     xlossval       (double ) When semi-orthogonality is maintained, and
%                              Diagnostics=true, xlossval(k) is the exact loss 
%                              of orthogonality at step k.
%
%     Internal variables:
%     -------------------
%     T              (complex) Tridiagonal matrix tridiag(beta,alpha,beta).
%     tau            (double ) Norm of r, the most recent unscaled Lanczos
%                              vector.
%     mu             (double ) Norm of B*r, where r is the most recent
%                              unscaled Lanczos vector.
%     nu             (double ) Frobenius norm of B*Q_j. 
%
%     w              (complex) B*q_j.
%     w1             (complex) B*q_{j-1}, needed for when maintaining 
%                              semi-orthogonality.
%     ww             (complex) When probtype='quadratic' and AltForm=true,
%                              ww is an intermediate value computed by the
%                              routine MATVEC which is later used by LINSOL.
%     h0,h1,h2,scl   (complex) Used in recurrence for monitoring the loss of
%                              orthogonality.
%
%     External Routines called: 
%     =========================
%     matvec, linsol, solvred, reortho
%
%     Intrinsic Routines called: 
%     ==========================
%     abs, diag, norm
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%--------Begin Lanczos procedure
%
%
%--------Initialize Q, omega, alpha, beta
%
Q = zeros(n,maxitr);
omega = zeros(maxitr,1);
beta  = zeros(maxitr,1);
alpha = zeros(maxitr,1);
%
% Further initializations for semi-orthogonality
%
w1 = 0;
ww = 0;
h0 = zeros(maxitr,1);
h1 = h0;
h2 = h0;
scl = zeros(maxitr,1);
nreorth = 0;
xlossval = 0;
elossval = 0;
%
%--------Preprocess initial vector, if desired
%
if (MustPreprocess)
   w = matvec( probtype, AltForm, m, r, B, K, C, M );
   Q(:,1) = r;
   [r, ww] = linsol( probtype, AltForm, m, sigma, Q(:,1), w, L, U, C, M );
end
tau = norm(r);
if (tau == 0.0)
   error( 'Initial vector is zero' );
end
%
%--------Initial scaling
%
Q(:,1) = r/tau;
w = matvec( probtype, AltForm, m, Q(:,1), B, K, C, M );
omega(1) = w.'*Q(:,1);
mu = norm(w);
if (abs(omega(1)) <  bdtol*mu)
   error( 'breakdown in initial scaling' );
end
%
%--------Start of Lanczos loop
%
nconv = 0;
nu = 0;
k = 0;

while ((k < maxitr) & (nconv < nwant))
   k = k+1;
   [r, ww] = linsol( probtype, AltForm, m, sigma, Q(:,k), w, L, U, C, M );
   if (k > 1)
      r = r - Q(:,k-1)*(beta(k)/omega(k-1));
   end
   alpha(k) = w.'*r;
   r = r - Q(:,k)*(alpha(k)/omega(k));
   w = matvec( probtype, AltForm, m, r, B, K, C, M );
%
% Update T
%
   T = diag(alpha(1:k));
   if (k > 1) 
      T = T + diag(beta(2:k),1) + diag(beta(2:k),-1);
   end
%
%--------Reorthogonalization 
%
   tau = norm(r);
   if ( (iorthog == 'full') | (iorthog == 'semi') )
      [nreorth, h0, h1, h2, r, Q, w, ww, scl, elossval, xlossval] = ... 
      reortho( k, nreorth, probtype, AltForm, Diagnostics, iorthog, ...
               orthtol, alpha(k), beta(k), tau, sigma, omega, w1, T, B, K, ...
               C, M, Q, h0, h1, h2, r, w, ww, scl, elossval, xlossval );
   end
%
%--------Solve reduced problem and check for convergence
%
   [ nconv, nu, indcon, pritz, rcond, Z, res ] = ...
     solvred( k, nu, tau, convtol, w, omega, T );
%
%--------Prepare for next iteration
%
    Q(:,k+1) = r/tau;
    w        = w/tau;
    omega(k+1) = w.' * Q(:,k+1);
    if (abs(omega(k+1)) < bdtol*( abs(alpha(k))+2*abs(beta(k)) )  )
       fprintf( '\n-----in Lanczos iteration k= %i,\n omega(k+1) = %g,\n ...
                 alpha(k) = %g,\n breakdown tolerance = %g\n', k, omega(k+1),...
                 alpha(k), bdtol );
       warning( 'breakdown in Lanczos iteration' );
       if (tau < bdtol*( abs(alpha(k))+2*abs(beta(k)) ) )
          fprintf( '\n... could be good news:\n');
          fprintf( '\tit seems that an invariant subspace has been found\n' );
       end
       break;
    end
    beta(k+1) = tau*omega(k+1);   
    if (AltForm)
       ww = ww/tau;
    end
    if (iorthog == 'semi')
       w1 = w;
    end
end
%
%--------End of Lanczos loop
%
return
