function   [r, ww] = linsol( probtype, AltForm, m, sigma, Qj, w, L, U, C, M );
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LINSOL - Linear solution of (A-sigma*B)*r = w,
%          where w = B*Qj.
%
% function   [r, ww] = linsol( probtype, AltForm, m, sigma, Qj, w, L, U, C, M );
%------------------------------------------------------------------------------
%     Variable Descriptions
%     =====================
%
%     Input variables:
%     ----------------
%     probtype       (string ) probtype ='quadratic' if underlying problem is
%                              quadratic, and probtype ='linear   ' if 
%                              underlying problem is linear.
%     AltForm        (logical) Determines linearization type used for quadratic
%                              eigenvalue problems.  See the remarks in the
%                              ``Further Details'' below and in the file 
%                              insydrv.m.
%     m              (integer) Order of matrices in quadratic problems.
%                              see ``Further Details'' in file insydrv.m
%     sigma          (complex) Shift value.
%     Qj             (complex) Right-hand side is B*Qj.  Required for
%                              quadratic problems.
%     w              (complex) Right-hand side.
%     L              (complex) Lower triangular factor of A-sigma*B (linear
%                              problems) or of K +sigma*C +sigma^2*M
%                              (quadratic problems).
%     U              (complex) Upper triangular factor of A-sigma*B (linear
%                              problems) or of K +sigma*C +sigma^2*M
%                              (quadratic problems).
%     C              (double ) For quadratic problems, coefficient matrix of
%                              linear term.
%     M              (double ) For quadratic problems, coefficient matrix of
%                              quadratic term.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if (probtype=='linear   ')
   r = U \ (L \ w);
   ww = 0;
else
   m2 = 2*m;
   r = zeros(m2,1);
   if (AltForm) 
      ww = C*Qj(1:m,:) + sigma*M*Qj(1:m,:);
      r(1:m) = U \ ( L \ ( -ww + w(m+1:m2) ) );
      r(m+1:m2) = Qj(1:m) + sigma*r(1:m);
   else
      ww = 0;
      r(1:m) = U \ ( L \ ( -w(1:m) - sigma*w(m+1:m2) ) );
      r(m+1:m2) = Qj(1:m) + sigma*r(1:m);
   end
end
