function [nreorth, h0, h1, h2, r, Q, w, ww, scl, elossval, xlossval] = ... 
reortho( k, nreorth, probtype, AltForm, Diagnostics, iorthog, orthtol, aj,...
         bj, tau, sigma, omega, w1, T, B, K, C, M, Q, ...
         h0, h1, h2, r, w, ww, scl, elossval, xlossval )
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  REORTHO  - monitors the loss of orthogonality and performs
%  any required reorthogonalizations.
%
% function [nreorth, h0, h1, h2, r, Q, w, ww, scl, elossval, xlossval] = ... 
% reortho( k, nreorth, probtype, AltForm, Diagnostics, iorthog, orthtol, aj,...
%          bj, tau, sigma, omega, w1, T, B, K, C, M, Q, ...
%          h0, h1, h2, r, w, ww, scl, elossval, xlossval )
%------------------------------------------------------------------------------
%     Variable Descriptions
%     =====================
%
%     Input variables:
%     ----------------
%     k              (integer) Lanczos step.
%     iorthog        (string ) Reorthogonalization option: "none", "full",
%                              or "semi".
%     orthtol        (double ) Reorthogonalization tolerance.
%     aj             (complex) Most recent diagonal value of T.
%     bj             (complex) Most recent subdiagonal value of T.
%     orthtol        (double ) Reorthogonalization tolerance.
%     tau            (double ) Norm of r, the most recent unscaled Lanczos
%                              vector.
%     probtype       (string ) probtype ='quadratic' if underlying problem is
%                              quadratic, and probtype ='linear   ' if 
%                              underlying problem is linear.
%     AltForm        (logical) Determines linearization type used for quadratic
%                              eigenvalue problems.  See the remarks in the
%                              ``Further Details'' section in file insydrv.m.
%     Diagnostics    (logical) Turns on diagnostic computation/calculations. 
%     sigma          (complex) Shift value.
%     omega          (complex) Diagonal entries of Omega.
%     w1             (complex) B*q_{j-1}, needed for when maintaining 
%                              semi-orthogonality.
%     Q              (complex) Matrix of Lanczos vectors.
%     T              (complex) Tridiagonal matrix tridiag(beta,alpha,beta).
%     B              (double ) The B matrix. Not explicitly formed for 
%                              quadratic eigenvalue problems.
%     K              (double ) For quadratic problems, coefficient matrix of
%                              constant term.
%     C              (double ) For quadratic problems, coefficient matrix of
%                              linear term.
%     M              (double ) For quadratic problems, coefficient matrix of
%                              quadratic term.
%     Updated variables:
%     ------------------
%     nreorth        (integer) Number of reorthogonalization performed.
%     h0             (complex) Estimated value of q_{k-1}^T*M*Q_{k-2}
%     h1             (complex) Estimated value of q_{k}^T*M*Q_{k-1}
%     h2             (complex) Estimated value of q_{k+1}^T*M*Q_{k}
%     r              (complex) Most recent unscaled Lanczos vector
%     Q              (complex) Matrix of Lanczos vectors.
%     ww             (complex) When probtype='quadratic' and AltForm=true,
%                              ww is an intermediate value computed by the
%                              routine MATVEC which is later used by LINSOL.
%     scl            (complex) Scale factor which takes into account the
%                              norm of w and Omega.
%     elossval       (double ) When semi-orthogonality is maintained,
%                              elossval(k) is the estimated loss of
%                              orthogonality at step k.
%     xlossval       (double ) When semi-orthogonality is maintained, and
%                              Diagnostics=true, xlossval(k) is the exact loss 
%                              of orthogonality at step k.
%
%     External Routines called: 
%     =========================
%     matvec
%
%     Intrinsic Routines called: 
%     ==========================
%     norm
%
%     Further Details
%     ===============
%     Refer to
%
%     @article(day:1997,
%       author  =  "D. Day",
%       title  =  "An Efficient Implementation of the Nonsymmetric Lanczos 
%                  Algorithm",
%       journal  =  "SIAM J. Matrix Anal. Appl.",
%       year  =  "1997",
%       volume  =  "18",
%       pages  =  "566-589")
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
m = size(Q,1)/2;
m2 =2*m;
if (iorthog == 'full') %-------- full reorthogonalization
   nreorth = nreorth + 1;
   for h=1:2      %-------- do classical GS twice; this is as stable as MGS
      s = Q(:,1:k).'*w;
      r = r - Q(:,1:k)*( diag(omega(1:k))\s );
      w = matvec( probtype, AltForm, m, r, B, K, C, M);
   end
elseif (iorthog == 'semi') %--- maintain semi-orthogonality
   h2(k) = Q(:,k).'*w;
   if (k>1)
      h2(k-1) = Q(:,k-1).'*w;
   end
   if (k > 2)
      h1(1:k-1) = h1(1:k-1)/tau;
      h2(1:k-2) = T(1:k-2,1:k-2)*((1./omega(1:k-2)).*h1(1:k-2));
      h2(1:k-2) = h2(1:k-2) - h1(1:k-2)*(aj/omega(k));
   end
   if (k > 3)
      h2(1:k-2) = h2(1:k-2) - h0(1:k-2)*(bj/omega(k-1));
   end
   normw = norm(w);
   scl(k) = sqrt(normw/omega(k));
   eloss = norm(scl(1:k).*(h2(1:k)/normw),1);
   elossval(k) = eloss;
   if (Diagnostics)
      xloss = Q(:,1:k).'*w/normw;
      xlossval(k) = norm(xloss,1);
   end
   if (elossval(k) > orthtol)
      nreorth = nreorth + 1;
      for h=1:2
         s = Q(:,1:k-1).'*[w1 w];
         Q(:,k+1) = r;
         Q(:,k:k+1) = Q(:,k:k+1) - Q(:,1:k-1)*( diag(omega(1:k-1))\s );
         r = Q(:,k+1);
         tmpin = [Q(:,k), r];
         tmpout = matvec( probtype, AltForm, m, tmpin, B, K, C, M);
         w1 = tmpout(:,1);
         w = tmpout(:,2);
      end
      h1(1:k-1) = s(:,1)*1.e1;
      h2(1:k-1) = s(:,2)*1.e1;
   end
   h0 = h1;
   h1 = h2;   
end
return
