function  [ nconv, nu, indcon, pritz, rcond, Z, res ] = ...
          solvred( k, nu, tau, convtol, w, omega, T );
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% SOLVRED - SOLVe REDuced eigenvalue problem T_k*z = (pritz)*Omega_k*z.
%           and test for convergence.
% function  [ nconv, nu, indcon, pritz, rcond, Z ] = ...
%             solvred( k, nu, tau, convtol, w, omega, T );
%
%------------------------------------------------------------------------------
%     Variable Descriptions
%     =====================
%
%     Input variables:
%     ----------------
%     k              (integer) Order of T and of Omega.
%     nu             (double ) Frobenius norm of B*Q_j. 
%     tau            (double ) Norm of r, the most recent unscaled Lanczos
%                              vector.
%     convtol        (double ) Convergence tolerance.
%     w              (complex) B*r where r is the most recent unscaled Lanczos
%                              vector.
%     omega          (complex) Diagonal entries of Omega.
%     T              (complex) Tridiagonal matrix tridiag(beta,alpha,beta).
%
%     Updated variables:
%     ------------------
%     nu             (double ) Frobenius norm of B*Q_{j+1}. 
%
%     Output variables:
%     -----------------
%     nconv          (integer) Number of converged Ritz values.
%     indcon         (integer) Indices of converged Ritz values.
%     pritz          (complex) Eigenvalues of (T, Omega).
%     rcond          (double ) Reciprocal of the eigenvalue condition numbers
%                              of the eigenvalues of (T, Omega).
%     Z              (complex) Eigenvectors of pencil (T, Omega).
%     res            (double ) Bound on estimated accuracy of pritz.
%
%     Internal variables:
%     -------------------
%     That           (complex) Modified tridiagonal matrix. 
%                              That = |Omega|^(-1/2)*T*pmOmega*|Omega|^(-1/2)
%                                  where 
%                               pmOmega(k,k) = Omega(k,k)/abs(Omega(k,k)).
%     absomega       (double ) Absolute value of omega. 
%     Zhat           (complex) Eigenvectors of That.
%     gamma          (complex) Algorithm residuals. xxx
%
%     Intrinsic Routines called: 
%     ==========================
%     abs, diag, norm, eig
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
mu = norm(w);
nu = norm([nu,mu/tau]);
absomega = abs(omega(1:k));
pmomega = (omega(1:k)./absomega).* sqrt(absomega);
That = diag(pmomega)\( T/diag(sqrt(absomega)) );
[Zhat, pritz] = eig(That);
Z = diag(sqrt(absomega))\Zhat;
%   That = diag(omega(1:k))\T;
%   [Z, pritz] = eig(That);
normThat = norm(That,1);
pritz = diag(pritz);
nconv = 0;
indcon = zeros(1,k);
for h = 1:k
    Z(:,h) = Z(:,h)/norm(Z(:,h));
    rcond(h) = abs( (Z(:,h).*omega(1:k)).'*Z(:,h) );
    gamma(h) =  abs(Z(k,h)*tau);
    res(h) = gamma(h)*max(nu, sqrt(k)*mu/tau)/rcond(h);
    res(h) = max( res(h), eps );
%
%--------Do not accept zero eigenvalues of (T, Omega)
%
    if ((res(h) < convtol) & (abs(pritz(h)) > eps*normThat))
       nconv = nconv+1;
       indcon(nconv) = h;
    end
end 
if (nconv > 0)
   indcon = indcon(1:nconv);
end
return
