#!/usr/bin/env python
import sys

########################################################
################ Mutation Effect Analysis ##############
########################################################

# RNA codon to Amino Acid conversion dictionary
RNA_codon_to_amino_acid_dict = {}

RNA_codon_to_amino_acid_dict["UUU"] = "Phe"
RNA_codon_to_amino_acid_dict["UUC"] = "Phe"
RNA_codon_to_amino_acid_dict["UUA"] = "Leu"
RNA_codon_to_amino_acid_dict["UUG"] = "Leu"

RNA_codon_to_amino_acid_dict["UCU"] = "Ser"
RNA_codon_to_amino_acid_dict["UCC"] = "Ser"
RNA_codon_to_amino_acid_dict["UCA"] = "Ser"
RNA_codon_to_amino_acid_dict["UCG"] = "Ser"

RNA_codon_to_amino_acid_dict["UAU"] = "Tyr"
RNA_codon_to_amino_acid_dict["UAC"] = "Tyr"
RNA_codon_to_amino_acid_dict["UAA"] = "Stop"
RNA_codon_to_amino_acid_dict["UAG"] = "Stop"

RNA_codon_to_amino_acid_dict["UGU"] = "Cys"
RNA_codon_to_amino_acid_dict["UGC"] = "Cys"
RNA_codon_to_amino_acid_dict["UGA"] = "Stop"
RNA_codon_to_amino_acid_dict["UGG"] = "Trp"


RNA_codon_to_amino_acid_dict["CUU"] = "Leu"
RNA_codon_to_amino_acid_dict["CUC"] = "Leu"
RNA_codon_to_amino_acid_dict["CUA"] = "Leu"
RNA_codon_to_amino_acid_dict["CUG"] = "Leu"

RNA_codon_to_amino_acid_dict["CCU"] = "Pro"
RNA_codon_to_amino_acid_dict["CCC"] = "Pro"
RNA_codon_to_amino_acid_dict["CCA"] = "Pro"
RNA_codon_to_amino_acid_dict["CCG"] = "Pro"

RNA_codon_to_amino_acid_dict["CAU"] = "His"
RNA_codon_to_amino_acid_dict["CAC"] = "His"
RNA_codon_to_amino_acid_dict["CAA"] = "Gln"
RNA_codon_to_amino_acid_dict["CAG"] = "Gln"

RNA_codon_to_amino_acid_dict["CGU"] = "Arg"
RNA_codon_to_amino_acid_dict["CGC"] = "Arg"
RNA_codon_to_amino_acid_dict["CGA"] = "Arg"
RNA_codon_to_amino_acid_dict["CGG"] = "Arg"


RNA_codon_to_amino_acid_dict["AUU"] = "Ile"
RNA_codon_to_amino_acid_dict["AUC"] = "Ile"
RNA_codon_to_amino_acid_dict["AUA"] = "Ile"
RNA_codon_to_amino_acid_dict["AUG"] = "Met"

RNA_codon_to_amino_acid_dict["ACU"] = "Thr"
RNA_codon_to_amino_acid_dict["ACC"] = "Thr"
RNA_codon_to_amino_acid_dict["ACA"] = "Thr"
RNA_codon_to_amino_acid_dict["ACG"] = "Thr"

RNA_codon_to_amino_acid_dict["AAU"] = "Asn"
RNA_codon_to_amino_acid_dict["AAC"] = "Asn"
RNA_codon_to_amino_acid_dict["AAA"] = "Lys"
RNA_codon_to_amino_acid_dict["AAG"] = "Lys"

RNA_codon_to_amino_acid_dict["AGU"] = "Ser"
RNA_codon_to_amino_acid_dict["AGC"] = "Ser"
RNA_codon_to_amino_acid_dict["AGA"] = "Arg"
RNA_codon_to_amino_acid_dict["AGG"] = "Arg"


RNA_codon_to_amino_acid_dict["GUU"] = "Val"
RNA_codon_to_amino_acid_dict["GUC"] = "Val"
RNA_codon_to_amino_acid_dict["GUA"] = "Val"
RNA_codon_to_amino_acid_dict["GUG"] = "Val"

RNA_codon_to_amino_acid_dict["GCU"] = "Ala"
RNA_codon_to_amino_acid_dict["GCC"] = "Ala"
RNA_codon_to_amino_acid_dict["GCA"] = "Ala"
RNA_codon_to_amino_acid_dict["GCG"] = "Ala"

RNA_codon_to_amino_acid_dict["GAU"] = "Asp"
RNA_codon_to_amino_acid_dict["GAC"] = "Asp"
RNA_codon_to_amino_acid_dict["GAA"] = "Glu"
RNA_codon_to_amino_acid_dict["GAG"] = "Glu"

RNA_codon_to_amino_acid_dict["GGU"] = "Gly"
RNA_codon_to_amino_acid_dict["GGC"] = "Gly"
RNA_codon_to_amino_acid_dict["GGA"] = "Gly"
RNA_codon_to_amino_acid_dict["GGG"] = "Gly"


# Amino acid abbreviation to Amino acid symbol dictionary
amino_acid_to_symbol_dict = {}
amino_acid_to_symbol_dict["Ala"] = 'A'
amino_acid_to_symbol_dict["Arg"] = 'R'
amino_acid_to_symbol_dict["Asn"] = 'N'
amino_acid_to_symbol_dict["Asp"] = 'D'
amino_acid_to_symbol_dict["Cys"] = 'C'
amino_acid_to_symbol_dict["Gln"] = 'Q'
amino_acid_to_symbol_dict["Glu"] = 'E'
amino_acid_to_symbol_dict["Gly"] = 'G'
amino_acid_to_symbol_dict["His"] = 'H'
amino_acid_to_symbol_dict["Ile"] = 'I'
amino_acid_to_symbol_dict["Leu"] = 'L'
amino_acid_to_symbol_dict["Lys"] = 'K'
amino_acid_to_symbol_dict["Met"] = 'M'
amino_acid_to_symbol_dict["Phe"] = 'F'
amino_acid_to_symbol_dict["Pro"] = 'P'
amino_acid_to_symbol_dict["Ser"] = 'S'
amino_acid_to_symbol_dict["Thr"] = 'T'
amino_acid_to_symbol_dict["Trp"] = 'W'
amino_acid_to_symbol_dict["Tyr"] = 'Y'
amino_acid_to_symbol_dict["Val"] = 'V'
amino_acid_to_symbol_dict["Stop"] = '*'


# DNA complementary base dictionary
DNA_base_complement_dict = {}
DNA_base_complement_dict['A'] = 'T'
DNA_base_complement_dict['C'] = 'G'
DNA_base_complement_dict['G'] = 'C'
DNA_base_complement_dict['T'] = 'A'
DNA_base_complement_dict['N'] = 'N'

# nucleic acid code dictionary
symbol_to_represented_nucleic_acids_dict = {}
symbol_to_represented_nucleic_acids_dict['A'] = ['A']
symbol_to_represented_nucleic_acids_dict['C'] = ['C']
symbol_to_represented_nucleic_acids_dict['T'] = ['T']
symbol_to_represented_nucleic_acids_dict['G'] = ['G']

symbol_to_represented_nucleic_acids_dict['M'] = ['A', 'C']
symbol_to_represented_nucleic_acids_dict['R'] = ['A', 'G']
symbol_to_represented_nucleic_acids_dict['W'] = ['A', 'T']
symbol_to_represented_nucleic_acids_dict['S'] = ['C', 'G']
symbol_to_represented_nucleic_acids_dict['Y'] = ['C', 'T']
symbol_to_represented_nucleic_acids_dict['K'] = ['G', 'T']
symbol_to_represented_nucleic_acids_dict['V'] = ['A', 'C', 'G']
symbol_to_represented_nucleic_acids_dict['H'] = ['A', 'C', 'T']
symbol_to_represented_nucleic_acids_dict['D'] = ['A', 'G', 'T']
symbol_to_represented_nucleic_acids_dict['B'] = ['C', 'G', 'T']
symbol_to_represented_nucleic_acids_dict['N'] = ['A', 'C', 'G', 'T']


# Takes in the 1-indexed position in the genomic sequence, with respect to the beginning of the coding sequence
# "global_coding_sequence" is 0-indexed, and is in the form of a list
def get_effect_of_base_change(position_wrt_start_of_CDS, reference_base, mutation_base, coding_sequence, genomic_sequence_starting_at_CDS, 
				BLOSUM62_matrix, BLOSUM62_possibly_severe_missense_mutation_threshold, 
				exon_list_wrt_start_of_CDS, gene_name):
	position_in_CDS, location_of_base_change = find_position_in_CDS(position_wrt_start_of_CDS, exon_list_wrt_start_of_CDS, gene_name)
	if location_of_base_change == '5\' non-coding':
		effect_of_base_change = '5\' non-coding'
		mutation_type = '5\' non-coding'
		BLOSUM62_substitution_score = 'NA'
	elif location_of_base_change == '3\' non-coding':
		effect_of_base_change = '3\' non-coding'
		mutation_type = '3\' non-coding'
		BLOSUM62_substitution_score = 'NA'
	elif location_of_base_change == 'intron':
		if base_change_disrupts_splice_site(position_wrt_start_of_CDS, reference_base, mutation_base, 
						    exon_list_wrt_start_of_CDS, genomic_sequence_starting_at_CDS):
			effect_of_base_change = 'splice'
			mutation_type = 'splice'
		else:
			effect_of_base_change = 'intron'
			mutation_type = 'intron'
		BLOSUM62_substitution_score = 'NA'
	elif location_of_base_change == 'exon':
		amino_acid_position = ((position_in_CDS - 1) / 3) + 1
		start_of_codon = (3 * amino_acid_position) - 2
        
		# start_of_codon is 1-indexed, while coding_sequence is 0-indexed
		DNA_codon_list = coding_sequence[(start_of_codon - 1):(start_of_codon + 2)]
		DNA_codon = ''.join(DNA_codon_list)
        
		# determine old amino acid symbol
		old_amino_acid_symbol = DNA_codon_to_amino_acid_symbol(DNA_codon)
		
		# determine new amino acid symbol
		codon_offset = position_in_CDS - start_of_codon	# (0, 1, or 2)
		if DNA_codon_list[codon_offset] != reference_base:
			print 'position in genomic sequence wrt start of CDS: ' + str(position_wrt_start_of_CDS)
			print 'position in CDS: ' + str(position_in_CDS)
			print 'refbase: ' + reference_base
			print 'Base found at corresponding position in codon: ' + DNA_codon_list[codon_offset]
			print 'DNA codon: ' + ''.join(DNA_codon_list)
			print 'Index w/in DNA codon (0-indexed): ' + str(codon_offset)
			raise IOError, 'Error: mutation reference base does not match genomic sequence starting at coding sequence'
		new_DNA_codon_list = DNA_codon_list
		new_DNA_codon_list[codon_offset] = mutation_base
		new_DNA_codon = ''.join(new_DNA_codon_list)
        
		new_amino_acid_symbol = DNA_codon_to_amino_acid_symbol(new_DNA_codon)
        
		if old_amino_acid_symbol == new_amino_acid_symbol:
			# silent mutation
			effect_of_base_change = old_amino_acid_symbol + str(amino_acid_position) + '='
			BLOSUM62_substitution_score = 'NA'
			mutation_type = 'silent'
		else:
			# missense or nonsense mutation
			effect_of_base_change = old_amino_acid_symbol + str(amino_acid_position) + new_amino_acid_symbol
			BLOSUM62_substitution_score = get_BLOSUM62_substitution_score(old_amino_acid_symbol, new_amino_acid_symbol, BLOSUM62_matrix)
			if old_amino_acid_symbol == '*':
				mutation_type = 'extension'
			elif new_amino_acid_symbol == '*':
				mutation_type = 'truncation'
			elif BLOSUM62_substitution_score <= BLOSUM62_possibly_severe_missense_mutation_threshold:
				mutation_type = 'PSM'
			else:
				mutation_type = 'NSM'
	else:
		raise IOError, 'Internal error: unrecognized base change location: %s' % location_of_base_change

	return effect_of_base_change, BLOSUM62_substitution_score, mutation_type

def find_position_in_CDS(position_wrt_start_of_CDS, exon_list_wrt_start_of_CDS, gene_name):
	found_position = False
	if len(exon_list_wrt_start_of_CDS) == 0:
		raise IOError, 'Internal error: did not find any exons in gene: %s' % gene_name

	first_exon = exon_list_wrt_start_of_CDS[0]
	first_exon_start = first_exon[0]
	if position_wrt_start_of_CDS < first_exon_start:
		# position is in 5' non-coding region (or in the 5' UTR)
		position_in_CDS = -1
		location_of_base_change = '5\' non-coding'
		found_position = True
	else:
		exon_base_count = 0
		for exon in exon_list_wrt_start_of_CDS:
			[exon_start, exon_stop] = exon
			if position_wrt_start_of_CDS > exon_stop:
				# position is after this exon
				exon_length = exon_stop - exon_start + 1
				exon_base_count += exon_length
				continue
			elif position_wrt_start_of_CDS < exon_start:
				# position is in an intron
				position_in_CDS = -1
				location_of_base_change = 'intron'
				found_position = True
				break
			else:
				# position is in this exon
				position_within_exon = position_wrt_start_of_CDS - exon_start + 1
				position_in_CDS = exon_base_count + position_within_exon
				location_of_base_change = 'exon'
				found_position = True
				break
        
	if found_position == False:
		# the position is after the last exon,
		# so it is in the 3' non-coding region (or in the 3' UTR)
		position_in_CDS = -1
		location_of_base_change = '3\' non-coding'
		
	return position_in_CDS, location_of_base_change

def base_change_disrupts_splice_site(position_wrt_start_of_CDS, reference_base, mutation_base, exon_list_wrt_start_of_CDS, genomic_sequence_starting_at_CDS):
	disrupts_splice_site = False

	intron_list_wrt_start_of_CDS = get_intron_list_from_exon_list(exon_list_wrt_start_of_CDS)
	for intron in intron_list_wrt_start_of_CDS:
		[intron_start, intron_stop] = intron
		if (position_wrt_start_of_CDS == intron_start) or (position_wrt_start_of_CDS == (intron_start + 1)) or \
		   (position_wrt_start_of_CDS == intron_stop) or (position_wrt_start_of_CDS == (intron_stop - 1)):
			disrupts_splice_site = True
			# check reference base
			if genomic_sequence_starting_at_CDS[position_wrt_start_of_CDS - 1] != reference_base:
				raise IOError, 'Conflict between the parsed pileup file and the genomic sequence'
			break
	return disrupts_splice_site

# Throws an error if exon list is not sorted in increasing order of position
def get_intron_list_from_exon_list(exon_list):
	intron_list = []
	for i in range(0, len(exon_list) - 1):
		current_exon_end = exon_list[i][1]
		next_exon_start = exon_list[i+1][0]
		if current_exon_end >= next_exon_start:
			raise IOError, 'In get_intron_list_from_exon_list(): exons are out of order'
		intron_start = current_exon_end + 1
		intron_end = next_exon_start - 1
		intron = [intron_start, intron_end]
		intron_list.append(intron)
		intron_length = intron_end - intron_start + 1
		if intron_length < 4:
			raise IOError, 'Intron is too small - overlapping splice sites'
		#if (not intron_has_standard_splice_sites(intron, global_genomic_sequence_starting_at_CDS)) and \
		#   (not intron_has_U12_type_splice_sites(intron, global_genomic_sequence_starting_at_CDS)):
		#	print 'WARNING: Intron ' + str(intron) + ' does not have standard or U12-type intron splice sites'
		#	print 'start splice site: ' + str(global_genomic_sequence_starting_at_CDS[(intron[0] - 1):(intron[0] + 1)])
		#	print 'end splice site: ' + str(global_genomic_sequence_starting_at_CDS[(intron[1] - 2):(intron[1])])
	return intron_list


# The intron indices are assumed to be 1-indexed,
# and the base sequence is assumed to be 0-indexed.

def intron_has_standard_splice_sites(intron_boundaries, base_sequence):
	starting_splice_site = ['G', 'T']
	ending_splice_site = ['A', 'G']
	return intron_has_splice_sites(intron_boundaries, base_sequence, starting_splice_site, ending_splice_site)

def intron_has_U12_type_splice_sites(intron_boundaries, base_sequence):
	starting_splice_site = ['A', 'T']
	ending_splice_site = ['A', 'C']
	return intron_has_splice_sites(intron_boundaries, base_sequence, starting_splice_site, ending_splice_site)

def intron_has_splice_sites(intron_boundaries, base_sequence, starting_splice_site, ending_splice_site):
	intron_start = intron_boundaries[0]
	intron_end = intron_boundaries[1]

	intron_length = intron_end - intron_start + 1
	if intron_length < 4:
		raise IOError, 'Intron is too small - overlapping splice sites'

	if (base_sequence[(intron_start - 1):(intron_start + 1)] == starting_splice_site) and (base_sequence[(intron_end - 2):intron_end] == ending_splice_site):
		result = True
	else:
		result = False
	return result

def get_BLOSUM62_substitution_matrix(BLOSUM62_filename):
	BLOSUM62_file = open(BLOSUM62_filename, 'r')
	header_line = BLOSUM62_file.readline().rstrip()
	header_field_list = header_line.split('\t')

	BLOSUM62_matrix = {}
	for line in BLOSUM62_file:
		line = line.strip()
		field_list = line.split('\t')

		from_aa = field_list[0]
		BLOSUM62_matrix[from_aa] = {}

		for i in range(1, len(field_list)):
			to_aa = header_field_list[i]
			score = int(field_list[i])
			BLOSUM62_matrix[from_aa][to_aa] = score
	return BLOSUM62_matrix

def get_BLOSUM62_substitution_score(old_amino_acid_symbol, new_amino_acid_symbol, BLOSUM62_matrix):
	if (old_amino_acid_symbol == '*') or (new_amino_acid_symbol == '*'):
		BLOSUM62_substitution_score = 'NA'
	else:
		if new_amino_acid_symbol == '=':
			new_amino_acid_symbol = old_amino_acid_symbol
		BLOSUM62_substitution_score = BLOSUM62_matrix[old_amino_acid_symbol][new_amino_acid_symbol]

	return BLOSUM62_substitution_score

def DNA_codon_to_amino_acid_symbol(DNA_codon):
	# replace 'T' with 'U' to change DNA codon to RNA codon
	RNA_codon = DNA_codon.replace('T', 'U')

	amino_acid_abbreviation = RNA_codon_to_amino_acid_dict[RNA_codon]
	amino_acid_symbol = amino_acid_to_symbol_dict[amino_acid_abbreviation]

	return amino_acid_symbol

def get_restriction_enzyme_to_recognition_sequence_dict(restriction_enzyme_recognition_sequence_filename):
	restriction_enzyme_recognition_sequence_file = open(restriction_enzyme_recognition_sequence_filename, 'r')

	restriction_enzyme_to_recognition_sequence_dict = {}
	for line in restriction_enzyme_recognition_sequence_file:
		line = line.rstrip('\n\r')
		field_list = line.split('\t')
		if len(field_list) != 3:
			raise IOError, 'There should be exactly three fields on each line of the restriction enzyme to recognition sequence file'
		restriction_enzyme = field_list[0]
		recognition_sequence = field_list[1]

		if restriction_enzyme in restriction_enzyme_to_recognition_sequence_dict:
			raise IOError, 'Saw restriction enzyme twice'
		restriction_enzyme_to_recognition_sequence_dict[restriction_enzyme] = recognition_sequence
	return restriction_enzyme_to_recognition_sequence_dict

# restriction_enzyme_gained_string, restriction_enzyme_lost_string
def get_restriction_enzymes_gained_and_lost(position_wrt_start_of_CDS, reference_base, mutation_base, restriction_enzyme_to_recognition_sequence_dict, 
					    CDS_offset_wrt_start_of_genomic_sequence, num_prefix_and_suffix_bases_added_to_genomic_sequence_of_global_gene,
					    genomic_sequence):
	#print 'position wrt start of CDS: ' + str(position_wrt_start_of_CDS)
	position_wrt_start_of_genomic_sequence = position_wrt_start_of_CDS + CDS_offset_wrt_start_of_genomic_sequence
	#print 'position wrt start of genomic: ' + str(position_wrt_start_of_genomic_sequence)
	#print 'position_wrt_start_of_CDS', position_wrt_start_of_CDS
	#print 'global_CDS_offset_wrt_start_of_genomic_sequence', global_CDS_offset_wrt_start_of_genomic_sequence
	#print 'position_wrt_start_of_genomic_sequence', position_wrt_start_of_genomic_sequence

	# consider prefix and suffix bases added to genomic sequence
	[num_prefix_bases, num_suffix_bases] = num_prefix_and_suffix_bases_added_to_genomic_sequence_of_global_gene

	if (position_wrt_start_of_genomic_sequence <= num_prefix_bases) or (position_wrt_start_of_genomic_sequence > (len(genomic_sequence) - num_suffix_bases)):
		raise IOError, 'Internal error: candidate position does not map to genomic sequence, or it maps to the prefix or suffix added to the genomic sequence.\n' + \
			       'The added prefix and suffix may not match the corresponding bases in the tilling sequence.'

	restriction_enzymes_gained = []
	restriction_enzymes_lost = []
	for RE in restriction_enzyme_to_recognition_sequence_dict:
		recognition_sequence = restriction_enzyme_to_recognition_sequence_dict[RE]
		recognition_sequence_length = len(recognition_sequence)
		start_index = (position_wrt_start_of_genomic_sequence - 1) - (recognition_sequence_length - 1)
		stop_index = (position_wrt_start_of_genomic_sequence - 1) + (recognition_sequence_length - 1)

		if start_index < num_prefix_bases:
			start_index = num_prefix_bases
		if stop_index > (len(genomic_sequence) - 1 - num_suffix_bases):
			stop_index = (len(genomic_sequence) - 1 - num_suffix_bases)

		search_sequence = genomic_sequence[start_index : (stop_index + 1)]

		
		mutation_index_in_search_sequence = (position_wrt_start_of_genomic_sequence - 1) - start_index
		if search_sequence[mutation_index_in_search_sequence] != reference_base:
			print 'start index of search sequence (wrt genomic): ' + str(start_index)
			print 'mutation index in search sequence: ' + str(mutation_index_in_search_sequence)
			print search_sequence
			print search_sequence[mutation_index_in_search_sequence], ', ', genomic_sequence[(position_wrt_start_of_genomic_sequence - 1)]
			print reference_base
			raise IOError, 'Internal error in finding position of mutation'

		search_sequence_after_mutation = [element for element in search_sequence]
		search_sequence_after_mutation[mutation_index_in_search_sequence] = mutation_base

		# get sequence of complementary strand, before and after mutation
		complement_of_search_sequence = [DNA_base_complement_dict[base] for base in search_sequence]
		complement_of_search_sequence_after_mutation = [DNA_base_complement_dict[base] for base in search_sequence_after_mutation]
		complement_of_search_sequence.reverse()
		complement_of_search_sequence_after_mutation.reverse()
		
		RE_binds_sequence_before = is_RE_sequence_in_sequence(recognition_sequence, ''.join(search_sequence)) or \
						is_RE_sequence_in_sequence(recognition_sequence, ''.join(complement_of_search_sequence))

		RE_binds_sequence_after = is_RE_sequence_in_sequence(recognition_sequence, ''.join(search_sequence_after_mutation)) or \
						is_RE_sequence_in_sequence(recognition_sequence, ''.join(complement_of_search_sequence_after_mutation))

		# DEBUGGING:
		#if (position == 232) and (recognition_sequence == 'GAGGAG'):
		#	print reference_base + '->' + mutation_base
		#	print recognition_sequence + ' : ' + ''.join(search_sequence) + ', ' + ''.join(search_sequence_after_mutation)
		#	print 'RE binds before mutation is ' + str(RE_binds_sequence_before)
		#	print 'RE binds after mutation is ' + str(RE_binds_sequence_after)
		#	raise IOError, 'Debugging'

		if (RE_binds_sequence_before == True) and (RE_binds_sequence_after == False):
			restriction_enzymes_lost.append(RE)
		elif (RE_binds_sequence_before == False) and (RE_binds_sequence_after == True):
			restriction_enzymes_gained.append(RE)
	
	restriction_enzymes_gained_string = ','.join(restriction_enzymes_gained)
	restriction_enzymes_lost_string = ','.join(restriction_enzymes_lost)
		
	return restriction_enzymes_gained_string, restriction_enzymes_lost_string

def is_RE_sequence_in_sequence(RE_sequence, sequence):
	for i in range(0, len(sequence) - len(RE_sequence) + 1):
		window_sequence = sequence[i:(i + len(RE_sequence))]
		if sequence_matches_template(window_sequence, RE_sequence):
			return True
	return False

def sequence_matches_template(sequence, template):
	if len(sequence) != len(template):
		return False
	for i in range(0, len(sequence)):
		if not(sequence[i] in symbol_to_represented_nucleic_acids_dict[template[i]]):
			return False
	return True


