/*
**++
**  FACILITY:  University of California, Davis
**              Computer Science Department
**              Theory Lab
**
**  PROGRAM NAME:       XPARAL
**
**  MODULE DESCRIPTION:
**
**      This is the header file for the dynamic string matching program.
**
*/

#include <ctype.h>
#include <errno.h>
extern int      errno;
#include <fcntl.h>
#include <float.h>
#include <math.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <float.h>

#include <X11/cursorfont.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

#define  GLOBAL                 1
#define  ENDGAPS_FREE           2
#define  LOCAL                  3
#define  SUBSTRING_1_OF_2       4
#define  GLOBAL_GAPS            5
#define  ENDGAPS_FREE_GAPS      6
#define  LOCAL_GAPS             7
#define  SUBSTRING_1_OF_2_GAPS  8
#define  GLOBAL_CONVEX		9
#define  ENDGAPS_FREE_CONVEX    10
#define  LOCAL_CONVEX           11
#define  SUBSTRING_1_OF_2_CONVEX 12

#ifndef INFINITY
#define  INFINITY DBL_MAX
#endif

#define  DIAG   'D'
#define  SIDE   'S'
#define  UP     'U'

inline
int     equal(  const double    d1,
                const double    d2)
{
    double      a1,a2,adiff;

    if  (d1 == d2)
        return 1;

    adiff = d2 - d1;

    if ( adiff < 0.0 )
        adiff = -adiff;

    if ( adiff < 1.0e-6 )
	return 1;
    
    a1 = (d1 >= 0) ? d1 : -d1;
    a2 = (d2 >= 0) ? d2 : -d2;

    if ( a1 < a2 )
        a1 = a2;

    if ( a1 < 1.0e6 )
	return 0;

    if ( (adiff/a1) < 1.0e-6)
	return 1;

    return 0;

}


inline
void    swap(   double  *d1,
                double  *d2)
{
    double      temp;

    temp = *d1;
    *d1 = *d2;
    *d2 = temp;
}

/*
**  This structure defines each cell in the array used for the
**  dynamic programming algorithm for finding the highest valued
**  approximate matching of str1 and str2.
*/
class   Max_node        {
public:
    double      value,
                side_value,
                top_value;
    short       e_gap_start,
		f_gap_start;
    union {
        int             flags;
        struct {
            unsigned    diag_opt : 1;
            unsigned    side_opt : 1;
            unsigned    side_cont : 1;
            unsigned    side_stop : 1;
            unsigned    top_opt : 1;
            unsigned    top_cont : 1;
            unsigned    top_stop : 1;
        }       flag;
    }   path;
};

/*
** this structure holds the energy values for 
** matches, misses, insertion or deletion, and gaps
** for a given function.
*/
class   Funct   {
public:
    double      match_val;
    double      mis_val;
    double      inordel_val;
    double      gaps_val;
    int         match_ct;
    int         mis_ct;
    int         inordel_ct;
    int         gaps_ct;
    int         is_2D_opt;
    int         opt_align_ct;
    char        *align1;
    char        *align2;

    Funct();
    Funct(const Funct*);
    ~Funct();
    void        clear();
    int         equal(const Funct*, const int) const;
};

inline
Funct::Funct()
{
    this->match_val = 0.0;
    this->mis_val = 0.0;
    this->inordel_val = 0.0;
    this->gaps_val = 0.0;
    this->match_ct = 0;
    this->mis_ct = 0;
    this->inordel_ct = 0;
    this->gaps_ct = 0;
    this->is_2D_opt = 0;
    this->opt_align_ct = 0;
    this->align1 = NULL;
    this->align2 = NULL;
}

inline
Funct::Funct(const Funct *fun)
{
    this->match_val = fun->match_val;
    this->mis_val = fun->mis_val;
    this->inordel_val = fun->inordel_val;
    this->gaps_val = fun->gaps_val;
    this->match_ct = fun->match_ct;
    this->mis_ct = fun->mis_ct;
    this->inordel_ct = fun->inordel_ct;
    this->gaps_ct = fun->gaps_ct;
    this->is_2D_opt = fun->is_2D_opt;
    this->opt_align_ct = fun->opt_align_ct;
    this->align1 = NULL;
    this->align2 = NULL;
}

inline
Funct::~Funct()
{
    if  (this->align1 != NULL)
	{
        delete this->align1;
	this->align1 = NULL;
	}
    if  (this->align2 != NULL)
	{
        delete this->align2;
	this->align2 = NULL;
	}
}

inline
void    Funct::clear()
{
    this->match_val = 0.0;
    this->mis_val = 0.0;
    this->inordel_val = 0.0;
    this->gaps_val = 0.0;
    this->match_ct = 0;
    this->mis_ct = 0;
    this->inordel_ct = 0;
    this->gaps_ct = 0;
    delete this->align1;
    this->align1 = NULL;
    delete this->align2;
    this->align2 = NULL;
}

inline
int     Funct::equal(   const Funct     *f1,
                        const int       opt_function = 0)
        const
{
    switch (opt_function)
    {
        case 1:
        case 2:
        case 3:
            return(   (::equal(this->match_val, f1->match_val))
                   && (::equal(this->mis_val, f1->mis_val))
                   && (::equal(this->inordel_val, f1->inordel_val)));
        case 4:
            return(   (::equal(this->match_val + this->mis_val,
                               f1->match_val + f1->mis_val))
                   && (::equal(this->inordel_val, f1->inordel_val))
                   && (::equal(this->gaps_val, f1->gaps_val)));
        case 5:
            return(   (::equal(this->match_val + this->inordel_val,
                               f1->match_val + f1->inordel_val))
                   && (::equal(this->mis_val, f1->mis_val))
                   && (::equal(this->gaps_val, f1->gaps_val)));
        case 6:
            return(   (::equal(this->match_val + this->gaps_val,
                               f1->match_val + f1->gaps_val))
                   && (::equal(this->mis_val, f1->mis_val))
                   && (::equal(this->inordel_val, f1->inordel_val)));
        case 7:
            return(   (::equal(this->match_val, f1->match_val))
                   && (::equal(this->mis_val + this->inordel_val,
                               f1->mis_val + f1->inordel_val))
                   && (::equal(this->gaps_val, f1->gaps_val)));
        case 8:
            return(   (::equal(this->match_val, f1->match_val))
                   && (::equal(this->mis_val + this->gaps_val,
                               f1->mis_val + f1->gaps_val))
                   && (::equal(this->inordel_val, f1->inordel_val)));
        case 9:
            return(   (::equal(this->match_val, f1->match_val))
                   && (::equal(this->mis_val, f1->mis_val))
                   && (::equal(this->inordel_val + this->gaps_val,
                               f1->inordel_val + f1->gaps_val)));
        default:
            return(   (::equal(this->match_val, f1->match_val))
                   && (::equal(this->mis_val, f1->mis_val))
                   && (::equal(this->inordel_val, f1->inordel_val))
                   && (::equal(this->gaps_val, f1->gaps_val)));
    }
}


class   Funct_list      {
public:
        Funct           *fun;
        Funct_list      *next;

        Funct_list() {};
        ~Funct_list();
};

inline
Funct_list::~Funct_list()
{
    delete this->fun;
    this->fun = NULL;
}


/*
** this structure defines a line in terms of 
** a slope m and a y-intercept b
*/
class   Line    {
public:
        int     m_is_inf;
        double  m;
        double  b;

        Line(const int, const double, const double);
        Line(const Line*);
        ~Line() {};
        int     equal(Line*);
};

inline
Line::Line(     const int       new_m_is_inf = 0,
                const double    new_m = 0.0,
                const double    new_b = 0.0)
{
    this->m_is_inf = new_m_is_inf;
    this->m = new_m;
    this->b = new_b;
}

inline
Line::Line(     const Line      *line)
{
    this->m_is_inf = line->m_is_inf;
    this->m = line->m;
    this->b = line->b;
}

inline
int     Line::equal(    Line    *l1)
{
    if  (this->m_is_inf)
    {
        if  (   (l1->m_is_inf)
             && (::equal(l1->b, this->b)))
            return 1;
        else
            return 0;
    }
    else if  (l1->m_is_inf)
        return 0;
    else if  (   (::equal(l1->m, this->m))
              && (::equal(l1->b, this->b)))
        return 1;
    else
        return 0;
}


/*
** this structure holds the coordinate values
** for a parameterized point
*/

class   Point   {
public:
    double      v;
    double      w;
    double      x;
    double      y;
    double      max_val;
    double      input_fun_val;
    double      diff_val;
    double      dist_sq_val;
    Point       *next;

    Point(const double, const double, const double, const double);
    Point(const Point*);
    Point(const Line*, const Line*, const double, const double);
    ~Point() {};
    int  equal(Point*) const;
    double eval(const Funct*, const int);
    void set_vals(const Funct*, const int, const double, const double, const Funct*);
    void trans_in(const int);
    void trans_out(const int);
};

inline
Point::Point(   const double    new_v,
                const double    new_w,
                const double    new_x = 0.0,
                const double    new_y = 0.0)
{
    this->v = new_v;
    this->w = new_w;
    this->x = new_x;
    this->y = new_y;
    this->max_val = 0.0;
    this->input_fun_val = 0.0;
    this->diff_val = 0.0;
    this->dist_sq_val = 0.0;
    this->next = NULL;
}

inline
Point::Point(   const Point*    pt)
{
    this->v = pt->v;
    this->w = pt->w;
    this->x = pt->x;
    this->y = pt->y;
    this->max_val = pt->max_val;
    this->input_fun_val = pt->input_fun_val;
    this->diff_val = pt->diff_val;
    this->dist_sq_val = pt->dist_sq_val;
    this->next = pt->next;
}

inline
Point::Point(   const Line      *line1,
                const Line      *line2,
                const double    v,
                const double    w)
{
    this->v = v;
    this->w = w;
    this->max_val = 0.0;
    this->input_fun_val = 0.0;
    this->diff_val = 0.0;
    this->dist_sq_val = 0.0;
    this->next = NULL;
    if  (line1->m_is_inf)
    {
        this->x = line1->b;
        this->y = (line2->m) * (this->x) + (line2->b);
    }
    else if  (line2->m_is_inf)
    {
        this->x = line2->b;
        this->y = (line1->m) * (this->x) + (line1->b);
    }
    else
    {
        this->x = (double)(line2->b - line1->b) / (line1->m - line2->m);
        this->y = (line1->m) * (this->x) + (line1->b);
    }
}

inline
int     Point::equal(   Point   *pt1)
        const
{
    return (   (::equal(this->x, pt1->x))
            && (::equal(this->y, pt1->y))
            && (::equal(this->v, pt1->v))
            && (::equal(this->w, pt1->w)));
}

inline
void    Point::trans_in(        const int       opt_function)
{
    switch (opt_function)
    {
        case 1:
        case 6:
            break;
        case 2:
        case 8:
            swap(&this->v, &this->x);
            break;
        case 3:
        case 9:
            swap(&this->v, &this->y);
            swap(&this->v, &this->x);
            break;
        case 4:
            swap(&this->w, &this->x);
            swap(&this->w, &this->y);
            break;
        case 5:
            swap(&this->w, &this->y);
            break;
        case 7:
            swap(&this->v, &this->x);
            swap(&this->w, &this->y);
            break;
    }
}

inline
void    Point::trans_out(       const int       opt_function)
{
    switch (opt_function)
    {
        case 1:
        case 6:
            break;
        case 2:
        case 8:
            swap(&this->v, &this->x);
            break;
        case 3:
        case 9:
            swap(&this->v, &this->x);
            swap(&this->v, &this->y);
            break;
        case 4:
            swap(&this->w, &this->y);
            swap(&this->w, &this->x);
            break;
        case 5:
            swap(&this->w, &this->y);
            break;
        case 7:
            swap(&this->w, &this->y);
            swap(&this->v, &this->x);
            break;
    }
}

inline
double  Point::eval(    const Funct     *fun,
                        const int       opt_function)
{
    double      val = 0.0;

    this->trans_in(opt_function);
    val = (fun->match_val * this->v) + (fun->mis_val * this->x)
        + (fun->inordel_val * this->y) + (fun->gaps_val * this->w);
    this->trans_out(opt_function);

    return val;
}

inline
void    Point::set_vals(        const Funct*    opt_fun,
                                const int       opt_function,
                                const double    min_x,
                                const double    min_y,
                                const Funct*    input_fun = NULL)
{
    this->max_val = this->eval(opt_fun, opt_function);

    if  (input_fun > 0)
    {
        this->input_fun_val = this->eval(input_fun, opt_function);
        this->diff_val = this->max_val - this->input_fun_val;
    }

    this->dist_sq_val = ((this->x - min_x) * (this->x - min_x))
                      + ((this->y - min_y) * (this->y - min_y));
}


/*
** this structure defines a polygon by its optimal function
** and a linked list of corner points
*/
class   Poly    {
public:
    double      diff_opt_pt;
    double      dist_sq_opt_pt;
    int         min_diff_region_dim;
    int         opt_align_ct;
    Funct       *fun;
    Point       *corners;
    Poly        *next;

    Poly();
    ~Poly();
};

inline
Poly::Poly()
{
    this->diff_opt_pt = 0.0;
    this->dist_sq_opt_pt = 0.0;
    this->min_diff_region_dim = 0;
    this->opt_align_ct = 0;
    this->fun = NULL;
    this->corners = NULL;
    this->next = NULL;
}

inline
Poly::~Poly()
{
    Point       *pt1,
                *pt2;

    delete this->fun;
    this->fun = NULL;
    pt1 = this->corners;
    while (pt1 != NULL)
    {
        pt2 = pt1;
        pt1 = pt1->next;
        delete pt2;
	pt2 = NULL;
    }
}

/*
** this structure lists the information needed to find the next
** corner of the polygon currently being found
*/
class   Todo;
class   Find_info       {
public:
    Funct       *fun;
    Line        *edgeline0;
    int         above0;
    int         incr0;
    Point       *corner1;
    Funct       *fun1;
    Line        *edgeline1;
    int         above1;
    int         incr1;
    Point       *corner2;
    Funct       *fun2;
    Line        *edgeline2;
    int         above2;
    int         incr2;

    Find_info();
    Find_info(Todo*);
    ~Find_info();
};

/*
** this structure lists the functions that are ready for
** the polygon to be found
*/
class   Todo    {
public:
    Funct       *fun;
    double      diff_from_input;
    double      distance_sq;
    Point       *start;
    Line        *edgeline;
    int         above;
    int         incr;
    Todo        *next;

    Todo(Funct*, Point*, Line*, int, int);
    Todo(Find_info*);
    ~Todo();
};
/*
** functions for class Find_info
*/
inline
Find_info::Find_info()
{
    this->fun = NULL;
    this->edgeline0 = NULL;
    this->above0 = 0;
    this->incr0 = 0;
    this->corner1 = NULL;
    this->fun1 = NULL;
    this->edgeline1 = NULL;
    this->above1 = 0;
    this->incr1 = 0;
    this->corner2 = NULL;
    this->fun2 = NULL;
    this->edgeline2 = NULL;
    this->above2 = 0;
    this->incr2 = 0;
}

inline
Find_info::Find_info(   Todo    *todoptr)
{
    this->fun = todoptr->fun;
    this->edgeline0 = NULL;
    this->above0 = 0;
    this->incr0 = 0;
    this->corner1 = NULL;
    this->fun1 = NULL;
    this->edgeline1 = NULL;
    this->above1 = 0;
    this->incr1 = 0;
    this->corner2 = todoptr->start;
    this->fun2 = NULL;
    this->edgeline2 = todoptr->edgeline;
    this->above2 = todoptr->above;
    this->incr2 = todoptr->incr;
}

inline
Find_info::~Find_info()
{
    delete this->fun;
    delete this->edgeline0;
    delete this->corner1;
    delete this->fun1;
    delete this->edgeline1;
    delete this->corner2;
    delete this->fun2;
    delete this->edgeline2;
    
    this->fun = NULL;
    this->edgeline0 = NULL;
    this->corner1 = NULL;
    this->fun1 = NULL;
    this->edgeline1 = NULL;
    this->corner2 = NULL;
    this->fun2 = NULL;
    this->edgeline2 = NULL;
}

/*
**      Functions for the class Todo
*/
inline
Todo::Todo(     Funct*  in_fun,
                Point*  in_pt,
                Line*   in_line,
                int     in_above,
                int     in_incr)
{
    this->fun = in_fun;
    this->diff_from_input = 0.0;
    this->distance_sq = 0.0;
    this->start = in_pt;
    this->edgeline = in_line;
    this->above = in_above;
    this->incr = in_incr;
    this->next = NULL;
}

inline
Todo::Todo(     Find_info       *findptr)
{
    this->fun = findptr->fun1;

    this->edgeline = new Line(findptr->edgeline1);
    this->above = 1 - findptr->above1;

    if  (   (findptr->corner1->diff_val < findptr->corner2->diff_val)
         || (   (findptr->corner1->diff_val == findptr->corner2->diff_val)
             && (findptr->corner1->dist_sq_val <= findptr->corner2->dist_sq_val)))
    {
        this->diff_from_input = findptr->corner1->diff_val;
        this->distance_sq = findptr->corner1->dist_sq_val;
        this->start = new Point(findptr->corner1);
        this->incr = findptr->incr1;
    }
    else
    /*  (   (findptr->corner1->diff_val > findptr->corner2->diff_val)              */
    /*   || (   (findptr->corner1->diff_val == findptr->corner2->diff_val)         */
    /*       && (findptr->corner1->dist_sq_val > findptr->corner2->dist_sq_val)))  */
    {
        this->diff_from_input = findptr->corner2->diff_val;
        this->distance_sq = findptr->corner2->dist_sq_val;
        this->start = new Point(findptr->corner2);
        this->incr = 1 - findptr->incr1;
    }

    this->next = NULL;
}

inline
Todo::~Todo()
{
    delete this->fun;
    delete this->start;
    delete this->edgeline;
    this->fun = NULL;
    this->start = NULL;
    this->edgeline = NULL;
}


class   Image_data      {
public:
    Display     *disp;
    Window      win;
    GC          gc;
    Pixmap      pix;
    Dimension   width,
                height;
};

// Mars - We need callbacks with char* and void* defined.
typedef struct  {
                char    *title;
                void    (*callback)(Widget, char*, XtPointer);
                char    *arg;
}       int_menu_struct;


typedef void    (*callback_type)(Widget, void*, XtPointer);
typedef struct  {
                 char    *title;
                 void    (*callback)(Widget, void*, XtPointer);
                 char    *arg;
}       int_menu_struct2;


/* structure for traceback stack nodes */
typedef struct _stpoint {
     int i;
     int j;
} stpoint;

/* structure for n^2logn time fill list */
typedef struct _lpoint {
     int p;
     int b;
} lpoint;
