/*
**++
**  FACILITY:  University of California, Davis
**              Computer Science Department
**              Theory Lab
**
**  PROGRAM NAME:       XPARAL
**
**  MODULE DESCRIPTION:
**
**      This file conatins the code for performing dynamic string
**      matching using constant costs.
**
*/

#include "header.h"
#include "globals_s.h"

#include "noweights.h"
#include "dynam.h"

extern void     get_first_align_path(Max_node*);
extern int      get_opt_align_ct(Max_node*);
extern void     calc_align_from_path(Max_node*, Funct*);

/*
**  Access to the array
*/

#define access_mat(a,row,column)  ((a)+((row)*(len2+1))+(column))


/****************************************************************/
/*  Initializes the dynamic array for                           */
/*  Global optimization without gaps.                           */
/****************************************************************/
inline
static void     init_mat1(      Max_node        *array,
                                const Point     *pt)
{
    Max_node    *curr;
    int         i;

    array->value = 0.0;
    array->path.flags = 0;
    curr = array;
    for  (i = 1; i <= len1; i++)
    {
//      curr = access_mat(array, i, 0);
        curr += (len2 + 1);
        curr->value = ((double)i) * pt->y * WEIGHT_INORDEL;
        curr->path.flags = 0;
        curr->path.flag.top_opt = 1;
    }
    curr = array;
    for  (i = 1; i <= len2; i++)
    {
//      curr = access_mat(array, 0, i);
        curr++;
        curr->value = ((double)i) * pt->y * WEIGHT_INORDEL;
        curr->path.flags = 0;
        curr->path.flag.side_opt = 1;
    }
}


/****************************************************************/
/*  Initializes the dynamic array for                           */
/*    End-gaps Free optimization without gaps.                  */
/****************************************************************/
inline
static void     init_mat2(      Max_node        *array,
                                const Point     *pt)
{
    Max_node    *curr;
    int         i;

    array->value = 0.0;
    array->path.flags = 0;
    curr = array;
    for  (i = 1; i <= len1; i++)
    {
//      curr = access_mat(array, i, 0);
        curr += (len2 + 1);
        curr->value = 0.0;
        curr->path.flags = 0;
    }
    curr = array;
    for  (i = 1; i <= len2; i++)
    {
//      curr = access_mat(array, 0, i);
        curr++;
        curr->value = 0.0;
        curr->path.flags = 0;
    }
}


/****************************************************************/
/*  Initializes the dynamic array for                           */
/*    Local optimization without gaps.                          */
/****************************************************************/
inline
static void     init_mat3(      Max_node        *array,
                                const Point     *pt)
{
    Max_node    *curr;
    int         i;

    array->value = 0.0;
    array->path.flags = 0;
    curr = array;
    for  (i = 1; i <= len1; i++)
    {
//      curr = access_mat(array, i, 0);
        curr += (len2 + 1);
        curr->value = 0.0;
        curr->path.flags = 0;
    }
    curr = array;
    for  (i = 1; i <= len2; i++)
    {
//      curr = access_mat(array, 0, i);
        curr++;
        curr->value = 0.0;
        curr->path.flags = 0;
    }
}


/****************************************************************/
/*  Initializes the dynamic array for                           */
/*    1st as Substring of 2nd optimization without gaps.        */
/****************************************************************/
inline
static void     init_mat4(      Max_node        *array,
                                const Point     *pt)
{
    Max_node    *curr;
    int         i;

    array->value = 0.0;
    array->path.flags = 0;
    curr = array;
    for  (i = 1; i <= len1; i++)
    {
//      curr = access_mat(array, i, 0);
        curr += (len2 + 1);
        curr->value = ((double)i) * pt->y * WEIGHT_INORDEL;
        curr->path.flags = 0;
        curr->path.flag.top_opt = 1;
    }
    curr = array;
    for  (i = 1; i <= len2; i++)
    {
//      curr = access_mat(array, 0, i);
        curr++;
        curr->value = 0.0;
        curr->path.flags = 0;
    }
}


/****************************************************************/
/*  Initializes the dynamic array for                           */
/*    Global optimization with gaps.                            */
/****************************************************************/
inline
static void     init_mat5gap(   Max_node        *array,
                                const Point     *pt)
{
    int         i;
    Max_node    *curr;

    array->value = 0.0;
    array->side_value = -DBL_MAX;
    array->top_value = -DBL_MAX;
    array->path.flags = 0;

    curr = array;
    for  (i = 1; i <= len1; i++)
    {
//      curr = access_mat(array, i, 0);
        curr += (len2 + 1);
        curr->value = (((double)i) * pt->y * WEIGHT_INORDEL)
                    + (pt->w * WEIGHT_GAP);
        curr->side_value = -DBL_MAX;
        curr->top_value = (((double)i) * pt->y * WEIGHT_INORDEL)
                    + (pt->w * WEIGHT_GAP);
        curr->path.flags = 0;
        curr->path.flag.top_opt = curr->path.flag.top_cont = 1;
    }

    curr = array;
    for  (i = 1; i <= len2; i++)
    {
//      curr = access_mat(array, 0, i);
        curr++;
        curr->value = (((double)i) * pt->y * WEIGHT_INORDEL)
                    + (pt->w * WEIGHT_GAP);
        curr->side_value = (((double)i) * pt->y * WEIGHT_INORDEL)
                    + (pt->w * WEIGHT_GAP);
        curr->top_value = -DBL_MAX;
        curr->path.flags = 0;
        curr->path.flag.side_opt = curr->path.flag.side_cont = 1;
    }
}


/****************************************************************/
/*  Initializes the dynamic array for                           */
/*    End-gaps Free optimization with gaps.                     */
/****************************************************************/
inline
static void     init_mat6gap(   Max_node        *array,
                                const Point     *pt)
{
    int         i;
    Max_node    *curr;

    array->value = 0.0;
    array->side_value = 0.0;
    array->top_value = 0.0;
    array->path.flags = 0;
    curr = array;
    for  (i = 1; i <= len1; i++)
    {
//      curr = access_mat(array, i, 0);
        curr += (len2 + 1);
        curr->value = 0.0;
        curr->side_value = -DBL_MAX;
        curr->top_value = 0.0;
        curr->path.flags = 0;
    }
    curr = array;
    for  (i = 1; i <= len2; i++)
    {
//      curr = access_mat(array, 0, i);
        curr++;
        curr->value = 0.0;
        curr->side_value = 0.0;
        curr->top_value = -DBL_MAX;
        curr->path.flags = 0;
    }
}


/****************************************************************/
/*  Initializes the dynamic array for                           */
/*    Local optimization with gaps.                             */
/****************************************************************/
inline
static void     init_mat7gap(   Max_node        *array,
                                const Point     *pt)
{
    int         i;
    Max_node    *curr;

    array->value = 0.0;
    array->side_value = 0.0;
    array->top_value = 0.0;
    array->path.flags = 0;
    curr = array;
    for  (i = 1; i <= len1; i++)
    {
//      curr = access_mat(array, i, 0);
        curr += (len2 + 1);
        curr->value = 0.0;
        curr->side_value = -DBL_MAX;
        curr->top_value = 0.0;
        curr->path.flags = 0;
    }
    curr = array;
    for  (i = 1; i <= len2; i++)
    {
//      curr = access_mat(array, 0, i);
        curr++;
        curr->value = 0.0;
        curr->side_value = 0.0;
        curr->top_value = -DBL_MAX;
        curr->path.flags = 0;
    }
}


/****************************************************************/
/*  Initializes the dynamic array for                           */
/*    1st as Substring of 2nd optimization with gaps.           */
/****************************************************************/
inline
static void     init_mat8gap(   Max_node        *array,
                                const Point     *pt)
{
    int         i;
    Max_node    *curr;

    array->value = 0.0;
    array->side_value = 0.0;
    array->top_value = pt->w * WEIGHT_GAP;
    array->path.flags = 0;

    curr = array;
    for  (i = 1; i <= len1; i++)
    {
//      curr = access_mat(array, i, 0);
        curr += (len2 + 1);
        curr->value = (((double)i) * pt->y * WEIGHT_INORDEL)
                    + (pt->w * WEIGHT_GAP);
        curr->side_value = -DBL_MAX;
        curr->top_value = (((double)i) * pt->y * WEIGHT_INORDEL)
                    + (pt->w * WEIGHT_GAP);
        curr->path.flags = 0;
        curr->path.flag.top_opt = curr->path.flag.top_cont = 1;
    }

    curr = array;
    for  (i = 1; i <= len2; i++)
    {
//      curr = access_mat(array, 0, i);
        curr++;
        curr->value = 0.0;
        curr->side_value = 0.0;
        curr->top_value = -DBL_MAX;
        curr->path.flags = 0;
    }
}

/****************************************************************/
/*  Initializes the dynamic array for                           */
/*    Global optimization with convex gaps.                     */
/****************************************************************/
inline
static void     init_mat9gap_logc(   Max_node        *array,
                                     const Point     *pt)
{
    int         i,j;
    Max_node    *curr, *temp;

    curr = access_mat(array, 0, 0);
    curr->value = 0.0;
    curr->path.flags = 0;
    curr->e_gap_start=0;
    curr->f_gap_start=0;

/* INITIALIZE FIRST COLUMN OF V & F TABLES    */
/* V = array.value,opts F=array.top_val,f_len */
    for  (i = 1; i <= len1; i++)
    {
        curr = access_mat(array, i, 0);
        curr->value = (pt->y*WEIGHT_INORDEL_CONV(i) ) + ( pt->w * WEIGHT_GAP_CONV );
        curr->f_gap_start = 0;
        curr->path.flags = 0;
        curr->path.flag.top_opt = 1;
    }
/* INITIALIZE FIRST ROW OF V & E TABLES        */
/* V = array.value,opts E=array.side_val,e_len */
    for  (j = 1; j <= len2; j++)
    {
        curr = access_mat(array, 0, j);
        curr->value = (pt->y*WEIGHT_INORDEL_CONV(j) ) + ( pt->w * WEIGHT_GAP_CONV );
	curr->e_gap_start = 0;
        curr->path.flags = 0;
        curr->path.flag.side_opt = 1;
    }
}

/****************************************************************/
/*  Initializes the dynamic array for                           */
/*  End gaps free optimization with convex gaps.                */
/****************************************************************/
inline
static void     init_mat10gap_logc(   Max_node        *array,
                                     const Point     *pt)
{
    int         i,j;
    Max_node    *curr, *temp;

    curr = access_mat(array, 0, 0);
    curr->value = 0.0;
    curr->path.flags = 0;
    curr->e_gap_start=0;
    curr->f_gap_start=0;

/* INITIALIZE FIRST COLUMN OF V & F TABLES    */
/* V = array.value,opts F=array.top_val,f_len */
    for  (i = 1; i <= len1; i++)
    {
        curr = access_mat(array, i, 0);
	/* endgaps free */
        curr->value = 0;
        curr->f_gap_start = 0;
        curr->path.flags = 0;
    }
/* INITIALIZE FIRST ROW OF V & E TABLES        */
/* V = array.value,opts E=array.side_val,e_len */
    for  (j = 1; j <= len2; j++)
    {
        curr = access_mat(array, 0, j);
	/*endgaps free*/
        curr->value = 0;
	curr->e_gap_start = 0;
        curr->path.flags = 0;
    }
}
/****************************************************************/
/*  Initializes the dynamic array for                           */
/*  Local optimization with convex gaps.                        */
/****************************************************************/
inline
static void     init_mat11gap_logc(   Max_node        *array,
                                     const Point     *pt)
{
    int         i,j;
    Max_node    *curr, *temp;

    curr = access_mat(array, 0, 0);
    curr->value = 0.0;
    curr->path.flags = 0;
    curr->e_gap_start=0;
    curr->f_gap_start=0;

/* INITIALIZE FIRST COLUMN OF V & F TABLES    */
/* V = array.value,opts F=array.top_val,f_len */
    for  (i = 1; i <= len1; i++)
    {
        curr = access_mat(array, i, 0);
        curr->value = 0;
        curr->f_gap_start = 0;
        curr->path.flags = 0;
    }
/* INITIALIZE FIRST ROW OF V & E TABLES        */
/* V = array.value,opts E=array.side_val,e_len */
    for  (j = 1; j <= len2; j++)
    {
        curr = access_mat(array, 0, j);
        curr->value = 0;
	curr->e_gap_start = 0;
        curr->path.flags = 0;
    }
}
/****************************************************************/
/*  Initializes the dynamic array for *uc*                      */
/*  1 substring of 2 optimization with convex gaps.             */
/****************************************************************/
inline
static void     init_mat12gap_logc(   Max_node        *array,
                                     const Point     *pt)
{
    int         i,j;
    Max_node    *curr, *temp;

    curr = access_mat(array, 0, 0);
    curr->value = 0.0;
    curr->path.flags = 0;
    curr->e_gap_start=0;
    curr->f_gap_start=0;

/* INITIALIZE FIRST COLUMN OF V & F TABLES    */
/* V = array.value,opts F=array.top_val,f_len */
    for  (i = 1; i <= len1; i++)
    {
        curr = access_mat(array, i, 0);
        curr->value = (pt->y*WEIGHT_INORDEL_CONV(i) ) + ( pt->w * WEIGHT_GAP_CONV );
        curr->f_gap_start = 0;
        curr->path.flags = 0;
        curr->path.flag.top_opt = 1;
    }
/* INITIALIZE FIRST ROW OF V & E TABLES        */
/* V = array.value,opts E=array.side_val,e_len */
    for  (j = 1; j <= len2; j++)
    {
        curr = access_mat(array, 0, j);
        curr->value = 0;
	curr->e_gap_start = 0;
        curr->path.flags = 0;
    }
}

/*********************************************************************/
/*  Calculates the dynamic array for                                 */
/*  global/endgapsfree/substring1of2 optimization with convex gaps.  */
/*********************************************************************/
inline
static void     calc_max9gap_logc(    Max_node        *curr,
				      const int       n,
				      const int       m,
				      const Point     *pt)
{
    int		col,row,i,j,k,l,r;

    double      diag_val,
		side_val,
		top_val,
                test_val,
		e_bar,
		f_bar;

    Max_node    *diag,
                *side,
                *top;

    lpoint      p, q, s, last;

      

    /* set global point to last point table filled for */
    /* this is used for doing all tracebacks           */
    convex_w=pt->w;
    convex_y=pt->y;

    
    /* empty f stacks */
    for (j=1; j<=m; j++) {
      fpt[j] = 0;
    }

    /* push first value on stacks */
    for (j=1; j<=m; j++) { 
      lpush(0,n,&fpt[j],flist[j]);
    }

    for (i=1; i <= n; i++) {
      /*for each row i*/

      /* empty the e stack */
        ept=0; 
        lpush(0,m,&ept,elist);
	for (j=1; j <= m; j++ ) {
	  curr=access_mat(array,i,j);
	  /*set p to be first pointer on pointer list*/
	  /*it could be past current node on the list*/
	  p=read_pt(&ept,elist);
	  if (p.p < j) {
                lpop(&ept,elist);
                p=read_pt(&ept,elist);
          }
	  /* E(i,j):=V(p.b)-w(j-p.b); */
	  side_val = candrow(p.b,j,i,j,pt);
	  /* G(i,j):=V(i-1,j-1) + match or mismatch value */
    	  diag = access_mat(array,i-1,j-1);
	  diag_val = diag->value + (str1[i - 1] == str2[j - 1]
               ? pt->v * WEIGHT_MATCH(str1[i-1], str2[j-1])
               : pt->x * WEIGHT_MISMATCH(str1[i-1], str2[j-1]));
	  /* F(i,j):=V(p.d)-w(i-p.d) */
	  q=read_pt(&fpt[j],flist[j]);
	  if (q.p < i) {
                lpop(&fpt[j],flist[j]);
                q=read_pt(&fpt[j],flist[j]);
          }
	  top_val = candcol(q.b,i,i,j,pt); 
	  /* set V(i,j) , as well as E & F b pointers */
	  curr->path.flags=0;
	  set_optimal(curr, diag_val, side_val, top_val);
	  curr->e_gap_start=p.b;
	  curr->f_gap_start=q.b;

	  /*send forward e values */	  
	  if ( j < m ) { 
	    /* j + 1 might be in a different block */
	    if ( p.p == j) {
	      lpop(&ept, elist);
	      p=read_pt(&ept,elist);
	    }
	    if ( candrow(p.b,j+1,i,j,pt) < candrow(j,j+1,i,j,pt) ) {
	    /*  will send forward at least one */
	      l = j+1;
	      while ( (!(list_empty(&ept,elist)))&&(candrow(p.b,p.p,i,j,pt)<candrow(j,p.p,i,j,pt)) ) {
		last=lpop(&ept,elist);
		l=last.p; // set l to the furthest left we know we can beat
		p=read_pt(&ept,elist);
	      }
	      if (list_empty(&ept,elist)) {
		lpush(j,m,&ept,elist);
	      }
	      else {
		s=read_pt(&ept,elist);
		// find first point p where cand(j,p) > cand(s.b,p)
		// lpush(j,p);
		// l is the last position we beat
		r = s.p; 
		// r is the position we didn't beat
		int cur;                        
		while (l != r) {
		  cur = (l + r + 1) / 2;
		if (candrow(j,cur,i,j,pt) > candrow(s.b,cur,i,j,pt))
		  l = cur;
		else
		  r = cur - 1;
		}                       
		// l is point p such that cand(j,l) > cand(s.b,l) 
		// and cand(j,l+1) <= cand(s.b,l+1)
		lpush(j,l,&ept,elist);
	      }
	    }
	  }

	  /*send forward f values */
	  if ( i < n ) {
	    /* i+1 might be in a different block */
	    if ( q.p == i) {
	      lpop(&fpt[j], flist[j]);
	      q=read_pt(&fpt[j],flist[j]);
	    }
	    if ( candcol(q.b,i+1,i,j,pt) < candcol(i,i+1,i,j,pt) ) {
	      /* will send forward at least one */
	      l = i + 1;
	      while ( (!(list_empty(&fpt[j],flist[j])))&&(candcol(q.b,q.p,i,j,pt) < candcol(i,q.p,i,j,pt)) ) {
		last=lpop(&fpt[j],flist[j]);
		l = last.p; // will send forward at least one
		q=read_pt(&fpt[j],flist[j]);
	      }
	      if (list_empty(&fpt[j],flist[j])) {
		lpush(i,n,&fpt[j],flist[j]);
	      }
	      else {
		s=read_pt(&fpt[j],flist[j]);
		// find first point p where cand(j,p) > cand(s.b,p)
		// lpush(j,p);
		// l is the last position we beat
		r = s.p;
		// r is the position we didn't beat
		int cur;                        
		while (l != r) {
		  cur = (l + r + 1) / 2;
		  if ( candcol(i,cur,i,j,pt) > candcol(s.b,cur,i,j,pt) )
		    l = cur;
		  else
		    r = cur - 1;
		}
		lpush(i,l,&fpt[j],flist[j]);
	      }
	    }
	  }
	}
    }
}

/*********************************************************************/
/*  Calculates the dynamic array for *uc*                            */
/*  local optimization with convex gaps. (compares with 0)           */
/*********************************************************************/
inline
static void     calc_max11gap_logc(    Max_node        *curr,
				      const int       n,
				      const int       m,
				      const Point     *pt)
{
    int		col,row,i,j,k,l,r;

    double      diag_val,
		side_val,
		top_val,
                test_val,
		e_bar,
		f_bar;

    Max_node    *diag,
                *side,
                *top;

    lpoint      p, q, s, last;

      

    /* set global point to last point table filled for */
    /* this is used for doing all tracebacks           */
    convex_w=pt->w;
    convex_y=pt->y;

    
    /* empty f stacks */
    for (j=1; j<=m; j++) {
      fpt[j] = 0;
    }

    /* push first value on stacks */
    for (j=1; j<=m; j++) { 
      lpush(0,n,&fpt[j],flist[j]);
    }

    for (i=1; i <= n; i++) {
      /*for each row i*/

      /* empty the e stack */
        ept=0; 
        lpush(0,m,&ept,elist);
	for (j=1; j <= m; j++ ) {
	  curr=access_mat(array,i,j);
	  /*set p to be first pointer on pointer list*/
	  /*it could be past current node on the list*/
	  p=read_pt(&ept,elist);
	  if (p.p < j) {
                lpop(&ept,elist);
                p=read_pt(&ept,elist);
          }
	  /* E(i,j):=V(p.b)-w(j-p.b); */
	  side_val = candrow(p.b,j,i,j,pt);
	  /* G(i,j):=V(i-1,j-1) + match or mismatch value */
    	  diag = access_mat(array,i-1,j-1);
	  diag_val = diag->value + (str1[i - 1] == str2[j - 1]
               ? pt->v * WEIGHT_MATCH(str1[i-1], str2[j-1])
               : pt->x * WEIGHT_MISMATCH(str1[i-1], str2[j-1]));
	  /* F(i,j):=V(p.d)-w(i-p.d) */
	  q=read_pt(&fpt[j],flist[j]);
	  if (q.p < i) {
                lpop(&fpt[j],flist[j]);
                q=read_pt(&fpt[j],flist[j]);
          }
	  top_val = candcol(q.b,i,i,j,pt); 
	  /* set V(i,j) , as well as E & F b pointers */
	  curr->path.flags=0;
	  set_optimal(curr, diag_val, side_val, top_val);
	  curr->e_gap_start=p.b;
	  curr->f_gap_start=q.b;
	  if  (curr->value <= 0.0) {
	      curr->value = 0.0;
	      curr->path.flags = 0;	      
	  }
	  /*send forward e values */	  
	  if ( j < m ) { 
	    /* j + 1 might be in a different block */
	    if ( p.p == j) {
	      lpop(&ept, elist);
	      p=read_pt(&ept,elist);
	    }
	    if ( candrow(p.b,j+1,i,j,pt) < candrow(j,j+1,i,j,pt) ) {
	    /*  will send forward at least one */
	      l = j+1;
	      while ( (!(list_empty(&ept,elist)))&&(candrow(p.b,p.p,i,j,pt)<candrow(j,p.p,i,j,pt)) ) {
		last=lpop(&ept,elist);
		l=last.p; // set l to the furthest left we know we can beat
		p=read_pt(&ept,elist);
	      }
	      if (list_empty(&ept,elist)) {
		lpush(j,m,&ept,elist);
	      }
	      else {
		s=read_pt(&ept,elist);
		// find first point p where cand(j,p) > cand(s.b,p)
		// lpush(j,p);
		// l is the last position we beat
		r = s.p; 
		// r is the position we didn't beat
		int cur;                        
		while (l != r) {
		  cur = (l + r + 1) / 2;
		if (candrow(j,cur,i,j,pt) > candrow(s.b,cur,i,j,pt))
		  l = cur;
		else
		  r = cur - 1;
		}                       
		// l is point p such that cand(j,l) > cand(s.b,l) 
		// and cand(j,l+1) <= cand(s.b,l+1)
		lpush(j,l,&ept,elist);
	      }
	    }
	  }

	  /*send forward f values */
	  if ( i < n ) {
	    /* i+1 might be in a different block */
	    if ( q.p == i) {
	      lpop(&fpt[j], flist[j]);
	      q=read_pt(&fpt[j],flist[j]);
	    }
	    if ( candcol(q.b,i+1,i,j,pt) < candcol(i,i+1,i,j,pt) ) {
	      /* will send forward at least one */
	      l = i + 1;
	      while ( (!(list_empty(&fpt[j],flist[j])))&&(candcol(q.b,q.p,i,j,pt) < candcol(i,q.p,i,j,pt)) ) {
		last=lpop(&fpt[j],flist[j]);
		l = last.p; // will send forward at least one
		q=read_pt(&fpt[j],flist[j]);
	      }
	      if (list_empty(&fpt[j],flist[j])) {
		lpush(i,n,&fpt[j],flist[j]);
	      }
	      else {
		s=read_pt(&fpt[j],flist[j]);
		// find first point p where cand(j,p) > cand(s.b,p)
		// lpush(j,p);
		// l is the last position we beat
		r = s.p;
		// r is the position we didn't beat
		int cur;                        
		while (l != r) {
		  cur = (l + r + 1) / 2;
		  if ( candcol(i,cur,i,j,pt) > candcol(s.b,cur,i,j,pt) )
		    l = cur;
		  else
		    r = cur - 1;
		}
		lpush(i,l,&fpt[j],flist[j]);
	      }
	    }
	  }
	}
    }
}

/****************************************************************/
/*  Calculates the maximum path for a given cell of the         */
/*    dynamic array without gaps.                               */
/****************************************************************/
inline
static void     calc_max(       Max_node        *curr,
                                const int       i,
                                const int       j,
                                const Point     *pt)
{
    Max_node    *diag,
                *side,
                *top;
    double      diag_val,
                side_val,
                top_val;

    /* curr is access_mat(array, i, j) */
    diag = curr - (len2 + 2);   /* (access_mat(array, i-1, j-1)) */
    side = curr - 1;            /* (access_mat(array, i, j-1))   */
    top =  curr - (len2 + 1);   /* (access_mat(array, i-1, j))   */

    diag_val = diag->value
             + (str1[i - 1] == str2[j - 1]
                   ? pt->v * WEIGHT_MATCH(str1[i-1], str2[j-1])
                   : pt->x * WEIGHT_MISMATCH(str1[i-1], str2[j-1]));
    side_val = side->value + (pt->y * WEIGHT_INORDEL);
    top_val = top->value + (pt->y * WEIGHT_INORDEL);

    curr->path.flags = 0;

    set_optimal(curr, diag_val, side_val, top_val);
}


/****************************************************************/
/*  Calculates the maximum path for a given cell of the         */
/*    dynamic array without gaps.                               */
/*  It also compares with 0 for Local optimization.             */
/****************************************************************/
inline
static void     calc_max0(      Max_node        *curr,
                                const int       i,
                                const int       j,
                                const Point     *pt)
{
    Max_node    *diag,
                *side,
                *top;
    double      diag_val,
                side_val,
                top_val;

    /* curr is access_mat(array, i, j) */
    diag = curr - (len2 + 2);   /* (access_mat(array, i-1, j-1)) */
    side = curr - 1;            /* (access_mat(array, i, j-1))   */
    top =  curr - (len2 + 1);   /* (access_mat(array, i-1, j))   */

    diag_val = diag->value
             + (str1[i - 1] == str2[j - 1]
                   ? pt->v * WEIGHT_MATCH(str1[i-1], str2[j-1])
                   : pt->x * WEIGHT_MISMATCH(str1[i-1], str2[j-1]));
    side_val = side->value + (pt->y * WEIGHT_INORDEL);
    top_val = top->value + (pt->y * WEIGHT_INORDEL);

    curr->path.flags = 0;

    set_optimal(curr, diag_val, side_val, top_val);

    if  (curr->value <= 0.0)
    {
        curr->value = 0.0;
        curr->path.flags = 0;
    }
}


/****************************************************************/
/*  Calculates the maximum path for a given cell of the         */
/*    dynamic array with gaps.                                  */
/****************************************************************/
inline
static void     calc_maxgap(    Max_node        *curr,
                                const int       i,
                                const int       j,
                                const Point     *pt)
{
    double      diag_val,
                test_val;
    Max_node    *diag,
                *side,
                *top;

    /* curr is access_mat(array, i, j) */
    diag = curr - (len2 + 2);   /* access_mat(array, i-1, j-1) */
    side = curr - 1;            /* access_mat(array, i, j-1)   */
    top =  curr - (len2 + 1);   /* access_mat(array, i-1, j)   */

    curr->path.flags = 0;

    if  (top->path.flag.top_cont)
        curr->top_value = top->top_value + (pt->y * WEIGHT_INORDEL);
    else
        curr->top_value = top->value
                        + (pt->w * WEIGHT_GAP)
                        + (pt->y * WEIGHT_INORDEL);

    if  (side->path.flag.side_cont)
        curr->side_value = side->side_value + (pt->y * WEIGHT_INORDEL);
    else
        curr->side_value = side->value
                        + (pt->w * WEIGHT_GAP)
                        + (pt->y * WEIGHT_INORDEL);

    diag_val = diag->value
             + (str1[i - 1] == str2[j - 1]
                   ? pt->v * WEIGHT_MATCH(str1[i-1], str2[j-1])
                   : pt->x * WEIGHT_MISMATCH(str1[i-1], str2[j-1]));

    set_optimal(curr, diag_val, curr->side_value, curr->top_value);

    test_val = curr->value + (pt->w * WEIGHT_GAP);

    if  (   (curr->path.flag.top_opt)
         || (curr->top_value >= test_val)
         || (equal(curr->top_value, test_val)))
        curr->path.flag.top_cont = 1;

    if  (   (   (curr->path.flag.diag_opt)
             || (curr->path.flag.side_opt))
         && (   (curr->top_value <= test_val)
             || (equal(curr->top_value, test_val))))
        curr->path.flag.top_stop = 1;

    if  (   (curr->path.flag.side_opt)
         || (curr->side_value >= test_val)
         || (equal(curr->side_value, test_val)))
        curr->path.flag.side_cont = 1;

    if  (   (   (curr->path.flag.diag_opt)
             || (curr->path.flag.top_opt))
         && (   (curr->side_value <= test_val)
             || (equal(curr->side_value, test_val))))
        curr->path.flag.side_stop = 1;

}


/****************************************************************/
/*  Calculates the maximum path for a given cell of the         */
/*    dynamic array with gaps.                                  */
/*  It also compares with 0 for Local optimization.             */
/****************************************************************/
inline
static void     calc_max0gap(   Max_node        *curr,
                                const int       i,
                                const int       j,
                                const Point     *pt)
{
    double      diag_val,
                test_val;
    Max_node    *diag,
                *side,
                *top;

    /* curr is access_mat(array, i, j) */
    diag = curr - (len2 + 2);   /* access_mat(array, i-1, j-1) */
    side = curr - 1;            /* access_mat(array, i, j-1)   */
    top =  curr - (len2 + 1);   /* access_mat(array, i-1, j)   */

    curr->path.flags = 0;

    if  (top->path.flag.top_cont)
        curr->top_value = top->top_value + (pt->y * WEIGHT_INORDEL);
    else
        curr->top_value = top->value
                        + (pt->w * WEIGHT_GAP)
                        + (pt->y * WEIGHT_INORDEL);

    if  (side->path.flag.side_cont)
        curr->side_value = side->side_value + (pt->y * WEIGHT_INORDEL);
    else
        curr->side_value = side->value
                        + (pt->w * WEIGHT_GAP)
                        + (pt->y * WEIGHT_INORDEL);

    diag_val = diag->value
             + (str1[i - 1] == str2[j - 1]
                   ? pt->v * WEIGHT_MATCH(str1[i-1], str2[j-1])
                   : pt->x * WEIGHT_MISMATCH(str1[i-1], str2[j-1]));

    set_optimal(curr, diag_val, curr->side_value, curr->top_value);

    test_val = curr->value + (pt->w * WEIGHT_GAP);

    if  (   (curr->path.flag.top_opt)
         || (curr->top_value >= test_val)
         || (equal(curr->top_value, test_val)))
        curr->path.flag.top_cont = 1;

    if  (   (   (curr->path.flag.diag_opt)
             || (curr->path.flag.side_opt))
         && (   (curr->top_value <= test_val)
             || (equal(curr->top_value, test_val))))
        curr->path.flag.top_stop = 1;

    if  (   (curr->path.flag.side_opt)
         || (curr->side_value >= test_val)
         || (equal(curr->side_value, test_val)))
        curr->path.flag.side_cont = 1;

    if  (   (   (curr->path.flag.diag_opt)
             || (curr->path.flag.top_opt))
         && (   (curr->side_value <= test_val)
             || (equal(curr->side_value, test_val))))
        curr->path.flag.side_stop = 1;

    if  (curr->value <= 0.0)
    {
        curr->value = 0.0;
        curr->side_value = 0.0;
        curr->top_value = 0.0;
        curr->path.flags = 0;
    }
    else
    {
        if  (curr->top_value <= 0.0)
        {
            curr->top_value = 0.0;
            curr->path.flag.top_opt = 0;
            curr->path.flag.top_cont = 0;
            curr->path.flag.top_stop = 0;
        }
        if  (curr->side_value <= 0.0)
        {
            curr->side_value = 0.0;
            curr->path.flag.side_opt = 0;
            curr->path.flag.side_cont = 0;
            curr->path.flag.side_stop = 0;
        }
    }
}


/****************************************************************/
/*  Calculate the function from the path.                       */
/****************************************************************/

void    calc_fun_from_path_c(   Funct   *fun)
{
    char        *align_path_wk;
    int         i,
                j;
    int         side_gap = 0,
                top_gap = 0;
    float       gap_weight;

    fun->clear();

    i = align_start_array_i;
    j = align_start_array_j;

    for (align_path_wk = align_path;
         align_path_wk <= align_path_last;
         align_path_wk++)
    {
        if  (*align_path_wk == UP)
        {
            i--;
            if  (!top_gap)
            {
                side_gap = 0;
                top_gap = 1;
                fun->gaps_ct++;
                fun->gaps_val += WEIGHT_GAP;
            }
            fun->inordel_ct++;
            fun->inordel_val += WEIGHT_INORDEL;
        }
        else if  (*align_path_wk == DIAG)
        {
            i--;
            j--;
            side_gap = 0;
            top_gap = 0;
            if  (str1[i] == str2[j])
            {
                fun->match_ct++;
                fun->match_val += WEIGHT_MATCH(str1[i], str2[j]);
            }
            else
            {
                fun->mis_ct++;
                fun->mis_val += WEIGHT_MISMATCH(str1[i], str2[j]);
            }
        }
        else //  => (*align_path_wk == SIDE)
        {
            j--;
            if  (!side_gap)
            {
                side_gap = 1;
                top_gap = 0;
                fun->gaps_ct++;
                fun->gaps_val += WEIGHT_GAP;
            }
            fun->inordel_ct++;
            fun->inordel_val += WEIGHT_INORDEL;
        }
    }
    if  (!with_gaps)
        fun->gaps_val = 0;
}

/****************************************************************/
/*  Calculate the function from the path.                       */
/****************************************************************/

void    calc_convex_fun_from_path_c(   Funct   *fun)
{
    char        *align_path_wk;
    int         i,
                j;
    int         side_gap = 0,
                top_gap = 0;
    int         side_len = 0,
                top_len = 0;

    fun->clear();

    i = align_start_array_i;
    j = align_start_array_j;

    for (align_path_wk = align_path;
         align_path_wk <= align_path_last;
         align_path_wk++)
    {
        if  (*align_path_wk == UP) {
            i--;
	    if (side_gap) { /* just ended a side gap */
		fun->gaps_ct++;
		fun->gaps_val += WEIGHT_GAP_CONV;
		fun->inordel_ct+=side_len;
		fun->inordel_val+=WEIGHT_INORDEL_CONV(side_len);
		side_len=0;
		side_gap=0;
	    }
            top_gap = 1;
	    top_len++;
        }
        else if  (*align_path_wk == DIAG)
        {
            i--;
            j--;
	    if (side_gap) { /* just ended a side gap */
		fun->gaps_ct++;
		fun->gaps_val += WEIGHT_GAP_CONV;
		fun->inordel_ct+=side_len;
		fun->inordel_val+=WEIGHT_INORDEL_CONV(side_len);
		side_len=0;
		side_gap=0;
	    }
	    if (top_gap) { /* just ended a top gap */
		fun->gaps_ct++;
		fun->gaps_val += WEIGHT_GAP_CONV;
		fun->inordel_ct+=top_len;
		fun->inordel_val+=WEIGHT_INORDEL_CONV(top_len);
		top_len=0;
		top_gap=0;
	    }
            if  (str1[i] == str2[j])
            {
                fun->match_ct++;
                fun->match_val += WEIGHT_MATCH(str1[i], str2[j]);
            }
            else
            {
                fun->mis_ct++;
                fun->mis_val += WEIGHT_MISMATCH(str1[i], str2[j]);
            }
        }
        else /* (*align_path_wk == SIDE) */
        {
            j--;
	    if (top_gap) { /* just ended a top gap */
		fun->gaps_ct++;
		fun->gaps_val += WEIGHT_GAP_CONV;
		fun->inordel_ct+=top_len;
		fun->inordel_val+=WEIGHT_INORDEL_CONV(top_len);
		top_len=0;
		top_gap=0;
	    }
            side_gap = 1;
	    side_len++;
        }
    }
    if (side_gap) { /* just ended a side gap */
	fun->gaps_ct++;
	fun->gaps_val += WEIGHT_GAP_CONV;
	fun->inordel_ct+=side_len;
	fun->inordel_val+=WEIGHT_INORDEL_CONV(side_len);
	side_len=0;
	side_gap=0;
    }
    if (top_gap) { /* just ended a top gap */
	fun->gaps_ct++;
	fun->gaps_val += WEIGHT_GAP_CONV;
	fun->inordel_ct+=top_len;
	fun->inordel_val+=WEIGHT_INORDEL_CONV(top_len);
	top_len=0;
	top_gap=0;
    }
    if  (!with_gaps)
        fun->gaps_val = 0;
}

/*******************************************************************/
/* fast calculation of the dp table for                            */
/* global/engapfree/substring1of2 optimization with convex gaps    */
/* this strictly maximizes avoiding rounding error but fails to    */
/* catch all of the cooptimals                                     */
/*******************************************************************/
inline
static void     calc_max9gap_logc_fast(    Max_node        *curr,
				      const int       n,
				      const int       m,
				      const Point     *pt)
{
    int		col,row,i,j,k,l,r;

    double      diag_val,
		side_val,
		top_val,
                test_val,
		e_bar,
		f_bar;

    Max_node    *diag,
                *side,
                *top;

    lpoint      p, q, s, last;

      

    /* set global point to last point table filled for */
    /* this is used for doing all tracebacks           */
    convex_w=pt->w;
    convex_y=pt->y;

    
    /* empty f stacks */
    for (j=1; j<=m; j++) {
      fpt[j] = 0;
    }

    /* push first value on stacks */
    for (j=1; j<=m; j++) { 
      lpush(0,n,&fpt[j],flist[j]);
    }

    for (i=1; i <= n; i++) {
      /*for each row i*/

      /* empty the e stack */
        ept=0; 
        lpush(0,m,&ept,elist);
	for (j=1; j <= m; j++ ) {
	  curr=access_mat(array,i,j);
	  /*set p to be first pointer on pointer list*/
	  /*it could be past current node on the list*/
	  p=read_pt(&ept,elist);
	  if (p.p < j) {
                lpop(&ept,elist);
                p=read_pt(&ept,elist);
          }
	  /* E(i,j):=V(p.b)-w(j-p.b); */
	  side_val = candrow(p.b,j,i,j,pt);
	  /* G(i,j):=V(i-1,j-1) + match or mismatch value */
    	  diag = access_mat(array,i-1,j-1);
	  diag_val = diag->value + (str1[i - 1] == str2[j - 1]
               ? pt->v * WEIGHT_MATCH(str1[i-1], str2[j-1])
               : pt->x * WEIGHT_MISMATCH(str1[i-1], str2[j-1]));
	  /* F(i,j):=V(p.d)-w(i-p.d) */
	  q=read_pt(&fpt[j],flist[j]);
	  if (q.p < i) {
                lpop(&fpt[j],flist[j]);
                q=read_pt(&fpt[j],flist[j]);
          }
	  top_val = candcol(q.b,i,i,j,pt); 
	  /* set V(i,j) , as well as E & F b pointers */
	  curr->path.flags=0;
	  set_optimal_fast(curr, diag_val, side_val, top_val);
	  curr->e_gap_start=p.b;
	  curr->f_gap_start=q.b;

	  /*send forward e values */	  
	  if ( j < m ) { 
	    /* j + 1 might be in a different block */
	    if ( p.p == j) {
	      lpop(&ept, elist);
	      p=read_pt(&ept,elist);
	    }
	    if ( candrow(p.b,j+1,i,j,pt) < candrow(j,j+1,i,j,pt) ) {
	    /*  will send forward at least one */
	      l = j+1;
	      while ( (!(list_empty(&ept,elist)))&&(candrow(p.b,p.p,i,j,pt)<candrow(j,p.p,i,j,pt)) ) {
		last=lpop(&ept,elist);
		l=last.p; // set l to the furthest left we know we can beat
		p=read_pt(&ept,elist);
	      }
	      if (list_empty(&ept,elist)) {
		lpush(j,m,&ept,elist);
	      }
	      else {
		s=read_pt(&ept,elist);
		// find first point p where cand(j,p) > cand(s.b,p)
		// lpush(j,p);
		// l is the last position we beat
		r = s.p; 
		// r is the position we didn't beat
		int cur;                        
		while (l != r) {
		  cur = (l + r + 1) / 2;
		if (candrow(j,cur,i,j,pt) > candrow(s.b,cur,i,j,pt))
		  l = cur;
		else
		  r = cur - 1;
		}                       
		// l is point p such that cand(j,l) > cand(s.b,l) 
		// and cand(j,l+1) <= cand(s.b,l+1)
		lpush(j,l,&ept,elist);
	      }
	    }
	  }

	  /*send forward f values */
	  if ( i < n ) {
	    /* i+1 might be in a different block */
	    if ( q.p == i) {
	      lpop(&fpt[j], flist[j]);
	      q=read_pt(&fpt[j],flist[j]);
	    }
	    if ( candcol(q.b,i+1,i,j,pt) < candcol(i,i+1,i,j,pt) ) {
	      /* will send forward at least one */
	      l = i + 1;
	      while ( (!(list_empty(&fpt[j],flist[j])))&&(candcol(q.b,q.p,i,j,pt) < candcol(i,q.p,i,j,pt)) ) {
		last=lpop(&fpt[j],flist[j]);
		l = last.p; // will send forward at least one
		q=read_pt(&fpt[j],flist[j]);
	      }
	      if (list_empty(&fpt[j],flist[j])) {
		lpush(i,n,&fpt[j],flist[j]);
	      }
	      else {
		s=read_pt(&fpt[j],flist[j]);
		// find first point p where cand(j,p) > cand(s.b,p)
		// lpush(j,p);
		// l is the last position we beat
		r = s.p;
		// r is the position we didn't beat
		int cur;                        
		while (l != r) {
		  cur = (l + r + 1) / 2;
		  if ( candcol(i,cur,i,j,pt) > candcol(s.b,cur,i,j,pt) )
		    l = cur;
		  else
		    r = cur - 1;
		}
		lpush(i,l,&fpt[j],flist[j]);
	      }
	    }
	  }
	}
    }
}

/*******************************************************************/
/* fast calculation of the dp table for *uc*                       */
/* local optimization with convex gaps (compares with 0)           */
/* this strictly maximizes avoiding rounding error but fails to    */
/* catch all of the cooptimals                                     */
/*******************************************************************/
inline
static void     calc_max11gap_logc_fast(    Max_node        *curr,
				      const int       n,
				      const int       m,
				      const Point     *pt)
{
    int		col,row,i,j,k,l,r;

    double      diag_val,
		side_val,
		top_val,
                test_val,
		e_bar,
		f_bar;

    Max_node    *diag,
                *side,
                *top;

    lpoint      p, q, s, last;

      

    /* set global point to last point table filled for */
    /* this is used for doing all tracebacks           */
    convex_w=pt->w;
    convex_y=pt->y;

    
    /* empty f stacks */
    for (j=1; j<=m; j++) {
      fpt[j] = 0;
    }

    /* push first value on stacks */
    for (j=1; j<=m; j++) { 
      lpush(0,n,&fpt[j],flist[j]);
    }

    for (i=1; i <= n; i++) {
      /*for each row i*/

      /* empty the e stack */
        ept=0; 
        lpush(0,m,&ept,elist);
	for (j=1; j <= m; j++ ) {
	  curr=access_mat(array,i,j);
	  /*set p to be first pointer on pointer list*/
	  /*it could be past current node on the list*/
	  p=read_pt(&ept,elist);
	  if (p.p < j) {
                lpop(&ept,elist);
                p=read_pt(&ept,elist);
          }
	  /* E(i,j):=V(p.b)-w(j-p.b); */
	  side_val = candrow(p.b,j,i,j,pt);
	  /* G(i,j):=V(i-1,j-1) + match or mismatch value */
    	  diag = access_mat(array,i-1,j-1);
	  diag_val = diag->value + (str1[i - 1] == str2[j - 1]
               ? pt->v * WEIGHT_MATCH(str1[i-1], str2[j-1])
               : pt->x * WEIGHT_MISMATCH(str1[i-1], str2[j-1]));
	  /* F(i,j):=V(p.d)-w(i-p.d) */
	  q=read_pt(&fpt[j],flist[j]);
	  if (q.p < i) {
                lpop(&fpt[j],flist[j]);
                q=read_pt(&fpt[j],flist[j]);
          }
	  top_val = candcol(q.b,i,i,j,pt); 
	  /* set V(i,j) , as well as E & F b pointers */
	  curr->path.flags=0;
	  set_optimal_fast(curr, diag_val, side_val, top_val);
	  if  (curr->value <= 0.0) {
	      curr->value = 0.0;
	      curr->path.flags = 0;	      
	  }
	  curr->e_gap_start=p.b;
	  curr->f_gap_start=q.b;

	  /*send forward e values */	  
	  if ( j < m ) { 
	    /* j + 1 might be in a different block */
	    if ( p.p == j) {
	      lpop(&ept, elist);
	      p=read_pt(&ept,elist);
	    }
	    if ( candrow(p.b,j+1,i,j,pt) < candrow(j,j+1,i,j,pt) ) {
	    /*  will send forward at least one */
	      l = j+1;
	      while ( (!(list_empty(&ept,elist)))&&(candrow(p.b,p.p,i,j,pt)<candrow(j,p.p,i,j,pt)) ) {
		last=lpop(&ept,elist);
		l=last.p; // set l to the furthest left we know we can beat
		p=read_pt(&ept,elist);
	      }
	      if (list_empty(&ept,elist)) {
		lpush(j,m,&ept,elist);
	      }
	      else {
		s=read_pt(&ept,elist);
		// find first point p where cand(j,p) > cand(s.b,p)
		// lpush(j,p);
		// l is the last position we beat
		r = s.p; 
		// r is the position we didn't beat
		int cur;                        
		while (l != r) {
		  cur = (l + r + 1) / 2;
		if (candrow(j,cur,i,j,pt) > candrow(s.b,cur,i,j,pt))
		  l = cur;
		else
		  r = cur - 1;
		}                       
		// l is point p such that cand(j,l) > cand(s.b,l) 
		// and cand(j,l+1) <= cand(s.b,l+1)
		lpush(j,l,&ept,elist);
	      }
	    }
	  }

	  /*send forward f values */
	  if ( i < n ) {
	    /* i+1 might be in a different block */
	    if ( q.p == i) {
	      lpop(&fpt[j], flist[j]);
	      q=read_pt(&fpt[j],flist[j]);
	    }
	    if ( candcol(q.b,i+1,i,j,pt) < candcol(i,i+1,i,j,pt) ) {
	      /* will send forward at least one */
	      l = i + 1;
	      while ( (!(list_empty(&fpt[j],flist[j])))&&(candcol(q.b,q.p,i,j,pt) < candcol(i,q.p,i,j,pt)) ) {
		last=lpop(&fpt[j],flist[j]);
		l = last.p; // will send forward at least one
		q=read_pt(&fpt[j],flist[j]);
	      }
	      if (list_empty(&fpt[j],flist[j])) {
		lpush(i,n,&fpt[j],flist[j]);
	      }
	      else {
		s=read_pt(&fpt[j],flist[j]);
		// find first point p where cand(j,p) > cand(s.b,p)
		// lpush(j,p);
		// l is the last position we beat
		r = s.p;
		// r is the position we didn't beat
		int cur;                        
		while (l != r) {
		  cur = (l + r + 1) / 2;
		  if ( candcol(i,cur,i,j,pt) > candcol(s.b,cur,i,j,pt) )
		    l = cur;
		  else
		    r = cur - 1;
		}
		lpush(i,l,&fpt[j],flist[j]);
	      }
	    }
	  }
	}
    }
}

/****************************************************************/
/*  Calculates the maximum path for a given cell of the         */
/*  dynamic array without gaps. Quickly as described above      */
/****************************************************************/
inline
static void     calc_max_fast(       Max_node        *curr,
                                const int       i,
                                const int       j,
                                const Point     *pt)
{
    Max_node    *diag,
                *side,
                *top;
    double      diag_val,
                side_val,
                top_val;

    /* curr is access_mat(array, i, j) */
    diag = curr - (len2 + 2);   /* (access_mat(array, i-1, j-1)) */
    side = curr - 1;            /* (access_mat(array, i, j-1))   */
    top =  curr - (len2 + 1);   /* (access_mat(array, i-1, j))   */

    diag_val = diag->value
             + (str1[i - 1] == str2[j - 1]
                   ? pt->v * WEIGHT_MATCH(str1[i-1], str2[j-1])
                   : pt->x * WEIGHT_MISMATCH(str1[i-1], str2[j-1]));
    side_val = side->value + (pt->y * WEIGHT_INORDEL);
    top_val = top->value + (pt->y * WEIGHT_INORDEL);

    curr->path.flags = 0;

    set_optimal_fast(curr, diag_val, side_val, top_val);
}


/****************************************************************/
/*  Calculates the maximum path for a given cell of the         */
/*  dynamic array without gaps. quickly as described above      */
/*  It also compares with 0 for Local optimization.             */
/****************************************************************/
inline
static void     calc_max0_fast(      Max_node        *curr,
                                const int       i,
                                const int       j,
                                const Point     *pt)
{
    Max_node    *diag,
                *side,
                *top;
    double      diag_val,
                side_val,
                top_val;

    /* curr is access_mat(array, i, j) */
    diag = curr - (len2 + 2);   /* (access_mat(array, i-1, j-1)) */
    side = curr - 1;            /* (access_mat(array, i, j-1))   */
    top =  curr - (len2 + 1);   /* (access_mat(array, i-1, j))   */

    diag_val = diag->value
             + (str1[i - 1] == str2[j - 1]
                   ? pt->v * WEIGHT_MATCH(str1[i-1], str2[j-1])
                   : pt->x * WEIGHT_MISMATCH(str1[i-1], str2[j-1]));
    side_val = side->value + (pt->y * WEIGHT_INORDEL);
    top_val = top->value + (pt->y * WEIGHT_INORDEL);

    curr->path.flags = 0;

    set_optimal_fast(curr, diag_val, side_val, top_val);

    if  (curr->value <= 0.0)
    {
        curr->value = 0.0;
        curr->path.flags = 0;
    }
}


/****************************************************************/
/*  Calculates the maximum path for a given cell of the         */
/*  dynamic array with gaps. quickly as described above         */
/****************************************************************/
inline
static void     calc_maxgap_fast(    Max_node        *curr,
                                const int       i,
                                const int       j,
                                const Point     *pt)
{
    double      diag_val,
                test_val;
    Max_node    *diag,
                *side,
                *top;

    /* curr is access_mat(array, i, j) */
    diag = curr - (len2 + 2);   /* access_mat(array, i-1, j-1) */
    side = curr - 1;            /* access_mat(array, i, j-1)   */
    top =  curr - (len2 + 1);   /* access_mat(array, i-1, j)   */

    curr->path.flags = 0;

    if  (top->path.flag.top_cont)
        curr->top_value = top->top_value + (pt->y * WEIGHT_INORDEL);
    else
        curr->top_value = top->value
                        + (pt->w * WEIGHT_GAP)
                        + (pt->y * WEIGHT_INORDEL);

    if  (side->path.flag.side_cont)
        curr->side_value = side->side_value + (pt->y * WEIGHT_INORDEL);
    else
        curr->side_value = side->value
                        + (pt->w * WEIGHT_GAP)
                        + (pt->y * WEIGHT_INORDEL);

    diag_val = diag->value
             + (str1[i - 1] == str2[j - 1]
                   ? pt->v * WEIGHT_MATCH(str1[i-1], str2[j-1])
                   : pt->x * WEIGHT_MISMATCH(str1[i-1], str2[j-1]));

    set_optimal_fast(curr, diag_val, curr->side_value, curr->top_value);

    test_val = curr->value + (pt->w * WEIGHT_GAP);

    if  (   (curr->path.flag.top_opt)
         || (curr->top_value >= test_val))
      curr->path.flag.top_cont = 1;

    if  (   (   (curr->path.flag.diag_opt)
             || (curr->path.flag.side_opt))
         && ( curr->top_value <= test_val ))
      curr->path.flag.top_stop = 1;

    if  ( (curr->path.flag.side_opt)
         ||(curr->side_value >= test_val))
      curr->path.flag.side_cont = 1;

    if  (((curr->path.flag.diag_opt)
	  || (curr->path.flag.top_opt))
          && (curr->side_value <= test_val))
      curr->path.flag.side_stop = 1;


}


/********************************************************/
/*  Calculates the maximum path for a given cell of the */
/*  dynamic array with gaps. quickly as described above */
/*  It also compares with 0 for Local optimization.     */
/********************************************************/
inline
static void     calc_max0gap_fast(   Max_node        *curr,
                                const int       i,
                                const int       j,
                                const Point     *pt)
{
    double      diag_val,
                test_val;
    Max_node    *diag,
                *side,
                *top;

    /* curr is access_mat(array, i, j) */
    diag = curr - (len2 + 2);   /* access_mat(array, i-1, j-1) */
    side = curr - 1;            /* access_mat(array, i, j-1)   */
    top =  curr - (len2 + 1);   /* access_mat(array, i-1, j)   */

    curr->path.flags = 0;

    if  (top->path.flag.top_cont)
        curr->top_value = top->top_value + (pt->y * WEIGHT_INORDEL);
    else
        curr->top_value = top->value
                        + (pt->w * WEIGHT_GAP)
                        + (pt->y * WEIGHT_INORDEL);

    if  (side->path.flag.side_cont)
        curr->side_value = side->side_value + (pt->y * WEIGHT_INORDEL);
    else
        curr->side_value = side->value
                        + (pt->w * WEIGHT_GAP)
                        + (pt->y * WEIGHT_INORDEL);

    diag_val = diag->value
             + (str1[i - 1] == str2[j - 1]
                   ? pt->v * WEIGHT_MATCH(str1[i-1], str2[j-1])
                   : pt->x * WEIGHT_MISMATCH(str1[i-1], str2[j-1]));

    set_optimal_fast(curr, diag_val, curr->side_value, curr->top_value);

    test_val = curr->value + (pt->w * WEIGHT_GAP);

    if  (   (curr->path.flag.top_opt)
         || (curr->top_value >= test_val)
         || (equal(curr->top_value, test_val)))
        curr->path.flag.top_cont = 1;

    if  (   (   (curr->path.flag.diag_opt)
             || (curr->path.flag.side_opt))
         && (   (curr->top_value <= test_val)
             || (equal(curr->top_value, test_val))))
        curr->path.flag.top_stop = 1;

    if  (   (curr->path.flag.side_opt)
         || (curr->side_value >= test_val)
         || (equal(curr->side_value, test_val)))
        curr->path.flag.side_cont = 1;

    if  (   (   (curr->path.flag.diag_opt)
             || (curr->path.flag.top_opt))
         && (   (curr->side_value <= test_val)
             || (equal(curr->side_value, test_val))))
        curr->path.flag.side_stop = 1;

    if  (curr->value <= 0.0)
    {
        curr->value = 0.0;
        curr->side_value = 0.0;
        curr->top_value = 0.0;
        curr->path.flags = 0;
    }
    else
    {
        if  (curr->top_value <= 0.0)
        {
            curr->top_value = 0.0;
            curr->path.flag.top_opt = 0;
            curr->path.flag.top_cont = 0;
            curr->path.flag.top_stop = 0;
        }
        if  (curr->side_value <= 0.0)
        {
            curr->side_value = 0.0;
            curr->path.flag.side_opt = 0;
            curr->path.flag.side_cont = 0;
            curr->path.flag.side_stop = 0;
        }
    }
}


/****************************************************************/
/* This is the dynamic programming routine used to find optimal */
/* functions and alignments at a given point.                   */
/* It calls the respective procedures for the given setting.    */
/****************************************************************/

Funct   *find_max_c(    Point           *pt,
                        const int       do_align)
{
    int         i,
                j;
    Funct       *fun;
    int         rel_scoring_method;
    Max_node    *curr;
  
    rel_scoring_method = scoring_method;
    if (with_gaps) {
	rel_scoring_method += 4;
	if (convex) {
		rel_scoring_method += 4;
	}
    }
/*
**  Above maps scoring_method with gaps to 5 - 8.
**  and convex gaps to 9 - 12.
*/
    pt->trans_in(opt_function);
    switch  (rel_scoring_method)
    {
        case GLOBAL:
            init_mat1(array, pt);
            break;
        case ENDGAPS_FREE:
            init_mat2(array, pt);
            break;
        case LOCAL:
            init_mat3(array, pt);   /* Just the same as init-1 */
            break;
        case SUBSTRING_1_OF_2:
            init_mat4(array, pt);
            break;
        case GLOBAL_GAPS:
            init_mat5gap(array, pt);
            break;
        case ENDGAPS_FREE_GAPS:
            init_mat6gap(array, pt);
            break;
        case LOCAL_GAPS:
            init_mat7gap(array, pt);  /* Just the same as init-5 */
            break;
        case SUBSTRING_1_OF_2_GAPS:
            init_mat8gap(array, pt);
            break;
        case GLOBAL_CONVEX:
            init_mat9gap_logc(array, pt);
            break;
        case ENDGAPS_FREE_CONVEX:
            init_mat10gap_logc(array, pt);
            break;
        case LOCAL_CONVEX:
            init_mat11gap_logc(array, pt);
            break;
        case SUBSTRING_1_OF_2_CONVEX:
            init_mat12gap_logc(array, pt);
            break;
    }

    curr = array + len2;
    switch  (rel_scoring_method)
    {
        case GLOBAL:
        case ENDGAPS_FREE:
        case SUBSTRING_1_OF_2:
            for  (i = 1; i <= len1; i++)
            {
                curr++;
                for  (j = 1; j <= len2; j++)
                {
                    curr++;
                    calc_max(curr, i, j, pt);
                }
            }
            break;
        case LOCAL:
            for  (i = 1; i <= len1; i++)
            {
                curr++;
                for  (j = 1; j <= len2; j++)
                {
                    curr++;
                    calc_max0(curr, i, j, pt);
                }
            }
            break;
        case GLOBAL_GAPS:
        case ENDGAPS_FREE_GAPS:
        case SUBSTRING_1_OF_2_GAPS:
            for  (i = 1; i <= len1; i++)
            {
                curr++;
                for  (j = 1; j <= len2; j++)
                {
                    curr++;
                    calc_maxgap(curr, i, j, pt);
                }
            }
            break;
        case LOCAL_GAPS:
            for  (i = 1; i <= len1; i++)
            {
                curr++;
                for  (j = 1; j <= len2; j++)
                {
                    curr++;
                    calc_max0gap(curr, i, j, pt);
                }
            }
            break;
        case GLOBAL_CONVEX:
        case ENDGAPS_FREE_CONVEX:
        case SUBSTRING_1_OF_2_CONVEX:
            calc_max9gap_logc(curr, len1, len2, pt);
            break;
        case LOCAL_CONVEX:
	    calc_max11gap_logc(curr, len1, len2, pt);
	    break;
    }
    curr->path.flag.side_cont = 0;
    curr->path.flag.side_stop = 0;
    curr->path.flag.top_cont = 0;
    curr->path.flag.top_stop = 0;

    fun = new Funct;

    if  (do_align)
    {
        fun->opt_align_ct = get_opt_align_ct(array);
    }

    get_first_align_path(array);

    if ( convex && with_gaps ) {
    	calc_convex_fun_from_path_c(fun);
    }
    else {
    	calc_fun_from_path_c(fun);
    }
	      
    if  (do_align)
    {
        calc_align_from_path(array, fun);
    }
    pt->trans_out(opt_function);

    return fun;
}

Funct   *find_max_c_fast(    Point           *pt,
			     const int       do_align)
{
    int         i,
                j;
    Funct       *fun;
    int         rel_scoring_method;
    Max_node    *curr;
  
    rel_scoring_method = scoring_method;
    if (with_gaps) {
	rel_scoring_method += 4;
	if (convex) {
		rel_scoring_method += 4;
	}
    }
/*
**  This maps scoring_method with gaps to 5 - 8.
**  and scoring_method for convex to 9-12.
*/
    pt->trans_in(opt_function);
    switch  (rel_scoring_method)
    {
        case GLOBAL:
            init_mat1(array, pt);
            break;
        case ENDGAPS_FREE:
            init_mat2(array, pt);
            break;
        case LOCAL:
            init_mat3(array, pt);   /* Just the same as init-1 */
            break;
        case SUBSTRING_1_OF_2:
            init_mat4(array, pt);
            break;
        case GLOBAL_GAPS:
            init_mat5gap(array, pt);
            break;
        case ENDGAPS_FREE_GAPS:
            init_mat6gap(array, pt);
            break;
        case LOCAL_GAPS:
            init_mat7gap(array, pt);  /* Just the same as init-5 */
            break;
        case SUBSTRING_1_OF_2_GAPS:
            init_mat8gap(array, pt);
            break;
        case GLOBAL_CONVEX:
            init_mat9gap_logc(array, pt);
            break;
        case ENDGAPS_FREE_CONVEX:
            init_mat10gap_logc(array, pt);
            break;
        case LOCAL_CONVEX:
            init_mat11gap_logc(array, pt);  /* Just the same as init-5 */
            break;
        case SUBSTRING_1_OF_2_CONVEX:
            init_mat12gap_logc(array, pt);
            break;
    }

    curr = array + len2;
    switch  (rel_scoring_method)
    {
        case GLOBAL:
        case ENDGAPS_FREE:
        case SUBSTRING_1_OF_2:
            for  (i = 1; i <= len1; i++)
            {
                curr++;
                for  (j = 1; j <= len2; j++)
                {
                    curr++;
                    calc_max_fast(curr, i, j, pt);
                }
            }
            break;
        case LOCAL:
            for  (i = 1; i <= len1; i++)
            {
                curr++;
                for  (j = 1; j <= len2; j++)
                {
                    curr++;
                    calc_max0_fast(curr, i, j, pt);
                }
            }
            break;
        case GLOBAL_GAPS:
        case ENDGAPS_FREE_GAPS:
        case SUBSTRING_1_OF_2_GAPS:
            for  (i = 1; i <= len1; i++)
            {
                curr++;
                for  (j = 1; j <= len2; j++)
                {
                    curr++;
                    calc_maxgap_fast(curr, i, j, pt);
                }
            }
            break;
        case LOCAL_GAPS:
            for  (i = 1; i <= len1; i++)
            {
                curr++;
                for  (j = 1; j <= len2; j++)
                {
                    curr++;
                    calc_max0gap_fast(curr, i, j, pt);
                }
            }
	    break;
	case GLOBAL_CONVEX:
        case ENDGAPS_FREE_CONVEX:
        case SUBSTRING_1_OF_2_CONVEX:
	    calc_max9gap_logc_fast(curr,len1,len2,pt);
            break;
        case LOCAL_CONVEX:
	    calc_max11gap_logc_fast(curr,len1,len2,pt);
            break; 
    }
    curr->path.flag.side_cont = 0;
    curr->path.flag.side_stop = 0;
    curr->path.flag.top_cont = 0;
    curr->path.flag.top_stop = 0;

    fun = new Funct;
    if  (do_align)
    {
        fun->opt_align_ct = 1;
    }

    get_first_align_path(array);
  
    if (convex && with_gaps )  {
    	calc_convex_fun_from_path_c(fun);
    }
    else  {
    	calc_fun_from_path_c(fun);
    }


    if  (do_align)
    {
        calc_align_from_path(array, fun);
    }
    pt->trans_out(opt_function);

    return fun;
}
