/* parallel -	execute multiple tasks in parallel
 *
 *	Parallel reads commands from standard input, one per line, and
 *	executes them using `/bin/sh -c'.  Output from the child processes
 *	may be combined in an unspecified order, so it is recommended that
 *	the output be redirected to a file.  Parallel takes one command
 *	line argument, a positive integer specifying the maximum number of
 *	tasks to run at a time.
 *
 *	Parallel exits after stdin is closed and all of its children
 *	have exited.  It returns 0 if all of its children exited with
 *	exit status 0; otherwise, 1 is returned.
 */

#include <sys/time.h>
#include <sys/resource.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <errno.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#define CMDBUFSIZE	ARG_MAX

#if !defined(__sun__)
#include <paths.h>
#else
#define _PATH_BSHELL		"/bin/sh"
#endif

/* This hack suppresses obnoxious compiler warnings. */
#if __GNUC__ > 2 || __GNUC__ == 2 && __GNUC_MINOR__ >= 5
#define __dead		__attribute__((__noreturn__))
#else
#define __dead
#endif

extern char **environ;

static void usage(void) __dead;
int spawn(char *command);

int
main (int argc, char *argv[])
{
    int jobs;
    int status = 0, childstatus = 1;
    char buf[CMDBUFSIZE];

    if (argc != 2 || (jobs = atoi(argv[1])) <= 0)
	usage();

    while (!feof(stdin) && !ferror(stdin)) {
	if (fgets(buf, CMDBUFSIZE, stdin) == NULL)
	    continue;
	spawn(buf);
	if (--jobs <= 0) {
	    waitpid(-1, &childstatus, 0);
	    if (childstatus != 0)
		status = 1;
	    jobs++;
	}
    }
    while (waitpid(-1, &childstatus, 0) > 0) {
	if (childstatus != 0)
	    status = 1;
    }

    exit(status);
}

/*
 * spawn -	spawn a child process and return immediately
 *
 *	The given string is executed via `sh -c'.  This routine
 *	returns the child's process ID, or -1 on failure.
 */
int
spawn(char *command)
{
    int pid;
    char *argv[] = {"sh", "-c", command, NULL};

    switch (pid = vfork()) {
    case -1:
	return -1;
    case 0:
	execve(_PATH_BSHELL, argv, environ);
	_exit(127);	/* bad shell */
    }
    return pid;
}

static void
usage(void)
{
    fprintf(stderr, "usage: parallel maxjobs\n");
    exit(1);
}
