#!/usr/bin/perl -w

use strict;
use Cwd;
use File::Basename;
use Getopt::Std;


$::indent = 15;
$::jump_offset = 15;
$::framesetname="index.html";
$::srcframe="source";
$::traceframe="trace";


%::BGColors = (normal0 => "#ffffff",
	       normal1 => "#e7e7e7",
	       call => "#00ffff",
	       state => "#6666ff");

# Note: if you change these, be sure to update trace.js as well
%::Img = (pad => "../pad.png",
	  exp => "../expand.png",
	  col => "../collapse.png",
	  nop => "../noop.png");

%::Class = (trans => "trans",
	    fcn => "fcn");

$::FramesetHTML = 
'<html>
<head>
<title>MOPS Output: @@TRACE@@</title>
</head>

<frameset cols="30%,*">
<frame name="@@TRACEFRAME@@" src="@@TRACE@@.html">
<frame name="@@SRCFRAME@@" src="@@FILE@@#@@LINE@@">
</frameset>

<noframes>
<p>The full MOPS output system uses frames. If your browser does not
allow frames, you may view the <a href="@@TRACE@@.html">trace</a> without
frames.
</noframes>
</html>
';

$::IndexHTML = 
'<html>
<head>
<title>MOPS Results</title>
</head>

<body bgcolor="#ffffff">
<center><h2>MOPS Results</h2></center><br>
<center><h3>Program: @@FILE@@<br>MFSA: @@MFSA@@</h3></center>
<center><h4>@@LABEL@@</h4></center>

<ul>
@@LIST@@
</ul>
<a href="../profile">View profiling data</a><br>
<a href="../../index.html">Return to program index</a>
<p>
<hr>
This file was generated on @@DATE@@.
</body>
</html>
';

$::SourcePreamble =
'<HTML>
<HEAD>
<TITLE>MOPS Trace</TITLE>
</HEAD>

<BODY BGCOLOR="#ffffff">
<TABLE border=0 cellspacing=0 cellpadding=0 width="100%">
';

$::SourcePostamble =
'</TABLE>
</BODY>
</HTML>
';

$::TracePreamble =
'<html>
<head>
<title>Test Stuff</title>
<link rel="stylesheet" href="../trace.css" type="text/css">

<script language="javascript" src="../trace.js">
</script>

<script language="javascript">
<!--
';

$::TraceMidamble = 
'
//-->
</SCRIPT>

</head>

<body bgcolor="#ffffff" onload="expandAll()">

<h2>MOPS Trace</h2>
<h4>Program: @@FILE@@<br>MFSA: @@MFSA@@</h4>

<nobr>
';

$::TracePostamble =
"</nobr>
<p>
<a href='javascript:expandAll()'>Expand All</a><br>
<a href='javascript:collapseAll()'>Collapse All</a>
<p>
<img width=8 height=8 src='$::Img{exp}'> Expand Subtrace<br>
<img width=8 height=8 src='$::Img{col}'> Collapse Subtrace<br>
<img width=8 height=8 src='$::Img{nop}'> Function call<br>

<p>
<a href='../index.html' target='_top'>Back to trace index</a>

</body>
</html>
";



######################################################################
#                          Output Generation                         #
######################################################################


# Parse a single trace, creating the HTMLified version of the trace
# and the associated source code. Expects to be called from the top
# of the output file hierarchy; will return there before returning.
sub parse_trace {
    my ($tracefile) = @_;
    my ($tracename);

    # Extract the relevant bits of information from the trace and
    # generate the strings to be printed to the trace file.
    &debug("Parsing subtrace $tracefile\n");
    if (! open TRACE, "< $tracefile") {
	warn("Unable to open trace $tracefile: $!\n");
	return;
    }
    <TRACE>; # Skip the compilation-mode header

    my ($errors) = (0);
    my (@nodes, @stack, %stuff);
    my ($laststate, $lastfile, $lastline, $lastdepth, $lastnode) = 
	("", "", 0, 0, 0);
    my ($entrypoint, $input, $reason, %states, %changes);

    $entrypoint = "main";
    while ($input = <TRACE>) {
	my ($file, $line, $state, $depth, $astnode);

	chomp $input;
	if ($input =~ /^error: (.+)/) {
	    &debug("Error reason restated on line $.\n") if $reason;
	    $reason = $1;
	    next;
	}
	if ($input =~ /^entry: (.+)/) {
	    $entrypoint = $1;
	    next;
	}

	if ($input !~ /^(.*):(\d+): "(.*)"  (\d+) ?(\d*)$/) {
	    warn("Line $. in $tracefile malformed, skipping\n") if $::verbose;
	    $errors++;
	    next;
	}
	$file = $1;
	$line = $2;
	$state = $3;
	$depth = $4;
	$astnode = $5;

	# Keep track of the longest common prefix for source files
	&update_prefix($file);

	# Handle the first line (the entry function) specially
	if (!$lastdepth) {
	    $stack[1] = $.;

	    my ($label);
	    $label = "fcn" . $.;
	    push @nodes, $label;

	    my ($decl, $i);
	    $decl = "var $label = new mkFcn(\"$label\", 1)\n";
	    $stuff{$label}{decl} = $decl;
	    
	    my ($srclink, $html_function, $w, $link);
	    $srclink = link_to_source($file, $line);
	    $html_function = escape_html($entrypoint . "()");
	    $link = fixup_heredoc(<<"		EOF");
	        @@<div class="$::Class{fcn}" id="$label">
		@@  <img width=8 height=8 border=0 src="$::Img{nop}">
		@@  <a $srclink>
		@@    $html_function
		@@  </a>
		@@</div>
		EOF
		;
	    $stuff{$label}{link} = $link;

	    $stuff{$label}{show} = 0;
	}

	# Function calls
	if ($lastdepth && $depth > $lastdepth) {
	    $changes{$lastfile}{$lastline} = "call";
	    $stack[$depth] = $.;

	    my ($label);
	    $label = "fcn" . $.;
	    push @nodes, $label;

	    my ($decl, $i);
	    $decl = "var $label = new mkFcn(\"$label\", $depth)\n";
	    $stuff{$label}{decl} = $decl;
	    
	    my ($srclink, $html_function, $w, $link);
	    $srclink = link_to_source($lastfile, $lastline);
	    $html_function = escape_html(resolve_astnode($lastnode));
	    $w = $::indent * $lastdepth;
	    $link = fixup_heredoc(<<"		EOF");
	        @@<div class="$::Class{fcn}" id="$label">
		@@  <img id="pad_$label" width=$w height=0 border=0 src="$::Img{pad}">
		@@  <img width=8 height=8 border=0 src="$::Img{nop}">
		@@  <a $srclink>
		@@    $html_function
		@@  </a>
		@@</div>
		EOF
		;
	    $stuff{$label}{link} = $link;

	    $stuff{$label}{show} = 0;
	}

	# State changes
	if ($laststate && $state ne $laststate) {
	    $changes{$lastfile}{$lastline} = "state";

	    my ($label);
	    $label = "trans" . $.;
	    push @nodes, $label;

	    my ($decl, $i);
	    $decl = "var $label = new mkTrans(\"$label\", $depth)\n";
	    for $i (1 .. $depth) {
		my $func = $stack[$i];
		my $funclabel = "fcn" . $func;
		$stuff{$funclabel}{decl} .= 
		    "$funclabel.trans[$funclabel.trans.length] = \"$label\";\n";
		$stuff{$funclabel}{show} = 1;
		$decl .= "$label.fcns[$label.fcns.length] = \"fcn$func\";\n";
	    }
	    $stuff{$label}{decl} = $decl;
	    
	    # We fake the initial expansion of the call stack here. If
	    # the user has Javascript enabled, the onload should take
	    # care of the underlying objects; if not, better to show the
	    # expanded trace rather than the collapsed one.
	    my ($srclink, $html_transition, $w, $link);
	    $srclink = link_to_source($lastfile, $lastline);
	    $html_transition = escape_html("$laststate -> $state");
	    $w = $::indent * $depth;
	    $link = fixup_heredoc(<<"		EOF");
		@@<div class="$::Class{trans}" id="$label">
		@@  <img id="pad_$label" width=$w height=0 border=0 src="$::Img{pad}">
		@@  <a href="javascript:toggle('$label')">
		@@    <img id="tog_$label" width=8 height=8 border=0 src="$::Img{col}"></a>
		@@  <a $srclink>
		@@    $html_transition
		@@  </a>
		@@</div>
		EOF

	    $stuff{$label}{link} = $link;

	    $stuff{$label}{show} = 1;
	}


	# Record all possible states for a given source line
	if (! grep /\Q$state\E/, @{$states{$file}{$line}}) {
	    push @{$states{$file}{$line}}, $state;
	}

	$lastfile = $file;
	$lastline = $line;
	$lastdepth = $depth;
	$laststate = $state;
	$lastnode = $astnode;
    }
    close TRACE;


    # $::errors{reason}{place} is an array of trace files used by the
    # index generator to group traces by final state and then by last
    # line. Populate it here.
    if (! $reason) {
	&debug("Error reason not stated in trace file");
	$reason = "Unspecified Error";
    }
    $lastfile =~ s/^$::prefix//;
    push @{$::errors{$reason}{$lastfile . ":" . $lastline}}, $tracefile;

  
    ##
    ## Per-trace output generation
    ##

    # Make the second-level trace directory, and descend into it.
    $tracename = basename($tracefile);
    if (! (-d $tracename || mkdir $tracename, 0777)) {
	warn("Unable to create output subdirectory $tracename: $!\n");
	close TRACE;
	return;
    }
    if (! chdir $tracename) {
	warn("Unable to enter output subdirectory $tracename: $!\n");
	close TRACE;
	return;
    }

    # Create the trace output file
    &debug("  Creating output in " . getcwd() . "\n");
    if (! open BRIEF, "> " . $tracename . ".html") {
	warn("Unable to create trace output ${tracename}.html: $!\n");
	chdir $::outputdir or 
	    die("Unable to return to output directory: $!\n");
	return;
    }

    print BRIEF $::TracePreamble;

    my ($chunk);
    for $chunk (@nodes) {
	next unless $stuff{$chunk}{show};
	print BRIEF $stuff{$chunk}{decl}, "\n";
    }

    my $midamble;
    $midamble = $::TraceMidamble;
    $midamble =~ s/\@\@FILE\@\@/$::filename/;
    $midamble =~ s/\@\@MFSA\@\@/$::mfsaname/;
    print BRIEF $midamble;

    # Remove the source prefix from all file links
    for $chunk (@nodes) {
	next unless $stuff{$chunk}{show};
	my $output = $stuff{$chunk}{link};
	$output =~ s/\@\@$::prefix//g;
	print BRIEF $output;
    }

    print BRIEF $::TracePostamble;

    close BRIEF;


    &parse_sources(\%changes, \%states);
    &generate_frameset($tracename, $errors, $lastfile, $lastline);


    # Put the current working directory back where we found it
    if (! chdir $::outputdir) {
	die("Unable to return to output directory: $!");
    }
    &debug("Subtrace $tracefile completed\n");

}


# Generate HTMLified versions of the sources for each trace
sub parse_sources {
    my ($changes_p, $states_p) = @_;
    my ($file);
    my %changes = %$changes_p;
    my %states = %$states_p;

    foreach $file (keys %changes) {
	my ($input, $output, $outputdir);

	$input = $file;
	($output = $file . ".html") =~ s/^$::prefix//;

	$outputdir = dirname($output);
	if (! (-d $outputdir || &mkdir_recursive($outputdir, 0777))) {
	    warn("Unable to create output subdirectory $outputdir: $!\n");
	    next;
	}
	&debug("  Creating $output from $input\n");

	if (! open INPUT, $input) {
	    warn("Unable to read from $input: $!\n");
	    next;
	}
	if (! open OUTPUT, ">" . $output) {
	    warn("Unable to write to $output: $!\n");
	    next;
	}

	print OUTPUT $::SourcePreamble;

	my ($line);
	while ($line = <INPUT>) {
	    my ($anchor, $number, $code, $states);
	    my ($bgcolor);
	    my ($output);
	    chomp $line;
	    
	    
	    $anchor = anchor_nofile($file, $.);
	    $code = escape_html($line);

	    if (defined($changes{$file}{$.})) {
		$number = $.;
		$bgcolor = $::BGColors{$changes{$file}{$.}}
	    }
	    else {
		$number = $.;
		$bgcolor = $::BGColors{'normal' . ($. % 2)};
	    }

	    $states = "";
	    if (defined($states{$file}{$.})) {
		$states = join ", ", @{$states{$file}{$.}};
		$states = escape_html($states);
	    }
	    
	    $output = "";
	    $output .= "<tr valign=\"middle\">";
	    $output .= "<td bgcolor=\"$bgcolor\" align=right> " . 
		"<pre>$anchor$number  </td>"; 
	    $output .= "<td bgcolor=\"$bgcolor\"><pre>$code&nbsp;</td>";
	    $output .= "<td bgcolor=\"$bgcolor\"><pre>$states&nbsp;</td>";
	    $output .= "</tr>\n";
	    print OUTPUT $output;
	} # end line

	print OUTPUT $::SourcePostamble;
	close OUTPUT;
	close INPUT;

    } # end file
}


# Make the per-trace frameset
sub generate_frameset {
    my ($tracename, $errors, $file, $line) = @_;
    my ($output);

    $file .= ".html";
    $line -= $::jump_offset;
    $line = 1 if $line < 1;

    $output = $::FramesetHTML;
    $output =~ s/\@\@TRACE\@\@/$tracename/g;
    $output =~ s/\@\@SRCFRAME\@\@/$::srcframe/g;
    $output =~ s/\@\@TRACEFRAME\@\@/$::traceframe/g;
    $output =~ s/\@\@ERRORS\@\@/$errors/g;
    $output =~ s/\@\@FILE\@\@/$file/g;
    $output =~ s/\@\@LINE\@\@/$line/g;

    if (! open FRAMESET, "> $::framesetname") {
	warn("Unabled to create frameset: $!\n");
	return;
    }
    print FRAMESET $output;
    close FRAMESET;
}


# Make the top-level index. 
# The top-level index is an HTML file containing a link to each trace,
# grouped by final MFSA state. Within each final MFSA state, the traces
# are grouped by last program point. This function assumes that the
# only traces we want to link to are the ones we just produced; any
# other traces (for example, from previous invocations of MOPS) are not
# linked in the top-level index. This is usually the desired behavior.
# The top-level index also includes links to ../{profile,log}, which
# should contain, respectively, profiling information and a copy of the
# log file for the build process for the program in question.
# (Profiling data is only available for programs run by MOPS ld.)
sub create_index {
    &debug("Creating index\n");
    open OUTPUT, ">" . $::outputdir . "/index.html" or
	(warn "Unable to create index: $!\n" && return 0);

    my ($list, $reason);
    $list = "";
    for $reason (sort keys %::errors) {
	$list .= "  <dt><b>Error detected:</b> " . escape_html($reason) . "\n";
	for my $place (sort by_position keys %{$::errors{$reason}}) {
	    my ($count, $trace, @traces);

	    $list .= "  <dd>" . escape_html($place) . "\n";
	    $count = 0;
	    @traces = @{$::errors{$reason}{$place}};
	    while ($trace = shift @traces) {
		$trace = basename($trace);
		$list .= "    <a href=\"" . escape_href($trace) . 
		    "/index.html\">" . ++$count . "</a>\n";
	    }
	    $list .= "<br>\n";
	}
    }

    my ($date, $output);

    $date = localtime();
    $output = $::IndexHTML;
    $output =~ s/\@\@MFSA\@\@/$::mfsaname/g;
    $output =~ s/\@\@LABEL\@\@/$::mfsalabel/g;
    $output =~ s/\@\@FILE\@\@/$::filename/g;
    $output =~ s/\@\@LIST\@\@/$list/g;
    $output =~ s/\@\@DATE\@\@/$date/g;
    print OUTPUT $output;
    close OUTPUT;
    &debug("Index creation complete\n");
}

######################################################################
#                          Initialization                            #
######################################################################


# Process command-line options
sub check_options {
    my ($options, $valid_options);

    $options = {};
    getopts('f:g:ho:v', $options);
    @::traces = @ARGV;

    usage() if ($options->{'h'});
    usage() if (!defined($options->{'o'}));

    $::outputdir = $options->{'o'};
    if ($::outputdir !~ m!^/!) {
      $::outputdir = getcwd() . "/" . $::outputdir;
    }
    ($::outputdir .= "/") =~ s!/+!/!g;
    $valid_options = 1;

    if (defined($options->{'v'})) {
	$::verbose = 1;
	$valid_options++;
    }

    if (defined($options->{'f'})) {
	$::filename = $options->{'f'};
	$valid_options++;
    }
    $::filename = "Unknown" unless $::filename;

    if (defined($options->{'g'})) {
	open $::msgfh, ">> $options->{'g'}" or
	    die "Unable to open log file: $!\n";
	$valid_options++;
    }

    &usage if (keys(%$options) != $valid_options);
    &usage if (scalar(@::traces) == 0);
}

sub usage {
    print STDERR "Usage: $0 [-v] [-m mfsaname] -c <cfg> " . 
	"-o <outputdir> <trace>+ ...\n";
    exit 1;
}



######################################################################
#                          Utility Functions                         #
######################################################################


# Generate an HTML anchor referencing a given file:line
sub anchor {
    my ($file, $line) = @_;
    return "<a name=\"$file:$line\">";
}

# Generate an HTML anchor with just a line number
sub anchor_nofile {
    my ($file, $line) = @_;
    return "<a name=\"$line\">";
}

# Sorting subroutine for positions within source files. If the positions
# are in different files, sorts lexically by filename; if the files are
# the same, sorts numerically by line number.
sub by_position {
    my ($a_file, $a_line) = split /:/, $a, 2;
    my ($b_file, $b_line) = split /:/, $b, 2;
    my $cmp = $a_file cmp $b_file;
    return $cmp if $cmp != 0;
    return $a_line <=> $b_line;
}

# Print diagnositic messages if the -v flag was specified
sub debug {
    if ($::msgfh) {
	print $::msgfh basename($0), ": ", @_ if $::verbose;
    }
    else {
	print STDERR @_ if $::verbose;
    }
}

# Turn a line of arbitrary text into presentable HTML
sub escape_html {
    my ($code) = @_;

    # Escape special characters
    $code =~ s/&/&amp;/g;
    $code =~ s/</&lt;/g;
    $code =~ s/>/&gt;/g;
    $code =~ s/\"/&quot;/g;

    return $code;
}

# Escape characters which are legal in filenames (and which may therefore
# appear as property or program names) but which are not allowed in URLs.
# XXX TODO: Currently only handles spaces (the most common case); should
# be extended to escape all invalid characters. Such a function probably
# already exists, on CPAN if nothing else; we could probably either steal
# one or import the appropriate module.
sub escape_href {
    my ($href) = @_;

    $href =~ s/%/%25/g; # Must do this first
    $href =~ s/ /%20/g;

    return $href;
}


# Allow proper indentation in heredocs
sub fixup_heredoc {
    local $_ = shift;
    my ($white, $leader);  # common white space and common leading string
    $white = "";
    $leader = "\@\@";
    s/^\s*?$leader//gm;
    return $_;
}

# Link to a file:line based on an AST node
sub link_to_function {
    my ($file, $line, $astnode) = @_;
    my ($function_name, $jumpline);
    $function_name = resolve_astnode($astnode);
    $jumpline = $line - $::jump_offset;
    $jumpline = 1 if $jumpline < 1;

    return "<a href=\"$file.html\#$jumpline\" target=\"$::srcframe\">" . 
	"$function_name</a>";
}

# Link to a file:line
# Put @@ before the filename so we can find it and remove the source
# path prefix before printing.
sub link_to_source {
    my ($file, $line) = @_;

    my $jumpline = $line - $::jump_offset;
    $jumpline = 1 if $jumpline < 1;
    return "href=\"\@\@$file.html\#$jumpline\" target=\"$::srcframe\"";
}

# Link to the line in the trace corresponding to a source code line
sub link_to_trace {
    my ($file, $line) = @_;

    my $jumpline = $line - $::jump_offset;
    $jumpline = 1 if $jumpline < 1;
    return "<a href=\"#$file:$jumpline\" target=\"$::traceframe\">" . 
	"$line</a>";
}

# Perl equivalent of mkdir -p
sub mkdir_recursive {
    my ($path, $perms) = @_;
    mkdir_recursive(dirname($path), $perms) unless -d dirname($path);
    mkdir($path, $perms);
}

# Find the function name associated with an AST node ID
{
    my %nodeids;
    sub resolve_astnode {
	my ($nodeid) = @_;
	my (@cmd, $node, $name);

	if ($nodeid =~ /\D/) {
	    &debug("Cannot resolve non-numeric AST node $nodeid\n");
	    return "**UNKNOWN**";
	}

	return $nodeids{$nodeid} if defined($nodeids{$nodeid});

	@cmd = ($::MOPS_JAVA, "-Xmx1024m", "-cp", $ENV{MOPS_CLASSPATH},
		"ResolveAstNode", "-i", $::cfg,
		"-n", $nodeid);
	($node) = safe_exec(@cmd);
	$node =~ s/^\d+://;
	chomp $node;
	unless ($node) {
	    $nodeids{$nodeid} = "**UNKNOWN**";
	    return "**UNKNOWN**";
	}
	$nodeids{$nodeid} = $node . "()";

	return "$node()";
    }
}

# Execute a program safely, without possibility of interference from
# the shell, and return a list of the results. 
sub safe_exec {
    my (@command) = @_;
    my ($pid, @lines);

    $pid = open(CHILD, "-|");
    if (!defined($pid)) {
	die("Unable to fork process $command[0]: $!\n");
    }

    if ($pid) {
	# parent
	@lines = <CHILD>;
	close(CHILD) or warn("$command[0] exited with code $?");
    }
    else {
	# child
	exec(@command) or die("Unable to run $command[0]: $!\n");
	# NOTREACHED
    }
    return @lines;
}

# Update the longest common source path prefix
sub update_prefix {
    my ($filename) = @_;
    if (! $::prefix) {
	$::prefix = dirname($filename) . "/";
	return;
    }

    while ($filename !~ /^$::prefix/) {
	$::prefix = dirname($::prefix) . "/";
	$::prefix = "/" if $::prefix eq "//";
    }
}


######################################################################
#                          Main control flow                         #
######################################################################

&check_options;

if (! (-d $::outputdir || mkdir_recursive $::outputdir, 0777)) {
    die("Unable to create output directory $::outputdir: $!");
}
if (! chdir $::outputdir) {
    die("Unable to enter output directory $::outputdir: $!");
}

$::MOPS_JAVA = "java";
if (defined($ENV{MOPS_JAVA})) {
    $::MOPS_JAVA = $ENV{MOPS_JAVA};
}
else {
    &debug("MOPS_JAVA not set in the environment; defaulting to 'java'\n");
}

my ($trace, $ntraces, $cfgName, $cfgTextName, @cmd);
$ntraces = 0;
&debug("Starting output generation\n");
foreach $trace (@::traces) {
    open INDEX, "$trace" or die "Unable to open trace $trace: $!\n";
    while (<INDEX>) {
	chomp;
	if (/mfsa: (.+)/) {
	    $::mfsaname = fileparse($1, ".mfsa");
	    &debug("Set MFSA to $::mfsaname from trace file\n");
	}
	elsif (/mfsalabel: (.+)/) {
	    $::mfsalabel = $1;
	    &debug("Set MFSA label to $::mfsalabel from trace file\n");
	}
	elsif (/cfg: (.+)/) {
	    $::cfg = $1;
	    &debug("Set CFG to $::cfg from trace file\n");
	}
	else {
	    &parse_trace($_);
	    $ntraces++;
	}
    }
    if (defined($cfgTextName))
    {
      unlink($cfgTextName);
      $cfgTextName = undef;
    }
    close INDEX;
}

&create_index;

&debug("Generated $ntraces trace(s)\n");
&debug("Output generation complete\n");
