#$Id: Util.pm,v 1.7 2004/10/25 00:55:39 hchen Exp $
package Util;
use Exporter();
@ISA = qw(Exporter);
@EXPORT = qw(%Verbose @Verbose
	     &error &warn &info &debug
	     &setProfile &getProfile &setLogFile &getLogFile
	     &getVerbosity &clean_dir &fixup_heredoc &runSysCmd
	     &setup_dirs &myBaseName &getInputFileName &begin &end
	     $ERROR $WARNING $INFO $DEBUG);
@EXPORT_OK = qw();

use strict;
use Cwd;
use FileHandle;
use File::Basename;
use vars qw (@Verbose %Verbose $ERROR $WARNING $INFO $DEBUG);
my ($verbosity, $programName, $profileName, $logFile, $logFileName);

BEGIN
{
  my ($i, $tempDir);
  $ERROR = 0;
  $WARNING = 1;
  $INFO = 2;
  $DEBUG = 3;
  @Verbose = ("ERROR", "WARNING", "INFO", "DEBUG");
  for $i (0 .. $#Verbose)
  {
    $Verbose{$Verbose[$i]} = $i;
  }
  $verbosity = $ENV{MOPS_VERBOSITY};
  $verbosity = 0 if (!defined($verbosity) ||
		     $verbosity < 0 || $verbosity > $#Verbose);
  $programName = basename($0);
  # Keep profileName undefined
  $logFile = \*STDERR;
  $logFileName = "-";
}

sub setProfile($)
{
  $profileName = $_[0];
}

sub getProfile()
{
  return $profileName;
}

# sub setLogFile($$)
# {
#   my ($dirname, $filename) = @_;
#   if (!defined($dirname) || !defined($filename) ||
#       $filename eq "STDERR" || $filename eq "-")
#   {
#     $logFile = \*STDERR;
#     $logFileName = "-";
#     STDERR->autoflush(1);
#   }
#   else
#   {
#     if (!(-d $dirname) && !mkdir($dirname))
#     {
#       &warn("Unable to create temporary directory $dirname: $!\n");
#     }
#     $logFileName = "$dirname/$filename";
#     $logFile = new FileHandle ">>$logFileName" or
#     die "Unable to open log file $logFileName: $!\n";
#     $logFile->autoflush(1);
#   }
# }

sub setLogFile($)
{
  $logFileName = $_[0];
  die "Log file undefined" if (!defined($logFileName));
  $logFile = new FileHandle ">>$logFileName" or
    die "Unable to open log file $logFileName: $!\n";
  $logFile->autoflush(1);
}

sub getLogFile()
{
  return $logFileName;
}

sub getVerbosity()
{
  return $verbosity;
}

sub printMessage(@)
{
  my $level = shift @_;
  
  print $logFile "$Verbose[$level]: $programName: ", @_
    if ($verbosity >= $level);
}

sub debug(@)
{
  printMessage($DEBUG, @_);
}

sub info(@)
{
  printMessage($INFO, @_);
}

sub warn(@)
{
  printMessage($WARNING, @_);
}

sub error(@) {
  printMessage($ERROR, @_);
}

# Clean out and recreate a directory
sub clean_dir($) {
    my ($dir) = @_;
    if (! -d $dir) {
	&debug("clean_dir called on non-directory $dir\n");
	return;
    }
    system("rm", "-r", "--", $dir);
    if ($? != 0) {
	&warn("rm returned ", sprintf("%hd", $?>>8), "\n");
	&warn("Unable to clear out MOPS temporary cfgs\n");
    }
    mkdir $dir;
}

# Allow proper indentation in heredocs
sub fixup_heredoc {
    local $_ = shift;
    my ($white, $leader);  # common white space and common leading string
    $white = "";
    $leader = "\@\@";
    s/^\s*?$leader//gm;
    return $_;
}

# Run a system command. Handles profiling, debug messages, and return
# status logic. Captures stdout and stderr from the subprogram and
# prints both to the logfile as well as MOPS's controlling terminal.
# If stdin or stdout should be redirected, specify a parameter as
# runSysCmd({ STDIN => "input.txt", STDOUT => "output.txt" }, ...)
sub runSysCmd(@)
{
    my @cmd = (@_);
    my @time = ();
    my ($i, $stdin, $stdout);

    # Detect if stdin or stdout should be redirected
    for $i (0 .. $#cmd)
    {
      if (ref($cmd[$i]) eq "HASH")
      {
	if (exists($cmd[$i]->{STDIN}))
	{
	  $stdin = $cmd[$i]->{STDIN};
	}
	if (exists($cmd[$i]->{STDOUT}))
	{
	  $stdout = $cmd[$i]->{STDOUT};
	}
	splice(@cmd, $i, 1);
	$i--;
      }
    }
    
    if (defined($profileName))
    {
	open PROFILE, ">> $profileName" or
	    &warn("Unable to append to profiling log: $!\n");
	print PROFILE "=-" x 30, "\n";
	print PROFILE join(" ", @cmd), "\n";
	close PROFILE;
	@time = ("/usr/bin/time", "-p", "-o", $profileName, "-a");
    }

    for $i (0 .. $#cmd)
    {
      print $logFile "bad\n" unless defined($cmd[$i]);
    }
#    &debug(join(" ", @cmd), "\n");
#    system @time, @cmd;
#    return int sprintf("%hd", $?>>8);

    if (!defined($stdout))
    {
      pipe(READOUT, WRITEOUT) or die("pipe1: $!\n");
    }
    pipe(READERR, WRITEERR) or die("pipe2: $!\n");
    
    my $pid = fork();
    die("fork: $!\n") if !defined($pid);
    if ($pid)
    {
      # parent
      my ($rin, $ein, $rout, $eout, $data, $len);

	$rin = $ein = '';
        if (!defined($stdout))
	{
	  close WRITEOUT;
	  vec($rin, fileno(READOUT), 1) = 1;
	}
        close WRITEERR;
	vec($rin, fileno(READERR), 1) = 1;
	$ein = $rin;
	while(1) {
	    select($rout = $rin, undef, $eout = $ein, undef);
	    if (!defined($stdout) && vec($rout, fileno(READOUT), 1)) {
		$len = sysread(READOUT, $data, 1024);
		last if $len == 0;
		print STDOUT $data;
		print $logFile $data;
	    }
	    if (vec($rout, fileno(READERR), 1)) {
		$len = sysread(READERR, $data, 1024);
		last if $len == 0;
		print STDERR $data;
		print $logFile $data;
	    }
	    last if (!defined($stdout) && vec($eout, fileno(READOUT), 1));
	    last if (vec($eout, fileno(READERR), 1));
	}
	close READOUT if (!defined($stdout));
        close READERR;
    } else { # child
	&debug(join(' ', @cmd), "\n");

	if (defined($stdin))
	{
	  close(STDIN);
	  open(STDIN, "<$stdin") or die("Cannot open $stdin for reading");
	}
	
	close STDOUT;
	if (defined($stdout))
	{
	  open(STDOUT, ">$stdout") or die("Cannot open $stdout for writing");
	}
	else
	{
	  close READOUT; 
	  open(STDOUT, ">&WRITEOUT") or die("open stdout: $!\n");
	  close WRITEOUT;
	}
	
	close STDERR;	
	close READERR;
	open(STDERR, ">&WRITEERR") or die("open stderr: $!\n");
	close WRITEERR;
	
	exec(@time, @cmd) or do
	{ 
	  error("exec failed: $!\n");
	  exit(1);
	}
    }

    my $rv;
    while ($pid != -1) {
	$rv = int sprintf("%hd", $?>>8);
	$pid = wait();
    }
    return $rv;

}

# Create all necessary directories
# sub setup_dirs($) {
#     my ($topdir) = @_;
#     if ($topdir && ! -d $topdir) {
# 	run_syscmd("mkdir", "-p", $topdir);
#     }

#     for my $subdir(qw(build cfgs traces)) {
# 	my $fullpath = $topdir . "/" . $subdir;
# 	if ((! -d $fullpath) && (! mkdir $fullpath)) {
# 	    &warn("Unable to create working subdirectory $fullpath: $!\n");
# 	}
#     }
# }

sub myBaseName($)
{
  return (fileparse($_[0], qr{\.[^.]+}))[0];
}

sub getInputFileName($)
{
  my ($argv) = @_;
  my ($inputFile);
  
  shift @$argv while ($#$argv >= 0 && $argv->[0] =~ /^-.+/);
  if (scalar(@$argv) > 0)
  {
    $inputFile = $argv->[0];
  }
  else
  {
    $inputFile = "-";
  }
  return $inputFile;
}

sub begin()
{
  debug("Start MOPS $programName @ARGV\n");
}

sub end($)
{
  my ($rc) = @_;

  debug("End MOPS $programName.  Return value is $rc\n\n");
  exit $rc;
}

1;
