// $Id: Ast.java,v 1.34 2003/10/20 23:29:34 hchen Exp $

import java.io.*;
import java.util.*;

/**
 * AST: Abstract Syntax Tree
 */
class Ast
{
  public Ast()
  {
    setKind(-1);
    setAddress(0);
    setChildren(null);
    setParent(null);
    visited = false;
  }

  public final void setKind(int kind)
  {
    this.kind = kind;
  }
  
  public final int getKind()
  {
    return kind;
  }

  public final void setAddress(int address)
  {
    this.address = address;
  }

  public final int getAddress()
  {
    return address;
  }

  public final void setChildren(Vector children)
  {
    this.children = children;
  }

  public final Vector getChildren()
  {
    return children;
  }

  public final void setParent(Ast parent)
  {
    this.parent = parent;
  }

  public final Ast getParent()
  {
    return parent;
  }
  
  /**
   * Read Ast from Input.
   *
   */
  public static Ast read(Input reader, IntHashtable localNodeTable,
			 IntHashtable globalNodeTable)
    throws IOException
  {
    int token;
    
    reader.nextToken();
    token = reader.tokenValue();
    if (!reader.isOk() || token != Constants.LP)
      Util.die(Util.FILE_FORMAT,
	       "expect {", reader.getFileName(), reader.getLineNumber());
    return read0(reader, localNodeTable, globalNodeTable);
  }

  /**
   * Read Ast from a String
   */
  public static Ast read(String buffer, IntHashtable localNodeTable,
			 IntHashtable globalNodeTable)
    throws IOException
  {
    return read(new TextInput(new StringReader(buffer), "(STRING)"),
		localNodeTable, globalNodeTable);
  }

  protected static Ast read0(Input reader, IntHashtable localNodeTable,
			     IntHashtable globalNodeTable)
    throws IOException
  {
    Ast ast, ast2;
    String str;
    int token, value;

    ast = new Ast();
    ast.children = new Vector();

    if (localNodeTable != null)
    // Each AST has an address
    {
      // Read address
      reader.nextToken();
      value = reader.intValue();
      if (!reader.isOk())
	Util.die(Util.FILE_FORMAT, "expect an integer", reader.getFileName(),
		 reader.getLineNumber());
      if (value != 0)
      {
	if (localNodeTable.containsKey(value) ||
	    (globalNodeTable != null && globalNodeTable.containsKey(value)))
	{
	  Util.warn(Util.WARNING, Util.FILE_FORMAT, 
		    "deny duplicate AST address " + value,
		    reader.getFileName(), reader.getLineNumber());
	}
        else
	{
          localNodeTable.put(value, ast);
	}
	ast.setAddress(value);
      }
    }
    
    // Read kind
    reader.nextToken();
    value = reader.byteValue();
    if (!reader.isOk())
    {
      //Util.stderr.println(reader.stringValue());
      Util.die(Util.FILE_FORMAT, "expect a kind",
	       reader.getFileName(), reader.getLineNumber());
    }
    ast.setKind(value);

    // Read children
    for (reader.nextToken(); reader.isOk(); reader.nextToken())
    {
      token = reader.tokenValue();
      if (!reader.isOk())
	token = Constants.STRING;
      switch(token)
      {
	case Constants.RP:
	  // End of current node
	  return ast;

	case Constants.LP:
	  // Begins a new node
	  ast2 = read0(reader, localNodeTable, globalNodeTable);
	  ast2.setParent(ast);
	  ast.children.add(ast2);
	  break;

	case Constants.ADDRESS:
	  // A pointer to an Ast
	  reader.nextToken();
	  value = reader.intValue();
	  if (!reader.isOk())
	    Util.die(Util.FILE_FORMAT, "expect an integer after *",
		     reader.getFileName(), reader.getLineNumber());
	  if (value == 0)
	  {
	    Util.warn(Util.WARNING, Util.FILE_FORMAT, "ignore a null AST reference",
		      reader.getFileName(), reader.getLineNumber());
	  }
	  else
	  {
	    ast2 = null;
	    if (localNodeTable != null)
	    {
	      ast2 = (Ast)localNodeTable.get(value);
	    }
	    if (ast2 == null && globalNodeTable != null)
	    {
	      ast2 = (Ast)globalNodeTable.get(value);
	    }
	    if (ast2 == null)
	    {
	      Util.warn(Util.WARNING, Util.FILE_FORMAT, "undefined AST reference " + value,
			reader.getFileName(), reader.getLineNumber());
	    }
	    else
	    {
	      ast.children.add(ast2);
	    }
	  }
	  break;

	default:
	  // A string
	  str = reader.stringValue();
	  if (!reader.isOk())
	    Util.die(Util.FILE_FORMAT, "expect a string",
		     reader.getFileName(), reader.getLineNumber());
	  ast.children.add(str);
	  break;
      }
    }

    Util.die(Util.FILE_FORMAT, "unexpected end of file",
	     reader.getFileName(), reader.getLineNumber());
    // unreachable
    return null;
  }

  /**
   * Print this AST and all its children
   *
   * @param printAddress Whether to write the address of this Ast node
   * @param hashCode Stores all Ast objects that have been written out
   *                 so that one node is not written out more than once
   *
   * If maxAddress != null, then assign an address to each node
   * else keep the node's address
   */
  public final void write(Output writer, boolean printAddress,
			  IntegerVar maxAddress)
    throws IOException
  {
    int i;
    Integer code;
    Object child;
    String str;

    if (getVisited())
    // We have printed this AST, so this time we only print a pointer to it.
    {
      writer.writeToken(Constants.ADDRESS);
      writer.writeInt(address);
      return;
    }
    setVisited();
    writer.writeToken(Constants.LP);
    if (printAddress)
      // need to print the address of the node	
    {
      if (maxAddress != null)
	// need to rewrite the address of the node
      {
        address = ++maxAddress.data;
      }
      writer.writeInt(address);
    }
    writer.writeByte(getKind());
    if (children != null)
    {
      for (i = 0; i < children.size(); i++)
      {
	child = children.get(i);
	if (child == null)
	{
	  Util.warn(Util.ERROR, Util.INTERNAL, "A child of an AST is null");
	}
	else if (child instanceof Ast)
	{
	  ((Ast)child).write(writer, printAddress, maxAddress);
	}
	else if (child instanceof String)
	{
	  writer.writeString((String)child);
	}
	else
	{
	  Util.warn(Util.WARNING, Util.FILE_FORMAT, "ignore the unknown AST object type " +
	  	     child.getClass().getName());
	}
      }
    }
    writer.writeToken(Constants.RP);
  }
  
  /**
   * Write the AST to a dot file for visualization
   */
  public final void writeToDot(PrintWriter writer, boolean isWriteAddress)
  {
    Vector nodeQueue, childQueue;
    HashSet nodeHash;
    Ast node, node2;
    Object child;
    int i;

    nodeQueue = new Vector();
    nodeHash = new HashSet();
    childQueue = new Vector();

    writer.println("digraph \"" + getKindLabel() + "\" {\nsize=\"8,11\";");
    nodeQueue.add(this);
    nodeHash.add(this);
    while (nodeQueue.size() > 0)
    {
      node = (Ast)nodeQueue.remove(0);
      writer.print(node.address);
      writer.print(" [label=\"");
      if (isWriteAddress)
	writer.print(node.getAddress() + ": ");
      writer.print(node.getKindLabel());
      for (i = 0; i < node.children.size(); i++)
      {
	child = node.children.get(i);
	if (child instanceof Ast)
	{
	  childQueue.add(child);
	  if (!nodeHash.contains(child))
	  {
	    nodeQueue.add(child);
	    nodeHash.add(child);
	  }
	}
	else if (child instanceof String)
	{
	  // need to escape the string.
	  //writer.print(' ');
	  //writeEscape(writer, (String)child);
	}
	else
	  Util.warn(Util.WARNING, Util.FILE_FORMAT, 
		    "unknown AST object type " + child.getClass().getName());
      }
      writer.println("\"]");
      for (i = 0; i < childQueue.size(); i++)
      {
	node2 = (Ast)childQueue.get(i);
	writer.println(node.address + " -> " + node2.address);
      }
      childQueue.clear();
    }
    writer.println("}");
    nodeHash.clear();    
  }
  
  public final void writeToDot(PrintWriter writer)
  {
    writeToDot(writer, false);
  }

  public final String getKindLabel()
  {
    String str;
    /*
    if (kind < 0 || kind >= kindString.length)
      str = "unmatched";
    else
      str = kindString[kind];
    */
    str = Constants.kindLabels[kind];
    if (str == null)
      str = "unmatched";
    return str;
  }

  public final String toString()
  {
    StringWriter writer = new StringWriter();

    if (!startVisit())
      return "(cannot print AST)";
    try
    {
      write(new TextOutput(new PrintWriter(writer)), false, null);
      writer.flush();
    }
    catch(IOException e)
    {
    }
    endVisit(this);
    return writer.toString();
  }

  /**
   * Match this AST against a list of pattern ASTs that are on all the
   * transitions from the same node in an FSA.
   *
   * @return The index of the first matched pattern.  If none of the
   * patterns match but one of them is of type kind_other, return its
   * index.  Otherwise, return -1 and warn, because in each FSA every
   * node is required to have a kind_other transition to itself which 
   * enables us to trace into user defined function.
   */
  public final int match(Vector patterns)
  {
    int otherIndex = -1, i;
    Ast pattern;
    
    for (i = 0; i < patterns.size(); i++)
    {
      pattern = (Ast)patterns.get(i);
      if (pattern.kind == Constants.kind_other)
      {
	if (otherIndex == -1)
	  otherIndex = i;
        else
	  Util.warn(Util.WARNING, Util.FILE_FORMAT, 
	     "Ignore duplicate { other } ASTs from the same node: indice "
		    + i + " and " + otherIndex + ")");
      }
      else
      {
	if (match(pattern, null))
	{
	  //Util.stderr.println("Matched: " + i + ": " + pattern.toString());
	  break;
	}
	else
	{
	  //Util.println("Failed to match " + this + " to " + pattern);
	}
      }
    }

    if (i < patterns.size())
    {
      return i;
    }
    else
    {
      return otherIndex;
    }
  }

  /**
   * This method operates in two modes.
   * <ul>
   * <li>If <code>assignment==null</code>, determine if this AST
   * matches the AST <code>pattern</code>.  Note that if <code>pattern</code>
   * is a pattern variable, it matches this unconditionally to make
   * Cfg.hasUsefulAst() happy.  There should be a better way to handle this.
   * </li>
   * <li>If <code>assignment!=null</code>, determine if this AST
   * matches the AST <code>pattern</code>.  However, if
   * <code>pattern</code> contains pattern variables, try to find the
   * binding from the pattern variables to concrete variables in the program
   * and store the binding in <code>assignment</code>.
   *
   * @param pattern A regular or pattern AST
   * @param assignment If not null, contains bindings from pattern variables
   * in the FSA to concrete variables in the program.
   *
   * assignment:
   * Key: name of a pattern variable in the FSA.
   * Value: Vector of matched concrete variables in the program.
   */
  public final boolean match(Ast pattern, Hashtable assignment)
  {
    Object a, b;
    boolean comparison, isFound;
    Object lastChild;
    boolean hasEllipsis;
    int i, numToCompare, index;
    String variable, str1, str2;
    Ast ast;
    Vector asts;

    // Util.stderr.println(toString() + " : " + pattern.toString());
    // Handle variable discovery
    if (pattern.kind == Constants.kind_var)
    {
      if (assignment == null)
	// this is so to make Cfg.hasUsefulAst() happy
	return true;
      variable = (String)pattern.children.get(0);
      if ((asts = (Vector)assignment.get(variable)) == null)
      {
	asts = new Vector();
	assignment.put(variable, asts);
      }
      // If the same variable appears in a pattern AST more than once,
      // then all of their instances in the concrete AST must match
      // each other.  E.g., if the pattern AST is f(x, x), then the
      // concrete AST f(3,3) matches, but the concrete AST f(3,2) does
      // not.
      //
      // June 9, 2002
      // The above comment is no longer valid.  I won't check for
      // consistent binding of each pattern variable in an AST here,
      // because I cannot check for consistent binding of each pattern
      // variable in all ASTs here anyway.  Instead, I leave
      // consistency checking to when a MetaFsa is instantiated, where
      // all instances of the same pattern variable will be replaced
      // by the same actual AST.
      // if (ast != null)
      //	return match(ast, null);
      asts.add(this);
      return true;
    }

    // kind_any matches any Ast or string
    if (pattern.kind == Constants.kind_any)
      return true;
    
    /* Handle the logical operators "not" and "or" */
    if (pattern.kind == Constants.kind_metanot)
    {
      if (pattern.children.size() != 1)
	Util.die(Util.FILE_FORMAT, "the operator 'not' must be followed by exactly one child in ASTs");
      return !match((Ast)pattern.children.get(0), assignment);
    }
    else if (pattern.kind == Constants.kind_or)
    {
      isFound = false;
      for (i = 0; i < pattern.children.size(); i++)
	if (match((Ast)pattern.children.get(i), assignment))
	{
	  if (assignment == null)
            return true;
	  else
	    isFound = true;
	}
      return isFound;
    }

    /* Match actual AST */
    if (kind != pattern.kind)
    {
      return false;
    }

    // Determine if two variables point to the same storage
    if (kind == Constants.kind_identifier)
    {
      // If both this Ast and the pattern Ast are variables (and therefore
      // contains ddecl) then compare ddecl only (ddecl points to the storage)
      if (pattern.children.size() >= 2 && children.size() >= 2)
	return (Ast)children.get(1) == (Ast)pattern.children.get(1);
      // Otherwise (both are not variables) compare their names only
      else
      {
	if (children.size() < 1 || pattern.children.size() < 1)
	  return false;
	str1 = (String)children.get(0);
	str2 = (String)pattern.children.get(0);
	// FIX ME: should also make sure that parent.kind == function_call
	return getUnmangledFunctionName(str1).equals(str2);
      }
    }
    
    // Determine if Ast pattern ends in an ellipsis
    hasEllipsis =
      (pattern.children.size() > 0) &&
      ((lastChild = pattern.children.get(pattern.children.size() - 1)) != null)
      && (lastChild instanceof Ast) &&
      (((Ast)lastChild).kind == Constants.kind_ellipsis);

    if ((hasEllipsis && children.size() < (pattern.children.size() - 1)) ||
	(!hasEllipsis && children.size() != pattern.children.size()))
      return false;

    numToCompare = pattern.children.size();
    if (hasEllipsis)
      numToCompare--;

    for (i = 0; i < numToCompare; i++)
    {
      a = children.get(i);
      b = pattern.children.get(i);
      if (b instanceof Ast && ((Ast)b).getKind() == Constants.kind_any)
      {
	comparison = true;
      }
      else
      {
	if (a instanceof Ast && b instanceof Ast)
	{
	  comparison = ((Ast)a).match((Ast)b, assignment);
	}
	else if (a instanceof String && b instanceof String)
	{
	  comparison = ((String)a).equals((String)b);
	}
	else
	{
	  comparison = false;
	}
      }
      if (!comparison)
	return false;
    }
    
    return true;
  }

  /**
   * Determine whether two pattern Asts match
   */
  public final boolean patternMatch(Ast pattern)
  {
    int i;
    Object a, b;
    boolean comparison;
    
    if (kind != pattern.kind ||
	children.size() != pattern.children.size())
      return false;
	
    for (i = 0; i < children.size(); i++)
    {
      a = children.get(i);
      b = pattern.children.get(i);
      if (a instanceof Ast && b instanceof Ast)
      {
        comparison = ((Ast)a).patternMatch((Ast)b);
      }
      else if (a instanceof String && b instanceof String)
      {
	comparison = ((String)a).equals((String)b);
      }
      else
	comparison = false;
      if (! comparison)
	return false;
    }
    
    return true;
  }

  /**
   * Find all variables in this pattern Ast.
   *
   * @param variables Stores the variables found.  Key: variable name.
   * Value: Vector of pointers to where the variable appears in the
   * pattern Ast
   */
  public final void findVariable(Hashtable variables)
  {
    String name;
    Vector entry;
    int i;
    Object child;
    
    if (kind == Constants.kind_var)
    {
      name = (String)children.get(0);
      if ((entry = (Vector)variables.get(name)) == null)
      {
	entry = new Vector();
	variables.put(name, entry);
      }
      entry.add(this);
    }
    else
    {
      for (i = 0; i < children.size(); i++)
      {
	child = children.get(i);
	if (child instanceof Ast)
	  ((Ast)child).findVariable(variables);
      }
    }
  }

  /**
   * If this Ast is kind_funciton_call, return the function name
   */
  public final String getFunctionCallName()
  {
    Object child;
    
    if (kind != Constants.kind_function_call)
      return null;

    if ((child = children.get(0)) instanceof Ast &&
	((Ast)child).kind == Constants.kind_identifier)
    {
      return (String)((Ast)child).children.get(0);
    }
    else
    {
      Util.warn(Util.INFO, Util.EXTERNAL, "Unknown syntax of function_call");
      return null;
    }
  }

  public static final String getUnmangledFunctionName(String name)
  {
    int index;

    if (name == null || (index = name.indexOf(':')) == -1)
      return name;
    else
      return name.substring(index + 1);
  }
  
  public final void setVisited()
  {
    visited = true;
  }

  public final boolean getVisited()
  {
    return visited;
  }
  
  public static final boolean startVisit()
  {
    if (visitInProgress)
    {
      return false;
    }
    else
    {
      visitInProgress = true;
      return true;
    }
  }

  public static final boolean endVisit(Ast ast)
  {
    if (!visitInProgress)
    {
      return false;
    }
    else
    {
      ast.clearVisited();
      visitInProgress = false;
      return true;
    }
  }

  public final static boolean endVisit(Vector list)
  {
    int i;
    
    if (!visitInProgress)
    {
      return false;
    }
    else
    {
      for (i = 0; i < list.size(); i++)
	((Ast)list.get(i)).clearVisited();
      visitInProgress = false;
      return true;
    }
  }
  
  private final void clearVisited()
  {
    int i;
    Object child;

    if (visited == false)
    {
      return;
    }
    visited = false;
    if (children != null)
    {
      for (i = 0; i < children.size(); i++)
      {
	child = children.get(i);
	if (child == null)
	{
	  Util.warn(Util.ERROR, Util.INTERNAL, "A child of an AST is null");
	}
	else if (child instanceof Ast)
	{
	  ((Ast)child).clearVisited();
	}
      }
    }
  }
  
  static
  {
    visitInProgress = false;
  }
  
  /**
   * Type of this AST
   */
  protected int kind;

  /**
   * Address of this AST in the CFG generated by rc
   */
  protected int address;

  /**
   * Children of this AST.  A child may be a String or an AST
   */
  protected Vector children;

  protected Ast parent;

  /**
   * This is used for traversing the tree.
   * Invariant: Before or after traversing, done == classDone
   */
  private boolean visited;

  private static boolean visitInProgress;
}
