// $Id: Cfg.java,v 1.46 2003/10/23 07:03:37 hchen Exp $

import java.io.*;
import java.util.*;

public class Cfg
{
  public Cfg()
  {
    functions = new Vector();
    dataDecls = new Vector();
    functionDefns = new Vector();
    variableDefns = new Vector();
    variableDecls = new Vector();
    moduleName = "";
  }

  public Vector getFunctions()
  {
    return functions;
  }

  public Vector getDataDecls()
  {
    return dataDecls;
  }
  
  public String getModuleName()
  {
    return moduleName;
  }

  public void clear()
  {
    functions.clear();
    dataDecls.clear();
    functionDefns.clear();
    variableDefns.clear();
    variableDecls.clear();
    moduleName = "";
  }

  /** 
   * Read CFG 
   */
  public void read(String filename, boolean isBinary) throws IOException
  {
    Input reader;
    String str, label;
    int ttype, address;
    IntHashtable nodeHash, localAstTable, globalAstTable;
    Node node, node2;
    Edge edge;
    CfgFunction function;
    Ast ast = null;
    Iterator iter;
    Map.Entry entry;
    Vector list = null;
    int token, value;
    boolean ok;
    final int initCapacity = 256;

    clear();

    nodeHash = new IntHashtable(initCapacity);
    localAstTable = new IntHashtable(initCapacity);
    globalAstTable = new IntHashtable(initCapacity);
    if (isBinary)
      reader = new BinaryInput(
        new BufferedInputStream(Util.openInputStream(filename)), filename);
    else
      reader = new TextInput(
        new BufferedReader(Util.openReader(filename)), filename);

    moduleName = filename;
    function = null;

    ok = false;
    token = reader.nextToken();
    if (reader.isOk() && token == Constants.STRING)
    {
      str = reader.stringValue();
      ok = str.equals(Constants.CFG_MAGIC_STRING);
    }
    if (!ok)
    {
      Util.die(Util.FILE_FORMAT, "Expect Constants.CFG_MAGIC_STRING at the beginning of file",
	       filename, reader.getLineNumber());
    }
    if (reader.skipToEOL() > 1)
      Util.warn(Util.WARNING, Util.FILE_FORMAT, "Ignore extra words at the end of line",
		filename, reader.getLineNumber());
    
    for (token = reader.nextToken(); reader.isOk(); token = reader.nextToken())
    {
      // skip empty line
      if (token == Constants.EOL)
	continue;

      token = reader.tokenValue();
      if (!reader.isOk())
	Util.die(Util.FILE_FORMAT, "expect a token", filename, reader.getLineNumber());
      if (token == Constants.EOF)
	break;
      switch(token)
      {
	case Constants.FB:
	  // function_decl
	  reader.nextToken();
	  label = reader.stringValue();
	  if (!reader.isOk())
	    Util.die(Util.FILE_FORMAT, "expect a function name", filename, reader.getLineNumber());
	  function = new CfgFunction();
	  function.label = label;
	  function.entry = function.exit = null;
	  function.cfg = this;
	  // read fileName
	  reader.nextToken();
	  if (!reader.isOk())
	    Util.die(Util.FILE_FORMAT, "Expect a file name", filename, reader.getLineNumber());
	  str = reader.stringValue();
	  function.fileName = str;
	  
	  if (reader.skipToEOL() > 1)
	    Util.warn(Util.WARNING, Util.FILE_FORMAT, "ignore extra words at the end of line",
		      filename, reader.getLineNumber());
	  function.ast = Ast.read(reader, localAstTable, globalAstTable);
	  break;

	case Constants.FE:
	  if (function.entry == null || function.exit == null)
	  {
	    Util.warn(Util.INFO, Util.EXTERNAL, "missing "
		      + (function.entry == null ? "entry " : "")
		      + (function.exit == null ? "exit " : "")
		      + "point in the function " +  function.label, filename,
		      reader.getLineNumber());
	  }	      
	  functions.add(function);
	  function = null;
	  localAstTable.clear();
	  nodeHash.clear();
	  break;
	  
	// node
	case Constants.N:
	case Constants.NN:
	case Constants.NX:  
	  if (function == null)
	    Util.die(Util.FILE_FORMAT, "nodes must be defined within a function", filename,
		     reader.getLineNumber());
  	  node = readNode(reader, nodeHash);
	  if (node == null)
	    Util.die(Util.FILE_FORMAT, "expect a node address", filename, reader.getLineNumber());
	  reader.nextToken();
	  value = reader.intValue();
	  if (reader.isOk())
	  {
	    node.setLabel(new Integer(value));
	  }
	  else
	  {
	    label = reader.stringValue();
	    if (!reader.isOk())
	      Util.die(Util.FILE_FORMAT, "expect a node label", filename, reader.getLineNumber());
	    node.setLabel(label);
	  }
	  if (token == Constants.NN)
	  {
	    if (function.entry == null)
	      function.entry = node;
	    else
	      Util.warn(Util.WARNING, Util.FILE_FORMAT, "ignore duplicate entry node",
			filename, reader.getLineNumber());
	  }
	  else if (token == Constants.NX)
	  {
	    if (function.exit == null)
	      function.exit = node;
	    else
	      Util.warn(Util.WARNING, Util.FILE_FORMAT, "ignore duplicate exit node",
			filename, reader.getLineNumber());
	    break;
	  }
	  break;

	// edge
	case Constants.E:
	  if (function == null)
	    Util.die(Util.FILE_FORMAT, "edges must be defined within a function", filename,
		     reader.getLineNumber());
	  node = readNode(reader, nodeHash);
	  node2 = readNode(reader, nodeHash);
	  if (node == null || node2 == null)
	    Util.die(Util.FILE_FORMAT, "expect a node address", filename, reader.getLineNumber());
	  reader.nextToken();
	  address = reader.intValue();
	  if (!reader.isOk())
	  {
	    Util.die(Util.FILE_FORMAT, "expect an integer AST address", filename, reader.getLineNumber());
	  }
	  if (address == 0)
	  {
	    Util.die(Util.FILE_FORMAT, "the address of an AST is zero", filename, reader.getLineNumber());
	  }
	  if ((ast = (Ast)localAstTable.get(address)) == null &&
	      (ast = (Ast)globalAstTable.get(address)) == null)
	  {
	    ast = new Ast();
	    ast.setAddress(address);
	    ast.setKind(Constants.kind_unmatched);
	    localAstTable.put(address, ast);
	    //Util.warn("Unresolved AST node address " + address, filename,
	    //      reader.getLineNumber());
	    //Util.stderr.print("U ");
	  }
	  edge = new Edge();
	  edge.setLabel(ast);
	  edge.setNodes(node, node2);
	  break;

        // global declaration
	case Constants.D:
	  //reader.skipToEOL();
	  ast = Ast.read(reader, globalAstTable, null);
	  if (ast == null)
	    Util.warn(Util.WARNING, Util.FILE_FORMAT, "expect a global declaration", filename,
		      reader.getLineNumber());
	  else
	    dataDecls.add(ast);
	  break;

	  /*
	// module name
	case 'm':
	  if (moduleName != null)
	    Util.warn(Util.WARNING, Util.FILE_FORMAT, filename,
		      reader.getLineNumber());
	  else
	  {
            moduleName = reader.nextToken();
	    if (moduleName == null)
	      Util.die(Util.FILE_FORMAT, "Module name expected", filename, reader.getLineNumber());
	  }
	  break;
	  */
	  
	// function definition
	case Constants.LF:
	case Constants.LC:
	case Constants.LU:
	  switch(token)
	  {
	    case Constants.LF:
	      list = functionDefns;
	      break;
	      
	    case Constants.LC:
	      list = variableDefns;
	      break;
	      
	    case Constants.LU:
	      list = variableDecls;
	      break;
	  }
	  
          while ((token = reader.nextToken()) != Constants.EOL &&
		 token != Constants.EOF)
	  {
	    str = reader.stringValue();
	    if (!reader.isOk())
	      Util.die(Util.FILE_FORMAT, "expect a string", filename, reader.getLineNumber());
	    list.add(str);
	  }
	  break;

	default:
	  Util.die(Util.FILE_FORMAT, "unknown syntax", filename, reader.getLineNumber());
      }
      /*
      // read eol
      if (reader.getTtype() != StreamTokenizer.TT_EOL &&
	  reader.getTtype() != StreamTokenizer.TT_EOF)
      {
        reader.nextToken();
        if (reader.getTtype() != StreamTokenizer.TT_EOL &&
	    reader.getTtype() != StreamTokenizer.TT_EOF)
        {
	  Util.warn(Util.WARNING, Util.FILE_FORMAT, "ignore extra words near the end of the line",
		    filename, reader.getLineNumber());
	  reader.skipToEOL();
        }
      }
      */
      if (reader.skipToEOL() > 1)
	Util.warn(Util.WARNING, Util.FILE_FORMAT, "ignore extra words near the end of the line",
		  filename, reader.getLineNumber());
    }

    if (function != null)
      Util.die(Util.FILE_FORMAT, "expect fe", filename, reader.getLineNumber());
    if (token != Constants.EOF)
      Util.die(Util.FILE_FORMAT, "Incomplete CFG file", filename, reader.getLineNumber());
    globalAstTable.clear();
    reader.close();
  }

  private Node readNode(Input reader, IntHashtable nodeHash)
    throws IOException
  {
    Node node;
    Integer address;
    int value;

    reader.nextToken();
    value = reader.intValue();
    if (!reader.isOk())
      Util.die(Util.FILE_FORMAT, "expect an integer", reader.getFileName(), reader.getLineNumber());
    if (value == 0)
      Util.warn(Util.WARNING, Util.FILE_FORMAT, "node address is 0");
    if ((node = (Node)nodeHash.get(value)) == null)
    {
      node = new Node();
      node.address = value;
      nodeHash.put(value, node);
    }
    return node;
  }

  private void writeLinkageNames(Output writer, Vector list, int token)
    throws IOException
  {
    int i;
    
    if (list.size() > 0)
    {
      writer.writeToken(token);
      for (i = 0; i < list.size(); i++)
        writer.writeString((String)list.get(i));
      writer.writeToken(Constants.EOL);
    }
  }
    
  /**
   * Write to file
   * Note: Different Ast node within different functions may have the same
   * address as a result of CfgMerge.
   */
  public void write(String filename, boolean isBinary, IntegerVar maxAddress)
    throws IOException
  {
    CfgFunction function;
    Vector nodeQueue, visitedAsts;
    Node node, node2;
    Edge edge;
    Vector outEdges, inEdges;
    int i, j;
    Output writer;
    Ast ast;
    Iterator iter;
    int nodeType;
    Object label;
    Calendar calendar;
    
    /*
    calendar = new GregorianCalendar();
    Util.stderr.println("begin traversing " + calendar.getTime());
    int count;
    for (j = 0; j < 10; j++)
    {
      count = 0;
    Ast.initDone();
    for (i = 0; i < dataDecls.size(); i++)
    {
      count += ((Ast)dataDecls.get(i)).traverse();
    }
    for (i = 0; i < functions.size(); i++)
    {
      function = (CfgFunction)functions.get(i);
      count += function.ast.traverse();
    }
    Util.stderr.println("count=" + count);
    }
    Util.stderr.println("end traversing " + new GregorianCalendar().getTime());
    */
    
    if (isBinary)
      writer = new BinaryOutput(
	new BufferedOutputStream(Util.openOutputStream(filename)));
    else
      writer = new TextOutput(
        new PrintWriter(new BufferedWriter(Util.openWriter(filename))));
    writer.writeString(Constants.CFG_MAGIC_STRING);
    writer.writeToken(Constants.EOL);
    
    if (!Ast.startVisit())
    {
      Util.die(Util.INTERNAL, "Ast.startVisit() failed");
    }
    Node.initDone();
    nodeQueue = new Vector();
    visitedAsts = new Vector();
    for (i = 0; i < dataDecls.size(); i++)
    {
      writer.writeToken(Constants.D);
      ast = (Ast)dataDecls.get(i);
      ast.write(writer, true, maxAddress);
      visitedAsts.add(ast);
      writer.writeToken(Constants.EOL);
    }
    
    for (i = 0; i < functions.size(); i++)
    {
      function = (CfgFunction)functions.get(i);
      writer.writeToken(Constants.FB);
      writer.writeString(function.label);
      writer.writeString(function.fileName);
      writer.writeToken(Constants.EOL);
      ast = function.ast;
      ast.write(writer, true, maxAddress);
      visitedAsts.add(ast);
      writer.writeToken(Constants.EOL);
      nodeQueue.add(function.entry);
      function.entry.setDone();
      while (nodeQueue.size() > 0)
      {
	node = (Node)nodeQueue.remove(nodeQueue.size() - 1);
	if (node == function.entry)
	  nodeType = Constants.NN;
	else if (node == function.exit)
	  nodeType = Constants.NX;
	else
	  nodeType = Constants.N;
	writer.writeToken(nodeType);
	writer.writeInt(node.getAddress());
	label = node.getLabel();
	if (label instanceof Integer)
	{
	  writer.writeInt(((Integer)label).intValue());
	}
	else
	{
	  writer.writeString((String)label);
	}
	writer.writeToken(Constants.EOL);
        outEdges = node.getOutEdges();
	for (j = 0; j < outEdges.size(); j++)
	{
	  edge = (Edge)outEdges.get(j);
	  node2 = (Node)edge.getDstNode();
	  writer.writeToken(Constants.E);
	  writer.writeInt(node.getAddress());
	  writer.writeInt(node2.getAddress());
	  writer.writeInt((ast = edge.getLabel()) == null ?
			  0 : ast.getAddress());
	  writer.writeToken(Constants.EOL);
	  if (!node2.getDone())
	  {
	    nodeQueue.add(node2);
	    node2.setDone();
	  }
	}
      }
      writer.writeToken(Constants.FE);
      writer.writeToken(Constants.EOL);
    }

    if (!Ast.endVisit(visitedAsts))
      Util.die(Util.INTERNAL, "Ast.endVisit() failed");
    writeLinkageNames(writer, functionDefns, Constants.LF);
    writeLinkageNames(writer, variableDefns, Constants.LC);
    writeLinkageNames(writer, variableDecls, Constants.LU);

    if (isBinary)
      writer.writeToken(Constants.EOF);
    writer.close();
  }

  /**
   * Write to .dot file
   */
  public void writeToDot(String filename, boolean isWriteAst,
			 boolean isWriteAddress, HashSet functionsToWrite)
    throws IOException
  {
    CfgFunction function;
    Vector nodeQueue;
    Node node, node2;
    Edge edge;
    Vector outEdges, inEdges;
    int i, j;
    PrintWriter writer;
    HashSet nodeHash;
    String label;
    Ast ast;
    final int maxLabelLength = 24;

    nodeHash = new HashSet();
    nodeQueue = new Vector();
    writer = new PrintWriter(new BufferedWriter(Util.openWriter(filename)));
    
    for (i = 0; i < functions.size(); i++)
    {
      function = (CfgFunction)functions.elementAt(i);
      if (functionsToWrite != null &&
	  !functionsToWrite.contains(function.label))
	continue;
      if (isWriteAst)
        function.ast.writeToDot(writer, isWriteAddress);

      writer.println("digraph \"" + function.label + "\" {\nsize=\"8,11\";");
      nodeQueue.add(function.entry);
      nodeHash.add(function.entry);
      while (nodeQueue.size() > 0)
      {
	node = (Node)nodeQueue.remove(nodeQueue.size() - 1);
	writer.print(node.getAddress());
	if (!isWriteAddress)
	{
	  label = node.getLabelString(function);
	  if (label.length() > maxLabelLength)
	    label = "..." + label.substring(label.length() - maxLabelLength + 3);
	  //label = label.substring(0, maxLabelLength);
 	  writer.print(" [label=");
	  TextOutput.writeEscape(writer, label);
	  if (node == function.entry)
	    writer.print(", shape=box");
	  else if (node == function.exit)
	    writer.print(", shape=trapezium");
	  writer.print("]");
	}
	writer.println(";");
        outEdges = node.getOutEdges();
	for (j = 0; j < outEdges.size(); j++)
	{
	  edge = (Edge)outEdges.get(j);
	  node2 = (Node)edge.getDstNode();
	  writer.print(node.getAddress() +  " -> " + node2.getAddress() +
		       " [label=");
	  ast = edge.getLabel();
	  if (ast == null)
	  {
	    writer.print("(unmatched)");
	  }
	  else if (isWriteAddress)
	  {
	    writer.print(ast.getAddress());
	  }
	  else
	  {
	    label = ast.getKindLabel();
	    if (ast.kind == Constants.kind_function_call)
	    {
	      label += " " +
		((Ast)ast.getChildren().get(0)).getChildren().get(0);
	    }
	    TextOutput.writeEscape(writer, label);
	  }
	  writer.println("];");
	  if (!nodeHash.contains(node2))
	  {
	    nodeQueue.add(node2);
	    nodeHash.add(node2);
	  }

	}
      }
      writer.println("}");
      nodeHash.clear();
    }

    writer.close();
    nodeQueue.clear();
  }

  // This part deals with compacting a CFG
  /**
   * Implements the phi function in DefUse.  Note: index 0 referes to bottom.
   */
  private static class Phi
  {
    public Phi()
    {
      bitSets = new Vector();
      // index 0 should not be used
      bitSets.add(null);
      indices = new Hashtable();
    }

    public void clear()
    {
      // Util.warn("Phi.clear(): bitSets.size()=" + bitSets.size());
      bitSets.clear();
      indices.clear();
    }

    /**
     * Make a new bit
     */
    public int newValue()
    {
      int index;
      BitSet bitSet;

      index = bitSets.size();
      bitSet = new BitSet(index + 1);
      bitSet.set(index);
      bitSets.add(bitSet);
      indices.put(bitSet, new Integer(index));
      return index;
    }

    /**
     * Join a list of BitSet
     */
    public int join(Vector list)
    {
      BitSet sum, element;
      int i, index;
      Integer indexInt;

      sum = new BitSet();
      for (i = 0; i < list.size(); i++)
      {
	index = ((Integer)list.get(i)).intValue();
	if (index == 0)
	  continue;
	else if (index >= bitSets.size())
	  Util.die(Util.INTERNAL, "index=" + index + " but bitSets.size()=" + bitSets.size());
	element = (BitSet)bitSets.get(index);
	sum.or(element);
      }
      if (sum.length() == 0)
      {
	Util.warn(Util.INFO, Util.INTERNAL, "join " + list.size() + " bottoms");
	return 0;
      }
      if ((indexInt = (Integer)indices.get(sum)) != null)
	return indexInt.intValue();
      else
      {
	bitSets.add(sum);
	index = bitSets.size() - 1;
	indices.put(sum, new Integer(index));
	return index;
      }
    }

    /**
     * Each element is a BitSet where each bit corresponds to a specific def.
     */
    private Vector bitSets;

    /**
     * Lookup table for elements in bitSets.  Key: an element in bitSet.
     * Value: its index in bitSet
     */
    private Hashtable indices;
  }

  /**
   * Compute DefUse value for each node in a CFG.
   * WARNING: Rely on the assumption that all node.getSN() == 0 initially
   */
  private void defUse(CfgFunction function, Vector asts,
		      Hashtable nameToFunction, CfgFunctionExt[] functionsExt,
		      boolean handleOther)
  {
    Vector nodeQueue, inEdges, outEdges, list;
    NodeExt node;
    Edge edge;
    int i, value, otherValue = 0;
    boolean isModified;
    Phi phi;

    nodeQueue = new Vector();
    node = (NodeExt)function.entry;
    nodeQueue.add(node);
    phi = new Phi();
    list = new Vector();
    // defuse value for the special transition "other"
    if (handleOther)
    {
      otherValue = phi.newValue();
    }
    
    while (nodeQueue.size() > 0)
    {
      node = (NodeExt)nodeQueue.remove(nodeQueue.size() - 1);
      //Util.stderr.print(node.label);
      inEdges = node.getInEdges();
      for (i = 0; i < inEdges.size(); i++)
      {
	edge = (Edge)inEdges.get(i);
	if (hasUsefulAst(edge, asts, nameToFunction, functionsExt))
	{
	  list.clear();
	  break;
	}
	else if (((Node)edge.getSrcNode()).getSN() != 0)
	{
	  list.add(new Integer(((Node)edge.getSrcNode()).getSN()));
	}
      }

      if (
	  // This node has an interesting incoming edge
	  i < inEdges.size() ||
	  // This node is the entry node
	  list.size() == 0)
      {
	if (node.getSN() == 0)
	// First time to visit this node.  Give it a new DefUse value
	{
	  node.setSN(phi.newValue());
	  isModified = true;
	}
	else
	{
	  // Already visited this node.
	  isModified = false;
	}
      }
      else
      {
	if (handleOther)
	{
	  list.add(new Integer(otherValue));
	}
	value = phi.join(list);
	if (value != node.getSN())
	{
	  node.setSN(value);
	  // Util.stderr.println(node.getAddress() + ": " + value);
	  isModified = true;
	}
	else
	{
	  isModified = false;
	}
        list.clear();
      }

      if (isModified)
      // Place all child nodes into work queue
      {
	// Util.stderr.println("  new value: " + node.getSN());
	outEdges = node.getOutEdges();
	for (i = 0; i < outEdges.size(); i++)
	  nodeQueue.add(((Edge)outEdges.get(i)).getDstNode());
      }
      /*
      else
	Util.stderr.println();
      */
    }

    phi.clear();
  }

  /**
   * Extends Node to record a set of numbers as the label.
   * Used in merging nodes
   */
  private static class NodeExt extends Node
  {
    public NodeExt(Node node, String functionLabel)
    {
      // shallow copy from node
      super(node.getInEdges(), node.getOutEdges());
      address = node.address;
      nodeSet = new IntHashSet(64);
      nodeSet.add(getAddress());
      this.functionLabel = functionLabel;
      functionSet = null;
      // for debugging
      label = node.label;
    }

    public Object getLabel()
    {
      StringBuffer strBuf;
      Iterator iter;

      strBuf = new StringBuffer();
      strBuf.append(functionLabel);
      strBuf.append(delimiter);
      iter = nodeSet.iterator();
      while (iter.hasNext())
      {
	strBuf.append(((IntHashBase.Entry)iter.next()).getKey());
	strBuf.append(delimiter);
      }
      if (functionSet != null)
      {
	iter = functionSet.iterator();
	while (iter.hasNext())
	{
	  strBuf.append('(');
	  strBuf.append(iter.next());
	  strBuf.append(')');
	  strBuf.append(delimiter);
	}
      }
      return strBuf.substring(0, strBuf.length() - 1);
    }

    /**
     * Add the label of another node to this node
     * The function entry node, if any, within the other node is ignored
     */
    public void addLabel(NodeExt node)
    {
      Iterator iter;

      if (node == null)
	return;
      iter = node.nodeSet.iterator();
      while (iter.hasNext())
      {
	nodeSet.add(((IntHashBase.Entry)iter.next()).getKey());
      }
      if (node.functionSet != null)
      {
	if (functionSet == null)
	  functionSet = new HashSet();
	iter = node.functionSet.iterator();
	while (iter.hasNext())
	{
	  functionSet.add(iter.next());
	}
      }
    }

    public void addFunctionLabel(String label)
    {
      if (functionSet == null)
	functionSet = new HashSet();
      functionSet.add(label);
    }

    // functionSet contains the name of useless functions that were removed.
    private HashSet functionSet;

    private IntHashSet nodeSet;

    private String functionLabel;

    private static char delimiter = ';';
  }

  /**
   * Provides additional information for each CfgFunction
   */
  private class CfgFunctionExt
  {
    /**
     * Whether this function is useful.  A function is useful if it contains
     * useful Asts or if it calls a useful function directly or indirectly.
     */
    public boolean isUseful;

    /**
     * All functions that call this function directly
     */
    public HashSet caller;

    /**
     * All functions that this function calls directly or indirectly
     * (transitive closure)
     */
    public HashSet callee;

    public CfgFunctionExt()
    {
      isUseful = false;
      caller = new HashSet();
      callee = new HashSet();
    }
  }

  /**
   * Insert ast into asts, sorted by ast.kind
   *
   * @return false if ast is already in asts, true otherwise
   */
  private boolean insertAst(Vector asts, Ast ast)
  {
    int i;
    Ast ast2;

    // Ideally, should use binary search
    for (i = 0; i < asts.size(); i++)
      if (((Ast)asts.get(i)).kind >= ast.kind)
	break;

    for (; i < asts.size(); i++)
    {
      ast2 = (Ast)asts.get(i);
      if (ast2.kind > ast.kind)
	break;
      if (ast2.patternMatch(ast))
	return false;
    }

    asts.add(i, ast);

    return true;
  }

  /**
   * Determine if edge.getLabel() is in asts.  asts is sorted by Ast.kind
   */
  private boolean hasUsefulAst(Edge edge, Vector asts, Hashtable nameToFunction,
			       CfgFunctionExt[] functionsExt)
  {
    int i, kind, kind2, start, end, index = -1;
    Ast ast;
    String str = null;
    Integer indexInt;

    if (asts.size() == 0 || (ast = edge.getLabel()) == null)
      return false;

    if (nameToFunction != null &&
	(str = ast.getFunctionCallName()) != null &&
	(indexInt = (Integer)nameToFunction.get(str)) != null &&
	functionsExt != null && 
	functionsExt[indexInt.intValue()].isUseful)
    // If ast is a call to a useful function, return true
    {
      return true;
    }

    // Binary search for matching Ast.kind in asts
    kind = edge.getLabel().kind;
    start = 0;
    end = asts.size() - 1;
    while (start <= end)
    {
      index = (start + end) / 2;
      kind2 = ((Ast)asts.get(index)).kind;
      if (kind < kind2)
	end = index - 1;
      else if (kind > kind2)
	start = index + 1;
      else
	break;
    }

    if (start <= end)
    // Found ast.kind in asts
    {
      // Check all ast in asts whose Ast.kind matches ast.kind
      start = end = index;
      for (start--; start >= 0; start--)
	if (((Ast)asts.get(start)).kind != kind)
	  break;
      start++;
      for (end++; end < asts.size(); end++)
	if (((Ast)asts.get(end)).kind != kind)
	  break;
      end--;

      for (i = start; i <= end; i++)
      {
	ast = (Ast)asts.get(i);
	// This handles pattern variable correctly, because a pattern variable
	// matches anything
	if (edge.getLabel().match(ast, null))
	  break;
      }
      if (i <= end)
      {
	return true;
      }
    }
    
    // Match ast against all pattern Asts.  Be sure to double check with all
    // possible pattern Asts in Ast.java
    /*
    for (i = asts.size() - 1; i >= 0; i--)
      if (((Ast)asts.get(i)).kind < Constants.kind_min_pattern)
	break;
    for (i++; i < asts.size(); i++)
      if (edge.getLabel().match((Ast)asts.get(i), null))
	return true;
    */
    for (i = 0; i < asts.size(); i++)
    {
      ast = (Ast)asts.get(i);
      if (ast.kind == Constants.kind_metanot ||
	  ast.kind == Constants.kind_or ||
	  ast.kind == Constants.kind_any ||
	  ast.kind == Constants.kind_ellipsis ||
	  ast.kind == Constants.kind_var)
      {
	if (edge.getLabel().match(ast, null))
	{
	  return true;
	}
      }
    }

    return false;
  }

  /**
   * If allowed, merge mergedEdge.dstNode into mergedEdge.srcNode and
   * discard the former.
   */
  private boolean mergeNode(Edge mergedEdge, Vector asts, CfgFunction function,
			Hashtable nameToFunction, CfgFunctionExt[] functionsExt,
			    IntegerVar intVar)
  {
    Vector inEdges1, inEdges2, outEdges1, outEdges2, deletedEdges;
    //HashSet labels;
    Node node1, node2;
    Edge edge;
    Ast ast;
    int i, j, index;
    String str = null;
    Iterator iter;
    Integer indexInt;
    boolean done;

    intVar.data = 0;
      
    node1 = (Node)mergedEdge.getSrcNode();
    node2 = (Node)mergedEdge.getDstNode();
    // Merge two adjacent nodes only if they have the same DefUse value
    // SN records DefUse value, which was computed by defUse()
    if (node1.getSN() != node2.getSN())
      return false;

    /*
    // Deny the merge of entry and exit nodes
    if ((node1 == function.entry && node2 == function.exit) ||
	(node1 == function.exit && node2 == function.entry))
      return false;

    // Deny the merge of exit node, because Pda.read() ignores exit node
    if (node1 == function.exit || node2 == function.exit)
      return false;
    */
    if (node2 == function.entry || node2 == function.exit)
    {
      return false;
    }
    
    //Util.stderr.println("Merging node " + node2.getAddress() + " into node "
    //       + node1.getAddress() + " ast="
    //		 + mergedEdge.getLabel().getAddress());
    
    inEdges1 = node1.getInEdges();
    inEdges2 = node2.getInEdges();
    outEdges1 = node1.getOutEdges();
    outEdges2 = node2.getOutEdges();

    // do not merge if it creates imfeasible paths
    for (i = 0; i < outEdges1.size(); i++)
    {
      edge = (Edge)outEdges1.get(i);
      if (edge.getDstNode() == node2)
	continue;
      else
	break;
    }
    for (j = 0; j < inEdges2.size(); j++)
    {
      edge = (Edge)inEdges2.get(j);
      if (edge.getSrcNode() == node1)
	continue;
      else
	break;
    }
    if (i < outEdges1.size() && j < inEdges2.size())
    {
      /*
      Util.stderr.println("Diagnosis: disallow merging node "
			 + node1.getAddress() + " and " + node2.getAddress() +
			 " " + outEdges1.size() + " " + i + " " +
			 + ((Edge)outEdges1.get(i)).getDstNode().getAddress()
			 + "; " + inEdges2.size() + " " + j + " " +
			  ((Edge)inEdges2.get(j)).getSrcNode().getAddress());
      */
      intVar.data = 1;
      return false;
    }
    //    Util.stderr.println("Merging node " + node1.getAddress() + " and " +
    //	       node2.getAddress());
    
    deletedEdges = new Vector();
    
    // Modify incoming edges to node2
    for (i = 0; i < inEdges2.size(); i++)
    {
      edge = (Edge)inEdges2.get(i);
      /*
      if (edge == mergedEdge)
	continue;
      */
      if (edge.getSrcNode() == node1)
      {
	deletedEdges.add(edge);
	edge.setSrcNode(null);
	edge.setDstNode(null);
	done = false;
	for (j = 0; j < outEdges1.size(); j++)
	  if ((Edge)outEdges1.get(j) == edge)
	  {
	    outEdges1.remove(j);
	    done = true;
	    break;
	  }
	if (!done)
	  Util.die(Util.INTERNAL, "cannot find the edge from SrcNode");
      }
      else
      {
        edge.setDstNode(node1);
        inEdges1.add(edge);
      }
    }

    // Modify outgoing edges from node2
    for (i = 0; i < outEdges2.size(); i++)
    {
      edge = (Edge)outEdges2.get(i);
      if (edge.getDstNode() == node1)
      {
	deletedEdges.add(edge);
	edge.setSrcNode(null);
	edge.setDstNode(null);
	done = false;
	for (j = 0; j < inEdges1.size(); j++)
	  if ((Edge)inEdges1.get(j) == edge)
	  {
	    inEdges1.remove(j);
	    done = true;
	    break;
	  }
	if (!done)
	  Util.die(Util.INTERNAL, "cannot find the edge from DstNode");
      }
      else
      {
        edge.setSrcNode(node1);
        outEdges1.add(edge);
      }
    }

    // Add the label of node2 to the label of node1
    ((NodeExt)node1).addLabel((NodeExt)node2);

    
    /*
      If any of the deleted edge contains a function call, add the label
      of this function and all of its callees to node1, because call
      to this function and all of its callees between node1 and node2
      are removed from the CFG.  The label of a function is
      represented by the label of its entry node, which happens to be
      the line number of the first line of the function definition.
    */
    for (i = 0; i < deletedEdges.size(); i++)
      if (nameToFunction != null &&
	  (ast = ((Edge)(deletedEdges.get(i))).getLabel()) != null &&
	  (str = ast.getFunctionCallName()) != null &&
	  (indexInt = (Integer)nameToFunction.get(str)) != null)
      {
	//Util.stderr.println("i=" + i + " callee=" + str);
	((NodeExt)node1).addFunctionLabel
	  (((CfgFunction)functions.get(indexInt.intValue())).label);
	iter = functionsExt[indexInt.intValue()].callee.iterator();
	while (iter.hasNext())
	{
	  index = ((Integer)iter.next()).intValue();
	  //Util.stderr.print(((CfgFunction)functions.get(index)).label + ":");
	  ((NodeExt)node1).addFunctionLabel
	    (((CfgFunction)functions.get(index)).label);
	}
	//Util.stderr.println();
      }

    // Remove node2 and mergedEdge
    node2.getInEdges().clear();
    node2.getOutEdges().clear();
    deletedEdges.clear();

    /*
    for (i = outEdges1.size() - 1; i >= 0; i--)
      if ((Edge)outEdges1.get(i) == mergedEdge)
      {
	outEdges1.remove(i);
	// should I break?
	// break;
      }
    mergedEdge.setSrcNode(null);
    mergedEdge.setDstNode(null);

    // Check if exit node is merged.  Entry node cannot be merged
    if (node2 == function.exit)
    {
      function.exit = node1;
    }
    else if (node2 == function.entry)
    {
      function.entry = node1;
    }
    */
    
    return true;
  }

  /**
   * Compact CFG.
   * <ul>
   * <li>Merge adjacent nodes if they have the same DefUse value</li>
   * <li>Remove useless function definitions.  A useless function is a function
   *     that do not have any useful Asts and do not call any interesting
   *     functions directly or indirectly.  Note that
   *     <code>entryFunction</code> is considered useful</li>
   * <ul>
   */
  public void compact(MetaFsa metaFsa, HashMap entryFunctions)
  {
    Vector asts, fsas, nodeQueue, inEdges, outEdges, list, edgeQueue,
      edgeQueue2, temp;
    IntegerVar intVar;
    CfgFunction function;
    Node node, node2;
    Edge edge, edge2;
    Hashtable nameToFunction;
    HashSet nodeHash, edgeHash, callee;
    Fsa fsa;
    Iterator iter, iter2;
    String name, str;
    Ast ast;
    int i, j, k, ret;
    Integer index, index2, index3;
    CfgFunctionExt[] functionsExt;
    CfgFunctionExt functionExt;
    boolean isUseful, isModified;
    NodeExt nodeExt;
    boolean handleOther = false;

    /*
      Get all useful Asts.  Note: variables are NOT instantiated here, but it
      does not matter, since a variable matches anything
    */
    fsas = metaFsa.getFsas();
    asts = new Vector();
    list = new Vector();
    for (i = 0; i < fsas.size(); i++)
    {
      fsa = (Fsa)fsas.get(i);
      switch (ret = fsa.checkTransitions())
      {
	case 0:
	  break;

	case 1:
	  handleOther = true;
	  break;

	case 2:
	  Util.warn(Util.WARNING, Util.FILE_FORMAT, "an FSA has consecutive non-self-loop { other } transitions, which will break CfgCompact");
	  handleOther = true;
	  break;

	case 3:
	  Util.warn(Util.WARNING, Util.FILE_FORMAT, "an FSA has an { any } transition, which will break CfgCompact");
	  break;

	default:
	  Util.die(Util.INTERNAL, "unknown return value from Fsa.checkTransitions(): " +
		   ret);
	  
      }
      // Util.stderr.println("ret=" + ret);
      fsa.getAllTransitions(list);
      for (j = 0; j < list.size(); j++)
      {
	ast = (Ast)((FsaTransition)list.get(j)).input;
	// Warning: assume kind_other is insignificant
	if (ast.kind != Constants.kind_other)
	  insertAst(asts, ast);
      }
      list.clear();
    }

    // Index each function name.  functionsExt[] will be used to hold extra
    // information about each function later.
    nameToFunction = new Hashtable();
    functionsExt = new CfgFunctionExt[functions.size()];
    for (i = 0; i < functionsExt.length; i++)
    {
      nameToFunction.put(((CfgFunction)functions.get(i)).label, new Integer(i));
      functionsExt[i] = new CfgFunctionExt();
    }

    // Change every Node into NodeExt while keeping the graph structure of CFG
    // This is a hack!
    nodeHash = new HashSet();
    nodeQueue = new Vector();
    for (i = 0; i < functions.size(); i++)
    {
      function = (CfgFunction)functions.get(i);
      nodeQueue.add(function.entry);
      nodeHash.add(function.entry);
      while (nodeQueue.size() > 0)
      {
	node = (Node)nodeQueue.remove(nodeQueue.size() - 1);
	nodeExt = new NodeExt(node, function.label);
	inEdges = node.getInEdges();
	for (j = 0; j < inEdges.size(); j++)
	{
	  edge = (Edge)inEdges.get(j);
	  edge.setDstNode(nodeExt);
	}
	outEdges = node.getOutEdges();
	for (j = 0; j < outEdges.size(); j++)
	{
	  edge = (Edge)outEdges.get(j);
	  edge.setSrcNode(nodeExt);
	  node2 = (Node)(edge.getDstNode());
	  if (!(node2 instanceof NodeExt) && !nodeHash.contains(node2))
          {
	    nodeQueue.add(node2);
	    nodeHash.add(node2);
	  }
	}
	if (node == function.exit)
	  function.exit = nodeExt;
	else if (node == function.entry)
	  function.entry = nodeExt;
      }
      if (!(function.entry instanceof NodeExt &&
	    function.exit instanceof NodeExt))
	Util.warn(Util.INFO, Util.INTERNAL, 
		  (function.entry instanceof NodeExt ? "" : "entry ") +
		  (function.exit instanceof NodeExt ? "" : "exit ") +
		  "node is not changed into nodeExt in function " +
		  function.label);

      nodeHash.clear();
    }

    // find out
    // 1. Which function has useful Ast
    // 2. The callers and callees of each function
    nodeHash.clear();
    for (i = 0; i < functions.size(); i++)
    {
      function = (CfgFunction)functions.get(i);
      nodeQueue.add(function.entry);
      nodeHash.add(function.entry);
      // the entry function is always useful.  Never remove it.
      isUseful = entryFunctions.containsKey(
	Ast.getUnmangledFunctionName(function.label));
      while (nodeQueue.size() > 0)
      {
	node = (Node)nodeQueue.remove(nodeQueue.size() - 1);
	outEdges = node.getOutEdges();
	for (j = 0; j < outEdges.size(); j++)
	{
	  edge = (Edge)outEdges.get(j);
	  if (edge.getLabel() != null)
	  {
	    //Util.stderr.print(edge.getLabel());
	    if (!isUseful && hasUsefulAst(edge, asts, null, null))
	    {
	      isUseful = true;
	      //Util.stderr.println("  isuseful");
	    }
	    //else
	    //Util.stderr.println(" is not useful");
	    if ((str = edge.getLabel().getFunctionCallName()) != null &&
		(index = (Integer)nameToFunction.get(str)) != null)
	    {
	      functionsExt[index.intValue()].caller.add(new Integer(i));
	      functionsExt[i].callee.add(index);
	    }
	  }
	  node2 = (Node)edge.getDstNode();
	  if (!nodeHash.contains(node2))
	  {
	    nodeQueue.add(node2);
	    nodeHash.add(node2);
	  }
	}
      }
      
      functionsExt[i].isUseful = isUseful;
      nodeHash.clear();
    }

    /*
    if (Util.isDebug())
    {
      int ind;
      
      for (i = 0; i < functionsExt.length; i++)
      {
	iter = functionsExt[i].caller.iterator();
	while (iter.hasNext())
	{
	  ind = ((Integer)iter.next()).intValue();
	  Util.warn(Util.DEBUG, Util.INTERNAL,
	    ((CfgFunction)functions.get(ind)).label + " -> " +
	    ((CfgFunction)functions.get(i)).label);
	}
      }
    }
    */
    // Compute the transitive closure of useful functions.  A function is
    // useful if it calls a useful function directly or indirectly
    for (i = 0; i < functionsExt.length; i++)
    {
      if (functionsExt[i].isUseful)
	nodeQueue.add(new Integer(i));
    }
    while (nodeQueue.size() > 0)
    {
      index = (Integer)nodeQueue.remove(nodeQueue.size() - 1);
      functionExt = (CfgFunctionExt)functionsExt[index.intValue()];
      iter = functionExt.caller.iterator();
      while (iter.hasNext())
      {
	index = (Integer)iter.next();
	if (!functionsExt[index.intValue()].isUseful)
	{
	  functionsExt[index.intValue()].isUseful = true;
	  nodeQueue.add(index);
	}
      }
    }

    // Computer transitive closure of callee
    nodeQueue.clear();
    nodeHash.clear();
    for (i = 0; i < functionsExt.length; i++)
      if (functionsExt[i].callee.size() > 0)
      {
	index = new Integer(i);
	nodeQueue.add(index);
	nodeHash.add(index);
      }
    
    while (nodeQueue.size() > 0)
    {
      // this function
      index = (Integer)nodeQueue.remove(nodeQueue.size() - 1);
      nodeHash.remove(index);
      iter = functionsExt[index.intValue()].caller.iterator();
      while (iter.hasNext())
      {
	// caller to this function
	index2 = (Integer)iter.next();
	// caller's callee list
	callee = functionsExt[index2.intValue()].callee;
	iter2 = functionsExt[index.intValue()].callee.iterator();
	isModified = false;
	while (iter2.hasNext())
	{
	  // One callee of this function
	  index3 = (Integer)iter2.next();
	  if (!callee.contains(index3))
	  {
	    callee.add(index3);
	    isModified = true;
	  }
	}
	if (isModified && !nodeHash.contains(index2))
	{
	  nodeQueue.add(index2);
	  nodeHash.add(index2);
	}
      }
    }

    /*
    nodeHash.clear();
    nodeQueue.clear();

    newNode = new HashSet();
    while (nodeQueue.size() > 0)
    {
      index = (Integer)nodeQueue.remove(0);
      nodeHash.remove(index);
      callee = functionsExt[index.intValue()].callee;
      iter = callee.iterator();
      while (iter.hasNext())
      {
	iter2 = functionsExt[((Integer)iter.next()).intValue()].callee.iterator();
	while (iter2.hasNext())
	{
	  index2 = (Integer)iter2.next();
	  if (!callee.contains(index2))
	  {
	    newNode.add(index2);
	  }
	}
      }
      if (newNode.size() > 0)
      {
	iter = newNode.iterator();
	while (iter.hasNext())
	  callee.add(iter.next());
	if (!nodeHash.contains(index))
	{
	  nodeHash.add(index);
	  nodeQueue.add(index);
	}
	newNode.clear();
      }
    }
    */
    
    // Compact CFG and remove useless functions
    edgeQueue = new Vector();
    edgeQueue2 = new Vector();
    edgeHash = new HashSet();
    intVar = new IntegerVar();
    for (i = 0; i < functions.size(); i++)
    {
      // Don't compact useless CfgFunction because it will be thrown away
      if (!functionsExt[i].isUseful)
	continue;

      function = (CfgFunction)functions.get(i);
      defUse(function, asts, nameToFunction, functionsExt, handleOther);
      // Util.stderr.println("compacting " + function.label);

      outEdges = function.entry.getOutEdges();
      for (j = 0; j < outEdges.size(); j++)
      {
	edge = (Edge)outEdges.get(j);
	if (!edgeHash.contains(edge))
	{
	  edgeQueue.add(edge);
	  edgeHash.add(edge);
	}
      }
      while (edgeQueue.size() > 0)
      {
	edge = (Edge)edgeQueue.remove(edgeQueue.size() - 1);
	if (edge.getSrcNode() == null && edge.getDstNode() == null)
	  // this edge has been removed in mergeNode()
	  continue;
	//Util.stderr.println(node.getAddress() + ": " + node.getSN());
	outEdges = edge.getDstNode().getOutEdges();
	for (j = 0; j < outEdges.size(); j++)
	{
	  edge2 = (Edge)outEdges.get(j);
	  if (!edgeHash.contains(edge2))
	  {
	    edgeQueue.add(edge2);
	    edgeHash.add(edge2);
	  }
	}
	if (!mergeNode(edge, asts, function, nameToFunction, functionsExt,
		      intVar)
	    && intVar.data == 1)
	{
	  edgeQueue2.add(edge);
	}
      }
      edgeHash.clear();

      // fix me
      for (temp = edgeQueue, edgeQueue = edgeQueue2, edgeQueue2 = temp,
	     isModified = true;
	   isModified;
	   temp = edgeQueue, edgeQueue = edgeQueue2, edgeQueue2 = temp)
      {
	//Util.stderr.println("Handling infeasible path once.  queue.size()="
	//	   + edgeQueue.size());
	isModified = false;
	while (edgeQueue.size() > 0)
	{
	  edge = (Edge)edgeQueue.remove(edgeQueue.size() - 1);
	  if (edge.getSrcNode() == null && edge.getDstNode() == null)
	    // this edge has been removed in mergeNode()
	    continue;
	  if (mergeNode(edge, asts, function, nameToFunction, functionsExt,
			intVar))
	  {
	    isModified = true;
	  }
	  else if (intVar.data == 1)
	  {
	    edgeQueue2.add(edge);
	  }
	}
	edgeQueue2.clear();
      }
    }

    // remove useless functions from this.functions
    for (i = functions.size() - 1; i >= 0; i--)
      if (!functionsExt[i].isUseful)
	functions.remove(i);

    nameToFunction.clear();
  }

  /**
   * Find external (function and variable) name reference
   * information from CFGs
   */
  private static void findExternalDefinitions(Vector cfgs,
					      Hashtable variableDefns)
  {
    int i, j, k;
    String label;
    HashSet hashSet;
    CfgFunction function;
    Cfg cfg;
    Ast dataDecl, variableDecl, ast;

    hashSet = new HashSet();
    for (i = 0; i < cfgs.size(); i++)
    {
      cfg = (Cfg)cfgs.get(i);
      /*
      // functions
      for (j = 0; j < cfg.functionDefns.size(); j++)
      {
	hashSet.add(cfg.functionDefns.get(j));
      }
      for (j = 0; j < cfg.functions.size(); j++)
      {
	function = (CfgFunction)cfg.functions.get(j);
	label = function.label;
	if (hashSet.contains(label))
	{
	  if (functionDefns.containsKey(label))
	    Util.warn(Util.WARNING, Util.FILE_FORMAT, "ignore duplicate function definition "
		      + label + " in module " + cfg.getModuleName());
	  else
	    functionDefns.put(label, function);
	  hashSet.remove(label);
	}
      }
      if (hashSet.size() > 0)
      {
	Util.warn(Util.WARNING, Util.FILE_FORMAT, "cannot find the definitions of some external functions in module " + cfg.getModuleName());
	hashSet.clear();
      }
      */
      
      // variables
      for (j = 0; j < cfg.variableDefns.size(); j++)
      {
	hashSet.add(cfg.variableDefns.get(j));
      }
      for (j = 0; j < cfg.dataDecls.size(); j++)
      {
	dataDecl = (Ast)cfg.dataDecls.get(j);
	for (k = 0; k < dataDecl.getChildren().size(); k++)
	{
	  variableDecl = (Ast)dataDecl.getChildren().get(k);
	  // sanity check
	  if (variableDecl.getChildren().size() < 2 ||
	      ((ast = (Ast)variableDecl.getChildren().get(1)).getKind()
	      != Constants.kind_identifier_declarator &&
	       ast.getKind() != Constants.kind_pointer_declarator) ||
	      ast.getChildren().size() < 1 ||
	      !(ast.getChildren().get(0) instanceof String))
	    Util.warn(Util.INFO, Util.EXTERNAL, 
		      "ignored invalid variable declaration: " + variableDecl);
	  else
	  {
	    label = (String)ast.getChildren().get(0);
	    if (hashSet.contains(label))
	    {
	      if (variableDefns.containsKey(label))
		Util.warn(Util.WARNING, Util.FILE_FORMAT, 
			  "ignored duplicate variable declaration " + label +
			  " in module " + cfg.getModuleName());
	      else
	      {
		variableDefns.put(label, variableDecl.getChildren().get(0));
	      }
	      hashSet.remove(label);
	    }
	  }
	}
      }
      if (hashSet.size() > 0)
      {
	Util.warn(Util.WARNING, Util.FILE_FORMAT, "cannot find the definitions of some external variables in module " + cfg.getModuleName());
	hashSet.clear();
      }
    }
  }
  
  /**
   * Resolve references to global names (variables and functions) in the AST
   * of a CFG.
   */
  private static void linkExternals(Ast rootAst, Hashtable localFunctionDefns)
  {
    Vector nodeQueue;
    Ast ast, ast2, identifier;
    Object object;
    String name;
    CfgFunction function;
    int i;

    nodeQueue = new Vector();
    nodeQueue.add(rootAst);
    rootAst.setVisited();
    while (nodeQueue.size() > 0)
    {
      ast = (Ast)nodeQueue.remove(nodeQueue.size() - 1);
      switch (ast.getKind())
      {
	case Constants.kind_identifier:
	  // Resolve an identifier pointing to a global variable.
	  if (ast.getChildren().size() >= 2 &&
	      (object = ast.getChildren().get(1)) instanceof Ast &&
	      ((Ast)object).getKind() == -1)
	  // object.kind == -1 indicates that object is in the AST of a
	  // global variable that is defined in another module
	  // (object.kind was set to -1 in merge())
	  {
	    ast.getChildren().set(1, ((Ast)object).getChildren().get(0));
	  }
	  break;

	case Constants.kind_function_call:
	  // Resolve calls to external functions
	  identifier = (Ast)ast.children.get(0);
	  if (identifier.kind == Constants.kind_identifier)
          {
            name = (String)identifier.children.get(0);
	    if ((function = (CfgFunction)localFunctionDefns.get(name)) != null)
	    {
	      // At this moment, function.label has been mangled in merge()
	      identifier.children.set(0, function.label);
	    }
	  }
	  break;
      }
      
      for (i = 0; i < ast.children.size(); i++)
      {
	if ((object = ast.children.get(i)) instanceof Ast &&
	    !(ast2 = (Ast)object).getVisited())
	{
	  nodeQueue.add(ast2);
	  ast2.setVisited();
	}
      }
    }
  }

  /**
   * Merge CFGs.  Resolve cross references of global variables and functions
   *
   * @param crefFileName Cross reference file generated by ld
   */
  public static Cfg merge(Vector cfgs)
    throws IOException
  {
    BufferedReader reader;
    Cfg cfg, newCfg;
    CfgFunction function;
    Ast dataDecl, variableDecl, dataDeclaration, ast;
    String str, symbolName, fileName, name = null;
    Map.Entry entry;
    Object object;
    int i, j, k, l;
    Hashtable newVariableDefns;
    HashSet newFunctionDefns, // collect all functionDefns
      newVariableDecls; // collect all resolved variableDecls
    HashSet variableDecls, functionDefns;
    Hashtable[] localFunctionDefns;
    Vector visitedAsts;
    Iterator iter;

    // Change 07/13/03: mangle static function names only.
    // Do NOT mangle external function names.
    // Scan Cfgs.  Resolve external names
    // externFunctionDefns = new Hashtable();
    newVariableDefns = new Hashtable();
    newVariableDecls = new HashSet();
    newFunctionDefns = new HashSet();
    localFunctionDefns = new Hashtable[cfgs.size()];
    functionDefns = new HashSet();
    variableDecls = new HashSet();
    
    for (i = 0; i < localFunctionDefns.length; i++)
      localFunctionDefns[i] = new Hashtable();

    findExternalDefinitions(cfgs, newVariableDefns);
    // Collect all locally defined functions.
    // Note: this must be done after findExternalDefinitions()
    for (i = 0; i < cfgs.size(); i++)
    {
      cfg = (Cfg)cfgs.get(i);
      for (j = 0; j < cfg.functionDefns.size(); j++)
      {
	name = (String)cfg.functionDefns.get(j);
	functionDefns.add(name);
	if (newFunctionDefns.contains(name))
	{
	  Util.warn(Util.WARNING, Util.FILE_FORMAT, "Ignore duplicate function definition "
		    + name + " in file " + cfg.getModuleName());
	}
	else
	{
	  newFunctionDefns.add(name);
	}
      }
      
      for (j = 0; j < cfg.functions.size(); j++)
      {
	function = (CfgFunction)cfg.functions.get(j);
	if (!functionDefns.contains(function.label))
	{
	  localFunctionDefns[i].put(function.label, function);
	  //mangle static function names to make each of them unique
	  function.label = i + ":" + function.label;
	}
      }
      functionDefns.clear();
    }
    
    // Link external variable references 
    if (!Ast.startVisit())
    {
      Util.die(Util.INTERNAL, "Ast.startVisit() failed");
    }
    visitedAsts = new Vector();
    for (i = 0; i < cfgs.size(); i++)
    {
      cfg = (Cfg)cfgs.get(i);
      for (j = 0; j < cfg.variableDecls.size(); j++)
      {
	variableDecls.add(cfg.variableDecls.get(j));
      }
      
      // resolve global variables.  Remove variable declarations
      for (j = cfg.dataDecls.size() - 1; j >= 0; j--)
      {
	// data_decl contains variable_decl contains data_declaration
	dataDecl = (Ast)cfg.dataDecls.get(j);
        for (k = dataDecl.getChildren().size() - 1; k >= 0; k--)
	  if ((object = dataDecl.getChildren().get(k)) instanceof Ast &&
	      ((Ast)object).getKind() == Constants.kind_variable_decl)
	  {
	    variableDecl = (Ast)object;
	    name = null;
	    dataDeclaration = null;
            for (l = 0; l < variableDecl.getChildren().size(); l++)
	      if ((object = variableDecl.getChildren().get(l)) instanceof Ast)
	      {
		ast = (Ast)object;
		switch(ast.getKind())
		{
		  case Constants.kind_data_declaration:
		    dataDeclaration = ast;
		    break;

		  case Constants.kind_identifier_declarator:
		    name = (String)ast.children.get(0);
		    break;

		  case -1:
		    dataDecl.children.remove(k);
		    break;

		    /*
		  case Constants.kind_lexical_cst:
		    break;
		    
		  default:
		    Util.warn("Unexpected kind " + ast.getKind() + ": "
			     + variableDecl);
		    */
		}
	      }

	    if (name != null && dataDeclaration != null
	      /*
	    {
	      Util.warn((name == null ? "declarator " : "") +
			(dataDeclaration == null ? "data_declaration " : "") +
	                " not found inside variable_decl in module "
			+ cfg.getModuleName() + ": " + variableDecl);
	    }
	      */
		&& variableDecls.contains(name))
	    // This is an external variable declaration.  It should be removed
	    {
	      if ((ast = (Ast)newVariableDefns.get(name)) == null)
	      {
		// Util.warn(Util.WARNING, Util.FILE_FORMAT, 
		//	  "cannot find definition for external variable "
		//  + name + " in module " + cfg.getModuleName());
		if (!newVariableDecls.contains(name))
		{
		  newVariableDecls.add(name);
		}
	      }
	      else
	      {
		// remove this variable_decl from parent data_decl
		dataDecl.children.remove(k);
		// mark the kind of data_declaration as -1 to indicate that
		// it should be resolved later.  Cannot resolve it now
		// because we may not have found its definition yet.  Record
		// its entry in crossRefs.
		dataDeclaration.kind = -1;
		dataDeclaration.children.clear();
		dataDeclaration.children.add(ast);
	      }
	    }
	  }
	// If all variable_decl inside data_decl have been removed
	if (dataDecl.children.size() == 0)
	  cfg.dataDecls.remove(j);
      }

      // modify cfgs
      for (j = 0; j < cfg.functions.size(); j++)
      {
	function = (CfgFunction)cfg.functions.get(j);
	linkExternals(function.ast, localFunctionDefns[i]);
	visitedAsts.add(function.ast);
      }

      // modify dataDecls
      for (j = 0; j < cfg.dataDecls.size(); j++)
      {
	dataDecl = (Ast)cfg.dataDecls.get(j);
	linkExternals(dataDecl, localFunctionDefns[i]);
	visitedAsts.add(dataDecl);
      }
      variableDecls.clear();
    }

    if (!Ast.endVisit(visitedAsts))
    {
      Util.die(Util.INTERNAL, "Ast.endVisit() failed");
    }
    for (i = 0; i < localFunctionDefns.length; i++)
      localFunctionDefns[i].clear();

    // merge Cfgs
    // uniquefyAddresses(cfgs);
    newCfg = new Cfg();
    for (i = 0; i < cfgs.size(); i++)
    {
      cfg = (Cfg)cfgs.get(i);
      for (j = 0; j < cfg.functions.size(); j++)
	newCfg.functions.add(cfg.functions.get(j));
      for (j = 0; j < cfg.dataDecls.size(); j++)
	newCfg.dataDecls.add(cfg.dataDecls.get(j));
    }
    for (iter = newFunctionDefns.iterator(); iter.hasNext(); )
    {
      newCfg.functionDefns.add(iter.next());
    }
    for (iter = newVariableDefns.keySet().iterator(); iter.hasNext(); )
    {
      newCfg.variableDefns.add(iter.next());
    }
    for (iter = newVariableDecls.iterator(); iter.hasNext(); )
    {
      newCfg.variableDecls.add(iter.next());
    }

    for (i = 0; i < cfgs.size(); i++)
    {
      ((Cfg)cfgs.get(i)).clear();
    }
    newFunctionDefns.clear();
    newVariableDefns.clear();
    newVariableDecls.clear();
    return newCfg;
  }

  /**
   * Find a shortest path starting with entryNode.  All nodes on the
   * path must come from nodeSet (however not all nodes in nodeSet
   * have to be on the path) and the last node must have an outgoing
   * edge whose ast is astInt.  A node cannot appear more than once on
   * the path.  Since multiple outgoing edges from one exit node may
   * carry the same astInt and I cannot determine which edge to follow
   * without looking at the next node, I put all the potential next
   * entry node in nextEntryNodes.
   * 
   */
  private void backtrackNode(IntHashSet nodeSet, Vector entryNodes, int astInt,
			     Vector path, Vector nextEntryNodes)
  {
    Vector nodeQueue, outEdges, nodes;
    HashSet nodeHash;
    Node node, node2, exitNode;
    Edge edge;
    int i, j;

    // Find an entry node that is in nodeSet
    node = null;
    for (i = 0; i < entryNodes.size(); i++)
      if (nodeSet.contains(((Node)entryNodes.get(i)).getAddress()))
      {
	if (node == null)
	  node = (Node)entryNodes.get(i);
	else
	  Util.warn(Util.INFO, Util.INTERNAL, "Ignore duplicate entry nodes.  Old entryNode=" + node.getAddress() + "  New entryNode=" + ((Node)entryNodes.get(i)).getAddress());
      }
    if (node == null)
    {
      return;
    }
    // If astInt == 0, then nodeSet contains the exit node
    if (astInt == 0)
    {
      path.add(node);
      return;
    }

    // Set up the priority queue for dijkstra's algorithm
    node.setDistance(0);
    node.setParent(null);
    nodeQueue = new Vector();
    nodeHash = new HashSet();
    nodes = new Vector();
    nodeQueue.add(node);
    nodeHash.add(node);
    while (nodeQueue.size() > 0)
    {
      node = (Node)nodeQueue.remove(nodeQueue.size() - 1);
      nodes.add(node);
      outEdges = node.getOutEdges();
      for (i = 0; i < outEdges.size(); i++)
      {
	edge = (Edge)outEdges.get(i);
	node2 = (Node)edge.getDstNode();
	if (nodeSet.contains(node2.getAddress()) && !nodeHash.contains(node2))
	{
	  node2.setDistance(-1);
	  node2.setParent(null);
	  nodeQueue.add(node2);
	  nodeHash.add(node2);
	}
      }
    }

    Util.dijkstra(nodes, false, null);

    // find the exit node
    exitNode = null;
    for (i = 0; i < nodes.size(); i++)
    {
      node = (Node)nodes.get(i);
      if (node.getDistance() != -1)
      {
	outEdges = node.getOutEdges();
	for (j = 0; j < outEdges.size(); j++)
	{
	  edge = (Edge)outEdges.get(j);
	  if (edge.getLabel().getAddress() == astInt)
	  {
	    if (exitNode == null)
	    {
	      exitNode = node;
	    }
	    else if (node.compareTo(exitNode) < 0)
	    {
	      exitNode = node;
	      // Util.stderr.println("Found a closer exit node");
	    }
	    else
	    {
	      // Util.stderr.println("Found a farther exit node");
	    }
	    break;
	  }
	}
      }
    }
    if (exitNode == null)
    {
      // clean up
      nodes.clear();
      nodeHash.clear();
      return;
    }
    // get next entry nodes
    outEdges = exitNode.getOutEdges();
    for (i = 0; i < outEdges.size(); i++)
    {
      edge = (Edge)outEdges.get(i);
      if (edge.getLabel().getAddress() == astInt)
	nextEntryNodes.add(edge.getDstNode());
    }
    node = exitNode;
    while (true)
    {
      path.add(0, node);
      edge = (Edge)node.getParent();
      if (edge == null)
	break;
      node = (Node)edge.getSrcNode();
    }
    nodes.clear();
    nodeHash.clear();
  }

  private String changePrefix(String str, Vector dirPrefixes)
  {
    int i;
    StringPair stringPair;

    for (i = 0; i < dirPrefixes.size(); i++)
    {
      stringPair = (StringPair)dirPrefixes.get(i);
      if (str.startsWith(stringPair.str0))
      {
	return stringPair.str1 + str.substring(stringPair.str0.length());
      }
    }
    return str;
  }
  
  /**
   * Input: a trace from the compacted CFG
   * Output: the corresponding trace in the original CFG in the format
   * of emacs compilation mode (can use "goto-next-error" to go to next line)
   */
  public void transformPath(Input reader, PrintWriter writer,
			    Vector dirPrefixes)
    throws IOException
  {
    CfgFunction function;
    StringTokenizer st;
    Hashtable entries;
    IntHashSet nodeSet;
    Vector callStack, path, entryNodes, nextEntryNodes, outEdges, temp;
    Node node;
    String str, str2, state, functionLabel, lastLabel, lastState;
    Iterator iter;
    int i, ast, depth, lastDepth, index, token;
    char ch;

    lastLabel = "";
    lastState = "";

    // hash all function labels
    entries = new Hashtable();
    for (i = 0; i < functions.size(); i++)
    {
      function = (CfgFunction)functions.get(i);
      if (entries.containsKey(function.label))
      {
	Util.warn(Util.WARNING, Util.FILE_FORMAT, "ignore duplicate function label " + function.label);
      }
      else
      {
        entries.put(function.label, function);
      }
    }

    nodeSet = new IntHashSet(64);
    callStack = new Vector();
    path = new Vector();
    entryNodes = new Vector();
    nextEntryNodes = new Vector();
    lastDepth = 0;
    function = null;

    writer.println("-*- compilation -*-");
    reader.nextToken();
    str = reader.stringValue();
    if (!reader.isOk())
    {
      entries.clear();
      writer.flush();
      return;
    }
    writer.println("error: " + str);
    if (reader.skipToEOL() > 1)
      Util.warn(Util.WARNING, Util.FILE_FORMAT, "Ignore extra words at the end of line",
		reader.getFileName(), reader.getLineNumber());
    for (reader.nextToken(); reader.isOk(); reader.nextToken())
    {
      state = reader.stringValue();
      reader.nextToken();
      str = reader.stringValue();
      reader.nextToken();
      depth = reader.intValue();
      if (!reader.isOk())
	Util.die(Util.FILE_IO, "Reading error", reader.getFileName(), reader.getLineNumber());
      if (reader.skipToEOL() > 1)
	Util.warn(Util.WARNING, Util.FILE_FORMAT, "Ignore extra words at the end of line",
		  reader.getFileName(), reader.getLineNumber());

      if (depth > lastDepth)
      {
	if (lastDepth > 0)
	{
	  callStack.add(entryNodes);
	  callStack.add(function);
	  entryNodes = new Vector();
	}
      }
      else if (depth < lastDepth)
      {
	if (callStack.size() <= 0)
	  Util.die(Util.INTERNAL, "the call stack is empty when returning from a function call",
		   reader.getFileName(), reader.getLineNumber());
	entryNodes.clear();
	function = (CfgFunction)callStack.remove(callStack.size() - 1);
	entryNodes = (Vector)callStack.remove(callStack.size() - 1);
      }

      index = str.indexOf(';');
      if (index == -1)
	Util.die(Util.FILE_FORMAT, "cannot find a function label", reader.getFileName(),
		 reader.getLineNumber());
      functionLabel = str.substring(0, index);
      str = str.substring(index + 1);
      
      // break node addresses separated by ';'
      st = new StringTokenizer(str, ";");
      while (st.hasMoreTokens())
      {
	str2 = st.nextToken();
	// ignore function names in brackets because they are useless functions
	// We don't backtrack useless functions
	if (str2.charAt(0) != '(')
	{
	//if (str2.charAt(0) == 'n')
	//{
	// str2 = str2.substring(1);
	//nodeInt = new Integer(str2);
	//if (entryNodeInt == null)
	//  entryNodeInt = nodeInt;
	//else
	//  Util.warn("Duplicate entry node " + nodeInt + " in " + str);
	//}
	//else
	//{
	//}
	  nodeSet.add(Integer.parseInt(str2));
	}
      }
      
      if (depth > lastDepth)
      // enter a function call
      {
	if ((function = (CfgFunction)entries.get(functionLabel)) == null)
	  Util.die(Util.FILE_FORMAT,
		   "cannot find the real entry node inside a composite function entry node",
		   reader.getFileName(), reader.getLineNumber());
	node = function.entry;
	entryNodes.add(node);
	if (lastDepth == 0)
	{
	  writer.println("entry: " + function.label);
	}
      }

      if (entryNodes.size() == 0)
      {
	Util.die(Util.INTERNAL, "No entry node found");
      }
      
      // Read AST
      token = reader.nextToken();
      ast = reader.intValue();
      if (!reader.isOk())
	// last transition in current function
      {
	str = ((Node)entryNodes.get(0)).getLabelString(function);
	writer.println(changePrefix(str, dirPrefixes)
		       + ": \"" + state + "\"  " + depth);
	break;
      }
      if (reader.skipToEOL() > 1)
	Util.warn(Util.WARNING, Util.FILE_FORMAT, "Ignore extra words at the end of line",
		  reader.getFileName(), reader.getLineNumber());
      if (ast == 0)
	// nodeSet contains a single node, i.e. the exit node of the function,
	// therefore no need to backtrack
      {
	if (entryNodes.size() > 1)
	{
	  Util.warn(Util.ERROR, Util.INTERNAL,
		   "The composite exit node contains more than one node");
	  // remove me
	  for (i = 0; i < entryNodes.size(); i++)
	    Util.stderr.print(((Node)entryNodes.get(i)).getAddress() + " ");
	  Util.stderr.println();
	}
	path.add(entryNodes.get(0));
      }
      else
      {
	backtrackNode(nodeSet, entryNodes, ast, path, nextEntryNodes);
      /*
      Util.stderr.print("nextEntryNodes=");
      for (i = 0; i < nextEntryNodes.size(); i++)
	Util.stderr.print(((Node)nextEntryNodes.get(i)).getAddress() + " ");
      Util.stderr.println();
      */
	if (path.size() == 0)
	{
	  StringWriter sw = new StringWriter();
	  PrintWriter pw = new PrintWriter(sw);
	  pw.print("backtrackNode() failed.  entryNodes=");
	  for (i = 0; i < entryNodes.size(); i++)
	    pw.print(((Node)entryNodes.get(i)).getAddress() + " ");
	  pw.println("ast=" + ast);
	  Util.die(Util.INTERNAL, sw.toString());
	}
      }
      
      for (i = 0; i < path.size(); i++)
      {
	str = ((Node)path.get(i)).getLabelString(function);
	if ((i == (path.size() - 1)) ||
	    !((i == 0 && str.equals(lastLabel) && state.equals(lastState)) ||
	      str.equals(((Node)path.get(i + 1)).getLabelString(function))))
	{
          writer.print(changePrefix(str, dirPrefixes)
		       + ": \"" + state + "\"  " + depth);
	  if (i == (path.size() - 1))
	  {
	    writer.println(" " + ast);
	    lastLabel = str;
	    lastState = state;
	  }
	  else
	    writer.println();
	}
	/*
	else
	  Util.stderr.println("Remove redundant line: " + str);
	*/
      }
      entryNodes.clear();
      temp = entryNodes;
      entryNodes = nextEntryNodes;
      nextEntryNodes = temp;
      lastDepth = depth;
      // debug
      // writer.flush();
      
      path.clear();
      nodeSet.clear();
    }

    entries.clear();
    writer.flush();
  }

  /**
   * "Show me all statements that may be executed in certain state"
   * Input: output from Fsa.writeInitialTransitions() on compacted CFG
   * Output: transformed into filename:lineno in this (uncompacted) CFG
   */
  public void transformNodes(Input reader, PrintWriter writer)
    throws IOException
  {
    String str, moduleName, label;
    Node node, node2;
    CfgFunction function;
    StringTokenizer st;
    HashSet nodeSet, nodeHash, hashSet, positions;
    Vector nodeQueue, outEdges, list;
    HashMap positionsPerFunc;
    Hashtable positionsPerModule;
    Integer nodeInt, lineNumber;
    Iterator iter, iter2;
    Map.Entry entry;
    Object object;
    int i, j, index;

    // Sort all transitions by function name
    positionsPerFunc = new HashMap();
    for (reader.nextToken(); reader.isOk(); reader.nextToken())
    {
      str = reader.stringValue();
      index = str.indexOf(';');
      label = str.substring(0, index);
      if (positionsPerFunc.containsKey(label))
      {
	if ((positions = (HashSet)positionsPerFunc.get(label)) == null)
	{
	  Util.warn(Util.INFO, Util.INTERNAL, 
		    "ignore the following transition because the function"
		    + label + " is useless and has been removed: " + str);
	  continue;
	}
      }
      else
      {
	positions = new HashSet();
	positionsPerFunc.put(label, positions);
      }
      st = new StringTokenizer(str.substring(index + 1), ";");
      while (st.hasMoreTokens())
      {
	str = st.nextToken();
	if (str.charAt(0) == '(')
	{
	  label = str.substring(1, str.length() - 1);
	  if (positionsPerFunc.containsKey(label))
	  {
	    if (positionsPerFunc.get(label) != null)
	      Util.warn(Util.INFO, Util.INTERNAL, "ignore the useless function " + label +
			" because I have seen its transitions");
	  }
	  else
	    positionsPerFunc.put(label, null);
	}
	else
	  positions.add(new Integer(str));
      }
      if (reader.skipToEOL() > 1)
	Util.warn(Util.WARNING, Util.FILE_FORMAT, "Ignore extra words at the end of line",
		  reader.getFileName(), reader.getLineNumber());
    }

    // turn each transition into module name and line numbers
    positionsPerModule = new Hashtable();
    nodeQueue = new Vector();
    nodeHash = new HashSet();
    // traverse CFG
    for (i = 0; i < functions.size(); i++)
    {
      function = (CfgFunction)functions.get(i);
      if (!positionsPerFunc.containsKey(function.label))
	continue;
      positions = (HashSet)positionsPerFunc.get(function.label);
      node = function.entry;
      nodeQueue.add(node);
      nodeHash.add(node);
      while (nodeQueue.size() > 0)
      {
	node = (Node)nodeQueue.remove(nodeQueue.size() - 1);
	nodeInt = new Integer(node.getAddress());
        if (positions == null || positions.contains(nodeInt))
	{
	  object = node.getLabel();
	  if (object instanceof Integer)
	  {
	    moduleName = function.fileName;
	    lineNumber = (Integer)object;
	  }
	  else
	  {
	    st = new StringTokenizer((String)object, ":");
	    moduleName = st.nextToken();
	    lineNumber = new Integer(st.nextToken());
	  }
	  if ((hashSet = (HashSet)positionsPerModule.get(moduleName)) == null)
	  {
	    hashSet = new HashSet();
	    positionsPerModule.put(moduleName, hashSet);
	  }
	  hashSet.add(lineNumber);
	  if (positions != null)
	    positions.remove(nodeInt);
	}
	outEdges = node.getOutEdges();
	for (j = 0; j < outEdges.size(); j++)
	{
	  node2 = (Node)((Edge)outEdges.get(j)).getDstNode();
	  if (!nodeHash.contains(node2))
	  {
	    nodeHash.add(node2);
	    nodeQueue.add(node2);
	  }
	}
      }
      nodeHash.clear();
      if (positions != null && positions.size() > 0)
      {
	StringBuffer sb = new StringBuffer();
	iter = positions.iterator();
	while (iter.hasNext())
	{
	  sb.append((Integer)iter.next());
	  sb.append(' ');
	}
	Util.warn(Util.INFO, Util.INTERNAL, "cannot dereference these node addresses: "
		  + sb.toString());
      }
      positionsPerFunc.remove(function.label);
    }

    if (positionsPerFunc.keySet().size() > 0)
    {
      StringBuffer sb = new StringBuffer();
      iter = positionsPerFunc.keySet().iterator();
      while (iter.hasNext())
      {
	sb.append(iter.next());
	sb.append(' ');
      }
      Util.warn(Util.WARNING, Util.FILE_FORMAT,
	"cannot resolve these functions in the source code: " + sb.toString());
    }
    
    iter = positionsPerModule.entrySet().iterator();
    list = new Vector();
    while (iter.hasNext())
    {
      entry = (Map.Entry)iter.next();
      moduleName = (String)entry.getKey();
      iter2 = ((HashSet)entry.getValue()).iterator();
      while (iter2.hasNext())
	list.add(iter2.next());
      Collections.sort(list);
      for (i = 0; i < list.size(); i++)
	writer.println(moduleName + ":" + list.get(i));
      list.clear();
    }
    writer.flush();
    positionsPerModule.clear();
    positionsPerFunc.clear();
  }

  /**
   * Function declarations
   */
  protected Vector functions;

  /**
   * Global variable declarations
   */
  protected Vector dataDecls;

  /**
   * Name of this module
   */
  protected String moduleName;

  /**
   * Linkage information
   */
  protected Vector functionDefns, variableDefns, variableDecls;
}

/*
  private static int getUniqueAddress
    (HashSet allAddresses, HashSet localAddresses, int address,
     Random random)
  {
    Integer addressInt;
    int count, value;

    addressInt = new Integer(address);
    if (allAddresses.contains(addressInt))
    {
      count = 0;
      do
      {
	value = random.nextInt();
	if (value < 0)
	  value = -value;
	else if (value == 0)
	  value = 1;
        addressInt = new Integer(value);
	count++;
      }
      while (allAddresses.contains(addressInt) ||
	     localAddresses.contains(addressInt));
      //Util.stderr.println("Random: " + address + " -> " + addressInt
      //		 + "  count=" + count);
      return addressInt.intValue();
    }
    else
    {
      if (!localAddresses.contains(addressInt))
	localAddresses.add(addressInt);
      return address;
    }
  }

   * Change colliding addresses among Ast objects and Node objects
   * from different Cfgs.  Within a Cfg, no two Ast objects have the
   * same address.  Within a function, no two Node objects have the
   * same address.  (An Ast object and a Node object may have the same
   * address, but this is ok since they are in separate address
   * spaces.)  When we merge several Cfgs, we must ensure that in the
   * merged Cfg no two Ast objects have the same address, and no two
   * Node objects have the same address.
  private static void uniquefyAddresses(Vector cfgs)
  {
    Ast ast;
    Vector nodeQueue, outEdges, functions, dataDecls;
    HashSet nodeHash, allAddresses, localAddresses;
    Object object;
    int i, j, k, address;
    Random random;
    Node node, node2;
    Cfg cfg;
    CfgFunction function;
    Iterator iter;

    random = new Random();
    nodeQueue = new Vector();
    nodeHash = new HashSet();
    allAddresses = new HashSet();
    localAddresses = new HashSet();

    // Uniquefy Ast objects
    for (i = 0; i < cfgs.size(); i++)
    {
      cfg = (Cfg)cfgs.get(i);
      functions = cfg.getFunctions();
      dataDecls = cfg.getDataDecls();
      for (j = 0; j < functions.size() + dataDecls.size(); j++)
      {
	if (j < functions.size())
	{
	  ast = ((CfgFunction)functions.get(j)).ast;
	}
	else
	{
	  ast = (Ast)dataDecls.get(j - functions.size());
	}
	nodeQueue.add(ast);
	nodeHash.add(ast);
	while (nodeQueue.size() > 0)
	{
	  ast = (Ast)nodeQueue.remove(0);
	  address = getUniqueAddress(allAddresses, localAddresses,
				     ast.getAddress(), random);
	  if (address != ast.getAddress())
	    ast.setAddress(address);
	  if (ast.children != null)
	    for (k = 0; k < ast.children.size(); k++)
	    {
	      object = ast.children.get(k);
	      if (object instanceof Ast && !nodeHash.contains(object))
	      {
	        nodeQueue.add(object);
   	        nodeHash.add(object);
	      }
	    }
	}
	nodeHash.clear();
      }
      // merge localAddresses into allAddresses
      iter = localAddresses.iterator();
      while (iter.hasNext())
      {
	allAddresses.add(iter.next());
      }
      localAddresses.clear();
    }
    allAddresses.clear();

    // uniquefy Node objects
    for (i = 0; i < cfgs.size(); i++)
    {
      cfg = (Cfg)cfgs.get(i);
      functions = cfg.getFunctions();
      for (j = 0; j < functions.size(); j++)
      {
	function = (CfgFunction)functions.get(j);
	node = function.entry;
	nodeQueue.add(node);
	nodeHash.add(node);
	while (nodeQueue.size() > 0)
	{
	  node = (Node)nodeQueue.remove(0);
	  address = getUniqueAddress(allAddresses, localAddresses,
				     node.getAddress(), random);
	  if (address != node.getAddress())
	    node.setAddress(address);

	  outEdges = node.getOutEdges();
	  for (k = 0; k < outEdges.size(); k++)
	  {
	    node2 = ((Edge)outEdges.get(k)).getDstNode();
	    if (!nodeHash.contains(node2))
	    {
	      nodeQueue.add(node2);
	      nodeHash.add(node2);
	    }
	  }
	}
	nodeHash.clear();
        iter = localAddresses.iterator();
        while (iter.hasNext())
        {
	  allAddresses.add(iter.next());
        }
        localAddresses.clear();
      }
    }
    allAddresses.clear();
  }

*/
