// $Id: Check.java,v 1.29 2003/09/22 21:31:56 hchen Exp $

import java.io.*;
import java.util.*;
import gnu.getopt.*;

/**
 * User application to check for reachability of some states, or all
 * statements that may be executed in these states.
 */
public class Check
{
  static final String optionString = "e:f:i:m:o:p" + Util.getOptionString(),
    optionUsage =
    "Usage: Check [options] [-e entry_function]+ -m meta_FSA\n" +
    "  -e <name>\tUse <name> as the entry function\n" +
    "  -f <file>\tRead entry function names from <file> (one name per line)\n" +
    "  -i <file>\tRead CFG from <file>\n" +
    "  -m <file>\tRead Meta-FSA from <file>\n" +
    "  -o <file>\tWrite the output to <file>\n" +
    "  -p \t\tWrite program points instead traces\n" +
    Util.getOptionUsage();

  public static void main(String[] args) throws IOException
  {
    Getopt opt;
    int ch;
    boolean isProgramPoint = false;
    String mfsaFilename = null, cfgFilename = "-", traceFilename = "-";
    HashMap entryFunctions;
    String label, entryFunctionFileName = null;
    
    entryFunctions = new HashMap();
    opt = new Getopt("Check", args, optionString, Util.getLongOpts());
    while ((ch = opt.getopt()) != -1)
    {
      switch(ch)
      {
	case 'e':
	  label = opt.getOptarg();
	  if (entryFunctions.containsKey(label))
	  {
	    Util.warn(Util.WARNING, Util.FILE_IO, "redundant entry function " + label);
	  }
	  else
	  {
	    entryFunctions.put(label, null);
	  }
	  break;

	case 'f':
	  entryFunctionFileName = opt.getOptarg();
	  break;
	  
	case 'i':
	  cfgFilename = opt.getOptarg();
	  break;

	case 'm':
	  mfsaFilename = opt.getOptarg();
	  break;

	case 'o':
	  traceFilename = opt.getOptarg();
	  break;
	  
	case 'p':
	  isProgramPoint = true;
	  break;
	  
	default:
	  Util.processOption(ch, opt, optionUsage);
      }
    }
    
    if (entryFunctionFileName != null)
      Util.readEntryFunctions(entryFunctionFileName, entryFunctions);
    
    if (mfsaFilename == null || entryFunctions.keySet().size() == 0)
    {
      Util.stderr.println(optionUsage);
      System.exit(1);
    }
    new Check().run(mfsaFilename, cfgFilename, entryFunctions,
		    traceFilename, isProgramPoint);
  }

  /**
   * @param list False: check for state reachability.  True: Gather
   * all statements that may be executed in these states
   */
  public void run(String metaFsaFilename, String cfgFilename,
		  HashMap entryFunctions, String outputFilename,
		  boolean list)
    throws IOException
  {
    Pda pda, composedPda;
    Cfg cfg;
    Vector path;
    MetaFsa metaFsa;
    Iterator iter, iter2, iter3;
    int i, j, counter;
    ModelChecker modelChecker;
    HashSet criticalEdges, sinkNodes;
    IntHashSet unsafePoints;
    BitSet keyFields;
    // for debug purpose only
    Vector variables;
    // for debug purpose only
    Variable variable;
    TextOutput writer;
    boolean hasTrace, isOk;
    int stackSymbol;
    Integer initialStackSymbol;
    Map.Entry entry;
    String str;
    
    // read in CFG
    cfg = new Cfg();
    cfg.read(cfgFilename, true);

    // read in MetaFsa
    metaFsa = new MetaFsa();
    metaFsa.read(metaFsaFilename);

    // construct PDA from CFG
    keyFields = new BitSet();
    keyFields.set(0);
    pda = new Pda(new BitSet[] { keyFields });
    pda.read(cfg, entryFunctions, metaFsa);
    isOk = true;
    iter = entryFunctions.entrySet().iterator();
    while (iter.hasNext())
    {
      entry = (Map.Entry)iter.next();
      if (entry.getValue() == null)
      {
	Util.stderr.println("Cannot find entry function " + entry.getKey());
	isOk = false;
      }
    }
    if (!isOk)
      System.exit(1);

    // variable instantiation
    metaFsa.resolveVariable(cfg);

    // this part is for debugging purposes only
    if (Util.isDebug())
    {
      variables = metaFsa.getVariables();
      if (variables.size() == 0)
	Util.warn(Util.DEBUG, Util.INTERNAL, "No variable is found/matched.");
      else
	for (i = 0; i < variables.size(); i++)
	{
	  variable = (Variable)variables.get(i);
	  Util.warn(Util.DEBUG, Util.INTERNAL,
		    "Variable " + i + ": in " + variable.instances.size() +
			     " Ast nodes");
	  for (j = 0; j < variable.instances.size(); j++)
	    Util.warn(Util.DEBUG, Util.INTERNAL,
		      j + ": " + (Ast)variable.instances.get(j));
	}
    }
    
    criticalEdges = new HashSet();
    modelChecker = new ModelChecker();
    Util.stdout.println("Checking the program for the property:");
    if (!list)
    {
      // the user asked for error traces
      hasTrace = false;
      path = new Vector();
      counter = 1;
      writer = new TextOutput(new PrintWriter(
		 new BufferedWriter(Util.openWriter(outputFilename))));
      writer.writeString(RAW_TRACE_MAGIC_STRING);
      writer.writeToken(Constants.EOL);
      writer.writeString(metaFsaFilename);
      writer.writeToken(Constants.EOL);
      writer.writeString(metaFsa.getLabel());
      writer.writeToken(Constants.EOL);

      iter = entryFunctions.entrySet().iterator();
      while (iter.hasNext())
      {
	initialStackSymbol = (Integer)((Map.Entry)iter.next()).getValue();
	pda.setInitialStackSymbol(initialStackSymbol);
	iter2 = metaFsa.iterator();
	while (iter2.hasNext())
	{
	  // in fact, iter.next() always returns the same object
	  // but its internal data have been changed
	  metaFsa = (MetaFsa)iter2.next();
	  // Compose pda with metaFsa
	  composedPda = pda.compose(metaFsa);
	  // Model checking
	  modelChecker.init(composedPda, criticalEdges);
	  for(; modelChecker.findPath(path); counter++)
	  {
	    if (!hasTrace)
	    {
	      hasTrace = true;
	      Util.stdout.print(
 	        "The program violates the property.  Writing error trace ");
	    }
	    if (path.size() == 0)
	    {
	      Util.warn(Util.ERROR, Util.INTERNAL, "Ignore an empty trace");
	    }
	    Util.stdout.print(counter);
	    Util.stdout.print(" ");
	    Util.stdout.flush();
	    if (path.size() == 0)
	      continue;
	    writer.writeString(RAW_TRACE_SEPARATOR);
	    writer.writeToken(Constants.EOL);
	    modelChecker.writePath(path, writer);
	    path.clear();
	  }

	  /*
	    iter2 = criticalEdges.iterator();
	    while (iter2.hasNext())
	    {
	    ExplicitEdge criticalEdge = (ExplicitEdge)iter2.next();
	    Util.stdout.println(
	    criticalEdge.srcNode.state0 + " " +
	    //composedPda.getStateLabelString(criticalEdge.srcState0) + " " +
	    //composedPda.getStackLabel((Integer)criticalEdge.srcInput) +
	    (Integer)criticalEdge.srcNode.input + " " +
	    criticalEdge.srcNode.state1 + "  -> " +
	    criticalEdge.dstNode.state0 + " " +
	    //composedPda.getStateLabelString(criticalEdge.dstState0) + " " +
	    //composedPda.getStackLabel((Integer)criticalEdge.dstInput) +
	    (Integer)criticalEdge.dstNode.input + " " + 
	    criticalEdge.dstNode.state1);
	    }
	  */
	  composedPda.clear();
	  modelChecker.clear();
	  path.clear();
	}
      }
      
      if (hasTrace)
      {
	Util.stdout.println();
      }
      else
      {
	Util.stdout.println("The program satisfies the property.");
      }
      writer.writeToken(Constants.EOF);
      writer.writeToken(Constants.EOL);
      writer.close();
    }
    else
    {
      // the user asked for unsafe program points.
      unsafePoints = new IntHashSet(1024);
      iter = entryFunctions.entrySet().iterator();
      while (iter.hasNext())
      {
	initialStackSymbol = (Integer)((Map.Entry)iter.next()).getValue();
	pda.setInitialStackSymbol(initialStackSymbol);
	iter2 = metaFsa.iterator();
	while (iter2.hasNext())
	{
	  metaFsa = (MetaFsa)iter2.next();
	  // Compose pda with metaFsa
	  composedPda = pda.compose(metaFsa);
	  // Model checking
	  modelChecker.init(composedPda, criticalEdges);
	  // Get sink nodes
	  sinkNodes = modelChecker.getSinkNodes();
	
	  iter3 = sinkNodes.iterator();
	  while (iter3.hasNext())
	  {
	    stackSymbol =
	      ((Integer)((TransitionNode)iter3.next()).input).intValue();
	    if (!unsafePoints.contains(stackSymbol))
	      unsafePoints.add(stackSymbol);
	  }
	  composedPda.clear();
	  modelChecker.clear();
	}
      }

      if (unsafePoints.size() == 0)
      {
	Util.stdout.println("The program satisfies the property.");
      }
      else
      {
	Util.stdout.println("The program violates the property.");
      }
      writer = new TextOutput(new PrintWriter(
 	       new BufferedWriter(Util.openWriter(outputFilename))));
      writer.writeString(RAW_PROGRAM_POINT_MAGIC_STRING);
      writer.writeToken(Constants.EOL);
      iter = unsafePoints.iterator();
      while (iter.hasNext())
      {
	stackSymbol = ((IntHashBase.Entry)iter.next()).getKey();
	writer.writeString(pda.getStackLabel(stackSymbol));
	writer.writeToken(Constants.EOL);
      }
      unsafePoints.clear();
      writer.close();
    }
  }

  public static final String RAW_TRACE_MAGIC_STRING = "MOPSRAWTRACE";
  public static final String RAW_TRACE_SEPARATOR = "TRACE";
  public static final String RAW_PROGRAM_POINT_MAGIC_STRING = "MOPSRAWPOINT";
}


