//$Id: Fsa.java,v 1.28 2003/10/21 18:51:51 hchen Exp $

import java.io.*;
import java.util.*;

/**
 * Finite State Automaton
 */
public class Fsa
{
  /**
   * The first index is on FsaTransition.state0, if it is indexed.
   * The last index is the most details index (therefore it is the
   * most efficient for querying.
   */
  public Fsa(BitSet[] keyFieldsArray)
  {
    int i;
    BitSet keyFields;
    
    allStates = new IntHashSet(64);
    initialStates = new IntHashSet(16);
    finalStates = new IntHashSet(16);
    stateLabels = new Vector();
    explicitStateLabels = new IntHashtable(16);
    transitions = new PolyHashtable[keyFieldsArray.length];
    for (i = 0; i < transitions.length; i++)
      transitions[i] = new PolyHashtable(keyFieldsArray[i]);

    keyFields = new BitSet();
    keyFields.set(0);
    for (i = 0; i < keyFieldsArray.length; i++)
      if (keyFieldsArray[i].equals(keyFields))
      {
	state0Index = i;
	break;
      }
    if (i >= keyFieldsArray.length)
    {
      state0Index = -1;
      //Util.warn(Util.WARNING, "keyFieldsArray.length=" + keyFieldsArray.length
      //	+ "  state0Index == -1");
    }
  }

  public void clear()
  {
    int i;

    allStates.clear();
    initialStates.clear();
    finalStates.clear();
    for (i = 0; i < transitions.length; i++)
    {
      transitions[i].clear();
    }
    /*
    for (i = 0; i < stateLabels.size(); i++)
      ((StateLabel)stateLabels.get(i)).labels = null;
    */
    stateLabels.clear();
    explicitStateLabels.clear();
  }

  /**
   * Add a transition.  Do not check for duplicates.
   */
  public void addTransition(FsaTransition transition)
  {
    int i;

    for (i = 0; i < transitions.length; i++)
      transitions[i].put(transition);
    addState(transition.state0);
    addState(transition.state1);
  }

  /**
   * Retrieve a transition by value
   */
  public FsaTransition getTransition(FsaTransition t)
  {
    return (FsaTransition)transitions[getQueryIndex()].getByValue(t);
  }

  /**
   * Retrieve all the transitions that match <code>pattern</code> from
   * PolyHashtable <code>keyFieldIndex</code>. It is crucial that we
   * don't call output.clear() because the client may want to
   * accumulate output by calling this function several times.
   */
  public void getTransitions(FsaTransition pattern, int keyFieldIndex,
			     AbstractCollection output)
  {
    transitions[keyFieldIndex].getByPattern(pattern, output);
  }

  /**
   * Retrieve all the transitions.
   */
  public void getAllTransitions(AbstractCollection output)
  {
    transitions[0].getAll(output);
  }

  /**
   * Whether contains the object t
   */
  public boolean containsTransitionObject(FsaTransition t)
  {
    return transitions[getQueryIndex()].containsObject(t);
  }

  /**
   * Whether contains the value t
   */
  public boolean containsTransitionValue(FsaTransition t)
  {
    return transitions[getQueryIndex()].containsValue(t);
  }

  /**
   * Remove a transition and return it
   */
  public FsaTransition popTransition()
  {
    FsaTransition t;
    int i;

    t = (FsaTransition)transitions[0].pop();
    if (t == null)
      return null;
    for (i = 1; i < transitions.length; i++)
      if (transitions[i].removeObject(t) == null)
	Util.die(Util.INTERNAL, "An FsaTransition object is in some PolyHashtable but not in others");

    return t;
  }

  /**
   * Remove the object t
   */
  public void removeTransitionObject(FsaTransition t)
  {
    int i, count;

    count = 0;
    for (i = 0; i < transitions.length; i++)
      if (transitions[i].removeObject(t) != null)
	count++;

    if (count != 0 && count != transitions.length)
      Util.die(Util.INTERNAL, "An FsaTransition object is in some PolyHashtable but not in others");
  }

  public void addState(int state)
  {
    if (allStates.contains(state))
    {
      //Util.warn(Util.ALERT, "ignore duplicate states");
    }
    else
    {
      allStates.add(state);
      if (state >= getStateSize())
      	setStateSize(state + 1);
    }
  }

  public IntHashSet getStates()
  {
    return allStates;
  }
  
  public boolean containsState(int state)
  {
    return allStates.contains(state);
  }

  public void addInitialState(int state)
  {
    addState(state);
    if (initialStates.contains(state))
      Util.warn(Util.INFO, Util.INTERNAL, "ignore duplicate initial states");
    else
      initialStates.add(state);
  }

  public IntHashSet getInitialStates()
  {
    return initialStates;
  }

  public void addFinalState(int state)
  {
    addState(state);
    if (finalStates.contains(state))
      Util.warn(Util.INFO, Util.INTERNAL, "ignore duplicate final states");
    else
      finalStates.add(state);
  }

  public IntHashSet getFinalStates()
  {
    return finalStates;
  }

  /**
   * Compute a single state value from its components in all dimensions
   */
  public int packState(int[] states)
  {
    int i, state;

    if (states.length != stateLabels.size())
      Util.die(Util.INTERNAL, "the dimensions of states mismatch: stateLabels().size()=" +
	       stateLabels.size() + " but states.length=" + states.length);

    state = states[0];
    for (i = 1; i < stateLabels.size(); i++)
      state += states[i] * ((StateLabel)stateLabels.get(i - 1)).size;

    return state;
  }
  
  public void addStateLabel(int state, String label)
  {
    if (stateLabels.size() != 1)
      Util.die(Util.INTERNAL, "cannot add a state label unless statelabels.size() == 1");
    
    IntHashtable labels = (IntHashtable)
      ((StateLabel)stateLabels.get(0)).labels;
    if (labels.get(state) == null)
      labels.put(state, label);
    addState(state);
  }

  protected String[] getStateLabel(int state)
  {
    String[] labels = new String[stateLabels.size()];
    int i, size, index, origState;

    if (labels.length == 0)
      return labels;

    if (state >= ((StateLabel)stateLabels.get(stateLabels.size() - 1)).size)
    {
      Util.warn(Util.INFO, Util.INTERNAL, "state > maxsize");
      return labels;
    }

    origState = state;
    for (i = labels.length - 1; i > 0; i--)
    {
      size = ((StateLabel)stateLabels.get(i - 1)).size;
      index = state / size;
      state %=  size;
      labels[i] = (String)((StateLabel)stateLabels.get(i)).labels.get(index);
      if (labels[i] == null)
      {
	return new String[] {"(" + origState + ")"};
	//labels[i] = "(" + index + ")";
      }
    }
    labels[0] = (String)((StateLabel)stateLabels.get(0)).labels.get(state);
    if (labels[0] == null)
      return new String[] {"(" + state + ")"};
      //labels[0] = "(" + state + ")";
    
    return labels;
  }

  public String getStateLabelString(int state)
  {
    String[] labels = getStateLabel(state);
    StringBuffer label = new StringBuffer();
    int i;

    if (labels[0] != null)
      label.append(labels[0]);
    for (i = 1; i < labels.length; i++)
    {
      label.append(",");
      if (labels[i] != null)
	label.append(labels[i]);
    }
    return label.toString();
  }

  public final Vector getStateLabels()
  {
    return stateLabels;
  }

  public final boolean addExplicitStateLabel(int state, String label)
  {
    if (!explicitStateLabels.containsKey(state))
    {
      explicitStateLabels.put(state, label);
      return true;
    }
    else
    {
      return false;
    }
  }

  public String getExplicitStateLabel(int state)
  {
    return (String)explicitStateLabels.get(state);
  }
  
  /**
   * Add the first dimension
   */
  public void addInitialDimension()
  {
    stateLabels.add(new StateLabel(0, new IntHashtable(16)));
  }
  
  public void setStateSize(int size)
  {
    StateLabel stateLabel;
    if (stateLabels.size() == 0)
      Util.die(Util.INTERNAL, "cannot set the state size when stateLabels.size() == 0");

    stateLabel = (StateLabel)stateLabels.get(stateLabels.size() - 1);
    if (stateLabel.size > size)
      Util.warn(Util.INFO, Util.INTERNAL, "shrink state size");
    stateLabel.size = size;
  }

  public int getStateSize()
  {
    if (stateLabels.size() == 0)
      // hack for pda where there is no state.  Be aware of all implications
      return 1;
    else
      return ((StateLabel)stateLabels.get(stateLabels.size() - 1)).size;
  }

  /**
   * Creates and returns a new state
   */
  public int getNewState()
  {
    if (stateLabels.size() == 0)
    {
      Util.die(Util.INTERNAL, "Cannot get new state when stateLabels.size() == 0");
    }
    return ((StateLabel)stateLabels.get(stateLabels.size() - 1)).size++;
  }

  /**
   * Get the index to the PolyHashtable whose keyFields equals
   * <code>keyFields</code>
   */
  public int getHashtableIndex(BitSet keyFields)
  {
    int i;

    for (i = 0; i < transitions.length; i++)
      if (transitions[i].getKeyFields().equals(keyFields))
	return i;

    return -1;
  }

  /**
   * Compose two FSAs or one PDA with one FSA.
   * this may be a PDA, but fsa2 must be an FSA.
   * this is a regular fsa(from CFG), but fsa2 may be a pattern fsa
   * epsilon transitions are NOT treated specially
   * initial and final states are NOT set.  You must set them manually
   *
   * fsa2 must be indexed by state0, newFsa must be indexed by state0
   * and input
   * 
   * @param newFsa Holds the product.  Must be an empty Fsa or Pda
   */
  public void compose(Fsa fsa2, Fsa newFsa)
  {
    Fsa fsa1 = this;
    Iterator iter;
    Vector fsa1Transitions, fsa2Transitions;
    Vector 
      inputs, // a list of all ASTs
      orderedTransitions, // a list of all transitions
      isUsed;
    FsaTransition fsaTransition, fsaPattern;
    FsaTransitionWithOrder transitionWithOrder;
    StateLabel stateLabel, stateLabel2;
    Ast ast;
    int i, j, index, size, order, state;
    HashSet unmatchedTransitions;
    BitSet bitSet;

    // set up state labels.  shallow copy
    for (i = 0; i < fsa1.stateLabels.size(); i++)
    {
      stateLabel = new StateLabel();
      stateLabel2 = (StateLabel)fsa1.stateLabels.get(i);
      stateLabel.size = stateLabel2.size;
      stateLabel.labels = stateLabel2.labels;
      newFsa.stateLabels.add(stateLabel);
    }
    size = fsa1.getStateSize();
    for (i = 0; i < fsa2.stateLabels.size(); i++)
    {
      stateLabel = new StateLabel();
      stateLabel2 = (StateLabel)fsa2.stateLabels.get(i);
      stateLabel.size = stateLabel2.size * size;
      stateLabel.labels = stateLabel2.labels;
      newFsa.stateLabels.add(stateLabel);
    }

    // set up transitions
    fsa1Transitions = new Vector();
    fsa2Transitions = new Vector();
    inputs = new Vector();
    orderedTransitions = new Vector();
    isUsed = new Vector();
    fsaPattern = new FsaTransition();
    
    fsa1.getAllTransitions(fsa1Transitions);
    iter = fsa2.getStates().iterator();
    while (iter.hasNext())
    {
      state = ((IntHashBase.Entry)iter.next()).getKey();
      fsaPattern.state0 = state;
      fsa2.getTransitions(fsaPattern, fsa2.getState0Index(), fsa2Transitions);
      // fill in "inputs" with all transitions in perState.  The order of
      // the transitions are recorded in FsaTransitionWithOrder.order
      inputs.setSize(fsa2Transitions.size());
      orderedTransitions.setSize(fsa2Transitions.size());
      isUsed.setSize(fsa2Transitions.size());
      for (i = 0; i < isUsed.size(); i++)
	isUsed.set(i, null);
      for (i = 0; i < fsa2Transitions.size(); i++)
      {
	transitionWithOrder = (FsaTransitionWithOrder)fsa2Transitions.get(i);
	ast = (Ast)transitionWithOrder.input;
	/*
	  Util.stderr.println(fsaTransition2.state0 + " " +
	  fsaTransition2.state1 + " " +
	  fsaTransition2.input + " " +
	  ast.address);
	*/
	order = transitionWithOrder.order;
	if (order < 0 || order >= inputs.size())
	{
	  Util.die(Util.FILE_FORMAT, "Ast.address out of range: " + order);
	}
	else if (inputs.get(order) != null)
	{
	  Util.die(Util.FILE_FORMAT,
		   "Duplicate order of transitions from the same state");
	}
	else
	{
	  inputs.set(order, ast);
	  orderedTransitions.set(order, transitionWithOrder);
	}
      }
      fsa2Transitions.clear();

      for (i = 0; i < fsa1Transitions.size(); i++)
      {
	fsaTransition = (FsaTransition)fsa1Transitions.get(i);
	if (fsaTransition.input == null)
	  Util.die(Util.INTERNAL, "fsaTransition.input == null");
	
	if ((index = ((Ast)fsaTransition.input).match(inputs)) >= 0)
	{
	  transitionWithOrder = (FsaTransitionWithOrder)
	    orderedTransitions.get(index);
	  newFsa.addTransition(fsaTransition.expand(transitionWithOrder.state0,
			  transitionWithOrder.state1, fsa1.getStateSize()));
	  isUsed.set(index, isUsed);
	}
	else
	{
	  newFsa.addTransition(fsaTransition.expand(state, state,
						    fsa1.getStateSize()));
	}
      }

      for (i = 0; i < inputs.size(); i++)
	if (((Ast)inputs.get(i)).kind == Constants.kind_other)
	  break;
      /*
      if (i >= inputs.size())
	Util.warn(Util.WARNING, Util.FILE_FORMAT,
		  "Added missing {other} transition from the state \"" +
		  fsa2.getStateLabelString(state) + "\"");
      */
      for (i = 0; i < isUsed.size(); i++)
      {
	if (isUsed.get(i) != null)
	{
	  isUsed.set(i, null);
	}
	else
	{
	  ast = (Ast)inputs.get(i);
	  if (ast.kind != Constants.kind_other)
	    Util.warn(Util.INFO, Util.EXTERNAL,
		      "Nothing in the program matches the AST " + ast);
	}
      }
      inputs.clear();
      orderedTransitions.clear();
      isUsed.clear();
    }
  }

  /**
   * Read from .fsa file.
   * Note: initialStates and finalStates in .fsa files are inconsequential
   * initialStates and finalStates in .mfsa overrides them.
   */
  public final void read(String filename, Hashtable stateHash)
    throws IOException
  {
    TextInput reader;
    String name, label, str;
    int state0, state1, number;
    Integer integer;
    Ast ast;
    Vector list;
    int token;
    BitSet keyFields;
    FsaTransition fsaPattern;
    Hashtable astHash;
    
    clear();
    addInitialDimension();
    fsaPattern = new FsaTransition();
    list = new Vector();
    astHash = new Hashtable();
    reader = new TextInput(new BufferedReader(Util.openReader(filename)),
			   filename);
    for (token = reader.nextToken(); reader.isOk(); token = reader.nextToken())
    {
      // skip empty line
      if (token == Constants.EOL)
	continue;

      token = reader.tokenValue();
      if (!reader.isOk())
	Util.die(Util.FILE_FORMAT, "unknown syntax", filename, reader.getLineNumber());

      switch(token)
      {
	case Constants.STATE:
	  name = Util.readString(reader, "Expect a state name");
	  if (stateHash.containsKey(name))
	    Util.die(Util.FILE_FORMAT, "Duplicate state name",
		     reader.getFileName(), reader.getLineNumber());
	  label = Util.readString(reader, "Expect a state label");
	  integer = new Integer(getNewState());
	  addState(integer.intValue());
	  addStateLabel(integer.intValue(), label);
	  stateHash.put(name, integer);
	  break;

	case Constants.AST:
	  name = Util.readString(reader, "Expect an AST name");
	  if (astHash.containsKey(name))
	    Util.die(Util.FILE_FORMAT, "Duplicate ast name",
		     reader.getFileName(), reader.getLineNumber());
 	  ast = Ast.read(reader, null, null);
	  if (ast == null)
	    Util.die(Util.FILE_FORMAT, "Expect an AST",
		     filename, reader.getLineNumber());
	  astHash.put(name, ast);
	  break;
	  
	// transitions
	case Constants.TRANSITION:
	  name = Util.readString(reader, "Expect a state name");
	  if ((integer = (Integer)stateHash.get(name)) == null)
	    Util.die(Util.FILE_FORMAT, "Unknown state name " + name,
		     reader.getFileName(), reader.getLineNumber());
	  state0 = integer.intValue();
	  name = Util.readString(reader, "Expect a state name");
	  if ((integer = (Integer)stateHash.get(name)) == null)
	    Util.die(Util.FILE_FORMAT, "Unknown state name " + name,
		     reader.getFileName(), reader.getLineNumber());
	  state1 = integer.intValue();
	  name = Util.readString(reader, "Expect a transition name");
	  if ((ast = (Ast)astHash.get(name)) == null)
	    Util.die(Util.FILE_FORMAT, "Unknown AST name " + name,
		     reader.getFileName(), reader.getLineNumber());

	  // record the order of this transition
	  fsaPattern.state0 = state0;
	  getTransitions(fsaPattern, getState0Index(), list);
          addTransition(new FsaTransitionWithOrder
			(state0, ast, state1, list.size()));
	  list.clear();
	  // read eol
	  // reader.nextToken();
	  break;

	default:
	  Util.die(Util.FILE_FORMAT, "unknown syntax",
		   reader.getFileName(), reader.getLineNumber());
      }

      number = reader.skipToEOL();
      if (number != 1)
      {
	if (number == 0)
	  str = "Incomplete line";
	else
	  str = "Ignore extra words at the end of the line";
	Util.warn(Util.WARNING, Util.FILE_FORMAT, str, filename, reader.getLineNumber());
      }
    }

    reader.close();
    astHash.clear();
  }

  public void read(String filename) throws IOException
  {
    Hashtable stateHash = new Hashtable();
    read(filename, stateHash);
    stateHash.clear();
  }

  /**
   * Write FSA
   */
  public final void write(String filename) throws IOException
  {
    TextOutput writer;

    writer = new TextOutput(new PrintWriter(new BufferedWriter(
	       Util.openWriter(filename))));
    writeStates(writer, getInitialStates().iterator(), Constants.INITIALSTATE);
    writeStates(writer, getFinalStates().iterator(), Constants.FINALSTATE);
    writeTransitions(writer);
    writer.close();
  }
    
  public final void writeToDot(String filename, boolean writeLabel)
    throws IOException
  {
    PrintWriter writer;
    Iterator iter;
    FsaTransition transition, pattern;
    int state;
    Vector transitions;
    StringBuffer strBuf;
    String str;
    char ch;
    int i, j;

    strBuf = new StringBuffer();
    writer = new PrintWriter(new BufferedWriter(Util.openWriter(filename)));
    writer.println("digraph FSA {\nsize=\"8,11\";");

    pattern = new FsaTransition();
    transitions = new Vector();
    iter = getStates().iterator();
    while (iter.hasNext())
    {
      state = ((IntHashBase.Entry)iter.next()).getKey();
      writer.print(state + " [label=\""
		   + getStateLabelString(state) + "\"");
      if (initialStates.contains(state))
	writer.print(", shape=box");
      else if (finalStates.contains(state))
	writer.print(", shape=trapezium");
      writer.println("];");

      pattern.state0 = state;
      getTransitions(pattern, getState0Index(), transitions);
      for (i = 0; i < transitions.size(); i++)
      {
	transition = (FsaTransition)transitions.get(i);
	str = transition.input.toString();
	// Sanitize str to make dot happy
	for (j = 0; j < str.length(); j++)
	  if ((ch = str.charAt(j)) != '"' && ch != '\\')
	    strBuf.append(ch);
        writer.print(transition.state0 + " -> " + transition.state1);
	if (writeLabel)
	  writer.print(" [label=\"" + strBuf + "\"];");
	writer.println();
	strBuf.setLength(0);
      }
      transitions.clear();
    }
    writer.println("}");
    writer.close();
  }

  protected void writeTransitions(TextOutput writer) throws IOException
  {
    FsaTransition pattern;
    FsaTransitionWithOrder transition;
    Vector map, transitions;
    int i, index, state;
    Iterator iter;

    pattern = new FsaTransition();
    transitions = new Vector();
    map = new Vector();
    iter = getStates().iterator();
    while (iter.hasNext())
    {
      state = ((IntHashBase.Entry)iter.next()).getKey();
      pattern.state0 = state;
      getTransitions(pattern, getState0Index(), transitions);
      map.setSize(transitions.size());
      for (i = 0; i < map.size(); i++)
	map.set(i, null);
      
      for (i = 0; i < transitions.size(); i++)
      {
	transition = (FsaTransitionWithOrder)transitions.get(i);
	index = transition.order;
	if (map.get(index) == null)
	{
	  map.set(index, transition);
	}
	else
	{
	  Util.die(Util.FILE_FORMAT, "duplicate ordering in Fsa");
	}
      }

      for (i = 0; i < map.size(); i++)
      {
	transition = (FsaTransitionWithOrder)map.get(i);
        writer.writeToken(Constants.TRANSITION);
	writer.writeString(getStateLabelString(transition.state0));
	writer.writeString(getStateLabelString(transition.state1));
	writer.writeString(transition.input.toString());
      }
      map.clear();
      transitions.clear();
    }
  }    

  protected void writeStates(TextOutput writer, Iterator iter, int prefix)
    throws IOException
  {
    while (iter.hasNext())
    {
      writer.writeToken(prefix);
      writer.writeString(getStateLabelString(((IntHashBase.Entry)iter.next()).getKey()));
    }
  }
  /**
   * Decide if the FSA contains non-self-loop "other" transitions or
   * consecutive such transitions.
   *
   * @return
   * <li>
   * <ul>0: has no non-self-loop "other" transition </ul>
   * <ul>1: has no consecutive non-self-loop "other" transitions </ul>
   * <ul>2: has consecutive non-self-loop "other" transitions </ul>
   * <ul>3: has "any" transition
   */
  public int checkTransitions()
  {
    IntHashSet states;
    Iterator iter;
    Vector allTransitions;
    int i, kind;
    FsaTransition transition, fsaTransPattern;
    int state;

    states = new IntHashSet(64);
    fsaTransPattern = new FsaTransition();
    allTransitions = new Vector();
    iter = getStates().iterator();
    while (iter.hasNext())
    {
      state = ((IntHashBase.Entry)iter.next()).getKey();
      fsaTransPattern.state0 = state;
      getTransitions(fsaTransPattern, getState0Index(), allTransitions);
      for (i = 0; i < allTransitions.size(); i++)
      {
	transition = (FsaTransition)allTransitions.get(i);
	kind = ((Ast)transition.input).kind;
	if (kind == Constants.kind_other &&
	    transition.state1 != transition.state0)
	{
	  states.add(state);
	  break;
	}
	else if (kind == Constants.kind_any)
	{
	  states.clear();
	  allTransitions.clear();
	  return 3;
	}
      }
      allTransitions.clear();    
    }
    if (states.isEmpty())
    {
      return 0;
    }

    iter = states.iterator();
    while (iter.hasNext())
    {
      fsaTransPattern.state0 = ((IntHashBase.Entry)iter.next()).getKey();
      getTransitions(fsaTransPattern, getState0Index(), allTransitions);
      for (i = 0; i < allTransitions.size(); i++)
      {
	transition = (FsaTransition)allTransitions.get(i);
	kind = ((Ast)transition.input).kind;
	if (kind == Constants.kind_other &&
	    transition.state0 != transition.state1 &&
	    states.contains(transition.state1))
	{
	  states.clear();
	  allTransitions.clear();
	  // Util.stderr.println(getStateLabelString(transition.state1));
	  return 2;
	}
      }
    }

    states.clear();
    allTransitions.clear();
    return 1;
  }

  /**
   * Which PolyHashtable is indexed on state0
   */
  public final int getState0Index()
  {
    return state0Index;
  }

  /**
   * Which PolyHashtable should be used for querying
   */
  public final int getQueryIndex()
  {
    return transitions.length - 1;
  }
  
  protected IntHashSet
    allStates, // all the states
    initialStates, // initial states
    finalStates; // final states

  /**
   * A list of PolyHashtable objects sorted by how specific their
   * keyFields are.
   * Important:
   * <ul>
   * <li>The first element is indexed on state0. </li>
   * <li>The last element has the most specific keyFields.</li>
   * </ul>
   */
  protected PolyHashtable[] transitions;
  
  /**
   * Records size and labels of each dimension of the state space
   * 0th element in stateLabels corresponds to LSB of state value
   * Note: state added by newState() cannot have labels
   */
  protected Vector stateLabels;

  /**
   * Stores explicit labels of states
   */
  protected IntHashtable explicitStateLabels;
    
  protected int state0Index;
}
