// $Id: IntHashBase.java,v 1.3 2003/09/22 21:31:56 hchen Exp $

import java.util.*;

public class IntHashBase
{
  public IntHashBase(int initCapacity, float loadFactor)
  {
    init(initCapacity, loadFactor);
  }
  
  public IntHashBase(int initCapacity)
  {
    init(initCapacity, (float)0.75); 
  }
  
  public final void clear()
  {
    numKeys = 0;
    Arrays.fill(buckets, null);
  }

  public final int size()
  {
    return numKeys;
  }
  
  public final boolean isEmpty()
  {
    return numKeys == 0;
  }
  
  public final Iterator iterator()
  {
    return new IntHashIterator();
  }
  
  public static class Entry
  {
    public Entry(int key)
    {
      this.key = key;
      next = null;
    }

    public final int getKey()
    {
      return key;
    }

    private int key;
    private Entry next;
  }
 
  private void init(int initCapacity, float loadFactor)
  {
    numKeys = 0;
    this.initCapacity = initCapacity;
    this.loadFactor = loadFactor;
    threshold = (int)(initCapacity * loadFactor);
    // temporary variable for holding a return value from getEntry();
    entryVarTemp = new EntryVar();
    buckets = new Entry[initCapacity];
    Arrays.fill(buckets, null);
  }

  protected final boolean containsEntry(int key)
  {
    return getEntry(key) != null;
  }
  
  protected final Entry getEntry(int key)
  {
    int index;

    index = getBucketIndex(key);
    return getEntry(index, key, null);
  }
  
  protected final boolean putEntry(Entry entry)
  {
    int index;

    index = getBucketIndex(entry.key);
    if (getEntry(index, entry.key, null) != null)
    {
      return false;
    }
    else
    {
      entry.next = buckets[index];
      buckets[index] = entry;
      numKeys++;
      if (numKeys > threshold)
      {
	rehash();
      }
      return true;
    }
  }
  
  protected final boolean removeEntry(int key)
  {
    int index;
    Entry entry, prev;

    index = getBucketIndex(key);
    entry = getEntry(index, key, entryVarTemp);
    if (entry == null)
    {
      return false;
    }
    else
    {
      prev = entryVarTemp.data;
      if (prev != null)
      {
	prev.next = entry.next;
      }
      else
      {
	buckets[index] = entry.next;
      }
      entry.next = null;
      numKeys--;
      return true;
    }
  }

  protected class IntHashIterator implements Iterator
  {
    public IntHashIterator()
    {
      bucket = -1;
      entry = null;
      probed = false;
      probeNext();
    }
	
    public final boolean hasNext()
    {
      probeNext();
      return entry != null;
    }

    public Object next()
    {
      probeNext();
      probed = false;
      return entry;
    }

    public final void remove()
    {
    }

    protected void probeNext()
    {
      if (probed)
	return;
      probed = true;
      if (entry != null)
      {
	entry = entry.next;
	if (entry != null)
	  return;
      }
      for (bucket++; bucket < buckets.length; bucket++)
      {
	if (buckets[bucket] != null)
	{
	  entry = buckets[bucket];
	  return;
	}
      }
      // entry == null
    }

    int bucket;
    Entry entry;
    boolean probed;
  }
  
  private final int getBucketIndex(int key)
  {
    return Util.remainder(key, buckets.length);
  }

  private final Entry getEntry(int bucketIndex, int key,
				   EntryVar prevVar)
  {
    Entry entry, prev;

    for (entry = buckets[bucketIndex], prev = null;
	 entry != null;
	 prev = entry, entry = entry.next)
    {
      if (entry.key == key)
	break;
    }
    if (prevVar != null)
    {
      prevVar.data = prev;
    }
    return entry;
  }

  private final void rehash()
  {
    Entry[] oldBuckets;
    Entry entry, next;
    int i;
      
    oldBuckets = buckets;    
    buckets = new Entry[oldBuckets.length * 2];
    Arrays.fill(buckets, null);
    numKeys = 0;
    threshold = (int)(buckets.length * loadFactor);
    for (i = 0; i < oldBuckets.length; i++)
    {
      for (entry = oldBuckets[i]; entry != null; entry = next)
      {
	next = entry.next;
	if (!putEntry(entry))
	  Util.warn(Util.ERROR, Util.INTERNAL, "FIX ME");
      }
      oldBuckets[i] = null;
    }
  }
  
  private static class EntryVar
  {
    Entry data;
  };

  Entry[] buckets;
  int numKeys, initCapacity, threshold;
  float loadFactor;
  EntryVar entryVarTemp;
}
