// $Id: Pda.java,v 1.35 2003/09/22 21:35:48 hchen Exp $

import java.io.*;
import java.util.*;

/**
 * Pushdown Automaton
 */
public class Pda extends Fsa
{ 
  public Pda(BitSet[] keyFieldsArray)
  {
    super(keyFieldsArray);
    stackLabels = new IntHashtable(1024);
    initialStackSymbols = new Vector();
  }

  public Pda(BitSet[] keyFieldsArray, IntHashtable stackLabels,
	     Vector initialStackSymbols)
  {
    super(keyFieldsArray);
    this.stackLabels = stackLabels;
    this.initialStackSymbols = initialStackSymbols;
  }
  
  public void clear()
  {
    super.clear();
    // WARNING: do not clear stackLabels and initialStackSymbols
    // because many PDAs share them
    // stackLabels.clear();
    // initialStackSymbols.clear();
  }
  
  public void addStackLabel(int symbol, String label)
  {
    if (symbol == PdaTransition.EPSILON)
      return;
    if (stackLabels.get(symbol) == null)
      stackLabels.put(symbol, label);
  }
 
  public IntHashtable getStackLabels()
  {
    return stackLabels;
  }

  protected String getStackLabel(int symbol)
  {
    String label;

    if ((label = (String)stackLabels.get(symbol)) == null)
      return "(" + symbol + ")";
    else
      return label;
  }

  public void addInitialStackSymbol(Integer symbol)
  {
    initialStackSymbols.add(symbol);
  }

  public void setInitialStackSymbol(Integer symbol)
  {
    initialStackSymbols.clear();
    initialStackSymbols.add(symbol);
  }

  public Vector getInitialStackSymbols()
  {
    return initialStackSymbols;
  }

  /**
   * Overrides Fsa.compose()
   * Compose this with an FSA
   *
   * @param fsa2 Must be a FSA
   * @param newFsa The composed PDA.  It is declared as Fsa for polymorphism
   */
  public void compose(Fsa fsa2, Fsa newFsa)
  {
    if (!(newFsa instanceof Pda))
      Util.die(Util.INTERNAL, "newFsa must be a Pda");
    super.compose(fsa2, newFsa);
  }

  /**
   * compose this with all FSAs in a MetaFsa
   */
  public Pda compose(MetaFsa metaFsa) throws IOException
  {
    Vector fsas, states;
    Pda oldPda, newPda;
    int i, state;
    BitSet[] keyFieldsArray;
    BitSet keyFields;
    StateWithLabel stateWithLabel;
    
    keyFieldsArray = new BitSet[1];
    fsas = metaFsa.getFsas();
    oldPda = this;
    newPda = null;
    for (i = 0; i < fsas.size(); i++)
    {
      keyFields = new BitSet();
      keyFields.set(0);  // PdaTransition.state0
      keyFields.set(3);  // PdaTransition.stack0
      keyFieldsArray[0] = keyFields;
      newPda = new Pda(keyFieldsArray, stackLabels, initialStackSymbols);
      oldPda.compose((Fsa)fsas.get(i), newPda);
      if (i > 0)
      {
	// destroy intermediate PDAs
	oldPda.clear();
      }
      oldPda = newPda;
    }
    // copy explicit state labels
    states = metaFsa.getInitialStates();
    for (i = 0; i < states.size(); i++)
    {
      stateWithLabel = (StateWithLabel)states.get(i);
      state = newPda.packState(stateWithLabel.states);
      newPda.addInitialState(state);
      newPda.addExplicitStateLabel(state, stateWithLabel.label);
    }
    states = metaFsa.getFinalStates();
    for (i = 0; i < states.size(); i++)
    {
      stateWithLabel = (StateWithLabel)states.get(i);
      state = newPda.packState(stateWithLabel.states);
      newPda.addFinalState(state);
      newPda.addExplicitStateLabel(state, stateWithLabel.label);
    }
    return newPda;
  }

  /**
   * Read in a CFG, from which construct a PDA
   */
  public void read(Cfg cfg, HashMap entryFunctions, MetaFsa metaFsa)
  {
    Hashtable functionHash;
    int counter, symbolNumber, i, j, index;
    Vector nodeQueue, functions, outEdges, list;
    CfgFunction function, function2;
    Node node, node2;
    Edge edge;
    String label;
    Ast ast, unmatchedAst;
    HashSet functionCalls;
    Vector fsas;
    Fsa fsa;
    //boolean isEntry;

    clear();

    functionCalls = new HashSet();
    list = new Vector();
    fsas = metaFsa.getFsas();
    for (i = 0; i < fsas.size(); i++)
    {
      fsa = (Fsa)fsas.get(i);
      fsa.getAllTransitions(list);
      for (j = 0; j < list.size(); j++)
      {
	ast = (Ast)((FsaTransition)list.get(j)).input;
	if ((label = ast.getFunctionCallName()) != null)
	{
	  functionCalls.add(label);
	}
      }
      list.clear();
    }

    // Number cfg nodes
    nodeQueue = new Vector();
    functionHash = new Hashtable();
    functions = cfg.getFunctions();
    counter = 0;
    Node.initDone();
    for (i = 0; i < functions.size(); i++)
    {
      function = (CfgFunction)functions.get(i);
      if (function.entry == null)
	Util.warn(Util.INFO, Util.EXTERNAL,
		  "No entry node in the function " + function.label);
      else
      {
	functionHash.put(function.label, function);
	if (function.exit == null)
	  Util.warn(Util.INFO, Util.EXTERNAL,
		    "No exit node in the function " + function.label);
	// number nodes
	nodeQueue.add(function.entry);
	//function.entry.setVisited(true);
	function.entry.setDone();
	while (nodeQueue.size() > 0)
	{
	  node = (Node)nodeQueue.remove(nodeQueue.size() - 1);
	  node.setSN(++counter);
	  outEdges = node.getOutEdges();
	  for (j = 0; j < outEdges.size(); j++)
	  {
	    node2 = (Node)((Edge)outEdges.get(j)).getDstNode();
	    //if (!node2.getVisited())
	    if (!node2.getDone())
	    {
	      nodeQueue.add(node2);
	      //node2.setVisited(true);
	      node2.setDone();
	    }
	  }
	}

	label = Ast.getUnmangledFunctionName(function.label);
	if (entryFunctions.containsKey(label))
	{
	  if (entryFunctions.get(label) == null)
	  {
	    entryFunctions.put(label, new Integer(function.entry.getSN()));
	  }
	  else
	  {
	    Util.warn(Util.WARNING, Util.FILE_FORMAT, "ignore duplicate entry function "
		      + function.label + " (was " + label + ")");
	  }
	}
      }
    }

    unmatchedAst = new Ast();
    unmatchedAst.setKind(Constants.kind_unmatched);
    Node.initDone();
    // create Pda transitions from CFG
    for (i = 0; i < functions.size(); i++)
    {
      function = (CfgFunction)functions.get(i);
      nodeQueue.add(function.entry);
      //function.entry.setVisited(false);
      function.entry.setDone();
      while (nodeQueue.size() > 0)
      {
	node = (Node)nodeQueue.remove(nodeQueue.size() - 1);
	addStackLabel(node.getSN(), node.getLabelString(function));
	outEdges = node.getOutEdges();
	for (j = 0; j < outEdges.size(); j++)
	{
	  edge = (Edge)outEdges.get(j);
	  node2 = (Node)edge.getDstNode();
	  ast = edge.getLabel();
	  //if (node2.getVisited())
	  if (!node2.getDone())
	  {
	    //node2.setVisited(false);
	    nodeQueue.add(node2);
	    node2.setDone();
	  }

          if (ast == null || (label = ast.getFunctionCallName()) == null ||
	      (function2 = (CfgFunction)functionHash.get(label)) == null ||
	      functionCalls.contains(Ast.getUnmangledFunctionName(label)))
	  // Do not trace into a function call if
	    // 1. it is not defined;
	    // 2. it is listed in an FSA
	  {
	    // Note the delicacy here.  If the type of an AST node is
	    // unmatched, it will match the "other" symbol in compose().
	    if (ast == null)
	    {
	      ast = unmatchedAst;
	    }
	    addTransition(new PdaTransition(PdaTransition.EPSILON,
 	      ast, PdaTransition.EPSILON, node.getSN(), node2.getSN(),
	      PdaTransition.EPSILON));
	  }
	  else
	  // Trace into function call
	  {
	    // use ast instead of EPSILON to be able to backtrack
	    // in function call
	      addTransition(new PdaTransition(PdaTransition.EPSILON,
	        ast, PdaTransition.EPSILON, node.getSN(),
	        function2.entry.getSN(), node2.getSN()));
	      //}
	  }
	  // if node2 is exit node
	  if (node2 == function.exit)
	    addTransition(new PdaTransition(PdaTransition.EPSILON,
	      unmatchedAst, PdaTransition.EPSILON, node2.getSN(),
	      PdaTransition.EPSILON, PdaTransition.EPSILON));
	}
      }
    }

    /*
    // add initialStackSymbols
    if (entryFunction == null)
      Util.die(Util.FILE_FORMAT, "cannot find the entry function " + entryFunctionName);
    else
      initialStackSymbols.add(new Integer(entryFunction.entry.getSN()));
    */
    
    functionHash.clear();
    functionCalls.clear();
  }

  /*
  public void print()
  {
    Iterator iter; 
    String label;
    Integer key;
    
    super.print();

    // print stack symbols
    iter = getStackLabels().keySet().iterator();
    Util.stdout.println("Stack Symbols");
    while (iter.hasNext())
      Util.stdout.print(getStackLabel((Integer)iter.next()) + "  ");
    Util.stdout.println();
  }

  public void writeTransitions(PrintWriter writer)
  {
    Iterator iter;
    PdaTransition transition;

    while (iter.hasNext())
    {
      transition = (PdaTransition)iter.next();
      writer.println("t " + getStateLabelString(transition.state0) + " " +
		     getStackLabel(transition.stack0) + " " +
		     getStateLabelString(transition.state1) + " " +
		     getStackLabel(transition.stack1) + " " +
		     getStackLabel(transition.stack2) + " " +
  		     transition.input);
    }
  }
  */

  protected IntHashtable stackLabels;

  /**
   * Each element is an Integer, an index to stackLabels
   */
  protected Vector initialStackSymbols;
}

