// $Id: PolyHashtable.java,v 1.3 2003/09/22 21:31:57 hchen Exp $
import java.io.*;
import java.util.*;

/* A hashtable that stores records.  The hashcode of each record is
 * computed from a fixed set of its fields, the key fields.
 * Multiple records may have the same hashcode.
 * Each record must define equals(Record) or equals(Object) properly.
 */
class PolyHashtable
{
  public PolyHashtable(BitSet keyFields)
  {
    this.keyFields = (BitSet)keyFields.clone();
    numRecords = 0;

    initCapacity = 65;
    initBucketCapacity = 4;
    loadFactor = 0.75;

    buckets = new Vector[initCapacity];
    numEmptyBuckets = initCapacity;
    minEmptyBuckets = (int)(initCapacity * (1 - loadFactor));
  }

  /**
   * Insert a record.  Do not check for duplicate.
   */
  public void put(Record record)
  {
    int index;
    Vector bucket;
    
    index = getBucketIndex(record);
    bucket = buckets[index];
    if (bucket == null)
    {
      bucket = new Vector(initBucketCapacity);
      buckets[index] = bucket;
      numEmptyBuckets--;
    }
    bucket.add(record);
    numRecords++;
    if (numEmptyBuckets < minEmptyBuckets)
      rehash();
  }

  /**
   * Retrieve a record that equals this record, decided by
   * record.equals(Record)
   */
  public Record getByValue(Record record)
  {
    int index1, index2;

    index1 = getBucketIndex(record);
    index2 = getRecordIndexByValue(index1, record, true);
    if (index2 >= 0)
      return (Record)buckets[index1].get(index2);
    else
      return null;
  }
  
  /**
   * Get all the records that match <code>pattern</code> masked by
   * <code>keyFields</code>.  It is crucial that this function doesn't
   * call output.clear() because the caller may want to accumulate
   * output by calling this function several times.
   */
  public void getByPattern(Record pattern, AbstractCollection output)
  {
    Vector bucket;
    int i;
    Record record;
    
    bucket = buckets[getBucketIndex(pattern)];
    if (bucket != null)
    {
      for (i = 0; i < bucket.size(); i++)
      {
	record = (Record)bucket.get(i);
	if (record.matches(pattern, keyFields))
	{
	  output.add(record);
	}
      }
    }
  }

  /**
   * Get all the records.
   */
  public void getAll(AbstractCollection output)
  {
    int i, j;
    Vector bucket;

    //output.setSize(numRecords);
    for (i = 0; i < buckets.length; i++)
    {
      bucket = buckets[i];
      if (bucket != null)
      {
	for (j = 0; j < bucket.size(); j++)
	  output.add((Record)bucket.get(j));
      }
    }
  }
  
  /**
   * Remove a record.  Return the removed object.
   */
  public Record removeObject(Record record)
  {
    int i, j;
    Vector bucket;

    i = getBucketIndex(record);
    bucket = buckets[i];
    if (bucket != null)
    {
      j = getRecordIndexByObject(i, record);
      if (j >= 0)
      {
	return remove(i, j);
      }
      else
      {
	return null;
      }
    }
    return null;
  }

  /**
   * Remove and return an arbitrary record
   */
  public Record pop()
  {
    int i;
    
    for (i = 0; i < buckets.length; i++)
      if (buckets[i] != null && buckets[i].size() > 0)
	break;
    if (i >= buckets.length)
      return null;
    else
      return remove(i, 0);
  }

  /*
   * Whether contains the object <code>record</code>, decided by the
   * == operator
   */
  public boolean containsObject(Record record)
  {
    return getRecordIndexByObject(getBucketIndex(record), record) >= 0;
  }

  /*
   * Whether contains an object whose value equals
   * <code>record</code>, decided by record.equals(Record).
   */
  public boolean containsValue(Record record)
  {
    return getRecordIndexByValue(getBucketIndex(record), record, true) >= 0;
  }

  /*
   * Whther contains an object that matches <code>pattern</code>, decided by
   * record.matches(Record)
   */
  public boolean containsPattern(Record pattern)
  {
    return getRecordIndexByValue(getBucketIndex(pattern), pattern, false) >= 0;
  }
  
  public BitSet getKeyFields()
  {
    return keyFields;
  }

  public int getSize()
  {
    return numRecords;
  }
  
  public void clear()
  {
    int i;

    for (i = 0; i < buckets.length; i++)
      if (buckets[i] != null)
	buckets[i].setSize(0);
    numEmptyBuckets = buckets.length;
    minEmptyBuckets = (int)(numEmptyBuckets * (1 - loadFactor));
    numRecords = 0;
  }
  
  int getBucketIndex(Record record)
  {
    int hashCode;

    hashCode = record.hashCode(keyFields);
    return Util.remainder(hashCode, buckets.length);
  }

  int getRecordIndexByObject(int bucketIndex, Record record)
  {
    Vector bucket;
    int i;
    
    bucket = buckets[bucketIndex];
    if (bucket != null)
    {
      for (i = 0; i < bucket.size(); i++)
      {
	if ((Record)bucket.get(i) == record)
	{
	  return i;
	}
      }
    }
    return -1;
  }

  int getRecordIndexByValue(int bucketIndex, Record record,
			    boolean allFields)
  {
    Vector bucket;
    Record record2;
    int i;
    
    bucket = buckets[bucketIndex];
    if (bucket != null)
    {
      for (i = 0; i < bucket.size(); i++)
      {
	record2 = (Record)bucket.get(i);
	if ((allFields && record2.equals(record)) ||
	    (!allFields && record2.matches(record, keyFields)))
	{
	  return i;
	}
      }
    }
    return -1;
  }
  
  /**
   * Pre-condition: i and j are valid indices.
   */
  Record remove(int i, int j)
  {
    Vector bucket = buckets[i];
    Record record = (Record)bucket.get(j);
    
    bucket.set(j, bucket.get(bucket.size() - 1));
    bucket.removeElementAt(bucket.size() - 1);
    numRecords--;
    return record;
  }
  
  void rehash()
  {
    int newCapacity;
    Vector[] oldBuckets = buckets;
    Vector bucket;
    int i, j;
    Record record;

    newCapacity = buckets.length * 2 + 1;
    buckets = new Vector[newCapacity];
    numEmptyBuckets = newCapacity;
    minEmptyBuckets = (int)(newCapacity * (1 - loadFactor));
    numRecords = 0;
    
    for (i = 0; i < oldBuckets.length; i++)
    {
      bucket = oldBuckets[i];
      if (bucket != null)
      {
	for (j = 0; j < bucket.size(); j++)
	{
	  record = (Record)bucket.get(j);
	  put(record);
	}
	bucket.clear();
	oldBuckets[i] = null;
      }
    }
  }

  /*
  void print()
  {
    int i, j;
    Vector bucket;

    Util.stdout.println("numRecords=" + numRecords + "  numEmptyBuckets=" +
		       numEmptyBuckets);
    for (i = 0; i < buckets.length; i++)
    {
      bucket = buckets[i];
      if (bucket != null)
      {
	Util.stdout.println(i + ": ");
	for (j = 0; j < bucket.size(); j++)
	{
	  Util.stdout.println(((Record)bucket.get(j)).toString());
	}
      }
    }
  }
  */
  
  Vector[] buckets;
  int numEmptyBuckets, minEmptyBuckets, numRecords;
  // semi-constants
  int initCapacity, initBucketCapacity;
  double loadFactor;
  BitSet keyFields;
}
