/* ====Descriptors.H=============================================================================
 *
 * Author: Patrice Koehl, May 2020
 * Department of Computer Science
 * University of California, Davis
 *
 * Computes 3D descriptors at each vertex of a mesh. Considered here:
 *      LD-SIFT descriptors 
 *      Spin Image descriptors
 *
 * based on the paper:
 *
 *      Tal Darom and Yosi Keller, “Scale Invariant Features for 3D Mesh Models”. 
 *      IEEE Transactions on Image Processing 21(5): 2758-2769 (2012)

 =============================================================================================== */

#ifndef _DESCRIPTORS_H_
#define _DESCRIPTORS_H_

  #define _USE_MATH_DEFINES // for M_PI

  /* =============================================================================================
     System and local includes
   =============================================================================================== */

  #include <vector>
  #include <cmath>
  #include "Adjacency.h"
  #include "renderpatch.h"
  #include "sift.h"
  #include "Zernike.h"

  Adjacency table;
  RenderPatch render;
  SIFT sift;
  Zernike zer;

  extern "C" {

        // BLAS2: perform Y := alpha*op( A )* B  + beta*Y
        void dgemv_(char * trans, int * m, int * n, double * alpha, double *A,
                int *lda, double * X, int * incx, double *beta, double * Y, int * incy);

	// BLAS3: perform C := alpha*op( A )* op(B)  + beta*C
	void dgemm_(char * transa, char * transb, int * m, int * n, int * k,
		double * alpha, double * A, int * lda,
		double * B, int * ldb, double * beta, double * C, int * ldc);

	// Lapack: SVD
	void dgesvd_(char *JOBU, char *JOBV, int *M, int *N, double *A, int *LDA,
	double *S, double *U, int *LDU, double *VT, int *LDVT, double *WORK, int *LWORK, int *info);

  }

  /* =============================================================================================
     Define class
   =============================================================================================== */

  class Descriptors {

  public:
	// generate shape descriptors at each vertex of a mesh
	void genDescriptors(Mesh& mesh, int desc_type, int flag_pts, std::vector<int>& keypoints,
		std::vector<std::vector<double> >& descriptors);

  private:
 
	// Difference of Gaussian
	void dog(Mesh& mesh, int n_octaves, int flag_scale, int flag_boundary,
		std::vector<int>& keypoints, std::vector<int>& octaves, double *scale);

	// SIFT descriptors
	void ldsift(Mesh& mesh, std::vector<int>& keypoints, std::vector<int>& octaves, double *scale,
		std::vector<std::vector<double> >& descriptors);

	// All SISI descriptors
	void sisi(Mesh& mesh, std::vector<int>& keypoints, std::vector<int>& octaves, double *scale,
		std::vector<std::vector<double> >& descriptors);

	// Zernike descriptors
	void zernike(Mesh& mesh, std::vector<int>& keypoints, double *scale,
		std::vector<std::vector<double> >& descriptors);

	// one SISI descriptors
	void compute_sisi_descriptors(int nvertices, double *vertex, double *normals , 
		int kpt, double scale, int imgsize, double *img, std::vector<double>& descriptors);

	// Compute weights of all edges in the mesh
	void computeMeshWeight(Mesh& mesh, int type, double *Weights);

	// find keypoints at each octave
	void find_local_maxima(int n, int n_octaves, int flag_boundary, int *boundary,
		int *ind_oct, double *diff, std::vector<int>& keypoints, std::vector<int>& octaves);

	// compute local density around each vertex
	void meshScale(Mesh& mesh, double *scale);

	// Create view matrix
	void create_viewmatrix(double *eyevec, double *center, double *up, double *viewmatrix);

	// Compute vertex normals
	void computeNormals(Mesh& mesh, double *normal);

	// Compute largest, and second largest elements in an array
	void findLarge(int npixels, double *img, double *largest, double *second);

	// Compute gradients of an image
	void imgGradients(int nx, int ny, double *img, double *Gx, double *Gy);

	// rotate a matrix by 180 degrees
	void rotate180(int nx, int ny, double *mat);

   protected:

	std::vector<std::set<int> > adj;

	std::vector<int> keypoints;
	std::vector<int> octaves;

  };

  /* ================================================================================================
   Gen local descriptors
   ==================================================================================================*/

  void Descriptors::genDescriptors(Mesh& mesh, int desc_type, int flag_pts, std::vector<int>& keypoints,
		std::vector<std::vector<double> >& descriptors)

   {

  /* ================================================================================================
   	Find keypoints
   ==================================================================================================*/

	int n_vertices = mesh.vertices.size();
	int n_octaves = 20;
	int flag_scale = 0;
	int flag_boundary = 1;

	double * scale = new double[n_vertices];

	if(flag_pts == 1) {

		keypoints.clear();
		octaves.clear();
	
		dog(mesh, n_octaves, flag_scale, flag_boundary, keypoints, octaves, scale);

	} else {

		int nkpts = keypoints.size();
		for(int i = 0; i < nkpts; i++) {
			octaves.push_back(1);
		}

		meshScale(mesh, scale);
	}

/*
	std::cout << "Number of keypoints:"  << keypoints.size() << std::endl;
	for(int i = 0; i < keypoints.size(); i++) {
		std::cout << keypoints[i] << " " << octaves[i] << std::endl;
	}
	std::cout << std::endl;
*/

  /* ================================================================================================
   	Compute descriptors
   ==================================================================================================*/

	descriptors.clear();
	if(desc_type==0) {
		ldsift(mesh, keypoints, octaves, scale, descriptors);
	} else if(desc_type == 1) {
		sisi(mesh, keypoints, octaves, scale, descriptors);
	} else if(desc_type == 2) {
		zernike(mesh, keypoints, scale, descriptors);
	}

	delete [] scale;
   }

  /* ===== Mesh DoG                 =================================================================
	difference of gaussians features
   ==================================================================================================*/

  void Descriptors::dog(Mesh& mesh, int n_octaves, int flag_scale, int flag_boundary,
		std::vector<int>& keypoints, std::vector<int>& octaves, double *scale)
  {

  /* ================================================================================================
   Set of local arrays
   ==================================================================================================*/

	int n = mesh.vertices.size();
	double *Weights = new double[n*n];
	double *ones = new double[n];
	double *sumr = new double[n];
	double *vertex = new double[3*n];
	double *vertex2 = new double[3*n];
	double *diff = new double[n*n_octaves];
	int *boundary = new int[n];
	int *ind_oct = new int[n];

  /* ================================================================================================
   Define convolution operator
   ==================================================================================================*/

	int type = 0;
	computeMeshWeight(mesh, type, Weights);

	for(int i = 0; i < n; i++) { Weights[i+i*n] = 1.0;}

	for(int i = 0; i < n; i++) { ones[i] = 1.0;}

	double alpha, beta;
	alpha = 1.0; beta = 0.0;
	int inc = 1;
	char NoTrans = 'N';

	dgemv_(&NoTrans, &n, &n, &alpha, Weights, &n, ones, &inc, &beta, sumr, &inc);

	for(int j = 0; j < n; j++) {
		for(int i = 0; i < n; i++) {
			Weights[i+j*n] /= sumr[i];
		}
	}


  /* ================================================================================================
   Apply convolution operator on all vertices over all octaves, and compute diff vectors
   ==================================================================================================*/

	int idx;
	Vector p;
	for (VertexIter v = mesh.vertices.begin(); v != mesh.vertices.end(); v++) {
		idx = v->index;
		p = v->position;
		for(int j = 0; j < 3; j++) {
			vertex[idx+j*n] = p[j];
		}
		boundary[idx] = 0;
		if(v->onBoundary()) boundary[idx] = 1;
	}

	int three = 3;
	double sum, d;
	for(int ind = 0; ind < n_octaves; ind++) 
	{
        	dgemm_(&NoTrans, &NoTrans, &n, &three, &n, &alpha, Weights, &n, vertex, &n, 
		&beta, vertex2, &n);

		sum = 0;
		for(int i = 0; i < n; i++) {
			double dx = vertex[i] - vertex2[i];
			double dy = vertex[i+n] - vertex2[i+n];
			double dz = vertex[i+2*n] - vertex2[i+2*n];
			d = dx*dx + dy*dy + dz*dz;
			sum += d;
			diff[i+n*ind] = d;
		}

		for(int i = 0; i < n; i++) {
			if(flag_scale == 1) {
				diff[i+n*ind] *= ind;
			} else {
				diff[i+n*ind] /= sum;
			}
		}

		for(int i = 0; i < 3*n; i++) {vertex[i] = vertex2[i];}
	}

  /* ================================================================================================
   Find highest value over all octaves, for all vertices
   ==================================================================================================*/


	for(int i = 0; i < n; i++) {

		idx = 0;
		d=diff[i];
		for(int ind = 1; ind < n_octaves; ind++) {
			if(diff[i+n*ind] > d) {
				d = diff[i+n*ind];
				idx = ind;
			}
		}
		ind_oct[i] = idx;
	}
		
  /* ================================================================================================
   Find local maxima == keypoints
   ==================================================================================================*/

	table.construct(mesh, adj);
	find_local_maxima(n, n_octaves, flag_boundary, boundary, ind_oct, diff, keypoints, octaves);

  /* ================================================================================================
   Compute local density == scale of each vertex
   ==================================================================================================*/

	meshScale(mesh, scale);

  /* ================================================================================================
   Clean up
   ==================================================================================================*/

	delete [] vertex; delete [] vertex2; delete [] diff; delete [] ind_oct;
	delete [] Weights; delete [] sumr; delete [] ones; delete [] boundary;
  }

  /* ================================================================================================
	Compute LD-SIFT descriptors

	Note that:
		vertex, faces, and normals are nx3 vectors
   ==================================================================================================*/
	
  void Descriptors::ldsift(Mesh& mesh, std::vector<int>& keypoints, std::vector<int>& octaves, 
		double *scale, std::vector<std::vector<double> >& descriptors)
{
  /* ================================================================================================
	Local parameters and arrays
   ==================================================================================================*/

	int nvertices = mesh.vertices.size();
	int nfaces = mesh.faces.size();

	int pix = 21;
	int npixels = pix*pix;
	double scale_fact = 3.0;
	double Large = 1e15;
	int ndesc = 128;

	double *vertex = new double[3*nvertices];
	double *normal = new double[3*nvertices];
	double *faces = new double[3*nfaces];
	double *vv = new double[3*nvertices];

	std::vector<int> neighbours;

	double center[3];
	double eyevec[3];
	double cent[3];
	double viewmatrix[16];
	double proj_matrix[16];
	double viewport[4];

	double *I = new double[6*npixels];
	double *Inew = new double[6*npixels];
	double *img = new double[npixels];
	double *Gx = new double[npixels];
	double *Gy = new double[npixels];
	float *grad = new float[2*npixels];

	std::vector<double> desc;
	desc.resize(ndesc);

	int *img_size = new int[3];
	img_size[0] = pix; img_size[1] = pix; img_size[2] = 6;

	double *U = new double[9];
	double *VT = new double[3*nvertices];
	double *S  = new double[3];
	double *Work;

	double support;
	double largest, second;
	double x, y, sigma, angle0;

  /* ================================================================================================
	Transfer mesh info into local structures
	Note: add +1 to vertex indices for faces as renderpatch used numbering starting at 1

	Compute normal, ensuring that they are outward
   ==================================================================================================*/

	int idx;
	for (VertexIter v = mesh.vertices.begin(); v != mesh.vertices.end(); v++) {
		idx = v->index;
		for(int j = 0; j < 3; j++) {
			vertex[idx + j*nvertices] = v->position[j];
		}
	}

	int idxA, idxB, idxC;
	HalfEdgeIter hAB, hBC, hCA;
	VertexIter vA, vB, vC;
	idx = 0;
	for(FaceIter f_iter = mesh.faces.begin(); f_iter != mesh.faces.end(); f_iter++)
	{
		hAB =f_iter->he; hBC =hAB->next; hCA =hBC->next;

		vA = hAB->vertex; vB = hBC->vertex; vC = hCA->vertex;

		idxA = vA->index + 1; idxB = vB->index + 1; idxC = vC->index + 1;

		faces[idx ] = (double) idxB;
		faces[idx+ nfaces] = (double) idxC;
		faces[idx+ 2*nfaces] = (double) idxA;
		idx++;
	}

	computeNormals(mesh, normal);

  /* ================================================================================================
	Loop over all keypoints
   ==================================================================================================*/

	int point;
	double v, dist;
	double s2, s;

	memset(proj_matrix, 0, 16*sizeof(double));
	for(int i = 0; i < 4; i++) proj_matrix[i+4*i]=1.0;

	int ksize = keypoints.size();
	for(int kpt = 0; kpt < ksize; kpt++)
	{
		point = keypoints[kpt];
		s = scale[point]*scale_fact*std::sqrt(octaves[kpt]+1);
		s2 = s*s;

  	/* ==========================================================================================
		Eliminate keypoint if neighborhood too small (contains less than 3 vertices)
		Otherwise, build local neighborhood
   	=============================================================================================*/

		neighbours.clear();
		for(int i = 0; i < nvertices; i++) {
			dist = 0;
			for(int k = 0; k < 3; k++) {
				v = vertex[i+k*nvertices]-vertex[point + k*nvertices];
				dist += v*v;
			}
			if(dist < s2) neighbours.push_back(i);
		}

		if(neighbours.size() < 3) continue;

  	/* ==========================================================================================
		Find local tangent plane
   	=============================================================================================*/

		int nn = neighbours.size();

		center[0] = 0; center[1] = 0; center[2] = 0;
		for(int in = 0; in < nn; in++) {
			int i = neighbours[in];
			vv[3*in+0] = vertex[i];
			vv[3*in+1] = vertex[i + nvertices];
			vv[3*in+2] = vertex[i+ 2*nvertices];
			center[0] += vv[3*in+0];
			center[1] += vv[3*in+1];
			center[2] += vv[3*in+2];
		}
		center[0] /= nn; center[1] /= nn; center[2] /= nn; 
		for(int in = 0; in < nn; in++) {
			vv[3*in+0] -= center[0];
			vv[3*in+1] -= center[1];
			vv[3*in+2] -= center[2];
		}

		char JOBU ='S';
        	char JOBVT='S';
        	int N     = 3;
		double size;
        	int lwork=-1;
        	int info;
        	dgesvd_(&JOBU, &JOBVT, &N, &nn, vv, &N, S, U, &N, VT, &N, &size, &lwork, &info);
		lwork = (int) size;
		Work = new double[lwork];
        	dgesvd_(&JOBU, &JOBVT, &N, &nn, vv, &N, S, U, &N, VT, &N, Work, &lwork, &info);
		delete [] Work;

  	/* ==========================================================================================
		Project mesh on plane and build image
   	=============================================================================================*/

		for(int i = 0; i < 3; i++) {
			eyevec[i] = vertex[point+i*nvertices] + s*normal[point+i*nvertices];
			cent[i] = vertex[point+i*nvertices];
		}

		support = pix/s;
		viewport[0] = -(support-pix)/2.0;
		viewport[1] = -(support-pix)/2.0;
		viewport[2] = support;
		viewport[3] = support;

		create_viewmatrix(eyevec, cent, U, viewmatrix);

		memset(I, 0, 6*npixels*sizeof(double));
		int offset = 4*npixels;
		for(int i = 0; i < npixels; i++) I[offset+i] = Large;

		render.renderpatch(nvertices, vertex, nfaces, faces,
		viewmatrix, proj_matrix, viewport, img_size, I, Inew);
		for(int i = 0; i < npixels; i++) img[i] = Inew[offset+i];

		img[10+10*pix] = (img[10+9*pix]+img[10+11*pix]+img[9+10*pix]+img[11+10*pix])/4;
		
		findLarge(npixels, img, &largest, &second);

		if(largest==Large) {
			for(int i = 0; i < npixels; i++) {
				if(img[i]==Large) img[i] = second;
			}
		}

		for(int i = 0; i < npixels; i++) img[i] = 10*img[i];

		imgGradients(pix, pix, img, Gx, Gy);

		for(int i = 0; i < npixels; i++) {
			grad[2*i] = (float) std::sqrt(Gx[i]*Gx[i] + Gy[i]*Gy[i]);
			grad[2*i+1] = (float) std::atan2(Gy[i], Gx[i]);
		}

		x = 9; y = 9; sigma = 2; angle0 = 0;

		for(int i = 0; i < ndesc; i++) desc[i] = 0.;
		sift.compute_descriptors(pix, pix, grad, x, y, sigma, angle0, desc);

		rotate180(pix, pix, img);
		imgGradients(pix, pix, img, Gx, Gy);
		for(int i = 0; i < npixels; i++) {
			grad[2*i] = (float) std::sqrt(Gx[i]*Gx[i] + Gy[i]*Gy[i]);
			grad[2*i+1] = (float) std::atan2(Gy[i], Gx[i]);
		}
		sift.compute_descriptors(pix, pix, grad, x, y, sigma, angle0, desc);

		descriptors.push_back(desc);
	}

	delete [] vertex; delete [] normal; delete [] faces; delete [] vv;
	delete [] I; delete [] Inew; delete [] img; delete [] Gx; delete [] Gy; delete [] grad;
	delete [] U; delete [] VT; delete [] S;
  }

  /* ================================================================================================
	Compute Zernike descriptors

	Note that:
		vertex, faces, and normals are nx3 vectors
   ==================================================================================================*/
	
  void Descriptors::zernike(Mesh& mesh, std::vector<int>& keypoints,
		double *scale, std::vector<std::vector<double> >& descriptors)
{
  /* ================================================================================================
	Local parameters and arrays
   ==================================================================================================*/

	int nvertices = mesh.vertices.size();
	int nfaces = mesh.faces.size();

	int pix = 21;
	int npixels = pix*pix;
	double scale_fact = 3.0;
	double Large = 1e15;
	int Nmax = 12;

	int ndesc = 0;
	for(int i = 0; i <= Nmax; i++) ndesc += i + 1;

	double *vertex = new double[3*nvertices];
	double *normal = new double[3*nvertices];
	double *faces = new double[3*nfaces];
	double *vv = new double[3*nvertices];

	std::vector<int> neighbours;

	double center[3];
	double eyevec[3];
	double cent[3];
	double viewmatrix[16];
	double proj_matrix[16];
	double viewport[4];

	double *I = new double[6*npixels];
	double *Inew = new double[6*npixels];
	double *img = new double[npixels];

	std::vector<double> desc;
	desc.resize(ndesc);

	int *img_size = new int[3];
	img_size[0] = pix; img_size[1] = pix; img_size[2] = 6;

	double *U = new double[9];
	double *VT = new double[3*nvertices];
	double *S  = new double[3];
	double *Work;

	double support;
	double largest, second;

  /* ================================================================================================
	Transfer mesh info into local structures
	Note: add +1 to vertex indices for faces as renderpatch used numbering starting at 1

	Compute normal, ensuring that they are outward
   ==================================================================================================*/

	int idx;
	for (VertexIter v = mesh.vertices.begin(); v != mesh.vertices.end(); v++) {
		idx = v->index;
		for(int j = 0; j < 3; j++) {
			vertex[idx + j*nvertices] = v->position[j];
		}
	}

	int idxA, idxB, idxC;
	HalfEdgeIter hAB, hBC, hCA;
	VertexIter vA, vB, vC;
	idx = 0;
	for(FaceIter f_iter = mesh.faces.begin(); f_iter != mesh.faces.end(); f_iter++)
	{
		hAB =f_iter->he; hBC =hAB->next; hCA =hBC->next;

		vA = hAB->vertex; vB = hBC->vertex; vC = hCA->vertex;

		idxA = vA->index + 1; idxB = vB->index + 1; idxC = vC->index + 1;

		faces[idx ] = (double) idxB;
		faces[idx+ nfaces] = (double) idxC;
		faces[idx+ 2*nfaces] = (double) idxA;
		idx++;
	}

	computeNormals(mesh, normal);

  /* ================================================================================================
	Loop over all keypoints
   ==================================================================================================*/

	int point;
	double v, dist;
	double s2, s;

	memset(proj_matrix, 0, 16*sizeof(double));
	for(int i = 0; i < 4; i++) proj_matrix[i+4*i]=1.0;

	int ksize = keypoints.size();
	for(int kpt = 0; kpt < ksize; kpt++)
	{
		point = keypoints[kpt];
		s = scale[point]*scale_fact*std::sqrt(2.0);
		s2 = s*s;

  	/* ==========================================================================================
		Eliminate keypoint if neighborhood too small (contains less than 3 vertices)
		Otherwise, build local neighborhood
   	=============================================================================================*/

		neighbours.clear();
		for(int i = 0; i < nvertices; i++) {
			dist = 0;
			for(int k = 0; k < 3; k++) {
				v = vertex[i+k*nvertices]-vertex[point + k*nvertices];
				dist += v*v;
			}
			if(dist < s2) neighbours.push_back(i);
		}

		if(neighbours.size() < 3) continue;

  	/* ==========================================================================================
		Find local tangent plane
   	=============================================================================================*/

		int nn = neighbours.size();

		center[0] = 0; center[1] = 0; center[2] = 0;
		for(int in = 0; in < nn; in++) {
			int i = neighbours[in];
			vv[3*in+0] = vertex[i];
			vv[3*in+1] = vertex[i + nvertices];
			vv[3*in+2] = vertex[i+ 2*nvertices];
			center[0] += vv[3*in+0];
			center[1] += vv[3*in+1];
			center[2] += vv[3*in+2];
		}
		center[0] /= nn; center[1] /= nn; center[2] /= nn; 
		for(int in = 0; in < nn; in++) {
			vv[3*in+0] -= center[0];
			vv[3*in+1] -= center[1];
			vv[3*in+2] -= center[2];
		}

		char JOBU ='S';
        	char JOBVT='S';
        	int N     = 3;
		double size;
        	int lwork=-1;
        	int info;
        	dgesvd_(&JOBU, &JOBVT, &N, &nn, vv, &N, S, U, &N, VT, &N, &size, &lwork, &info);
		lwork = (int) size;
		Work = new double[lwork];
        	dgesvd_(&JOBU, &JOBVT, &N, &nn, vv, &N, S, U, &N, VT, &N, Work, &lwork, &info);
		delete [] Work;

  	/* ==========================================================================================
		Project mesh on plane and build image
   	=============================================================================================*/

		for(int i = 0; i < 3; i++) {
			eyevec[i] = vertex[point+i*nvertices] + s*normal[point+i*nvertices];
			cent[i] = vertex[point+i*nvertices];
		}

		support = pix/s;
		viewport[0] = -(support-pix)/2.0;
		viewport[1] = -(support-pix)/2.0;
		viewport[2] = support;
		viewport[3] = support;

		create_viewmatrix(eyevec, cent, U, viewmatrix);

		memset(I, 0, 6*npixels*sizeof(double));
		int offset = 4*npixels;
		for(int i = 0; i < npixels; i++) I[offset+i] = Large;

		render.renderpatch(nvertices, vertex, nfaces, faces,
		viewmatrix, proj_matrix, viewport, img_size, I, Inew);
		for(int i = 0; i < npixels; i++) img[i] = Inew[offset+i];

		
		findLarge(npixels, img, &largest, &second);

		if(largest==Large) {
			for(int i = 0; i < npixels; i++) {
				if(img[i]==Large) img[i] = second;
			}
		}
		img[10+10*pix] = (img[10+9*pix]+img[10+11*pix]+img[9+10*pix]+img[11+10*pix])/4;

		for(int i = 0; i < npixels; i++) img[i] = 10*img[i];

		zer.compute_descriptors(Nmax, pix, pix, img, desc);

		descriptors.push_back(desc);
	}

	delete [] vertex; delete [] normal; delete [] faces; delete [] vv;
	delete [] I; delete [] Inew; delete [] img; 
	delete [] U; delete [] VT; delete [] S;
  }

  /* ================================================================================================
	Compute SISI descriptors

	Note that:
		vertex and normals are nx3 vectors
   ==================================================================================================*/
	
  void Descriptors::sisi(Mesh& mesh, std::vector<int>& keypoints, std::vector<int>& octaves, 
		double *scale, std::vector<std::vector<double> >& descriptors)
{
  /* ================================================================================================
	Local parameters and arrays
   ==================================================================================================*/

	int nvertices = mesh.vertices.size();

	int imgsize = 10;
	int ndesc = imgsize*imgsize;
	double scale_fact = 3.0;

	double *vertex = new double[3*nvertices];
	double *normal = new double[3*nvertices];

	double *img = new double[ndesc];

	std::vector<double> desc;
	desc.resize(ndesc);

  /* ================================================================================================
	Transfer mesh info into local structures
	Note: add +1 to vertex indices for faces as renderpatch used numbering starting at 1

	Compute normal, ensuring that they are outward
   ==================================================================================================*/

	int idx;
	for (VertexIter v = mesh.vertices.begin(); v != mesh.vertices.end(); v++) {
		idx = v->index;
		for(int j = 0; j < 3; j++) {
			vertex[idx + j*nvertices] = v->position[j];
		}
	}

	computeNormals(mesh, normal);

  /* ================================================================================================
	Loop over all keypoints
   ==================================================================================================*/

	int point;
	double s, bin_size;

	int ksize = keypoints.size();
	for(int kpt = 0; kpt < ksize; kpt++)
	{
		point = keypoints[kpt];
		s = scale[point]*scale_fact*std::sqrt(octaves[kpt]+1);

		bin_size = s/imgsize;
		compute_sisi_descriptors(nvertices, vertex, normal, 
		point, bin_size, imgsize, img, desc);
		
		descriptors.push_back(desc);
	}

  }

  /* ===== Mesh edge weights        =================================================================
   ==================================================================================================*/

  void Descriptors::computeMeshWeight(Mesh& mesh, int type, double *Weights)
  {


	Vector pA, pB, pC;

	double length;
	double cotan_A, cotan_B, cotan_C;

	int idxA, idxB, idxC;
	HalfEdgeIter hAB, hBA, hBC, hCA;
	VertexIter vA, vB, vC;

  /* ================================================================================================
	Build adjacency table
   ==================================================================================================*/

	int nvertices = mesh.vertices.size();

  /* ================================================================================================
	Compute weights for each edge (i,j):
		- itype = 0: uniform weight, 1
		- itype = 1: 1/d_ij^2
		- itype = 2: cotan Laplacian
   ==================================================================================================*/

	if(type==0) {

		for (EdgeIter e = mesh.edges.begin(); e != mesh.edges.end(); e++) {
			hAB = e->he;
			hBA = hAB->flip;

			vA = hAB->vertex;
			vB = hBA->vertex;

			idxA = vA->index;
			idxB = vB->index;
			Weights[idxA+idxB*nvertices] = 1.0;
			Weights[idxB+idxA*nvertices] = 1.0;
		}

	} else if(type==1) {

		for (EdgeIter e = mesh.edges.begin(); e != mesh.edges.end(); e++) {
			hAB = e->he;
			hBA = hAB->flip;

			vA = hAB->vertex;
			vB = hBA->vertex;
			pA = vA->position;
			pB = vB->position;
			length = (pA-pB).norm2();

			idxA = vA->index;
			idxB = vB->index;
			Weights[idxA+idxB*nvertices] = 1.0/length;
			Weights[idxB+idxA*nvertices] = 1.0/length;
		}

	} else if(type==2) {

		for(FaceIter f_iter = mesh.faces.begin(); f_iter != mesh.faces.end(); f_iter++)
		{
			hAB =f_iter->he;
			hBC =hAB->next;
			hCA =hBC->next;

			vA = hAB->vertex;
			vB = hBC->vertex;
			vC = hCA->vertex;

			pA = vA->position;
			pB = vB->position;
			pC = vC->position;

			idxA = vA->index;
			idxB = vB->index;
			idxC = vC->index;

			cotan_A = dot(pB-pA, pC-pA) / ( cross(pB-pA, pC-pA).norm());
			cotan_B = dot(pA-pB, pC-pB) / ( cross(pA-pB, pC-pB).norm());
			cotan_C = dot(pA-pC, pB-pC) / ( cross(pA-pC, pB-pC).norm());

			Weights[idxA+nvertices*idxB] += cotan_C;
			Weights[idxB+nvertices*idxA] += cotan_C;
			Weights[idxA+nvertices*idxC] += cotan_B;
			Weights[idxC+nvertices*idxA] += cotan_B;
			Weights[idxB+nvertices*idxC] += cotan_A;
			Weights[idxC+nvertices*idxB] += cotan_A;
		}

	}

  }


  /* ================================================================================================
	Find keypoints at each octave   
   ==================================================================================================*/

  void Descriptors::find_local_maxima(int n, int n_octaves, int flag_boundary,
		int *boundary, int *ind_oct, double *diff, std::vector<int>& keypoints, 
		std::vector<int>& octaves)
  {

	double maxDiff;
	int offset;

	for(int nOct = 0; nOct < n_octaves-1; nOct++) 
	{
		offset = nOct*n;

		for(int i = 0; i < n; i++) 
		{
			if(flag_boundary==1 && boundary[i]==1) continue;

			if(std::abs(ind_oct[i] - nOct) > 0.1) continue;

			if((nOct < n_octaves-1) && (diff[offset+i] < diff[offset+n+i])) continue;
			if((nOct > 0) && (diff[offset+i] < diff[offset-n+i])) continue;

			maxDiff = 0;
			for(int nO = std::max(nOct-1, 0); nO < std::min(nOct+2, n_octaves); nO++) {
				for (std::set<int>::iterator it = adj[i].begin(); it != adj[i].end(); it++) {
					maxDiff = std::max(maxDiff, diff[nO*n + *it]);
				}
			}
			if(diff[offset+i] < maxDiff) continue;

			keypoints.push_back(i);
			octaves.push_back(nOct);
		}
	}
  }

  /* =============================================================================================
     Compute local density at each vertex
   =============================================================================================== */

  void Descriptors::meshScale(Mesh& mesh, double *scale)
  {
	int idxA, idxB;

	Vector pA, pB;

	double length;

	HalfEdgeIter hAB, hBA; 
	VertexIter vA, vB;

	int n = mesh.vertices.size();
	memset(scale, 0, n*sizeof(double));


	for (EdgeIter e = mesh.edges.begin(); e != mesh.edges.end(); e++) {
		hAB = e->he;
		hBA = hAB->flip;

		vA = hAB->vertex;
		vB = hBA->vertex;
		pA = vA->position;
		pB = vB->position;
		length = (pA-pB).norm();

		idxA = vA->index;
		idxB = vB->index;

		scale[idxA] += length/vA->degree();
		scale[idxB] += length/vB->degree();
	}
  }

  /* =============================================================================================
     Create view matrix
   =============================================================================================== */

  void Descriptors::create_viewmatrix(double *eyevec, double *center, double *up, double *viewmatrix)
  {

	Vector zaxis;
	for(int i = 0; i < 3; i++) {
		zaxis[i] = center[i] - eyevec[i];
	}
	zaxis.normalize();

	Vector Up(up[0], up[1], up[2]);
	Vector xaxis = cross(Up, zaxis);
	xaxis.normalize();

	Vector yaxis = cross(zaxis, xaxis);

	double temp1[16], temp2[16];

	memset(temp1, 0, 16*sizeof(double));
	memset(temp2, 0, 16*sizeof(double));
	for(int i = 0; i < 4; i++) temp1[i+4*i] = 1;
	for(int i = 0; i < 4; i++) temp2[i+4*i] = 1;

	for(int i = 0; i < 3; i++) {
		temp1[0+4*i] = xaxis[i];
		temp1[1+4*i] = yaxis[i];
		temp1[2+4*i] = zaxis[i];
		temp2[i+12] = -eyevec[i];
	}

	double alpha, beta;
	alpha = 1.0; beta = 0.0;
	char NoTrans = 'N';
	int n = 4;

       	dgemm_(&NoTrans, &NoTrans, &n, &n, &n, &alpha, temp1, &n, temp2, &n, 
		&beta, viewmatrix, &n);

  }

  /* =============================================================================================
     Compute vertex normals
   =============================================================================================== */

  void Descriptors::computeNormals(Mesh& mesh, double *normal)
  {
	int nvertices = mesh.vertices.size();
	int idx;
	double center[3];
	std::vector<Vector> Normals;

	for(int i=0; i < nvertices; i++) {
		Vector p;
		Normals.push_back(p);
	}

	int idxA, idxB, idxC;
	HalfEdgeIter hAB, hBC, hCA;
	Vector A, B, C;
	idx = 0;
	for(FaceIter f_iter = mesh.faces.begin(); f_iter != mesh.faces.end(); f_iter++)
	{
		hAB =f_iter->he;
		hBC =hAB->next;
		hCA =hBC->next;

		A = hAB->vertex->position;
		B = hBC->vertex->position;
		C = hCA->vertex->position;

		idxA = hAB->vertex->index;
		idxB = hBC->vertex->index;
		idxC = hCA->vertex->index;

	        Vector n = cross(B - A, C - A);
		n.normalize();

		Normals[idxA] += n;
		Normals[idxB] += n;
		Normals[idxC] += n;


	}

	for(int i = 0; i < nvertices; i++) {
		A = Normals[i];
		A.normalize();
		normal[i] = A[0];
		normal[i+nvertices] = A[1];
		normal[i+2*nvertices] = A[2];
	}

  /* ================================================================================================
	Make sure all normals are outward
   ==================================================================================================*/

	center[0] = 0; center[1] = 0; center[2] = 0;
	for (VertexIter v = mesh.vertices.begin(); v != mesh.vertices.end(); v++) {
		idx = v->index;
		for(int j = 0; j < 3; j++) {
			center[j] += v->position[j];
		}
	}
	center[0] /= nvertices; center[1] /= nvertices; center[2] /= nvertices;

	int npos = 0, nneg = 0;
	double dot;
	for (VertexIter v = mesh.vertices.begin(); v != mesh.vertices.end(); v++) {
		idx = v->index;
		dot = 0;
		for(int j = 0; j < 3; j++) {
			dot += (v->position[j]-center[j])*normal[idx+j*nvertices];
		}
		if(dot > 0) npos++;
		else nneg++;
	}
	if(npos < nneg) {
		for(int i = 0; i < 3*nvertices; i++) normal[i] *= -1.0;
	}

  }
  /* =============================================================================================
     Find largest, and second largest element in an array
   =============================================================================================== */

   void Descriptors::findLarge(int npixels, double *img, double *largest, double *second)
   {

	double l, s;

	l = img[0];
	s = l;
	for(int i = 0; i < npixels; i++) {
		if(img[i] < l) {
			s = img[i];
			break;
		}
	}

	for(int i = 0; i < npixels; i++) 
	{
		if(img[i] > l) {
			s = l;
			l = img[i];
		}
		else if (img[i] > s && img[i] != l) {
			s = img[i];
		}
	}

	*largest = l;
	*second = s;

  }
  /* =============================================================================================
	Compute gradients of an image
   =============================================================================================== */

  void Descriptors::imgGradients(int nx, int ny, double *img, double *Gx, double *Gy)
  {

	int npixels = nx*ny;
	memset(Gx, 0, npixels*sizeof(double));
	memset(Gy, 0, npixels*sizeof(double));

  /* =============================================================================================
		Gradient in x direction
   =============================================================================================== */

	for(int i = 0; i < nx; i++) {
		Gx[i] = img[i+nx] - img[i];
	}
	for(int j = 1; j < ny -1 ; j++) {
		for(int i = 0; i < nx; i++) {
			Gx[i+j*nx] = 0.5*(img[i+(j+1)*nx] - img[i+(j-1)*nx]);
		}
	}
	for(int i = 0; i < nx; i++) {
		Gx[i+(ny-1)*nx] = img[i+(ny-1)*nx] - img[i+(ny-2)*nx];
	}

  /* =============================================================================================
		Gradient in y direction
   =============================================================================================== */

	for(int j = 0; j < ny; j++) {
		Gy[j*nx] = img[1+j*nx] - img[j*nx];
	}
	for(int i = 1; i < nx -1 ; i++) {
		for(int j = 0; j < ny; j++) {
			Gy[i+j*nx] = 0.5*(img[i+1+j*nx] - img[i-1+j*nx]);
		}
	}
	for(int j = 0; j < ny; j++) {
		Gy[nx-1+j*nx] = img[nx-1+j*nx] - img[nx-2+j*nx];
	}

  }
  /* =============================================================================================
	Rotate in place a matrix by 180 degrees
   =============================================================================================== */

  void Descriptors::rotate180(int nx, int ny, double *mat)
  {

  /* =============================================================================================
		Reverse rows
   =============================================================================================== */

	double a;

	for(int j = 0; j < ny; j++) {
		for(int i = 0; i < nx/2; i++) {
			a = mat[nx-1-i + j*nx];
			mat[nx-1-i + j*nx] = mat[i + j*nx];
			mat[i+j*nx] = a;
		}
	}

  /* =============================================================================================
		Reverse cols
   =============================================================================================== */

	for(int i = 0; i < nx; i++) {
		for(int j = 0; j < ny/2; j++) {
			a = mat[i + (ny-1-j)*nx];
			mat[i + (ny-1-j)*nx] = mat[i + j*nx];
			mat[i+j*nx] = a;
		}
	}
  }

/* ===============================================================================================
 *
 * Compute the SISI descriptor on one vertex
 *
 * @param nvertices: 	# of vertices in the mesh
 * @param descriptors:  SISI descriptor (output).
 * @param vertex   array of vertex coordinates
 * @param normals  array of vertex normals
 * @param kpt      index of the keypoint considered
 * @param bin_size  size of each bin in Spin image
 * @param imgsize  image size (# of bins in each direction)
 * @param img      work array of size imgsize*imgsize that stores the spin image
 =============================================================================================== */

  void Descriptors::compute_sisi_descriptors(int nvertices, double *vertex, double *normals , 
		int kpt, double bin_size, int imgsize, double *img, std::vector<double>& descriptors)
  {

	/* ====================================================================================
	Current point 
 	======================================================================================= */

	double KeyPt[3], KeyNormal[3];
	double Pt[3], Normal[3];
	double Diff[3], Cross[3];

	for(int i = 0; i < 3; i++) {
		KeyPt[i] = vertex[kpt+i*nvertices];
		KeyNormal[i] = normals[kpt+i*nvertices];
	}

	/* ====================================================================================
	Initialize spin image
 	======================================================================================= */

	memset(img, 0, imgsize*imgsize*sizeof(double));

	/* ====================================================================================
	Loop over all vertices in the mesh
 	======================================================================================= */

	double ddot, dang;
	double Alpha, Beta;
	double A, B;

	int idx;

	int sum = 0;
	for(int i = 0; i < nvertices; i++) {
		for(int k = 0; k < 3; k++) {
			Pt[k] = vertex[i+k*nvertices];
			Normal[k] = normals[i+k*nvertices];
		}
		ddot = Normal[0]*KeyNormal[0] + Normal[1]*KeyNormal[1] + Normal[2]*KeyNormal[2];
		ddot = std::max(-1.0, std::min(1.0, ddot));
		dang = std::acos(ddot);

		if(dang > M_PI/2) continue;

		for(int k = 0; k < 3; k++) Diff[k] = Pt[k] - KeyPt[k];

		Cross[0] = Diff[1] * KeyNormal[2] - Diff[2]*KeyNormal[1]; 
		Cross[1] = Diff[2] * KeyNormal[0] - Diff[0]*KeyNormal[2]; 
		Cross[2] = Diff[0] * KeyNormal[1] - Diff[1]*KeyNormal[0]; 

		Alpha = std::sqrt(Cross[0] * Cross[0] + Cross[1] * Cross[1] + Cross[2] * Cross[2]);
		Beta = KeyNormal[0] *Diff[0] + KeyNormal[1] * Diff[1] + KeyNormal[2]*Diff[2];

		Beta -= 0.5* imgsize * bin_size/2.0;

		if(Beta > 0 || Beta <= (-imgsize*bin_size)) continue;

		A = std::floor( Alpha / bin_size);
		if(A >= imgsize) continue;

		B = std::floor(-Beta/bin_size);

		idx = (int) (A + B*imgsize);

		img[idx] += 1.0;
		sum++;
	}

	for (int i = 0 ; i < imgsize*imgsize ; i++) {
		descriptors[i] = img[i]/sum;
	}
  }

#endif
