/* ====BelkinLaplace=============================================================================
 *
 * Author: Patrice Koehl, May 2020
 * Department of Computer Science
 * University of California, Davis
 *
 * Computes the discrete Laplace operator on a surface mesh using:
 *
 *	M. Belkin, J. Sum and Y. Wang. "Discrete Laplace Operator on Meshed Surfaces",
 *	Proceedings of the Twenty-Fourth Annual Symposium on Computational Geometry,
 *	pages 278-287 (2008)
 *
 =============================================================================================== */

#ifndef _BELKINLAPLACE_
#define _BELKINLAPLACE_

/*================================================================================================
 BLAS / LAPACK prototypes
================================================================================================== */

  extern "C" {

	void dsyevd_(char * JOBZ, char * UPLO, int *N, double *A, int *LDA, double *W, 
	double *WORK, int *LWORK, int *IWORK, int *LIWORK, int *INFO);

  }

  #define _USE_MATH_DEFINES // for M_PI

  #include <vector>
  #include <cmath>
  #include "geodesic_algorithm_exact.h"

  /* =============================================================================================
     Define class
   =============================================================================================== */

  class BelkinLaplace {

  public:
	// generate Laplace operator
	void laplace(Mesh& mesh, double *L);

	// eigen decomposition of Laplace operator
	void laplaceEigen(int nvertices, double *eigVect, double *eigVal);

  private:
 
	// Find mesh scale
	double aveEdgeLength(Mesh& mesh);

	// Compute area associated with each vertex
	void computeArea(Mesh& mesh, double *Area);

	// Computes surface area of a triangle
	double trigArea(Vector& A, Vector& B, Vector& C);

	// build "geodesic" mesh
	void buildGeodesicMesh(Mesh& mesh, geodesic::Mesh& gMesh);

	// get geodesic neighbours
	void geodesicNeighbours(int vidx, geodesic::Mesh& gMesh, geodesic::GeodesicAlgorithmExact& algorithm, 
   		std::vector<std::pair<int, double> >& neighbours, double maxdist);

  };

  /* ===============================================================================================
   Generate a mesh structure to be used by the code "geodesic"
   =============================================================================================== */

   void BelkinLaplace::buildGeodesicMesh(Mesh& mesh, geodesic::Mesh& gMesh)
   {

	std::vector<double> points;
	std::vector<unsigned> faces;

	for(VertexIter v = mesh.vertices.begin(); v != mesh.vertices.end(); v++) {
		Vector ve = v->position;
		points.push_back(ve.x);
		points.push_back(ve.y);
		points.push_back(ve.z);
	}

	HalfEdgeIter hAB, hBC, hCA;
	int idxA, idxB, idxC;
	for(FaceIter f = mesh.faces.begin(); f != mesh.faces.end(); f++) {
		hAB =f->he;
		hBC =hAB->next;
		hCA =hBC->next;
		idxA = hAB->vertex->index;
		idxB = hBC->vertex->index;
		idxC = hCA->vertex->index;
		faces.push_back(idxA);
		faces.push_back(idxB);
		faces.push_back(idxC);
	}
	gMesh.initialize_mesh_data(points, faces);


  }

  /* ===============================================================================================
   Find closest neighbours (based on geodesic distance) of a given vertex
   =============================================================================================== */

   void BelkinLaplace::geodesicNeighbours(int vidx, geodesic::Mesh& gMesh, geodesic::GeodesicAlgorithmExact& algorithm, 
   			std::vector<std::pair<int, double> >& neighbours, double maxdist)
    {

	// Define current vertex as "source"
	geodesic::SurfacePoint source(&gMesh.vertices()[vidx]);

	// set "all_sources" to be that source
	std::vector<geodesic::SurfacePoint> all_sources(1,source);

	// Propagate over the whole mesh
	algorithm.propagate(all_sources);
	
	double distance;
	unsigned int best_source;

	neighbours.clear();
	for(int i = 0; i < gMesh.vertices().size(); i++)
	{
		geodesic::SurfacePoint p(&gMesh.vertices()[i]); 
		best_source = algorithm.best_source(p,distance); 
		if(distance <= maxdist) {
			neighbours.push_back(std::make_pair(i, distance));
		}
	}

  }
  /* ===============================================================================================
   Build Discrete Laplacian
   =============================================================================================== */

  void BelkinLaplace::laplace(Mesh& Mesh, double *Laplace)
  {

	int nvertices = Mesh.vertices.size();

  	/* ========================================================================================
   	Compute area for each vertex
   	=========================================================================================== */

	double *Area = new double[nvertices];
	computeArea(Mesh, Area);

  	/* ========================================================================================
   	Prepare for "geodesic: build mesh structure and define algorithm
   	=========================================================================================== */

	geodesic::Mesh gMesh;
	buildGeodesicMesh(Mesh, gMesh);

	geodesic::GeodesicAlgorithmExact algorithm(&gMesh); 

  	/* ========================================================================================
   	Parameter for Gaussian kernel 
   	=========================================================================================== */

	double h = 2.0;
	double rho = 3.0;

	double alength = aveEdgeLength(Mesh);
	h = h*alength;

	double hh = h*h; 		// variance
	double maxdist = h*rho;		// neighbourhood size
	double fact = 4.0 / (M_PI * hh * hh);

  	/* ========================================================================================
   	Local array + initialization
   	=========================================================================================== */

	std::vector<std::pair<int, double> > neighbours;
	double *totalWeight = new double[nvertices];
	
	memset(Laplace, 0, nvertices*nvertices*sizeof(double));
	memset(totalWeight, 0, nvertices*sizeof(double));

  	/* ========================================================================================
   	Loop over all vertices
   	=========================================================================================== */

	int jdx;
	double dist, weight;

	for(int i = 0; i < nvertices; i++) {

		neighbours.clear();
		geodesicNeighbours(i, gMesh, algorithm, neighbours, maxdist);

		for(int j = 0; j < neighbours.size(); j++) {

			jdx = neighbours[j].first;
			if(jdx <= i) continue;

			dist = neighbours[j].second;

			weight = -std::exp(-dist*dist/hh) * fact;
			weight *= Area[i]*Area[jdx];

			Laplace[i+jdx*nvertices] = weight;
			Laplace[jdx+i*nvertices] = weight;

			totalWeight[i] -= weight;
			totalWeight[jdx] -= weight;

		}

	}

	for(int i = 0; i < nvertices; i++) {
		Laplace[i+nvertices*i] = totalWeight[i];
	}

	delete [] totalWeight;
	neighbours.clear();

  }

/*================================================================================================
 Compute all eigenpairs of Laplace Operator using LAPACK routine dsyevd

 Note: eigVect as input contains the Laplace operator
================================================================================================== */

  void BelkinLaplace::laplaceEigen(int N, double *eigVect, double *eigVal)
  {

/*================================================================================================
	Declare some variables
================================================================================================== */

	int lwork, info;
	int liwork, isizeopt;
	double sizeopt;
	char U = 'U';
	char V = 'V';

/*================================================================================================
	Create all temporary arrays
================================================================================================== */

	lwork = -1;
	liwork = -1;
	dsyevd_(&V, &U, &N, eigVect, &N, eigVal, &sizeopt, &lwork, &isizeopt, &liwork, &info);

	lwork = (int) sizeopt;
	liwork = isizeopt;

	int *iwork = new int[liwork];
	double *dwork = new double[lwork];

/*================================================================================================
	Compute eigenvalues / eigenvectors
================================================================================================== */

	dsyevd_(&V, &U, &N, eigVect, &N, eigVal, dwork, &lwork, iwork, &liwork, &info);

/*================================================================================================
	Clean up all temporary arrays
================================================================================================== */

	delete [] dwork;
	delete [] iwork;
 }
  /* ===============================================================================================
   trigArea: Computes surface area of a triangle given by 3 points.

         if the three points are A, B, and C,
	 S = norm( A-B) x (A-C) )/2
	 where x is the cross product

   Input:
	  A, B, C: the three vertices of the triangle
   Output:
	  Area: surface area of the triangle
   =============================================================================================== */

  double BelkinLaplace::trigArea(Vector& A, Vector& B, Vector& C)
  {
	double Area;
	Area = 0.5 * cross(A-B , A-C).norm() ;
	return Area;
  }


  /* ===============================================================================================
   computeArea: computes the area of all vertices ( 1/3 of the area of each of its adjacent triangles)

   Input:
	  mesh:	the mesh data structure (pointer to the structure)
   =============================================================================================== */

  void BelkinLaplace::computeArea(Mesh& mesh, double *Area)
  {
	HalfEdgeCIter he;
	VertexCIter va, vb, vc;
	Vector a, b, c;

	int idxa, idxb, idxc;

	double Tarea;

	int n_vertices = mesh.vertices.size();
	memset(Area, 0, n_vertices*sizeof(double));

/* 	====================================================================================
	Iterate over each triangle:
       	=================================================================================== */

	for (FaceCIter f = mesh.faces.begin(); f != mesh.faces.end(); f++)
	{
		he = f->he;
		va = he->vertex;
		vb = he->next->vertex;
		vc = he->prev->vertex;
		idxa = va->index;
		idxb = vb->index;
		idxc = vc->index;
		a = va->position;
		b = vb->position;
		c = vc->position;

		Tarea = trigArea(a, b, c);

		Area[idxa] += Tarea/3;
		Area[idxb] += Tarea/3;
		Area[idxc] += Tarea/3;
	}

   }

  /* ===============================================================================================
	Find mesh scale: average edge length
   =============================================================================================== */

  double BelkinLaplace::aveEdgeLength(Mesh& mesh)
  {

	double ave = 0;
	double length;
        for(EdgeIter e = mesh.edges.begin(); e != mesh.edges.end(); e++)
        {
                HalfEdgeIter h = e->he;
                Vector p0 = h->vertex->position;
                Vector p1 = h->flip->vertex->position;
                length = (p0-p1).norm();
                ave += length;
        }
	int nedges = mesh.edges.size();
	ave /= nedges;

	return ave;
  }

#endif
