/* ====RENDERPATCH_H =============================================================================
 *
 * Author: Patrice Koehl (based on code from Dirk-Jan-Kroon), May 2020
 * Department of Computer Science
 * University of California, Davis
 *
 * Project a mesh onto an image
 *
 * based on: 
 * https://www.mathworks.com/matlabcentral/fileexchange/27084-patch-software-render?s_tid=prof_contriblnk
 =============================================================================================== */

#ifndef _RENDERPATCH_H_
#define _RENDERPATCH_H_

 #include "math.h"
 #include "string.h"
 #include <iostream>
 #include "stdlib.h"

 #include "transformation.h"
 #include "vertices.h"
 #include "texture.h"
 #include "renderimage.h"
 #include "shading.h"
 #include "scene.h"
 #include "fragments.h"
 #include "faces.h"
 #include "mesh.h"

/* ===============================================================================================
	Define class
 =============================================================================================== */

  class RenderPatch{

	public:

		void renderpatch(int nvertices, double *vertices, int nfaces, double *faces,
		double *viewmatrix, double *proj_matrix, double *viewport,
		int *img_size, double *Img_in, double *Img_out);

	private:

		void LoadObject(RMesh *H, int nvertices, double *vertices, int nfaces, double *faces,
			double *viewmatrix);

		void LoadScene(Scene *S, RenderImage *I, double *proj_matrix, double *viewport);

  };

/* ===============================================================================================
	load current mesh object into internal mesh structure
 =============================================================================================== */

 void RenderPatch::LoadObject(RMesh *H, int nvertices, double *vertices, int nfaces, double *faces,
	double *viewmatrix)
 {

/* ===============================================================================================
	load faces and vertices in mesh
 =============================================================================================== */

	int Fdims[2], Vdims[2];
	Fdims[0] = nfaces; Fdims[1] = 3;
	Vdims[0] = nvertices; Vdims[1] = 3;


	H[0].setVerticesFaces(faces, Fdims, vertices, Vdims);

/* ===============================================================================================
	load Model view matrix
 =============================================================================================== */

	int Mdims[2];
	Mdims[0] = 4; Mdims[1] = 4;
	H[0].setModelMatrix(viewmatrix, Mdims);
 }

/* ===============================================================================================
	Initialize scene
 =============================================================================================== */

 void RenderPatch::LoadScene(Scene *S, RenderImage *I, double *proj_matrix, double *viewport)
 {

/* ===============================================================================================
	load Current image on scene
 =============================================================================================== */

	S[0].I = I[0];

/* ===============================================================================================
    	Set Scene projection Matrix
 =============================================================================================== */

	S[0].TT.setProjectionMatrix(proj_matrix);
        
/* ===============================================================================================
    	Set Scene viewport
 =============================================================================================== */

	S[0].TT.setViewport(viewport);

/* ===============================================================================================
    	Enable shading and depth test
 =============================================================================================== */

	S[0].enableshading=true;
	S[0].enabledepthtest=true;
	S[0].culling=0;
    
 }

/* ===============================================================================================
	render patch into image
 =============================================================================================== */

  void RenderPatch::renderpatch(int nvertices, double *vertices, int nfaces, double *faces,
		double *viewmatrix, double *proj_matrix, double *viewport,
		int *img_size, double *Img_in, double *Iout)
 {


	int dim_tot = img_size[0] * img_size[1] * img_size[2];

	std::memcpy(Iout, Img_in, dim_tot*sizeof(double));

	RenderImage *I;
	I=new RenderImage(Iout, img_size[0], img_size[1]);

	Scene *S = new Scene[1];
	LoadScene(S, I, proj_matrix, viewport);

	RMesh *H = new RMesh();
	LoadObject(H, nvertices, vertices, nfaces, faces, viewmatrix);

	H[0].drawMesh(S);

	delete [] I;
	delete [] S;
 }

#endif
