/* ====SISI=======================================================================================
 *
 * Author: Patrice Koehl,  May 2020
 * Department of Computer Science
 * University of California, Davis
 *
 * Computes the SI spin image descriptors of keypoints of an image
 *
 =============================================================================================== */

#ifndef _SISI_H_
#define _SISI_H_

 #include "math.h"
 #include "string.h"
 #include <iostream>
 #include "stdlib.h"
 #include <vector>

/* ===============================================================================================
   The SISI class
 =============================================================================================== */

  class SISI {

	public:

		void compute_sisi_descriptors(int nvertices, double *vertex, double *normals , 
			int kpt, double scale, int imgsize, double *img, std::vector<double>& descriptors);

	private:

  };

/* ===============================================================================================
 *
 * Compute the SISI descriptor on one vertex
 *
 * @param nvertices: 	# of vertices in the mesh
 * @param descriptors:  SISI descriptor (output).
 * @param vertex   array of vertex coordinates
 * @param normals  array of vertex normals
 * @param kpt      index of the keypoint considered
 * @param scale    scale of keypoint
 * @param imgsize  image size (# of bins in each direction)
 * @param img      work array of size imgsize*imgsize that stores the spin image
 =============================================================================================== */

  void SISI::compute_sisi_descriptors(int nvertices, double *vertex, double *normals , 
		int kpt, double scale, int imgsize, std::vector<double>& descriptors)
  {

	/* ====================================================================================
	Current point 
 	======================================================================================= */

	double KeyPt[3], KeyNormal[3];
	double Pt[3], Normal[3];
	double Diff[3], Cross[3];

	for(int i = 0; i < 3; i++) {
		KeyPt[i] = vertex[kpt+i*nvertices];
		KeyNormal[i] = normals[kpt+i*nvertices];
	}

	/* ====================================================================================
	Initialize spin image
 	======================================================================================= */

	memset(img, 0, imgsize*imgsize*sizeof(double));

	/* ====================================================================================
	Loop over all vertices in the mesh
 	======================================================================================= */

	double ddot, dang;
	double Alpha, Beta;
	double A, B;

	int idx;

	int sum = 0;
	for(int i = 0; i < nvertices; i++) {
		for(int k = 0; k < 3; k++) {
			Pt[k] = vertex[i+k*nvertices];
			Normal[k] = normals[i+k*nvertices];
		}
		ddot = Normal[0]*KeyNormal[0] + Normal[1]*KeyNormal[1] + Normal[2]*KeyNormal[2];
		ddot = std::max(-1.0, std::min(1.0, ddot));
		dang = std::acos(ddot);

		if(dAng > M_PI/2) continue;

		for(int k = 0; k < 3; k++) Diff[k] = Pt[k] - KeyPt[k];

		Cross[0] = Diff[1] * KeyNormal[2] - Diff[2]*KeyNormal[1]; 
		Cross[1] = Diff[2] * KeyNormal[0] - Diff[0]*KeyNormal[2]; 
		Cross[2] = Diff[0] * KeyNormal[1] - Diff[1]*KeyNormal[0]; 

		Alpha = std::sqrt(Cross[0] * Cross[0] + Cross[1] * Cross[1] + Cross[2] * Cross[2]);
		Beta = KeyNormal[0] *Diff[0] + KeyNormal[1] * Diff[1] + KeyNormal[2]*Diff[2];

		Beta -= 0.5* imgsize * scale/2.0;

		if(Beta > 0 || Beta <= (-imgsize*scale)) continue

		A = std::floor( Alpha / scale);
		if(A >= imgsize) continue;

		B = std::floor(-Beta/scale);

		idx = (int) (A + B*imgsize);

		img[idx] += 1.0;
		sum++;
	}

	for (int i = 0 ; i < imgsize*imgsize ; i++) {
		img[i] /= sum;
	}
  }

#endif

