#pragma once

#include "Types.h"
class Vertex {
public:
	// constructor
	Vertex();

	// outgoing halfedge
	HalfEdgeIter he;

	// position
	Vector position;

	// second position
	Vector position2;

	// flag to indicate whether this vertex is a neighbor of
	// the north pole of a stereographic projection from the disk to a sphere
	bool inNorthPoleVicinity;

	// flag to indicate whether this vertex is
	// the north pole of a stereographic projection from the disk to a sphere
	bool NorthPole;

	// id between 0 and |V|-1
	int index;

	// id of the reference vertex this vertex is a duplicate of
	int referenceIndex;

	// id of the vertex after North pole is removed
	int indexN;

	// checks if this vertex is on the boundary
	bool onBoundary() const;

	// returns degree
	int degree() const;

	// returns angle weighted average of adjacent face normals
	Vector normal() const;

	// returns angle weighted average of adjacent face normals, based on position2
	Vector normal2() const;

	// returns 2π minus sum of incident angles. Note: only valid for interior vertices
	double angleDefect() const;

	// returns exterior angle. Note: only valid for boundary vertices
	double exteriorAngle() const;
};

#include "Face.h"

inline Vertex::Vertex():
inNorthPoleVicinity(false),
NorthPole(false),
index(-1),
referenceIndex(-1),
indexN(-1)
{

}

inline bool Vertex::onBoundary() const
{
	if (inNorthPoleVicinity) return true;

	HalfEdgeCIter h = he;
	do {
		if (h->onBoundary) return true;
		h = h->flip->next;
	} while (h != he);

	return false;
}

inline int Vertex::degree() const
{
	int k = 0;
	HalfEdgeCIter h = he;
	do {
		k++;

		h = h->flip->next;
	} while (h != he);

	return k;
}

inline Vector Vertex::normal() const
{
	Vector n;
	Vector a, b, c, u, v;
	double angle;
	HalfEdgeCIter h = he;
	do {
		if (!h->onBoundary) {
        		a = h->prev->vertex->position;
        		b = h->vertex->position;
        		c = h->next->vertex->position;
			u = b - a;
			u.normalize();
			v = c - a;
			v.normalize();
			angle = std::acos(std::max(-1.0, std::min(1.0, dot(u, v))));

			n += h->face->normal(false)*angle;
		}

		h = h->flip->next;
	} while (h != he);

	n.normalize();

	return n;
}

inline Vector Vertex::normal2() const
{
	Vector n;
	Vector a, b, c, u, v;
	double angle;
	HalfEdgeCIter h = he;
	do {
		if (!h->onBoundary) {
        		a = h->prev->vertex->position2;
        		b = h->vertex->position2;
        		c = h->next->vertex->position2;
			u = b - a;
			u.normalize();
			v = c - a;
			v.normalize();
			angle = std::acos(std::max(-1.0, std::min(1.0, dot(u, v))));

			n += h->face->normal2(false)*angle;
		}

		h = h->flip->next;
	} while (h != he);

	n.normalize();

	return n;
}

inline double Vertex::angleDefect() const
{
	double sum = 0.0;
	if (onBoundary()) return sum;

	Vector a, b, c, u, v;
	double angle;

	HalfEdgeCIter h = he;
	do {
        	a = h->prev->vertex->position;
        	b = h->vertex->position;
        	c = h->next->vertex->position;
		u = b - a;
		u.normalize();
		v = c - a;
		v.normalize();
		angle = std::acos(std::max(-1.0, std::min(1.0, dot(u, v))));
		sum += angle;

		h = h->flip->next;
	} while (h != he);

	return 2*M_PI - sum;
}

inline double Vertex::exteriorAngle() const
{
	double sum = 0.0;
	if (!onBoundary()) return sum;

	Vector a, b, c, u, v;
	double angle;

	HalfEdgeCIter h = he;
	do {
		if (!h->onBoundary) {
        		a = h->prev->vertex->position;
        		b = h->vertex->position;
        		c = h->next->vertex->position;
			u = b - a;
			u.normalize();
			v = c - a;
			v.normalize();
			angle = std::acos(std::max(-1.0, std::min(1.0, dot(u, v))));
			sum += angle;
		}

		h = h->flip->next;
	} while (h != he);

	return M_PI - sum;
}
