/*================================================================================================
  OptimBfactor.h
  Version 1: 12/1/2017

Copyright (c) Patrice Koehl.

================================================================================================== */

#ifndef _OPTIMBFACTOR_KI_H_
#define _OPTIMBFACTOR_KI_H_

/*================================================================================================
 Includes
================================================================================================== */

#include <math.h>
#include <cstdlib>
#include "derivs_ki_thread.h"

#ifndef M_PI
    #define M_PI 3.14159265358979323846
#endif

/*================================================================================================
  Protoflag_aves for BLAS
================================================================================================== */

extern "C" {

	void daxpy_(int * n ,double *alpha , double * X, int *incx, double * Y,int *incy);
	double dnrm2_(int * n, double * X, int *incx);
	void dscal_(int * n, double * alpha, double * X, int *incx);
	void dcopy_(int * n, double * X, int *incx, double * Y, int *incy);
	double ddot_(int * n, double * u, int * incu, double * v, int *incv);

	void setulb_(int *N, int *M, double *X, double *L, double *U, int *NBD, double *F, 
		double *G, double *FACTR, double *PGTOL, double *WA, int *IWA, char *TASK,
		int *IPRINT, char *CSAVE, int *LSAVE, int *ISAVE, double *DSAVE);

}

/*================================================================================================
  class
================================================================================================== */

  class OptimBfactor_ki {

	public:

		// Optim bfactors
		double optimkval(std::vector<Atoms>& atoms, std::vector<Links>& pairs, int nm1,
		int nm2, int nm3, double *Uij, double *U1ij, double *Uijk, double *Uijkl,
		double *hessian, double *eigVal, double *eigVect, double *bfact, int flag_ave, 
		int flag_ent, int flag_rigid, double *rig, int nthreads);

		// Check derivatives
		void checkDeriv(std::vector<Atoms>& atoms, std::vector<Links>& pairs, 
		int nm1, int nm2, double *Uij, double *U1ij, double *Uijk, double *Uijkl,
		double *hessian, double *eigVal, double *eigVect, double *bfact, int flag_ave, 
		int flag_ent, int flag_rigid, int nthreads);

	private:

		// init
		void init(int natoms, int nlinks, int nm1, int nm2, int nm3, int potential, int flag_ave,
		int flag_rigid);

		// InitKconst
		double initKconst(std::vector<Atoms>& atoms, std::vector<Links>& pairs, 
		double *Uij, double *U1ij, double *Uijk, double *Uijkl, double *hessian,
		double *eigVal, double *eigVect, int flag_ave, int flag_rigid);

		//
		void resetKconst(std::vector<Atoms>& atoms, double *X, int potential);

		// entropy
		double entropy(double *eigVal);

		// Energy function
		double energy(std::vector<Atoms>& atoms, int flag_rigid, double *eigVal, double *eigVect, 
			double *bfact);

		// derivatives of Go-specific terms
		void deriv_Go(std::vector<Atoms>& atoms, double *U1ij, double *Uijk, double *Uijkl, 
		double *eigVal, double *eigVect, double *bfact);

		void deriv_rigid(std::vector<Atoms>& atoms, double *bfact);

		// Energy and derivatives
		double eneAndDer(std::vector<Atoms>& atoms, std::vector<Links>& pairs, 
		int flag_rigid, double *Uij, double *U1ij, double *Uijk, double *Uijkl,
		double *eigVal, double *eigVect, double *bfact, int flag_ave, int flag_ent, int nthreads);

		// Recompute Hessian and diagonalize
		void diagHessian(std::vector<Atoms>& atoms, std::vector<Links>& pairs, 
		double *Uij, double *U1ij, double *Uijk, double *Uijkl, double *hessian,
		double *eigVal, double *eigVect, int flag_ave);

		// One step of LBFGSB
		double oneStep_LBFGSB(std::vector<Atoms>& atoms, std::vector<Links>& pairs, 
		double *Uij, double *U1ij, double *Uijk, double *Uijkl, double *hessian,
		double *eigVal, double *eigVect, double *bfact, int *IFLAG, int iter, 
		int flag_ave, int flag_ent, int flag_rigid, int nthreads);

		int nmode1, nmode2, nmodeB;
		double *DiagHm1, *dDiagHm1, *r2_rigid, *rigid;
		double facb;

		int N, M;
		double *X, *G;
		double *L, *U, *WA, *DSAVE;
		int *NBD, *IWA, *ISAVE;
		char *TASK, *CSAVE;
		int *LSAVE;

		int IPRINT = -1;
		double FACTR = 1.e7;
		double PGTOL = 1.e-4;

		int potential = 0;
		double lambda, lambda_ent;

  };

/* ===============================================================================================
   Initialize parameters
   =============================================================================================== */

void OptimBfactor_ki::init(int natoms, int nlinks, int nm1, int nm2, int nm3, int potential, int flag_ave,
	int flag_rigid)
{
	nmode1 = nm1;
	nmode2 = nm2;
	nmodeB = nm3;

	double kT = 0.593;
	facb = 8.0*kT*M_PI*M_PI/3.0;

	DiagHm1 = new double[3*natoms];
	dDiagHm1 = new double[3*natoms];
	r2_rigid = new double[natoms];
	rigid = new double[10];

	N = natoms;
	if(flag_rigid==1) N += 10;
	M = std::min(200, natoms/2);

	X = new double[N];
	G = new double[N];

	L = new double[N];
	U = new double[N];
	NBD = new int[N];
	WA = new double[2*M*N+5*N+12*M*M+12*M];
	IWA = new int[3*N];
	ISAVE = new int[44];
	DSAVE = new double[29];
	LSAVE = new int[4];
	TASK = new char[60];
	CSAVE = new char[60];
	memset(L, 0, N*sizeof(double));
	memset(U, 0, N*sizeof(double));

	double kmax;
	if(potential==0) {
		if(flag_ave == 1) {
			kmax = 100;
		} else {
			kmax = 500;
		}
	} else {
		if(flag_ave == 1) {
			kmax = 1000;
		} else {
			kmax = 5000;
		}
	}
	for(int i = 0; i < natoms; i++) {
		NBD[i] = 2;
		L[i] = 0.0001;
		U[i] = kmax;
	}
	if(flag_rigid==1) {
	for(int i = 0; i < 10; i++) {
		NBD[i+natoms] = 0;
	}
/*
		NBD[natoms] = 2;
		L[natoms] = 0.0001;
		U[natoms] = kmax;
		for(int i = 1; i < 10; i++) {
			NBD[i+natoms] = 2;
			L[i+natoms] = -10;
			U[i] = 10;
		}
*/
	}

	for(int i = 0; i < 60; i++) {
		TASK[i] = ' ';
		CSAVE[i] = ' ';
	}

	memset(G, 0, N*sizeof(double));

	std::cout << "        " << "=======================================================================" << std::endl;
	std::cout << "        " << "       Iter           FitB           F           chi2_tot       NormG  " << std::endl;
	std::cout << "        " << "=======================================================================" << std::endl;

  }

/*================================================================================================
 initKconst: initialize all ki to the same value
================================================================================================== */

 double OptimBfactor_ki::initKconst(std::vector<Atoms>& atoms, std::vector<Links>& pairs, 
	double *Uij, double *U1ij, double *Uijk, double *Uijkl, double *hessian,
	double *eigVal, double *eigVect, int flag_ave, int flag_rigid)
  {
	int natoms = atoms.size();

	for(int i = 0; i < natoms; i++) {
		if(U1ij) {
			X[i] = 120.0*0.36;
		} else {
			X[i] = 1.;
		}
	}

	if(flag_rigid==1) {
		for(int i = 0; i < 10; i++) {
			X[i+natoms] = 0;
		}
	}

//	return 1.0;

	diagHessian(atoms, pairs, Uij, U1ij, Uijk, Uijkl, hessian, eigVal, eigVect, flag_ave);

	int npar = 3*natoms;
	memset(DiagHm1, 0, npar*sizeof(double));

	double sum, val;

	for(int k = 0; k < npar; k++) {
		sum = 0.0;
		for (int i = nmode1; i < nmodeB; i++) 
		{
			val = facb/eigVal[i];
			sum = sum + val*eigVect[i*npar + k]*eigVect[i*npar +k];
		}
		DiagHm1[k] = sum;
	}

	double sum1=0, sum2=0;
	double b_exp, b_calc;
	for(int i = 0; i < natoms; i++)
	{
		b_exp = atoms[i].bfact;
		b_calc = DiagHm1[3*i] + DiagHm1[3*i+1] + DiagHm1[3*i+2];
		sum1 += b_exp*b_calc;
		sum2 += b_calc*b_calc;
	}
	double kval = sum2/sum1;

	for(int i = 0; i < natoms; i++) {
		X[i] = kval;
	}
	std::cout << "kval = " << kval << std::endl;

 	return kval;
  }

/*================================================================================================
 entropy: computes entropy regularisation
================================================================================================== */

  double OptimBfactor_ki::entropy(double *eigVal)
  {
	double sum = 0.0;
	for(int i = nmode1; i < nmodeB; i++) {
		sum += std::log(eigVal[i]);
	}
	return sum;
  }

/*================================================================================================
 Energy: computes difference between computed and experimental B-factors
================================================================================================== */

double OptimBfactor_ki::energy(std::vector<Atoms>& atoms, int flag_rigid, double *eigVal, 
		double *eigVect, double *bfact)
{
	int natoms = atoms.size();
	int npar = 3*natoms;
	memset(DiagHm1, 0, npar*sizeof(double));

	double sum, val;

	if(flag_rigid==0) {
		memset(r2_rigid, 0, natoms*sizeof(double));
	} else {
		double x[3];
		for(int i = 0; i < natoms; i++) 
		{
			for(int k = 0; k < 3; k++) x[k] = atoms[i].coord[k];
			val = rigid[0] + rigid[1]*x[0] + rigid[2]*x[1] + rigid[3]*x[2];
			val += rigid[4]*x[0]*x[0] + rigid[5]*x[0]*x[1] + rigid[6]*x[0]*x[2];
			val += rigid[7]*x[1]*x[1] + rigid[8]*x[1]*x[2] + rigid[9]*x[2]*x[2];
			r2_rigid[i] = facb*val;
		}
	}

	for(int k = 0; k < npar; k++) {
		sum = 0.0;
		for (int i = nmode1; i < nmodeB; i++) 
		{
			val = facb/eigVal[i];
			sum = sum + val*eigVect[i*npar + k]*eigVect[i*npar +k];
		}
		DiagHm1[k] = sum;
	}

	double chi2=0;
	double b_exp, b_calc, diff;
	for(int i = 0; i < natoms; i++)
	{
		b_exp = atoms[i].bfact;
		b_calc = r2_rigid[i] + DiagHm1[3*i] + DiagHm1[3*i+1] + DiagHm1[3*i+2];
		diff = b_calc - b_exp;
		chi2 += diff*diff;
		bfact[i] = b_calc;
	}

	return chi2;
}

/*================================================================================================
  derivatives of the rigid body terms
================================================================================================== */

  void OptimBfactor_ki::deriv_rigid(std::vector<Atoms>& atoms, double *bfact)
  {
	int natoms = atoms.size();
	double x[3];
	double val;
	for(int k = 0; k < 10; k++) G[natoms+k] = 0;
	for(int i = 0; i < natoms; i++) 
	{
		val = 2.0*facb*(bfact[i]-atoms[i].bfact);
		for(int k = 0; k < 3; k++) x[k] = atoms[i].coord[k];
		G[natoms+0] += val;
		G[natoms+1] += val*x[0];
		G[natoms+2] += val*x[1];
		G[natoms+3] += val*x[2];
		G[natoms+4] += val*x[0]*x[0];
		G[natoms+5] += val*x[0]*x[1];
		G[natoms+6] += val*x[0]*x[2];
		G[natoms+7] += val*x[1]*x[1];
		G[natoms+8] += val*x[1]*x[2];
		G[natoms+9] += val*x[2]*x[2];
	}
  }

/*================================================================================================
  derivatives of the specific Go terms
================================================================================================== */

  void OptimBfactor_ki::deriv_Go(std::vector<Atoms>& atoms,
	double *U1ij, double *Uijk, double *Uijkl, double *eigVal, double *eigVect, double *bfact)
  {

	int natoms = atoms.size();
	int npar = 3*natoms;

	int iat, jat;
	double a, b, sum;
	int inc=1;
	double *u, *e1, *e2;

	// Bond term
	G[0] = 0.;
	for(int k = 0; k < natoms-1; k++) {
		u = &U1ij[3*k];
		iat = k;
		jat = k+1;

		memset(dDiagHm1, 0, npar*sizeof(double));
		for(int i = nmode1; i < nmode2; i++) {
			e1 = &eigVect[i*npar + 3*iat];
			e2 = &eigVect[i*npar + 3*jat];
			a = u[0]*(e1[0]-e2[0]) + u[1]*(e1[1]-e2[1]) + u[2]*(e1[2]-e2[2]);
			a = a/eigVal[i];
			daxpy_(&npar, &a, &eigVect[i*npar], &inc, dDiagHm1, &inc);
		}

		for(int i = 0; i < npar; i++) {
			a = dDiagHm1[i];
			dDiagHm1[i] = -a * a;
		}

		sum = 0.;
		for(int i = 0; i < natoms; i++) {
			a = bfact[i] - atoms[i].bfact;
			b = facb*(dDiagHm1[3*i] + dDiagHm1[3*i+1] + dDiagHm1[3*i+2]);
			sum += a*b;
		}
		sum *= lambda;
		G[0] += 2*sum;

		sum = 0.;
		for(int i = nmode1; i < nmode2; i++) {
			e1 = &eigVect[i*npar + 3*iat];
			e2 = &eigVect[i*npar + 3*jat];
			a = u[0]*(e1[0]-e2[0]) + u[1]*(e1[1]-e2[1]) + u[2]*(e1[2]-e2[2]);
			a = (a*a)/eigVal[i];
			sum += a;
		}
		sum *= lambda_ent;
		G[0] += sum;
	}

	// Angle term
	int nval = 9;
	G[1] = 0;
	for(int k = 0; k < natoms-2; k++) {
		u = &Uijk[9*k];

		memset(dDiagHm1, 0, npar*sizeof(double));
		for(int i = nmode1; i < nmode2; i++) {
			e1 = &eigVect[i*npar + 3*k];
			a = ddot_(&nval, u, &inc, e1, &inc);
			a = a/eigVal[i];
			daxpy_(&npar, &a, &eigVect[i*npar], &inc, dDiagHm1, &inc);
		}

		for(int i = 0; i < npar; i++) {
			a = dDiagHm1[i];
			dDiagHm1[i] = -a * a;
		}

		sum = 0.;
		for(int i = 0; i < natoms; i++) {
			a = bfact[i] - atoms[i].bfact;
			b = facb*(dDiagHm1[3*i] + dDiagHm1[3*i+1] + dDiagHm1[3*i+2]);
			sum += a*b;
		}
		sum *= lambda;
		G[1] += 2*sum;

		sum = 0.;
		for(int i = nmode1; i < nmode2; i++) {
			e1 = &eigVect[i*npar + 3*k];
			a = ddot_(&nval, u, &inc, e1, &inc);
			a = (a*a)/eigVal[i];
			sum += a;
		}
		sum *= lambda_ent;
		G[1] += sum;
	}

	// dihedral term
	nval = 12;
	G[2] = 0;
	for(int k = 0; k < natoms-3; k++) {
		u = &Uijkl[12*k];

		memset(dDiagHm1, 0, npar*sizeof(double));
		for(int i = nmode1; i < nmode2; i++) {
			e1 = &eigVect[i*npar + 3*k];
			a = ddot_(&nval, u, &inc, e1, &inc);
			a = a/eigVal[i];
			daxpy_(&npar, &a, &eigVect[i*npar], &inc, dDiagHm1, &inc);
		}

		for(int i = 0; i < npar; i++) {
			a = dDiagHm1[i];
			dDiagHm1[i] = -a * a;
		}

		sum = 0.;
		for(int i = 0; i < natoms; i++) {
			a = bfact[i] - atoms[i].bfact;
			b = facb*(dDiagHm1[3*i] + dDiagHm1[3*i+1] + dDiagHm1[3*i+2]);
			sum += a*b;
		}
		sum *= lambda;
		G[2] += 2*sum;

		sum = 0.;
		for(int i = nmode1; i < nmode2; i++) {
			e1 = &eigVect[i*npar + 3*k];
			a = ddot_(&nval, u, &inc, e1, &inc);
			a = (a*a)/eigVal[i];
			sum += a;
		}
		sum *= lambda_ent;
		G[2] += sum;
	}

  }

/*================================================================================================
 EneAndDer: computes difference between computed and experimental B-factors, as well as derivatives
	    wrt force constants
================================================================================================== */

 double OptimBfactor_ki::eneAndDer(std::vector<Atoms>& atoms, std::vector<Links>& pairs, 
	int flag_rigid, double *Uij, double *U1ij, double *Uijk, double *Uijkl,
	double *eigVal, double *eigVect, double *bfact, int flag_ave, int flag_ent, int nthreads)
  {

	double chi2 = energy(atoms, flag_rigid, eigVal, eigVect, bfact);

	int natoms = atoms.size();
	int npairs = pairs.size();

/*	==========================================================================================
	Break list to all threads and send jobs
	========================================================================================== */

	int nval = npairs/nthreads;
	int N1, N2;

	for (int i=0; i < nthreads; i++)
	{
		N1 = i*nval;
		N2 = N1 + nval;
		if(i==nthreads-1) N2 = npairs;
		threadids[i] = i;

		derivs_ki[i].N1 = N1;
		derivs_ki[i].N2 = N2;
		derivs_ki[i].flag_ave  = flag_ave;
		derivs_ki[i].facb      = facb;
		derivs_ki[i].nmode1    = nmode1;
		derivs_ki[i].nmode2    = nmodeB;
		derivs_ki[i].lambda    = lambda;
		derivs_ki[i].lambda_ent= lambda_ent;
		derivs_ki[i].eigVect   = eigVect;
		derivs_ki[i].eigVal    = eigVal;
		derivs_ki[i].bfact     = bfact;
		derivs_ki[i].atoms     = atoms;
		derivs_ki[i].pairs     = pairs;
		derivs_ki[i].Uij       = Uij;
		derivs_ki[i].X         = X;
		double *Work = new double[pairs.size()];
		memset(Work, 0., pairs.size()*sizeof(double));
		derivs_ki[i].Work      = Work;
		double *WG = new double[atoms.size()];
		memset(WG, 0., atoms.size()*sizeof(double));
		derivs_ki[i].G         = WG;

		pthread_create(&threads[i], NULL, deriv_ki_thread, (void*) &threadids[i]);
	}

/*	==========================================================================================
	Compute bonded derivatives on main thread
	========================================================================================== */

	memset(G, 0, natoms*sizeof(double));
//	if(potential==1) {
//		deriv_Go(atoms, U1ij, Uijk, Uijkl, eigVal, eigVect, bfact);
//	}

/*	==========================================================================================
	Join all the threads (to make sure they are all finished)
	========================================================================================== */

	int inc = 1;
	double a = 1.;
	for (int i=0; i < nthreads; i++)
	{
		pthread_join(threads[i], NULL);
		daxpy_(&natoms, &a, derivs_ki[i].G, &inc, G, &inc);
	}

	if(flag_rigid==1) deriv_rigid(atoms, bfact);

	return chi2;

  }

/*================================================================================================
 diagHessian: recompute Hessian and its eigenvalues and vectors for new kval
================================================================================================== */

 void OptimBfactor_ki::diagHessian(std::vector<Atoms>& atoms, std::vector<Links>& pairs, 
	double *Uij, double *U1ij, double *Uijk, double *Uijkl, double *hessian,
	double *eigVal, double *eigVect, int flag_ave)
 {
        hess.buildK2(pairs, atoms, X, flag_ave);
	int natoms = atoms.size();
        hess.fullHessian(atoms, pairs, Uij, U1ij, Uijk, Uijkl, hessian);
        int N2 = 9*natoms*natoms;
        int inc = 1;
        dcopy_(&N2, hessian, &inc, eigVect, &inc);
        hess.fullEigen(natoms, eigVal, eigVect);
	hess.rescaleEigVect(atoms, nmode2, eigVect);
  }

/*================================================================================================
 From X to atoms information
================================================================================================== */

   void OptimBfactor_ki::resetKconst(std::vector<Atoms>& atoms, double *X, int potential)
  {
	int natoms = atoms.size();
	for(int i = 0; i < natoms; i++) {
		atoms[i].kconst = X[i];
	}
   }

  /* ==============================================================================================
	One step of the Optimization of kconst with lbfgsb
   =============================================================================================== */

  double OptimBfactor_ki::oneStep_LBFGSB(std::vector<Atoms>& atoms, std::vector<Links>& pairs, 
		double *Uij, double *U1ij, double *Uijk, double *Uijkl, double *hessian,
		double *eigVal, double *eigVect, double *bfact, int *IFLAG, int iter, 
		int flag_ave, int flag_ent, int flag_rigid, int nthreads)
  {

  /* ==================================================================================================
	Variables for lbfgs
   ==================================================================================================*/

	int natoms = atoms.size();

	if(*IFLAG == 0) {
		TASK[0] = 'S'; TASK[1] = 'T'; TASK[2]='A'; TASK[3]='R'; TASK[4]='T';
	}

	double dist, dist_ent, dist_tot;
	do {
		diagHessian(atoms, pairs, Uij, U1ij, Uijk, Uijkl, hessian, eigVal, eigVect, flag_ave);
		dist = eneAndDer(atoms, pairs, flag_rigid, Uij, U1ij, Uijk, Uijkl, eigVal, eigVect, bfact, 
			flag_ave, flag_ent, nthreads);
		dist_ent = entropy(eigVal);
		dist_tot = dist*lambda + lambda_ent*dist_ent;
		setulb_(&N, &M, X, L, U, NBD, &dist_tot, G, &FACTR, &PGTOL, WA, IWA,
		TASK, &IPRINT, CSAVE, LSAVE, ISAVE, DSAVE);
		if(flag_rigid==1) for(int k = 0; k < 10; k++) rigid[k] = X[natoms+k];
//		std::cout << TASK[0] << " " << TASK[1] << " " << TASK[2] << " " << TASK[3] << " " << TASK[4] << std::endl;
	} while(TASK[0]=='F');

	int inc = 1;
	double gnorm = dnrm2_(&N, G, &inc);
        std::cout << "     " << "   " << std::setw(8)<< iter+1;
        std::cout << "     " << std::fixed << std::setprecision(6) << std::setw(8) << dist;
        std::cout << "     " << std::fixed << std::setprecision(6) << std::setw(8) << dist_ent;
        std::cout << "     " << std::fixed << std::setprecision(6) << std::setw(8) << dist_tot;
        std::cout << "     " << std::fixed << std::setprecision(6) << std::setw(8) << gnorm;
	std::cout << std::endl;

	if(TASK[0]=='C') {
		*IFLAG=0;
	} else if( TASK[0]=='A' || TASK[0]=='E') {
		*IFLAG=1;
	} else {
		*IFLAG=2;
	}

	return dist;
  }

  /* ==============================================================================================
	Check derivatives of Bfactors
   =============================================================================================== */

  void OptimBfactor_ki::checkDeriv(std::vector<Atoms>& atoms, std::vector<Links>& pairs, int nm1, int nm2,
		double *Uij, double *U1ij, double *Uijk, double *Uijkl, double *hessian,
		double *eigVal, double *eigVect, double *bfact, int flag_ave, int flag_ent, 
		int flag_rigid, int nthreads)
  {

	int natoms = atoms.size();
	int npairs = pairs.size();

	if(U1ij) potential = 1;
	init(natoms, npairs, nm1, nm2, nm2, potential, flag_ave, flag_rigid);

	int nval = natoms;
	if(flag_rigid==1) nval += 10;
	double *anal = new double[nval];

 	initKconst(atoms, pairs, Uij, U1ij, Uijk, Uijkl, hessian, eigVal, eigVect, flag_ave, flag_rigid);
	memset(rigid, 0, 10*sizeof(double));


	lambda = 4.0;
	lambda_ent = 1.0;
	if(flag_ent==1) {
		lambda = 4.0;
		lambda_ent = 0.0;
	}

	diagHessian(atoms, pairs, Uij, U1ij, Uijk, Uijkl, hessian, eigVal, eigVect, flag_ave);

	eneAndDer(atoms, pairs, flag_rigid, Uij, U1ij, Uijk, Uijkl, eigVal, eigVect, bfact, 
		flag_ave, flag_ent, nthreads);


	int inc = 1;
	dcopy_(&nval, G, &inc, anal, &inc);

	double dx=1.e-5;

	double *num = new double[nval];
	double dist1, dist1_ent, dist2, dist2_ent;
	for(int i = 0; i < natoms; i++) {
		X[i] += dx;
		diagHessian(atoms, pairs, Uij, U1ij, Uijk, Uijkl, hessian, eigVal, eigVect, flag_ave);
		dist1 = energy(atoms, flag_rigid, eigVal, eigVect, bfact);
		dist1_ent = entropy(eigVal);
		dist1 = lambda*dist1 + lambda_ent*dist1_ent;
		X[i] -= 2*dx;
		diagHessian(atoms, pairs, Uij, U1ij, Uijk, Uijkl, hessian, eigVal, eigVect, flag_ave);
		dist2 = energy(atoms, flag_rigid, eigVal, eigVect, bfact);
		dist2_ent = entropy(eigVal);
		dist2 = lambda*dist2 + lambda_ent*dist2_ent;
		X[i] += dx;
		num[i] = (dist1-dist2)/(2*dx);
	}

	for(int i = 0; i < natoms; i++) {
		std::cout << "i = " << i << " anal = " << anal[i] << " num[i] = " << num[i] << std::endl;
	}

	diagHessian(atoms, pairs, Uij, U1ij, Uijk, Uijkl, hessian, eigVal, eigVect, flag_ave);
	if(flag_rigid==1) {
		for(int i = 0; i < 10; i++)
		{
			rigid[i] += dx;
			dist1 = energy(atoms, flag_rigid, eigVal, eigVect, bfact);
			rigid[i] -= 2*dx;
			dist2 = energy(atoms, flag_rigid, eigVal, eigVect, bfact);
			num[i+natoms] = (dist1-dist2)/(2*dx);
			rigid[i] += dx;
		}
	}
	for(int i = 0; i < 10; i++) {
		std::cout << "i = " << i << " anal = " << anal[i+natoms] << " num[i] = " << num[i+natoms] << std::endl;
	}

	exit(1);

   }

  /* ==============================================================================================
	Optimization of kconst with lbfgsb
   =============================================================================================== */

   double OptimBfactor_ki::optimkval(std::vector<Atoms>& atoms, std::vector<Links>& pairs, int nm1,
		int nm2, int nm3, double *Uij, double *U1ij, double *Uijk, double *Uijkl, double *hessian,
		double *eigVal, double *eigVect, double *bfact, int flag_ave, int flag_ent, int flag_rigid,
		double *rig, int nthreads)
   {

	int natoms = atoms.size();
	int npairs = pairs.size();

	if(U1ij) potential = 1;
	init(natoms, npairs, nm1, nm2, nm3, potential, flag_ave, flag_rigid);

 	initKconst(atoms, pairs, Uij, U1ij, Uijk, Uijkl, hessian, eigVal, eigVect, flag_ave, flag_rigid);

	int niter = 2000;
	if(flag_ent==2) niter=200;
	int IFLAG = 0;
	double chi2;

	lambda = 4.0;
	lambda_ent = 1.0;
	if(flag_ent==1) {
		lambda = 1.0;
		lambda_ent = 0;
	} 

	do {
		IFLAG = 0;
		for(int i = 0; i < niter; i++) {
			chi2 = oneStep_LBFGSB(atoms, pairs, Uij, U1ij, Uijk, Uijkl, hessian, eigVal, eigVect, 
			bfact, &IFLAG, i, flag_ave, flag_ent, flag_rigid, nthreads);
			if(IFLAG==0 || IFLAG == 1) break;
			if(chi2 < 1.e-4) break;
		}
		lambda = 2*lambda;
	} while (chi2 > 0.03 && flag_ent == 2);

	if(flag_rigid==1) for(int k = 0; k < 10; k++) rig[k] = X[natoms+k];

	return chi2;

  }

#endif
