/* ====YAMABEFLOWIDT.H============================================================================
 *
 * Author: Patrice Koehl, June 2019
 * Department of Computer Science
 * University of California, Davis
 *
 * This file applies a variational method for computing a conformal map from a genus zero
 * surface onto the sphere
 *
 * It is based on the paper:
 *      B. Springborn, P. Schroeder, and U. Pinkall. "Conformal equivalence of triangle meshes",
 *      ACM Trans. Graph. 27, 3, Article 77 (2008).
 =============================================================================================== */

#pragma once

  /* ===== INCLUDES AND TYPEDEFS =================================================================
   *
   * Third party libraries included: Eigen
   *
   =============================================================================================== */

  #define _USE_MATH_DEFINES // for M_PI

  #include <vector>
  #include <cmath>

  #include "iDT.h"
  #include "Layout2D.h"

  #include <Eigen/SparseLU>

  typedef Eigen::Triplet<double> Triplet;
  Eigen::SimplicialLDLT<Eigen::SparseMatrix<double>> solverYD;
  int nYD = 0;


  /* =============================================================================================
     Define class
   =============================================================================================== */

  class YamabeIDT{

  public:
	// select vertex from genus 0 surface that will be remove temperarily
	VertexIter pickPole(Mesh& mesh, int type);

	// Removes a vertex from the mesh and set the tags.
	void punctureMesh(Mesh& mesh, VertexIter pole);
	
	// Performs Yamabe flow to plane with preprocessing IDT
	VertexIter yamabeFlowIDT(Mesh& mesh, int vtype, bool *SUCCESS);

  private:
 
	// init Hessian
	void initHessian(Mesh& mesh);

	// Compute Hessian and right hand side
	void computeHessian(Mesh& mesh);

	// Solve Jacobian linear system
	bool solveSystem(Mesh& mesh);

	// compute Lengths
	void computeLength0(Mesh& mesh);

 	// Init radius for all vertices
	void initU(Mesh& mesh);

 	// Compute new lengths for all edges
	void computeLength(Mesh& mesh);

 	// Compute curvatures using edge lengths
	double computeCurv(Mesh& mesh, double *maxK, int *nbad, std::vector<EdgeIter>& list_bad);

	// Projection to sphere
	void stereo2Sphere(Mesh& mesh);

	// Select which edge of a triangle to flip
	EdgeIter pickEdge(EdgeIter e1, EdgeIter e2, EdgeIter e3, int *ierr);

	// flip Mesh
	void flipMesh(Mesh& mesh, std::vector<EdgeIter>& list_bad);

   protected:

	Eigen::SparseMatrix<double> H;
	Eigen::VectorXd B, Sol;

	double *V_curv, *U_fact, *U_keep;
  };

  /* ===== PICK VERTEX    ========================================================================
   * Select vertex that will be removed from the mesh to enable planar Tutte embedding.
   * At this stage, selection is based on finding the vertex with either the highest valence
   * (type = 1), or the lowest valence (type = 2). If type is set to 0, vertex # 0 is picked
   * arbitrarily
   =============================================================================================== */

  VertexIter YamabeIDT::pickPole(Mesh& mesh, int type)
  {
	if(type==0) return mesh.vertices.begin();

	/* Find vertex with largest, or smallest valence */

	int valence;
	int val_ref = mesh.vertices.begin()->degree();
	VertexIter pole = mesh.vertices.begin();

	for (VertexIter v = mesh.vertices.begin(); v != mesh.vertices.end(); v++)
	{
		valence = v->degree();

		if(type==2) {
			if(valence < val_ref) {
				val_ref = valence;
				pole = v;
			}
		} else if(type==1) {
			if(valence > val_ref) {
				val_ref = valence;
				pole = v;
			}
		}
	}

	return pole;
  }

  /* ===== PUNCTURE MESH  ========================================================================
   * Remove vertex VR.  All incident
   * vertices and faces  are set to inNorthPoleVicinity
   =============================================================================================== */

  void YamabeIDT::punctureMesh(Mesh& mesh, VertexIter pole)
  {

	for (VertexIter v = mesh.vertices.begin(); v != mesh.vertices.end(); v++)
	{
		v->NorthPole = false;
		v->inNorthPoleVicinity = false;
	}

	int idx = 0;
	pole->NorthPole = true;
	HalfEdgeIter he = pole->he;
	do {
		he->face->inNorthPoleVicinity = true;
		he->flip->vertex->indexN = idx;
		he->flip->vertex->inNorthPoleVicinity = true;
		idx++;

		he = he->flip->next;
	} while (he != pole->he);

	idx=0;
	for (VertexIter v = mesh.vertices.begin(); v != mesh.vertices.end(); v++)
	{
		bool b1 = v->NorthPole;
		bool b2 = v->inNorthPoleVicinity;

		if(!b1 && !b2) {
			v->indexN = idx;
			idx++;
		}
	}
  }

  /* ===== Stereo2Sphere   ============================================================================
   * Project planar mesh onto unit sphere
   ==================================================================================================*/

  void YamabeIDT::stereo2Sphere(Mesh& mesh)
  {

	double U, V, val, den;
	int idx;

        for (VertexIter v = mesh.vertices.begin(); v != mesh.vertices.end(); v++)
	{
		bool b1 = v->NorthPole;
		idx = v->index;

		if(b1) {
			v->position2.x = 0.;
			v->position2.y = 0.;
			v->position2.z = -1.;
		} else {


			U = v->position2.x;
			V = v->position2.y;

			val = U*U + V*V;
			den = 1.0/(1.0+val);

			v->position2.x = 2.0*U*den;
			v->position2.y = 2.0*V*den;
			v->position2.z = (1.0-val)*den;

		}
	}
  }


  /* ===============================================================================================
   computeLength0: computes initial length of all edges
   =============================================================================================== */

  void YamabeIDT::computeLength0(Mesh& mesh)
  {

	for(EdgeIter e = mesh.edges.begin(); e != mesh.edges.end(); e++)
	{
		HalfEdgeIter h = e->he;
		Vector p0 = h->vertex->position;
		Vector p1 = h->flip->vertex->position;
		double length = (p0-p1).norm();
		e->length = length;
	}
  }
  /* =============================================================================================
   Init Conformal factors
   =============================================================================================== */

  void YamabeIDT::initU(Mesh& mesh)
  {
	int n_vertices = mesh.vertices.size();
	for(int i = 0; i < n_vertices; i++) U_fact[i] = 0.0;
  }

  /* ===============================================================================================
   YamabeNewLength: compute edge length based on new conformal factors
   =============================================================================================== */

  void YamabeIDT::computeLength(Mesh& mesh) 
  {

	int idx;
	double length0, length;
	double u0, u1;
	VertexIter v0, v1;

	for(EdgeIter e_iter = mesh.edges.begin(); e_iter != mesh.edges.end(); e_iter++)
	{
		idx = e_iter->index;
		v0 = e_iter->he->vertex;
		v1 = e_iter->he->flip->vertex;

		u0 = U_fact[v0->index];
		u1 = U_fact[v1->index];
//		u0 = std::max(-50.0, std::min(50.0, u0));
//		u1 = std::max(-50.0, std::min(50.0, u1));

		Vector p0 = e_iter->he->vertex->position;
		Vector p1 = e_iter->he->flip->vertex->position;
		length0 = (p0-p1).norm();
		length = length0*std::exp((u0 + u1) / 2.0);
		e_iter->length = length;
	}
   }


  /* ===============================================================================================
   pickEdge: select the largest flippable edge of a face
   =============================================================================================== */

 EdgeIter YamabeIDT::pickEdge(EdgeIter e1, EdgeIter e2, EdgeIter e3, int *ierr)
 {

	*ierr = 0;

	double lAB, lBC, lCA;
	lAB = e1->length;
	lBC = e2->length;
	lCA = e3->length;

	bool b1 = IDT::isFlippable(e1);
	bool b2 = IDT::isFlippable(e2);
	bool b3 = IDT::isFlippable(e3);

	int n=0;
	if(b1) n++; if(b2) n++; if(b3) n++;

	if(n==3) {

		if(lAB > std::max(lBC, lCA)) {
			return e1;
		} else {
			if(lBC > std::max(lAB, lCA)) {
				return e2;
			} else {
				return e3;
			}
		}

	} else if (n==2) {

		if(!b1) {
			if(lBC>lCA) {
				return e2;
			} else {
				return e3;
			}
		} else if(!b2) {
			if(lAB > lCA) {
				return e1;
			} else {
				return e3;
			}
		} else {
			if(lAB > lBC) {
				return e1;
			} else {
				return e2;
			}
		}
	} else if(n==1) {
		if(b1) {
			return e1;
		} else if(b2) {
			return e2;
		} else {
			return e3;
		}
	} else {
		*ierr = 1;
		return e1;
	}

	*ierr = 1;
	return e1;
 }

  /* ===============================================================================================
   flipMesh: flip "bad" triangles
   =============================================================================================== */

  void YamabeIDT::flipMesh(Mesh& mesh, std::vector<EdgeIter>& list_bad)
  {

	for(int i = 0; i < list_bad.size(); i++)
	{
		EdgeIter e = list_bad[i];
//		if(e->he->vertex->inNorthPoleVicinity && e->he->flip->vertex->inNorthPoleVicinity) continue;
		if(e->he->vertex->degree() == 3 || e->he->flip->vertex->degree() == 3) continue;
		IDT::flipEdge(e);
	}
  }

  /* ===============================================================================================
   computeCurv: computes the curvatures of all vertices

   The curvature is defined as the excess angle at each vertex i, i.e. to
   2pi - sum_t Theta(i,ijk)
   where sum_t indicates a sum over all triangles (ijk) that are incident to i
         Theta(i,ijk) is the angle at vertex i in the triangle (i,jk)
   =============================================================================================== */

  double YamabeIDT::computeCurv(Mesh& mesh, double *maxK, int *nbad, std::vector<EdgeIter>& list_bad)
  {
	double TwoPI = 2*M_PI;

/* 	==========================================================================================
	Initialize Curvature to 2*PI
        ========================================================================================== */

	HalfEdgeIter hAB, hBC, hCA;
	EdgeIter eAB, eBC, eCA;
	VertexCIter va, vb, vc;
	double lAB, lCA, lBC;

	int idxa, idxb, idxc;

	for (VertexCIter v = mesh.vertices.begin(); v != mesh.vertices.end(); v++)
	{
		idxa = v->index;
		V_curv[idxa] = TwoPI;
	}

/* 	==========================================================================================
	Iterate over all triangles and remove angle for each vertex
        ========================================================================================== */

	int nb = 0;
	int ierr;
	list_bad.clear();
	int n;
	double val, alpha_A, alpha_B, alpha_C;

	for (FaceCIter f = mesh.faces.begin(); f != mesh.faces.end(); f++)
	{
		hAB = f->he;
		hBC = hAB->next;
		hCA = hBC->next;
		
		va = hAB->vertex;
		vb = hBC->vertex;
		vc = hCA->vertex;
		idxa = va->index;
		idxb = vb->index;
		idxc = vc->index;

		lAB = hAB->edge->length;
		lBC = hBC->edge->length;
		lCA = hCA->edge->length;

		n = 0;
		if( lAB > lBC + lCA) {
			alpha_A = 0; alpha_B = 0; alpha_C = M_PI;
			n++;
		} else if (lBC > lAB + lCA) {
			alpha_A = M_PI; alpha_B = 0; alpha_C = 0;
			n++;
		} else if (lCA > lAB + lBC) {
			alpha_A = 0; alpha_B = M_PI; alpha_C = 0;
			n++;
		} else {
			val = ((lBC + lAB-lCA)*(lBC-lAB+lCA))/((lAB+lBC+lCA)*(-lBC+lAB+lCA));
			val = std::sqrt(val);
			alpha_A = 2.0*atan(val);
			val = ((lCA + lAB-lBC)*(lCA-lAB+lBC))/((lAB+lBC+lCA)*(-lCA+lAB+lBC));
			val = std::sqrt(val);
			alpha_B = 2.0*atan(val);
			val = ((lAB + lBC-lCA)*(lAB-lBC+lCA))/((lAB+lBC+lCA)*(-lAB+lBC+lCA));
			val = std::sqrt(val);
			alpha_C = 2.0*atan(val);
		}

		V_curv[idxa] -= alpha_A;
		V_curv[idxb] -= alpha_B;
		V_curv[idxc] -= alpha_C;
		nb += n;

		if(n > 0) {
			EdgeIter h=pickEdge(hAB->edge, hBC->edge, hCA->edge, &ierr);
			if(ierr==0) list_bad.push_back(h);
		}
	}

	double err=0, errM=0;
	for (VertexCIter v = mesh.vertices.begin(); v != mesh.vertices.end(); v++)
	{
		idxa = v->index;
		if(!v->inNorthPoleVicinity && !v->NorthPole) {
			val = V_curv[idxa];
			err += val*val;
			errM = std::max(errM, std::abs(val));
		}
	}
	err = std::sqrt(err);
	*maxK = errM;
	*nbad = nb;
	return err;

  }

  /* =============================================================================================
   Init Hessian matrix
   =============================================================================================== */

  void YamabeIDT::initHessian(Mesh& mesh)
  {

	H.setZero();

  /* ==================================================================================================
	Initialize all values for regular constraints to 0
   ==================================================================================================*/

	int idx, jdx;
	std::vector<Triplet> Mat_coefficients;
	double zero = 0;
	bool b1, b2, b3, b4;

	Mat_coefficients.clear();

	for (VertexIter v = mesh.vertices.begin(); v != mesh.vertices.end(); v++)
	{
		b1 = v->NorthPole;
		b2 = v->inNorthPoleVicinity;
		if(!b1 && !b2) {
			idx = v->indexN;
			Mat_coefficients.push_back(Triplet(idx, idx, zero));
			B[idx] = 0.0;
		}
	}

	HalfEdgeIter he, he2;
	VertexIter v_i, v_j;
	
	for (EdgeIter e = mesh.edges.begin(); e != mesh.edges.end(); e++)
	{
		he = e->he;
		he2 = he->flip;

		v_i = he->vertex;
		v_j = he2->vertex;

		b1 = v_i->NorthPole;
		b2 = v_i->inNorthPoleVicinity;
		b3 = v_j->NorthPole;
		b4 = v_j->inNorthPoleVicinity;

		if(!b1 && !b2 && !b3 && !b4) {
			idx = v_i->indexN;
			jdx = v_j->indexN;
			Mat_coefficients.push_back(Triplet(idx, jdx, zero));
			Mat_coefficients.push_back(Triplet(jdx, idx, zero));
		}
	}

	H.setFromTriplets(Mat_coefficients.begin(), Mat_coefficients.end());

  }
  /* ===== Jacobian system          =================================================================
	Hessian 
   ==================================================================================================*/

  void YamabeIDT::computeHessian(Mesh& mesh)
  {

	initHessian(mesh);

	double lAB, lBC, lCA;
	int idxA, idxB, idxC;

	double TwoPI = 2*M_PI;

	double val, alpha_A, alpha_B, alpha_C;
	double cotan_A, cotan_B, cotan_C;
	bool b1, b2, b3;

	HalfEdgeIter hAB, hBC, hCA;
	EdgeIter e1, eAB, eBC, eCA;
	VertexIter v1, v2, v3;

	for (VertexIter v_it = mesh.vertices.begin(); v_it != mesh.vertices.end(); v_it++)
	{
		if(!v_it->inNorthPoleVicinity && !v_it->NorthPole)
		{
			idxA = v_it->indexN;
			B[idxA] = -TwoPI;
		}
	}

	for(FaceIter f_iter = mesh.faces.begin(); f_iter != mesh.faces.end(); f_iter++)
	{
		hAB =f_iter->he;
		hBC =hAB->next;
		hCA =hBC->next;

		v1 = hAB->vertex;
		v2 = hBC->vertex;
		v3 = hCA->vertex;

		b1 = v1->inNorthPoleVicinity || v1->NorthPole;
		b2 = v2->inNorthPoleVicinity || v2->NorthPole;
		b3 = v3->inNorthPoleVicinity || v3->NorthPole;

		eAB = hAB->edge;
		eBC = hBC->edge;
		eCA = hCA->edge;

		idxA = v1->indexN;
		idxB = v2->indexN;
		idxC = v3->indexN;

		lAB = eAB->length;
		lBC = eBC->length;
		lCA = eCA->length;


		if( lAB > lBC + lCA) {
			alpha_A = 0; alpha_B = 0; alpha_C = M_PI;
			cotan_A = 0; cotan_B = 0; cotan_C = 0;
		} else if (lBC > lAB + lCA) {
			alpha_A = M_PI; alpha_B = 0; alpha_C = 0;
			cotan_A = 0; cotan_B = 0; cotan_C = 0;
		} else if (lCA > lAB + lBC) {
			alpha_A = 0; alpha_B = M_PI; alpha_C = 0;
			cotan_A = 0; cotan_B = 0; cotan_C = 0;
		} else {
			val = ((lBC + lAB-lCA)*(lBC-lAB+lCA))/((lAB+lBC+lCA)*(-lBC+lAB+lCA));
			val = std::sqrt(val);
			alpha_A = 2.0*atan(val); cotan_A = (1-val*val)/(2*val);
			val = ((lCA + lAB-lBC)*(lCA-lAB+lBC))/((lAB+lBC+lCA)*(-lCA+lAB+lBC));
			val = std::sqrt(val);
			alpha_B = 2.0*atan(val); cotan_B = (1-val*val)/(2*val);
			val = ((lAB + lBC-lCA)*(lAB-lBC+lCA))/((lAB+lBC+lCA)*(-lAB+lBC+lCA));
			val = std::sqrt(val);
			alpha_C = 2.0*atan(val); cotan_C = (1-val*val)/(2*val);
		}
		if(std::isnan(alpha_A) || std::isnan(alpha_B) || std::isnan(alpha_C)) {
			std::cout << "Problem with angle for face # : " << f_iter->index << std::endl;
		}
		if(std::isnan(cotan_A) || std::isnan(cotan_B) || std::isnan(cotan_C)) {
			std::cout << "Problem with cotan for face # : " << f_iter->index << std::endl;
		}

		if(!b1) {
			B[idxA] += alpha_A;
			H.coeffRef(idxA, idxA) -= cotan_B+cotan_C;
		}
		if(!b2) {
			B[idxB] += alpha_B;
			H.coeffRef(idxB, idxB) -= cotan_A+cotan_C;
		}
		if(!b3) {
			B[idxC] += alpha_C;
			H.coeffRef(idxC, idxC) -= cotan_A+cotan_B;
		}

		if( !b1 && !b2 ) {
			H.coeffRef(idxA, idxB) += cotan_C;
			H.coeffRef(idxB, idxA) += cotan_C;
		}

		if( !b1 && !b3 ) {
			H.coeffRef(idxA, idxC) += cotan_B;
			H.coeffRef(idxC, idxA) += cotan_B;
		}

		if( !b2 && !b3 ) {
			H.coeffRef(idxB, idxC) += cotan_A;
			H.coeffRef(idxC, idxB) += cotan_A;
		}
	}
	H = 0.25*H;
	B = -0.5*B;

	double eps = 1.e-8;
	for(VertexIter v = mesh.vertices.begin(); v != mesh.vertices.end(); v++) {
		b1 = v->inNorthPoleVicinity || v->NorthPole;
		idxA = v->indexN;
		if(!b1) {
			H.coeffRef(idxA, idxA) += eps;
		}
	}
  }

  /* =============================================================================================
  Solve Linear System
   =============================================================================================== */

  bool YamabeIDT::solveSystem(Mesh& mesh)
  {

  /* ==================================================================================================
	Compute Hessian and gradient
   ==================================================================================================*/

	computeHessian(mesh);

  /* ==================================================================================================
	Solve system
   ==================================================================================================*/

        solverYD.analyzePattern(H);
	solverYD.factorize(H);

	double TOL = 1.e-8;

	if(solverYD.info()==Eigen::NumericalIssue)
	{
		std::cout << "Problem with Cholesky factorization!!" << std::endl;
		return false;
	}
	if(solverYD.info()!=Eigen::Success)
	{
		std::cout << "Problem with Cholesky factorization!!" << std::endl;
		return false;
	}
	Sol = solverYD.solve(B); 
	if(solverYD.info()!=Eigen::Success)
	{
		std::cout << "Problem with Cholesky solver!!" << std::endl;
		return false;
	}
	double err = (H*Sol - B).norm();

	if(err > TOL || std::isnan(err) ) return false;

	return true;

  }

  /* ===== MinimizeYamabe  ===========================================================================
   *
   ==================================================================================================*/

  VertexIter YamabeIDT::yamabeFlowIDT(Mesh& mesh, int vtype, bool *SUCCESS)
  {

	Mesh *model = new Mesh(mesh);
	Mesh mesh2 = *model;

	int niter_max = 300;
	double TOL = 1.e-10;

	*SUCCESS = true;

	int n_vertices = mesh.vertices.size();

  /* ==================================================================================================
        Compute initial edge lengths
   ==================================================================================================*/

	computeLength0(mesh2);

  /* ==================================================================================================
        apply iDT on initial mesh
   ==================================================================================================*/

	IDT::iDT(mesh2);

  /* ==================================================================================================
        Puncture Mesh
   ==================================================================================================*/

        VertexIter pole = pickPole(mesh2, vtype);
        punctureMesh(mesh2, pole);

  /* ==================================================================================================
	Count number of active vertices
   ==================================================================================================*/

	int idx;
	int ntot = 0;
	for (VertexIter v_it = mesh2.vertices.begin(); v_it != mesh2.vertices.end(); v_it++)
	{
		if(!v_it->inNorthPoleVicinity && !v_it->NorthPole) ntot++;
	}

  /* ==================================================================================================
	Create Eigen matrices and arrays needed to minimize energy
   ==================================================================================================*/

	H.resize(ntot, ntot);
	B.resize(ntot);
	Sol.resize(ntot);

	U_fact    = new double[n_vertices];
	U_keep    = new double[n_vertices];
	V_curv    = new double[n_vertices];

	H.setZero();


  /* ==================================================================================================
	Initialize Yamabe discrete metric
   ==================================================================================================*/

	initU(mesh2);

  /* ==================================================================================================
	Initial energy that will be refined
   ==================================================================================================*/

	double ene0, ene, ene1;
	int nbad;
	std::vector<EdgeIter> list_bad;
	computeLength(mesh2);
	ene = computeCurv(mesh2, &ene0, &nbad, list_bad);

  /* ==================================================================================================
	Now perform refinement
   ==================================================================================================*/

	double step = 0;
	double u;
	int zero = 0;

	std::cout << " " << std::endl;
	std::cout << "Minimize Yamabe functional (with IDT):" << std::endl;
	std::cout << "======================================" << std::endl;
	std::cout << " " << std::endl;

	std::cout << "        " << "===========================================================================" << std::endl;
	std::cout << "        " << "       Iter       Step size          Max curv.     Bad triangles           " << std::endl;
        std::cout << "        " << "===========================================================================" << std::endl;
        std::cout << "        " << "   " << std::setw(8)<< zero << "    " << std::setw(12) << step;
	std::cout << "    " << std::setw(12) << ene0 ;
	std::cout << "      " << std::setw(8) << zero << std::endl;

	std::vector<double> save_length;

	bool info;
	for(int niter = 0; niter < niter_max; niter++)
	{

		save_length.clear();
		for(EdgeIter e = mesh2.edges.begin(); e != mesh2.edges.end(); e++) save_length.push_back(e->length);

		for(int i = 0; i < n_vertices; i++) U_keep[i] = U_fact[i];

  		/* ====================================================================================
		Solve Jacobian system
   		======================================================================================*/

		info = solveSystem(mesh2);
		if(!info) break;

  		/* ====================================================================================
		Find "optimal" step size:
			- try first step = 1; if corresponding energy ene1 < ene0, apply step
			- if ene1 > ene0, set step = 0.5*step and try again
   		======================================================================================*/

		step = 1.0;
		for (VertexIter v_it = mesh2.vertices.begin(); v_it != mesh2.vertices.end(); v_it++)
		{
			idx = v_it->index;
			if(!v_it->inNorthPoleVicinity && !v_it->NorthPole) {
				u = U_keep[idx] + step*Sol[v_it->indexN]; 
				U_fact[idx] = u;
			}
		}
		computeLength(mesh2);
		ene = computeCurv(mesh2, &ene1, &nbad, list_bad);

	for(EdgeIter e_iter = mesh.edges.begin(); e_iter != mesh.edges.end(); e_iter++)
	{
		if(std::isnan(e_iter->length)) {
			std::cout << "Problem with edge : " << e_iter->index << std::endl;
		}
	}

		double step_min = 1.0;

		while (((ene1 > ene0) || (nbad > 0)) && (step > step_min)) {
			step = 0.5*step;
			for (VertexIter v_it = mesh2.vertices.begin(); v_it != mesh2.vertices.end(); v_it++)
			{
				idx = v_it->index;
				if(!v_it->inNorthPoleVicinity && !v_it->NorthPole) {
					u = U_keep[idx] + step*Sol[v_it->indexN]; 
					U_fact[idx] = u;
				}
			}
			computeLength(mesh2);
			ene = computeCurv(mesh2, &ene1, &nbad, list_bad);
		}

        	std::cout << "        " << "   " << std::setw(8)<< niter+1 << "    " << std::setw(12) << step;
		std::cout << "    " << std::setw(12) << ene1 ;
		std::cout << "      " << std::setw(8) << nbad << std::endl;

		if(std::abs(ene1 - ene0) < TOL*ene1 && ene1 < TOL) break;
		if(ene1 < TOL && nbad == 0) break;
		ene0 = ene1;

		if(nbad > 0) {
			int ie = 0;
			for(EdgeIter e = mesh2.edges.begin(); e != mesh2.edges.end(); e++) {
				e->length = save_length[ie];
				ie++;
			}
			for(int i = 0; i < n_vertices; i++) U_fact[i] = U_keep[i];
			flipMesh(mesh2, list_bad);
		}
	for(EdgeIter e_iter = mesh.edges.begin(); e_iter != mesh.edges.end(); e_iter++)
	{
		if(std::isnan(e_iter->length)) {
			std::cout << "Problem with edge : " << e_iter->index << std::endl;
		}
	}
	}
        std::cout << "        " << "===========================================================================" << std::endl;
	std::cout << " " << std::endl;

	if(std::abs(ene1) > TOL || nbad > 0 || !info) {
		std::cout << " " << std::endl;
		std::cout << "Yamabe flow has failed!" << std::endl;
		std::cout << " " << std::endl;
		*SUCCESS = false;
		return pole;
	}

  /* ==================================================================================================
	Now layout on plane, and project onto sphere
   ==================================================================================================*/

	info = Layout::layout2D(mesh2);

	if(!info) {
		*SUCCESS = false;
		return pole;
	}
	std::vector<Vector> positions;
	for(VertexIter v2 = mesh2.vertices.begin() ; v2 != mesh2.vertices.end(); v2++) {
		positions.push_back(v2->position2);
	}			
	int i=0;
	for(VertexIter v = mesh.vertices.begin() ; v != mesh.vertices.end(); v++) {
		v->position2 = positions[i];
		i++;
	}
	positions.clear();

	// Project onto sphere
	stereo2Sphere(mesh);

	// restore vertex star
	pole->NorthPole = false;
	HalfEdgeIter he = pole->he;
	do {
		he->face->inNorthPoleVicinity = false;

		he = he->flip->next;
	} while (he != pole->he);


	return pole;

  }

