#pragma once

#include <fstream>
#include <sstream>
#include <tuple>

#include "MeshIO.h"

class AdjacencyTable {
public:
	// constructs table
	void construct(int n, const std::vector<int>& indices);

	// returns unique index corresponding to entry (i, j)
	int getIndex(int i, int j) const;

	// returns table size
	int getSize() const;

private:
	// members
	std::vector<std::set<int> > data;
	std::vector<int> iMap;
	int size;
};

class Soup {
public:
        std::vector<Vector> positions;
        std::vector<Vector> positions2;
        std::vector<int> indices;
        AdjacencyTable table; // construct after filling positions and indices
};

class FixSoup {
public:
	// remove small faces from soup
	static void collapseSmallFaces(Soup& soup);

private:
	// total area of "soup"
	static double soupArea(Soup& soup);

	// detect smallest face
	static void smallestFace(Soup& soup, int *iface, double *amin);

	// make hole in mesh when removing a face
	static void makeHole(Soup& soup, std::vector<int> indices_save, std::vector<Vector> positions_save, 
	std::vector<Vector> positions2_save, int iface, std::vector<int>& border_list);

	// Sanity check: make sure mesh is still genus 0....
	static bool sanityCheck(Soup& soup);

	// find face in mesh adjacent to a given edge
	static void findFace(Soup& soup, int i0, int j0, int *iface, int *orient);

	// find if a face exists
	static bool isFace(Soup& soup, int i0, int j0, int k0);

	// find normal of polygon formed by the hole
	static Vector holeNormal(Soup& soup, std::vector<int> hole_list);

	// check if a vertex along the polygon is convex (i.e. define an ear)
	static bool isConvex(Soup& soup, std::vector<int> hole_list, int idx, Vector Normal);

	// check if the triangle formed by an ear crosses any edge along the polygon
	static bool crossEdge(Soup& soup, std::vector<int> hole_list, int idx, Vector Normal);

	// check if a triangle is crossed by an edge
	static bool checkTrig(Vector t0, Vector t1, Vector t2, Vector e0, Vector e1);

	// check if a triangle is crossed by an edge
	static void earClipping(Soup& soup, std::vector<int> hole_list);
};

void AdjacencyTable::construct(int n, const std::vector<int>& indices)
{
	data.resize(n);
	iMap.resize(n);
	size = 0;

	// build table
	for (int I = 0; I < (int)indices.size(); I += 3) {
		for (int J = 0; J < 3; J++) {
			int K = (J + 1) % 3;
			int i = indices[I + J];
			int j = indices[I + K];

			if (i > j) std::swap(i, j);
			data[i].insert(j);
		}
	}

	// build iMap
	for (int i = 0; i < n; i++) {
		iMap[i] = size;
		size += data[i].size();
	}
}

int AdjacencyTable::getIndex(int i, int j) const
{
	if (i > j) std::swap(i, j);

	int k = 0;
	for (std::set<int>::iterator it = data[i].begin(); it != data[i].end(); it++) {
		if (*it == j) break;
		k++;
	}

	return iMap[i] + k;
}

int AdjacencyTable::getSize() const
{
	return size;
}

double FixSoup::soupArea(Soup& soup)
{
	int i0, j0, k0;
	Vector p0, p1, p2;
	double Atot=0;
	for(int f = 0; f < soup.indices.size()/3; f++) 
	{
		i0 = soup.indices[3*f];
		j0 = soup.indices[3*f+1];
		k0 = soup.indices[3*f+2];
		p0 = soup.positions[i0];
		p1 = soup.positions[j0];
		p2 = soup.positions[k0];
		Atot += 0.5*(cross(p0-p1,p0-p2).norm());
	}
	return Atot;
}

void FixSoup::smallestFace(Soup& soup, int *iface, double *areamin)
{
	int i0, j0, k0;
	Vector p0, p1, p2;
	double area;
	double amin = 0;
	int fmin=0;
	int idx=0;
	for(int f = 0; f < soup.indices.size()/3; f++) 
	{
		i0 = soup.indices[3*f];
		j0 = soup.indices[3*f+1];
		k0 = soup.indices[3*f+2];
		p0 = soup.positions[i0];
		p1 = soup.positions[j0];
		p2 = soup.positions[k0];
		area = 0.5*(cross(p0-p1,p0-p2).norm());
		if(idx==0) {
			fmin = 0; amin = area;
		} else {
			if(area < amin) {
				fmin = f; amin = area;
			}
		}
		idx++;
	}
	*iface = fmin;
	*areamin = amin;
}

void FixSoup::makeHole(Soup& soup, std::vector<int> indices_save, std::vector<Vector> positions_save, 
	std::vector<Vector> positions2_save, int iface, std::vector<int>& border_list)
{
	std::vector<int> corresp;
	std::vector<int> border;
	std::vector<int> border_flag;

	int nvertices = positions_save.size();
	int nfaces    = indices_save.size()/3;

	int i0, j0, k0, i1, j1, k1;

	i0 = soup.indices[3*iface]; j0 = soup.indices[3*iface+1]; k0 = soup.indices[3*iface+2];
	i1 = std::min(i0, std::min(j0, k0)); k1 = std::max(i0, std::max(j0, k0)); j1 = i0+j0+k0-i1-k1;

	// Recreate vertex soup, removing vertices j1 and k1
	int idx = 0;
	corresp.clear();
	for(int i = 0; i < nvertices; i++) {
		if(i!= i1 && i!= j1 && i!= k1) {
			corresp.push_back(idx);
			soup.positions.push_back(positions_save[i]);
			soup.positions2.push_back(positions2_save[i]);
			idx++;
		} else {
			corresp.push_back(-1);
		}
	}

	// Find border edges
	border.clear();
	for(int f = 0; f < nfaces; f++) {
		i0 = indices_save[3*f];
		j0 = indices_save[3*f+1];
		k0 = indices_save[3*f+2];
		int nmatch = 0;
		if(corresp[i0] < 0) nmatch++;
		if(corresp[j0] < 0) nmatch++;
		if(corresp[k0] < 0) nmatch++;
		if(nmatch==0) {
			soup.indices.push_back(corresp[i0]);
			soup.indices.push_back(corresp[j0]);
			soup.indices.push_back(corresp[k0]);
		} else if(nmatch==1) {
			if(corresp[i0] < 0) {
				border.push_back(corresp[j0]); border.push_back(corresp[k0]);
			} else if (corresp[j0] < 0) {
				border.push_back(corresp[i0]); border.push_back(corresp[k0]);
			} else {
				border.push_back(corresp[i0]); border.push_back(corresp[j0]);
			}
		}
	}
	int nborder_edge = border.size()/2;
	border_list.clear();
	border_list.push_back(border[0]);
	border_list.push_back(border[1]);
	border_flag.clear();
	for(int b = 0; b < nborder_edge; b++) border_flag.push_back(1);
	border_flag[0] = 0;
	int npoint_border = 2;
	while(npoint_border < nborder_edge) {
		for(int b = 0; b < nborder_edge; b++) 
		{
			if(border_flag[b]==0) continue;
			if(border[2*b] == border_list[npoint_border-1]) {
				border_list.push_back(border[2*b+1]);
				npoint_border++;
				border_flag[b] = 0;
				break;
			} else if(border[2*b+1] == border_list[npoint_border-1]) {
				border_list.push_back(border[2*b]);
				npoint_border++;
				border_flag[b] = 0;
				break;
			}
		}
	}
}

bool FixSoup::isFace(Soup& soup, int i0, int j0, int k0)
{
	int i1, j1, k1;
	i1 = std::min(i0, std::min(j0, k0)); k1 = std::max(i0, std::max(j0, k0)); j1 = i0+j0+k0-i1-k1;
	int i2, j2, k2, i3, j3, k3;
	for (int f = 0; f < soup.indices.size()/3; f++)
	{
		i2 = soup.indices[3*f]; j2 = soup.indices[3*f+1]; k2 = soup.indices[3*f+2];
		i3 = std::min(i2, std::min(j2, k2)); k3 = std::max(i2, std::max(j2, k2)); j3 = i2+j2+k2-i3-k3;
		if(i1==i3 && j1 == j3 && k1 == k3) return true;
	}
	return false;
}

void FixSoup::findFace(Soup& soup, int i0, int j0, int *iface, int *orient)
{
	int diff = 2;
	int i2, j2, k2, id1, id2;
	int face=0;
	for (int f = 0; f < soup.indices.size()/3; f++)
	{
		i2 = soup.indices[3*f]; j2 = soup.indices[3*f+1]; k2 = soup.indices[3*f+2];
		if((i0==i2||i0==j2||i0==k2) && (j0==i2||j0==j2||j0==k2)) {
			face = f;
			if(i0==i2) {
				id1 = 1; 
			} else if(i0==j2) {
				id1 = 2;
			} else {
				id1 = 3;
			}
			if(j0==i2) {
				id2 = 1; 
			} else if(j0==j2) {
				id2 = 2;
			} else {
				id2 = 3;
			}
			diff = id1-id2; if(diff<0) diff += 3;
			break;
		}
	}
	*iface = face;
	*orient = diff;
}

bool FixSoup::sanityCheck(Soup& soup)
{
	std::vector<std::tuple<int, int, int> > check_faces;
	check_faces.clear();

	int i0, j0, k0, i1, j1, k1;
	for(int f = 0; f < soup.indices.size()/3; f++) {
		i0 = soup.indices[3*f]; j0 = soup.indices[3*f+1]; k0 = soup.indices[3*f+2];
		i1 = std::min(i0, std::min(j0, k0)); k1 = std::max(i0, std::max(j0, k0)); j1 = i0+j0+k0-i1-k1;
		check_faces.push_back(std::make_tuple(i1, j1, k1));
	}
	std::sort(check_faces.begin(), check_faces.end());
	int ndouble = 0;
	for(int i = 0; i < check_faces.size()-1; i++) {
		if(check_faces[i] == check_faces[i+1]) {
			ndouble++;
		}
	}

	int nvertices = soup.positions.size();
	int nfaces = soup.indices.size()/3;
	int nedges = 3*nfaces/2;
	int euler = nvertices - nedges + nfaces;

	if(ndouble > 0 || euler !=2) return false;

	return true;
}

			
void FixSoup::collapseSmallFaces(Soup& soup)
{
	int nvertices, nfaces;
	int nremove=0;

	std::vector<int> indices_save;
	std::vector<Vector> positions_save;
	std::vector<Vector> positions2_save;
	std::vector<int> indices_keep;
	std::vector<Vector> positions_keep;
	std::vector<Vector> positions2_keep;
	std::vector<int> border_list;


	double areamin;
	int iface;
	double Area = soupArea(soup);
	smallestFace(soup, &iface, &areamin);


	while (areamin/Area < 1.e-8) {

		nremove++;

		// current soup size
		nvertices = soup.positions.size();
		nfaces = soup.indices.size()/3;

		// print info
		std::cout << " " << std::endl;
		std::cout << " Face # " << iface << " has near 0 surface area. " << std::endl;
		std::cout << " Face vertices: " << soup.indices[3*iface] << " " << soup.indices[3*iface+1];
		std::cout << " " << soup.indices[3*iface+2] << std::endl;
		std::cout << " Face surface area: " << areamin << std::endl;
		std::cout << " This face is removed for numerical stability." << std::endl;
		std::cout << " " << std::endl;

		// Keep a copy of current "soup" info and empty "soup" itself
		indices_save.clear(); positions_save.clear(); positions2_save.clear();
		for (int j = 0; j < soup.indices.size() ; j++) indices_save.push_back(soup.indices[j]);
		for (int j = 0; j < soup.positions.size() ; j++) {
			positions_save.push_back(soup.positions[j]);
			positions2_save.push_back(soup.positions2[j]);
		}
		indices_keep.clear(); positions_keep.clear(); positions2_keep.clear();
		for (int j = 0; j < soup.indices.size() ; j++) indices_keep.push_back(soup.indices[j]);
		for (int j = 0; j < soup.positions.size() ; j++) {
			positions_keep.push_back(soup.positions[j]);
			positions2_keep.push_back(soup.positions2[j]);
		}
		soup.positions.clear(); soup.indices.clear(); soup.positions2.clear();

		// Update vertex list and create hole (closure of the vertices i0, j0, k0);
		// hole is defined by the list of its vertices
		border_list.clear();
		makeHole(soup, indices_save, positions_save, positions2_save, iface, border_list);

		// Fill in hole; use basic earClipping algorithm

		earClipping(soup, border_list);

		if(!sanityCheck(soup)) {
			soup.positions.clear(); soup.indices.clear(); soup.positions2.clear();
			for (int j = 0; j < indices_keep.size() ; j++) soup.indices.push_back(indices_keep[j]);
			for (int j = 0; j < positions_keep.size() ; j++) {
				soup.positions.push_back(positions_keep[j]);
				soup.positions2.push_back(positions2_keep[j]);
			}
			break;
		}
		// Recompute total area and area of smallest face
		Area = soupArea(soup);
		smallestFace(soup, &iface, &areamin);

	}
	std::cout << "Number of triangles removed: " << nremove << std::endl;

 }

Vector FixSoup::holeNormal(Soup& soup, std::vector<int> hole_list)
{
	Vector n =  Vector(0., 0., 0.);
	int npoint = hole_list.size();

	Vector Curr, Next;
	for(int i = 0; i < npoint; i++)
	{
		int j=i+1;
		if(i==npoint-1) j = 0;
		Curr = soup.positions[hole_list[i]];
		Next = soup.positions[hole_list[j]];
		n[0] += (Curr[1]-Next[1])*(Curr[2]+Next[2]);
		n[1] += (Curr[2]-Next[2])*(Curr[0]+Next[0]);
		n[2] += (Curr[0]-Next[0])*(Curr[1]+Next[1]);
	}
	return n;
}

bool FixSoup::isConvex(Soup& soup, std::vector<int> hole_list, int idx, Vector Normal)
{
	int npoint = hole_list.size();
	int next = idx + 1; 
	if(idx==npoint-1) next = 0;
	int prev = idx - 1;
	if(idx==0) prev = npoint-1;
	Vector p1 = soup.positions[hole_list[idx]];
	Vector p0 = soup.positions[hole_list[prev]];
	Vector p2 = soup.positions[hole_list[next]];
	Vector n = cross(p0-p1, p2-p1);
	double val = dot(n, Normal);
	return (val < 0);
}

bool FixSoup::crossEdge(Soup& soup, std::vector<int> hole_list, int idx, Vector Normal)
{
	int npoint = hole_list.size();
	int next = idx + 1; 
	if(idx==npoint-1) next = 0;
	int prev = idx - 1;
	if(idx==0) prev = npoint-1;

	Vector p1 = soup.positions[hole_list[idx]];
	Vector p0 = soup.positions[hole_list[prev]];
	Vector p2 = soup.positions[hole_list[next]];

	Vector e0, e1;
	bool check;
	for(int i = 0; i < prev-2; i++) {
		e0 = soup.positions[hole_list[i]];
		e1 = soup.positions[hole_list[i+1]];
		check = checkTrig(p0, p1, p2, e0, e1);
		if(check) return true;
	}

	int j;
	for(int i = next+1; i < npoint; i++) {
		e0 = soup.positions[hole_list[i]];
		j = i+1;
		if(i==npoint-1) j = 0;
		e1 = soup.positions[hole_list[j]];
		check = checkTrig(p0, p1, p2, e0, e1);
		if(check) return true;
	}

	return false;
}

bool FixSoup::checkTrig(Vector t0, Vector t1, Vector t2, Vector e0, Vector e1)
{
	Vector BA = t0-t1;
	Vector BC = t2-t1;
	Vector n = cross(BA, BC);
	n.normalize();
	
	Vector e = e1-e0;
	Vector e0A = t0-e0;

	if(dot(e, n)==0) return false;

	double t = dot(e0A,n)/dot(e,n);

	if(t<0 || t>1) return false;

	Vector P = e0 + t*e;

	double a1 = dot(t0-t2, t0-t2); double a2 = dot(t1-t0, t2-t0);
	double b1 = dot(t2-t0, t1-t0); double b2 = dot(t1-t0, t1-t0);
	double c1 = dot(P-t0, t2-t0); double c2 = dot(P-t0, t1-t0);

	double det = a1*b2-b1*a2;
	double alpha = (c1*b2-c2*a2)/det;
	double beta = (a1*c2-b1*c1)/det;

	if(alpha>0 && beta > 0 && (alpha+beta<1)) return true;

	return false;
}

void FixSoup::earClipping(Soup& soup, std::vector<int> hole_list)
{
	int n;
	int prev, next;

	//Orient hole
	int i0 = hole_list[0];
	int j0 = hole_list[1];
	int k0 = hole_list[2];
	int f, orient;
	findFace(soup, i0, j0, &f, &orient);
	if(orient == 2) {
		std::reverse(hole_list.begin(), hole_list.end());
	}

	Vector normal;

	while(hole_list.size() > 3)
	{
		normal = holeNormal(soup, hole_list);
		n = hole_list.size();
		int idx=0;
		for(int i = 0; i < hole_list.size(); i++)
		{
			if(!isConvex(soup, hole_list, i, normal)) continue;
//			if(crossEdge(soup, hole_list, i, normal)) continue;

			prev = i-1;
			if(i==0) prev = n-1;
			next = i+1;
			if(i==n-1) next = 0;
			soup.indices.push_back(hole_list[prev]);
			soup.indices.push_back(hole_list[i]);
			soup.indices.push_back(hole_list[next]);
			idx = i;
			break;
		}
		hole_list.erase(hole_list.begin()+idx);
	}
	i0 = hole_list[0]; j0 = hole_list[1]; k0 = hole_list[2];
	soup.indices.push_back(hole_list[0]);
	soup.indices.push_back(hole_list[1]);
	soup.indices.push_back(hole_list[2]);
}
