#pragma once

#if defined(NANOGUI_GLAD)
	#if defined(NANOGUI_SHARED) && !defined(GLAD_GLAPI_EXPORT)
		#define GLAD_GLAPI_EXPORT
	#endif
	#include <glad/glad.h>
#else
	#if defined(__APPLE__)
		#define GLFW_INCLUDE_GLCOREARB
	#else
		#define GL_GLEXT_PROTOTYPES
	#endif
#endif

#include <GLFW/glfw3.h>
#include <stdlib.h>
#include <string>
#include <iostream>

class Shader {
public:
	// constructor
	Shader();

	// destructor
	~Shader();

	// loads vertex and fragment shaders
	void load(const std::string& vertexSource, const std::string& fragmentSource);

	// use
	void use();

	// member
	GLuint program;

private:
	// compiles shader
	GLuint compileShader(const std::string& code, GLenum type);

	// members
	GLuint vertex;
	GLuint fragment;
};

Shader::Shader():
program(0),
vertex(0),
fragment(0)
{

}

Shader::~Shader()
{
	if (vertex) {
		glDetachShader(program, vertex);
		glDeleteShader(vertex);
	}

	if (fragment) {
		glDetachShader(program, fragment);
		glDeleteShader(fragment);
	}

	if (program) {
		glDeleteProgram(program);
	}
}

void Shader::load(const std::string& vertexSource, const std::string& fragmentSource)
{
	vertex = compileShader(vertexSource, GL_VERTEX_SHADER);
	fragment = compileShader(fragmentSource, GL_FRAGMENT_SHADER);

	// create program and attach shaders
	program = glCreateProgram();
	if (vertex) glAttachShader(program, vertex);
	if (fragment) glAttachShader(program, fragment);

	GLint success;
	GLchar infoLog[512];

	glLinkProgram(program);
	glGetProgramiv(program, GL_LINK_STATUS, &success);
	if (!success) {
		glGetProgramInfoLog(program, 512, NULL, infoLog);
		std::cerr << infoLog << std::endl;
	}
}

void Shader::use()
{
	glUseProgram(program);
}

GLuint Shader::compileShader(const std::string& code, GLenum type)
{
	GLint success;
	GLchar infoLog[512];
	GLuint shader = 0;

	const GLchar *shaderCode = code.c_str();
	shader = glCreateShader(type);
	glShaderSource(shader, 1, &shaderCode, NULL);
	glCompileShader(shader);

	glGetShaderiv(shader, GL_COMPILE_STATUS, &success);
	if (!success) {
		glGetProgramInfoLog(program, 512, NULL, infoLog);
		std::cerr << infoLog << std::endl;
		shader = 0;
	}

	return shader;
}
