/*
 * Copyright (c) 1997-2005 Erez Zadok <ezk@cs.stonybrook.edu>
 * Copyright (c) 2001-2005 Stony Brook University
 *
 * For specific licensing information, see the COPYING file distributed with
 * this package, or get one from ftp://ftp.filesystems.org/pub/fistgen/COPYING.
 *
 * This Copyright notice must be kept intact and distributed with all
 * fistgen sources INCLUDING sources generated by fistgen.
 */
%{
/*
 * ftok.l: Lexical analyzer for FIST parser
 * Fistgen sources.
 * Copyright (c) 1997-2005 Erez Zadok <ezk@cs.stonybrook.edu>
 * Copyright (c) 2001-2005 Stony Brook University
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */


/*
 * Some systems include a definition for the macro ECHO in <sys/ioctl.h>,
 * and their (bad) version of lex defines it too at the very beginning of
 * the generated lex.yy.c file (before it can be easily undefined),
 * resulting in a conflict.  So undefine it here before needed.
 * Luckily, it does not appear that this macro is actually used in the rest
 * of the generated lex.yy.c file.
 */
#ifdef ECHO
# undef ECHO
#endif /* ECHO */
static void ll_echo(void);	/* forward definition */
#define ECHO ll_echo()
#include <fparse.h>

/*
 * There are some things that need to be defined only if useing GNU flex.
 * These must not be defined if using standard lex
 */
#ifdef FLEX_SCANNER
int fyylineno;
#endif /* FLEX_SCANNER */

int parsing_fist_rules;
extern ecl_t *tmp_ecl;
extern int num_func_args, parsing_fist_fxn;

int yylex(void);
int yywrap(void);

#define TOK_DEBUG 0

#if TOK_DEBUG
# define dprintf(f,s) fprintf(stderr, (f), fyylineno, (s))
# define fist_return(v)
#else
# define dprintf(f,s)
# define fist_return(v) return((v))
#endif /* TOK_DEBUG */

/* no need for yywrap() */
#define YY_SKIP_YYWRAP

%}

/* no need to use yyunput() */
%option nounput

%x OPT_DECLS FIST_DECLS FIST_RULES OPT_CODE
%x FUNC_ARGS_IN_RULES FUNC_ARGS_IN_OPT_CODE

WS		[[:blank:]]+
OPTWS		[[:blank:]]*
NOT_WS		[^[:blank:]\n]
NL		\r?\n
NON_NL		[^\r\n]
NON_SEMI_NL	[^\r\n;]+
WORD		([[:alpha:]_][[:alnum:]_]*)
ALPHA		([[:alpha:]_]+){3,}
NON_WORD	[^[:alpha:]_*\n]+
NUMBER		[0-9][0-9]*
NON_FIST_DECL	[^[:alpha:]_;:{}\[\]\r\n]
VARG		{ALPHA}
ATTR		{WORD}
FUNCTIONS	fist(Foo|Bar|MemCpy|SetErr|LastErr|ReturnErr|Malloc|Free|StrEq|Lookup|SkipName|GetIoctlData|SetIoctlData|GetFileData|SetFileData)
FUNCARG		[^[:blank:]\n\(\),]+

%%

<INITIAL>{

	{NL}		{
			fyylineno++;
			}

	"%{"		{
			dprintf("%8d: pct_left \"%s\"\n", yytext);
			fprintf(out_fp_h, "/* OPTIONAL DECLARATIONS FOLLOW */\n");
			BEGIN(OPT_DECLS);
			fist_return(PCT_LEFT);
			}

	"%%"		{
			dprintf("%8d: double_pct \"%s\"\n", yytext);
			BEGIN(FIST_RULES);
			parsing_fist_rules = TRUE;
			tmp_ecl = (ecl_t *) calloc(1, sizeof(ecl_t));
			if (!tmp_ecl)
			  YY_FATAL_ERROR("no more memory");
			fist_return(DOUBLE_PCT);
			}

	.*		{
			dprintf("%8d: fist_decls0 line \"%s\"\n", yytext);
			/* go to fist declarations, and rescan input line */
			BEGIN(FIST_DECLS);
			yyless(0);
			}
}

<OPT_DECLS>{

	{NL}		{
			dprintf("%8d: %s\n", "opt_decls newline");
			fprintf(out_fp_h, "\n");
			fyylineno++;
			}

	"%}"		{
			dprintf("%8d: pct_right \"%s\"\n", yytext);
			BEGIN(FIST_DECLS);
			fist_return(PCT_RIGHT);
			}

	.*		{
			dprintf("%8d: opt_decls line: %s\n", yytext);
			fyylval.strtype = strdup(yytext);
			fist_return(CODELINE);
			}
}

<FIST_DECLS>{

	{NL}		{
			dprintf("%8d: %s\n", "fist_decls newline");
			fyylineno++;
			}

	{WS}		{
				/* ignore white-space */
			}

	"//".*		{
				/* ignore C++ style comments */
			}

	"%%"		{
			dprintf("%8d: double_pct \"%s\"\n", yytext);
			BEGIN(FIST_RULES);
			parsing_fist_rules = TRUE;
			tmp_ecl = (ecl_t *) calloc(1, sizeof(ecl_t));
			if (!tmp_ecl)
			  YY_FATAL_ERROR("no more memory");
			fist_return(DOUBLE_PCT);
			}

	";"		{
			dprintf("%8d: semicolon \"%s\"\n", yytext);
			fist_return(SEMICOLON);
			}

	":"		{
			dprintf("%8d: colon \"%s\"\n", yytext);
			fist_return(COLON);
			}

	"{"		{
			dprintf("%8d: left-brace \"%s\"\n", yytext);
			fist_return(LEFT_BRACE);
			}

	"}"		{
			dprintf("%8d: right-brace \"%s\"\n", yytext);
			fist_return(RIGHT_BRACE);
			}

	"["		{
			dprintf("%8d: left-bracket \"%s\"\n", yytext);
			fist_return(LEFT_BRACKET);
			}

	"]"		{
			dprintf("%8d: right-bracket \"%s\"\n", yytext);
			fist_return(RIGHT_BRACKET);
			}

	{WORD}		{
			dprintf("%8d: word \"%s\"\n", yytext);
			fyylval.strtype = strdup(yytext);
			fist_return(WORD);
			}

	{NUMBER}	{
			dprintf("%8d: number \"%s\"\n", yytext);
			fyylval.inttype = atoi(yytext);
			fist_return(NUMBER);
			}

	{NON_FIST_DECL}	{
			dprintf("%8d: something else \"%s\"\n", yytext);
			fyylval.strtype = strdup(yytext);
			fist_return(MISC);
			}

	^"mod_src"{WS}{NON_SEMI_NL}	{
			  char *cp = strchr(yytext, ' ');
			  /* skip any spaces after "sources" */
			  while (cp && isspace((int)*cp))
			    cp++;
			  dprintf("%8d: mod_src \"%s\"\n", cp);
			  fyylval.strtype = strdup(cp);
			  fist_return(MSOURCE_LIST);
			}
	^"mod_hdr"{WS}{NON_SEMI_NL}	{
			  char *cp = strchr(yytext, ' ');
			  /* skip any spaces after "headers" */
			  while (cp && isspace((int)*cp))
			    cp++;
			  dprintf("%8d: mod_hdr \"%s\"\n", cp);
			  fyylval.strtype = strdup(cp);
			  fist_return(MHEADER_LIST);
			}
	^"user_src"{WS}{NON_SEMI_NL}	{
			  char *cp = strchr(yytext, ' ');
			  /* skip any spaces after "sources" */
			  while (cp && isspace((int)*cp))
			    cp++;
			  dprintf("%8d: user_src \"%s\"\n", cp);
			  fyylval.strtype = strdup(cp);
			  fist_return(USOURCE_LIST);
			}
	^"add_mk"{WS}{NON_SEMI_NL}	{
			  char *cp = strchr(yytext, ' ');
			  /* skip any spaces after "makefiles" */
			  while (cp && isspace((int)*cp))
			    cp++;
			  dprintf("%8d: add_mk \"%s\"\n", cp);
			  fyylval.strtype = strdup(cp);
			  fist_return(ADDMK_LIST);
			}
	^"license"{WS}{NON_SEMI_NL}	{
			  char *cp = strchr(yytext, ' ');
			  /* skip any spaces after "licence info" */
			  while (cp && isspace((int)*cp))
			    cp++;
			  dprintf("%8d: license \"%s\"\n", cp);
			  fyylval.strtype = strdup(cp);
			  fist_return(LICENSE);
			}
}

<FIST_RULES>{

	{NL}		{
			dprintf("%8d: %s\n", "fist_rules newline");
			ecl_strcat(tmp_ecl, "\n");
			fyylineno++;
			}

	"//".*		{
				/* ignore C++ style comments */
			}

	"/*"		{
			  register int c;
			  int comment_start_line = fyylineno;
			  dprintf("%8d: start of comment \"%s\"\n", yytext);
			  ecl_strcat(tmp_ecl, "/*");
			  while (1) {
			      /* eat up text of comment */
			      while ((c = input()) != '*' &&
				     c != '\n' &&
				     c != EOF )
				ecl_putc(tmp_ecl, c);
			      if (c == '\n') {
				ecl_putc(tmp_ecl, c);
				fyylineno++;
			      }
			      if (c == '*') {
				ecl_putc(tmp_ecl, c);
				while ( (c = input()) == '*' )
				  ecl_putc(tmp_ecl, c);
				ecl_putc(tmp_ecl, c);
				if (c == '/')
				  break;	/* found the end */
			      }
			      if (c == EOF) {
				char buf[80];
				sprintf(buf, "%s:%d: start of unterminated comment (EOF on line %d)", in_file, comment_start_line, fyylineno);
				YY_FATAL_ERROR(buf);
				break;
			      }
			  }
			}

	"\""		{
			  register int c;
			  int quote_start_line = fyylineno;
			  dprintf("%8d: start of quote \"%s\"\n", yytext);
			  ecl_putc(tmp_ecl, '"');
			  while (1) {
			      /* eat up text quote */
			      while ((c = input()) != '"' &&
				     c != '\n' &&
				     c != EOF )
				ecl_putc(tmp_ecl, c);
			      if (c == '\n') {
				ecl_putc(tmp_ecl, c);
				fyylineno++;
			      }
			      if (c == '"') {
				ecl_putc(tmp_ecl, c);
				break; /* found the end */
			      }
			      if (c == EOF) {
				char buf[80];
				sprintf(buf, "%s:%d: start of unterminated quote (EOF on line %d)", in_file, quote_start_line, fyylineno);
				YY_FATAL_ERROR(buf);
				break;
			      }
			  }
			}

	"%%"		{
			dprintf("%8d: double_pct \"%s\"\n", yytext);
			fprintf(out_fp_c, "/*\n" \
" * %s.c\n" \
" * Do not edit by hand.\n" \
" * Automatically generated by fistgen.\n" \
" */\n" \
"\n" \
"#ifdef HAVE_CONFIG_H\n" \
"# include <config.h>\n" \
"#endif /* HAVE_CONFIG_H */\n" \
"#ifdef FISTGEN\n" \
"# include \"fist_%s.h\"\n" \
"#endif /* FISTGEN */\n" \
"#include \"fist.h\"\n" \
"#include \"%s.h\"\n" \
"\n" \
"/* OPTIONAL CODE FOLLOWS */\n",
				fist_globals.fg_fsname,
				fist_globals.fg_fsname,
				fist_globals.fg_fsname);
			BEGIN(OPT_CODE);
			parsing_fist_rules = FALSE;
			fist_return(DOUBLE_PCT);
			}

	";"		{
			dprintf("%8d: semicolon \"%s\"\n", yytext);
			ecl_putc(tmp_ecl, ';');
			fist_return(SEMICOLON);
			}

	"{"		{
			dprintf("%8d: left-brace \"%s\"\n", yytext);
			ecl_putc(tmp_ecl, '{');
			fist_return(LEFT_BRACE);
			}

	"}"		{
			dprintf("%8d: right-brace \"%s\"\n", yytext);
			ecl_putc(tmp_ecl, '}');
			fist_return(RIGHT_BRACE);
			}

	{FUNCTIONS}	{
			dprintf("%8d: fist_fxn1 \"%s\"\n", yytext);
			parsing_fist_fxn = TRUE;
			num_func_args = 0; /* reset */
			BEGIN(FUNC_ARGS_IN_RULES);
			fyylval.strtype = strdup(yytext);
			fist_return(FIST_FXN);
			}

	"%"{WORD}":"{WORD}":"{WORD}		{
			dprintf("%8d: fist_rule_def \"%s\"\n", yytext);
			fyylval.strtype = strdup(yytext);
			fist_return(FIST_RULE_DEF);
			}
}

<FIST_RULES,OPT_CODE,FUNC_ARGS_IN_RULES,FUNC_ARGS_IN_OPT_CODE>{
	"%"{ALPHA}	{
			dprintf("%8d: pct_alpha \"%s\"\n", yytext);
			fyylval.strtype = strdup(yytext);
			fist_return(PCT_ALPHA);
			}

	"$"{VARG}	{
			dprintf("%8d: $V \"%s\"\n", yytext);
			fyylval.strtype = strdup(yytext);
			fist_return(DOLLAR_V);
			}

	"$"{NUMBER}	{
			dprintf("%8d: $A \"%s\"\n", yytext);
			fyylval.strtype = strdup(yytext);
			fist_return(DOLLAR_N);
			}

	"$"{VARG}"."{ATTR}	{
			dprintf("%8d: $VA \"%s\"\n", yytext);
			fyylval.strtype = strdup(yytext);
			fist_return(DOLLAR_VA);
			}

	"$"{NUMBER}"."{ATTR}	{
			dprintf("%8d: $NA \"%s\"\n", yytext);
			fyylval.strtype = strdup(yytext);
			fist_return(DOLLAR_NA);
			}

	"$"{VARG}":"{NUMBER}	{
			dprintf("%8d: $VN \"%s\"\n", yytext);
			fyylval.strtype = strdup(yytext);
			fist_return(DOLLAR_VN);
			}

	"$"{VARG}":"{NUMBER}"."{ATTR}	{
			dprintf("%8d: $VNA \"%s\"\n", yytext);
			fyylval.strtype = strdup(yytext);
			fist_return(DOLLAR_VNA);
			}
}

<FUNC_ARGS_IN_RULES>{

	"("		{
			dprintf("%8d: left-paren \"%s\"\n", yytext);
			fist_return(LEFT_PAREN);
			}

	")"		{
			dprintf("%8d: right-paren \"%s\"\n", yytext);
			parsing_fist_fxn = FALSE;
			BEGIN(FIST_RULES);
			fist_return(RIGHT_PAREN);
			}

	{FUNCARG}	{
			dprintf("%8d: funcarg \"%s\"\n", yytext);
			fyylval.strtype = strdup(yytext);
			fist_return(FUNCARG);
			}
}

<OPT_CODE>{

	{NL}		{
			fyylineno++;
			fprintf(out_fp_c, "\n");
			}

	"/*"		{
			  register int c;
			  int comment_start_line = fyylineno;
			  dprintf("%8d: start of comment \"%s\"\n", yytext);
			  fprintf(out_fp_c, "/*");
			  while (1) {
			      /* eat up text of comment */
			      while ((c = input()) != '*' &&
				     c != '\n' &&
				     c != EOF )
				fputc(c, out_fp_c);
			      if (c == '\n') {
				fputc(c, out_fp_c);
				fyylineno++;
			      }
			      if (c == '*') {
				fputc(c, out_fp_c);
				while ( (c = input()) == '*' )
				  fputc(c, out_fp_c);
				fputc(c, out_fp_c);
				if (c == '/')
				  break;	/* found the end */
			      }
			      if (c == EOF) {
				char buf[80];
				sprintf(buf, "%s:%d: start of unterminated comment (EOF on line %d)", in_file, comment_start_line, fyylineno);
				YY_FATAL_ERROR(buf);
				break;
			      }
			  }
			}

	"\""		{
			  register int c;
			  int quote_start_line = fyylineno;
			  dprintf("%8d: start of quote \"%s\"\n", yytext);
			  fprintf(out_fp_c, "\"");
			  while (1) {
			      /* eat up text quote */
			      while ((c = input()) != '"' &&
				     c != '\n' &&
				     c != EOF )
				fputc(c, out_fp_c);
			      if (c == '\n') {
				fputc(c, out_fp_c);
				fyylineno++;
			      }
			      if (c == '"') {
				fputc(c, out_fp_c);
				break; /* found the end */
			      }
			      if (c == EOF) {
				char buf[80];
				sprintf(buf, "%s:%d: start of unterminated quote (EOF on line %d)", in_file, quote_start_line, fyylineno);
				YY_FATAL_ERROR(buf);
				break;
			      }
			  }
			}

	{FUNCTIONS}	{
			dprintf("%8d: fist_fxn3 \"%s\"\n", yytext);
			parsing_fist_fxn = TRUE;
			num_func_args = 0; /* reset */
			BEGIN(FUNC_ARGS_IN_OPT_CODE);
			fyylval.strtype = strdup(yytext);
			fist_return(FIST_FXN);
			}
	/* we set yyout=out_fp_c to copy unmatched text to output file */
}

<FUNC_ARGS_IN_OPT_CODE>{

	"("		{
			dprintf("%8d: left-paren \"%s\"\n", yytext);
			fist_return(LEFT_PAREN);
			}

	")"		{
			dprintf("%8d: right-paren \"%s\"\n", yytext);
			parsing_fist_fxn = FALSE;
			BEGIN(OPT_CODE);
			fist_return(RIGHT_PAREN);
			}

	{FUNCARG}	{
			dprintf("%8d: funcarg \"%s\"\n", yytext);
			fyylval.strtype = strdup(yytext);
			fist_return(FUNCARG);
			}
}
%%


int
yywrap(void)
{
  return 1;
}


static void
ll_echo(void)
{
  /* do nothing if parsing functions.  they will be processed later */
  if (parsing_fist_fxn)
    return;
  if (parsing_fist_rules) {
    ecl_strcat(tmp_ecl, yytext);
  } else {
    fwrite( yytext, yyleng, 1, yyout );
  }
}
