/*
 * Copyright (c) 1997-2005 Erez Zadok <ezk@cs.stonybrook.edu>
 * Copyright (c) 2001-2005 Stony Brook University
 *
 * For specific licensing information, see the COPYING file distributed with
 * this package, or get one from ftp://ftp.filesystems.org/pub/fistgen/COPYING.
 *
 * This Copyright notice must be kept intact and distributed with all
 * fistgen sources INCLUDING sources generated by fistgen.
 */
/*
 *  $Id: sca_code.c,v 1.10 2005/01/03 21:10:41 ezk Exp $
 */

#include "sca_code.h"
#include "sca_aux.h"

#define DEFLATE_LEVEL 9


int
sca_encode_page(unsigned char *in, int inlen, unsigned char **out, int *outlen)
     /* in is provided by the caller, and is a inlen long array of bytes
        out is malloc'ed by this procedure (remember to free it!), and is
	the result of processing this page. process_page returns the length of
	out, or -1 for error

	Note: Normally, we would pass in a buffer for this function to encode
	onto, but as we don't know how big it might be, we pass a pointer
	that it can re-alloc as it needs.
	*/
{
  z_stream stream;		/* Compression stream */
  int rc, outsize;

  encode_counter++;
  stream.zalloc = (alloc_func)0; /* Custom memory allocator called with
				      opaque */
  stream.zfree = (free_func)0; /* Custom memory free-er called with
				    opaque as an argument */
  stream.opaque = (voidpf)0;	/* Opaque argument to zalloc/zfree */

  if ((rc = deflateInit(&stream, /* First arg is a stream object */
		   DEFLATE_LEVEL))/* Second arg is compression level (0-9) */
      != Z_OK ) {
    fprintf(stderr, "deflateInit error %d\n", rc);
    return(-1);
  }

  (*out) = (unsigned char *) malloc (inlen * sizeof(unsigned char));
  if ((*out) == NULL) {
    fprintf(stderr, "malloc failed on %d with %d\n", inlen, errno);
    return(-1);
  }

  stream.next_in = in;		/* Initialize in stream */
  stream.avail_in = inlen;
  stream.next_out = (*out);	/* Initialize out stream */
  stream.avail_out = inlen;

  outsize = inlen;
  rc=0;
  while (rc != Z_STREAM_END) {
    rc = deflate(&stream, Z_FULL_FLUSH);
    if (stream.avail_in == 0) {	/* We've finished all available input data */
      rc = deflate(&stream, Z_FINISH); /* End the stream */
    } else if (rc == Z_OK) {	/* Didn't finish, needs more space! */
      if (((*out) = realloc((*out), outsize+inlen)) == NULL) {
	perror("realloc");
	return(-1);
      }
      stream.next_out = &((*out)[outsize]); /* The end of our new array */
      outsize += inlen;
      stream.avail_out = inlen;	/* Increments of inlen bytes */
    } else if (rc != Z_STREAM_END) {
      fprintf(stderr, "deflate returns %d!\n", rc);
      return(-1);
    }
  }
  deflateEnd(&stream);
  *outlen = stream.total_out;
  return(stream.total_out);
}


int
sca_decode_page(unsigned char *in, int inlen, unsigned char **out, int *outlen)
     /* Takes the encoded data (of length len) on in, and decodes it,
        and writes the data into out. out is allocated by this procedure,
	and should be freed when you finish with it. It returns the length
	of out, or -1 for an error.
	*/
{
  z_stream stream;
  int rc, outsize = chunksize;

  decode_counter++;
  stream.zalloc = (alloc_func)0; /* Custom memory allocator called with
				      opaque */
  stream.zfree = (free_func)0; /* Custom memory free-er called with
				    opaque as an argument */
  stream.opaque = (voidpf)0;	/* Opaque data in the stream object */

  if (((*out) = (unsigned char *) malloc (chunksize * sizeof(unsigned char)))
      == NULL)
    return(-1);

  stream.next_in = in;
  stream.avail_in = inlen;  	/* How much data in rawdata */
  stream.next_out = (*out);
  stream.avail_out = chunksize;

  if ((rc = inflateInit(&stream)) /* First arg is a stream object */
      != Z_OK ) {
    fprintf(stderr, "inflateInit error %d\n", rc);
    return(-1);
  }

  rc = 0;
  while (rc != Z_STREAM_END) {
    rc = inflate(&stream, Z_FINISH);
    if (rc == Z_OK) {	/* Didn't finish, needs more space */
      fprintf(stderr, "Needed more than %d space!\n", outsize);
      if (((*out) = realloc((*out), outsize+chunksize)) == NULL) {
	perror("realloc");
	return(-1);
      }
      stream.next_out = &((*out)[outsize]); /* The end of our new array */
      outsize += chunksize;
      stream.avail_out = chunksize;	/* Increments of inlen bytes */

      return(-1);
    } else if (rc != Z_STREAM_END) {
      fprintf(stderr, "inflate returns %d!\n", rc);
      return(-1);
    }
  }
  inflateEnd(&stream);
  *outlen = stream.total_out;
  return(stream.total_out);
}


int
sca_fix_idx(char *name)
{
  int srcfd;
  unsigned char *ptr, *data = NULL;
  char filename[MAXPATHLEN];
  int cnt, rc, offset = 0, remainder = 0;
  struct fistfs_header hdr;
  struct stat sb;
  unsigned char *outpage;
  z_stream stream;

  hdr.num_pages = 0;
  hdr.real_size = 0;
  hdr.offsets = NULL;

  if (name == NULL)
    return(-1);
  if ((srcfd = open(name,O_RDONLY)) < 0)
    return(-1);

  if (do_fast_tails)
    hdr.flags |= 0x80000000;

  if ((outpage = (unsigned char *)
       malloc(chunksize * (sizeof(unsigned char)))) == NULL)
    return(-1);

  if ((ptr = strrchr(name,'.')) == NULL) {
    fprintf(stderr, "No .gzf suffix, creating %s.idx\n", name);
    sprintf(filename, "%s.idx", name);
  } else {
    ptr[0] = '\0';
    strncpy(filename, name, MAXPATHLEN-5);
    cnt = strlen(filename);
    filename[cnt] = '.';
    filename[cnt+1] = 'i';
    filename[cnt+2] = 'd';
    filename[cnt+3] = 'x';
    fprintf(stderr, "Creating %s\n", filename);
  }

  if ((fstat(srcfd,&sb)) < 0)
    return(-1);

  if ((data = mmap((void *)data,sb.st_size, PROT_READ,
		   MAP_PRIVATE,srcfd,0)) == NULL)
    return(errno);

  stream.zalloc = (alloc_func)0; /* Custom memory allocator called with
				      opaque */
  stream.zfree = (free_func)0; /* Custom memory free-er called with
				    opaque as an argument */
  stream.opaque = (voidpf)0;	/* Opaque argument to zalloc/zfree */

  cnt = 0;

  if (do_fast_tails) {		/* Round to nerest chunksize */
    rc = sb.st_size / chunksize;
    remainder = sb.st_size % chunksize;
    sb.st_size = rc * chunksize;
  }

  while (cnt < sb.st_size) {
    if ((rc = inflateInit(&stream)) /* Arg is a stream object */
	!= Z_OK ) {
      fprintf(stderr, "inflateInit error %d\n", rc);
      return(-1);
    }

    stream.next_in = &(data[cnt]);
    stream.avail_in = sb.st_size - cnt;
    stream.next_out = outpage;
    stream.avail_out = chunksize;

    rc = inflate(&stream, Z_FINISH);
    if (rc == Z_STREAM_END) {
      cnt += ((sb.st_size - cnt) - stream.avail_in);
      hdr.num_pages++;
      hdr.real_size += stream.total_out;
      if ((hdr.offsets = realloc(hdr.offsets,
				 (hdr.num_pages * sizeof(off_t))))
	  == NULL) {
	return(-1);
      }
      hdr.offsets[hdr.num_pages-1] = cnt;

    } else {
      fprintf(stderr, "Bad inflate (returns %d, decoded %ld)\n",
	      rc, stream.total_out);
      return(-1);
    }
    inflateEnd(&stream);
  }

  if (do_fast_tails)
    hdr.real_size += remainder;

  munmap(data, sb.st_size);
  close(srcfd);
  if (write_idx(filename, &hdr) < 0) {
    fprintf(stderr, "Error writing %s (%d)\n", filename, errno);
    return(-1);
  }
  return(0);
}
