/*
 * Copyright (c) 1997-2005 Erez Zadok <ezk@cs.stonybrook.edu>
 * Copyright (c) 2001-2005 Stony Brook University
 *
 * For specific licensing information, see the COPYING file distributed with
 * this package, or get one from ftp://ftp.filesystems.org/pub/fistgen/COPYING.
 *
 * This Copyright notice must be kept intact and distributed with all
 * fistgen sources INCLUDING sources generated by fistgen.
 */
/*
 * Copyright (c) 1992, 1993
 *      The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from software donated to Berkeley by
 * Jan-Simon Pendry.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the University of
 *      California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *      @(#)wrapfs_subr.c       8.7 (Berkeley) 5/14/95
 *
 * $Id: subr.c,v 1.8 2005/01/03 21:10:41 ezk Exp $
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */
#ifdef FISTGEN
# include "fist_wrapfs.h"
#endif /* FISTGEN */
#include <fist.h>
#include <wrapfs.h>

#if 0
#include "opt_debug.h"

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/proc.h>
#include <sys/vnode.h>
#include <sys/mount.h>
#include <sys/malloc.h>
#include <sys/namei.h>
#include <vm/vm_zone.h>
#include <wrapfs.h>
#endif

#define LOG2_SIZEVNODE 7	/* log2(sizeof struct vnode) */
#define	NWRAPFSNODECACHE 16

/*
 * Cryptfs layer cache:
 * Each cache entry holds a reference to the lower vnode
 * along with a pointer to the alias vnode.  When an
 * entry is added the lower vnode is VREF'd.  When the
 * alias is removed the lower vnode is vrele'd.
 */

#define	WRAPFS_NHASH(vp) \
	(&wrapfs_node_hashtbl[(((uintptr_t)vp)>>LOG2_SIZEVNODE) & wrapfs_node_hash])
static
LIST_HEAD(wrapfs_node_hashhead, wrapfs_node) * wrapfs_node_hashtbl;
     static u_long wrapfs_node_hash;

     static int wrapfs_node_alloc __P((struct mount *mp, struct vnode * lowervp,
				       struct vnode ** vpp));
     static struct vnode *
       wrapfs_node_find __P((struct mount *mp, struct vnode * lowervp));

/*
 * Initialise cache headers
 */
int
wrapfs_init(vfsp)
     struct vfsconf *vfsp;
{

  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);

#ifdef WRAPFS_DIAGNOSTIC
  printf("wrapfs_init\n");	/* printed during system boot */
#endif
  wrapfs_node_hashtbl = hashinit(NWRAPFSNODECACHE, M_CACHE, &wrapfs_node_hash);
  return (0);
}

/*
 * Return a VREF'ed alias for lower vnode if already exists, else 0.
 */
static struct vnode *
wrapfs_node_find(mp, lowervp)
     struct mount *mp;
     struct vnode *lowervp;
{
  struct proc *p = curproc;	/* XXX */
  struct wrapfs_node_hashhead *hd;
  struct wrapfs_node *a;
  struct vnode *vp;

  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);

  /*
   * Find hash base, and then search the (two-way) linked
   * list looking for a wrapfs_node structure which is referencing
   * the lower vnode.  If found, the increment the wrapfs_node
   * reference count (but NOT the lower vnode's VREF counter).
   */
  hd = WRAPFS_NHASH(lowervp);
loop:
  for (a = hd->lh_first; a != 0; a = a->wrapfs_hash.le_next) {
    if (a->wrapfs_lowervp == lowervp && WRAPFS_TO_VP(a)->v_mount == mp) {
      vp = WRAPFS_TO_VP(a);
      /*
       * We need vget for the VXLOCK
       * stuff, but we don't want to lock
       * the lower node.
       */
      if (vget(vp, 0, p)) {
	printf("wrapfs_node_find: vget failed.\n");
	goto loop;
      };
      return (vp);
    }
  }

  return NULLVP;
}

/*
 * Make a new wrapfs_node node.
 * Vp is the alias vnode, lofsvp is the lower vnode.
 * Maintain a reference to (lowervp).
 */
static int
wrapfs_node_alloc(mp, lowervp, vpp)
     struct mount *mp;
     struct vnode *lowervp;
     struct vnode **vpp;
{
  struct wrapfs_node_hashhead *hd;
  struct wrapfs_node *xp;
  struct vnode *othervp, *vp;
  int error;

  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);

  /*
   * Do the MALLOC before the getnewvnode since doing so afterward
   * might cause a bogus v_data pointer to get dereferenced
   * elsewhere if MALLOC should block.
   */
  MALLOC(xp, struct wrapfs_node *, sizeof(struct wrapfs_node), M_TEMP, M_WAITOK);
  error = getnewvnode(VT_WRAPFS, mp, wrapfs_vnodeop_p, vpp);
  if (error) {
    FREE(xp, M_TEMP);
    return (error);
  }
  vp = *vpp;

  vp->v_type = lowervp->v_type;
  xp->wrapfs_vnode = vp;
  vp->v_data = xp;
  xp->wrapfs_lowervp = lowervp;
  /*
   * Before we insert our new node onto the hash chains,
   * check to see if someone else has beaten us to it.
   * (We could have slept in MALLOC.)
   */
  othervp = wrapfs_node_find(mp, lowervp);
  if (othervp) {
    FREE(xp, M_TEMP);
    vp->v_type = VBAD;		/* node is discarded */
    vp->v_usecount = 0;		/* XXX */
    *vpp = othervp;
    return 0;
  };
  VREF(lowervp);		/* Extra VREF will be vrele'd in
				 * wrapfs_node_create */
  hd = WRAPFS_NHASH(lowervp);
  LIST_INSERT_HEAD(hd, xp, wrapfs_hash);
  return 0;
}

/*
 * Try to find an existing wrapfs_node vnode refering
 * to it, otherwise make a new wrapfs_node vnode which
 * contains a reference to the lower vnode.
 */
int
wrapfs_node_create(mp, lowervp, newvpp)
     struct mount *mp;
     struct vnode *lowervp;
     struct vnode **newvpp;
{
  struct vnode *aliasvp;

  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);

  aliasvp = wrapfs_node_find(mp, lowervp);
  if (aliasvp) {
    /*
     * wrapfs_node_find has taken another reference
     * to the alias vnode.
     */
#ifdef WRAPFS_DIAGNOSTIC
    vprint("wrapfs_node_create: exists", aliasvp);
#endif
    /* VREF(aliasvp); --- done in wrapfs_node_find */
  } else {
    int error;

    /*
     * Get new vnode.
     */
#ifdef WRAPFS_DIAGNOSTIC
    printf("wrapfs_node_create: create new alias vnode\n");
#endif

    /*
     * Make new vnode reference the wrapfs_node.
     */
    error = wrapfs_node_alloc(mp, lowervp, &aliasvp);
    if (error)
      return error;

    /*
     * aliasvp is already VREF'd by getnewvnode()
     */
  }

  vrele(lowervp);

#ifdef DIAGNOSTIC
  if (lowervp->v_usecount < 1) {
    /* Should never happen... */
    vprint("wrapfs_node_create: alias ", aliasvp);
    vprint("wrapfs_node_create: lower ", lowervp);
    panic("wrapfs_node_create: lower has 0 usecount.");
  };
#endif

#ifdef WRAPFS_DIAGNOSTIC
  vprint("wrapfs_node_create: alias", aliasvp);
  vprint("wrapfs_node_create: lower", lowervp);
#endif

  *newvpp = aliasvp;
  return (0);
}

#ifdef WRAPFS_DIAGNOSTIC
#include "opt_ddb.h"

#ifdef DDB
#define	wrapfs_checkvp_barrier	1
#else
#define	wrapfs_checkvp_barrier	0
#endif

struct vnode *
wrapfs_checkvp(vp, fil, lno)
     struct vnode *vp;
     char *fil;
     int lno;
{
  struct wrapfs_node *a = VP_TO_WRAPFS(vp);
  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);

#ifdef notyet
  /*
   * Can't do this check because vop_reclaim runs
   * with a funny vop vector.
   */
  if (vp->v_op != wrapfs_vnodeop_p) {
    printf("wrapfs_checkvp: on non-wrapfs-node\n");
    while (wrapfs_checkvp_barrier)	/* WAIT */
      ;
    panic("wrapfs_checkvp");
  };
#endif
  if (a->wrapfs_lowervp == NULLVP) {
    /* Should never happen */
    int i;
    u_long *p;
    printf("vp = %p, ZERO ptr\n", vp);
    for (p = (u_long *) a, i = 0; i < 8; i++)
      printf(" %lx", p[i]);
    printf("\n");
    /* wait for debugger */
    while (wrapfs_checkvp_barrier)	/* WAIT */
      ;
    panic("wrapfs_checkvp");
  }
  if (a->wrapfs_lowervp->v_usecount < 1) {
    int i;
    u_long *p;
    printf("vp = %p, unref'ed lowervp\n", vp);
    for (p = (u_long *) a, i = 0; i < 8; i++)
      printf(" %lx", p[i]);
    printf("\n");
    /* wait for debugger */
    while (wrapfs_checkvp_barrier)	/* WAIT */
      ;
    panic("wrapfs with unref'ed lowervp");
  };
#ifdef notyet
  printf("wrapfs %x/%d -> %x/%d [%s, %d]\n",
	 WRAPFSTOV(a), WRAPFSTOV(a)->v_usecount,
	 a->wrapfs_lowervp, a->wrapfs_lowervp->v_usecount,
	 fil, lno);
#endif
  return a->wrapfs_lowervp;
}
#endif

/****************************************************************************/
/* ADDED BY EZK */

/* allocate memory and zero it */
void *
kmem_zalloc(unsigned long size)
{
  void *addr;

  addr = malloc(size, M_TEMP, M_WAITOK);
  if (addr)
    bzero(addr, size);
  return addr;
}

/* do an I/O move */
int
fist_uiomove(caddr_t cp, int n, enum uio_rw rwflag, struct uio *uio)
{
  enum uio_rw saved_rwflag;
  int ret;

  if (uio->uio_rw != rwflag) {
    printf("UIOMOVE mismatching flags: uio->uio_rw=%d, rwflag=%d\n",
	   uio->uio_rw, rwflag);
  }
  /* save uio's rwflag */
  saved_rwflag = uio->uio_rw;
  uio->uio_rw = rwflag;
  ret = uiomove(cp, n, uio);
  uio->uio_rw = saved_rwflag;
  return ret;
}

#if 0
#ifdef FIST_FILTER_DATA
/* return the number of bytes written */
int
wrapfs_encode_block(char *func, int line, caddr_t base, int len, const vnode_t *vp, cred_t *cr)
{
  /* blowfish variables */
  unsigned char iv[8];
  int n;

  /* ASSERT(fist_get_userpass(vp->v_vfsp,cr) != NULL); */

  if (len > PAGE_SIZE || len < 0)
    printf("CEB: %s:%d: base=%x, len=%d, vp=%x\n",
	   func, line, (int) base, len, (int) vp);

  bcopy(cbc_iv, iv, 8);
  n = 0;		/* opaque variable for blowfish's internal stat */
  BF_cfb64_encrypt(base,
		   base,
		   len, fist_get_userpass(vp->v_mount, cr), iv, &n,
		   BF_ENCRYPT);
  return len;
}

int
wrapfs_decode_block(char *func, int line, caddr_t base, int len, const vnode_t *vp, cred_t *cr)
{
  /* blowfish variables */
  unsigned char iv[8];
  int n;

  /* ASSERT(fist_get_userpass(vp->v_vfsp,cr) != NULL); */

  if (len > PAGE_SIZE || len < 0)
    printf("CDB: %s:%d: base=%x, len=%d, vp=%x\n",
	   func, line, (int) base, len, (int) vp);

  bcopy(cbc_iv, iv, 8);
  n = 0;		/* opaque variable for blowfish's internal stat */
  BF_cfb64_encrypt(base,
		   base,
		   len, fist_get_userpass(vp->v_mount, cr), iv, &n,
		   BF_DECRYPT);

  return len;
}
#endif /* FIST_FILTER_DATA */
#endif

#ifdef FIST_FILTER_NAME
#if 0
void
wrapfs_encode_filename(
		       vfs_t *vfsp,
		       char *name,
		       char **uuencoded_name, /* null-terminated */
		       int *uuencoded_length, /* w/ terminating null */
		       int skip_dots,
		       int use_namei_zone,
		       cred_t *cr)
{
  char *crypted_name,*ptr;
  int length, rounded_length, n, i, j;
  unsigned char iv[8];
  short csum;

  /* ASSERT(fist_get_userpass(vfsp, cr) != NULL); */
  if (skip_dots && (name[0] == '.' &&
		    (name[1] == '\0' ||
		     (name[1] == '.' && name[2] == '\0')))) {
    *uuencoded_length = strlen(name) + 1;
    *uuencoded_name = kmem_zalloc(*uuencoded_length);
    strcpy(*uuencoded_name, name);
    return;
  }
  for (csum = 0, length = 0, ptr = name; *ptr; ptr++, length++)
    csum += *ptr;
  /*
   * rounded_length is an multiple of 3 rounded-up length
   * the uuencode algorithm processes 3 source bytes at a time
   * so we have to make sure we don't read past the memory
   * we have allocated
   *
   * it uses length + 3 to provide 2 bytes for the checksum
   * and one byte for the length
   */
  rounded_length = (((length + 3) + 2) / 3) * 3;
  crypted_name = kmem_zalloc(rounded_length);

  bcopy(cbc_iv, iv, 8);
  n = 0;
  *(short *)crypted_name = csum;
  crypted_name[2] = length;
  BF_cfb64_encrypt(name, crypted_name + 3,
		   length, fist_get_userpass(vfsp, cr), iv, &n,
		   BF_ENCRYPT);

  *uuencoded_length = (((length + 3) + 2) / 3) * 4 + 1;
  if (use_namei_zone)
    *uuencoded_name = zalloc(namei_zone); /* 'z' stands for Zone Alloc */
  else
    *uuencoded_name = kmem_alloc(*uuencoded_length);

  for (i = 0, j = 0; i < rounded_length; i += 3, j += 4) {
    (*uuencoded_name)[j] = 48 + ((crypted_name[i] >> 2) & 63);
    (*uuencoded_name)[j + 1] = 48 + (((crypted_name[i] << 4) & 48) | ((crypted_name[i + 1] >> 4) & 15));
    (*uuencoded_name)[j + 2] = 48 + (((crypted_name[i + 1] << 2) & 60) | ((crypted_name[i + 2] >> 6) & 3));
    (*uuencoded_name)[j + 3] = 48 + (crypted_name[i + 2] & 63);
  }
  (*uuencoded_name)[j] = '\0';

  kmem_free(crypted_name);
}


int
wrapfs_decode_filename(vfs_t *vfsp,
		       char *name,
		       int length,
		       char **decrypted_name,
		       int *decrypted_length, /* w/ terminating null */
		       int skip_dots,
		       cred_t *cr)
{
  int n, i, j, saved_length, saved_csum, csum;
  int uudecoded_length, error = 0;
  unsigned char iv[8];
  char *uudecoded_name;

  /* ASSERT(fist_get_userpass(vfsp, cr) != NULL); */

  if (skip_dots && (name[0] == '.' &&
		    (name[1] == '\0' ||
		     (name[1] == '.' && name[2] == '\0')))) {
    /* used to add 1 to following */
    *decrypted_length = length + 1;
    *decrypted_name = kmem_alloc(*decrypted_length);
    for (i = 0; i < length; i++)
      (*decrypted_name)[i] = name[i];
    (*decrypted_name)[i] = 0;
    return 0;
  }

  uudecoded_length = ((length + 3) / 4) * 3;
  uudecoded_name = kmem_alloc(uudecoded_length);

  for (i = 0, j = 0; i < length; i += 4, j += 3) {
    uudecoded_name[j] = ((name[i] - 48) <<2) | ((name[i + 1] - 48) >>4);
    uudecoded_name[j + 1] = (((name[i + 1] - 48) <<4) & 240) | ((name[i + 2] - 48) >>2);
    uudecoded_name[j + 2] = (((name[i + 2] - 48) <<6) & 192) | ((name[i + 3] - 48) &63);
  }
  saved_csum = *(short *)uudecoded_name;
  saved_length = uudecoded_name[2];
  if (saved_length > uudecoded_length) {
    printf("Problems with the length - too big: %d", saved_length);
    error = -1;
    goto out;
  }

  *decrypted_name = (char *)kmem_alloc(saved_length + 1);
  bcopy(cbc_iv, iv, 8);
  n = 0;
  BF_cfb64_encrypt(uudecoded_name + 3, *decrypted_name,
		   saved_length, fist_get_userpass(vfsp, cr), iv, &n,
		   BF_DECRYPT);
  for (csum = 0, i = 0; i < saved_length; i++)
    csum += (*decrypted_name)[i];
  if (csum != saved_csum) {
    fist_dprint(6,"Checksum error\n");
    kmem_free(*decrypted_name);
    error = -1;
    goto out;
  }
  (*decrypted_name)[saved_length] = 0;
  *decrypted_length = saved_length + 1;
out:
  kmem_free(uudecoded_name);
  return error;
}
#endif

/*
 * create a new (lower) componentname from an existing one
 * (this level) and encode the inner.
 */
cn_t *
wrapfs_new_cnp(const vnode_t *thisvp, const cn_t *thiscnp)
{
  cn_t *lowercnp;
#ifdef FIST_ENCODE
  static char buf[MAXNAMLEN];
#else /* not FIST_ENCODE */
  int len;
  char *name = NULL;
#endif /* not FIST_ENCODE */

  if (!thiscnp)
    return NULL;

  lowercnp = kmem_alloc(sizeof(cn_t));
  if (!lowercnp)
    panic("wrapfs_new_cnp: no more memory\n");
  /* copy all fields */
  bcopy(thiscnp, lowercnp, sizeof(cn_t));
  /* fix certain fields temporarily */
  lowercnp->cn_pnbuf = NULL;
  lowercnp->cn_nameptr = NULL;
  lowercnp->cn_namelen = 0;

  /* prepare new path name for lookup */
#ifdef FIST_ENCODE
  bcopy(thiscnp->cn_nameptr, buf, thiscnp->cn_namelen);
  buf[thiscnp->cn_namelen] = '\0';
  wrapfs_encode_filename(thisvp->v_mount,
			 buf,
			 &lowercnp->cn_pnbuf,
			 (int *) &lowercnp->cn_namelen,
			 SKIP_DOTS,
			 TRUE, /* use namei zone alloc */
			 thiscnp->cn_cred);
  /* updated other cnp fields */
  lowercnp->cn_namelen--;	/* don't count terminating null */
  lowercnp->cn_nameptr = lowercnp->cn_pnbuf;
  /* I always allocate my own lowercnp buffer */
  // lowercnp->cn_flags |= HASBUF;
#else /* not FIST_ENCODE */
  len = thiscnp->cn_namelen;
  name = zalloc(namei_zone);
  if (!name)
    panic("wrapfs_new_cnp: no more memory for name\n");
  bcopy(thiscnp->cn_nameptr, name, len+1);
  name[len] = '\0';		/* just in case (not needed) */
  /* updated other cnp fields */
  name[0]++;			/* update first char of name */
  lowercnp->cn_namelen = len;
  lowercnp->cn_nameptr = lowercnp->cn_pnbuf = name;
#endif /* not FIST_ENCODE */

  /* return formed string */
  return lowercnp;
}

/*
 * Update an old cnp based on a newer one.
 * Also free previously allocated lowercnp and its fields.
 */
void
wrapfs_update_cnp(const vnode_t *thisvp, cn_t **lowercnpp, cn_t *thiscnp, int error)
{
  /*
   * free thiscnp if there no error, HASBUF was on, and SAVENAME was off.
   * XXX: what about SAVESTART (from <sys/namei.h>)
   */
  if (!error &&
      (thiscnp->cn_flags & (HASBUF|SAVENAME|SAVESTART)) == HASBUF) {
    fist_dprint(2, "UPDATE_CNP: freeing thiscnp->cn_pnbuf \"%s\"\n",
		thiscnp->cn_pnbuf);
    zfree(namei_zone, thiscnp->cn_pnbuf);
  }

  if (!fist_isdeadcode(*lowercnpp)) {
    fist_dprint(2, "UPDATE_CNP: lowercnp flags <%s>\n",
		fist_cn_flags((*lowercnpp)->cn_flags));
  }

  /* always free space used by lowercnp, if not already free'd */
  if (fist_isdeadcode(*lowercnpp))
    return;



  if (!fist_isdeadcode((*lowercnpp)->cn_pnbuf)) {
    if (((*lowercnpp)->cn_flags & (HASBUF|SAVENAME|SAVESTART)) == HASBUF) {
      fist_dprint(2, "UPDATE_CNP: freeing lowercnp->cn_pnbuf \"%s\"\n",
		  (*lowercnpp)->cn_pnbuf);
      zfree(namei_zone, (*lowercnpp)->cn_pnbuf);
    } else {
      fist_dprint(1, "UPDATE_CNP: not freeing 0x%x \"%s\" <%s>\n",
		  (int) *lowercnpp,
		  (*lowercnpp)->cn_pnbuf,
		  fist_cn_flags((*lowercnpp)->cn_flags));
    }
  }
  fist_dprint(2, "UPDATE_CNP: freeing lowercnp 0x%x\n", (int) *lowercnpp);
  kmem_free(*lowercnpp);

}
#endif /* FIST_FILTER_NAME */
