/*
 * Copyright (c) 1997-2005 Erez Zadok <ezk@cs.stonybrook.edu>
 * Copyright (c) 2001-2005 Stony Brook University
 *
 * For specific licensing information, see the COPYING file distributed with
 * this package, or get one from ftp://ftp.filesystems.org/pub/fistgen/COPYING.
 *
 * This Copyright notice must be kept intact and distributed with all
 * fistgen sources INCLUDING sources generated by fistgen.
 */
/*
 * Copyright (c) 1992, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from software donated to Berkeley by
 * Jan-Simon Pendry.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	@(#)wrapfs_subr.c	8.7 (Berkeley) 5/14/95
 *
 * $FreeBSD: src/sys/miscfs/wrapfs/wrapfs_subr.c,v 1.21.2.4 2001/06/26 04:20:09 bp Exp $
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */
#ifdef FISTGEN
# include "fist_wrapfs.h"
#endif /* FISTGEN */
#include <fist.h>
#include <wrapfs.h>

#if 0
#include "opt_debug.h"

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/proc.h>
#include <sys/vnode.h>
#include <sys/mount.h>
#include <sys/malloc.h>
#include <miscfs/wrapfs/wrapfs.h>
#endif

#define LOG2_SIZEVNODE 7		/* log2(sizeof struct vnode) */
#define	NWRAPFSNODECACHE 16

/*
 * Wrapfs layer cache:
 * Each cache entry holds a reference to the lower vnode
 * along with a pointer to the alias vnode.  When an
 * entry is added the lower vnode is VREF'd.  When the
 * alias is removed the lower vnode is vrele'd.
 */

#define	WRAPFS_NHASH(vp) \
	(&wrapfs_node_hashtbl[(((uintptr_t)vp)>>LOG2_SIZEVNODE) & wrapfs_node_hash])

static LIST_HEAD(wrapfs_node_hashhead, wrapfs_node) *wrapfs_node_hashtbl;
static u_long wrapfs_node_hash;
struct lock wrapfs_hashlock;

static MALLOC_DEFINE(M_WRAPFSHASH, "WRAPFS hash", "WRAPFS hash table");
MALLOC_DEFINE(M_WRAPFSNODE, "WRAPFS node", "WRAPFS vnode private part");

int wrapfs_init(struct vfsconf *vfsp);
int wrapfs_uninit(struct vfsconf *vfsp);
int wrapfs_node_create(struct mount *mp, struct vnode *lowervp, struct vnode **newvpp);
/* defined by EZK */
void * kmem_zalloc(unsigned long size);
int fist_uiomove(caddr_t cp, int n, enum uio_rw rwflag, struct uio *uio);
#ifdef FIST_FILTER_NAME
cn_t * wrapfs_new_cnp(const vnode_t *thisvp, const cn_t *thiscnp);
void wrapfs_update_cnp(const vnode_t *thisvp, cn_t **lowercnpp, cn_t *thiscnp, int error);
#endif /* FIST_FILTER_NAME */
#ifdef FIST_FILTER_DATA
void fist_copy_object_attr(vm_object_t dest, const vm_object_t src);
void fist_copy_page_attr(vm_page_t dest, const vm_page_t src);
int wrapfs_fill_zeros(vnode_t *vp, vattr_t *vap, cred_t *cred, proc_t *p);
#endif /* FIST_FILTER_DATA */

static int	wrapfs_node_alloc(struct mount *mp, struct vnode *lowervp,
				     struct vnode **vpp);
static struct vnode *
		wrapfs_node_find(struct mount *mp, struct vnode *lowervp);

/*
 * Initialise cache headers
 */
int
wrapfs_init(vfsp)
	struct vfsconf *vfsp;
{

  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);
  fist_dprint(2, "wrapfs_init\n");		/* printed during system boot */
  wrapfs_node_hashtbl = hashinit(NWRAPFSNODECACHE, M_WRAPFSHASH, &wrapfs_node_hash);
  lockinit(&wrapfs_hashlock, PVFS, "wrapfs", 0, 0);
  return (0);
}

int
wrapfs_uninit(vfsp)
	struct vfsconf *vfsp;
{

  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);
  if (wrapfs_node_hashtbl) {
    free(wrapfs_node_hashtbl, M_WRAPFSHASH);
  }
  return (0);
}

/*
 * Return a VREF'ed alias for lower vnode if already exists, else 0.
 * Lower vnode should be locked on entry and will be left locked on exit.
 */
static struct vnode *
wrapfs_node_find(mp, lowervp)
	struct mount *mp;
	struct vnode *lowervp;
{
  struct proc *p = curproc;	/* XXX */
  struct wrapfs_node_hashhead *hd;
  struct wrapfs_node *a;
  struct vnode *vp;

  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);

  /*
   * Find hash base, and then search the (two-way) linked
   * list looking for a wrapfs_node structure which is referencing
   * the lower vnode.  If found, the increment the wrapfs_node
   * reference count (but NOT the lower vnode's VREF counter).
   */
  hd = WRAPFS_NHASH(lowervp);
loop:
  lockmgr(&wrapfs_hashlock, LK_EXCLUSIVE, NULL, p);
  LIST_FOREACH(a, hd, wrapfs_hash) {
    if (a->wrapfs_lowervp == lowervp && WRAPFS_TO_VP(a)->v_mount == mp) {
      vp = WRAPFS_TO_VP(a);
      lockmgr(&wrapfs_hashlock, LK_RELEASE, NULL, p);
      /*
       * We need vget for the VXLOCK
       * stuff, but we don't want to lock
       * the lower node.
       */
      if (vget(vp, LK_EXCLUSIVE | LK_CANRECURSE, p)) {
	printf ("wrapfs_node_find: vget failed.\n");
	goto loop;
      }
      VOP_UNLOCK(lowervp, 0, p);
      return (vp);
    }
  }
  lockmgr(&wrapfs_hashlock, LK_RELEASE, NULL, p);

  print_location();
  return NULLVP; /* kiran ??? WRAPFSVP*/
}


/*
 * Make a new wrapfs_node node.
 * Vp is the alias vnode, lofsvp is the lower vnode.
 * Maintain a reference to (lowervp).
 */
static int
wrapfs_node_alloc(mp, lowervp, vpp)
	struct mount *mp;
	struct vnode *lowervp;
	struct vnode **vpp;
{
  struct proc *p = curproc;	/* XXX */
  struct wrapfs_node_hashhead *hd;
  struct wrapfs_node *xp;
  struct vnode *othervp, *vp;
  int error;

  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);

  /*
   * Do the MALLOC before the getnewvnode since doing so afterward
   * might cause a bogus v_data pointer to get dereferenced
   * elsewhere if MALLOC should block.
   */
  MALLOC(xp, struct wrapfs_node *, sizeof(struct wrapfs_node),
	 M_WRAPFSNODE, M_WAITOK);

  error = getnewvnode(VT_WRAPFS, mp, wrapfs_vnodeop_p, vpp);
  if (error) {
    FREE(xp, M_WRAPFSNODE);
    return (error);
  }
  vp = *vpp;

  vp->v_type = lowervp->v_type;
  lockinit(&xp->wrapfs_lock, PINOD, "wrapfsnode", 0, LK_CANRECURSE);
  xp->wrapfs_vnode = vp;
  vp->v_data = xp;
  xp->wrapfs_lowervp = lowervp;
  /*
   * Before we insert our new node onto the hash chains,
   * check to see if someone else has beaten us to it.
   * (We could have slept in MALLOC.)
   */
  othervp = wrapfs_node_find(mp, lowervp);
  if (othervp) {
    vp->v_data = NULL;
    FREE(xp, M_WRAPFSNODE);
    vp->v_type = VBAD;	/* node is discarded */
    vrele(vp);
    *vpp = othervp;
    return 0;
  }

  /*
   * From NetBSD:
   * Now lock the new node. We rely on the fact that we were passed
   * a locked vnode. If the lower node is exporting a struct lock
   * (v_vnlock != NULL) then we just set the upper v_vnlock to the
   * lower one, and both are now locked. If the lower node is exporting
   * NULL, then we copy that up and manually lock the new vnode.
   */

  lockmgr(&wrapfs_hashlock, LK_EXCLUSIVE, NULL, p);
  vp->v_vnlock = lowervp->v_vnlock;
  error = VOP_LOCK(vp, LK_EXCLUSIVE | LK_THISLAYER, p);
  if (error)
    panic("wrapfs_node_alloc: can't lock new vnode\n");

  VREF(lowervp);
  hd = WRAPFS_NHASH(lowervp);
  LIST_INSERT_HEAD(hd, xp, wrapfs_hash);
  lockmgr(&wrapfs_hashlock, LK_RELEASE, NULL, p);
  print_location();
  return 0;
}


/*
 * Try to find an existing wrapfs_node vnode refering to the given underlying
 * vnode (which should be locked). If no vnode found, create a new wrapfs_node
 * vnode which contains a reference to the lower vnode.
 */
int
wrapfs_node_create(mp, lowervp, newvpp)
	struct mount *mp;
	struct vnode *lowervp;
	struct vnode **newvpp;
{
  struct vnode *aliasvp;
  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);

  aliasvp = wrapfs_node_find(mp, lowervp);
  if (aliasvp) {
    /*
     * wrapfs_node_find has taken another reference
     * to the alias vnode.
     */
    vrele(lowervp);
#ifdef FIST_DEBUG
    fist_dprint(4, "wrapfs_node_create: exists", aliasvp);
#endif
  } else {
    int error;

    /*
     * Get new vnode.
     */
    fist_dprint(2, "wrapfs_node_create: create new alias vnode\n");

    /*
     * Make new vnode reference the wrapfs_node.
     */
    error = wrapfs_node_alloc(mp, lowervp, &aliasvp);
    if (error)
      return error;

    /*
     * aliasvp is already VREF'd by getnewvnode()
     */
  }
  /* vrele(lowervp); - was there in 4.0  - check */

#ifdef DIAGNOSTIC
  if (lowervp->v_usecount < 1) {
    /* Should never happen... */
    vprint ("wrapfs_node_create: alias ", aliasvp);
    vprint ("wrapfs_node_create: lower ", lowervp);
    panic ("wrapfs_node_create: lower has 0 usecount.");
  };
#endif

#ifdef FIST_DEBUG
  fist_dprint(2, "wrapfs_node_create: alias", aliasvp);
  fist_dprint(2, "wrapfs_node_create: lower", lowervp);
#endif

  *newvpp = aliasvp;
  print_location();
  return (0);
}

#ifdef DIAGNOSTIC

#ifdef DDB
#define	wrapfs_checkvp_barrier	1
#else
#define	wrapfs_checkvp_barrier	0
#endif

struct vnode *
wrapfs_checkvp(vp, fil, lno)
	struct vnode *vp;
	char *fil;
	int lno;
{
  struct wrapfs_node *a = VP_TO_WRAPFS(vp);
  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);
#ifdef notyet
  /*
   * Can't do this check because vop_reclaim runs
   * with a funny vop vector.
   */
  if (vp->v_op != wrapfs_vnodeop_p) {
    printf ("wrapfs_checkvp: on non-wrapfs-node\n");
    while (wrapfs_checkvp_barrier) /*WAIT*/ ;
    panic("wrapfs_checkvp");
  };
#endif
  if (a->wrapfs_lowervp == NULLVP) { /* kiran - should it be WRAPFSVP ? */
    /* Should never happen */
    int i;
    u_long *p;
    printf("vp = %p, ZERO ptr\n", (void *)vp);
    for (p = (u_long *) a, i = 0; i < 8; i++)
      printf(" %lx", p[i]);
    printf("\n");
    /* wait for debugger */
    while (wrapfs_checkvp_barrier) /*WAIT*/ ;
    panic("wrapfs_checkvp");
  }
  if (a->wrapfs_lowervp->v_usecount < 1) {
    int i;
    u_long *p;
    printf("vp = %p, unref'ed lowervp\n", (void *)vp);
    for (p = (u_long *) a, i = 0; i < 8; i++)
      printf(" %lx", p[i]);
    printf("\n");
    /* wait for debugger */
    while (wrapfs_checkvp_barrier) /*WAIT*/ ;
    panic ("wrapfs with unref'ed lowervp");
  };
#ifdef notyet
  printf("wrapfs %x/%d -> %x/%d [%s, %d]\n",
	 WRAPFS_TO_VP(a), WRAPFS_TO_VP(a)->v_usecount,
	 a->wrapfs_lowervp, a->wrapfs_lowervp->v_usecount,
	 fil, lno);
#endif
  print_location();
  return a->wrapfs_lowervp;
}
#endif

/****************************************************************************/
/* ADDED BY EZK */

/* allocate memory and zero it */
void *
kmem_zalloc(unsigned long size)
{
  void *addr;

  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);
  addr = malloc(size, M_TEMP, M_WAITOK);
  if (addr)
    bzero(addr, size);
  print_location();
  return addr;
}

/* do an I/O move */
int
fist_uiomove(caddr_t cp, int n, enum uio_rw rwflag, struct uio *uio)
{
  enum uio_rw saved_rwflag;
  int ret;

  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);
  if (uio->uio_rw != rwflag) {
    printf("UIOMOVE mismatching flags: uio->uio_rw=%d, rwflag=%d\n",
	   uio->uio_rw, rwflag);
  }
  /* save uio's rwflag */
  saved_rwflag = uio->uio_rw;
  uio->uio_rw = rwflag;
  ret = uiomove(cp, n, uio);
  uio->uio_rw = saved_rwflag;
  print_location();
  return ret;
}

#ifdef FIST_FILTER_NAME
/*
 * create a new (lower) componentname from an existing one
 * (this level) and encode the inner.
 */
cn_t *
wrapfs_new_cnp(const vnode_t *thisvp, const cn_t *thiscnp)
{
  cn_t *lowercnp;
#ifdef FIST_FILTER_NAME
  static char buf[MAXNAMLEN];
#else /* not FIST_FILTER_NAME */
  int len;
  char *name = NULL;
#endif /* not FIST_FILTER_NAME */

  if (!thiscnp)
    return NULL;

  lowercnp = kmem_alloc(sizeof(cn_t));
  if (!lowercnp)
    panic("wrapfs_new_cnp: no more memory\n");
  /* copy all fields */
  bcopy(thiscnp, lowercnp, sizeof(cn_t));
  /* fix certain fields temporarily */
  lowercnp->cn_pnbuf = NULL;
  lowercnp->cn_nameptr = NULL;
  lowercnp->cn_namelen = 0;

  /* prepare new path name for lookup */
#ifdef FIST_FILTER_NAME
  bcopy(thiscnp->cn_nameptr, buf, thiscnp->cn_namelen);
  buf[thiscnp->cn_namelen] = '\0';
  lowercnp->cn_namelen = wrapfs_encode_filename(buf,
			 thiscnp->cn_namelen,
			 &lowercnp->cn_pnbuf,
			 SKIP_DOTS,
			 thisvp, /* use namei zone alloc */
			 thisvp->v_mount);
  /* updated other cnp fields */
  lowercnp->cn_namelen--;	/* don't count terminating null */
  lowercnp->cn_nameptr = lowercnp->cn_pnbuf;
  /* I always allocate my own lowercnp buffer */
#else /* not FIST_FILTER_NAME */
  len = thiscnp->cn_namelen;
  name = zalloc(namei_zone);
  if (!name)
    panic("wrapfs_new_cnp: no more memory for name\n");
  bcopy(thiscnp->cn_nameptr, name, len+1);
  name[len] = '\0';		/* just in case (not needed) */
  /* updated other cnp fields */
  //name[0]++;			/* update first char of name ???*/
  lowercnp->cn_namelen = len;
  lowercnp->cn_nameptr = lowercnp->cn_pnbuf = name;
#endif /* not FIST_FILTER_NAME */

  /* return formed string */
  return lowercnp;
}

/*
 * Update an old cnp based on a newer one.
 * Also free previously allocated lowercnp and its fields.
 */
void
wrapfs_update_cnp(const vnode_t *thisvp, cn_t **lowercnpp, cn_t *thiscnp, int error)
{
  /*
   * free thiscnp if there no error, HASBUF was on, and SAVENAME was off.
   * XXX: what about SAVESTART (from <sys/namei.h>)
   */
  if (!error &&
      (thiscnp->cn_flags & (HASBUF|SAVENAME|SAVESTART)) == HASBUF) {
    fist_dprint(2, "UPDATE_CNP: freeing thiscnp->cn_pnbuf \"%s\"\n",
		thiscnp->cn_pnbuf);
    zfree(namei_zone, thiscnp->cn_pnbuf);
  }

  if (!fist_isdeadcode(*lowercnpp)) {
    fist_dprint(2, "UPDATE_CNP: lowercnp flags <%s>\n",
		fist_cn_flags((*lowercnpp)->cn_flags));
  }

  /* always free space used by lowercnp, if not already free'd */
  if (fist_isdeadcode(*lowercnpp))
    return;



  if (!fist_isdeadcode((*lowercnpp)->cn_pnbuf)) {
    if (((*lowercnpp)->cn_flags & (HASBUF|SAVENAME|SAVESTART)) == HASBUF) {
      fist_dprint(2, "UPDATE_CNP: freeing lowercnp->cn_pnbuf \"%s\"\n",
		  (*lowercnpp)->cn_pnbuf);
      zfree(namei_zone, (*lowercnpp)->cn_pnbuf);
    } else {
      fist_dprint(1, "UPDATE_CNP: not freeing 0x%x \"%s\" <%s>\n",
		  (int) *lowercnpp,
		  (*lowercnpp)->cn_pnbuf,
		  fist_cn_flags((*lowercnpp)->cn_flags));
    }
  }
  fist_dprint(2, "UPDATE_CNP: freeing lowercnp 0x%x\n", (int) *lowercnpp);
  kmem_free(*lowercnpp);

}
#endif /* FIST_FILTER_NAME */

#ifdef FIST_FILTER_DATA

void fist_copy_object_attr(vm_object_t dest, const vm_object_t src)
{
    dest->flags = src->flags;
}

void fist_copy_page_attr(vm_page_t dest, const vm_page_t src)
{
  dest->flags = src->flags;
  dest->dirty = src->dirty;
  dest->valid = src->valid;
}

/*
 * used for truncating a file to an offset beyond its current length
 */
int
wrapfs_fill_zeros(vnode_t *vp, vattr_t *vap, cred_t * cred, proc_t * p)
{
  int error = 0;
  vnode_t *hidden_vp;
  vattr_t hidden_vap;
  uio_t temp_uio;
  iovec_t *temp_iovec, *free_iovec;
  int i;
  caddr_t current_base;
  int resid, bytes_read, num_pages, first_page_bytes, real_first_page;
  long long start_loffset, end_loffset, page_start_loffset;
  long long page_end_loffset, current_loffset;
  int hidden_ioflag = IO_NODELOCKED;
  vm_object_t object;

  fist_dprint(4, "fist_wrapfs_fill_zeros vp=0x%x new size=0x%x \n",
	      (int) vp, vap->va_size);

  hidden_vp = WRAPFS_VP_TO_LOWERVP(vp);
  /* set the size of the vm object */
  vnode_pager_setsize(vp, vap->va_size);
  vnode_pager_setsize(hidden_vp, vap->va_size);

  /* get the attributes, length is necessary for correct updates */
  if ((error = VOP_GETATTR(hidden_vp, &hidden_vap, cred, p))) {
    fist_dprint(4, "VOP_GETATTR returned error - not good\n");
    goto out;
  }

  if (hidden_vap.va_size >= vap->va_size) {
    return 0;
  }

  start_loffset = hidden_vap.va_size;
  end_loffset = vap->va_size;

  page_start_loffset = start_loffset & ~(PAGE_SIZE - 1);
  page_end_loffset = end_loffset & ~(PAGE_SIZE - 1);

  /*
   * if not multiple of PAGE_SIZE, then the above formula loses one page.
   * adjust for it
   */
  if (page_end_loffset < end_loffset)
    page_end_loffset += PAGE_SIZE;

  num_pages = (page_end_loffset - page_start_loffset) >> PAGE_SHIFT;

  temp_iovec = kmem_zalloc(num_pages * sizeof(iovec_t));
  free_iovec = kmem_zalloc(num_pages * sizeof(iovec_t));
  for (i = 0; i < num_pages; i++) {
    temp_iovec[i].iov_len  = free_iovec[i].iov_len  = PAGE_SIZE;
    temp_iovec[i].iov_base = free_iovec[i].iov_base  = kmem_zalloc(PAGE_SIZE);
  }

  /* read first block XXX check length of file */
  temp_uio.uio_iov = temp_iovec;
  temp_uio.uio_iovcnt = 1;
  temp_uio.uio_offset = page_start_loffset;
  temp_uio.uio_segflg = UIO_SYSSPACE;
  temp_uio.uio_rw = UIO_READ;
  temp_uio.uio_procp = p;
  temp_uio.uio_resid = start_loffset - page_start_loffset;
  fist_print_uios("FILL_ZEROS (before VOP_READ 1)", &temp_uio);
  error = VOP_READ(hidden_vp, &temp_uio, hidden_ioflag, cred);
  if (error) {
    fist_dprint(5, "VOP_READ returned error - not good\n");
    goto out_free;
  }

  fist_print_uios("FILL_ZEROS (after VOP_READ 1)", &temp_uio);
  bytes_read = PAGE_SIZE - temp_iovec[0].iov_len;
  temp_iovec[0].iov_base -= bytes_read;
  temp_iovec[0].iov_len = PAGE_SIZE;

  /* decode block read */
  wrapfs_decode_block(temp_iovec[0].iov_base, temp_iovec[0].iov_base,
		      bytes_read, vp, vp->v_mount, OFF_TO_IDX(start_loffset));
  bzero(temp_iovec[0].iov_base + bytes_read, PAGE_SIZE - bytes_read);
  /*
   * Now we are ready to write the bytes within the start/end
   * cleartext offsets in the buffers we allocated.
   */
  current_loffset = page_start_loffset;
  for (i = 0; i < num_pages ; i++) {
    current_base = temp_iovec[i].iov_base;
    wrapfs_fill_page(vp, temp_iovec[i].iov_base, current_loffset);
    wrapfs_encode_block(temp_iovec[i].iov_base, temp_iovec[i].iov_base,
			PAGE_SIZE, vp, vp->v_mount, OFF_TO_IDX(current_loffset));
    wrapfs_fill_page(hidden_vp, temp_iovec[i].iov_base, current_loffset);
    current_loffset += PAGE_SIZE;
  }

  resid = end_loffset - page_start_loffset;

  /* XXX: no need for full initialization here */
  temp_uio.uio_iov = temp_iovec;
  temp_uio.uio_iovcnt = num_pages;
  temp_uio.uio_offset = page_start_loffset;
  temp_uio.uio_segflg = UIO_SYSSPACE;
  temp_uio.uio_rw = UIO_WRITE;
  temp_uio.uio_procp = p;
  temp_uio.uio_resid = resid;

  /*
   * pass operation to hidden filesystem, and return status
   */
  fist_print_uios("FILL_ZEROS (before write)", &temp_uio);
  error = VOP_WRITE(hidden_vp, &temp_uio, hidden_ioflag, cred);
  fist_print_uios("FILL_ZEROS (after write)", &temp_uio);

out_free:
  for (i = 0; i < num_pages; i++) {
    fist_dprint(6, "PRINT_BASE1 %d: 0x%x (len=%d)\n", i,
		temp_iovec[i].iov_base,
		temp_iovec[i].iov_len);
    kmem_free(free_iovec[i].iov_base);
  }
  kmem_free(temp_iovec);

out:

  print_location();
  return (error);
}
#endif /* FIST_FILTER_DATA */
