/*
 * Copyright (c) 1997-2005 Erez Zadok <ezk@cs.stonybrook.edu>
 * Copyright (c) 2001-2005 Stony Brook University
 *
 * For specific licensing information, see the COPYING file distributed with
 * this package, or get one from ftp://ftp.filesystems.org/pub/fistgen/COPYING.
 *
 * This Copyright notice must be kept intact and distributed with all
 * fistgen sources INCLUDING sources generated by fistgen.
 */
/*
 * Copyright (c) 1992, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * John Heidemann of the UCLA Ficus project.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	@(#)wrapfs_vnops.c	8.6 (Berkeley) 5/27/95
 *
 * Ancestors:
 *	@(#)lofs_vnops.c	1.2 (Berkeley) 6/18/92
 * $FreeBSD: src/sys/miscfs/wrapfs/wrapfs_vnops.c,v 1.38.2.5 2001/06/26 04:20:10 bp Exp $
 *	...and...
 *	@(#)wrapfs_vnodeops.c 1.20 92/07/07 UCLA Ficus project
 *
 * $FreeBSD: src/sys/miscfs/wrapfs/wrapfs_vnops.c,v 1.38.2.5 2001/06/26 04:20:10 bp Exp $
 */

/*
 * Wrapfs Layer
 *
 * (See mount_wrapfs(8) for more information.)
 *
 * The wrapfs layer duplicates a portion of the file system
 * name space under a new name.  In this respect, it is
 * similar to the loopback file system.  It differs from
 * the loopback fs in two respects:  it is implemented using
 * a stackable layers techniques, and its "wrapfs-node"s stack above
 * all lower-layer vnodes, not just over directory vnodes.
 *
 * The wrapfs layer has two purposes.  First, it serves as a demonstration
 * of layering by proving a layer which does nothing.  (It actually
 * does everything the loopback file system does, which is slightly
 * more than nothing.)  Second, the wrapfs layer can serve as a prototype
 * layer.  Since it provides all necessary layer framework,
 * new file system layers can be created very easily be starting
 * with a wrapfs layer.
 *
 * The remainder of this man page examines the wrapfs layer as a basis
 * for constructing new layers.
 *
 *
 * INSTANTIATING NEW WRAPFS LAYERS
 *
 * New wrapfs layers are created with mount_wrapfs(8).
 * Mount_wrapfs(8) takes two arguments, the pathname
 * of the lower vfs (target-pn) and the pathname where the wrapfs
 * layer will appear in the namespace (alias-pn).  After
 * the wrapfs layer is put into place, the contents
 * of target-pn subtree will be aliased under alias-pn.
 *
 *
 * OPERATION OF A WRAPFS LAYER
 *
 * The wrapfs layer is the minimum file system layer,
 * simply bypassing all possible operations to the lower layer
 * for processing there.  The majority of its activity centers
 * on the bypass routine, through which nearly all vnode operations
 * pass.
 *
 * The bypass routine accepts arbitrary vnode operations for
 * handling by the lower layer.  It begins by examing vnode
 * operation arguments and replacing any wrapfs-nodes by their
 * lower-layer equivlants.  It then invokes the operation
 * on the lower layer.  Finally, it replaces the wrapfs-nodes
 * in the arguments and, if a vnode is return by the operation,
 * stacks a wrapfs-node on top of the returned vnode.
 *
 * Although bypass handles most operations, vop_getattr, vop_lock,
 * vop_unlock, vop_inactive, vop_reclaim, and vop_print are not
 * bypassed. Vop_getattr must change the fsid being returned.
 * Vop_lock and vop_unlock must handle any locking for the
 * current vnode as well as pass the lock request down.
 * Vop_inactive and vop_reclaim are not bypassed so that
 * they can handle freeing wrapfs-layer specific data. Vop_print
 * is not bypassed to avoid excessive debugging information.
 * Also, certain vnode operations change the locking state within
 * the operation (create, mknod, remove, link, rename, mkdir, rmdir,
 * and symlink). Ideally these operations should not change the
 * lock state, but should be changed to let the caller of the
 * function unlock them. Otherwise all intermediate vnode layers
 * (such as union, umapfs, etc) must catch these functions to do
 * the necessary locking at their layer.
 *
 *
 * INSTANTIATING VNODE STACKS
 *
 * Mounting associates the wrapfs layer with a lower layer,
 * effect stacking two VFSes.  Vnode stacks are instead
 * created on demand as files are accessed.
 *
 * The initial mount creates a single vnode stack for the
 * root of the new wrapfs layer.  All other vnode stacks
 * are created as a result of vnode operations on
 * this or other wrapfs vnode stacks.
 *
 * New vnode stacks come into existance as a result of
 * an operation which returns a vnode.
 * The bypass routine stacks a wrapfs-node above the new
 * vnode before returning it to the caller.
 *
 * For example, imagine mounting a wrapfs layer with
 * "mount_wrapfs /usr/include /dev/layer/wrapfs".
 * Changing directory to /dev/layer/wrapfs will assign
 * the root wrapfs-node (which was created when the wrapfs layer was mounted).
 * Now consider opening "sys".  A vop_lookup would be
 * done on the root wrapfs-node.  This operation would bypass through
 * to the lower layer which would return a vnode representing
 * the UFS "sys".  wrapfs_bypass then builds a wrapfs-node
 * aliasing the UFS "sys" and returns this to the caller.
 * Later operations on the wrapfs-node "sys" will repeat this
 * process when constructing other vnode stacks.
 *
 *
 * CREATING OTHER FILE SYSTEM LAYERS
 *
 * One of the easiest ways to construct new file system layers is to make
 * a copy of the wrapfs layer, rename all files and variables, and
 * then begin modifing the copy.  Sed can be used to easily rename
 * all variables.
 *
 * The umap layer is an example of a layer descended from the
 * wrapfs layer.
 *
 *
 * INVOKING OPERATIONS ON LOWER LAYERS
 *
 * There are two techniques to invoke operations on a lower layer
 * when the operation cannot be completely bypassed.  Each method
 * is appropriate in different situations.  In both cases,
 * it is the responsibility of the aliasing layer to make
 * the operation arguments "correct" for the lower layer
 * by mapping an vnode arguments to the lower layer.
 *
 * The first approach is to call the aliasing layer's bypass routine.
 * This method is most suitable when you wish to invoke the operation
 * currently being handled on the lower layer.  It has the advantage
 * that the bypass routine already must do argument mapping.
 * An example of this is wrapfs_getattrs in the wrapfs layer.
 *
 * A second approach is to directly invoke vnode operations on
 * the lower layer with the VOP_OPERATIONNAME interface.
 * The advantage of this method is that it is easy to invoke
 * arbitrary operations on the lower layer.  The disadvantage
 * is that vnode arguments must be manualy mapped.
 *
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */
#ifdef FISTGEN
# include "fist_wrapfs.h"
#endif /* FISTGEN */
#include <fist.h>
#include <wrapfs.h>

#if 0
#include "opt_debug.h"

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/sysctl.h>
#include <sys/vnode.h>
#include <sys/mount.h>
#include <sys/namei.h>
#include <sys/malloc.h>
#include <sys/buf.h>
#include <sys/proc.h>
#include <sys/ioccom.h>
#include <sys/dirent.h>

#include <vm/vm.h>
#include <vm/vm_extern.h>
#include <vm/vm_zone.h>
#include <vm/vnode_pager.h>
#include <vm/vm_prot.h>
#include <vm/vm_page.h>
#include <vm/vm_pageout.h>
#include <vm/vm_object.h>
#include <vm/vm_pager.h>
#include <wrapfs.h>
#endif

static int wrapfs_bug_bypass = 0;   /* for debugging: enables bypass printf'ing */
SYSCTL_INT(_debug, OID_AUTO, wrapfs_bug_bypass, CTLFLAG_RW, &wrapfs_bug_bypass, 0, "");

static int	wrapfs_access(struct vop_access_args *ap);
       int      wrapfs_bypass(struct vop_generic_args *ap);
static int	wrapfs_createvobject(struct vop_createvobject_args *ap);
static int	wrapfs_destroyvobject(struct vop_destroyvobject_args *ap);
static int	wrapfs_getattr(struct vop_getattr_args *ap);
static int	wrapfs_getvobject(struct vop_getvobject_args *ap);
static int	wrapfs_inactive(struct vop_inactive_args *ap);
static int      wrapfs_ioctl(struct vop_ioctl_args *ap);
static int	wrapfs_islocked(struct vop_islocked_args *ap);
static int	wrapfs_lock(struct vop_lock_args *ap);
static int	wrapfs_lookup(struct vop_lookup_args *ap);
static int      wrapfs_mmap(struct vop_mmap_args *ap);
static int	wrapfs_open(struct vop_open_args *ap);
static int	wrapfs_print(struct vop_print_args *ap);
static int	wrapfs_reclaim(struct vop_reclaim_args *ap);
static int	wrapfs_rename(struct vop_rename_args *ap);
static int	wrapfs_setattr(struct vop_setattr_args *ap);
static int	wrapfs_unlock(struct vop_unlock_args *ap);


#ifdef FIST_FILTER_DATA
void wrapfs_fill_page(vnode_t *vp, char *buf, long long offset);
static int wrapfs_read(struct vop_read_args *ap);
static int wrapfs_write(struct vop_write_args *ap);
static int wrapfs_getpages(struct vop_getpages_args *ap);
static int wrapfs_putpages(struct vop_putpages_args *ap);
#endif /* FIST_FILTER_DATA */
#ifdef FIST_FILTER_NAME
static int wrapfs_mkdir(struct vop_mkdir_args *ap);
static int wrapfs_rmdir(struct vop_rmdir_args *ap);
static int wrapfs_create(struct vop_create_args *ap);
static int wrapfs_remove(struct vop_remove_args *ap);
static int wrapfs_rename(struct vop_rename_args *ap);
static int wrapfs_link(struct vop_link_args *ap);
static int wrapfs_symlink(struct vop_symlink_args *ap);
static int wrapfs_readlink(struct vop_readlink_args *ap);
static int wrapfs_whiteout(struct vop_whiteout_args *ap);
static int wrapfs_mknod(struct vop_mknod_args *ap);
static int wrapfs_readdir(struct vop_readdir_args *ap);
#endif /* FIST_FILTER_NAME */

/*
 * This is the 10-Apr-92 bypass routine.
 *    This version has been optimized for speed, throwing away some
 * safety checks.  It should still always work, but it's not as
 * robust to programmer errors.
 *
 * In general, we map all vnodes going down and unmap them on the way back.
 * As an exception to this, vnodes can be marked "unmapped" by setting
 * the Nth bit in operation's vdesc_flags.
 *
 * Also, some BSD vnode operations have the side effect of vrele'ing
 * their arguments.  With stacking, the reference counts are held
 * by the upper node, not the lower one, so we must handle these
 * side-effects here.  This is not of concern in Sun-derived systems
 *
 *
 * This makes the following assumptions:
 * - only one returned vpp
 * - no INOUT vpp's (Sun's vop_open has one of these)
 * - the vnode operation vector of the first vnode should be used
 *   to determine what implementation of the op should be invoked
 * - all mapped vnodes are of our vnode-type (NEEDSWORK:
 *   problems on rmdir'ing mount points and renaming?)
 */

int
wrapfs_bypass(ap)
     struct vop_generic_args /* {
				struct vnodeop_desc *a_desc;
				<other random data follows, presumably>
				} */ *ap;
{
  register struct vnode **this_vp_p;
  int error;
  struct vnode *old_vps[VDESC_MAX_VPS];
  struct vnode **vps_p[VDESC_MAX_VPS];
  struct vnode ***vppp;
  struct vnodeop_desc *descp = ap->a_desc;
  int reles, i;

  fist_dprint(4, "FXN=%s FILE=%s LINE=%d FOR=%s\n",
	      __FUNCTION__,__FILE__,__LINE__,descp->vdesc_name);

  if (wrapfs_bug_bypass)
    fist_dprint(5, "wrapfs_bypass: %s\n", descp->vdesc_name);

#ifdef DIAGNOSTIC
  /*
   * We require at least one vp.
   */
  if (descp->vdesc_vp_offsets == NULL ||
      descp->vdesc_vp_offsets[0] == VDESC_NO_OFFSET)
    panic ("wrapfs_bypass: no vp's in map");
#endif

  /*
   * Map the vnodes going in.
   * Later, we'll invoke the operation based on
   * the first mapped vnode's operation vector.
   */
  reles = descp->vdesc_flags;
  for (i = 0; i < VDESC_MAX_VPS; reles >>= 1, i++) {
    if (descp->vdesc_vp_offsets[i] == VDESC_NO_OFFSET)
      break;   /* bail out at end of list */
    vps_p[i] = this_vp_p =
      VOPARG_OFFSETTO(struct vnode**,descp->vdesc_vp_offsets[i],ap);
    /*
     * We're not guaranteed that any but the first vnode
     * are of our type.  Check for and don't map any
     * that aren't.  (We must always map first vp or vclean fails.)
     */
    if (i && (*this_vp_p == NULLVP ||
	      (*this_vp_p)->v_op != wrapfs_vnodeop_p)) {
      old_vps[i] = NULLVP;
    } else {
      old_vps[i] = *this_vp_p;
      *(vps_p[i]) = WRAPFS_VP_TO_LOWERVP(*this_vp_p);
      /*
       * XXX - Several operations have the side effect
       * of vrele'ing their vp's.  We must account for
       * that.  (This should go away in the future.)
       */
      if (reles & VDESC_VP0_WILLRELE)
	VREF(*this_vp_p);
    }

  }

  /*
   * Call the operation on the lower layer
   * with the modified argument structure.
   */
  if (vps_p[0] && *vps_p[0])
    error = VCALL(*(vps_p[0]), descp->vdesc_offset, ap);
  else {
    printf("wrapfs_bypass: no map for %s\n", descp->vdesc_name);
    error = EINVAL;
  }

  /*
   * Maintain the illusion of call-by-value
   * by restoring vnodes in the argument structure
   * to their original value.
   */
  reles = descp->vdesc_flags;
  for (i = 0; i < VDESC_MAX_VPS; reles >>= 1, i++) {
    if (descp->vdesc_vp_offsets[i] == VDESC_NO_OFFSET)
      break;   /* bail out at end of list */
    if (old_vps[i]) {
      *(vps_p[i]) = old_vps[i];
#if 0
      if (reles & VDESC_VP0_WILLUNLOCK)
	VOP_UNLOCK(*(vps_p[i]), LK_THISLAYER, curproc);
#endif
      if (reles & VDESC_VP0_WILLRELE)
	vrele(*(vps_p[i]));
    }
  }

  /*
   * Map the possible out-going vpp
   * (Assumes that the lower layer always returns
   * a VREF'ed vpp unless it gets an error.)
   */
  if (descp->vdesc_vpp_offset != VDESC_NO_OFFSET &&
      !(descp->vdesc_flags & VDESC_NOMAP_VPP) &&
      !error) {
    /*
     * XXX - even though some ops have vpp returned vp's,
     * several ops actually vrele this before returning.
     * We must avoid these ops.
     * (This should go away when these ops are regularized.)
     */
    if (descp->vdesc_flags & VDESC_VPP_WILLRELE)
      goto out;
    vppp = VOPARG_OFFSETTO(struct vnode***,
			   descp->vdesc_vpp_offset,ap);
    if (*vppp)
      error = wrapfs_node_create(old_vps[0]->v_mount, **vppp, *vppp);
  }

 out:
  print_location();
  return (error);
}

/*
 * We have to carry on the locking protocol on the wrapfs layer vnodes
 * as we progress through the tree. We also have to enforce read-only
 * if this layer is mounted read-only.
 */
static int
wrapfs_lookup(ap)
     struct vop_lookup_args /* {
			       struct vnode * a_dvp;
			       struct vnode ** a_vpp;
			       struct componentname * a_cnp;
			       } */ *ap;
{
  struct componentname *cnp = ap->a_cnp;
  struct vnode *dvp = ap->a_dvp;
  struct proc *p = cnp->cn_proc;
  int flags = cnp->cn_flags;
  struct vnode *vp, *ldvp, *lvp;
  int error;
  CNP_VARS;

  fist_dprint(2, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);
  if ((flags & ISLASTCN) && (dvp->v_mount->mnt_flag & MNT_RDONLY) &&
      (cnp->cn_nameiop == DELETE || cnp->cn_nameiop == RENAME))
    return (EROFS);
  /*
   * Although it is possible to call wrapfs_bypass(), we'll do
   * a direct call to reduce overhead
   */
  ldvp = WRAPFS_VP_TO_LOWERVP(dvp);
  vp = lvp = NULL;

  CNP_BEFORE( ap->a_dvp);

#ifdef FIST_FILTER_NAME
  error = VOP_LOOKUP(ldvp, &lvp, lowercnp);
#else  /* not FIST_FILTER_NAME */
  error = VOP_LOOKUP(ldvp, &lvp, cnp);
#endif /* not FIST_FILTER_NAME */

#ifdef FIST_FILTER_NAME
  thiscnp->cn_flags = lowercnp->cn_flags;
#endif /* FIST_FILTER_NAME */
  CNP_AFTER(ap->a_dvp);

#ifdef FIST_FILTER_NAME
  /*if (!error &&
    (thiscnp->cn_flags & MAKEENTRY) &&
    thiscnp->cn_nameiop != CREATE) {
    fist_dprint(1, "LK cache_enter\n");
    cache_enter(ap->a_dvp, *(ap->a_vpp), thiscnp);
    }*/
#endif /* FIST_FILTER_NAME */

  if (error == EJUSTRETURN && (flags & ISLASTCN) &&
      (dvp->v_mount->mnt_flag & MNT_RDONLY) &&
      (cnp->cn_nameiop == CREATE || cnp->cn_nameiop == RENAME))
    error = EROFS;

  /*
   * Rely only on the PDIRUNLOCK flag which should be carefully
   * tracked by underlying filesystem.
   */
  if (cnp->cn_flags & PDIRUNLOCK)
    VOP_UNLOCK(dvp, LK_THISLAYER, p);
  if ((error == 0 || error == EJUSTRETURN) && lvp != NULL) {
    if (ldvp == lvp) {
      *ap->a_vpp = dvp;
      VREF(dvp);
      vrele(lvp);
    } else {
      error = wrapfs_node_create(dvp->v_mount, lvp, &vp);
      if (error == 0)
	*ap->a_vpp = vp;
    }
  }

  print_location();
  return (error);
}

/*
 * Setattr call. Disallow write attempts if the layer is mounted read-only.
 */
int
wrapfs_setattr(ap)
     struct vop_setattr_args /* {
				struct vnodeop_desc *a_desc;
				struct vnode *a_vp;
				struct vattr *a_vap;
				struct ucred *a_cred;
				struct proc *a_p;
				} */ *ap;
{
  struct vnode *vp = ap->a_vp;
  struct vattr *vap = ap->a_vap;
  int error = 0;

  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);

  if ((vap->va_flags != VNOVAL || vap->va_uid != (uid_t)VNOVAL ||
       vap->va_gid != (gid_t)VNOVAL || vap->va_atime.tv_sec != VNOVAL ||
       vap->va_mtime.tv_sec != VNOVAL || vap->va_mode != (mode_t)VNOVAL) &&
      (vp->v_mount->mnt_flag & MNT_RDONLY))
    return (EROFS);
  if (vap->va_size != VNOVAL) {
    switch (vp->v_type) {
    case VDIR:
      return (EISDIR);
    case VCHR:
    case VBLK:
    case VSOCK:
    case VFIFO:
      if (vap->va_flags != VNOVAL)
	return (EOPNOTSUPP);
      return (0);
    case VREG:
    case VLNK:
    default:
      /*
       * Disallow write attempts if the filesystem is
       * mounted read-only.
       */
      if (vp->v_mount->mnt_flag & MNT_RDONLY)
	return (EROFS);
#ifdef FIST_FILTER_DATA
      error = wrapfs_fill_zeros(vp, vap, ap->a_cred, ap->a_p);
      if (error)
	return error;
#endif /* FIST_FILTER_DATA */
    }
  }

  print_location();
  return wrapfs_bypass((struct vop_generic_args *)ap);
}

/*
 *  We handle getattr only to change the fsid.
 */
static int
wrapfs_getattr(ap)
     struct vop_getattr_args /* {
				struct vnode *a_vp;
				struct vattr *a_vap;
				struct ucred *a_cred;
				struct proc *a_p;
				} */ *ap;
{
  int error;
  vnode_t *vp = ap->a_vp;

  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);
  if ((error = wrapfs_bypass((struct vop_generic_args *)ap)) != 0)
    return (error);

  ap->a_vap->va_fsid = ap->a_vp->v_mount->mnt_stat.f_fsid.val[0]; /* ??? */
#ifdef FIST_FILTER_DATA
  if (vp->v_object) {
    if (ap->a_vap->va_size > vp->v_object->un_pager.vnp.vnp_size)
      vnode_pager_setsize(vp, ap->a_vap->va_size);
  }
#endif /* FIST_FILTER_DATA */
  print_location();
  return (0);
}

/*
 * Handle to disallow write access if mounted read-only.
 */
static int
wrapfs_access(ap)
     struct vop_access_args /* {
			       struct vnode *a_vp;
			       int  a_mode;
			       struct ucred *a_cred;
			       struct proc *a_p;
			       } */ *ap;
{
  struct vnode *vp = ap->a_vp;
  mode_t mode = ap->a_mode;

  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);
  /*
   * Disallow write attempts on read-only layers;
   * unless the file is a socket, fifo, or a block or
   * character device resident on the file system.
   */
  if (mode & VWRITE) {
    switch (vp->v_type) {
    case VDIR:
    case VLNK:
    case VREG:
      if (vp->v_mount->mnt_flag & MNT_RDONLY)
	return (EROFS);
      break;
    default:
      break;
    }
  }

  print_location();
  return wrapfs_bypass((struct vop_generic_args *)ap);
}

/*
 * We must handle open to be able to catch MNT_NODEV and friends.
 */
static int
wrapfs_open(ap)
     struct vop_open_args /* {
			     struct vnode *a_vp;
			     int  a_mode;
			     struct ucred *a_cred;
			     struct proc *a_p;
			     } */ *ap;
{
  struct vnode *vp = ap->a_vp;
  struct vnode *lvp = WRAPFS_VP_TO_LOWERVP(ap->a_vp);

  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);

  if ((vp->v_mount->mnt_flag & MNT_NODEV) &&
      (lvp->v_type == VBLK || lvp->v_type == VCHR))
    return ENXIO;

  print_location();
  return wrapfs_bypass((struct vop_generic_args *)ap);
}

/*
 * We need to process our own vnode lock and then clear the
 * interlock flag as it applies only to our vnode, not the
 * vnodes below us on the stack.
 */
static int
wrapfs_lock(ap)
 	 struct vop_lock_args /* {
			     struct vnode *a_vp;
			     int a_flags;
			     struct proc *a_p;
			     } */ *ap;
{
  struct vnode *vp = ap->a_vp;
  int flags = ap->a_flags;
  struct proc *p = ap->a_p;
  struct wrapfs_node *np = VP_TO_WRAPFS(vp);
  struct vnode *lvp;
  int error;

  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);

  if (flags & LK_THISLAYER) {
    if (vp->v_vnlock != NULL)
      return 0;	/* lock is shared across layers */
    error = lockmgr(&np->wrapfs_lock, flags & ~LK_THISLAYER,
		    &vp->v_interlock, p);
    return (error);
  }

  if (vp->v_vnlock != NULL) {
    /*
     * The lower level has exported a struct lock to us. Use
     * it so that all vnodes in the stack lock and unlock
     * simultaneously. Note: we don't DRAIN the lock as DRAIN
     * decommissions the lock - just because our vnode is
     * going away doesn't mean the struct lock below us is.
     * LK_EXCLUSIVE is fine.
     */
    if ((flags & LK_TYPE_MASK) == LK_DRAIN) {
      fist_dprint(2, "wrapfs_lock: avoiding LK_DRAIN\n");
      return(lockmgr(vp->v_vnlock,
		     (flags & ~LK_TYPE_MASK) | LK_EXCLUSIVE,
		     &vp->v_interlock, p));
    }
    return(lockmgr(vp->v_vnlock, flags, &vp->v_interlock, p));
  }
  /*
   * To prevent race conditions involving doing a lookup
   * on "..", we have to lock the lower node, then lock our
   * node. Most of the time it won't matter that we lock our
   * node (as any locking would need the lower one locked
   * first). But we can LK_DRAIN the upper lock as a step
   * towards decomissioning it.
   */
  lvp = WRAPFS_VP_TO_LOWERVP(vp);
  if (lvp == NULL)
    return (lockmgr(&np->wrapfs_lock, flags, &vp->v_interlock, p));
  if (flags & LK_INTERLOCK) {
    VI_UNLOCK(vp);
    flags &= ~LK_INTERLOCK;
  }
  if ((flags & LK_TYPE_MASK) == LK_DRAIN) {
    error = VOP_LOCK(lvp,
		     (flags & ~LK_TYPE_MASK) | LK_EXCLUSIVE, p);
  } else
    error = VOP_LOCK(lvp, flags, p);
  if (error)
    return (error);
  error = lockmgr(&np->wrapfs_lock, flags, &vp->v_interlock, p);
  if (error)
    VOP_UNLOCK(lvp, 0, p);
  print_location();
  return (error);
}

/*
 * We need to process our own vnode unlock and then clear the
 * interlock flag as it applies only to our vnode, not the
 * vnodes below us on the stack.
 */
static int
wrapfs_unlock(ap)
     struct vop_unlock_args /* {
			       struct vnode *a_vp;
			       int a_flags;
			       struct proc *a_p;
			       } */ *ap;
{
  struct vnode *vp = ap->a_vp;
  int flags = ap->a_flags;
  struct proc *p = ap->a_p;
  struct wrapfs_node *np = VP_TO_WRAPFS(vp);
  struct vnode *lvp;

  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);

  if (vp->v_vnlock != NULL) {
    if (flags & LK_THISLAYER)
      return 0;	/* the lock is shared across layers */
    flags &= ~LK_THISLAYER;
    return (lockmgr(vp->v_vnlock, flags | LK_RELEASE,
		    &vp->v_interlock, p));
  }
  lvp = WRAPFS_VP_TO_LOWERVP(vp);
  if (lvp == NULL)
    return (lockmgr(&np->wrapfs_lock, flags | LK_RELEASE, &vp->v_interlock, p));
  if ((flags & LK_THISLAYER) == 0) {
    if (flags & LK_INTERLOCK) {
      VI_UNLOCK(vp);
      flags &= ~LK_INTERLOCK;
    }
    VOP_UNLOCK(lvp, flags, p);
  } else
    flags &= ~LK_THISLAYER;
  ap->a_flags = flags;
  print_location();
  return (lockmgr(&np->wrapfs_lock, flags | LK_RELEASE, &vp->v_interlock, p));
}

static int
wrapfs_islocked(ap)
     struct vop_islocked_args /* {
				 struct vnode *a_vp;
				 struct proc *a_p;
				 } */ *ap;
{
  struct vnode *vp = ap->a_vp;
  struct proc *p = ap->a_p;

  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);

  if (vp->v_vnlock != NULL)
    return (lockstatus(vp->v_vnlock, p));
  print_location();
  return (lockstatus(&VP_TO_WRAPFS(vp)->wrapfs_lock, p));
}


/*
 * There is no way to tell that someone issued remove/rmdir operation
 * on the underlying filesystem. For now we just have to release lowevrp
 * as soon as possible.
 */
static int
wrapfs_inactive(ap)
     struct vop_inactive_args /* {
				 struct vnode *a_vp;
				 struct proc *a_p;
				 } */ *ap;
{
  struct vnode *vp = ap->a_vp;
  struct proc *p = ap->a_p;
  struct wrapfs_node *xp = VP_TO_WRAPFS(vp);
  struct vnode *lowervp = xp->wrapfs_lowervp;

  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);

  lockmgr(&wrapfs_hashlock, LK_EXCLUSIVE, NULL, p);
  LIST_REMOVE(xp, wrapfs_hash);
  lockmgr(&wrapfs_hashlock, LK_RELEASE, NULL, p);

  xp->wrapfs_lowervp = NULLVP;
  if (vp->v_vnlock != NULL) {
    vp->v_vnlock = &xp->wrapfs_lock;	/* we no longer share the lock */
  } else
    VOP_UNLOCK(vp, LK_THISLAYER, p);

  if (lowervp) {
    vput(lowervp);
    /*
     * Now it is safe to drop references to the lower vnode.
     * VOP_INACTIVE() will be called by vrele() if necessary.
     */
    vrele (lowervp);
  }

  print_location();
  return (0);
}

/*
 * We can free memory in wrapfs_inactive, but we do this
 * here. (Possible to guard vp->v_data to point somewhere)
 */
static int
wrapfs_reclaim(ap)
     struct vop_reclaim_args /* {
				struct vnode *a_vp;
				struct proc *a_p;
				} */ *ap;
{
  struct vnode *vp = ap->a_vp;
  void *vdata = vp->v_data;

  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);

  vp->v_data = NULL;
  FREE(vdata, M_WRAPFSNODE);

  print_location();
  return (0);
}

static int
wrapfs_print(ap)
     struct vop_print_args /* {
			      struct vnode *a_vp;
			      } */ *ap;
{
  struct vnode *vp = ap->a_vp;

  printf ("\ttag VT_WRAPFS, vp=%p, lowervp=%p\n", vp, WRAPFS_VP_TO_LOWERVP(vp));
  if (vp->v_vnlock != NULL) {
    printf("\tvnlock: ");
    lockmgr_printinfo(vp->v_vnlock);
  } else {
    printf("\twrapfs_lock: ");
    lockmgr_printinfo(&VP_TO_WRAPFS(vp)->wrapfs_lock);
  }
  printf("\n");
  return (0);
}

#ifdef FIST_FILTER_DATA
/*
 * if FIST_FILTER_DATA, then we need to create 2 objects, 1 for
 * lower level vnode , another for wrapfs vnode
 */
static int
wrapfs_createvobject(ap)
     struct vop_createvobject_args /* {
				      struct vnode *vp;
				      struct ucred *cred;
				      struct proc *p;
				      } */ *ap;
{
  struct vnode *vp = ap->a_vp;
  struct vnode *lowervp = VP_TO_WRAPFS(vp) ? WRAPFS_VP_TO_LOWERVP(vp) : NULL;
  int error = 0;

  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);

  if (vp->v_type == VNON || lowervp == NULL) {
    error = EINVAL; /* xxx - is this right */
    goto retn;
  }
  /* create object for higher level vnode */
  error = vop_stdcreatevobject(ap);
  if (error)
    goto retn;
  /* now create the object for the lower layer */
  error = VOP_CREATEVOBJECT(lowervp, ap->a_cred, ap->a_p);
  if (error) {
    struct vop_destroyvobject_args destroy;
    destroy.a_vp = vp;
    vop_stddestroyvobject(&destroy);
  }

retn:
  print_location();
  return (error);
}
#else /* not FIST_FILTER_DATA */
/* let the underlying filesystem do the work */
static int
wrapfs_createvobject(ap)
     struct vop_createvobject_args /* {
				      struct vnode *vp;
				      struct ucred *cred;
				      struct proc *p;
				      } */ *ap;
{
  struct vnode *vp = ap->a_vp;
  struct vnode *lowervp = VP_TO_WRAPFS(vp) ? WRAPFS_VP_TO_LOWERVP(vp) : NULL;
  int error = 0;

  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);

  if (vp->v_type == VNON || lowervp == NULL) {
    goto retn; /* returns 0! */
  }
  error = VOP_CREATEVOBJECT(lowervp, ap->a_cred, ap->a_p);
  if (error)
    goto retn;
  vp->v_flag |= VOBJBUF;
retn:
  print_location();
  return (error);
}
#endif /* not FIST_FILTER_DATA */

#ifdef FIST_FILTER_DATA
/*
 * if FIST_FILTER_DATA, then if the object of the higher vnode
 * or lower level vnode exists, then delete them
 */
static int
wrapfs_destroyvobject(ap)
     struct vop_destroyvobject_args /* {
                                       struct vnode *vp;
                                       } */ *ap;
{
  int error = 0;
  struct vnode *vp = ap->a_vp;
  struct vnode *lvp = WRAPFS_VP_TO_LOWERVP(vp);
  vm_object_t obj = vp->v_object;
  vm_object_t lower_obj;
  struct vop_destroyvobject_args destroy;

  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);

  /* if the lower vnode exists destroy its object */
  if (lvp) {
    error = VOP_GETVOBJECT(lvp, &lower_obj);
    if (!error)
      error = VOP_DESTROYVOBJECT(lvp);
    if (error) {
      printk(" error destroying the lowervp \n");
    }
  }
  destroy.a_vp = vp;
  vop_stddestroyvobject(&destroy);

  print_location();
  return (error);
}
#else /* not FIST_FILTER_DATA */
/* do nothing */
static int
wrapfs_destroyvobject(ap)
     struct vop_destroyvobject_args /* {
                                       struct vnode *vp;
                                       } */ *ap;
{
  int error = 0;
  struct vnode *vp = ap->a_vp;

  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);

  vp->v_flag &= ~VOBJBUF;

  print_location();
  return (error);
}
#endif /* not FIST_FILTER_DATA */

#ifdef FIST_FILTER_DATA
static int
wrapfs_getvobject(ap)
     struct vop_getvobject_args /* {
                                   struct vnode *vp;
                                   struct vm_object **objpp;
                                   } */ *ap;
{
  struct vnode *vp = ap->a_vp;
  struct vnode *lowervp = WRAPFS_VP_TO_LOWERVP(ap->a_vp);
  struct vm_object **objpp = ap->a_objpp;

  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);
  /* XXX hack */
  if (lowervp == NULL)
    return EINVAL;

  if (objpp)
    *objpp = vp->v_object;
  print_location();
  return (vp->v_object ? 0 : EINVAL);
}
#else /* not FIST_FILTER_DATA */
static int
wrapfs_getvobject(ap)
     struct vop_getvobject_args /* {
				   struct vnode *vp;
				   struct vm_object **objpp;
				   } */ *ap;
{
  struct vnode *lvp = WRAPFS_VP_TO_LOWERVP(ap->a_vp);

  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);
  if (lvp == NULL)
    return EINVAL;
  print_location();
  return (VOP_GETVOBJECT(lvp, ap->a_objpp));
}
#endif /* not FIST_FILTER_DATA */

/****************************************************************************/

#ifdef FIST_FILTER_DATA

/*
 * Create a page in the vnode and insert data into it.
 * This is used to synchronize data between the VM and read/write interface.
 * That that was written using write() gets pages containing the same data, so
 * that subsequent mmap() ops get valid data.
 */
void
wrapfs_fill_page(vnode_t *vp, char *buf, long long offset)
{
  vm_page_t pp;
  vm_offset_t kva;
  caddr_t ca;

  fist_dprint(4, "FILL_PAGE: vp=0x%x, buf=0x%x, offset=0x%x\n",
              (int) vp, (int) buf, buf[0], buf[1], buf[2], (int) offset);
  fist_dprint(1, "FILL_PAGE: vp=0x%x, buf=0x%x [%d,%d,%d], offset=0x%x\n",
              (int) vp, (int) buf, buf[0], buf[1], buf[2], (int) offset);

  pp = vm_page_grab(vp->v_object, OFF_TO_IDX(offset),
                    VM_ALLOC_NORMAL | VM_ALLOC_RETRY);
  if (!pp) {
    printf("vm_page_grab returned NULL for offset 0x%x!\n", (int) offset);
    return;
  }
  kva = vm_pager_map_page(pp);
  ca = (caddr_t) kva;
  bcopy(buf, ca, PAGE_SIZE);
  vm_pager_unmap_page(kva);
  vm_page_set_validclean(pp, 0, PAGE_SIZE);

  if (pp->wire_count == 0 && pp->hold_count == 0 &&
      pp->busy == 0  && pp->flags & PG_BUSY ) {
      vm_page_free(pp);
  }

  /*
  if (pp->flags & PG_WANTED) {
    vm_page_activate(pp);
  } else {
    vm_page_deactivate(pp);
    }*/

  vm_page_wakeup(pp);

  print_location();
}

static int
wrapfs_read(ap)
     struct vop_read_args /* {
                             struct vnode *a_vp;
                             struct uio *a_uio;
                             int  a_ioflag;
                             struct ucred *a_cred;
                             } */ *ap;
{
  /* easy mappings */
  vnode_t *vp = ap->a_vp;
  uio_t *uiop = ap->a_uio;
  int ioflag = ap->a_ioflag;
  cred_t *cr = ap->a_cred;

  int error = EPERM;
  vnode_t *hidden_vp;
  uio_t temp_uio;
  iovec_t *temp_iovec;
  caddr_t current_base;
  int i, bytes_read;
  int num_pages, resid;
  long long start_loffset, end_loffset;
  long long cleartext_start_loffset, cleartext_end_loffset, current_loffset;

  fist_dprint(4, "fist_wrapfs_read vp %x\n", (int) vp);

#ifdef FIST_DEBUG
  fist_print_uios("fist_wrapfs_read", uiop);
#endif /* FIST_DEBUG */

  cleartext_start_loffset = uiop->uio_offset;
  cleartext_end_loffset = uiop->uio_offset + uiop->uio_resid;
  start_loffset = cleartext_start_loffset & ~(PAGE_SIZE - 1);
  end_loffset = cleartext_end_loffset & ~(PAGE_SIZE - 1);
  /* if not multiple of PAGE_SIZE, then the above formula loses one page.
   * adjust for it */
  if (cleartext_end_loffset > end_loffset)
    end_loffset += PAGE_SIZE;
  resid = end_loffset - start_loffset;
  num_pages = resid >> PAGE_SHIFT;

  fist_dprint(6,
              "READ: so=%d eo=%d cs=%d es=%d res=%d np=%d ps=%d\n",
              (int) start_loffset,
              (int) end_loffset,
              (int) cleartext_start_loffset,
              (int) cleartext_end_loffset,
              resid,
              num_pages,
              PAGE_SIZE);

  temp_iovec = kmem_zalloc(num_pages * sizeof(iovec_t));
  for (i = 0; i < num_pages; i++) {
    temp_iovec[i].iov_len = PAGE_SIZE;
    temp_iovec[i].iov_base = kmem_zalloc(PAGE_SIZE);
    fist_dprint(6, "READ allocated %d address 0x%x\n",
		i, temp_iovec[i].iov_base);
  }

  temp_uio.uio_iov = temp_iovec;
  temp_uio.uio_iovcnt = num_pages;
  temp_uio.uio_offset = start_loffset;
  temp_uio.uio_segflg = UIO_SYSSPACE;
  temp_uio.uio_rw = uiop->uio_rw;
  temp_uio.uio_procp = uiop->uio_procp;
  temp_uio.uio_resid = resid;

  hidden_vp = WRAPFS_VP_TO_LOWERVP(vp);

  /*
   * pass operation to hidden filesystem, and return status
   */

  error = VOP_READ(hidden_vp, &temp_uio, ioflag, cr);

  if (error) {
    fist_dprint(4, "VOP_READ in read returned error - not good\n");
    /* XXX to be checked */
    goto out_free;
  }


  current_loffset = start_loffset;
  for (i = 0; i < num_pages; i++) {

    bytes_read = PAGE_SIZE - temp_iovec[i].iov_len;

    if (bytes_read == 0)
      break;

    temp_iovec[i].iov_base -= bytes_read;
    current_base = temp_iovec[i].iov_base;

    /* decode the page/block */
    wrapfs_decode_block( current_base, current_base, bytes_read, vp,
			 vp->v_mount, OFF_TO_IDX(current_loffset));

    /*
     * save the original size, for kmem_free.
     * no need for it w/ wrapfs; size is always PAGE_SIZE, hence this line
     * is commented out:
     *		temp_iovec[i].iov_len = uiop->uio_iov[i].iov_len;
     */
    /* treat first and last iovec separately, not all data in them is needed */
    if (current_loffset + bytes_read > cleartext_end_loffset) {
      bytes_read = cleartext_end_loffset - current_loffset;
    }
    if (i == 0) {
      bytes_read -= cleartext_start_loffset - start_loffset;
      current_loffset += cleartext_start_loffset - start_loffset;
      current_base += cleartext_start_loffset - start_loffset;
    }
    if ((error = fist_uiomove(current_base, bytes_read, UIO_READ, uiop)))
      /*
       * XXX: we have to see the exact semantics of returning with an
       * EFAULT from read
       */
      break;
    current_loffset += bytes_read;
  }

out_free:
  for (i = 0; i < num_pages; i++) {
    fist_dprint(6, "READ freeing %d address 0x%x\n",
		i, temp_iovec[i].iov_base);
    kmem_free(temp_iovec[i].iov_base);
  }
  kmem_free(temp_iovec);

#ifdef FIST_DEBUG
  fist_print_uios("fist_wrapfs_read (END)", uiop);
#endif /* FIST_DEBUG */

  print_location();
  return (error);
}

static int
wrapfs_write(ap)
     struct vop_write_args /* {
			     struct vnode *a_vp;
			     struct uio *a_uio;
			     int  a_ioflag;
			     struct ucred *a_cred;
			     } */ *ap;
{
  /* easy mappings */
  vnode_t *vp = ap->a_vp;
  uio_t *uiop = ap->a_uio;
  int ioflag = ap->a_ioflag;
  cred_t *cr = ap->a_cred;
  struct proc *p = curproc;	/* XXX */

  int error = EPERM;
  vnode_t *hidden_vp;
  vattr_t va;
  uio_t temp_uio;
  iovec_t *temp_iovec;
  iovec_t *free_iovec;		/* for freeing allocated memory */
  int i;
  caddr_t current_base;
  int resid, bytes_read, num_pages, first_page_bytes, real_first_page;
  long long start_loffset, end_loffset, real_start_loffset;
  long long cleartext_start_loffset, cleartext_end_loffset, current_loffset;
  long long page_loffset; /* maintain offset to be passed to wrapfs_fill_page */
  int hidden_ioflag = (ioflag & ~IO_APPEND);

  fist_dprint(4, "fist_wrapfs_write vp=0x%x ioflag=0x%x offset=0x%x resid=%d iovcnt=%x\n",
	      (int) vp, ioflag, (int) uiop->uio_offset, uiop->uio_resid, uiop->uio_iovcnt);
#ifdef FIST_DEBUG
  fist_print_uios("fist_wrapfs_write (START)", uiop);
#endif /* FIST_DEBUG */

  hidden_vp = WRAPFS_VP_TO_LOWERVP(vp);

  /* we don't want anybody to do updates while we write, so lock the vnode */
#ifdef DO_WLOCK
  VREF(hidden_vp);		/* XXX: is this needed? */
  vn_lock(hidden_vp, LK_EXCLUSIVE | LK_RETRY | LK_CANRECURSE, p);
#endif

  /* get the attributes, length is necessary for correct updates */
  if ((error = VOP_GETATTR(hidden_vp, &va, cr, p))) {
    fist_dprint(4, "VOP_GETATTR returned error - not good\n");
    /* XXX to be checked */
    goto out;
  }

  /* just in case someone tries to pull a fast one */
  if (uiop->uio_resid == 0) {
    error = 0;
    goto out;
  }

  cleartext_start_loffset = uiop->uio_offset;
  cleartext_end_loffset = uiop->uio_offset + uiop->uio_resid;

  if (ioflag & IO_APPEND) {
    fist_dprint(6, "WRITE: turning off append flag\n");
    cleartext_start_loffset += va.va_size;
    cleartext_end_loffset += va.va_size;
  }

  start_loffset = MIN(cleartext_start_loffset, va.va_size) & ~(PAGE_SIZE - 1);
  real_start_loffset = cleartext_start_loffset & ~(PAGE_SIZE - 1);
  first_page_bytes = MIN(cleartext_start_loffset, va.va_size) - start_loffset;
  /* must use this to avoid shifting a quad w/ gcc */
  real_first_page = (int)(real_start_loffset - start_loffset) >> PAGE_SHIFT;
  end_loffset = cleartext_end_loffset & ~(PAGE_SIZE - 1);
  ASSERT(first_page_bytes <= PAGE_SIZE);
  /*
   * if not multiple of PAGE_SIZE, then the above formula loses one page.
   * adjust for it
   */
  if (cleartext_end_loffset > end_loffset)
    end_loffset += PAGE_SIZE;
  resid = end_loffset - start_loffset;
  num_pages = resid >> PAGE_SHIFT;

  if (num_pages == 1)
    first_page_bytes = PAGE_SIZE;

  temp_iovec = kmem_zalloc(num_pages * sizeof(iovec_t));
  free_iovec = kmem_zalloc(num_pages * sizeof(iovec_t));
  for (i = 0; i < num_pages; i++) {
    temp_iovec[i].iov_len = free_iovec[i].iov_len = PAGE_SIZE;
    /* we need the pages to be zeroed out */
    temp_iovec[i].iov_base = free_iovec[i].iov_base = kmem_zalloc(PAGE_SIZE);
  }

  fist_dprint(6,
	      "WRITE: so=%d eo=%d cso=%d ceo=%d rso=%d res=%d np=%d rfp=%d\n",
	      (int) start_loffset,
	      (int) end_loffset,
	      (int) cleartext_start_loffset,
	      (int) cleartext_end_loffset,
	      (int) real_start_loffset,
	      resid,
	      num_pages,
	      real_first_page
	      );

  current_loffset = start_loffset;
  page_loffset = start_loffset;

  /* read first block XXX check length of file */
  temp_uio.uio_iov = temp_iovec;
  temp_uio.uio_iovcnt = 1;
  temp_uio.uio_offset = start_loffset;
  temp_uio.uio_segflg = UIO_SYSSPACE;
  temp_uio.uio_rw = UIO_READ;
  temp_uio.uio_procp = uiop->uio_procp;
  temp_uio.uio_resid = first_page_bytes;
  fist_print_uios("WRITE (before VOP_READ 1)", &temp_uio);
  error = VOP_READ(hidden_vp, &temp_uio, hidden_ioflag, cr);
  if (error) {
    fist_dprint(5, "VOP_READ returned error - not good\n");
    /* XXX to be checked */
    goto out_free;
  }

  fist_print_uios("WRITE (after VOP_READ 1)", &temp_uio);
  bytes_read = PAGE_SIZE - temp_iovec[0].iov_len;
  temp_iovec[0].iov_base -= bytes_read;
  temp_iovec[0].iov_len = PAGE_SIZE;

  /* decode block read */
  wrapfs_decode_block(temp_iovec[0].iov_base, temp_iovec[0].iov_base, bytes_read,
		      vp, vp->v_mount, OFF_TO_IDX(start_loffset));

  /*
   * if num_pages == 1, we already read the page... don't clobber it
   * if num_pages > 1, then we must read the last page, and decode it
   * completely, before clobbering it.
   * XXX: if end offset is on page boundary, we don't have to do this.
   */
  if ( (num_pages > 1) && ( va.va_size > (end_loffset - PAGE_SIZE) ) ) {
    /* read last block XXX check length of file */
    temp_uio.uio_iov = temp_iovec + (num_pages - 1);
    temp_uio.uio_iovcnt = 1;
    temp_uio.uio_offset = end_loffset - PAGE_SIZE;
    temp_uio.uio_segflg = UIO_SYSSPACE;
    temp_uio.uio_rw = UIO_READ;
    temp_uio.uio_procp = uiop->uio_procp;
    temp_uio.uio_resid = PAGE_SIZE;

    fist_print_uios("WRITE (before VOP_READ 2)", &temp_uio);
    error = VOP_READ(hidden_vp, &temp_uio, hidden_ioflag, cr);
    fist_print_uios("WRITE (after VOP_READ 3)", &temp_uio);
    if (error) {
      fist_dprint(4, "VOP_READ returned error - not good\n");
      /* XXX to be checked */
      goto out_free;
    }
    bytes_read = PAGE_SIZE - temp_iovec[num_pages - 1].iov_len;
    temp_iovec[num_pages - 1].iov_base -= bytes_read;
    temp_iovec[num_pages - 1].iov_len = PAGE_SIZE;
    /* decode block read */
    wrapfs_decode_block(temp_iovec[num_pages-1].iov_base,
			temp_iovec[num_pages-1].iov_base, bytes_read,
			vp, vp->v_mount, OFF_TO_IDX(end_loffset - PAGE_SIZE));
  }

  /*
   * update the size of the vm object associated with both
   * the upper and lower vnode
   */
  if (cleartext_end_loffset > va.va_size) {
    vnode_pager_setsize(vp, cleartext_end_loffset );
    vnode_pager_setsize(hidden_vp, cleartext_end_loffset );
  }

  /*
   * Now we are ready to write the bytes within the start/end
   * cleartext offsets in the buffers we allocated.
   */
  for (i = 0; i < num_pages; i++) {
    if (i >= real_first_page) {
      bytes_read = PAGE_SIZE;
      current_base = temp_iovec[i].iov_base;
      if (i == real_first_page) {
#define real_first_page_offset (cleartext_start_loffset - real_start_loffset)
	bytes_read -= real_first_page_offset;
	current_loffset += real_first_page_offset;
	current_base += real_first_page_offset;
#undef real_first_page_offset
      }
      if (current_loffset + bytes_read > cleartext_end_loffset) {
	bytes_read = cleartext_end_loffset - current_loffset;
      }
      if ((error = fist_uiomove(current_base, bytes_read, UIO_WRITE, uiop)))
	break;
    }
    wrapfs_fill_page(vp, temp_iovec[i].iov_base, page_loffset);
    wrapfs_encode_block(temp_iovec[i].iov_base, temp_iovec[i].iov_base,
			PAGE_SIZE, vp, vp->v_mount, OFF_TO_IDX(page_loffset));
    wrapfs_fill_page(hidden_vp, temp_iovec[i].iov_base, page_loffset);
    current_loffset += bytes_read;
    page_loffset += PAGE_SIZE;
  }
  fist_print_uios("WRITE (after for loop 4)", &temp_uio);

  if (va.va_size < end_loffset) {
    if (va.va_size < cleartext_end_loffset)
      resid -= end_loffset - cleartext_end_loffset;
    else
      resid -= end_loffset - va.va_size;
  }

  /* XXX: no need for full initialization here */
  temp_uio.uio_iov = temp_iovec;
  temp_uio.uio_iovcnt = num_pages;
  temp_uio.uio_offset = start_loffset;
  temp_uio.uio_segflg = UIO_SYSSPACE;
  temp_uio.uio_rw = uiop->uio_rw;
  temp_uio.uio_procp = uiop->uio_procp;
  temp_uio.uio_resid = resid;

  /*
   * pass operation to hidden filesystem, and return status
   */
  fist_print_uios("WRITE (before write)", &temp_uio);
  error = VOP_WRITE(hidden_vp, &temp_uio, hidden_ioflag, cr);
  fist_print_uios("WRITE (after write)", &temp_uio);

  if (temp_uio.uio_offset < cleartext_end_loffset) {
    /* incomplete write: this case is an error and should not happen */
    uiop->uio_offset = temp_uio.uio_offset;
    uiop->uio_resid = cleartext_end_loffset - temp_uio.uio_offset;
  } else {
    /*
     * we may have written more than what was asked of us to preserve the
     * encoding over a whole page
     */
    uiop->uio_offset = cleartext_end_loffset;
    uiop->uio_resid = 0;
  }
  /* if IO_APPEND was used, return offset of 0 to upper level */
  if (ioflag & IO_APPEND) {
    uiop->uio_offset = 0;
  }

out_free:
  for (i = 0; i < num_pages; i++) {
    fist_dprint(6, "PRINT_BASE1 %d: 0x%x (len=%d)\n", i,
		temp_iovec[i].iov_base,
		temp_iovec[i].iov_len);
    fist_dprint(6, "PRINT_BASE2 %d: 0x%x (len=%d)\n", i,
		free_iovec[i].iov_base,
		free_iovec[i].iov_len);
    kmem_free(free_iovec[i].iov_base);
  }
  kmem_free(free_iovec);
  kmem_free(temp_iovec);

out:

#ifdef FIST_DEBUG
  fist_print_uios("fist_wrapfs_write (END)", uiop);
#endif /* FIST_DEBUG */

  print_location();
  return (error);
}

#endif /* FIST_FILTER_DATA */

static int
wrapfs_ioctl(ap)
     struct vop_ioctl_args /* {
			      struct vnode *a_vp;
			      u_long  a_command;
			      caddr_t  a_data;
			      int  a_fflag;
			      struct ucred *a_cred;
			      struct proc *a_p;
			      } */ *ap;
{
  int error = 0;
  struct vnode *vp = ap->a_vp;
  vnode_t *this_vnode = ap->a_vp;
  struct vnode *lowervp = WRAPFS_VP_TO_LOWERVP(vp);
  int val = 0;
  caddr_t arg = ap->a_data;
  cred_t *cr = curproc->p_cred->pc_ucred;
  vfs_t *this_vfs = this_vnode->v_mount;

  fist_dprint(4, "WRAPFS_IOCTL: vp=0x%x, lowervp=0x%x\n", (int) vp, (int) lowervp);

  switch (ap->a_command) {
#ifdef FIST_DEBUG
  case FIST_IOCTL_GET_DEBUG_VALUE:
    val = fist_get_debug_value();
    *ap->a_data = val;
    error = 0;
    goto out;
    break;

  case FIST_IOCTL_SET_DEBUG_VALUE:
    val = (int) (*ap->a_data);
    if (val < 0 || val > 20) {
      error = EINVAL;
      goto out;
    }
    fist_dprint(6, "IOCTL: got arg %d\n", val);
    fist_set_debug_value(val);
    error = 0;
    goto out;
    break;
#endif /* FIST_DEBUG */

    /* add non-debugging fist ioctl's here */
FIST_IOCTL_ECLS

  default:

    /* default to repeating ioctl on lowervp */
    error = VOP_IOCTL(lowervp,
		      ap->a_command,
		      ap->a_data,
		      ap->a_fflag,
		      ap->a_cred,
		      ap->a_p);

  } /* end of switch statement */

out:
  print_location();
  return (error);
}

#ifdef FIST_FILTER_DATA
/****************************************************************************/
/*
 * get page routine
 */

static int
wrapfs_getpages(ap)
     struct vop_getpages_args /* {
				 struct vnode *a_vp;
				 vm_page_t *a_m;
				 int a_count;
				 int a_reqpage;
				 vm_ooffset_t a_offset;
			      } */ *ap;
{
  int error = 0;
  struct vnode *vp = ap->a_vp;
  struct vnode *lowervp = WRAPFS_VP_TO_LOWERVP(vp);
  int reqpage = ap->a_reqpage;
  int bytecount = ap->a_count;
  vm_page_t *lowerm;
  vm_page_t mreq = ap->a_m[reqpage];
  int i, resid;
  int pagecount = round_page(bytecount) / PAGE_SIZE;
  vm_offset_t thiskva, lowerkva;
  caddr_t thisca, lowerca;
  vm_object_t obj = vp->v_object;
  vm_object_t lower_obj;

  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);
  fist_print_object(__FUNCTION__, obj);

  if (vp->v_mount == NULL)
    return VM_PAGER_BAD;

  if (mreq->valid) {
    for (i = 0; i < pagecount; i++) {
      if (i != reqpage) {
	vnode_pager_freepage(ap->a_m[i]);
      } else if (ap->a_m[i]->valid != VM_PAGE_BITS_ALL) {
	vm_page_zero_invalid(mreq,TRUE);
      }
    }
    return VM_PAGER_OK;
  }

  error = VOP_GETVOBJECT(lowervp, &lower_obj);
  if (error) {
    printk(" unable to get lower vm_object \n");
    return error;
  }


  vm_object_reference(lower_obj);
  vm_object_pip_add(lower_obj, obj->paging_in_progress);

  /* get the pages for the lower file system */
  lowerm = kmem_zalloc(pagecount * sizeof(vm_page_t));
  for ( i = 0; i< pagecount ; i++) {
    lowerm[i] = vm_page_grab(lower_obj, ap->a_m[i]->pindex,
			     VM_ALLOC_NORMAL | VM_ALLOC_RETRY);

    if (lowerm[i]->valid == 0) {
      /* call VOP_GETPAGES on the lower page - offset is ignored */
      error = VOP_GETPAGES(lowervp, &lowerm[i], 1, 0, ap->a_offset );
      if (error != VM_PAGER_OK) {
	printk(" wrapfs_getpages failed \n");
	continue;
      }
    }

    lowerkva = vm_pager_map_page(lowerm[i]);
    lowerca = (caddr_t) lowerkva;
    thiskva = vm_pager_map_page(ap->a_m[i]);
    thisca = (caddr_t) thiskva;
    wrapfs_decode_block(lowerca, thisca, PAGE_SIZE, vp,
			vp->v_mount, ap->a_m[i]->pindex);
    vm_pager_unmap_page(lowerkva);
    vm_pager_unmap_page(thiskva);
    ap->a_m[i]->valid = lowerm[i]->valid;

    if (lowerm[i]->wire_count == 0 && lowerm[i]->hold_count == 0 &&
	lowerm[i]->busy == 0 && lowerm[i]->flags & PG_BUSY) {
      vm_page_free(lowerm[i]);
    }

    vm_page_wakeup(lowerm[i]);
    if ( i == reqpage) {
      vm_page_busy(ap->a_m[i]);
    } else {
      if (ap->a_m[i]->flags & PG_BUSY)
	vnode_pager_freepage(ap->a_m[i]);
    }
    fist_print_page("wrapfs_getpages: thispp0", ap->a_m[i]);
  }

  vm_object_pip_subtract(lowervp->v_object, lowervp->v_object->paging_in_progress);
  vm_object_deallocate(lower_obj);
  error = VM_PAGER_OK;
out:
  kmem_free(lowerm);
  fist_print_object(__FUNCTION__, vp->v_object);
  print_location();
  return error;
}

static int
wrapfs_putpages(ap)
	 struct vop_putpages_args /* {
				 struct vnode *a_vp;
				 vm_page_t *a_m;
				 int a_count;
				 int a_sync;
				 int *a_rtvals;
				 vm_ooffset_t a_offset;
				 } */ *ap;
{
  int error = 0;
  struct vnode *vp = ap->a_vp;
  struct vnode *lowervp = WRAPFS_VP_TO_LOWERVP(vp);
  int bytecount = ap->a_count;
  vm_page_t *lowerm;
  int i, resid;
  int pagecount = round_page(bytecount) / PAGE_SIZE;
  vm_offset_t thiskva, lowerkva;
  caddr_t thisca, lowerca;
  vm_object_t obj = vp->v_object;
  vm_object_t lower_obj;

  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);
  fist_print_object(__FUNCTION__, obj);

  if (!lowervp) {
    printk("no lowervp \n");
    return VM_PAGER_BAD;
  }

  error = VOP_GETVOBJECT(lowervp, &lower_obj);
  if (error) {
    printk(" unable to get lower vm_object \n");
    return error;
  }

  fist_copy_object_attr(lower_obj, obj);
  vm_object_pip_add(lowervp->v_object, vp->v_object->paging_in_progress);

  lowerm = kmem_zalloc(pagecount * sizeof(vm_page_t));

  for ( i = 0; i< pagecount ; i++)
     lowerm[i] = vm_page_grab(lower_obj, ap->a_m[i]->pindex,
			      VM_ALLOC_NORMAL | VM_ALLOC_RETRY);

  for (i = 0; i < pagecount ; ++i) {

    /* copy the flags to the lower level vm_page_t */
    fist_copy_page_attr(lowerm[i], ap->a_m[i]);

    lowerkva = vm_pager_map_page(lowerm[i]);
    lowerca = (caddr_t) lowerkva;

    thiskva = vm_pager_map_page(ap->a_m[i]);
    thisca = (caddr_t) thiskva;

    wrapfs_encode_block( thisca, lowerca, PAGE_SIZE, vp,
			 vp->v_mount, ap->a_m[i]->pindex);
    vm_pager_unmap_page(lowerkva);
    vm_pager_unmap_page(thiskva);
  }

  error = VOP_PUTPAGES(lowervp, lowerm, bytecount, ap->a_sync, ap->a_rtvals, ap->a_offset);
  if (error)
    printf("wrapfs_putpages : error %d ", error);

  fist_copy_object_attr(vp->v_object, lowervp->v_object);
  vm_object_pip_subtract(lowervp->v_object, lowervp->v_object->paging_in_progress);
  /* set status of pages to valid and non-busy as needed */
  for (i = 0; i < pagecount; i++) {
    fist_copy_page_attr(ap->a_m[i], lowerm[i]);
    if (lowerm[i]->wire_count == 0 && lowerm[i]->hold_count == 0 &&
	lowerm[i]->busy == 0 && lowerm[i]->flags & PG_BUSY ) {
      vm_page_free(lowerm[i]);
    }
    vm_page_wakeup(lowerm[i]);
    vm_page_flag_clear(ap->a_m[i], PG_BUSY);
    fist_print_page("wrapfs_putpages: thispp0", ap->a_m[i]);
  }

  vp->v_flag &= ~VOBJDIRTY;
out:
  kmem_free(lowerm);
  fist_print_object(__FUNCTION__, vp->v_object);
  print_location();
  return error;
}

#endif /* FIST_FILTER_DATA */


#ifdef HAVE_BMAP
static int
wrapfs_bmap(ap)
     struct vop_bmap_args /* {
			     struct vnode *a_vp;
			     daddr_t a_bn;
			     struct vnode **a_vpp;
			     daddr_t *a_bnp;
			     int *a_runp;
			     int *a_runb;
			     } */ *ap;
{
  int error = 0;
  struct vnode *vp = ap->a_vp;
  struct vnode *lowervp = WRAPFS_VP_TO_LOWERVP(vp);
  printk(" in bmap \n");

  fist_dprint(4, "WRAPFS_BMAP:0x%x:0x%x\n", (int) ap->a_vpp, (int) lowervp);
  printf("WRAPFS_BMAP:0x%x:0x%x\n", (int) ap->a_vpp, (int) lowervp);


  if (error)
    return error;

#if 0
  ap->a_vp = lowervp;
  error = VCALL(lowervp, VOFFSET(vop_bmap), ap);
  return (error);
#endif

#if 1
  error = wrapfs_bypass((struct vop_generic_args *) ap);
  return (error);
#endif

#if 0
  /* this code also doesn't panic */
  if (ap->a_vpp != NULL)
    *ap->a_vpp = ap->a_vp;
  if (ap->a_bnp != NULL)
    *ap->a_bnp = ap->a_bn;
  if (ap->a_runp != NULL)
    *ap->a_runp = 0;
  if (ap->a_runb != NULL)
    *ap->a_runb = 0;
  return (error);
#endif

#if 0
  error = EINVAL;
  return (error);
#endif
}
#endif /* HAVE_BMAP */

static int
wrapfs_mmap(ap)
     struct vop_mmap_args /* {
			     struct vnode *a_vp;
			     int  a_fflags;
			     struct ucred *a_cred;
			     struct proc *a_p;
			     } */ *ap;
{
  return EINVAL;
}


#ifdef FIST_FILTER_NAME

static int
wrapfs_mkdir(ap)
     struct vop_mkdir_args /* {
			      struct vnode *a_dvp;
			      struct vnode **a_vpp;
			      struct componentname *a_cnp;
			      struct vattr *a_vap;
			      } */ *ap;
{
  int error;
  vnode_t *thisdvp = ap->a_dvp;
  vnode_t *lowerdvp = WRAPFS_VP_TO_LOWERVP(thisdvp);
  CNP_VARS;

  fist_dprint(2, "WRAPFS_MKDIR: lowerdvp=0x%x\n", (int) lowerdvp);

  CNP_BEFORE(thisdvp);
  error = wrapfs_bypass((struct vop_generic_args *) ap);
  CNP_AFTER(thisdvp);

  print_location();
  return (error);
}

static int
wrapfs_rmdir(ap)
     struct vop_rmdir_args /* {
			      struct vnode *a_dvp;
			      struct vnode *a_vp;
			      struct componentname *a_cnp;
			      } */ *ap;
{
  int error;
  vnode_t *thisdvp = ap->a_dvp;
  vnode_t *thisvp = ap->a_vp;
  vnode_t *lowerdvp = WRAPFS_VP_TO_LOWERVP(thisdvp);
  vnode_t *lowervp = WRAPFS_VP_TO_LOWERVP(thisvp);
  CNP_VARS;

  fist_dprint(2, "WRAPFS_RMDIR: lowerdvp=0x%x lowervp=0x%x\n",
	      (int) lowerdvp, (int) lowervp);

  CNP_BEFORE(thisvp);
  error = wrapfs_bypass((struct vop_generic_args *) ap);
  CNP_AFTER(thisvp);

  print_location();
  return (error);
}

static int
wrapfs_create(ap)
     struct vop_create_args /* {
			       struct vnode *a_dvp;
			       struct vnode **a_vpp;
			       struct componentname *a_cnp;
			       struct vattr *a_vap;
			       } */ *ap;
{
  int error;
  vnode_t *thisdvp = ap->a_dvp;
  vnode_t *lowerdvp = WRAPFS_VP_TO_LOWERVP(thisdvp);
  CNP_VARS;

  fist_dprint(2, "WRAPFS_CREATE: lowerdvp=0x%x\n", (int) lowerdvp);

  CNP_BEFORE(thisdvp);
  error = wrapfs_bypass((struct vop_generic_args *) ap);
  CNP_AFTER(thisdvp);

  print_location();
  return (error);
}

static int
wrapfs_remove(ap)
     struct vop_remove_args /* {
			       struct vnode *a_dvp;
			       struct vnode *a_vp;
			       struct componentname *a_cnp;
			       } */ *ap;
{
  int error;
  vnode_t *thisdvp = ap->a_dvp;
  vnode_t *lowerdvp = WRAPFS_VP_TO_LOWERVP(ap->a_dvp);
  vnode_t *lowervp = WRAPFS_VP_TO_LOWERVP(ap->a_vp);
  CNP_VARS;

  fist_dprint(2, "WRAPFS_REMOVE: lowerdvp=0x%x lowervp=0x%x\n",
	      (int) lowerdvp, (int) lowervp);

  CNP_BEFORE(thisdvp);
  error = wrapfs_bypass((struct vop_generic_args *) ap);
  CNP_AFTER(thisdvp);

  return (error);
}

static int
wrapfs_rename(ap)
     struct vop_rename_args  /* {
				struct vnode *a_fdvp;
				struct vnode *a_fvp;
				struct componentname *a_fcnp;
				struct vnode *a_tdvp;
				struct vnode *a_tvp;
				struct componentname *a_tcnp;
				} */ *ap;
{
  int error;
  vnode_t *thisfdvp = ap->a_fdvp;
  vnode_t *thisfvp = ap->a_fvp;
  vnode_t *thistdvp = ap->a_tdvp;
  vnode_t *thistvp = ap->a_tvp;	/* this one comes in as NULL! */
  vnode_t *lowerfdvp = WRAPFS_VP_TO_LOWERVP(thisfdvp);
  vnode_t *lowerfvp = WRAPFS_VP_TO_LOWERVP(thisfvp);
  vnode_t *lowertdvp = WRAPFS_VP_TO_LOWERVP(thistdvp);
  vnode_t *lowertvp = NULL;
  /* CNP_VARS; */
  cn_t *thisfcnp, *lowerfcnp, *thistcnp, *lowertcnp;

  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);
  /* Check for cross-device rename. */
  if ((thisfvp->v_mount != thistdvp->v_mount) ||
      (thistvp && (thisfvp->v_mount != thistvp->v_mount))) {
    if (thistdvp == thistvp)
      vrele(thistdvp);
    else
      vput(thistdvp);
    if (thistvp)
      vput(thistvp);
    vrele(thisfdvp);
    vrele(thisfvp);
    return (EXDEV);
  }

  if (thistvp != NULL)		/* comes in as null to this function */
    lowertvp = WRAPFS_VP_TO_LOWERVP(thistvp);

  fist_dprint(2, "WRAPFS_RENAME: thisfdvp=0x%x thisfvp=0x%x thistdvp=0x%x thistvp=0x%x\n",
	      (int) thisfdvp, (int) thisfvp,
	      (int) thistdvp, (int) thistvp);
  fist_dprint(2, "WRAPFS_RENAME: lowerfdvp=0x%x lowerfvp=0x%x lowertdvp=0x%x lowertvp=0x%x\n",
	      (int) lowerfdvp, (int) lowerfvp,
	      (int) lowertdvp, (int) lowertvp);

  /* CNP_BEFORE(thisvp); */
  thisfcnp = ap->a_fcnp;	/* save original */
  lowerfcnp = wrapfs_new_cnp((thisfdvp), thisfcnp);
  ap->a_fcnp = lowerfcnp;	/* prepare for bypass */
  thistcnp = ap->a_tcnp;	/* save original */
  lowertcnp = wrapfs_new_cnp((thistdvp), thistcnp);
  ap->a_tcnp = lowertcnp;	/* prepare for bypass */

  error = wrapfs_bypass((struct vop_generic_args *) ap);

  /* CNP_AFTER(thisvp); */
  if ((thisfcnp->cn_flags & PARAMASK) != (lowerfcnp->cn_flags & PARAMASK))
    panic("%s: FLAGS CHANGED fthis:0x%x flower:0x%x",
	  __FUNCTION__, (int)thisfcnp->cn_flags, (int)lowerfcnp->cn_flags);
  if ((thistcnp->cn_flags & PARAMASK) != (lowertcnp->cn_flags & PARAMASK))
    panic("%s: FLAGS CHANGED tthis:0x%x tlower:0x%x",
	  __FUNCTION__, (int)thistcnp->cn_flags, (int)lowertcnp->cn_flags);
  wrapfs_update_cnp((thisfdvp), &lowerfcnp, thisfcnp, error);
  ap->a_fcnp = thisfcnp;	/* update for caller */
  wrapfs_update_cnp((thistdvp), &lowertcnp, thistcnp, error);
  ap->a_tcnp = thistcnp;	/* update for caller */

  print_location();
  return (error);
}

static int
wrapfs_link(ap)
     struct vop_link_args /* {
			     struct vnode *a_tdvp;
			     struct vnode *a_vp;
			     struct componentname *a_cnp;
			     } */ *ap;
{
  int error;
  vnode_t *thistdvp = ap->a_tdvp;
  vnode_t *thisvp = ap->a_vp;
  vnode_t *lowertdvp = NULL;
  vnode_t *lowervp = NULL;
  CNP_VARS;

  /* MUST make sure we only hard link into our own file system! */
  if (thisvp->v_op != wrapfs_vnodeop_p)
    return EXDEV;

  lowertdvp = WRAPFS_VP_TO_LOWERVP(thistdvp);
  lowervp = WRAPFS_VP_TO_LOWERVP(thisvp);
  fist_dprint(2, "WRAPFS_LINK: lowertdvp=0x%x lowervp=0x%x\n",
	      (int) lowertdvp, (int) lowervp);

  CNP_BEFORE(thistdvp);
  error = wrapfs_bypass((struct vop_generic_args *) ap);
  CNP_AFTER(thistdvp);

  print_location();
  return (error);
}

static int
wrapfs_symlink(ap)
     struct vop_symlink_args /* {
				struct vnode *a_dvp;
				struct vnode **a_vpp;
				struct componentname *a_cnp;
				struct vattr *a_vap;
				char *a_target;
				} */ *ap;
{
  int error;
  vnode_t *thisdvp = ap->a_dvp;
  vnode_t *lowerdvp = WRAPFS_VP_TO_LOWERVP(thisdvp);
  char *lower_target, *this_target;
  int lower_target_len = 0;
  CNP_VARS;

  fist_dprint(2, "WRAPFS_SYMLINK: lowerdvp=0x%x target=\"%s\"\n",
	      (int) lowerdvp, ap->a_target);

  lower_target_len = wrapfs_encode_filename(ap->a_target,
					    strlen(ap->a_target),
					    &lower_target,
					    DO_DOTS,
					    thisdvp,
					    thisdvp->v_mount);

  if (lower_target_len < 0) {
    /* error values in FreeBSD should be positive*/
    error = -lower_target_len;
    goto out;
  }
  this_target = ap->a_target;	/* save pointer */
  ap->a_target = lower_target;	/* set to encoded name */
  CNP_BEFORE(thisdvp);
  error = wrapfs_bypass((struct vop_generic_args *) ap);
  CNP_AFTER(thisdvp);
  ap->a_target = this_target;	/* restore pointer */
  kmem_free(lower_target);

  print_location();

out:
  return (error);
}

static int
wrapfs_readlink(ap)
     struct vop_readlink_args /* {
				 struct vnode *a_vp;
				 struct uio *a_uio;
				 struct ucred *a_cred;
				 } */ *ap;
{
  int error;
  vnode_t *thisvp = ap->a_vp;
  vnode_t *lowervp = WRAPFS_VP_TO_LOWERVP(thisvp);
  uio_t *uiop = ap->a_uio;
  cred_t *cr = ap->a_cred;
  iovec_t temp_iovec;
  uio_t temp_uio;
  caddr_t temp_addr2free;
  int bytes_read;
  int temp_length, target_real_length;
  char *temp_name;

  fist_dprint(2, "WRAPFS_READLINK: lowervp=0x%x\n",
	      (int) lowervp);

  temp_iovec.iov_len = PAGE_SIZE;
  temp_iovec.iov_base = temp_addr2free = kmem_zalloc(PAGE_SIZE);
  if (!temp_iovec.iov_base) {
    printf("no more memory in readlink\n");
    error = ENOMEM;
    goto out;
  }
  temp_uio.uio_iov = &temp_iovec;
  temp_uio.uio_iovcnt = 1;
  temp_uio.uio_offset = 0;
  temp_uio.uio_segflg = UIO_SYSSPACE;
  temp_uio.uio_rw = uiop->uio_rw;
  temp_uio.uio_procp = uiop->uio_procp;
  temp_uio.uio_resid = uiop->uio_resid;

  error = VOP_READLINK(lowervp, &temp_uio, cr);
  if (error)
    goto out_free;

  bytes_read = PAGE_SIZE - temp_iovec.iov_len;

  temp_length = wrapfs_decode_filename(temp_iovec.iov_base - bytes_read,
				       bytes_read + 1,
				       &temp_name,
				       DO_DOTS,
				       thisvp,
				       thisvp->v_mount);


  if (temp_length < 0) {
    /* a checksum error had occured: skip entry */
    printf("symlink value encoded with different key than link itself\n");
    /* error values in FreeBSD are positive*/
    error = -temp_length;
    goto out_free;
  }
  temp_name[temp_length - 1] = '\0';
  /* must find real string length, which is guaranteed null terminated here */
  target_real_length = strlen(temp_name) + 1;
  fist_dprint(4, "wrapfs_readlink DECODE len=%d, real_len=%d, bytes_read=%d, name=\"%s\"",
	      temp_length, target_real_length, bytes_read, temp_name);

  fist_uiomove(temp_name, target_real_length, UIO_READ, uiop);
  /* already OK: uiop->uio_resid and uiop->uio_loffset */

out_free:
  kmem_free(temp_name);
  kmem_free(temp_addr2free);

out:
  print_location();
  return (error);
}

#if 0
static int
wrapfs_cachedlookup(ap)
     struct vop_cachedlookup_args /* {
				     struct vnode *a_dvp;
				     struct vnode **a_vpp;
				     struct componentname *a_cnp;
				     } */ *ap;
{
  int error;
  vnode_t *thisdvp = ap->a_dvp;
  vnode_t *lowerdvp = WRAPFS_VP_TO_LOWERVP(thisdvp);
  CNP_VARS;

  fist_dprint(2, "WRAPFS_CACHEDLOOKUP: lowerdvp=0x%x\n",
	      (int) lowerdvp);
  CNP_BEFORE(thisdvp);
  error = wrapfs_bypass((struct vop_generic_args *) ap);
  CNP_AFTER(thisdvp);

  print_location();
  return (error);
}
#endif

static int
wrapfs_whiteout(ap)
     struct vop_whiteout_args /* {
				 struct vnode *a_dvp;
				 struct componentname *a_cnp;
				 int a_flags;
				 } */ *ap;
{
  int error;
  vnode_t *thisdvp = ap->a_dvp;
  vnode_t *lowerdvp = WRAPFS_VP_TO_LOWERVP(thisdvp);
  CNP_VARS;

  fist_dprint(2, "WRAPFS_WHITEOUT: lowerdvp=0x%x\n",
	      (int) lowerdvp);

  CNP_BEFORE(thisdvp);
  error = wrapfs_bypass((struct vop_generic_args *) ap);
  CNP_AFTER(thisdvp);

  print_location();
  return (error);
}

static int
wrapfs_mknod(ap)
     struct vop_mknod_args /* {
			      struct vnode *a_dvp;
			      struct vnode **a_vpp;
			      struct componentname *a_cnp;
			      struct vattr *a_vap;
			      } */ *ap;
{
  int error;
  vnode_t *thisdvp = ap->a_dvp;
  vnode_t *lowerdvp = WRAPFS_VP_TO_LOWERVP(thisdvp);
  CNP_VARS;

  fist_dprint(2, "WRAPFS_MKNOD: lowerdvp=0x%x\n",
	      (int) lowerdvp);

  CNP_BEFORE(thisdvp);
  error = wrapfs_bypass((struct vop_generic_args *) ap);
  CNP_AFTER(thisdvp);

  print_location();
  return (error);
}


static int
wrapfs_readdir(ap)
     struct vop_readdir_args /* {
				struct vnode *a_vp;
				struct uio *a_uio;
				struct ucred *a_cred;
				int *a_eofflag;
				int *a_ncookies;
				u_long **a_cookies;
				} */ *ap;
{
  int error = EPERM;
  vnode_t *thisvp = ap->a_vp;
  vnode_t *lowervp = WRAPFS_VP_TO_LOWERVP(thisvp);
  cred_t *cr = ap->a_cred;
  uio_t *uiop = ap->a_uio;
  uio_t temp_uio;
  iovec_t temp_iovec;
  int aux, bytes_read, length, temp_length, tmp, old_reclen;
  char *temp_name;

  fist_dprint(2, "WRAPFS_READDIR: lowervp=0x%x ncookies=0x%x\n",
	      (int) lowervp, (int) ap->a_ncookies);

  if (ap->a_ncookies)
    panic("WRAPFS_READDIR0: *ncookies=%d (fix the code)\n", *ap->a_ncookies);

  if (uiop->uio_iovcnt != 1)
    panic("WRAPFS_READDIR: iovecnt not 1 (=%d)", uiop->uio_iovcnt);

  temp_iovec.iov_len = uiop->uio_resid;
  temp_iovec.iov_base = kmem_zalloc(uiop->uio_resid);

  temp_uio.uio_iov = &temp_iovec;
  temp_uio.uio_iovcnt = 1;
  temp_uio.uio_offset = uiop->uio_offset;
  temp_uio.uio_segflg = UIO_SYSSPACE;
  temp_uio.uio_rw = uiop->uio_rw;
  temp_uio.uio_procp = uiop->uio_procp;
  temp_uio.uio_resid = uiop->uio_resid;

  /* pass operation to hidden filesystem, and return status */
  error = VOP_READDIR(lowervp, &temp_uio, cr,
		      ap->a_eofflag, ap->a_ncookies, ap->a_cookies);

  bytes_read = uiop->uio_resid - temp_uio.uio_resid;
  temp_iovec.iov_base -= bytes_read;
  temp_iovec.iov_len += bytes_read;

  if (error)
    goto clean_up;

#define crt_dirent ((struct dirent *)(temp_iovec.iov_base + aux))
  for (aux = 0; aux < bytes_read; aux += old_reclen) {
    char *name = temp_name;
    old_reclen = crt_dirent->d_reclen;
    fist_dprint(2, "\nRD: old_reclen=%d old_namlen=%d\n", old_reclen, crt_dirent->d_namlen);

    temp_length = wrapfs_decode_filename(crt_dirent->d_name,
					 crt_dirent->d_namlen + 1,
					 &temp_name, /* null terminated */
					 SKIP_DOTS,
					 thisvp,
					 thisvp->v_mount);
    if (temp_length >= 0) {

      temp_name[temp_length - 1] = '\0';

      FIST_OP_READDIR_CALL;

      /*
       * We copy the dirent to userspace only if the csum matched
       */
      strcpy(crt_dirent->d_name, temp_name); /* ok, b/c of terminating null */
      length = temp_length - 256 + sizeof(struct dirent);
      fist_dprint(2, "RD: new length is %d, temp_length %d, struct dirent: %d\n",
		  length, temp_length, sizeof(struct dirent));
      if ((tmp = length & 3))
	length += 4 - tmp;
      crt_dirent->d_reclen = length;
      crt_dirent->d_namlen = temp_length - 1;
      kmem_free(temp_name);
      fist_dprint(2, "RD: adding entry \"%s\" of length %d\n",
		  crt_dirent->d_name, crt_dirent->d_reclen);
      error = fist_uiomove(temp_iovec.iov_base + aux,
			   crt_dirent->d_reclen, UIO_READ, uiop);
      if (error)
	goto clean_up;
    }
  }
  uiop->uio_offset = temp_uio.uio_offset;

clean_up:
  kmem_free(temp_iovec.iov_base);
  print_location();
  return (error);
}
#endif /* FIST_FILTER_NAME */

/*
 * Global vfs data structures
 */
vop_t **wrapfs_vnodeop_p;
static struct vnodeopv_entry_desc wrapfs_vnodeop_entries[] =
{
  {&vop_default_desc,		(vop_t *) wrapfs_bypass},
  {&vop_access_desc,		(vop_t *) wrapfs_access},
  {&vop_createvobject_desc,	(vop_t *) wrapfs_createvobject },
  {&vop_destroyvobject_desc,	(vop_t *) wrapfs_destroyvobject },
  {&vop_getattr_desc,		(vop_t *) wrapfs_getattr},
  {&vop_getvobject_desc,	(vop_t *) wrapfs_getvobject },
  {&vop_inactive_desc,		(vop_t *) wrapfs_inactive},
  {&vop_lock_desc,		(vop_t *) wrapfs_lock},
  {&vop_islocked_desc,          (vop_t *) wrapfs_islocked},
#ifdef FIST_FILTER_NAME
  {&vop_lookup_desc,		(vop_t *) wrapfs_lookup}, /* FIST_FILTER_NAME */
#else /* not FIST_FILTER_NAME */
  {&vop_lookup_desc,		(vop_t *) vfs_cache_lookup}, /* FIST_FILTER_NAME */
#endif /* not FIST_FILTER_NAME */
  {&vop_print_desc,		(vop_t *) wrapfs_print},
  {&vop_reclaim_desc,		(vop_t *) wrapfs_reclaim},
  {&vop_setattr_desc,		(vop_t *) wrapfs_setattr},
  {&vop_unlock_desc,		(vop_t *) wrapfs_unlock},
  /* EZK added these */
#ifdef FIST_FILTER_DATA
  { &vop_read_desc,		(vop_t *) wrapfs_read },
  { &vop_write_desc,		(vop_t *) wrapfs_write },
  { &vop_getpages_desc,		(vop_t *) wrapfs_getpages },
  { &vop_putpages_desc,		(vop_t *) wrapfs_putpages },
#endif /* FIST_FILTER_DATA */
  { &vop_ioctl_desc,		(vop_t *) wrapfs_ioctl },
#ifdef HAVE_BMAP
  { &vop_bmap_desc,		(vop_t *) wrapfs_bmap },
#endif /* HAVE_BMAP */
#ifdef FIST_FILTER_NAME
  { &vop_mkdir_desc,		(vop_t *) wrapfs_mkdir },
  { &vop_rmdir_desc,		(vop_t *) wrapfs_rmdir },
  { &vop_create_desc,		(vop_t *) wrapfs_create },
  { &vop_remove_desc,		(vop_t *) wrapfs_remove },
  { &vop_rename_desc,		(vop_t *) wrapfs_rename },
  { &vop_link_desc,		(vop_t *) wrapfs_link },
  { &vop_symlink_desc,		(vop_t *) wrapfs_symlink },
  { &vop_readlink_desc,		(vop_t *) wrapfs_readlink },
  { &vop_readdir_desc,		(vop_t *) wrapfs_readdir },
  { &vop_whiteout_desc,		(vop_t *) wrapfs_whiteout },
  { &vop_mknod_desc,		(vop_t *) wrapfs_mknod },
#endif /* FIST_FILTER_NAME */
  /* KIRAN added this */
  { &vop_open_desc,		(vop_t *) wrapfs_open },
  {NULL, NULL}
};

static struct vnodeopv_desc wrapfs_vnodeop_opv_desc = {
  &wrapfs_vnodeop_p, wrapfs_vnodeop_entries
};

VNODEOP_SET(wrapfs_vnodeop_opv_desc);
