/*
 * Copyright (c) 1992, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from software donated to Berkeley by
 * Jan-Simon Pendry.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	@(#)wrapfs_subr.c	8.7 (Berkeley) 5/14/95
 *
 * $FreeBSD: src/sys/fs/wrapfs/wrapfs_subr.c,v 1.38 2002/10/14 03:20:33 mckusick Exp $
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */
#ifdef FISTGEN
# include "fist_wrapfs.h"
#endif /* FISTGEN */
#include <fist.h>
#include <wrapfs.h>

#if 0
#include "opt_debug.h"

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/lock.h>
#include <sys/mutex.h>
#include <sys/malloc.h>
#include <sys/mount.h>
#include <sys/proc.h>
#include <sys/vnode.h>
#include <fs/wrapfs/wrapfs.h>
#endif

#define LOG2_SIZEVNODE 7		/* log2(sizeof struct vnode) */
#define	NWRAPFSNODECACHE 16

/*
 * Wrapfs layer cache:
 * Each cache entry holds a reference to the lower vnode
 * along with a pointer to the alias vnode.  When an
 * entry is added the lower vnode is VREF'd.  When the
 * alias is removed the lower vnode is vrele'd.
 */

#define	WRAPFS_NHASH(vp) \
	(&wrapfs_node_hashtbl[(((uintptr_t)vp)>>LOG2_SIZEVNODE) & wrapfs_node_hash])

static LIST_HEAD(wrapfs_node_hashhead, wrapfs_node) *wrapfs_node_hashtbl;
static u_long wrapfs_node_hash;
struct mtx wrapfs_hashmtx;

static MALLOC_DEFINE(M_WRAPFSHASH, "WRAPFS hash", "WRAPFS hash table");
MALLOC_DEFINE(M_WRAPFSNODE, "WRAPFS node", "WRAPFS vnode private part");

static struct vnode * wrapfs_hashget(struct vnode *);
static struct vnode * wrapfs_hashins(struct wrapfs_node *);

int wrapfs_init(struct vfsconf *vfsp);
int wrapfs_uninit(struct vfsconf *vfsp);

/* defined by EZK */
void * kmem_zalloc(unsigned long size);
int fist_uiomove(caddr_t cp, int n, enum uio_rw rwflag, struct uio *uio);
#ifdef FIST_FILTER_NAME
cn_t * wrapfs_new_cnp(const vnode_t *thisvp, const cn_t *thiscnp);
void wrapfs_update_cnp(const vnode_t *thisvp, cn_t **lowercnpp, cn_t *thiscnp, int error);
#endif /* FIST_FILTER_NAME */
#ifdef FIST_FILTER_DATA
void fist_copy_object_attr(vm_object_t dest, const vm_object_t src);
void fist_copy_page_attr(vm_page_t dest, const vm_page_t src);
int wrapfs_fill_zeros(vnode_t *vp, vattr_t *vap, cred_t *cred, thread_t *p);
#endif /* FIST_FILTER_DATA */

/*
 * Initialise cache headers
 */
int
wrapfs_init(vfsp)
	struct vfsconf *vfsp;
{

   fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);
  fist_dprint(2, "wrapfs_init\n");		/* printed during system boot */
   wrapfs_node_hashtbl = hashinit(NWRAPFSNODECACHE, M_WRAPFSHASH, &wrapfs_node_hash);
	mtx_init(&wrapfs_hashmtx, "wrapfs", NULL, MTX_DEF);
	return (0);
}

int
wrapfs_uninit(vfsp)
	struct vfsconf *vfsp;
{
   fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);
   mtx_destroy(&wrapfs_hashmtx);
   if (wrapfs_node_hashtbl) {
    free(wrapfs_node_hashtbl, M_WRAPFSHASH);
   }
   return (0);
}

/*
 * Return a VREF'ed alias for lower vnode if already exists, else 0.
 * Lower vnode should be locked on entry and will be left locked on exit.
 */
static struct vnode *
wrapfs_hashget(lowervp)
	struct vnode *lowervp;
{
  struct thread *td = curthread;	/* XXX */
  struct wrapfs_node_hashhead *hd;
  struct wrapfs_node *a;
  struct vnode *vp;

  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);

  /*
   * Find hash base, and then search the (two-way) linked
   * list looking for a wrapfs_node structure which is referencing
   * the lower vnode.  If found, the increment the wrapfs_node
   * reference count (but NOT the lower vnode's VREF counter).
   */
  hd = WRAPFS_NHASH(lowervp);
loop:
   mtx_lock(&wrapfs_hashmtx);
   LIST_FOREACH(a, hd, wrapfs_hash) {
     if (a->wrapfs_lowervp == lowervp) {
       vp = WRAPFS_TO_VP(a);
       mtx_lock(&vp->v_interlock);
       mtx_unlock(&wrapfs_hashmtx);
       /*
        * We need vget for the VXLOCK
        * stuff, but we don't want to lock
        * the lower node.
        */
       if (vget(vp, LK_EXCLUSIVE | LK_THISLAYER | LK_INTERLOCK, td)) {
	 printf ("wrapfs_node_find: vget failed.\n");
	 goto loop;
       }
       return (vp);
     }
   }
   mtx_unlock(&wrapfs_hashmtx);

   print_location();
   return (NULLVP);
}

/*
 * Act like wrapfs_hashget, but add passed wrapfs_node to hash if no existing
 * node found.
 */
static struct vnode *
wrapfs_hashins(xp)
	struct wrapfs_node *xp;
{
  struct thread *td = curthread;	/* XXX */
  struct wrapfs_node_hashhead *hd;
  struct wrapfs_node *oxp;
  struct vnode *ovp;
  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);
  hd = WRAPFS_NHASH(xp->wrapfs_lowervp);
loop:
  mtx_lock(&wrapfs_hashmtx);
  LIST_FOREACH(oxp, hd, wrapfs_hash) {
    if (oxp->wrapfs_lowervp == xp->wrapfs_lowervp) {
      ovp = WRAPFS_TO_VP(oxp);
      mtx_lock(&ovp->v_interlock);
      mtx_unlock(&wrapfs_hashmtx);
      if (vget(ovp, LK_EXCLUSIVE | LK_THISLAYER | LK_INTERLOCK, td))
	goto loop;

      return (ovp);
    }
  }
  LIST_INSERT_HEAD(hd, xp, wrapfs_hash);
  mtx_unlock(&wrapfs_hashmtx);
  print_location();
  return (NULLVP);
}

/*
 * Make a new or get existing wrapfs node.
 * Vp is the alias vnode, lowervp is the lower vnode.
 *
 * The lowervp assumed to be locked and having "spare" reference. This routine
 * vrele lowervp if wrapfs node was taken from hash. Otherwise it "transfers"
 * the caller's "spare" reference to created wrapfs vnode.
 */
int
wrapfs_nodeget(mp, lowervp, vpp)
	struct mount *mp;
	struct vnode *lowervp;
	struct vnode **vpp;
{
  struct thread *td = curthread;	/* XXX */
  struct wrapfs_node *xp;
  struct vnode *vp;
  int error;

  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);
  /* Lookup the hash firstly */
  *vpp = wrapfs_hashget(lowervp);
  if (*vpp != NULL) {
    vrele(lowervp);
    return (0);
  }

  /*
   * We do not serialize vnode creation, instead we will check for
   * duplicates later, when adding new vnode to hash.
   *
   * Note that duplicate can only appear in hash if the lowervp is
   * locked LK_SHARED.
   */

  /*
   * Do the MALLOC before the getnewvnode since doing so afterward
   * might cause a bogus v_data pointer to get dereferenced
   * elsewhere if MALLOC should block.
   */
  MALLOC(xp, struct wrapfs_node *, sizeof(struct wrapfs_node),
	 M_WRAPFSNODE, M_WAITOK);

  error = getnewvnode("wrapfs", mp, wrapfs_vnodeop_p, &vp);
  if (error) {
    FREE(xp, M_WRAPFSNODE);
    return (error);
  }

  xp->wrapfs_vnode = vp;
  xp->wrapfs_lowervp = lowervp;

  vp->v_type = lowervp->v_type;
  vp->v_data = xp;

  /*
   * From NetBSD:
   * Now lock the new node. We rely on the fact that we were passed
   * a locked vnode. If the lower node is exporting a struct lock
   * (v_vnlock != NULL) then we just set the upper v_vnlock to the
   * lower one, and both are now locked. If the lower node is exporting
   * NULL, then we copy that up and manually lock the new vnode.
   */

  vp->v_vnlock = lowervp->v_vnlock;
  error = VOP_LOCK(vp, LK_EXCLUSIVE | LK_THISLAYER, td);
  if (error)
    panic("wrapfs_nodeget: can't lock new vnode\n");

  /*
   * Atomically insert our new node into the hash or vget existing
   * if someone else has beaten us to it.
   */
  *vpp = wrapfs_hashins(xp);
  if (*vpp != NULL) {
    vrele(lowervp);
    VOP_UNLOCK(vp, LK_THISLAYER, td);
    vp->v_vnlock = NULL;
    xp->wrapfs_lowervp = NULL;
    vrele(vp);
    return (0);
  }

  /*
   * XXX We take extra vref just to workaround UFS's XXX:
   * UFS can vrele() vnode in VOP_CLOSE() in some cases. Luckily, this
   * can only happen if v_usecount == 1. To workaround, we just don't
   * let v_usecount be 1, it will be 2 or more.
   */
  VREF(lowervp);

  *vpp = vp;

  print_location();
  return (0);
}

/*
 * Remove node from hash.
 */
void
wrapfs_hashrem(xp)
	struct wrapfs_node *xp;
{

	mtx_lock(&wrapfs_hashmtx);
	LIST_REMOVE(xp, wrapfs_hash);
	mtx_unlock(&wrapfs_hashmtx);
}

#ifdef DIAGNOSTIC
#include "opt_ddb.h"

#ifdef DDB
#define	wrapfs_checkvp_barrier	1
#else
#define	wrapfs_checkvp_barrier	0
#endif

struct vnode *
wrapfs_checkvp(vp, fil, lno)
	struct vnode *vp;
	char *fil;
	int lno;
{
  struct wrapfs_node *a = VP_TO_WRAPFS(vp);
  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);
#ifdef notyet
  /*
   * Can't do this check because vop_reclaim runs
   * with a funny vop vector.
   */
  if (vp->v_op != wrapfs_vnodeop_p) {
    printf ("wrapfs_checkvp: on non-wrapfs-node\n");
    while (wrapfs_checkvp_barrier) /*WAIT*/ ;
    panic("wrapfs_checkvp");
  };
#endif
  if (a->wrapfs_lowervp == NULLVP) {
    /* Should never happen */
    int i;
    u_long *p;
    printf("vp = %p, ZERO ptr\n", (void *)vp);
    for (p = (u_long *) a, i = 0; i < 8; i++)
      printf(" %lx", p[i]);
    printf("\n");
    /* wait for debugger */
    while (wrapfs_checkvp_barrier) /*WAIT*/ ;
    panic("wrapfs_checkvp");
  }
  if (vrefcnt(a->wrapfs_lowervp) < 1) {
    int i;
    u_long *p;
    printf("vp = %p, unref'ed lowervp\n", (void *)vp);
    for (p = (u_long *) a, i = 0; i < 8; i++)
      printf(" %lx", p[i]);
    printf("\n");
    /* wait for debugger */
    while (wrapfs_checkvp_barrier) /*WAIT*/ ;
       panic ("null with unref'ed lowervp");
  };
#ifdef notyet
  printf("null %x/%d -> %x/%d [%s, %d]\n",
	 WRAPFS_TO_VP(a), vrefcnt(WRAPFS_TO_VP(a)),
	 a->wrapfs_lowervp, vrefcnt(a->wrapfs_lowervp),fil, lno);
#endif
  return a->wrapfs_lowervp;
}
#endif

/****************************************************************************/
/* ADDED BY EZK */

/* allocate memory and zero it */
void *
kmem_zalloc(unsigned long size)
{
  void *addr;

  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);
  addr = malloc(size, M_TEMP, M_WAITOK);
  if (addr)
    bzero(addr, size);
  print_location();
  return addr;
}

/* do an I/O move */
int
fist_uiomove(caddr_t cp, int n, enum uio_rw rwflag, struct uio *uio)
{
  enum uio_rw saved_rwflag;
  int ret;

  fist_dprint(4, "FXN=%s FILE=%s LINE=%d\n",__FUNCTION__,__FILE__,__LINE__);
  if (uio->uio_rw != rwflag) {
    printf("UIOMOVE mismatching flags: uio->uio_rw=%d, rwflag=%d\n",
	   uio->uio_rw, rwflag);
  }
  /* save uio's rwflag */
  saved_rwflag = uio->uio_rw;
  uio->uio_rw = rwflag;
  ret = uiomove(cp, n, uio);
  uio->uio_rw = saved_rwflag;
  print_location();
  return ret;
}

#ifdef FIST_FILTER_NAME
/*
 * create a new (lower) componentname from an existing one
 * (this level) and encode the inner.
 */
cn_t *
wrapfs_new_cnp(const vnode_t *thisvp, const cn_t *thiscnp)
{
  cn_t *lowercnp;
#ifdef FIST_FILTER_NAME
  static char buf[MAXNAMLEN];
#else /* not FIST_FILTER_NAME */
  int len;
  char *name = NULL;
#endif /* not FIST_FILTER_NAME */

  if (!thiscnp)
    return NULL;

  lowercnp = kmem_alloc(sizeof(cn_t));
  if (!lowercnp)
    panic("wrapfs_new_cnp: no more memory\n");
  /* copy all fields */
  bcopy(thiscnp, lowercnp, sizeof(cn_t));
  /* fix certain fields temporarily */
  lowercnp->cn_pnbuf = NULL;
  lowercnp->cn_nameptr = NULL;
  lowercnp->cn_namelen = 0;

  /* prepare new path name for lookup */
#ifdef FIST_FILTER_NAME
  bcopy(thiscnp->cn_nameptr, buf, thiscnp->cn_namelen);
  buf[thiscnp->cn_namelen] = '\0';
  lowercnp->cn_namelen = wrapfs_encode_filename(buf,
			 thiscnp->cn_namelen,
			 &lowercnp->cn_pnbuf,
			 SKIP_DOTS,
			 thisvp, /* use namei zone alloc */
			 thisvp->v_mount);
  /* updated other cnp fields */
  lowercnp->cn_namelen--;	/* don't count terminating null */
  lowercnp->cn_nameptr = lowercnp->cn_pnbuf;
  /* I always allocate my own lowercnp buffer */
#else /* not FIST_FILTER_NAME */
  len = thiscnp->cn_namelen;
  name = zalloc(namei_zone);
  if (!name)
    panic("wrapfs_new_cnp: no more memory for name\n");
  bcopy(thiscnp->cn_nameptr, name, len+1);
  name[len] = '\0';		/* just in case (not needed) */
  /* updated other cnp fields */
  //name[0]++;			/* update first char of name ???*/
  lowercnp->cn_namelen = len;
  lowercnp->cn_nameptr = lowercnp->cn_pnbuf = name;
#endif /* not FIST_FILTER_NAME */

  /* return formed string */
  return lowercnp;
}

/*
 * Update an old cnp based on a newer one.
 * Also free previously allocated lowercnp and its fields.
 */
void
wrapfs_update_cnp(const vnode_t *thisvp, cn_t **lowercnpp, cn_t *thiscnp, int error)
{
  /*
   * free thiscnp if there no error, HASBUF was on, and SAVENAME was off.
   * XXX: what about SAVESTART (from <sys/namei.h>)
   */
  if (!error &&
      (thiscnp->cn_flags & (HASBUF|SAVENAME|SAVESTART)) == HASBUF) {
    fist_dprint(2, "UPDATE_CNP: freeing thiscnp->cn_pnbuf \"%s\"\n",
		thiscnp->cn_pnbuf);
    uma_zfree(namei_zone, thiscnp->cn_pnbuf);
  }

  if (!fist_isdeadcode(*lowercnpp)) {
    fist_dprint(2, "UPDATE_CNP: lowercnp flags <%s>\n",
		fist_cn_flags((*lowercnpp)->cn_flags));
  }

  /* always free space used by lowercnp, if not already free'd */
  if (fist_isdeadcode(*lowercnpp))
    return;



  if (!fist_isdeadcode((*lowercnpp)->cn_pnbuf)) {
    if (((*lowercnpp)->cn_flags & (HASBUF|SAVENAME|SAVESTART)) == HASBUF) {
      fist_dprint(2, "UPDATE_CNP: freeing lowercnp->cn_pnbuf \"%s\"\n",
		  (*lowercnpp)->cn_pnbuf);
      uma_zfree(namei_zone, (*lowercnpp)->cn_pnbuf);
    } else {
      fist_dprint(1, "UPDATE_CNP: not freeing 0x%x \"%s\" <%s>\n",
		  (int) *lowercnpp,
		  (*lowercnpp)->cn_pnbuf,
		  fist_cn_flags((*lowercnpp)->cn_flags));
    }
  }
  fist_dprint(2, "UPDATE_CNP: freeing lowercnp 0x%x\n", (int) *lowercnpp);
  kmem_free(*lowercnpp);

}
#endif /* FIST_FILTER_NAME */

#ifdef FIST_FILTER_DATA

void fist_copy_object_attr(vm_object_t dest, const vm_object_t src)
{
    dest->flags = src->flags;
}

void fist_copy_page_attr(vm_page_t dest, const vm_page_t src)
{
  dest->flags = src->flags;
  dest->dirty = src->dirty;
  dest->valid = src->valid;
}

/*
 * used for truncating a file to an offset beyond its current length
 */
int
wrapfs_fill_zeros(vnode_t *vp, vattr_t *vap, cred_t * cred, thread_t * td)
{
  int error = 0;
  vnode_t *hidden_vp;
  vattr_t hidden_vap;
  uio_t temp_uio;
  iovec_t *temp_iovec, *free_iovec;
  int i;
  caddr_t current_base;
  int resid, bytes_read, num_pages, first_page_bytes, real_first_page;
  long long start_loffset, end_loffset, page_start_loffset;
  long long page_end_loffset, current_loffset;
  int hidden_ioflag = IO_NODELOCKED;
  vm_object_t object;

  fist_dprint(4, "fist_wrapfs_fill_zeros vp=0x%x new size=0x%x \n",
	      (int) vp, vap->va_size);

  hidden_vp = WRAPFS_VP_TO_LOWERVP(vp);
  /* set the size of the vm object */
  vnode_pager_setsize(vp, vap->va_size);
  vnode_pager_setsize(hidden_vp, vap->va_size);

  /* get the attributes, length is necessary for correct updates */
  if ((error = VOP_GETATTR(hidden_vp, &hidden_vap, cred, td))) {  /* HARI */
    fist_dprint(4, "VOP_GETATTR returned error - not good\n");
    goto out;
  }

  if (hidden_vap.va_size >= vap->va_size) {
    return 0;
  }

  start_loffset = hidden_vap.va_size;
  end_loffset = vap->va_size;

  page_start_loffset = start_loffset & ~(PAGE_SIZE - 1);
  page_end_loffset = end_loffset & ~(PAGE_SIZE - 1);

  /*
   * if not multiple of PAGE_SIZE, then the above formula loses one page.
   * adjust for it
   */
  if (page_end_loffset < end_loffset)
    page_end_loffset += PAGE_SIZE;

  num_pages = (page_end_loffset - page_start_loffset) >> PAGE_SHIFT;

  temp_iovec = kmem_zalloc(num_pages * sizeof(iovec_t));
  free_iovec = kmem_zalloc(num_pages * sizeof(iovec_t));
  for (i = 0; i < num_pages; i++) {
    temp_iovec[i].iov_len  = free_iovec[i].iov_len  = PAGE_SIZE;
    temp_iovec[i].iov_base = free_iovec[i].iov_base  = kmem_zalloc(PAGE_SIZE);
  }

  /* read first block XXX check length of file */
  temp_uio.uio_iov = temp_iovec;
  temp_uio.uio_iovcnt = 1;
  temp_uio.uio_offset = page_start_loffset;
  temp_uio.uio_segflg = UIO_SYSSPACE;
  temp_uio.uio_rw = UIO_READ;
  temp_uio.uio_td = td;  /* HARI */
  temp_uio.uio_resid = start_loffset - page_start_loffset;
 fist_print_uios("FILL_ZEROS (before VOP_READ 1)", &temp_uio);
  error = VOP_READ(hidden_vp, &temp_uio, hidden_ioflag, cred);
  if (error) {
    fist_dprint(5, "VOP_READ returned error - not good\n");
    goto out_free;
  }

  fist_print_uios("FILL_ZEROS (after VOP_READ 1)", &temp_uio);
  bytes_read = PAGE_SIZE - temp_iovec[0].iov_len;
  temp_iovec[0].iov_base -= bytes_read;
  temp_iovec[0].iov_len = PAGE_SIZE;

  /* decode block read */
  wrapfs_decode_block(temp_iovec[0].iov_base, temp_iovec[0].iov_base,
		      bytes_read, vp, vp->v_mount, OFF_TO_IDX(start_loffset));
  bzero(temp_iovec[0].iov_base + bytes_read, PAGE_SIZE - bytes_read);
  /*
   * Now we are ready to write the bytes within the start/end
   * cleartext offsets in the buffers we allocated.
   */
  current_loffset = page_start_loffset;
  for (i = 0; i < num_pages ; i++) {
    current_base = temp_iovec[i].iov_base;
    wrapfs_fill_page(vp, temp_iovec[i].iov_base, current_loffset);
    wrapfs_encode_block(temp_iovec[i].iov_base, temp_iovec[i].iov_base,
			PAGE_SIZE, vp, vp->v_mount, OFF_TO_IDX(current_loffset));
    wrapfs_fill_page(hidden_vp, temp_iovec[i].iov_base, current_loffset);
    current_loffset += PAGE_SIZE;
  }

  resid = end_loffset - page_start_loffset;

  /* XXX: no need for full initialization here */
  temp_uio.uio_iov = temp_iovec;
  temp_uio.uio_iovcnt = num_pages;
  temp_uio.uio_offset = page_start_loffset;
  temp_uio.uio_segflg = UIO_SYSSPACE;
  temp_uio.uio_rw = UIO_WRITE;
  temp_uio.uio_td = td;
  temp_uio.uio_resid = resid;

  /*
   * pass operation to hidden filesystem, and return status
   */
  fist_print_uios("FILL_ZEROS (before write)", &temp_uio);
  error = VOP_WRITE(hidden_vp, &temp_uio, hidden_ioflag, cred);
  fist_print_uios("FILL_ZEROS (after write)", &temp_uio);

out_free:
  for (i = 0; i < num_pages; i++) {
    fist_dprint(6, "PRINT_BASE1 %d: 0x%x (len=%d)\n", i,
		temp_iovec[i].iov_base,
		temp_iovec[i].iov_len);
    kmem_free(free_iovec[i].iov_base);
  }
  kmem_free(temp_iovec);

out:

  print_location();
  return (error);
}
#endif /* FIST_FILTER_DATA */
