/*
 * Copyright (c) 1997-2005 Erez Zadok <ezk@cs.stonybrook.edu>
 * Copyright (c) 2001-2005 Stony Brook University
 *
 * For specific licensing information, see the COPYING file distributed with
 * this package, or get one from ftp://ftp.filesystems.org/pub/fistgen/COPYING.
 *
 * This Copyright notice must be kept intact and distributed with all
 * fistgen sources INCLUDING sources generated by fistgen.
 */
/*
 *  $Id: inode.c,v 1.27 2005/01/03 21:10:41 ezk Exp $
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */
#ifdef FISTGEN
# include "fist_wrapfs.h"
#endif /* FISTGEN */
#include "fist.h"
#include "wrapfs.h"


STATIC int
wrapfs_create(inode_t *dir, dentry_t *dentry, int mode)
{
    int err = -EACCES;
    dentry_t *hidden_dentry = wrapfs_hidden_dentry(dentry);
    dentry_t *hidden_dir_dentry;

    print_entry_location();
    ASSERT(hidden_dentry != NULL);

    fist_checkinode(dir, "wrapfs_create");

    hidden_dir_dentry = lock_parent(hidden_dentry);
    if (IS_ERR(hidden_dir_dentry)) {
	err = PTR_ERR(hidden_dir_dentry);
	goto out;
    }
    if (!hidden_dir_dentry->d_inode->i_op || !hidden_dir_dentry->d_inode->i_op->create)
	goto out_lock;

    err = hidden_dir_dentry->d_inode->i_op->create(hidden_dir_dentry->d_inode,
						   hidden_dentry, mode);
    if (err || !hidden_dentry->d_inode)
	goto out_lock;

    err = wrapfs_interpose(hidden_dentry, dentry, dir->i_sb, 0);
    if (err)
	goto out_lock;

    fist_copy_attr_timesizes(dir, hidden_dir_dentry->d_inode);
    fist_checkinode(dir, "post wrapfs_create");

 out_lock:
    unlock_dir(hidden_dir_dentry);
    if (!dentry->d_inode)
	d_drop(dentry);

 out:
    print_exit_status(err);
    return err;
}


/*
 * Before 2.2.7: lookup must fill in dentry and return errno (0 if ok)
 * Since 2.2.7: if err, returns ERR_PTR, else 0.
 */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,2,7)
STATIC dentry_t *
#else /* kernel before 2.2.7 */
STATIC int
#endif /* kernel before 2.2.7 */
wrapfs_lookup(inode_t *dir, dentry_t *dentry)
{
    int err = 0;
    dentry_t *hidden_dir_dentry = wrapfs_hidden_dentry(dentry->d_parent);
    dentry_t *hidden_dentry;
    const char *name = dentry->d_name.name;
    vnode_t *this_vnode;
    dentry_t *this_dir;
#ifdef FIST_FILTER_NAME
    unsigned int namelen = dentry->d_name.len;
    char *encoded_name;
    unsigned int encoded_namelen;
#endif /* FIST_FILTER_NAME */

    print_entry_location();
    fist_checkinode(dir, "wrapfs_lookup");

    this_vnode = dir;
    this_dir = hidden_dir_dentry;

    fist_print_dentry("LOOKUP: dentry IN", dentry);
    fist_print_dentry("LOOKUP: dentry->d_parent IN", dentry->d_parent);
    fist_print_dentry("LOOKUP: hidden_dir_dentry IN", hidden_dir_dentry);
    fist_print_inode("LOOKUP: dir IN", dir);

    if (hidden_dir_dentry->d_inode)
	fist_print_inode("LOOKUP: hidden_dir_dentry->d_inode",
			 hidden_dir_dentry->d_inode);

    /* must initialize dentry operations */
    dentry->d_op = &wrapfs_dops;

    FIST_OP_LOOKUP_PRECALL;

    /* increase refcount of base dentry (lookup_dentry will decrement) */
    // THIS IS RIGHT! (don't "fix" it)
    dget(hidden_dir_dentry);

#ifndef FIST_FILTER_NAME
    /* will allocate a new hidden dentry if needed */
    hidden_dentry = lookup_dentry(name, hidden_dir_dentry, 0);
#else /* FIST_FILTER_NAME */
    encoded_namelen = wrapfs_encode_filename(name,
					     namelen,
					     &encoded_name,
					     SKIP_DOTS, dir, dir->i_sb);
    /* will allocate a new hidden dentry if needed */
    hidden_dentry = lookup_dentry(encoded_name, hidden_dir_dentry, 0);
    kfree_s(encoded_name, encoded_namelen);
#endif /* FIST_FILTER_NAME */

    if (IS_ERR(hidden_dentry)) {
	/*
	 * this produces an unusual dentry: one that has neither an
	 * inode, nor a private structure attached to it. All cleanup
	 * methods (d_delete, d_release, etc) must be prepared to deal
	 * with such dentries. Ion 09/29/2001
	 */
	printk("ERR from hidden_dentry!!!\n");
	err = PTR_ERR(hidden_dentry);
	goto out;
    }

    FIST_OP_LOOKUP_POSTCALL;

    /* update parent directory's atime */
    fist_copy_attr_atime(dir, hidden_dir_dentry->d_inode);
    /* link the upper and lower dentries */
    /* XXX allocate dentry_private_data if necessary */
    dtohd(dentry) = hidden_dentry;

    /* lookup is special: it needs to handle negative dentries */
    if (!hidden_dentry->d_inode) {
	d_add(dentry, NULL);
	fist_print_dentry("lookup hidden", hidden_dentry);
	goto out;
    }

    fist_dprint(6, "lookup \"%s\" -> inode %d\n", name, hidden_dentry->d_inode->i_ino);
    err = wrapfs_interpose(hidden_dentry, dentry, dir->i_sb, 1);
    if (err)
	goto out_free;

    fist_checkinode(dentry->d_inode, "wrapfs_lookup OUT: dentry->d_inode:");
    fist_checkinode(dir, "wrapfs_lookup OUT: dir:");

    fist_print_dentry(__FUNCTION__ " OUT hidden_dentry", hidden_dentry);
    fist_print_dentry(__FUNCTION__ " OUT hidden_dir_dentry", hidden_dir_dentry);
    /* All is well */
    goto out;

 out_free:
    d_drop(dentry);
    /* XXX free dentry_private_data if necessary */
    dtopd(dentry) = NULL;

out:
    fist_print_dentry("LOOKUP: dentry OUT", dentry);
    print_exit_status(err);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,2,7)
    return ERR_PTR(err);
#else /* kernel before 2.2.7 */
    return err;
#endif /* kernel before 2.2.7 */
}


STATIC int
wrapfs_link(dentry_t *old_dentry, inode_t *dir, dentry_t *new_dentry)
{
    int err;
    dentry_t *hidden_old_dentry = wrapfs_hidden_dentry(old_dentry);
    dentry_t *hidden_new_dentry = wrapfs_hidden_dentry(new_dentry);
    dentry_t *hidden_dir_dentry;

    print_entry_location();

    fist_checkinode(dir, "wrapfs_link-dir");
    fist_checkinode(old_dentry->d_inode, "wrapfs_link-oldinode");

    dget(hidden_old_dentry);
    dget(hidden_new_dentry);
    hidden_dir_dentry = lock_parent(hidden_new_dentry);
    err = -ENOENT;
    if (!check_parent(hidden_dir_dentry, hidden_new_dentry))
	goto out_lock;

    err = -EXDEV;
    if (hidden_dir_dentry->d_inode->i_dev != hidden_old_dentry->d_inode->i_dev)
	goto out_lock;

    err = -EPERM;
    if (!hidden_dir_dentry->d_inode->i_op || !hidden_dir_dentry->d_inode->i_op->link)
	goto out_lock;

    DQUOT_INIT(hidden_dir_dentry->d_inode);
    err = hidden_dir_dentry->d_inode->i_op->link(hidden_old_dentry,
						 hidden_dir_dentry->d_inode,
						 hidden_new_dentry);

    if (err || !hidden_new_dentry->d_inode)
	goto out_lock;
    err = wrapfs_interpose(hidden_new_dentry, new_dentry, dir->i_sb, 0);
    if (err)
	goto out_lock;
    /*
     * XXX: we have to do a wrapfs_copy_inode (or subset thereof) in
     * *ALL* functions, all file systems, and all OSs!!!
     */
    fist_copy_attr_timesizes(dir, hidden_new_dentry->d_inode);

 out_lock:
    unlock_dir(hidden_dir_dentry);
    dput(hidden_new_dentry);
    dput(hidden_old_dentry);
    if (!new_dentry->d_inode)
	d_drop(new_dentry);

    print_exit_status(err);
    return err;
}


STATIC int
wrapfs_unlink(inode_t *dir, dentry_t *dentry)
{
    int err = 0;
    inode_t *hidden_dir = itohi(dir);
    dentry_t *hidden_dentry = wrapfs_hidden_dentry(dentry);
    dentry_t *hidden_dir_dentry;

    print_entry_location();
    ASSERT(hidden_dentry != NULL);
    fist_checkinode(dir, "wrapfs_unlink-dir");

    dget(dentry);
    hidden_dir_dentry = lock_parent(hidden_dentry);

    if (!check_parent(hidden_dir_dentry, hidden_dentry))
	goto out_lock;

    /* avoid destroying the hidden inode if the file is in use */
    dget(hidden_dentry);
    err = vfs_unlink(hidden_dir, hidden_dentry);
    dput(hidden_dentry);

    if (!err)
	d_delete(hidden_dentry);

 out_lock:
    fist_copy_attr_times(dir, hidden_dir);
    /* propagate number of hard-links */
    dentry->d_inode->i_nlink = itohi(dentry->d_inode)->i_nlink;

    unlock_dir(hidden_dir_dentry);

    /*
     * call d_drop so the system "forgets" about us
     */
    if (!err)
	d_drop(dentry);

    dput(dentry);

    print_exit_status(err);
    return err;
}


STATIC int
wrapfs_symlink(inode_t *dir, dentry_t *dentry, const char *symname)
{
    int err;
    dentry_t *hidden_dentry = wrapfs_hidden_dentry(dentry);
    dentry_t *hidden_dir_dentry;
    const char *hidden_symname = symname; /* XXX: can modify this */
#ifdef FIST_FILTER_NAME
    char *encoded_symname;
    unsigned int encoded_symlen;
#endif /* FIST_FILTER_NAME */

    print_entry_location();
    fist_checkinode(dir, "wrapfs_symlink-dir");

    dget(hidden_dentry);
    hidden_dir_dentry = lock_parent(hidden_dentry);
    err = -ENOENT;
    if (!check_parent(hidden_dir_dentry, hidden_dentry))
	goto out_lock;

    if (!hidden_dir_dentry->d_inode->i_op ||
	!hidden_dir_dentry->d_inode->i_op->symlink) {
	err = -EPERM;
	goto out_lock;
    }

#ifndef FIST_FILTER_NAME
    err = hidden_dir_dentry->d_inode->i_op->symlink(hidden_dir_dentry->d_inode,
						    hidden_dentry,
						    hidden_symname);
#else /* FIST_FILTER_NAME */
    encoded_symlen = wrapfs_encode_filename(symname,
					    strlen(symname),
					    &encoded_symname,
					    DO_DOTS, dir, dir->i_sb);
    err = hidden_dir_dentry->d_inode->i_op->symlink(hidden_dir_dentry->d_inode,
						    hidden_dentry,
						    encoded_symname);
    kfree_s(encoded_symname, encoded_symlen);
#endif /* FIST_FILTER_NAME */
    if (err || !hidden_dentry->d_inode)
	goto out_lock;
    err = wrapfs_interpose(hidden_dentry, dentry, dir->i_sb, 0);
    if (err)
	goto out_lock;

    fist_copy_attr_timesizes(dir, hidden_dir_dentry->d_inode);
    fist_checkinode(dir, "post wrapfs_symlink-dir");

 out_lock:
    unlock_dir(hidden_dir_dentry);
    dput(hidden_dentry);
    if (!dentry->d_inode)
	d_drop(dentry);

    print_exit_status(err);
    return err;
}


STATIC int
wrapfs_mkdir(inode_t *dir, dentry_t *dentry, int mode)
{
    int err;
    dentry_t *hidden_dentry = wrapfs_hidden_dentry(dentry);
    dentry_t *hidden_dir_dentry;

    print_entry_location();
    fist_checkinode(dir, "wrapfs_mkdir-dir");

    hidden_dir_dentry = lock_parent(hidden_dentry);
    err = -ENOENT;
    if (!check_parent(hidden_dir_dentry, hidden_dentry))
	goto out_lock;

    if (!hidden_dir_dentry->d_inode->i_op ||
	!hidden_dir_dentry->d_inode->i_op->mkdir) {
	err = -EPERM;
	goto out_lock;
    }

    err = hidden_dir_dentry->d_inode->i_op->mkdir(hidden_dir_dentry->d_inode,
						  hidden_dentry,
						  mode);
    if (err || !hidden_dentry->d_inode)
	goto out_lock;
    err = wrapfs_interpose(hidden_dentry, dentry, dir->i_sb, 0);
    if (err)
	goto out_lock;
    fist_copy_attr_timesizes(dir, hidden_dir_dentry->d_inode);
    /* update number of links on parent directory */
    dir->i_nlink = hidden_dir_dentry->d_inode->i_nlink;

    fist_checkinode(dir, "post wrapfs_mkdir-dir");

 out_lock:
    unlock_dir(hidden_dir_dentry);
    if (!dentry->d_inode)
	d_drop(dentry);

 out:
    print_exit_status(err);
    return err;
}


STATIC int
wrapfs_rmdir(inode_t *dir, dentry_t *dentry)
{
    int err = 0;
    dentry_t *hidden_dentry = wrapfs_hidden_dentry(dentry);
    dentry_t *hidden_dir_dentry;

    print_entry_location();
    fist_checkinode(dir, "wrapfs_rmdir-dir");

    dget(dentry);
    hidden_dir_dentry = dget(hidden_dentry->d_parent);

    /*
     * The dentry->d_count stuff confuses d_delete() enough to
     * not kill the inode from under us while it is locked. This
     * wouldn't be needed, except the dentry semaphore is really
     * in the inode, not in the dentry..
     */
    fist_print_dentry("wrapfs_rmdir-hidden_dentry1", hidden_dentry);
    hidden_dentry->d_count++;
    double_lock(hidden_dir_dentry, hidden_dentry);

    if (!check_parent(hidden_dir_dentry, hidden_dentry))
	goto out_lock;

    /* avoid destroying the hidden inode if the file is in use */
    dget(hidden_dentry);
    err = vfs_rmdir(hidden_dir_dentry->d_inode, hidden_dentry);
    dput(hidden_dentry);

    if (!err)
	d_delete(hidden_dentry);

 out_lock:
    fist_copy_attr_times(dir, hidden_dir_dentry->d_inode);
    /* copy the nlink count for our dentry and our parent's dentry */
    dir->i_nlink =  hidden_dir_dentry->d_inode->i_nlink;

    double_unlock(hidden_dentry, hidden_dir_dentry);

    fist_print_dentry("wrapfs_rmdir-dentry", dentry);
    fist_print_inode("wrapfs_rmdir-inode", dentry->d_inode);
    /*
     * call d_drop so the system "forgets" about us
     */
    if (!err)
	d_drop(dentry);

    dput(dentry);

    print_exit_status(err);
    return err;
}


STATIC int
wrapfs_mknod(inode_t *dir, dentry_t *dentry, int mode, int dev)
{
    int err;
    dentry_t *hidden_dentry = wrapfs_hidden_dentry(dentry);
    dentry_t *hidden_dir_dentry;

    print_entry_location();
    fist_checkinode(dir, "wrapfs_mknod-dir");

    hidden_dir_dentry = lock_parent(hidden_dentry);
    err = -ENOENT;
    if (!check_parent(hidden_dir_dentry, hidden_dentry))
	goto out_lock;

    if (!hidden_dir_dentry->d_inode->i_op || !hidden_dir_dentry->d_inode->i_op->mknod) {
	err = -EPERM;
	goto out_lock;
    }

    DQUOT_INIT(hidden_dir_dentry->d_inode);
    err = hidden_dir_dentry->d_inode->i_op->mknod(hidden_dir_dentry->d_inode,
						  hidden_dentry,
						  mode,
						  dev);
    if (err || !hidden_dentry->d_inode)
	goto out_lock;

    err = wrapfs_interpose(hidden_dentry, dentry, dir->i_sb, 0);
    if (err)
	goto out_lock;
    fist_copy_attr_timesizes(dir, hidden_dir_dentry->d_inode);

 out_lock:
    unlock_dir(hidden_dir_dentry);
    if (!dentry->d_inode)
	d_drop(dentry);

 out:
    fist_checkinode(dir, "post wrapfs_mknod-dir");
    print_exit_status(err);
    return err;
}


#if LINUX_VERSION_CODE < KERNEL_VERSION(2,2,6)
/* this is for kernels before 2.2.6 */
STATIC int
wrapfs_rename(inode_t *old_dir, dentry_t *old_dentry,
	      inode_t *new_dir, dentry_t *new_dentry)
{
    int err = -EPERM;
    dentry_t *hidden_old_dentry = wrapfs_hidden_dentry(old_dentry);
    dentry_t *hidden_new_dentry = wrapfs_hidden_dentry(new_dentry);
    dentry_t *hidden_old_dir_dentry;
    dentry_t *hidden_new_dir_dentry;
    int (*rename_fxn) (inode_t *, dentry_t *, inode_t *, dentry_t *);

    print_entry_location();

    hidden_old_dir_dentry = get_parent(hidden_old_dentry);
    hidden_new_dir_dentry = get_parent(hidden_new_dentry);
    dget(hidden_old_dentry);
    dget(hidden_new_dentry);
    double_lock(hidden_old_dir_dentry, hidden_new_dir_dentry);

    fist_checkinode(old_dir, "wrapfs_rename-old_dir");
    fist_checkinode(new_dir, "wrapfs_rename-new_dir");

    if (!hidden_old_dir_dentry->d_inode->i_op ||
	!(rename_fxn = hidden_old_dir_dentry->d_inode->i_op->rename)) {
	err = -EPERM;
	goto out_lock;
    }

    err = rename_fxn(hidden_old_dir_dentry->d_inode, hidden_old_dentry,
		     hidden_new_dir_dentry->d_inode, hidden_new_dentry);
    if (err)
	goto out_lock;
    //    ASSERT(new_dentry->d_inode == NULL);
    //    new_dentry->d_inode = old_dentry->d_inode;
    //    err = wrapfs_interpose(hidden_new_dentry, new_dentry, new_dir->i_sb, 0);
    d_move(old_dentry, new_dentry);

    fist_copy_attr_all(new_dir, hidden_new_dir_dentry->d_inode);
    if (new_dir != old_dir)
	fist_copy_attr_times(old_dir, hidden_old_dir_dentry->d_inode);

 out_lock:
    double_unlock(hidden_old_dir_dentry, hidden_new_dir_dentry);
 out:
    dput(hidden_new_dentry);
    dput(hidden_old_dentry);

    fist_checkinode(new_dir, "post wrapfs_mknod-new_dir");
    print_exit_status(err);
    return err;
}
#else /* not LINUX_VERSION_CODE < KERNEL_VERSION(2,2,6) */
/* this is for 2.2.6 kernels and later (big rename changes) */
STATIC int
wrapfs_rename(inode_t *old_dir, dentry_t *old_dentry,
	      inode_t *new_dir, dentry_t *new_dentry)
{
    int err = -EPERM;
    dentry_t *hidden_old_dentry = wrapfs_hidden_dentry(old_dentry);
    dentry_t *hidden_new_dentry = wrapfs_hidden_dentry(new_dentry);
    dentry_t *hidden_old_dir_dentry;
    dentry_t *hidden_new_dir_dentry;
    int (*rename_fxn) (inode_t *, dentry_t *, inode_t *, dentry_t *);

    print_entry_location();
    fist_checkinode(old_dir, "wrapfs_rename-old_dir");
    fist_checkinode(new_dir, "wrapfs_rename-new_dir");

    dget(hidden_old_dentry);
    dget(hidden_new_dentry);
    hidden_old_dir_dentry = get_parent(hidden_old_dentry);
    hidden_new_dir_dentry = get_parent(hidden_new_dentry);
    double_lock(hidden_old_dir_dentry, hidden_new_dir_dentry);

    err = -ENOENT;
    if (check_parent(hidden_old_dir_dentry, hidden_old_dentry) &&
	check_parent(hidden_new_dir_dentry, hidden_new_dentry))
	err = vfs_rename(hidden_old_dir_dentry->d_inode, hidden_old_dentry,
			 hidden_new_dir_dentry->d_inode, hidden_new_dentry);
    if (err)
	goto out_lock;

    fist_copy_attr_all(new_dir, hidden_new_dir_dentry->d_inode);
    if (new_dir != old_dir)
	fist_copy_attr_all(old_dir, hidden_old_dir_dentry->d_inode);

 out_lock:
    double_unlock(hidden_old_dir_dentry, hidden_new_dir_dentry);
    dput(hidden_new_dentry);
    dput(hidden_old_dentry);

    fist_checkinode(new_dir, "post wrapfs_mknod-new_dir");
    print_exit_status(err);
    return err;
}
#endif /* not LINUX_VERSION_CODE < KERNEL_VERSION(2,2,6) */


STATIC int
wrapfs_readlink(dentry_t *dentry, char *buf, int bufsiz)
{
    int err;
    dentry_t *hidden_dentry = wrapfs_hidden_dentry(dentry);
#ifdef FIST_FILTER_NAME
    char *decoded_name, *hidden_buf;
    mm_segment_t old_fs;
#endif /* FIST_FILTER_NAME */

    print_entry_location();
    fist_print_dentry("wrapfs_readlink IN", dentry);

    if (!hidden_dentry->d_inode->i_op ||
	!hidden_dentry->d_inode->i_op->readlink) {
	err = -EINVAL;
	goto out;
    }

#ifndef FIST_FILTER_NAME
    err = hidden_dentry->d_inode->i_op->readlink(hidden_dentry,
						 buf,
						 bufsiz);
    if (err > 0)
	fist_copy_attr_atime(dentry->d_inode, hidden_dentry->d_inode);
#else /* FIST_FILTER_NAME */
    hidden_buf = kmalloc(bufsiz, GFP_KERNEL);
    if (hidden_buf == NULL) {
	printk("Out of memory.\n");
	err = -ENOMEM;
	goto out;
    }

    old_fs = get_fs();
    set_fs(KERNEL_DS);
    err = hidden_dentry->d_inode->i_op->readlink(hidden_dentry,
						 hidden_buf,
						 bufsiz);
    set_fs(old_fs);
    if (err >= 0) {
	fist_dprint(7, "READLINK: link \"%s\", length %d\n", hidden_buf, err);
	err = wrapfs_decode_filename(hidden_buf, err,
				     &decoded_name, DO_DOTS, dentry->d_inode, dentry->d_sb);
	if (err > 0) {
	    if (copy_to_user(buf, decoded_name, err))
		err = -EFAULT;
	    kfree_s(decoded_name, err);
	}
	fist_copy_attr_atime(dentry->d_inode, hidden_dentry->d_inode);
    }

    kfree_s(hidden_buf, bufsiz);
#endif /* FIST_FILTER_NAME */

 out:
    print_exit_status(err);
    return err;
}


/* inode is from wrapfs, dir could be anything */
STATIC dentry_t *
wrapfs_follow_link(dentry_t *dentry, dentry_t *base, unsigned int follow)
{
    char *buf;
    int len = PAGE_SIZE, err;
    dentry_t *ret_dentry = dentry;
    dentry_t *hidden_dentry = wrapfs_hidden_dentry(dentry);
    mm_segment_t old_fs;

    print_entry_location();
    //    fist_print_dentry("wrapfs_follow_link dentry IN", dentry);
    //    fist_print_dentry("wrapfs_follow_link base IN", base);

    if (!hidden_dentry->d_inode->i_op ||
	!hidden_dentry->d_inode->i_op->readlink ||
	!hidden_dentry->d_inode->i_op->follow_link) {
	/*
	 * do_follow_link will do a dput() on the dentry passed to us
	 * and if that is the same as ret_dentry, and we want it kept,
	 * then we must increment the dentry's refcount to avoid having
	 * it deallocate/destroyed.
	 */
	dget(ret_dentry);
	/*
	 * Looking at do_follow_link, if we implement follow_link, then
	 * we have to dput(base).  Normally that's done by lookup_dentry
	 * but here we don't run that code.
	 */
	dput(base);
	goto out;
    }

    buf = kmalloc(len, GFP_KERNEL);
    if (!buf) {
	ret_dentry = ERR_PTR(-ENOMEM);
	dput(base); // for the same reason as above
	goto out;
    }
    /* read the hidden symlink, and then we will follow it */
    old_fs = get_fs();
    set_fs(KERNEL_DS);
    err = dentry->d_inode->i_op->readlink(dentry, buf, len);
    set_fs(old_fs);
    if (err < 0) {
	ret_dentry = ERR_PTR(err);
	dput(base); // for the same reason as above
	goto out_free;
    }
    fist_copy_attr_atime(dentry->d_inode, hidden_dentry->d_inode);
    buf[err] = 0;	// terminate the buffer for lookup_dentry's sake

    /*
     * We do not dget(base) b/c the way we were called here,
     * the base (passed to us) was already dget'ed.
     */
    ret_dentry = lookup_dentry(buf, base, follow);
    //    fist_print_dentry("wrapfs_follow_link base OUT", base);

 out_free:
    kfree_s(buf, len);
 out:
    print_exit_location();
    return ret_dentry;
}


#if 0
STATIC void
fist_page_mkclean(page_t *page)
{
    unsigned long address;
    pmd_t *pmd;
    pgd_t *pgd;
    pte_t *ppte;

    ASSERT(page != NULL);
    ASSERT(page->inode != NULL);
    ASSERT(page->inode->i_mmap != NULL);
    ASSERT(page->inode->i_mmap->vm_mm != NULL);
    ASSERT(page->inode->i_mmap->vm_mm->pgd != NULL);

    address = page->offset;
    pgd = page->inode->i_mmap->vm_mm->pgd;
    pmd = pmd_offset(pgd, address);
    ppte = pte_offset(pmd, address);
    set_pte(ppte, pte_mkclean(*ppte));
    ASSERT(!pte_dirty(*ppte));
}

STATIC void
fist_page_assertclean(page_t *page)
{
    unsigned long address;
    pmd_t *pmd;
    pgd_t *pgd;
    pte_t *ppte;

    ASSERT(page != NULL);
    ASSERT(page->inode != NULL);
    ASSERT(page->inode->i_mmap != NULL);
    ASSERT(page->inode->i_mmap->vm_mm != NULL);
    ASSERT(page->inode->i_mmap->vm_mm->pgd != NULL);

    address = page->offset;
    pgd = page->inode->i_mmap->vm_mm->pgd;
    pmd = pmd_offset(pgd, address);
    ppte = pte_offset(pmd, address);
    //    set_pte(ppte, pte_mkclean(*ppte));
    ASSERT(!pte_dirty(*ppte));
}
#endif


#if LINUX_VERSION_CODE < KERNEL_VERSION(2,2,18)
static inline
struct page *__read_cache_page(struct inode *inode,
			       unsigned long offset,
			       int (*filler)(void *,struct page*),
			       void *data)
{
	struct page **hash = page_hash(inode, offset);
	struct page *page;
	unsigned long cached_page = 0;
	int err;

	offset &= PAGE_CACHE_MASK;
repeat:
	page = __find_page(inode, offset, *hash);
	if (!page) {
		if (!cached_page) {
			cached_page = page_cache_alloc();
			if (!cached_page)
				return ERR_PTR(-ENOMEM);
			goto repeat;
		}
		page = page_cache_entry(cached_page);
		cached_page = 0;
		add_to_page_cache(page, inode, offset, hash);
		set_bit(PG_locked, &page->flags);
		err = filler(data, page);
		if (err < 0) {
			page_cache_release(page);
			page = ERR_PTR(err);
		}
	}
	if (cached_page)
		page_cache_free(cached_page);
	return page;
}

/*
 * Read into the page cache. If a page already exists,
 * and Page_Uptodate() is not set, try to fill the page.
 */
struct page *read_cache_page(struct inode *inode,
				unsigned long offset,
				int (*filler)(void *,struct page*),
				void *data)
{
	struct page *page = __read_cache_page(inode, offset, filler, data);
	int err;

	if (IS_ERR(page) || PageUptodate(page))
		goto out;

	wait_on_page(page);
	if (PageUptodate(page))
		goto out;

	set_bit(PG_locked, &page->flags);
	err = filler(data, page);
	if (err < 0) {
		page_cache_release(page);
		page = ERR_PTR(err);
	}
 out:
	return page;
}
#endif /* kernels older than 2.2.18 */


STATIC int
wrapfs_readpage(file_t *file, page_t *page)
{
    int err = 0;
    inode_t *inode = file->f_dentry->d_inode;
    inode_t *hidden_inode = itohi(inode);
    file_t *hidden_file = ftohf(file);
    page_t *hidden_page;
    unsigned long page_data, hidden_page_data;

    print_entry_location();

    /*
     * readpage is called from generic_file_read and the fault handler.
     * If your file system uses generic_file_read for the read op, it
     * must implement readpage.
     */

    ASSERT(hidden_inode->i_op != NULL);

    atomic_inc(&page->count);
    set_bit(PG_locked, &page->flags);
    set_bit(PG_free_after, &page->flags);

    fist_print_page_flags("READPAGE flags (entry)", page);
    fist_dprint(6, "READPAGE: f_pos:0x%x inode:0x%x i_size:0x%x hidden_inode:0x%x page->inode:0x%x page->offset:%lu page->count:%d\n",
		(int) file->f_pos, (int) inode, inode->i_size, (int) hidden_inode,
		(int) page->inode, page->offset, atomic_read(&page->count));


    hidden_page = read_cache_page(hidden_inode, page->offset,
				  (filler_t *)hidden_inode->i_op->readpage,
				  hidden_file);

    if (IS_ERR(hidden_page)) {
	err = PTR_ERR(hidden_page);
	/* XXX: we might have to do some unlocking on this error path */
	goto out;
    }

    /*
     * decode hidden_page data onto page
     */
    page_data = page_address(page);
    hidden_page_data = page_address(hidden_page);

    wrapfs_decode_block((char *) hidden_page_data, (char *) page_data, PAGE_SIZE, inode, inode->i_sb);

    /* release our reference to the hidden page */
    page_cache_release(hidden_page);

    /*
     * adjust flags and wake up processes waiting on the page
     */
    set_bit(PG_uptodate, &page->flags);

out:
    /*
     * we have to unlock our page, b/c we locked it above
     */
    clear_bit(PG_locked, &page->flags);
    wake_up(&page->wait);
    after_unlock_page(page);

    fist_dprint(6, "READPAGE (exit/page): f_pos:0x%x inode:0x%x i_size:0x%x hidden_inode:0x%x page->inode:0x%x page->offset:%lu page->count:%d\n",
		(int) file->f_pos, (int) inode, inode->i_size, (int) hidden_inode,
		(int) page->inode, page->offset, atomic_read(&page->count));
    fist_dprint(6, "READPAGE (exit/hidden_page): f_pos:0x%x inode:0x%x i_size:0x%x hidden_inode:0x%x page->inode:0x%x page->offset:%lu page->count:%d\n",
		(int) file->f_pos, (int) inode, inode->i_size, (int) hidden_inode,
		(int) hidden_page->inode, hidden_page->offset, atomic_read(&hidden_page->count));

    fist_print_page_flags("READPAGE flags (exit)", page);
    fist_print_page_flags("READPAGE hidden_flags (exit)", hidden_page);
    print_exit_status(err);
    return err;
}


#ifdef NOT_NEEDED
#error ARE YOU SURE YOU WANT TO DEFINE WRITEPAGE???
/* XXX: the VFS never calls writepage */
STATIC int
wrapfs_writepage(file_t *file, page_t *page)
{
    int err = -EIO;
    file_t *hidden_file = ftohf(file);
    inode_t *hidden_inode = hidden_file->f_dentry->d_inode;

    print_entry_location();

    if (hidden_inode->i_op && hidden_inode->i_op->writepage) {
	fist_dprint(2, "non-generic writepage\n");
	err = hidden_inode->i_op->writepage(hidden_file, page);
	if (err >= 0)
	    fist_copy_attr_times(file->f_dentry->d_inode, hidden_inode);
    }

    print_exit_status(err);
    return err;
}
#endif /* NOT_NEEDED */


STATIC int
wrapfs_bmap(inode_t *inode, int block)
{
    int err = -EINVAL;
    inode_t *hidden_inode = itohi(inode);

    print_entry_location();

    if (hidden_inode->i_op && hidden_inode->i_op->bmap)
	err = hidden_inode->i_op->bmap(hidden_inode, block);

    print_exit_status(err);
    return err;
}


STATIC void
wrapfs_truncate(inode_t *inode)
{
    inode_t *hidden_inode = itohi(inode);

    print_entry_location();

    fist_checkinode(inode, "wrapfs_truncate");
    if (!hidden_inode->i_op || !hidden_inode->i_op->truncate)
	goto out;

    down(&hidden_inode->i_sem);
    hidden_inode->i_op->truncate(hidden_inode);
    up(&hidden_inode->i_sem);

    fist_copy_attr_timesizes(inode, hidden_inode);

 out:
    print_exit_location();
}


STATIC int
wrapfs_permission(inode_t *inode, int mask)
{
    inode_t *hidden_inode = itohi(inode);
    int mode = inode->i_mode;
    int err;

    print_entry_location();
    if ((mask & S_IWOTH) && IS_RDONLY(inode) &&
	(S_ISREG(mode) || S_ISDIR(mode) || S_ISLNK(mode))) {
	err = -EROFS; /* Nobody gets write access to a read-only fs */
	goto out;
    }
    err = permission(hidden_inode, mask);
 out:
    print_exit_status(err);
    return err;
}


STATIC int
wrapfs_smap(inode_t *inode, int block)
{
    int err = -EINVAL;
    inode_t *hidden_inode = itohi(inode);

    print_entry_location();

    if (hidden_inode->i_op && hidden_inode->i_op->smap)
	err = hidden_inode->i_op->smap(hidden_inode, block);
    else
	err = bmap(hidden_inode, block);

    print_exit_status(err);
    return err;
}


#ifdef NOT_NEEDED
/*
 * We don't need this because we will implement file_write, which will
 * call the hidden file_write, and the latter, if set to generic_file_write,
 * will call hidden's update_page (if implemented by lower file system).
 */
STATIC int
wrapfs_updatepage(file_t *file, page_t *page, const char *buf,
		  unsigned long offset, unsigned int count, int sync)
{
    int err = -EIO;
    file_t *hidden_file = ftohf(file);
    char *hidden_buf, *page_data;
    mm_segment_t old_fs;
    loff_t hidden_file_offset = file->f_pos;

    print_entry_location();
    fist_dprint(6, "UPDATEPAGE1: buf[0]=%d, offset=0x%x, count=%d, sync=%d\n",
		(int) buf[0], (int) offset, count, sync);

    if (!hidden_file->f_op || !hidden_file->f_op->write)
	goto out;

    fist_print_file("Updatepage BEFORE file", file);
    fist_print_file("Updatepage BEFORE hidden_file", hidden_file);

    hidden_buf = kmalloc(count, GFP_KERNEL);
    if (!hidden_buf) {
	err = -ENOMEM;
	goto out;
    }
    wrapfs_encode_block(buf, hidden_buf, count, page->inode, page->inode->i_sb);

    old_fs = get_fs();
    set_fs(KERNEL_DS);
    hidden_file->f_pos = file->f_pos;
    err = hidden_file->f_op->write(hidden_file,
				   hidden_buf,
				   count,
				   &hidden_file_offset);
    set_fs(old_fs);
    fist_print_file("Updatepage AFTER file", file);
    fist_print_file("Updatepage AFTER hidden_file", hidden_file);
    if (err >= 0) {
	//	file->f_pos = hidden_file_offset;
	page_data = (char *) page_address(page);
	memcpy(page_data, buf, count);
	//	file->f_pos = hidden_file->f_pos;
    }
    kfree_s(hidden_buf, count);

 out:
    print_exit_status(err);
    return err;
}
#endif /* NOT_NEEDED */


STATIC int
wrapfs_inode_revalidate(dentry_t *dentry)
{
    int err = 0;
    dentry_t *hidden_dentry = wrapfs_hidden_dentry(dentry);
    inode_t *hidden_inode = hidden_dentry->d_inode;

    print_entry_location();
    //    fist_print_dentry("inode_revalidate IN", dentry);

    if (hidden_inode->i_op && hidden_inode->i_op->revalidate) {
	err = hidden_inode->i_op->revalidate(hidden_dentry);
	if (!err)
	    fist_copy_attr_all(dentry->d_inode, hidden_inode);
    }

    //    fist_print_dentry("inode_revalidate OUT", dentry);
    print_exit_status(err);
    return err;
}


#if 0
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,2,18)
/*
 * XXX: we may not need this function if not FIST_FILTER_DATA.
 * XXX: for FIST_FILTER_SCA, get all lower pages and sync them each.
 */
STATIC int
wrapfs_sync_page(page_t *page)
{
    int err = 0;
    inode_t *inode = (inode_t *) page->inode;
    inode_t *hidden_inode = itohi(inode);
    page_t *hidden_page;

    print_entry_location();

    /*
     * find lower page. returns a locked page, and may also return
     * a newly allocated page.
     * XXX: if page is allocated, it may not have any data in it.  What do
     * we do then? (What are the exact semantics of ->sync_page?
     */
    hidden_page = get_cached_page(hidden_inode, page->offset, 1);
    if (!hidden_page)
	goto out;

    err = sync_page(hidden_page);

    UnlockPage(hidden_page);	/* b/c get_cached_page locked it */
    page_cache_release(hidden_page); /* b/c get_cached_page increased refcnt */

out:
    print_exit_location();
    return err;
}


STATIC int
wrapfs_prepare_write(file_t *file, page_t *page, unsigned from, unsigned to)
{
#error need to be ported correctly from 2.4. For now, wrapfs over nfs in 2.2
#error will not work.
}
#endif /* kernel 2.2.18 or newer */
#endif

struct inode_operations wrapfs_main_iops =
{
    default_file_ops:	&wrapfs_main_fops,
    create:		wrapfs_create,
    lookup:		wrapfs_lookup,
    link:		wrapfs_link,
    unlink:		wrapfs_unlink,
    symlink:		wrapfs_symlink,
    mkdir:		wrapfs_mkdir,
    rmdir:		wrapfs_rmdir,
    mknod:		wrapfs_mknod,
    rename:		wrapfs_rename,
    readlink:		wrapfs_readlink,
    follow_link:	wrapfs_follow_link,
    readpage:		wrapfs_readpage,
#ifdef NOT_NEEDED
    write_page:		wrapfs_writepage,
#endif /* NOT_NEEDED */
    bmap:		wrapfs_bmap,
    truncate:		wrapfs_truncate,
    permission:		wrapfs_permission,
    smap:		wrapfs_smap,
#ifdef NOT_NEEDED
    updatepage:		wrapfs_updatepage,
#endif /* NOT_NEEDED */
    revalidate:		wrapfs_inode_revalidate,
#if 0
#error need proper porting
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,2,18)
    prepare_write:	wrapfs_prepare_write,
    sync_page:		wrapfs_sync_page,
#endif /* kernel 2.2.18 or newer */
#endif
};

struct inode_operations wrapfs_dir_iops =
{
    default_file_ops:	&wrapfs_dir_fops,
    create:		wrapfs_create,
    lookup:		wrapfs_lookup,
    link:		wrapfs_link,
    unlink:		wrapfs_unlink,
    symlink:		wrapfs_symlink,
    mkdir:		wrapfs_mkdir,
    rmdir:		wrapfs_rmdir,
    mknod:		wrapfs_mknod,
    rename:		wrapfs_rename,
    readlink:		wrapfs_readlink,
    follow_link:	wrapfs_follow_link,
    readpage:		wrapfs_readpage,
#ifdef NOT_NEEDED
    write_page:		wrapfs_writepage,
#endif /* NOT_NEEDED */
    bmap:		wrapfs_bmap,
    truncate:		wrapfs_truncate,
    permission:		wrapfs_permission,
    smap:		wrapfs_smap,
#ifdef NOT_NEEDED
    updatepage:		wrapfs_updatepage,
#endif /* NOT_NEEDED */
    revalidate:		wrapfs_inode_revalidate,
#if 0
#error need proper porting
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,2,18)
    prepare_write:	wrapfs_prepare_write,
    sync_page:		wrapfs_sync_page,
#endif /* kernel 2.2.18 or newer */
#endif
};

/*
 * Local variables:
 * c-basic-offset: 4
 * End:
 */
